module FileFormat where

import Parsers
import CfFormat
import TypeFormat
import MorphFormat


-- notation for functional grammar files. AR 1998

type CatList = [(String,Int)]
type VarList = [(String,Int)]
type ArgList = [(String,(Int,Cat))]
type ParList = [(String,[String])]


-- quasi context free grammars

PGrammar ::  Parser Char Grammar
PCats ::     Parser Char CatList
PCat ::      CatList -> Parser Char Cat
PFunction :: CatList -> Parser Char (Function,VarList)
PPattern ::  VarList -> Parser Char Pattern 
PCFRule ::   CatList -> Parser Char Rule
PRule ::     CatList -> Parser Char Rule

PGrammar =
 PJ PCats .>. (\C -> 
 PTList ";" 
  (PJunk +.. PConst ... 
  (PJ (literal ':') +.. PRule C ||| 
   PJ (literal '.') +.. PCFRule C) ..+ PJunk) .>. (\R ->
 succeed R)) 

PCats =
 PTList "," (PJunk +.. some alphanum) .>. (\C ->
 succeed (zip C [0..]))

PRule C = 
 PFunction C .>. (\ (F,V) ->
 (PJunk +.. literal '"' +.. PPattern V ..+ literal '"') .>. (\P ->
 succeed (F,P)))

PCFRule C =
 (PCat C ..+ PJ (literals "->")) .>. (\A ->
 PTList " " (PCat C *** Left ||| PWord *** Right) .>. (\B ->
 succeed (([X | Left X <- B], A), patt 0 B)))
  where patt _ []            = []
        patt n (Left C  : L) = Left n  : Right " " : patt (n+1) L
        patt n (Right s : L) = Right s : Right " " : patt n L

PFunction C =
 (succeed [] ||| (literal '(' +.. 
 PTList ")(" (PVar ... literal ':' +.. PCat C) ..+
 literal ')' )) .>. (\L ->
 PCat C .>. (\A ->
 succeed ((map snd L, A), zip (map fst L) [0..])))

PPattern V =
 many (PLookup V   *** Left  ||| 
       PWord       *** Right)

PCat C = 
 PLookup C *** Cat



-- grammar with morphological parametres

PMGrammar :: Parser Char MGrammar
PMGrammar =
 literals "Categories" +.. PJ PCats .>. (\C ->
 literals "Parametres" +.. PJ PMPar .>. (\M ->
 literals "Operations" +.. PJ (PMOper M) .>. (\O ->
 PTList ";" 
  (PJunk +.. PConst ... PJ (literal ':') +.. PMRule C M O ..+ PJunk) .>. (\R ->
 succeed R))))

PMOper :: ParList -> Parser Char [(String,Str)]
PMOper M =
 succeed [] |||
 PTList ";" 
  (PJunk +.. some alphanum ... 
             PArgList (PElem (map fst M)) ... 
             PJ (literal '=') +.. PTList "," PStr ..+ PSpace
   *** (\ (f,(X,t)) -> (f, makeTable X t)))
  where 
   makeTable X t = zip (tags X) t
   tags (X:P)    = [ x : p | x <- pars X, p <- tags P]
   tags []       = [[]]
   pars X        = case lookup X M of Just L -> tail L   --head is a variable
--  e.g. etre(Num,Mod) = est, sont, soit, soient

PStr =
  PWord ... literal '/' +.. PWord *** vocpair
 |||
  PWord ... PArgList PWord ... literal '/' +.. PWord *** excpair
 |||
  PWord *** trivStr
   where
    vocpair (x,y)     = ([(["a","e","i","o","u","y","\\"],y)],x)
    trivStr x         = ([],x)
    excpair (x,(y,z)) = ([(y,x)],z) 

PMRule :: CatList -> ParList -> [(String,Str)] -> Parser Char MRule
PMRule C M O = 
 PFunction C .>. (\ (F,V) ->
 (PJunk +.. literal '-' +.. PJunk +.. PMEntry V M O) .>. (\P ->
 succeed (F,P)))

PMEntry :: VarList -> ParList -> [(String,Str)] -> Parser Char MEntry
PMEntry V M O = PMPatterns V M O ... PMParam V M

PMPatterns :: VarList -> ParList -> [(String,Str)] -> 
                                    Parser Char [(MTags,MPattern)]
PMPatterns V M O =
 (succeed [] ||| 
  (literal '(' +.. PTList "," (PMVar M) ..+ literal ')')) .>. (\W ->
 (literal '"' +.. PMPattern V M O ..+ literal '"') .>. (\P ->
 succeed (OpenPF W P)))
  where
   OpenPF W P     = map contractMPatt [(m,sPattern W m P) | m <- tags W]
   tags (X:W)     = [x : p | x <- pars X, p <- tags W]
   tags []        = [[]]
   pars X         = case lookup X [(head K,tail K) | (c,K) <- M] of Just L -> L
   sPattern W m P = map (sItem (zip W m)) P 
   sItem g (Left (i,p))   = Left  (i, sMParam g p) 
   sItem g (Right (t,p))  = Right (t, sMParam g p)
   sMParam g (Left z : p) = Left z : sMParam g p 
   sMParam g (Right z : p) = case lookup z g of Just c -> Right c : sMParam g p
                                                _      -> Right z : sMParam g p
   sMParam _ []            = []
   contractMPatt (m,K) = 
    case K of 
      [Right (T,m')] -> if eqm m m' then (m,[Right ([([],s T)],[])]) else (m,K)
      _              -> (m,K)
     where eqm x y = case (x,y) of ([],[])            -> True
                                   (a:l, Right b : k) -> a==b && eqm l k
                                   _                  -> False
           s T     = case lookup m T of Just s -> s

PMPattern :: VarList -> ParList -> [(String,Str)] -> Parser Char MPattern
PMPattern V M O = 
 PTList " " (PLookup V ... PMArgs V M *** Left  ||| 
             PLookup O ... PMArgs V M *** Right |||
             PWord ... literal '+' +.. PLookup O ... PMArgs V M *** suffixPatt |||
             PStr                     *** (\s -> Right ([([],s)],[])))
  where 
   suffixPatt (s,(f,x)) = Right ([(m,(gg s l,s++d)) | (m,(l,d)) <- f],x)
   gg s l               = [(c,s++t) | (c,t) <- l]

PMArgs :: VarList -> ParList -> Parser Char MArgs
PMArgs V M = 
  succeed [] |||
  PArgList (PElem (map fst M) ... PParenth (PLookup V) *** Left . makeTag |||
            PMTag *** Right)
   where 
    makeTag (m,x) = (x,m)
    PMTag         = PElem (foldl (++) [] (map snd M))

PMParam :: VarList -> ParList -> Parser Char MParam
PMParam V M = 
  PSp (literal '-') +..
           PTList ","
            (PElem (map fst M) ... PParenth (PLookup V) *** makeVTag |||
             PMTag                                      *** makeCTag)
 ||| succeed []
   where
    makeVTag (X,x) = (X,Left x)
    makeCTag x     = (type_of x, Right x)
    type_of x = case lookup x [(m,T) | (T,l) <- M, m <- l] of Just T -> T
    PMTag     = PElem (foldl (++) [] (map (tail . snd) M))




PGramm ::    Parser Char Gramm
PMGramm ::   Parser Char MGramm
--PType ::     CatList -> Parser Char Type 
--PFunct ::    CatList -> Parser Char (Funct,VarList)
--PRul ::      CatList -> Parser Char Rul
--PTerm ::     Parser Char Term

-- grammar using functions with dependent types

PGramm =
 PJ PCats .>. (\C -> 
 PTList ";" 
  (PJunk +.. PConst ... 
   (PJ (literal ':') +.. PRul C) ..+ PJunk) .>. (\R ->
 succeed R)) 

PMGramm =
 literals "Categories" +.. PJ PCats .>. (\C ->
 literals "Parametres" +.. PJ PMPar .>. (\M ->
 literals "Operations" +.. PJ (PMOper M) .>. (\O ->
 PTList ";" 
  (PJunk +.. PConst ... PJ (literal ':') +.. PMRul C M O ..+ PJunk) .>. (\R ->
 succeed R))))

PRul C = 
 PFunct C .>. (\ (F,V) ->
 (PJunk +.. literal '"' +.. PPattern V ..+ literal '"') .>. (\P ->
 succeed (F,P)))

PMRul :: CatList -> ParList -> [(String,Str)] -> Parser Char MRul
PMRul C M O = 
 PFunct C .>. (\ (F,V) ->
 (PJunk +.. literal '-' +.. PJunk +.. PMEntry V M O) .>. (\P ->
 succeed (F,P)))

PFunct C = 
 (succeed [] ||| (literal '(' +.. 
 PTList ")(" ((PVar *** Var) ... literal ':' +.. PType C) ..+
 literal ')' )) .>. (\L ->
 PCat C .>. (\A ->
 (succeed [] ||| literal '(' +.. PTList "," PTerm ..+ literal ')') .>. (\X ->
 succeed ((L,(A,X)), zip [x | (Var x,_) <- L] [0..]))))

PType C = 
 PCont C ... PCat C ... 
 (succeed [] ||| literal '(' +.. PTList "," PTerm ..+ literal ')')

PCont C = 
 succeed [] |||
 PArgList ((PVar *** Var) ... PJ (literal ':') +.. PCat C ... PArgList PTerm)

PTerm = 
 PTree *** ITree |||
 PParenth PVar ... PTerm *** (\ (x,b) -> Abs (Var x) b)

PTree =
 PVar *** Place . Var |||
 Pfun ... (succeed [] ||| literal '(' +.. PTList "," PTree ..+ literal ')')
   *** ( \ (c,l) -> Apply c l)
   where Pfun   = PConst *** Fun




-- generally used parsers

PMVar M = PElem [head x | (p,x) <- M]

PParenth P = literal '(' +.. P ..+ literal ')'
PArgList P = literal '(' +.. PTList "," (PSpace +.. P) ..+ literal ')'

PMPar :: Parser Char ParList
PMPar = 
 succeed [] ||| PTList "," (PJunk +.. some alphanum ... PArgList (some alphanum))
-- e.g. Gen(g,masc,fem), Num(n,pl,sg)

         
PVar = some alphanum |> varsymb
PConst = some alphanum |> (not . varsymb)

varsymb s =
 case s of a:b:[] -> elem a (['A'..'Z'] ++ ['a'..'z']) && elem b "0123456789"
           a:[]   -> elem a (['A'..'Z'] ++ ['a'..'z']) 
           _      -> False

alphanum = satisfy (\x -> elem x (['0'..'9'] ++ ['A'..'Z'] ++ ['a'..'z'])) 

PWord = 
 some (satisfy (\x -> 
            elem x (['0'..'9'] ++ ['A'..'Z'] ++ ['a'..'z'] ++ "'`\\"))
       ||| literal '-' *** (\x -> ' '))
 ||| literal '_' *** (\x -> "")

