% Prolog lexicon for SHARDS, from OALD machine-readable dictionary
% Produced by asc2lex, Matthew Purver 19/04/2001
%
% Manually edited for irregulars, CMTs, determiners etc.
%
% As this material is obtained from the OALD, it is freely available
% for RESEARCH PURPOSES ONLY. See the OTA's TEI header in
% ascii_0710-2.txt for more details.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Morphological interface predicates
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% noun/6
% noun( ?Stem, ?Word, ?Type, ?Number, ?Gender, ?Case )
%
% Morphological interface to noun/4 - relates word stem
% Stem to surface word Word.
% Number will be 'sing' or 'plur'
% Gender currently undefined
% Case currently always 'case' (i.e. any)
% Type will be 'mass' or 'count' (both may succeed)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

noun( Word, Word, Type, sing, _Gender, case ) :-
      noun( Word, _Plural, RawType, _SemClass ),
      noun_type( RawType, Type ).

noun( Stem, Word, Type, plur, _Gender, case ) :-
      noun( Stem, Word, RawType, _SemClass ),
      noun_type( RawType, Type ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% noun_type( +RawType, ?Type )
%
% Returns type as defined in lexicon, except for 'both'
% which gets converted to 'mass' or 'count'
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

noun_type( Type, Type ) :-
	\+ Type = both.
noun_type( both, mass ).
noun_type( both, count ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% verb/5
% verb( ?Stem, ?Word, ?VForm, ?Number, ?Type )
%
% Morphological interface to verb/7 - relates word stem
% Stem to surface word Word.
% VForm currently 'inf', 'pres' or 'past'
% Number currently 's3', 'nons3' or 'person' (i.e. undefined)
% Type will be 'intran', 'tran', 'ditran', etc.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

verb( Word, Word, inf, person, Type ) :-
	verb( Word, _, _, _, _, Cat, CatList ),
	verb_type( Cat, CatList, Type ).

verb( Word, Word, pres, nons3, Type ) :-
	verb( Word, _, _, _, _, Cat, CatList ),
	verb_type( Cat, CatList, Type ).

verb( Stem, Word, pres, s3, Type ) :-
	verb( Stem, Word, _, _, _, Cat, CatList ),
	verb_type( Cat, CatList, Type ).

verb( Stem, Word, past, person, Type ) :-
	verb( Stem, _, _, Word, _, Cat, CatList ),
	verb_type( Cat, CatList, Type ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% verb_type( +RawType, +CatList, ?Type )
%
% Type is a verb subcategory type determined from the
% RawType atom and CatList list of numbers defined in
% the lexicon
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

verb_type( intran, _, intran ).        % VP -> V
verb_type( tran, _, tran ).            % VP -> V NP
verb_type( _, CatList, ditran ) :-     % VP -> V NP NP
	memberchk( '12A', CatList );
	memberchk( '12B', CatList );
	memberchk( '12C', CatList ).
verb_type( _, CatList, stran ) :-      % VP -> V [question]
	memberchk( '10', CatList ).
verb_type( _, CatList, pp_to ) :-      % VP -> V PP
	memberchk( '3A', CatList ).
verb_type( _, CatList, pp_for ) :-     % VP -> V PP
	memberchk( '3A', CatList ).
verb_type( _, CatList, subjraise ) :-  % VP -> V [to+inf] (subject raising)
	memberchk( '4E', CatList ).
verb_type( _, CatList, subjcon ) :-    % VP -> V [to+inf] (subject control)
	memberchk( '7A', CatList ).
verb_type( _, CatList, aux ) :-        % VP -> V [inf] (auxiliary)
	memberchk( '5', CatList ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% adj/3
% adj( ?Stem, ?Word, ?Type )
%
% Morphological interface to adj/4 - relates word stem
% Stem to surface word Word.
% Type will be 'simple', 'comparative' or 'superlative'
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

adj( Word, Word, simple ) :-
	adj( Word, _, _, _ ).

adj( Stem, Word, comparative ) :-
	adj( Stem, Word, _, _ ).

adj( Stem, Word, superlative ) :-
	adj( Stem, _, Word, _ ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% General predicates
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% word( ?Word )
%
% Succeeds if Word is a word defined in the lexicon
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
word( Word ) :-
	noun( _, Word, _, _, _, _ );
	pron( Word, _, _, _ );
	verb( _, Word, _, _, _ );
	adj( _, Word, _ );
	adv( Word, _ );
	prep( Word, _ );
	conj( Word, _ );
	det( Word, _, _ );
	misc( Word, _, _ ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% open_class_word( ?Word )
%
% Succeeds if Word is a noun/verb/adj/adv in the lexicon
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
open_class_word( Word ) :-
	noun( _, Word, _, _, _, _ );
	verb( _, Word, _, _, _ );
	adj( _, Word, _ );
	adv( Word, _ ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% unknown_word( +Word )
%
% Succeeds if Word is NOT defined in the lexicon
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
unknown_word( Word ) :-
	\+ word( Word ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Lexicon
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

verb( 'hoover', 'hoovers', 'hoovering', 'hoovered', 'hoovered', tran, ['6A'] ).
verb( 'latinize', 'latinizes', 'latinizing', 'latinized', 'latinized', tran, [] ).
verb( 'roneo', 'roneos', 'roneoing', 'roneoed', 'roneoed', tran, [] ).
verb( 'x-ray', 'x-rays', 'x-raying', 'x-rayed', 'x-rayed', tran, ['6A'] ).
verb( 'xerox', 'xeroxes', 'xeroxing', 'xeroxed', 'xeroxed', tran, [] ).
verb( 'abandon', 'abandons', 'abandoning', 'abandoned', 'abandoned', tran, ['6A','14'] ).
verb( 'abase', 'abases', 'abasing', 'abased', 'abased', tran, ['6B'] ).
verb( 'abash', 'abashes', 'abashing', 'abashed', 'abashed', tran, ['6A'] ).
verb( 'abate', 'abates', 'abating', 'abated', 'abated', _, ['2A','6A'] ).
verb( 'abbreviate', 'abbreviates', 'abbreviating', 'abbreviated', 'abbreviated', tran, ['6A','14'] ).
verb( 'abdicate', 'abdicates', 'abdicating', 'abdicated', 'abdicated', _, ['2A','6A'] ).
verb( 'abduct', 'abducts', 'abducting', 'abducted', 'abducted', tran, ['6A'] ).
verb( 'abet', 'abets', 'abetting', 'abetted', 'abetted', tran, ['6A','14'] ).
verb( 'abhor', 'abhors', 'abhorring', 'abhorred', 'abhorred', tran, ['6A'] ).
verb( 'abide', 'abides', 'abiding', 'abided', 'abided', _, ['2C','3A','6A'] ).
verb( 'abjure', 'abjures', 'abjuring', 'abjured', 'abjured', tran, ['6A'] ).
verb( 'abolish', 'abolishes', 'abolishing', 'abolished', 'abolished', tran, ['6A'] ).
verb( 'abominate', 'abominates', 'abominating', 'abominated', 'abominated', tran, ['6A','6C'] ).
verb( 'abort', 'aborts', 'aborting', 'aborted', 'aborted', _, ['2A','6A'] ).
verb( 'abound', 'abounds', 'abounding', 'abounded', 'abounded', intran, ['3A'] ).
verb( 'about-face', 'about-faces', 'about-facing', 'about-faced', 'about-faced', intran, [] ).
verb( 'abrade', 'abrades', 'abrading', 'abraded', 'abraded', tran, ['6A'] ).
verb( 'abridge', 'abridges', 'abridging', 'abridged', 'abridged', tran, ['6A'] ).
verb( 'abrogate', 'abrogates', 'abrogating', 'abrogated', 'abrogated', tran, ['6A'] ).
verb( 'abscond', 'absconds', 'absconding', 'absconded', 'absconded', intran, ['2A','3A'] ).
verb( 'absent', 'absents', 'absenting', 'absented', 'absented', tran, ['6B','14'] ).
verb( 'absolve', 'absolves', 'absolving', 'absolved', 'absolved', tran, ['6A','14'] ).
verb( 'absorb', 'absorbs', 'absorbing', 'absorbed', 'absorbed', tran, ['6A'] ).
verb( 'abstain', 'abstains', 'abstaining', 'abstained', 'abstained', intran, ['2A','3A'] ).
verb( 'abstract', 'abstracts', 'abstracting', 'abstracted', 'abstracted', tran, ['6A','14'] ).
verb( 'abuse', 'abuses', 'abusing', 'abused', 'abused', tran, ['6A'] ).
verb( 'abut', 'abuts', 'abutting', 'abutted', 'abutted', intran, ['3A'] ).
verb( 'accede', 'accedes', 'acceding', 'acceded', 'acceded', intran, ['2A','3A'] ).
verb( 'accelerate', 'accelerates', 'accelerating', 'accelerated', 'accelerated', _, ['2A','6A'] ).
verb( 'accent', 'accents', 'accenting', 'accented', 'accented', tran, ['6A'] ).
verb( 'accentuate', 'accentuates', 'accentuating', 'accentuated', 'accentuated', tran, ['6A'] ).
verb( 'accept', 'accepts', 'accepting', 'accepted', 'accepted', _, ['2A','6A','9','16B'] ).
verb( 'acclaim', 'acclaims', 'acclaiming', 'acclaimed', 'acclaimed', tran, ['6A','16B','23'] ).
verb( 'acclimate', 'acclimates', 'acclimating', 'acclimated', 'acclimated', _, [] ).
verb( 'acclimatize', 'acclimatizes', 'acclimatizing', 'acclimatized', 'acclimatized', _, ['2A','14'] ).
verb( 'accommodate', 'accommodates', 'accommodating', 'accommodated', 'accommodated', tran, ['6A','14'] ).
verb( 'accompany', 'accompanies', 'accompanying', 'accompanied', 'accompanied', tran, ['6A','14'] ).
verb( 'accomplish', 'accomplishes', 'accomplishing', 'accomplished', 'accomplished', tran, ['6A'] ).
verb( 'accord', 'accords', 'according', 'accorded', 'accorded', _, ['2A','2C','3A','12A','13A'] ).
verb( 'accost', 'accosts', 'accosting', 'accosted', 'accosted', tran, ['6A'] ).
verb( 'account', 'accounts', 'accounting', 'accounted', 'accounted', _, ['3A','25'] ).
verb( 'accredit', 'accredits', 'accrediting', 'accredited', 'accredited', tran, ['14'] ).
verb( 'accrue', 'accrues', 'accruing', 'accrued', 'accrued', intran, ['2A','3A'] ).
verb( 'accumulate', 'accumulates', 'accumulating', 'accumulated', 'accumulated', _, ['2A','6A'] ).
verb( 'accuse', 'accuses', 'accusing', 'accused', 'accused', tran, ['6A','14'] ).
verb( 'accustom', 'accustoms', 'accustoming', 'accustomed', 'accustomed', tran, ['14'] ).
verb( 'ache', 'aches', 'aching', 'ached', 'ached', intran, ['2A','3A','4A'] ).
verb( 'achieve', 'achieves', 'achieving', 'achieved', 'achieved', tran, ['6A'] ).
verb( 'acidify', 'acidifies', 'acidifying', 'acidified', 'acidified', _, ['2A','6A'] ).
verb( 'acknowledge', 'acknowledges', 'acknowledging', 'acknowledged', 'acknowledged', tran, ['6A','6C','9','16B','24A','25'] ).
verb( 'acquaint', 'acquaints', 'acquainting', 'acquainted', 'acquainted', tran, ['14'] ).
verb( 'acquiesce', 'acquiesces', 'acquiescing', 'acquiesced', 'acquiesced', intran, ['2A','3A'] ).
verb( 'acquire', 'acquires', 'acquiring', 'acquired', 'acquired', tran, ['6A'] ).
verb( 'acquit', 'acquits', 'acquitting', 'acquitted', 'acquitted', tran, ['6A','14','16B'] ).
verb( 'act', 'acts', 'acting', 'acted', 'acted', _, ['2A','2C','3A','6A','15B'] ).
verb( 'activate', 'activates', 'activating', 'activated', 'activated', tran, ['6A'] ).
verb( 'actuate', 'actuates', 'actuating', 'actuated', 'actuated', tran, ['6A'] ).
verb( 'ad-lib', 'ad-libs', 'ad-libbing', 'ad-libbed', 'ad-libbed', intran, ['2A'] ).
verb( 'adapt', 'adapts', 'adapting', 'adapted', 'adapted', tran, ['6A','14'] ).
verb( 'add', 'adds', 'adding', 'added', 'added', _, ['2C','3A','6A','9','14','15B'] ).
verb( 'addict', 'addicts', 'addicting', 'addicted', 'addicted', tran, [] ).
verb( 'addle', 'addles', 'addling', 'addled', 'addled', _, ['2A','6A'] ).
verb( 'address', 'addresses', 'addressing', 'addressed', 'addressed', tran, ['6A','14','16B'] ).
verb( 'adduce', 'adduces', 'adducing', 'adduced', 'adduced', tran, ['6A'] ).
verb( 'adhere', 'adheres', 'adhering', 'adhered', 'adhered', intran, ['2A','3A'] ).
verb( 'adjoin', 'adjoins', 'adjoining', 'adjoined', 'adjoined', _, ['2A','6A'] ).
verb( 'adjourn', 'adjourns', 'adjourning', 'adjourned', 'adjourned', _, ['2A','2C','6A'] ).
verb( 'adjudge', 'adjudges', 'adjudging', 'adjudged', 'adjudged', tran, ['9','14','25'] ).
verb( 'adjudicate', 'adjudicates', 'adjudicating', 'adjudicated', 'adjudicated', _, ['2A','3A','6A','14','25'] ).
verb( 'adjure', 'adjures', 'adjuring', 'adjured', 'adjured', tran, ['17'] ).
verb( 'adjust', 'adjusts', 'adjusting', 'adjusted', 'adjusted', tran, ['6A','14'] ).
verb( 'administer', 'administers', 'administering', 'administered', 'administered', _, ['6A','14'] ).
verb( 'admire', 'admires', 'admiring', 'admired', 'admired', tran, ['6A'] ).
verb( 'admit', 'admits', 'admitting', 'admitted', 'admitted', _, ['3A','6A','6C','9','14','25'] ).
verb( 'admix', 'admixes', 'admixing', 'admixed', 'admixed', _, ['2A','6A'] ).
verb( 'admonish', 'admonishes', 'admonishing', 'admonished', 'admonished', tran, ['6A','14'] ).
verb( 'adopt', 'adopts', 'adopting', 'adopted', 'adopted', tran, ['6A'] ).
verb( 'adore', 'adores', 'adoring', 'adored', 'adored', tran, ['6A','6C'] ).
verb( 'adorn', 'adorns', 'adorning', 'adorned', 'adorned', tran, ['6A','14'] ).
verb( 'adulterate', 'adulterates', 'adulterating', 'adulterated', 'adulterated', tran, ['6A','14'] ).
verb( 'adumbrate', 'adumbrates', 'adumbrating', 'adumbrated', 'adumbrated', tran, ['6A'] ).
verb( 'advance', 'advances', 'advancing', 'advanced', 'advanced', _, ['2A','2B','3A','6A','12A','13A','14'] ).
verb( 'advantage', 'advantages', 'advantaging', 'advantaged', 'advantaged', tran, ['6A'] ).
verb( 'adventure', 'adventures', 'adventuring', 'adventured', 'adventured', tran, [] ).
verb( 'advert', 'adverts', 'adverting', 'adverted', 'adverted', intran, ['3A'] ).
verb( 'advertise', 'advertises', 'advertising', 'advertised', 'advertised', _, ['2A','3A','6A'] ).
verb( 'advise', 'advises', 'advising', 'advised', 'advised', _, ['3A','6A','6C','14','17','20','21'] ).
verb( 'advocate', 'advocates', 'advocating', 'advocated', 'advocated', tran, ['6A','6C'] ).
verb( 'aerate', 'aerates', 'aerating', 'aerated', 'aerated', tran, ['6A'] ).
verb( 'affect', 'affects', 'affecting', 'affected', 'affected', tran, ['6A','7A'] ).
verb( 'affiance', 'affiances', 'affiancing', 'affianced', 'affianced', tran, ['6A'] ).
verb( 'affiliate', 'affiliates', 'affiliating', 'affiliated', 'affiliated', _, ['2A','6A','14'] ).
verb( 'affirm', 'affirms', 'affirming', 'affirmed', 'affirmed', _, ['2A','6A','9','14'] ).
verb( 'affix', 'affixes', 'affixing', 'affixed', 'affixed', tran, ['6A','14'] ).
verb( 'afflict', 'afflicts', 'afflicting', 'afflicted', 'afflicted', tran, ['6A','14'] ).
verb( 'afford', 'affords', 'affording', 'afforded', 'afforded', tran, ['6A','7A','12A','13A'] ).
verb( 'afforest', 'afforests', 'afforesting', 'afforested', 'afforested', tran, ['6A'] ).
verb( 'affranchise', 'affranchises', 'affranchising', 'affranchised', 'affranchised', tran, ['6A'] ).
verb( 'affront', 'affronts', 'affronting', 'affronted', 'affronted', tran, ['6A'] ).
verb( 'age', 'ages', 'aging', 'aged', 'aged', _, ['2A','6A'] ).
verb( 'agglomerate', 'agglomerates', 'agglomerating', 'agglomerated', 'agglomerated', _, ['2A','6A'] ).
verb( 'agglutinate', 'agglutinates', 'agglutinating', 'agglutinated', 'agglutinated', tran, ['2A','6A'] ).
verb( 'aggrandize', 'aggrandizes', 'aggrandizing', 'aggrandized', 'aggrandized', tran, ['6A'] ).
verb( 'aggravate', 'aggravates', 'aggravating', 'aggravated', 'aggravated', tran, ['6A'] ).
verb( 'aggregate', 'aggregates', 'aggregating', 'aggregated', 'aggregated', _, ['2A','2E','6A'] ).
verb( 'aggrieve', 'aggrieves', 'aggrieving', 'aggrieved', 'aggrieved', tran, [] ).
verb( 'agitate', 'agitates', 'agitating', 'agitated', 'agitated', _, ['3A','6A'] ).
verb( 'agree', 'agrees', 'agreeing', 'agreed', 'agreed', _, ['2A','2C','3A','3B','4C','6A','7A'] ).
verb( 'aid', 'aids', 'aiding', 'aided', 'aided', tran, ['6A','14','17'] ).
verb( 'ail', 'ails', 'ailing', 'ailed', 'ailed', _, ['2A','2B','6A'] ).
verb( 'aim', 'aims', 'aiming', 'aimed', 'aimed', _, ['2A','3A','4A','6A','14'] ).
verb( 'air', 'airs', 'airing', 'aired', 'aired', tran, ['6A'] ).
verb( 'alarm', 'alarms', 'alarming', 'alarmed', 'alarmed', tran, ['6A'] ).
verb( 'alert', 'alerts', 'alerting', 'alerted', 'alerted', tran, ['6A'] ).
verb( 'alienate', 'alienates', 'alienating', 'alienated', 'alienated', tran, ['6A','14'] ).
verb( 'alight', 'alights', 'alighting', 'alighted', 'alighted', intran, ['2A','3A'] ).
verb( 'align', 'aligns', 'aligning', 'aligned', 'aligned', _, ['2A','3A','6A','14'] ).
verb( 'allay', 'allays', 'allaying', 'allayed', 'allayed', tran, ['6A'] ).
verb( 'allege', 'alleges', 'alleging', 'alleged', 'alleged', tran, ['6A','9'] ).
verb( 'alleviate', 'alleviates', 'alleviating', 'alleviated', 'alleviated', tran, ['6A'] ).
verb( 'allocate', 'allocates', 'allocating', 'allocated', 'allocated', tran, ['6A','14'] ).
verb( 'allot', 'allots', 'allotting', 'allotted', 'allotted', tran, ['6A','12A','13A','14'] ).
verb( 'allow', 'allows', 'allowing', 'allowed', 'allowed', _, ['3A','6A','6C','9','12A','13A','14','15B','17','25'] ).
verb( 'alloy', 'alloys', 'alloying', 'alloyed', 'alloyed', tran, ['6A'] ).
verb( 'allude', 'alludes', 'alluding', 'alluded', 'alluded', intran, ['3A'] ).
verb( 'allure', 'allures', 'alluring', 'allured', 'allured', tran, ['6A','14','17'] ).
verb( 'ally', 'allies', 'allying', 'allied', 'allied', tran, ['14'] ).
verb( 'alter', 'alters', 'altering', 'altered', 'altered', _, ['2A','6A'] ).
verb( 'alternate', 'alternates', 'alternating', 'alternated', 'alternated', _, ['3A','6A','14'] ).
verb( 'amalgamate', 'amalgamates', 'amalgamating', 'amalgamated', 'amalgamated', _, ['2A','6A'] ).
verb( 'amass', 'amasses', 'amassing', 'amassed', 'amassed', tran, ['6A'] ).
verb( 'amaze', 'amazes', 'amazing', 'amazed', 'amazed', tran, ['6A'] ).
verb( 'amble', 'ambles', 'ambling', 'ambled', 'ambled', intran, ['2A','2C'] ).
verb( 'ambuscade', 'ambuscades', 'ambuscading', 'ambuscaded', 'ambuscaded', tran, [] ).
verb( 'ambush', 'ambushes', 'ambushing', 'ambushed', 'ambushed', tran, ['6A'] ).
verb( 'ameliorate', 'ameliorates', 'ameliorating', 'ameliorated', 'ameliorated', _, ['2A','6A'] ).
verb( 'amend', 'amends', 'amending', 'amended', 'amended', _, ['2A','6A'] ).
verb( 'amortize', 'amortizes', 'amortizing', 'amortized', 'amortized', tran, ['6A'] ).
verb( 'amount', 'amounts', 'amounting', 'amounted', 'amounted', intran, ['3A'] ).
verb( 'amplify', 'amplifies', 'amplifying', 'amplified', 'amplified', tran, ['6A'] ).
verb( 'amputate', 'amputates', 'amputating', 'amputated', 'amputated', tran, ['6A'] ).
verb( 'amuse', 'amuses', 'amusing', 'amused', 'amused', tran, ['6A'] ).
verb( 'anaesthetize', 'anaesthetizes', 'anaesthetizing', 'anaesthetized', 'anaesthetized', tran, ['6A'] ).
verb( 'analyse', 'analyses', 'analysing', 'analysed', 'analysed', tran, ['6A'] ).
verb( 'analyze', 'analyzes', 'analyzing', 'analyzed', 'analyzed', tran, ['6A'] ).
verb( 'anathematize', 'anathematizes', 'anathematizing', 'anathematized', 'anathematized', _, [] ).
verb( 'anchor', 'anchors', 'anchoring', 'anchored', 'anchored', _, ['2A','6A'] ).
verb( 'anesthetize', 'anesthetizes', 'anesthetizing', 'anesthetized', 'anesthetized', tran, ['6A'] ).
verb( 'anger', 'angers', 'angering', 'angered', 'angered', tran, ['6A'] ).
verb( 'angle', 'angles', 'angling', 'angled', 'angled', _, ['2A','3A','6A'] ).
verb( 'angle-park', 'angle-parks', 'angle-parking', 'angle-parked', 'angle-parked', _, [] ).
verb( 'anglicize', 'anglicizes', 'anglicizing', 'anglicized', 'anglicized', tran, ['6A'] ).
verb( 'animadvert', 'animadverts', 'animadverting', 'animadverted', 'animadverted', intran, ['3A'] ).
verb( 'animate', 'animates', 'animating', 'animated', 'animated', tran, ['6A','14'] ).
verb( 'anneal', 'anneals', 'annealing', 'annealed', 'annealed', tran, ['6A'] ).
verb( 'annex', 'annexes', 'annexing', 'annexed', 'annexed', tran, ['6A','14'] ).
verb( 'annihilate', 'annihilates', 'annihilating', 'annihilated', 'annihilated', tran, ['6A'] ).
verb( 'annotate', 'annotates', 'annotating', 'annotated', 'annotated', tran, ['6A'] ).
verb( 'announce', 'announces', 'announcing', 'announced', 'announced', tran, ['6A','9','14'] ).
verb( 'annoy', 'annoys', 'annoying', 'annoyed', 'annoyed', tran, ['6A'] ).
verb( 'annul', 'annuls', 'annulling', 'annulled', 'annulled', tran, ['6A'] ).
verb( 'annunciate', 'annunciates', 'annunciating', 'annunciated', 'annunciated', tran, ['6A'] ).
verb( 'anoint', 'anoints', 'anointing', 'anointed', 'anointed', tran, ['6A','14','23'] ).
verb( 'answer', 'answers', 'answering', 'answered', 'answered', _, ['2A','2C','3A','6A','9','12A','15B'] ).
verb( 'antagonize', 'antagonizes', 'antagonizing', 'antagonized', 'antagonized', tran, ['6A'] ).
verb( 'antedate', 'antedates', 'antedating', 'antedated', 'antedated', tran, ['6A'] ).
verb( 'anticipate', 'anticipates', 'anticipating', 'anticipated', 'anticipated', tran, ['6A','6C','9'] ).
verb( 'ape', 'apes', 'aping', 'aped', 'aped', tran, [] ).
verb( 'apologize', 'apologizes', 'apologizing', 'apologized', 'apologized', intran, ['2A','3A'] ).
verb( 'apostrophize', 'apostrophizes', 'apostrophizing', 'apostrophized', 'apostrophized', tran, [] ).
verb( 'appal', 'appals', 'appalling', 'appalled', 'appalled', tran, ['6A'] ).
verb( 'apparel', 'apparels', 'apparelling', 'apparelled', 'apparelled', tran, [] ).
verb( 'appeal', 'appeals', 'appealing', 'appealed', 'appealed', intran, ['2A','3A'] ).
verb( 'appear', 'appears', 'appearing', 'appeared', 'appeared', intran, ['2A','2C','4D','4E'] ).
verb( 'appease', 'appeases', 'appeasing', 'appeased', 'appeased', tran, ['6A'] ).
verb( 'append', 'appends', 'appending', 'appended', 'appended', tran, ['6A','14'] ).
verb( 'appertain', 'appertains', 'appertaining', 'appertained', 'appertained', intran, ['3A'] ).
verb( 'applaud', 'applauds', 'applauding', 'applauded', 'applauded', _, ['2A','2B','6A'] ).
verb( 'appliqu_e', 'appliqu_es', 'appliqu_eing', 'appliqu_eed', 'appliqu_eed', tran, [] ).
verb( 'apply', 'applies', 'applying', 'applied', 'applied', _, ['2C','3A','6A','14'] ).
verb( 'appoint', 'appoints', 'appointing', 'appointed', 'appointed', tran, ['6A','9','14','16A','23','25'] ).
verb( 'apportion', 'apportions', 'apportioning', 'apportioned', 'apportioned', tran, ['6A','12B','13B','14'] ).
verb( 'appraise', 'appraises', 'appraising', 'appraised', 'appraised', tran, ['6A'] ).
verb( 'appreciate', 'appreciates', 'appreciating', 'appreciated', 'appreciated', _, ['2B','6A','9','10'] ).
verb( 'apprehend', 'apprehends', 'apprehending', 'apprehended', 'apprehended', tran, ['6A','9'] ).
verb( 'apprentice', 'apprentices', 'apprenticing', 'apprenticed', 'apprenticed', tran, ['6A','14'] ).
verb( 'apprise', 'apprises', 'apprising', 'apprised', 'apprised', tran, ['14'] ).
verb( 'approach', 'approaches', 'approaching', 'approached', 'approached', _, ['2A','6A'] ).
verb( 'appropriate', 'appropriates', 'appropriating', 'appropriated', 'appropriated', tran, ['6A','14'] ).
verb( 'approve', 'approves', 'approving', 'approved', 'approved', _, ['3A','6A'] ).
verb( 'approximate', 'approximates', 'approximating', 'approximated', 'approximated', _, ['3A','6A'] ).
verb( 'aquaplane', 'aquaplanes', 'aquaplaning', 'aquaplaned', 'aquaplaned', intran, [] ).
verb( 'arbitrate', 'arbitrates', 'arbitrating', 'arbitrated', 'arbitrated', _, ['2A','6A'] ).
verb( 'arch', 'arches', 'arching', 'arched', 'arched', _, ['2C','6A'] ).
verb( 'argue', 'argues', 'arguing', 'argued', 'argued', _, ['2A','2C','3A','6A','9','14'] ).
verb( 'arise', 'arises', 'arising', 'arose', 'arisen', intran, ['2A','3A'] ).
verb( 'arm', 'arms', 'arming', 'armed', 'armed', _, ['2A','6A','14'] ).
verb( 'arouse', 'arouses', 'arousing', 'aroused', 'aroused', tran, ['6A','14'] ).
verb( 'arraign', 'arraigns', 'arraigning', 'arraigned', 'arraigned', tran, ['6A','14'] ).
verb( 'arrange', 'arranges', 'arranging', 'arranged', 'arranged', _, ['3A','4C','6A','14','15A'] ).
verb( 'array', 'arrays', 'arraying', 'arrayed', 'arrayed', tran, ['6A','15A'] ).
verb( 'arrest', 'arrests', 'arresting', 'arrested', 'arrested', tran, ['6A'] ).
verb( 'arrive', 'arrives', 'arriving', 'arrived', 'arrived', intran, ['2A','2C','3A'] ).
verb( 'arrogate', 'arrogates', 'arrogating', 'arrogated', 'arrogated', tran, ['14'] ).
verb( 'article', 'articles', 'articling', 'articled', 'articled', tran, [] ).
verb( 'articulate', 'articulates', 'articulating', 'articulated', 'articulated', _, ['2C','6A','15A'] ).
verb( 'ascend', 'ascends', 'ascending', 'ascended', 'ascended', _, ['2A','2C','6A'] ).
verb( 'ascertain', 'ascertains', 'ascertaining', 'ascertained', 'ascertained', tran, ['6A','8','9','10','17'] ).
verb( 'ascribe', 'ascribes', 'ascribing', 'ascribed', 'ascribed', tran, ['14'] ).
verb( 'ask', 'asks', 'asking', 'asked', 'asked', _, ['3A','6A','7A','8','9','10','12C','14','15B','17','20','21'] ).
verb( 'asperse', 'asperses', 'aspersing', 'aspersed', 'aspersed', tran, ['6A'] ).
verb( 'asphalt', 'asphalts', 'asphalting', 'asphalted', 'asphalted', tran, ['6A'] ).
verb( 'asphyxiate', 'asphyxiates', 'asphyxiating', 'asphyxiated', 'asphyxiated', tran, ['6A'] ).
verb( 'aspirate', 'aspirates', 'aspirating', 'aspirated', 'aspirated', tran, [] ).
verb( 'aspire', 'aspires', 'aspiring', 'aspired', 'aspired', intran, ['3A','4A'] ).
verb( 'assail', 'assails', 'assailing', 'assailed', 'assailed', tran, ['6A','14'] ).
verb( 'assassinate', 'assassinates', 'assassinating', 'assassinated', 'assassinated', tran, ['6A'] ).
verb( 'assault', 'assaults', 'assaulting', 'assaulted', 'assaulted', tran, ['6A'] ).
verb( 'assay', 'assays', 'assaying', 'assayed', 'assayed', tran, ['6A','7A'] ).
verb( 'assemble', 'assembles', 'assembling', 'assembled', 'assembled', _, ['2A','6A'] ).
verb( 'assent', 'assents', 'assenting', 'assented', 'assented', intran, ['2A','3A'] ).
verb( 'assert', 'asserts', 'asserting', 'asserted', 'asserted', tran, ['6A','9','25'] ).
verb( 'assess', 'assesses', 'assessing', 'assessed', 'assessed', tran, ['6A','14'] ).
verb( 'asseverate', 'asseverates', 'asseverating', 'asseverated', 'asseverated', tran, ['6A','9'] ).
verb( 'assign', 'assigns', 'assigning', 'assigned', 'assigned', tran, ['12A','13A','13B','14','17'] ).
verb( 'assimilate', 'assimilates', 'assimilating', 'assimilated', 'assimilated', _, ['2A','3A','6A'] ).
verb( 'assist', 'assists', 'assisting', 'assisted', 'assisted', _, ['2A','3A','6A','14','17'] ).
verb( 'associate', 'associates', 'associating', 'associated', 'associated', _, ['3A','14'] ).
verb( 'assuage', 'assuages', 'assuaging', 'assuaged', 'assuaged', tran, ['6A'] ).
verb( 'assume', 'assumes', 'assuming', 'assumed', 'assumed', tran, ['6A','9','25'] ).
verb( 'assure', 'assures', 'assuring', 'assured', 'assured', tran, ['6A','11','14'] ).
verb( 'astonish', 'astonishes', 'astonishing', 'astonished', 'astonished', tran, ['6A'] ).
verb( 'astound', 'astounds', 'astounding', 'astounded', 'astounded', tran, ['6A'] ).
verb( 'atomize', 'atomizes', 'atomizing', 'atomized', 'atomized', tran, [] ).
verb( 'atone', 'atones', 'atoning', 'atoned', 'atoned', intran, ['2A','3A'] ).
verb( 'atrophy', 'atrophies', 'atrophying', 'atrophied', 'atrophied', _, ['2A','2B','6A'] ).
verb( 'attach', 'attaches', 'attaching', 'attached', 'attached', _, ['3A','6A','14'] ).
verb( 'attack', 'attacks', 'attacking', 'attacked', 'attacked', tran, ['6A'] ).
verb( 'attain', 'attains', 'attaining', 'attained', 'attained', _, ['3A','6A'] ).
verb( 'attempt', 'attempts', 'attempting', 'attempted', 'attempted', tran, ['6A','7A'] ).
verb( 'attend', 'attends', 'attending', 'attended', 'attended', _, ['2A','3A','6A'] ).
verb( 'attenuate', 'attenuates', 'attenuating', 'attenuated', 'attenuated', tran, ['6A'] ).
verb( 'attest', 'attests', 'attesting', 'attested', 'attested', _, ['2A','3A','6A'] ).
verb( 'attire', 'attires', 'attiring', 'attired', 'attired', tran, ['6A'] ).
verb( 'attitudinize', 'attitudinizes', 'attitudinizing', 'attitudinized', 'attitudinized', intran, ['2A'] ).
verb( 'attract', 'attracts', 'attracting', 'attracted', 'attracted', tran, ['6A'] ).
verb( 'attribute', 'attributes', 'attributing', 'attributed', 'attributed', tran, ['14'] ).
verb( 'attune', 'attunes', 'attuning', 'attuned', 'attuned', tran, ['14'] ).
verb( 'auction', 'auctions', 'auctioning', 'auctioned', 'auctioned', tran, ['6A','15B'] ).
verb( 'audit', 'audits', 'auditing', 'audited', 'audited', tran, ['6A'] ).
verb( 'audition', 'auditions', 'auditioning', 'auditioned', 'auditioned', tran, ['6A'] ).
verb( 'augment', 'augments', 'augmenting', 'augmented', 'augmented', _, ['2A','6A'] ).
verb( 'augur', 'augurs', 'auguring', 'augured', 'augured', _, ['2A','6A'] ).
verb( 'authenticate', 'authenticates', 'authenticating', 'authenticated', 'authenticated', tran, ['6A'] ).
verb( 'authorize', 'authorizes', 'authorizing', 'authorized', 'authorized', tran, ['6A','17'] ).
verb( 'autograph', 'autographs', 'autographing', 'autographed', 'autographed', tran, ['6A'] ).
verb( 'automate', 'automates', 'automating', 'automated', 'automated', tran, [] ).
verb( 'avail', 'avails', 'availing', 'availed', 'availed', _, ['2A','3A','14'] ).
verb( 'avenge', 'avenges', 'avenging', 'avenged', 'avenged', tran, ['6A','14'] ).
verb( 'aver', 'avers', 'averring', 'averred', 'averred', tran, ['6A','9'] ).
verb( 'average', 'averages', 'averaging', 'averaged', 'averaged', _, ['2B','6A'] ).
verb( 'avert', 'averts', 'averting', 'averted', 'averted', tran, ['6A','14'] ).
verb( 'avoid', 'avoids', 'avoiding', 'avoided', 'avoided', tran, ['6A','6C'] ).
verb( 'avouch', 'avouches', 'avouching', 'avouched', 'avouched', _, ['3A','6A','9'] ).
verb( 'avow', 'avows', 'avowing', 'avowed', 'avowed', tran, ['6A','25'] ).
verb( 'await', 'awaits', 'awaiting', 'awaited', 'awaited', tran, ['6A'] ).
verb( 'awake', 'awakes', 'awaking', 'awoke', 'awoken', intran, ['2A','3A','4B'] ).
verb( 'awaken', 'awakens', 'awakening', 'awakened', 'awakened', tran, ['6A','14'] ).
verb( 'award', 'awards', 'awarding', 'awarded', 'awarded', tran, ['6A','12A','13A'] ).
verb( 'awe', 'awes', 'awing', 'awed', 'awed', tran, ['6A','14'] ).
verb( 'ax', 'axes', 'axing', 'axed', 'axed', tran, ['6A'] ).
verb( 'axe', 'axes', 'axing', 'axed', 'axed', tran, ['6A'] ).
verb( 'baa', 'baas', 'baaing', 'baaed', 'baaed', intran, [] ).
verb( 'babble', 'babbles', 'babbling', 'babbled', 'babbled', _, ['2A','2B','2C','6A','15B'] ).
verb( 'baby', 'babies', 'babying', 'babied', 'babied', tran, ['6A'] ).
verb( 'babysit', 'babysits', 'babysitting', 'babysat', 'babysat', intran, [] ).
verb( 'back', 'backs', 'backing', 'backed', 'backed', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'backbite', 'backbites', 'backbiting', 'backbit', 'backbitten', _, [] ).
verb( 'backdate', 'backdates', 'backdating', 'backdated', 'backdated', tran, [] ).
verb( 'backfire', 'backfires', 'backfiring', 'backfired', 'backfired', intran, [] ).
verb( 'backpedal', 'backpedals', 'backpedalling', 'backpedalled', 'backpedalled', intran, [] ).
verb( 'backslide', 'backslides', 'backsliding', 'backslid', 'backslid', intran, ['2A'] ).
verb( 'backspace', 'backspaces', 'backspacing', 'backspaced', 'backspaced', intran, [] ).
verb( 'badger', 'badgers', 'badgering', 'badgered', 'badgered', tran, ['6A','14','16A'] ).
verb( 'baffle', 'baffles', 'baffling', 'baffled', 'baffled', tran, ['6A'] ).
verb( 'bag', 'bags', 'bagging', 'bagged', 'bagged', _, ['2A','2C','6A','15B'] ).
verb( 'bail', 'bails', 'bailing', 'bailed', 'bailed', _, ['2A','2C','6A','15B'] ).
verb( 'bait', 'baits', 'baiting', 'baited', 'baited', _, ['2A','6A'] ).
verb( 'bake', 'bakes', 'baking', 'baked', 'baked', _, ['2A','2C','6A','22'] ).
verb( 'balance', 'balances', 'balancing', 'balanced', 'balanced', _, ['2A','6A','14','15A'] ).
verb( 'bale', 'bales', 'baling', 'baled', 'baled', tran, ['6A'] ).
verb( 'balk', 'balks', 'balking', 'balked', 'balked', _, ['2A','3A','6A','14'] ).
verb( 'ball', 'balls', 'balling', 'balled', 'balled', _, [] ).
verb( 'ballast', 'ballasts', 'ballasting', 'ballasted', 'ballasted', tran, ['6A'] ).
verb( 'balloon', 'balloons', 'ballooning', 'ballooned', 'ballooned', intran, ['2A','2C'] ).
verb( 'ballot', 'ballots', 'balloting', 'balloted', 'balloted', intran, ['2A','3A'] ).
verb( 'bamboozle', 'bamboozles', 'bamboozling', 'bamboozled', 'bamboozled', tran, ['6A','14'] ).
verb( 'ban', 'bans', 'banning', 'banned', 'banned', tran, ['6A','14'] ).
verb( 'band', 'bands', 'banding', 'banded', 'banded', _, ['2C','6A','14','15B'] ).
verb( 'bandage', 'bandages', 'bandaging', 'bandaged', 'bandaged', tran, ['6A','15B'] ).
verb( 'bandy', 'bandies', 'bandying', 'bandied', 'bandied', tran, ['6A','14','15B'] ).
verb( 'bang', 'bangs', 'banging', 'banged', 'banged', _, ['2A','2C','6A','15B'] ).
verb( 'banish', 'banishes', 'banishing', 'banished', 'banished', tran, ['6A','14'] ).
verb( 'bank', 'banks', 'banking', 'banked', 'banked', _, ['2A','2C','3A','6A','15B'] ).
verb( 'bankrupt', 'bankrupts', 'bankrupting', 'bankrupted', 'bankrupted', tran, ['6A'] ).
verb( 'banquet', 'banquets', 'banqueting', 'banqueted', 'banqueted', _, ['2A','6A'] ).
verb( 'bant', 'bants', 'banting', 'banted', 'banted', intran, ['2A'] ).
verb( 'banter', 'banters', 'bantering', 'bantered', 'bantered', _, ['2A','6A'] ).
verb( 'baptize', 'baptizes', 'baptizing', 'baptized', 'baptized', tran, ['6A','23'] ).
verb( 'bar', 'bars', 'barring', 'barred', 'barred', tran, ['6A','6C','12C','14','15B'] ).
verb( 'barbarize', 'barbarizes', 'barbarizing', 'barbarized', 'barbarized', tran, ['6A'] ).
verb( 'barbecue', 'barbecues', 'barbecuing', 'barbecued', 'barbecued', tran, [] ).
verb( 'bare', 'bares', 'baring', 'bared', 'bared', tran, ['6A'] ).
verb( 'bargain', 'bargains', 'bargaining', 'bargained', 'bargained', _, ['2A','3A','9','15B'] ).
verb( 'barge', 'barges', 'barging', 'barged', 'barged', intran, ['2C','3A'] ).
verb( 'bark', 'barks', 'barking', 'barked', 'barked', _, ['2A','2C','3A','6A','15B'] ).
verb( 'barnstorm', 'barnstorms', 'barnstorming', 'barnstormed', 'barnstormed', intran, [] ).
verb( 'barrack', 'barracks', 'barracking', 'barracked', 'barracked', _, ['2A','6A'] ).
verb( 'barrel', 'barrels', 'barrelling', 'barrelled', 'barrelled', tran, [] ).
verb( 'barricade', 'barricades', 'barricading', 'barricaded', 'barricaded', tran, ['6A','15B'] ).
verb( 'barter', 'barters', 'bartering', 'bartered', 'bartered', _, ['2A','14','15B'] ).
verb( 'base', 'bases', 'basing', 'based', 'based', tran, ['14'] ).
verb( 'bash', 'bashes', 'bashing', 'bashed', 'bashed', tran, ['6A','15A','15B'] ).
verb( 'bask', 'basks', 'basking', 'basked', 'basked', intran, ['2C'] ).
verb( 'bastardize', 'bastardizes', 'bastardizing', 'bastardized', 'bastardized', tran, [] ).
verb( 'baste', 'bastes', 'basting', 'basted', 'basted', tran, ['6A','15B'] ).
verb( 'bastinado', 'bastinados', 'bastinadoing', 'bastinadoed', 'bastinadoed', tran, ['6A'] ).
verb( 'bat', 'bats', 'batting', 'batted', 'batted', _, ['2A','2B','2C','6A'] ).
verb( 'bate', 'bates', 'bating', 'bated', 'bated', tran, [] ).
verb( 'bath', 'baths', 'bathing', 'bathed', 'bathed', _, ['2A','6A'] ).
verb( 'bathe', 'bathes', 'bathing', 'bathed', 'bathed', _, ['2A','6A'] ).
verb( 'batten', 'battens', 'battening', 'battened', 'battened', _, ['3A','6A','15B'] ).
verb( 'batter', 'batters', 'battering', 'battered', 'battered', _, ['2C','6A','15A','15B'] ).
verb( 'battle', 'battles', 'battling', 'battled', 'battled', intran, ['3A'] ).
verb( 'baulk', 'baulks', 'baulking', 'baulked', 'baulked', _, ['2A','3A','6A','14'] ).
verb( 'bawl', 'bawls', 'bawling', 'bawled', 'bawled', _, ['2C','3A','6A','15A'] ).
verb( 'bay', 'bays', 'baying', 'bayed', 'bayed', intran, ['2A'] ).
verb( 'bayonet', 'bayonets', 'bayoneting', 'bayoneted', 'bayoneted', tran, ['6A'] ).
verb( 'be', 'is', 'being', 'was', 'been', unknown, ['1','4F'] ).
verb( 'be', 'is', 'being', 'was', 'been', intran, ['1','4F'] ).
verb( 'beach', 'beaches', 'beaching', 'beached', 'beached', tran, ['6A'] ).
verb( 'beam', 'beams', 'beaming', 'beamed', 'beamed', _, ['2C','6A','15A'] ).
verb( 'bear', 'bears', 'bearing', 'bore', 'borne', _, ['2A','2C','3A','6A','6D','6E','7A','11','12C','14','15B','16B','17'] ).
verb( 'beard', 'beards', 'bearding', 'bearded', 'bearded', tran, ['6A'] ).
verb( 'beat', 'beats', 'beating', 'beat', 'beaten', _, ['2A','2C','6A','14','15A','15B','22'] ).
verb( 'beatify', 'beatifies', 'beatifying', 'beatified', 'beatified', tran, ['6A'] ).
verb( 'beautify', 'beautifies', 'beautifying', 'beautified', 'beautified', tran, ['6A'] ).
verb( 'beaver', 'beavers', 'beavering', 'beavered', 'beavered', intran, ['2A','2C'] ).
verb( 'beckon', 'beckons', 'beckoning', 'beckoned', 'beckoned', _, ['2A','6A','15B','16A'] ).
verb( 'become', 'becomes', 'becoming', 'became', 'become', _, ['2D','3A','6A'] ).
verb( 'bed', 'beds', 'bedding', 'bedded', 'bedded', tran, ['6A','15A','15B'] ).
verb( 'bedevil', 'bedevils', 'bedevilling', 'bedevilled', 'bedevilled', tran, [] ).
verb( 'beef', 'beefs', 'beefing', 'beefed', 'beefed', intran, [] ).
verb( 'beeswax', 'beeswaxes', 'beeswaxing', 'beeswaxed', 'beeswaxed', tran, [] ).
verb( 'beetle', 'beetles', 'beetling', 'beetled', 'beetled', intran, ['2A'] ).
verb( 'befall', 'befalls', 'befalling', 'befell', 'befallen', _, ['6A'] ).
verb( 'befit', 'befits', 'befitting', 'befitted', 'befitted', tran, ['6A'] ).
verb( 'befoul', 'befouls', 'befouling', 'befouled', 'befouled', tran, [] ).
verb( 'befriend', 'befriends', 'befriending', 'befriended', 'befriended', tran, ['6A'] ).
verb( 'beg', 'begs', 'begging', 'begged', 'begged', _, ['2A','2C','3A','6A','7A','9','14','17'] ).
verb( 'beget', 'begets', 'begetting', 'begat', 'begotten', tran, ['6A'] ).
verb( 'beggar', 'beggars', 'beggaring', 'beggared', 'beggared', tran, ['6A'] ).
verb( 'begin', 'begins', 'beginning', 'began', 'begun', _, ['2A','3A','6A','6D','7A'] ).
verb( 'begrudge', 'begrudges', 'begrudging', 'begrudged', 'begrudged', tran, ['6C','12A','13A'] ).
verb( 'beguile', 'beguiles', 'beguiling', 'beguiled', 'beguiled', tran, ['6A','14'] ).
verb( 'behave', 'behaves', 'behaving', 'behaved', 'behaved', intran, ['2A','2C','6B'] ).
verb( 'behead', 'beheads', 'beheading', 'beheaded', 'beheaded', tran, ['6A'] ).
verb( 'behold', 'beholds', 'beholding', 'beheld', 'beheld', tran, ['6A'] ).
verb( 'behove', 'behoves', 'behoving', 'behoved', 'behoved', tran, [] ).
verb( 'belabour', 'belabours', 'belabouring', 'belaboured', 'belaboured', tran, ['6A'] ).
verb( 'belay', 'belays', 'belaying', 'belayed', 'belayed', tran, ['6A'] ).
verb( 'belch', 'belches', 'belching', 'belched', 'belched', _, ['2A','6A','15B'] ).
verb( 'beleaguer', 'beleaguers', 'beleaguering', 'beleaguered', 'beleaguered', tran, ['6A'] ).
verb( 'belie', 'belies', 'belying', 'belied', 'belied', tran, ['6A'] ).
verb( 'believe', 'believes', 'believing', 'believed', 'believed', _, ['3A','6A','9','10','25'] ).
verb( 'belittle', 'belittles', 'belittling', 'belittled', 'belittled', tran, ['6A'] ).
verb( 'bell', 'bells', 'belling', 'belled', 'belled', tran, [] ).
verb( 'bellow', 'bellows', 'bellowing', 'bellowed', 'bellowed', _, ['2A','6A','15B'] ).
verb( 'belly', 'bellies', 'bellying', 'bellied', 'bellied', _, ['2A','2C','6A','15B'] ).
verb( 'bellyache', 'bellyaches', 'bellyaching', 'bellyached', 'bellyached', intran, [] ).
verb( 'bellyland', 'bellylands', 'bellylanding', 'bellylanded', 'bellylanded', intran, [] ).
verb( 'bellylaugh', 'bellylaughs', 'bellylaughing', 'bellylaughed', 'bellylaughed', intran, [] ).
verb( 'belong', 'belongs', 'belonging', 'belonged', 'belonged', intran, ['2C','3A'] ).
verb( 'belt', 'belts', 'belting', 'belted', 'belted', tran, ['2C','3A','6A','15B'] ).
verb( 'bemoan', 'bemoans', 'bemoaning', 'bemoaned', 'bemoaned', tran, ['6A'] ).
verb( 'bend', 'bends', 'bending', 'bended', 'bended', _, ['2A','2C','6A','15A','15B'] ).
verb( 'benefit', 'benefits', 'benefiting', 'benefited', 'benefited', _, ['3A','6A'] ).
verb( 'bequeath', 'bequeaths', 'bequeathing', 'bequeathed', 'bequeathed', tran, ['6A','12A','13A'] ).
verb( 'berate', 'berates', 'berating', 'berated', 'berated', tran, ['6A'] ).
verb( 'bereave', 'bereaves', 'bereaving', 'bereaved', 'bereaved', tran, ['14'] ).
verb( 'berth', 'berths', 'berthing', 'berthed', 'berthed', _, ['2C','6A','15A'] ).
verb( 'beseech', 'beseeches', 'beseeching', 'beseeched', 'beseeched', tran, ['6A','11','13B','17'] ).
verb( 'beseem', 'beseems', 'beseeming', 'beseemed', 'beseemed', tran, [] ).
verb( 'beset', 'besets', 'besetting', 'beset', 'beset', tran, ['6A'] ).
verb( 'beshrew', 'beshrews', 'beshrewing', 'beshrewed', 'beshrewed', tran, [] ).
verb( 'besiege', 'besieges', 'besieging', 'besieged', 'besieged', tran, ['6A','14'] ).
verb( 'besmear', 'besmears', 'besmearing', 'besmeared', 'besmeared', tran, [] ).
verb( 'besmirch', 'besmirches', 'besmirching', 'besmirched', 'besmirched', tran, [] ).
verb( 'bespeak', 'bespeaks', 'bespeaking', 'bespoke', 'bespoken', tran, ['6A','25'] ).
verb( 'best', 'bests', 'besting', 'bested', 'bested', tran, ['6A'] ).
verb( 'bestir', 'bestirs', 'bestirring', 'bestirred', 'bestirred', tran, ['6A','17'] ).
verb( 'bestow', 'bestows', 'bestowing', 'bestowed', 'bestowed', tran, ['6A','14'] ).
verb( 'bestrew', 'bestrews', 'bestrewing', 'bestrewed', 'bestrewed', tran, ['6A','14'] ).
verb( 'bestride', 'bestrides', 'bestriding', 'bestrode', 'bestridden', tran, ['6A'] ).
verb( 'bet', 'bets', 'betting', 'betted', 'betted', _, ['2A','3A','9','11','12C'] ).
verb( 'betake', 'betakes', 'betaking', 'betook', 'betaken', tran, ['14'] ).
verb( 'bethink', 'bethinks', 'bethinking', 'bethought', 'bethought', tran, ['11','14','17','20','21'] ).
verb( 'betide', 'betides', 'betiding', 'betided', 'betided', tran, [] ).
verb( 'betoken', 'betokens', 'betokening', 'betokened', 'betokened', tran, ['6A'] ).
verb( 'betray', 'betrays', 'betraying', 'betrayed', 'betrayed', tran, ['6A','14','25'] ).
verb( 'betroth', 'betroths', 'betrothing', 'betrothed', 'betrothed', tran, ['6A','14'] ).
verb( 'better', 'betters', 'bettering', 'bettered', 'bettered', tran, ['6A'] ).
verb( 'bevel', 'bevels', 'bevelling', 'bevelled', 'bevelled', tran, [] ).
verb( 'bewail', 'bewails', 'bewailing', 'bewailed', 'bewailed', tran, ['6A'] ).
verb( 'beware', '-', '-', '-', '-', _, ['2A','3A','10'] ).
verb( 'bewilder', 'bewilders', 'bewildering', 'bewildered', 'bewildered', tran, ['6A'] ).
verb( 'bewitch', 'bewitches', 'bewitching', 'bewitched', 'bewitched', tran, ['6A'] ).
verb( 'bias', 'biases', 'biasing', 'biased', 'biased', tran, ['6A','14'] ).
verb( 'bib', 'bibs', 'bibbing', 'bibbed', 'bibbed', intran, [] ).
verb( 'bicker', 'bickers', 'bickering', 'bickered', 'bickered', intran, ['2A','2C','3A'] ).
verb( 'bicycle', 'bicycles', 'bicycling', 'bicycled', 'bicycled', intran, ['2A','2C'] ).
verb( 'bid', 'bids', 'bidding', 'bid', 'bid', _, ['2A','3A','6A','12A','13A','14','15B','17','18B'] ).
verb( 'bide', 'bides', 'biding', 'bided', 'bided', tran, [] ).
verb( 'biff', 'biffs', 'biffing', 'biffed', 'biffed', tran, [] ).
verb( 'bifurcate', 'bifurcates', 'bifurcating', 'bifurcated', 'bifurcated', _, ['2A','6A'] ).
verb( 'bike', 'bikes', 'biking', 'biked', 'biked', intran, [] ).
verb( 'bilk', 'bilks', 'bilking', 'bilked', 'bilked', tran, ['6A','14'] ).
verb( 'bill', 'bills', 'billing', 'billed', 'billed', _, ['6A','14'] ).
verb( 'billet', 'billets', 'billeting', 'billeted', 'billeted', tran, ['6A','14'] ).
verb( 'billow', 'billows', 'billowing', 'billowed', 'billowed', intran, ['2C'] ).
verb( 'bind', 'binds', 'binding', 'bound', 'bound', _, ['2A','6A','14','15A','15B','16B','17'] ).
verb( 'birch', 'birches', 'birching', 'birched', 'birched', tran, ['6A'] ).
verb( 'bisect', 'bisects', 'bisecting', 'bisected', 'bisected', tran, ['6A'] ).
verb( 'bitch', 'bitches', 'bitching', 'bitched', 'bitched', intran, ['2A'] ).
verb( 'bite', 'bites', 'biting', 'bit', 'bitten', _, ['2A','3A','6A','15B'] ).
verb( 'bivouac', 'bivouacs', 'bivouacking', 'bivouacked', 'bivouacked', intran, ['2A'] ).
verb( 'blab', 'blabs', 'blabbing', 'blabbed', 'blabbed', _, ['2A','6A','15B'] ).
verb( 'blabber', 'blabbers', 'blabbering', 'blabbered', 'blabbered', _, [] ).
verb( 'black', 'blacks', 'blacking', 'blacked', 'blacked', tran, ['6A'] ).
verb( 'black-lead', 'black-leads', 'black-leading', 'black-leaded', 'black-leaded', tran, [] ).
verb( 'blackball', 'blackballs', 'blackballing', 'blackballed', 'blackballed', tran, [] ).
verb( 'blacken', 'blackens', 'blackening', 'blackened', 'blackened', _, ['2A','6A'] ).
verb( 'blackguard', 'blackguards', 'blackguarding', 'blackguarded', 'blackguarded', tran, [] ).
verb( 'blackleg', 'blacklegs', 'blacklegging', 'blacklegged', 'blacklegged', _, [] ).
verb( 'blacklist', 'blacklists', 'blacklisting', 'blacklisted', 'blacklisted', tran, [] ).
verb( 'blackmail', 'blackmails', 'blackmailing', 'blackmailed', 'blackmailed', tran, [] ).
verb( 'blame', 'blames', 'blaming', 'blamed', 'blamed', tran, ['6A','14'] ).
verb( 'blanch', 'blanches', 'blanching', 'blanched', 'blanched', _, ['2A','6A'] ).
verb( 'blanket', 'blankets', 'blanketing', 'blanketed', 'blanketed', tran, ['6A','14'] ).
verb( 'blare', 'blares', 'blaring', 'blared', 'blared', _, ['2A','2C','15B'] ).
verb( 'blaspheme', 'blasphemes', 'blaspheming', 'blasphemed', 'blasphemed', _, ['2A','6A'] ).
verb( 'blast', 'blasts', 'blasting', 'blasted', 'blasted', tran, ['2A','2C','6A','15B'] ).
verb( 'blather', 'blathers', 'blathering', 'blathered', 'blathered', intran, [] ).
verb( 'blaze', 'blazes', 'blazing', 'blazed', 'blazed', _, ['2A','2C','6A','15B'] ).
verb( 'blazon', 'blazons', 'blazoning', 'blazoned', 'blazoned', tran, [] ).
verb( 'bleach', 'bleaches', 'bleaching', 'bleached', 'bleached', _, ['2A','6A'] ).
verb( 'bleat', 'bleats', 'bleating', 'bleated', 'bleated', _, ['2A','6A','15B'] ).
verb( 'bleed', 'bleeds', 'bleeding', 'bled', 'bled', _, ['2A','2C','3A','6A','14'] ).
verb( 'bleep', 'bleeps', 'bleeping', 'bleeped', 'bleeped', intran, [] ).
verb( 'blemish', 'blemishes', 'blemishing', 'blemished', 'blemished', tran, ['6A'] ).
verb( 'blench', 'blenches', 'blenching', 'blenched', 'blenched', intran, ['2A'] ).
verb( 'blend', 'blends', 'blending', 'blended', 'blended', _, ['2A','3A','6A'] ).
verb( 'bless', 'blesses', 'blessing', 'blessed', 'blessed', tran, ['6A'] ).
verb( 'blether', 'blethers', 'blethering', 'blethered', 'blethered', intran, ['2A','2C'] ).
verb( 'blight', 'blights', 'blighting', 'blighted', 'blighted', tran, ['6A'] ).
verb( 'blind', 'blinds', 'blinding', 'blinded', 'blinded', tran, ['6A','14'] ).
verb( 'blindfold', 'blindfolds', 'blindfolding', 'blindfolded', 'blindfolded', tran, ['6A'] ).
verb( 'blink', 'blinks', 'blinking', 'blinked', 'blinked', _, ['2A','2C','6A','15B'] ).
verb( 'blister', 'blisters', 'blistering', 'blistered', 'blistered', _, ['2A','6A'] ).
verb( 'blitz', 'blitzes', 'blitzing', 'blitzed', 'blitzed', tran, ['6A'] ).
verb( 'block', 'blocks', 'blocking', 'blocked', 'blocked', tran, ['6A','15B'] ).
verb( 'blockade', 'blockades', 'blockading', 'blockaded', 'blockaded', tran, ['6A'] ).
verb( 'blood', 'bloods', 'blooding', 'blooded', 'blooded', tran, ['6A'] ).
verb( 'bloom', 'blooms', 'blooming', 'bloomed', 'bloomed', intran, ['2A','2C'] ).
verb( 'blossom', 'blossoms', 'blossoming', 'blossomed', 'blossomed', intran, ['2A','2C'] ).
verb( 'blot', 'blots', 'blotting', 'blotted', 'blotted', tran, ['6A','15B'] ).
verb( 'blow', 'blows', 'blowing', 'blew', 'blown', _, ['2A','2B','2C','2E','3A','6A','12A','15A','15B'] ).
verb( 'blow-dry', 'blow-dries', 'blow-drying', 'blow-dried', 'blow-dried', tran, ['6A'] ).
verb( 'blubber', 'blubbers', 'blubbering', 'blubbered', 'blubbered', _, ['2A','15B'] ).
verb( 'bludgeon', 'bludgeons', 'bludgeoning', 'bludgeoned', 'bludgeoned', tran, ['6A','14'] ).
verb( 'blue', 'blues', 'bluing', 'blued', 'blued', tran, [] ).
verb( 'blue-pencil', 'blue-pencils', 'blue-pencilling', 'blue-pencilled', 'blue-pencilled', tran, [] ).
verb( 'bluff', 'bluffs', 'bluffing', 'bluffed', 'bluffed', _, ['2A','6A','14','15B'] ).
verb( 'blunder', 'blunders', 'blundering', 'blundered', 'blundered', _, ['2A','2C','3A'] ).
verb( 'blunt', 'blunts', 'blunting', 'blunted', 'blunted', tran, ['6A'] ).
verb( 'blur', 'blurs', 'blurring', 'blurred', 'blurred', _, ['2A','6A'] ).
verb( 'blurt', 'blurts', 'blurting', 'blurted', 'blurted', tran, ['15B'] ).
verb( 'blush', 'blushes', 'blushing', 'blushed', 'blushed', intran, ['2A','2C','3A','4B'] ).
verb( 'bluster', 'blusters', 'blustering', 'blustered', 'blustered', _, ['2A','2C','15B'] ).
verb( 'board', 'boards', 'boarding', 'boarded', 'boarded', _, ['3A','6A','15B'] ).
verb( 'boast', 'boasts', 'boasting', 'boasted', 'boasted', _, ['2A','3A','3B','6A'] ).
verb( 'boat', 'boats', 'boating', 'boated', 'boated', intran, ['2A','2C'] ).
verb( 'bob', 'bobs', 'bobbing', 'bobbed', 'bobbed', _, ['2C','6A'] ).
verb( 'bode', 'bodes', 'boding', 'boded', 'boded', _, ['12B','13B'] ).
verb( 'bog', 'bogs', 'bogging', 'bogged', 'bogged', _, ['2E','15B'] ).
verb( 'boggle', 'boggles', 'boggling', 'boggled', 'boggled', intran, ['2A','3A'] ).
verb( 'boil', 'boils', 'boiling', 'boiled', 'boiled', _, ['2A','2B','2C','2D','6A','15B','22'] ).
verb( 'bolster', 'bolsters', 'bolstering', 'bolstered', 'bolstered', tran, ['6A','15B'] ).
verb( 'bolt', 'bolts', 'bolting', 'bolted', 'bolted', _, ['2A','2C','6A','15B'] ).
verb( 'bomb', 'bombs', 'bombing', 'bombed', 'bombed', _, ['2C','6A','15B'] ).
verb( 'bombard', 'bombards', 'bombarding', 'bombarded', 'bombarded', tran, ['6A','14'] ).
verb( 'bond', 'bonds', 'bonding', 'bonded', 'bonded', tran, ['6A'] ).
verb( 'bone', 'bones', 'boning', 'boned', 'boned', tran, ['6A'] ).
verb( 'boo', 'boos', 'booing', 'booed', 'booed', _, ['2A','6A','15B'] ).
verb( 'boob', 'boobs', 'boobing', 'boobed', 'boobed', intran, [] ).
verb( 'book', 'books', 'booking', 'booked', 'booked', tran, ['6A'] ).
verb( 'boom', 'booms', 'booming', 'boomed', 'boomed', _, ['2A','2C','15B'] ).
verb( 'boost', 'boosts', 'boosting', 'boosted', 'boosted', tran, ['6A'] ).
verb( 'boot', 'boots', 'booting', 'booted', 'booted', tran, ['6A','15A','15B'] ).
verb( 'bootleg', 'bootlegs', 'bootlegging', 'bootlegged', 'bootlegged', tran, [] ).
verb( 'booze', 'boozes', 'boozing', 'boozed', 'boozed', intran, ['2A','2C'] ).
verb( 'border', 'borders', 'bordering', 'bordered', 'bordered', _, ['3A','6A'] ).
verb( 'bore', 'bores', 'boring', 'bored', 'bored', _, ['2A','2C','3A','6A','6D','6E','7A','11','12C','14','15A','15B','16B','17'] ).
verb( 'borrow', 'borrows', 'borrowing', 'borrowed', 'borrowed', tran, ['6A','14'] ).
verb( 'boss', 'bosses', 'bossing', 'bossed', 'bossed', tran, ['6A','15B'] ).
verb( 'botanize', 'botanizes', 'botanizing', 'botanized', 'botanized', intran, [] ).
verb( 'botch', 'botches', 'botching', 'botched', 'botched', tran, ['6A','15B'] ).
verb( 'bother', 'bothers', 'bothering', 'bothered', 'bothered', _, ['2A','3A','4C','6A','14','16A'] ).
verb( 'bottle', 'bottles', 'bottling', 'bottled', 'bottled', tran, ['6A','15B'] ).
verb( 'bottom', 'bottoms', 'bottoming', 'bottomed', 'bottomed', intran, [] ).
verb( 'bounce', 'bounces', 'bouncing', 'bounced', 'bounced', _, ['2A','2C','6A'] ).
verb( 'bound', 'bounds', 'bounding', 'bounded', 'bounded', _, ['2A','2C','4A','6A','14','15A','15B','16B','17'] ).
verb( 'bow', 'bows', 'bowing', 'bowed', 'bowed', tran, [] ).
verb( 'bow', 'bows', 'bowing', 'bowed', 'bowed', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'bowdlerize', 'bowdlerizes', 'bowdlerizing', 'bowdlerized', 'bowdlerized', tran, [] ).
verb( 'bowl', 'bowls', 'bowling', 'bowled', 'bowled', _, ['2A','2C','6A','15B'] ).
verb( 'box', 'boxes', 'boxing', 'boxed', 'boxed', _, ['2A','3A','6A','15B'] ).
verb( 'boycott', 'boycotts', 'boycotting', 'boycotted', 'boycotted', tran, ['6A'] ).
verb( 'brace', 'braces', 'bracing', 'braced', 'braced', _, ['6A','15B','16A'] ).
verb( 'bracket', 'brackets', 'bracketing', 'bracketed', 'bracketed', tran, ['6A','15B'] ).
verb( 'brag', 'brags', 'bragging', 'bragged', 'bragged', intran, ['2A','3A'] ).
verb( 'braid', 'braids', 'braiding', 'braided', 'braided', tran, ['6A'] ).
verb( 'brain', 'brains', 'braining', 'brained', 'brained', tran, ['6A'] ).
verb( 'brainwash', 'brainwashes', 'brainwashing', 'brainwashed', 'brainwashed', tran, ['6A'] ).
verb( 'braise', 'braises', 'braising', 'braised', 'braised', tran, ['6A'] ).
verb( 'brake', 'brakes', 'braking', 'braked', 'braked', _, ['2A','2C','2D','3A','6A','15A','15B','22'] ).
verb( 'branch', 'branches', 'branching', 'branched', 'branched', intran, ['2A','2C'] ).
verb( 'brand', 'brands', 'branding', 'branded', 'branded', tran, ['6A','16B'] ).
verb( 'brandish', 'brandishes', 'brandishing', 'brandished', 'brandished', tran, ['6A'] ).
verb( 'brave', 'braves', 'braving', 'braved', 'braved', tran, ['6A','15B'] ).
verb( 'brawl', 'brawls', 'brawling', 'brawled', 'brawled', intran, ['2A'] ).
verb( 'bray', 'brays', 'braying', 'brayed', 'brayed', tran, ['2A'] ).
verb( 'braze', 'brazes', 'brazing', 'brazed', 'brazed', tran, ['6A'] ).
verb( 'brazen', 'brazens', 'brazening', 'brazened', 'brazened', tran, [] ).
verb( 'breach', 'breaches', 'breaching', 'breached', 'breached', tran, ['6A'] ).
verb( 'break', 'breaks', 'breaking', 'broke', 'broken', _, ['2A','2C','2D','3A','6A','15A','15B','22'] ).
verb( 'breakfast', 'breakfasts', 'breakfasting', 'breakfasted', 'breakfasted', intran, [] ).
verb( 'breast', 'breasts', 'breasting', 'breasted', 'breasted', tran, ['6A'] ).
verb( 'breastfeed', 'breastfeeds', 'breastfeeding', 'breastfed', 'breastfed', _, [] ).
verb( 'breathe', 'breathes', 'breathing', 'breathed', 'breathed', _, ['2A','2C','6A','15B'] ).
verb( 'breed', 'breeds', 'breeding', 'bred', 'bred', _, ['2A','6A'] ).
verb( 'breeze', 'breezes', 'breezing', 'breezed', 'breezed', intran, ['2C'] ).
verb( 'brew', 'brews', 'brewing', 'brewed', 'brewed', _, ['2A','2C','6A'] ).
verb( 'bribe', 'bribes', 'bribing', 'bribed', 'bribed', tran, ['6A','15A','17'] ).
verb( 'brick', 'bricks', 'bricking', 'bricked', 'bricked', tran, ['15B'] ).
verb( 'bridge', 'bridges', 'bridging', 'bridged', 'bridged', tran, ['6A','15B'] ).
verb( 'bridle', 'bridles', 'bridling', 'bridled', 'bridled', _, ['2A','2C','6A'] ).
verb( 'brief', 'briefs', 'briefing', 'briefed', 'briefed', tran, ['6A'] ).
verb( 'brighten', 'brightens', 'brightening', 'brightened', 'brightened', _, ['2A','2C','6A','15B'] ).
verb( 'brim', 'brims', 'brimming', 'brimmed', 'brimmed', intran, ['2A','2C'] ).
verb( 'bring', 'brings', 'bringing', 'brought', 'brought', tran, ['6A','12A','12C','13A','14','15B','17','19B'] ).
verb( 'bristle', 'bristles', 'bristling', 'bristled', 'bristled', intran, ['2A','2C'] ).
verb( 'broach', 'broaches', 'broaching', 'broached', 'broached', _, ['2C','6A','15B'] ).
verb( 'broadcast', 'broadcasts', 'broadcasting', 'broadcasted', 'broadcasted', _, ['2A','6A'] ).
verb( 'broaden', 'broadens', 'broadening', 'broadened', 'broadened', _, ['2A','2C','6A','15B'] ).
verb( 'brocade', 'brocades', 'brocading', 'brocaded', 'brocaded', tran, ['6A'] ).
verb( 'broil', 'broils', 'broiling', 'broiled', 'broiled', _, ['2A','6A'] ).
verb( 'bronze', 'bronzes', 'bronzing', 'bronzed', 'bronzed', _, ['2A','6A'] ).
verb( 'brood', 'broods', 'brooding', 'brooded', 'brooded', intran, ['2A','2C','3A'] ).
verb( 'brook', 'brooks', 'brooking', 'brooked', 'brooked', tran, ['6A','6B'] ).
verb( 'browbeat', 'browbeats', 'browbeating', 'browbeat', 'browbeaten', tran, ['6A','14'] ).
verb( 'brown', 'browns', 'browning', 'browned', 'browned', _, ['2A','6A'] ).
verb( 'browse', 'browses', 'browsing', 'browsed', 'browsed', intran, ['2A','2C'] ).
verb( 'bruise', 'bruises', 'bruising', 'bruised', 'bruised', _, ['2A','6A'] ).
verb( 'bruit', 'bruits', 'bruiting', 'bruited', 'bruited', tran, ['15B'] ).
verb( 'brush', 'brushes', 'brushing', 'brushed', 'brushed', _, ['2A','2C','6A','15B','22'] ).
verb( 'brutalize', 'brutalizes', 'brutalizing', 'brutalized', 'brutalized', tran, ['6A'] ).
verb( 'bubble', 'bubbles', 'bubbling', 'bubbled', 'bubbled', intran, ['2A','2C'] ).
verb( 'buck', 'bucks', 'bucking', 'bucked', 'bucked', _, ['2A','2C','6A','15B'] ).
verb( 'bucket', 'buckets', 'bucketing', 'bucketed', 'bucketed', intran, [] ).
verb( 'buckle', 'buckles', 'buckling', 'buckled', 'buckled', _, ['2A','2C','6A','15B'] ).
verb( 'bud', 'buds', 'budding', 'budded', 'budded', intran, [] ).
verb( 'budge', 'budges', 'budging', 'budged', 'budged', _, ['2A','2C','6A'] ).
verb( 'budget', 'budgets', 'budgeting', 'budgeted', 'budgeted', intran, ['3A'] ).
verb( 'buff', 'buffs', 'buffing', 'buffed', 'buffed', tran, [] ).
verb( 'buffet', 'buffets', 'buffeting', 'buffeted', 'buffeted', _, ['3A','6A'] ).
verb( 'bug', 'bugs', 'bugging', 'bugged', 'bugged', tran, ['6A'] ).
verb( 'bugger', 'buggers', 'buggering', 'buggered', 'buggered', _, ['6A'] ).
verb( 'build', 'builds', 'building', 'built', 'built', _, ['2C','3A','6A','12B','13B','14','15B'] ).
verb( 'bulge', 'bulges', 'bulging', 'bulged', 'bulged', _, ['2A','2C','6A'] ).
verb( 'bulk', 'bulks', 'bulking', 'bulked', 'bulked', intran, [] ).
verb( 'bulldoze', 'bulldozes', 'bulldozing', 'bulldozed', 'bulldozed', tran, ['6A','14'] ).
verb( 'bullshit', 'bullshits', 'bullshitting', 'bullshitted', 'bullshitted', intran, [] ).
verb( 'bully', 'bullies', 'bullying', 'bullied', 'bullied', _, ['2C','6A','14'] ).
verb( 'bum', 'bums', 'bumming', 'bummed', 'bummed', _, ['2C','6A','14'] ).
verb( 'bump', 'bumps', 'bumping', 'bumped', 'bumped', _, ['2C','3A','6A','14','15B'] ).
verb( 'bunch', 'bunches', 'bunching', 'bunched', 'bunched', _, ['2A','2C','15B'] ).
verb( 'bundle', 'bundles', 'bundling', 'bundled', 'bundled', _, ['2C','15A','15B'] ).
verb( 'bung', 'bungs', 'bunging', 'bunged', 'bunged', tran, ['6A','15B'] ).
verb( 'bungle', 'bungles', 'bungling', 'bungled', 'bungled', _, ['2A','6A'] ).
verb( 'bunk', 'bunks', 'bunking', 'bunked', 'bunked', intran, [] ).
verb( 'bunker', 'bunkers', 'bunkering', 'bunkered', 'bunkered', _, ['2A','6A'] ).
verb( 'buoy', 'buoys', 'buoying', 'buoyed', 'buoyed', intran, ['6A','15B'] ).
verb( 'burble', 'burbles', 'burbling', 'burbled', 'burbled', intran, [] ).
verb( 'burden', 'burdens', 'burdening', 'burdened', 'burdened', tran, ['6A','14'] ).
verb( 'burgeon', 'burgeons', 'burgeoning', 'burgeoned', 'burgeoned', intran, [] ).
verb( 'burgle', 'burgles', 'burgling', 'burgled', 'burgled', _, ['2A','6A'] ).
verb( 'burke', 'burkes', 'burking', 'burked', 'burked', tran, ['6A'] ).
verb( 'burlesque', 'burlesques', 'burlesquing', 'burlesqued', 'burlesqued', tran, ['6A'] ).
verb( 'burn', 'burns', 'burning', 'burned', 'burned', _, ['2A','2B','2C','4A','6A','14','15B'] ).
verb( 'burnish', 'burnishes', 'burnishing', 'burnished', 'burnished', _, ['2A','6A'] ).
verb( 'burp', 'burps', 'burping', 'burped', 'burped', _, ['2A','6A'] ).
verb( 'burrow', 'burrows', 'burrowing', 'burrowed', 'burrowed', _, ['2A','2C','6A'] ).
verb( 'burst', 'bursts', 'bursting', 'burst', 'burst', _, ['2A','2C','3A','6A','22'] ).
verb( 'burthen', 'burthens', 'burthening', 'burthened', 'burthened', tran, [] ).
verb( 'bury', 'buries', 'burying', 'buried', 'buried', tran, ['6A','15A','22'] ).
verb( 'bus', 'buses', 'busing', 'bused', 'bused', _, ['2A','6A'] ).
verb( 'bust', 'busts', 'busting', 'busted', 'busted', _, [] ).
verb( 'bustle', 'bustles', 'bustling', 'bustled', 'bustled', _, ['2A','2C','15B'] ).
verb( 'busy', 'busies', 'busying', 'busied', 'busied', tran, ['6A','14'] ).
verb( 'butcher', 'butchers', 'butchering', 'butchered', 'butchered', tran, ['6A'] ).
verb( 'butt', 'butts', 'butting', 'butted', 'butted', _, ['2C','3A','6A','15A'] ).
verb( 'butter', 'butters', 'buttering', 'buttered', 'buttered', tran, ['6A','15B'] ).
verb( 'button', 'buttons', 'buttoning', 'buttoned', 'buttoned', _, ['2A','6A','15B'] ).
verb( 'buttonhole', 'buttonholes', 'buttonholing', 'buttonholed', 'buttonholed', tran, ['6A'] ).
verb( 'buttress', 'buttresses', 'buttressing', 'buttressed', 'buttressed', tran, ['6A','15B'] ).
verb( 'buy', 'buys', 'buying', 'bought', 'bought', _, ['2A','2C','6A','12B','13B','15B'] ).
verb( 'buzz', 'buzzes', 'buzzing', 'buzzed', 'buzzed', _, ['2A','2C','6A'] ).
verb( 'bypass', 'bypasses', 'bypassing', 'bypassed', 'bypassed', tran, ['6A'] ).
verb( 'cable', 'cables', 'cabling', 'cabled', 'cabled', _, ['2A','6A'] ).
verb( 'cache', 'caches', 'caching', 'cached', 'cached', tran, [] ).
verb( 'cackle', 'cackles', 'cackling', 'cackled', 'cackled', intran, [] ).
verb( 'cadge', 'cadges', 'cadging', 'cadged', 'cadged', _, ['2A','6A','14'] ).
verb( 'cage', 'cages', 'caging', 'caged', 'caged', tran, ['6A'] ).
verb( 'cajole', 'cajoles', 'cajoling', 'cajoled', 'cajoled', tran, ['6A','14'] ).
verb( 'cake', 'cakes', 'caking', 'caked', 'caked', _, ['2A','6A'] ).
verb( 'calcify', 'calcifies', 'calcifying', 'calcified', 'calcified', _, ['2A','6A'] ).
verb( 'calcine', 'calcines', 'calcining', 'calcined', 'calcined', _, ['2A','6A'] ).
verb( 'calculate', 'calculates', 'calculating', 'calculated', 'calculated', _, ['2A','3A','6A','8','9','10'] ).
verb( 'calender', 'calenders', 'calendering', 'calendered', 'calendered', tran, ['6A'] ).
verb( 'calibrate', 'calibrates', 'calibrating', 'calibrated', 'calibrated', tran, ['6A'] ).
verb( 'calk', 'calks', 'calking', 'calked', 'calked', tran, [] ).
verb( 'call', 'calls', 'calling', 'called', 'called', _, ['2A','2B','2C','3A','4A','6A','12B','13B','15B','22','23'] ).
verb( 'calm', 'calms', 'calming', 'calmed', 'calmed', _, ['2C','6A','15B'] ).
verb( 'calumniate', 'calumniates', 'calumniating', 'calumniated', 'calumniated', tran, ['6A'] ).
verb( 'calve', 'calves', 'calving', 'calved', 'calved', intran, [] ).
verb( 'camber', 'cambers', 'cambering', 'cambered', 'cambered', _, [] ).
verb( 'camouflage', 'camouflages', 'camouflaging', 'camouflaged', 'camouflaged', tran, ['6A'] ).
verb( 'camp', 'camps', 'camping', 'camped', 'camped', _, ['2A','2C'] ).
verb( 'campaign', 'campaigns', 'campaigning', 'campaigned', 'campaigned', intran, ['2A','3A'] ).
verb( 'can', 'can', '-', 'could', '-', unknown, ['5','6A'] ).
verb( 'can', 'cans', 'canning', 'canned', 'canned', tran, ['5','6A'] ).
verb( 'canalize', 'canalizes', 'canalizing', 'canalized', 'canalized', tran, ['6A','14'] ).
verb( 'cancel', 'cancels', 'cancelling', 'cancelled', 'cancelled', _, ['2C','6A','15B'] ).
verb( 'candy', 'candies', 'candying', 'candied', 'candied', _, ['2A','6A'] ).
verb( 'cane', 'canes', 'caning', 'caned', 'caned', tran, ['6A'] ).
verb( 'canker', 'cankers', 'cankering', 'cankered', 'cankered', tran, [] ).
verb( 'cannibalize', 'cannibalizes', 'cannibalizing', 'cannibalized', 'cannibalized', tran, [] ).
verb( 'canoe', 'canoes', 'canoeing', 'canoed', 'canoed', tran, ['2A','2C'] ).
verb( 'canonize', 'canonizes', 'canonizing', 'canonized', 'canonized', tran, ['6A'] ).
verb( 'cant', 'cants', 'canting', 'canted', 'canted', _, ['2A','2C','6A','15B'] ).
verb( 'canter', 'canters', 'cantering', 'cantered', 'cantered', _, [] ).
verb( 'canvass', 'canvasses', 'canvassing', 'canvassed', 'canvassed', _, ['2A','3A','6A'] ).
verb( 'cap', 'caps', 'capping', 'capped', 'capped', tran, ['6A'] ).
verb( 'caparison', 'caparisons', 'caparisoning', 'caparisoned', 'caparisoned', tran, [] ).
verb( 'caper', 'capers', 'capering', 'capered', 'capered', intran, [] ).
verb( 'capitalize', 'capitalizes', 'capitalizing', 'capitalized', 'capitalized', _, ['3A','6A'] ).
verb( 'capitulate', 'capitulates', 'capitulating', 'capitulated', 'capitulated', tran, ['2A'] ).
verb( 'capsize', 'capsizes', 'capsizing', 'capsized', 'capsized', _, ['2A','6A'] ).
verb( 'captain', 'captains', 'captaining', 'captained', 'captained', tran, ['6A'] ).
verb( 'captivate', 'captivates', 'captivating', 'captivated', 'captivated', tran, ['6A'] ).
verb( 'capture', 'captures', 'capturing', 'captured', 'captured', tran, ['6A'] ).
verb( 'carbonize', 'carbonizes', 'carbonizing', 'carbonized', 'carbonized', tran, ['6A'] ).
verb( 'card', 'cards', 'carding', 'carded', 'carded', tran, [] ).
verb( 'care', 'cares', 'caring', 'cared', 'cared', intran, ['2A','3A','3B','4C'] ).
verb( 'careen', 'careens', 'careening', 'careened', 'careened', _, ['2A','6A'] ).
verb( 'career', 'careers', 'careering', 'careered', 'careered', intran, ['2C','3A'] ).
verb( 'caress', 'caresses', 'caressing', 'caressed', 'caressed', tran, ['6A'] ).
verb( 'caricature', 'caricatures', 'caricaturing', 'caricatured', 'caricatured', tran, ['6A'] ).
verb( 'carol', 'carols', 'carolling', 'carolled', 'carolled', tran, [] ).
verb( 'carouse', 'carouses', 'carousing', 'caroused', 'caroused', tran, ['2A'] ).
verb( 'carp', 'carps', 'carping', 'carped', 'carped', tran, ['2A','3A'] ).
verb( 'carpet', 'carpets', 'carpeting', 'carpeted', 'carpeted', tran, ['6A'] ).
verb( 'carry', 'carries', 'carrying', 'carried', 'carried', _, ['2B','2C','6A','15A','15B','16B'] ).
verb( 'cart', 'carts', 'carting', 'carted', 'carted', tran, ['6A','15B'] ).
verb( 'cartoon', 'cartoons', 'cartooning', 'cartooned', 'cartooned', tran, [] ).
verb( 'carve', 'carves', 'carving', 'carved', 'carved', _, ['6A','14','15A','15B'] ).
verb( 'cascade', 'cascades', 'cascading', 'cascaded', 'cascaded', intran, [] ).
verb( 'case', 'cases', 'casing', 'cased', 'cased', tran, ['6A'] ).
verb( 'cash', 'cashes', 'cashing', 'cashed', 'cashed', _, ['2C','6A','12B','13B'] ).
verb( 'cashier', 'cashiers', 'cashiering', 'cashiered', 'cashiered', tran, ['6A'] ).
verb( 'cast', 'casts', 'casting', 'cast', 'cast', _, ['2C','6A','15A','15B'] ).
verb( 'castigate', 'castigates', 'castigating', 'castigated', 'castigated', tran, ['6A'] ).
verb( 'castle', 'castles', 'castling', 'castled', 'castled', intran, [] ).
verb( 'castrate', 'castrates', 'castrating', 'castrated', 'castrated', tran, ['6A'] ).
verb( 'catalogue', 'catalogues', 'cataloguing', 'catalogued', 'catalogued', tran, ['6A'] ).
verb( 'catapult', 'catapults', 'catapulting', 'catapulted', 'catapulted', tran, [] ).
verb( 'catcall', 'catcalls', 'catcalling', 'catcalled', 'catcalled', intran, [] ).
verb( 'catch', 'catches', 'catching', 'caught', 'caught', _, ['2C','3A','6A','12C','14','15A','15B','19B','22'] ).
verb( 'catechize', 'catechizes', 'catechizing', 'catechized', 'catechized', tran, ['6A'] ).
verb( 'categorize', 'categorizes', 'categorizing', 'categorized', 'categorized', tran, ['6A'] ).
verb( 'cater', 'caters', 'catering', 'catered', 'catered', intran, ['3A'] ).
verb( 'caterwaul', 'caterwauls', 'caterwauling', 'caterwauled', 'caterwauled', intran, [] ).
verb( 'caulk', 'caulks', 'caulking', 'caulked', 'caulked', tran, ['6A'] ).
verb( 'cause', 'causes', 'causing', 'caused', 'caused', tran, ['6A','12A','13A','17'] ).
verb( 'cauterize', 'cauterizes', 'cauterizing', 'cauterized', 'cauterized', tran, ['6A'] ).
verb( 'caution', 'cautions', 'cautioning', 'cautioned', 'cautioned', tran, ['6A','14','17'] ).
verb( 'cave', 'caves', 'caving', 'caved', 'caved', _, ['2C','15B'] ).
verb( 'cavil', 'cavils', 'cavilling', 'cavilled', 'cavilled', intran, ['2A','3A'] ).
verb( 'cavort', 'cavorts', 'cavorting', 'cavorted', 'cavorted', intran, [] ).
verb( 'caw', 'caws', 'cawing', 'cawed', 'cawed', _, ['2A','15B'] ).
verb( 'cease', 'ceases', 'ceasing', 'ceased', 'ceased', _, ['2A','3A','6A','6D','7A'] ).
verb( 'cede', 'cedes', 'ceding', 'ceded', 'ceded', tran, ['6A','14'] ).
verb( 'celebrate', 'celebrates', 'celebrating', 'celebrated', 'celebrated', tran, ['6A'] ).
verb( 'cement', 'cements', 'cementing', 'cemented', 'cemented', tran, ['6A','15B'] ).
verb( 'censor', 'censors', 'censoring', 'censored', 'censored', tran, ['6A'] ).
verb( 'censure', 'censures', 'censuring', 'censured', 'censured', tran, ['6A','14'] ).
verb( 'centralize', 'centralizes', 'centralizing', 'centralized', 'centralized', _, ['2A','6A'] ).
verb( 'centre', 'centres', 'centring', 'centred', 'centred', _, ['3A','6A','14','15A'] ).
verb( 'certificate', 'certificates', 'certificating', 'certificated', 'certificated', tran, [] ).
verb( 'certify', 'certifies', 'certifying', 'certified', 'certified', _, ['3A','6A','9','16B','25'] ).
verb( 'chafe', 'chafes', 'chafing', 'chafed', 'chafed', _, ['2A','3A','6A'] ).
verb( 'chaff', 'chaffs', 'chaffing', 'chaffed', 'chaffed', tran, ['2A','2C','6A','15A'] ).
verb( 'chagrin', 'chagrins', 'chagrining', 'chagrined', 'chagrined', tran, ['6A'] ).
verb( 'chain', 'chains', 'chaining', 'chained', 'chained', tran, ['6A','15A','15B'] ).
verb( 'chair', 'chairs', 'chairing', 'chaired', 'chaired', tran, ['6A'] ).
verb( 'chalk', 'chalks', 'chalking', 'chalked', 'chalked', tran, ['15B'] ).
verb( 'challenge', 'challenges', 'challenging', 'challenged', 'challenged', tran, ['6A','14','17'] ).
verb( 'champ', 'champs', 'champing', 'champed', 'champed', _, ['2A','2C','4A','6A'] ).
verb( 'champion', 'champions', 'championing', 'championed', 'championed', tran, ['6A'] ).
verb( 'chance', 'chances', 'chancing', 'chanced', 'chanced', _, ['2A','3A','4E','6A','6C'] ).
verb( 'change', 'changes', 'changing', 'changed', 'changed', _, ['2A','2C','6A','14'] ).
verb( 'channel', 'channels', 'channelling', 'channelled', 'channelled', tran, ['6A','14'] ).
verb( 'chant', 'chants', 'chanting', 'chanted', 'chanted', _, ['2A','6A'] ).
verb( 'chap', 'chaps', 'chapping', 'chapped', 'chapped', _, ['2A','6A'] ).
verb( 'chaperon', 'chaperons', 'chaperoning', 'chaperoned', 'chaperoned', tran, ['6A'] ).
verb( 'char', 'chars', 'charring', 'charred', 'charred', _, ['2A','6A'] ).
verb( 'characterize', 'characterizes', 'characterizing', 'characterized', 'characterized', tran, ['6A'] ).
verb( 'charge', 'charges', 'charging', 'charged', 'charged', _, ['2A','2B','2C','6A','14','15A','15B','17'] ).
verb( 'charm', 'charms', 'charming', 'charmed', 'charmed', _, ['2A','6A','15A'] ).
verb( 'chart', 'charts', 'charting', 'charted', 'charted', tran, ['6A'] ).
verb( 'charter', 'charters', 'chartering', 'chartered', 'chartered', tran, ['6A'] ).
verb( 'chase', 'chases', 'chasing', 'chased', 'chased', _, ['2C','3A','6A','15A','15B'] ).
verb( 'chasten', 'chastens', 'chastening', 'chastened', 'chastened', tran, ['6A'] ).
verb( 'chastise', 'chastises', 'chastising', 'chastised', 'chastised', tran, ['6A'] ).
verb( 'chat', 'chats', 'chatting', 'chatted', 'chatted', _, ['2A','2C','15B'] ).
verb( 'chatter', 'chatters', 'chattering', 'chattered', 'chattered', intran, ['2A','2C'] ).
verb( 'chaw', 'chaws', 'chawing', 'chawed', 'chawed', tran, [] ).
verb( 'cheapen', 'cheapens', 'cheapening', 'cheapened', 'cheapened', _, ['2A','6A'] ).
verb( 'cheat', 'cheats', 'cheating', 'cheated', 'cheated', _, ['2A','2C','6A','14'] ).
verb( 'check', 'checks', 'checking', 'checked', 'checked', _, ['2C','6A','15B'] ).
verb( 'checker', 'checkers', 'checkering', 'checkered', 'checkered', tran, [] ).
verb( 'checkmate', 'checkmates', 'checkmating', 'checkmated', 'checkmated', tran, ['6A'] ).
verb( 'cheek', 'cheeks', 'cheeking', 'cheeked', 'cheeked', tran, ['6A'] ).
verb( 'cheep', 'cheeps', 'cheeping', 'cheeped', 'cheeped', intran, [] ).
verb( 'cheer', 'cheers', 'cheering', 'cheered', 'cheered', _, ['2A','2C','6A','15B'] ).
verb( 'chequer', 'chequers', 'chequering', 'chequered', 'chequered', tran, [] ).
verb( 'cherish', 'cherishes', 'cherishing', 'cherished', 'cherished', tran, ['6A'] ).
verb( 'chew', 'chews', 'chewing', 'chewed', 'chewed', _, ['2A','2C','3A','6A','15B'] ).
verb( 'chide', 'chides', 'chiding', 'chided', 'chided', _, ['6A','14'] ).
verb( 'chill', 'chills', 'chilling', 'chilled', 'chilled', _, ['2A','2C','6A'] ).
verb( 'chime', 'chimes', 'chiming', 'chimed', 'chimed', _, ['2A','2C','6A','15A'] ).
verb( 'chink', 'chinks', 'chinking', 'chinked', 'chinked', _, ['2A','2C','6A','15B'] ).
verb( 'chip', 'chips', 'chipping', 'chipped', 'chipped', _, ['2A','2C','6A','15A','15B'] ).
verb( 'chirp', 'chirps', 'chirping', 'chirped', 'chirped', _, [] ).
verb( 'chirrup', 'chirrups', 'chirruping', 'chirruped', 'chirruped', _, [] ).
verb( 'chisel', 'chisels', 'chiselling', 'chiselled', 'chiselled', tran, ['6A','14','15B'] ).
verb( 'chivvy', 'chivvies', 'chivvying', 'chivvied', 'chivvied', tran, ['15B'] ).
verb( 'chivy', 'chivies', 'chivying', 'chivied', 'chivied', tran, ['15B'] ).
verb( 'chlorinate', 'chlorinates', 'chlorinating', 'chlorinated', 'chlorinated', tran, [] ).
verb( 'chock', 'chocks', 'chocking', 'chocked', 'chocked', tran, ['6A','15B'] ).
verb( 'choke', 'chokes', 'choking', 'choked', 'choked', _, ['2A','3A','6A','14','15B'] ).
verb( 'choose', 'chooses', 'choosing', 'chose', 'chosen', _, ['2A','2C','3A','6A','7A','16A','16B','23'] ).
verb( 'chop', 'chops', 'chopping', 'chopped', 'chopped', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'chortle', 'chortles', 'chortling', 'chortled', 'chortled', intran, [] ).
verb( 'chorus', 'choruses', 'chorusing', 'chorused', 'chorused', tran, [] ).
verb( 'christen', 'christens', 'christening', 'christened', 'christened', tran, ['6A','23'] ).
verb( 'chronicle', 'chronicles', 'chronicling', 'chronicled', 'chronicled', tran, ['6A'] ).
verb( 'chuck', 'chucks', 'chucking', 'chucked', 'chucked', tran, ['6A','12A','13A','15A','15B'] ).
verb( 'chuckle', 'chuckles', 'chuckling', 'chuckled', 'chuckled', intran, ['2A','2C'] ).
verb( 'chug', 'chugs', 'chugging', 'chugged', 'chugged', intran, ['2C'] ).
verb( 'chum', 'chums', 'chumming', 'chummed', 'chummed', intran, ['2C'] ).
verb( 'churn', 'churns', 'churning', 'churned', 'churned', _, ['2A','2C','6A','15B'] ).
verb( 'cipher', 'ciphers', 'ciphering', 'ciphered', 'ciphered', _, ['2A','6A'] ).
verb( 'circle', 'circles', 'circling', 'circled', 'circled', _, ['2A','2C','6A'] ).
verb( 'circularize', 'circularizes', 'circularizing', 'circularized', 'circularized', tran, ['6A'] ).
verb( 'circulate', 'circulates', 'circulating', 'circulated', 'circulated', _, ['2A','2C','6A'] ).
verb( 'circumcise', 'circumcises', 'circumcising', 'circumcised', 'circumcised', tran, [] ).
verb( 'circumnavigate', 'circumnavigates', 'circumnavigating', 'circumnavigated', 'circumnavigated', tran, ['6A'] ).
verb( 'circumscribe', 'circumscribes', 'circumscribing', 'circumscribed', 'circumscribed', tran, ['6A'] ).
verb( 'circumvent', 'circumvents', 'circumventing', 'circumvented', 'circumvented', tran, ['6A'] ).
verb( 'cite', 'cites', 'citing', 'cited', 'cited', tran, ['6A'] ).
verb( 'civilize', 'civilizes', 'civilizing', 'civilized', 'civilized', tran, ['6A'] ).
verb( 'clack', 'clacks', 'clacking', 'clacked', 'clacked', intran, [] ).
verb( 'claim', 'claims', 'claiming', 'claimed', 'claimed', _, ['2A','4A','7A','9'] ).
verb( 'clam', 'clams', 'clamming', 'clammed', 'clammed', intran, ['2A','2C'] ).
verb( 'clamber', 'clambers', 'clambering', 'clambered', 'clambered', intran, ['2C'] ).
verb( 'clamour', 'clamours', 'clamouring', 'clamoured', 'clamoured', _, ['2A','2C','4A'] ).
verb( 'clamp', 'clamps', 'clamping', 'clamped', 'clamped', _, ['2C','6A','15B'] ).
verb( 'clang', 'clangs', 'clanging', 'clanged', 'clanged', _, [] ).
verb( 'clank', 'clanks', 'clanking', 'clanked', 'clanked', _, [] ).
verb( 'clap', 'claps', 'clapping', 'clapped', 'clapped', _, ['2A','2B','6A','14','15A','15B'] ).
verb( 'clarify', 'clarifies', 'clarifying', 'clarified', 'clarified', _, ['2A','6A'] ).
verb( 'clash', 'clashes', 'clashing', 'clashed', 'clashed', _, ['2A','2C','3A','6A','15B'] ).
verb( 'clasp', 'clasps', 'clasping', 'clasped', 'clasped', _, ['2A','6A','15A'] ).
verb( 'class', 'classes', 'classing', 'classed', 'classed', tran, ['6A','14'] ).
verb( 'classify', 'classifies', 'classifying', 'classified', 'classified', tran, ['6A'] ).
verb( 'clatter', 'clatters', 'clattering', 'clattered', 'clattered', _, ['2A','2C','6A'] ).
verb( 'claw', 'claws', 'clawing', 'clawed', 'clawed', tran, ['3A','6A','15B'] ).
verb( 'clean', 'cleans', 'cleaning', 'cleaned', 'cleaned', _, ['2A','2C','6A','15B'] ).
verb( 'cleanse', 'cleanses', 'cleansing', 'cleansed', 'cleansed', tran, ['6A'] ).
verb( 'clear', 'clears', 'clearing', 'cleared', 'cleared', _, ['2C','6A','14','15B'] ).
verb( 'cleave', 'cleaves', 'cleaving', 'cleaved', 'cleaved', _, ['2A','3A','6A','14','15A','15B','22'] ).
verb( 'clench', 'clenches', 'clenching', 'clenched', 'clenched', tran, ['6A','14'] ).
verb( 'clerk', 'clerks', 'clerking', 'clerked', 'clerked', intran, [] ).
verb( 'clew', 'clews', 'clewing', 'clewed', 'clewed', tran, [] ).
verb( 'click', 'clicks', 'clicking', 'clicked', 'clicked', intran, ['2A'] ).
verb( 'climax', 'climaxes', 'climaxing', 'climaxed', 'climaxed', _, [] ).
verb( 'climb', 'climbs', 'climbing', 'climbed', 'climbed', _, ['2A','2C','6A'] ).
verb( 'clinch', 'clinches', 'clinching', 'clinched', 'clinched', _, ['2A','6A'] ).
verb( 'cling', 'clings', 'clinging', 'clung', 'clung', intran, ['2C','3A'] ).
verb( 'clink', 'clinks', 'clinking', 'clinked', 'clinked', _, [] ).
verb( 'clip', 'clips', 'clipping', 'clipped', 'clipped', tran, ['2C','6A','15A','15B','22'] ).
verb( 'cloak', 'cloaks', 'cloaking', 'cloaked', 'cloaked', tran, ['6A'] ).
verb( 'clobber', 'clobbers', 'clobbering', 'clobbered', 'clobbered', tran, ['6A'] ).
verb( 'clock', 'clocks', 'clocking', 'clocked', 'clocked', _, ['2C','6A','15B'] ).
verb( 'clog', 'clogs', 'clogging', 'clogged', 'clogged', _, ['2A','2C','6A','15B'] ).
verb( 'cloister', 'cloisters', 'cloistering', 'cloistered', 'cloistered', tran, ['6A'] ).
verb( 'close', 'closes', 'closing', 'closed', 'closed', _, ['2A','2C','3A','6A','15B'] ).
verb( 'closet', 'closets', 'closeting', 'closeted', 'closeted', tran, [] ).
verb( 'clot', 'clots', 'clotting', 'clotted', 'clotted', _, ['2A','6A'] ).
verb( 'clothe', 'clothes', 'clothing', 'clothed', 'clothed', tran, ['6A'] ).
verb( 'cloud', 'clouds', 'clouding', 'clouded', 'clouded', _, ['2A','2C','3A','6A'] ).
verb( 'clout', 'clouts', 'clouting', 'clouted', 'clouted', tran, [] ).
verb( 'clown', 'clowns', 'clowning', 'clowned', 'clowned', intran, ['2A'] ).
verb( 'cloy', 'cloys', 'cloying', 'cloyed', 'cloyed', _, ['2A','6A'] ).
verb( 'club', 'clubs', 'clubbing', 'clubbed', 'clubbed', _, ['2C','6A','16A'] ).
verb( 'cluck', 'clucks', 'clucking', 'clucked', 'clucked', intran, [] ).
verb( 'clump', 'clumps', 'clumping', 'clumped', 'clumped', _, ['2A','2C'] ).
verb( 'clunk', 'clunks', 'clunking', 'clunked', 'clunked', intran, [] ).
verb( 'cluster', 'clusters', 'clustering', 'clustered', 'clustered', intran, ['2A','2C','3A'] ).
verb( 'clutch', 'clutches', 'clutching', 'clutched', 'clutched', _, ['2A','3A','15A'] ).
verb( 'clutter', 'clutters', 'cluttering', 'cluttered', 'cluttered', tran, ['6A','15B'] ).
verb( 'co-opt', 'co-opts', 'co-opting', 'co-opted', 'co-opted', tran, ['6A','14'] ).
verb( 'co-star', 'co-stars', 'co-starring', 'co-starred', 'co-starred', _, ['6A','14'] ).
verb( 'coach', 'coaches', 'coaching', 'coached', 'coached', _, ['2A','6A','14'] ).
verb( 'coagulate', 'coagulates', 'coagulating', 'coagulated', 'coagulated', _, ['2A','6A'] ).
verb( 'coal', 'coals', 'coaling', 'coaled', 'coaled', _, ['2A','6A'] ).
verb( 'coalesce', 'coalesces', 'coalescing', 'coalesced', 'coalesced', intran, [] ).
verb( 'coarsen', 'coarsens', 'coarsening', 'coarsened', 'coarsened', _, ['2A','6A'] ).
verb( 'coast', 'coasts', 'coasting', 'coasted', 'coasted', _, ['2A','2C','3A','6A'] ).
verb( 'coat', 'coats', 'coating', 'coated', 'coated', tran, ['6A','14'] ).
verb( 'coax', 'coaxes', 'coaxing', 'coaxed', 'coaxed', _, ['2A','6A','14','15B','17'] ).
verb( 'cobble', 'cobbles', 'cobbling', 'cobbled', 'cobbled', tran, ['6A'] ).
verb( 'cock', 'cocks', 'cocking', 'cocked', 'cocked', tran, ['6A','15B'] ).
verb( 'cocoon', 'cocoons', 'cocooning', 'cocooned', 'cocooned', tran, ['6A'] ).
verb( 'cod', 'cods', 'codding', 'codded', 'codded', _, ['2A','6A'] ).
verb( 'coddle', 'coddles', 'coddling', 'coddled', 'coddled', tran, ['6A'] ).
verb( 'code', 'codes', 'coding', 'coded', 'coded', tran, [] ).
verb( 'codify', 'codifies', 'codifying', 'codified', 'codified', tran, ['6A'] ).
verb( 'coerce', 'coerces', 'coercing', 'coerced', 'coerced', tran, ['6A','14'] ).
verb( 'coexist', 'coexists', 'coexisting', 'coexisted', 'coexisted', intran, ['2A','3A'] ).
verb( 'cogitate', 'cogitates', 'cogitating', 'cogitated', 'cogitated', _, ['2A','3A','6A','14'] ).
verb( 'cohabit', 'cohabits', 'cohabiting', 'cohabited', 'cohabited', intran, [] ).
verb( 'cohere', 'coheres', 'cohering', 'cohered', 'cohered', intran, ['2A'] ).
verb( 'coil', 'coils', 'coiling', 'coiled', 'coiled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'coin', 'coins', 'coining', 'coined', 'coined', tran, ['6A'] ).
verb( 'coincide', 'coincides', 'coinciding', 'coincided', 'coincided', intran, ['2A','3A'] ).
verb( 'coke', 'cokes', 'coking', 'coked', 'coked', tran, [] ).
verb( 'cold-shoulder', 'cold-shoulders', 'cold-shouldering', 'cold-shouldered', 'cold-shouldered', tran, ['6A'] ).
verb( 'collaborate', 'collaborates', 'collaborating', 'collaborated', 'collaborated', intran, ['2A','3A'] ).
verb( 'collapse', 'collapses', 'collapsing', 'collapsed', 'collapsed', intran, ['2A','6A'] ).
verb( 'collar', 'collars', 'collaring', 'collared', 'collared', tran, ['6A'] ).
verb( 'collate', 'collates', 'collating', 'collated', 'collated', tran, ['6A'] ).
verb( 'collect', 'collects', 'collecting', 'collected', 'collected', _, ['2A','2C','6A','15B'] ).
verb( 'collectivize', 'collectivizes', 'collectivizing', 'collectivized', 'collectivized', tran, ['6A'] ).
verb( 'collide', 'collides', 'colliding', 'collided', 'collided', intran, ['2A','2C','3A'] ).
verb( 'collocate', 'collocates', 'collocating', 'collocated', 'collocated', intran, ['2A','3A'] ).
verb( 'colonize', 'colonizes', 'colonizing', 'colonized', 'colonized', tran, ['6A'] ).
verb( 'colour', 'colours', 'colouring', 'coloured', 'coloured', _, ['2A','2C','6A','22'] ).
verb( 'comb', 'combs', 'combing', 'combed', 'combed', _, ['2C','3A','6A','15B'] ).
verb( 'combat', 'combats', 'combating', 'combated', 'combated', _, ['3A','6A'] ).
verb( 'combine', 'combines', 'combining', 'combined', 'combined', _, ['2A','3A','4A','6A','14'] ).
verb( 'come', 'comes', 'coming', 'came', 'come', intran, ['2A','2B','2C','2D','2E','3A','4A'] ).
verb( 'comfort', 'comforts', 'comforting', 'comforted', 'comforted', tran, ['6A'] ).
verb( 'command', 'commands', 'commanding', 'commanded', 'commanded', _, ['2A','6A','9','17'] ).
verb( 'commandeer', 'commandeers', 'commandeering', 'commandeered', 'commandeered', tran, ['6A'] ).
verb( 'commemorate', 'commemorates', 'commemorating', 'commemorated', 'commemorated', tran, ['6A'] ).
verb( 'commence', 'commences', 'commencing', 'commenced', 'commenced', _, ['2A','3A','6A','6C'] ).
verb( 'commend', 'commends', 'commending', 'commended', 'commended', tran, ['6A','14'] ).
verb( 'comment', 'comments', 'commenting', 'commented', 'commented', intran, ['2A','3A'] ).
verb( 'commentate', 'commentates', 'commentating', 'commentated', 'commentated', intran, ['3A'] ).
verb( 'commercialize', 'commercializes', 'commercializing', 'commercialized', 'commercialized', tran, ['6A'] ).
verb( 'commingle', 'commingles', 'commingling', 'commingled', 'commingled', _, ['2A','6A'] ).
verb( 'commiserate', 'commiserates', 'commiserating', 'commiserated', 'commiserated', intran, ['3A'] ).
verb( 'commission', 'commissions', 'commissioning', 'commissioned', 'commissioned', tran, ['6A','17'] ).
verb( 'commit', 'commits', 'committing', 'committed', 'committed', tran, ['6A','14','16A'] ).
verb( 'commune', 'communes', 'communing', 'communed', 'communed', intran, ['2C','3A'] ).
verb( 'communicate', 'communicates', 'communicating', 'communicated', 'communicated', _, ['2A','3A','6A','14'] ).
verb( 'commute', 'commutes', 'commuting', 'commuted', 'commuted', _, ['2A','6A','14'] ).
verb( 'comp`ere', 'comp`eres', 'comp`ering', 'comp`ered', 'comp`ered', tran, [] ).
verb( 'compact', 'compacts', 'compacting', 'compacted', 'compacted', tran, [] ).
verb( 'compare', 'compares', 'comparing', 'compared', 'compared', _, ['3A','6A','14'] ).
verb( 'compartmentalize', 'compartmentalizes', 'compartmentalizing', 'compartmentalized', 'compartmentalized', tran, [] ).
verb( 'compass', 'compasses', 'compassing', 'compassed', 'compassed', tran, [] ).
verb( 'compel', 'compels', 'compelling', 'compelled', 'compelled', tran, ['6A','14','17'] ).
verb( 'compensate', 'compensates', 'compensating', 'compensated', 'compensated', _, ['3A','6A','14'] ).
verb( 'compete', 'competes', 'competing', 'competed', 'competed', intran, ['2A','3A'] ).
verb( 'compile', 'compiles', 'compiling', 'compiled', 'compiled', tran, ['6A'] ).
verb( 'complain', 'complains', 'complaining', 'complained', 'complained', intran, ['2A','3A','3B'] ).
verb( 'complement', 'complements', 'complementing', 'complemented', 'complemented', tran, ['6A'] ).
verb( 'complete', 'completes', 'completing', 'completed', 'completed', tran, ['6A'] ).
verb( 'complicate', 'complicates', 'complicating', 'complicated', 'complicated', tran, ['6A'] ).
verb( 'compliment', 'compliments', 'complimenting', 'complimented', 'complimented', tran, ['6A','14'] ).
verb( 'comply', 'complies', 'complying', 'complied', 'complied', tran, ['2A','3A'] ).
verb( 'comport', 'comports', 'comporting', 'comported', 'comported', _, ['3A','15A'] ).
verb( 'compose', 'composes', 'composing', 'composed', 'composed', _, ['2A','6A','16A'] ).
verb( 'compost', 'composts', 'composting', 'composted', 'composted', tran, ['6A'] ).
verb( 'compound', 'compounds', 'compounding', 'compounded', 'compounded', _, ['2A','3A','6A'] ).
verb( 'comprehend', 'comprehends', 'comprehending', 'comprehended', 'comprehended', tran, ['6A'] ).
verb( 'compress', 'compresses', 'compressing', 'compressed', 'compressed', tran, ['6A','14'] ).
verb( 'comprise', 'comprises', 'comprising', 'comprised', 'comprised', tran, ['6A'] ).
verb( 'compromise', 'compromises', 'compromising', 'compromised', 'compromised', _, ['2A','6A'] ).
verb( 'compute', 'computes', 'computing', 'computed', 'computed', _, ['2A','6A','14'] ).
verb( 'computerize', 'computerizes', 'computerizing', 'computerized', 'computerized', tran, ['6A'] ).
verb( 'con', 'cons', 'conning', 'conned', 'conned', tran, ['6A','14'] ).
verb( 'conceal', 'conceals', 'concealing', 'concealed', 'concealed', tran, ['6A','14'] ).
verb( 'concede', 'concedes', 'conceding', 'conceded', 'conceded', tran, ['6A','9','12A','13A'] ).
verb( 'conceive', 'conceives', 'conceiving', 'conceived', 'conceived', _, ['2A','3A','6A','9','10','14'] ).
verb( 'concentrate', 'concentrates', 'concentrating', 'concentrated', 'concentrated', _, ['2A','3A','6A','14'] ).
verb( 'concern', 'concerns', 'concerning', 'concerned', 'concerned', tran, ['6A','14'] ).
verb( 'concert', 'concerts', 'concerting', 'concerted', 'concerted', tran, [] ).
verb( 'conciliate', 'conciliates', 'conciliating', 'conciliated', 'conciliated', tran, ['6A'] ).
verb( 'conclude', 'concludes', 'concluding', 'concluded', 'concluded', _, ['2A','3A','6A','7A','9','14'] ).
verb( 'concoct', 'concocts', 'concocting', 'concocted', 'concocted', tran, ['6A'] ).
verb( 'concrete', 'concretes', 'concreting', 'concreted', 'concreted', _, ['2A','6A'] ).
verb( 'concur', 'concurs', 'concurring', 'concurred', 'concurred', intran, ['2A','3A','4A'] ).
verb( 'concuss', 'concusses', 'concussing', 'concussed', 'concussed', tran, ['6A'] ).
verb( 'condemn', 'condemns', 'condemning', 'condemned', 'condemned', tran, ['6A','14','16B','17'] ).
verb( 'condense', 'condenses', 'condensing', 'condensed', 'condensed', _, ['2A','3A','6A','14'] ).
verb( 'condescend', 'condescends', 'condescending', 'condescended', 'condescended', intran, ['2A','3A','4A'] ).
verb( 'condition', 'conditions', 'conditioning', 'conditioned', 'conditioned', tran, ['6A'] ).
verb( 'condole', 'condoles', 'condoling', 'condoled', 'condoled', intran, ['3A'] ).
verb( 'condone', 'condones', 'condoning', 'condoned', 'condoned', tran, ['6A','6C'] ).
verb( 'conduce', 'conduces', 'conducing', 'conduced', 'conduced', intran, ['3A'] ).
verb( 'conduct', 'conducts', 'conducting', 'conducted', 'conducted', _, ['2A','6A','14','15A','15B','16A'] ).
verb( 'cone', 'cones', 'coning', 'coned', 'coned', tran, ['15B'] ).
verb( 'confab', 'confabs', 'confabbing', 'confabbed', 'confabbed', intran, [] ).
verb( 'confabulate', 'confabulates', 'confabulating', 'confabulated', 'confabulated', intran, ['2A','3A'] ).
verb( 'confederate', 'confederates', 'confederating', 'confederated', 'confederated', _, ['2A','3A','6A','14'] ).
verb( 'confer', 'confers', 'conferring', 'conferred', 'conferred', _, ['2A','3A','14'] ).
verb( 'confess', 'confesses', 'confessing', 'confessed', 'confessed', _, ['2A','3A','3B','6A','9','14','25'] ).
verb( 'confide', 'confides', 'confiding', 'confided', 'confided', _, ['3A','14'] ).
verb( 'configure', 'configures', 'configuring', 'configured', 'configured', tran, ['6A'] ).
verb( 'confine', 'confines', 'confining', 'confined', 'confined', tran, ['6A','14'] ).
verb( 'confirm', 'confirms', 'confirming', 'confirmed', 'confirmed', tran, ['6A','9'] ).
verb( 'confiscate', 'confiscates', 'confiscating', 'confiscated', 'confiscated', tran, ['6A'] ).
verb( 'conflict', 'conflicts', 'conflicting', 'conflicted', 'conflicted', intran, ['2A','3A'] ).
verb( 'conform', 'conforms', 'conforming', 'conformed', 'conformed', _, ['2A','3A','14'] ).
verb( 'confound', 'confounds', 'confounding', 'confounded', 'confounded', tran, ['6A','14'] ).
verb( 'confront', 'confronts', 'confronting', 'confronted', 'confronted', tran, ['6A','14'] ).
verb( 'confuse', 'confuses', 'confusing', 'confused', 'confused', tran, ['6A','14'] ).
verb( 'confute', 'confutes', 'confuting', 'confuted', 'confuted', tran, ['6A'] ).
verb( 'congeal', 'congeals', 'congealing', 'congealed', 'congealed', _, ['2A','6A'] ).
verb( 'conglomerate', 'conglomerates', 'conglomerating', 'conglomerated', 'conglomerated', _, ['2A','6A'] ).
verb( 'congratulate', 'congratulates', 'congratulating', 'congratulated', 'congratulated', tran, ['6A','14'] ).
verb( 'congregate', 'congregates', 'congregating', 'congregated', 'congregated', _, ['2A','2C','6A'] ).
verb( 'conjecture', 'conjectures', 'conjecturing', 'conjectured', 'conjectured', _, ['2A','6A','9','25'] ).
verb( 'conjoin', 'conjoins', 'conjoining', 'conjoined', 'conjoined', _, ['2A','6A'] ).
verb( 'conjugate', 'conjugates', 'conjugating', 'conjugated', 'conjugated', _, ['2A','6A'] ).
verb( 'conjure', 'conjures', 'conjuring', 'conjured', 'conjured', _, ['2A','15A','15B','17'] ).
verb( 'conk', 'conks', 'conking', 'conked', 'conked', intran, [] ).
verb( 'connect', 'connects', 'connecting', 'connected', 'connected', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'connive', 'connives', 'conniving', 'connived', 'connived', intran, ['3A'] ).
verb( 'connote', 'connotes', 'connoting', 'connoted', 'connoted', tran, ['6A'] ).
verb( 'conquer', 'conquers', 'conquering', 'conquered', 'conquered', tran, ['6A'] ).
verb( 'conscript', 'conscripts', 'conscripting', 'conscripted', 'conscripted', tran, ['6A','14'] ).
verb( 'consecrate', 'consecrates', 'consecrating', 'consecrated', 'consecrated', tran, ['6A','14','23'] ).
verb( 'consent', 'consents', 'consenting', 'consented', 'consented', intran, ['2A','3A','7A'] ).
verb( 'conserve', 'conserves', 'conserving', 'conserved', 'conserved', tran, ['6A'] ).
verb( 'consider', 'considers', 'considering', 'considered', 'considered', tran, ['6A','6C','8','9','10','25'] ).
verb( 'consign', 'consigns', 'consigning', 'consigned', 'consigned', tran, ['6A','14'] ).
verb( 'consist', 'consists', 'consisting', 'consisted', 'consisted', intran, ['3A'] ).
verb( 'console', 'consoles', 'consoling', 'consoled', 'consoled', tran, ['6A','14'] ).
verb( 'consolidate', 'consolidates', 'consolidating', 'consolidated', 'consolidated', _, ['2A','6A'] ).
verb( 'consort', 'consorts', 'consorting', 'consorted', 'consorted', intran, ['2C','3A'] ).
verb( 'conspire', 'conspires', 'conspiring', 'conspired', 'conspired', _, ['2A','3A','4A','6A'] ).
verb( 'constipate', 'constipates', 'constipating', 'constipated', 'constipated', tran, ['6A'] ).
verb( 'constitute', 'constitutes', 'constituting', 'constituted', 'constituted', tran, ['6A','23'] ).
verb( 'constitutionalize', 'constitutionalizes', 'constitutionalizing', 'constitutionalized', 'constitutionalized', tran, [] ).
verb( 'constrain', 'constrains', 'constraining', 'constrained', 'constrained', tran, ['6A','17'] ).
verb( 'constrict', 'constricts', 'constricting', 'constricted', 'constricted', tran, ['6A'] ).
verb( 'construct', 'constructs', 'constructing', 'constructed', 'constructed', tran, ['6A'] ).
verb( 'construe', 'construes', 'construing', 'construed', 'construed', _, ['2A','6A'] ).
verb( 'consult', 'consults', 'consulting', 'consulted', 'consulted', _, ['3A','6A','14'] ).
verb( 'consume', 'consumes', 'consuming', 'consumed', 'consumed', _, ['6A'] ).
verb( 'consummate', 'consummates', 'consummating', 'consummated', 'consummated', tran, ['6A'] ).
verb( 'contact', 'contacts', 'contacting', 'contacted', 'contacted', tran, ['6A'] ).
verb( 'contain', 'contains', 'containing', 'contained', 'contained', tran, ['6A'] ).
verb( 'contaminate', 'contaminates', 'contaminating', 'contaminated', 'contaminated', tran, ['6A'] ).
verb( 'contemn', 'contemns', 'contemning', 'contemned', 'contemned', tran, ['6A'] ).
verb( 'contemplate', 'contemplates', 'contemplating', 'contemplated', 'contemplated', _, ['2A','6A','6C','19C'] ).
verb( 'contend', 'contends', 'contending', 'contended', 'contended', _, ['3A','9'] ).
verb( 'content', 'contents', 'contenting', 'contented', 'contented', tran, ['6A','14'] ).
verb( 'contest', 'contests', 'contesting', 'contested', 'contested', _, ['3A','6A','9'] ).
verb( 'continue', 'continues', 'continuing', 'continued', 'continued', _, ['2A','2B','2D','2E','6A','6D','7A','14'] ).
verb( 'contort', 'contorts', 'contorting', 'contorted', 'contorted', tran, ['6A','14'] ).
verb( 'contour', 'contours', 'contouring', 'contoured', 'contoured', tran, ['6A'] ).
verb( 'contract', 'contracts', 'contracting', 'contracted', 'contracted', _, ['2A','2C','4A','6A','14'] ).
verb( 'contradict', 'contradicts', 'contradicting', 'contradicted', 'contradicted', tran, ['6A'] ).
verb( 'contradistinguish', 'contradistinguishes', 'contradistinguishing', 'contradistinguished', 'contradistinguished', tran, ['14'] ).
verb( 'contrast', 'contrasts', 'contrasting', 'contrasted', 'contrasted', _, ['2C','3A','6A','14'] ).
verb( 'contravene', 'contravenes', 'contravening', 'contravened', 'contravened', tran, ['6A'] ).
verb( 'contribute', 'contributes', 'contributing', 'contributed', 'contributed', _, ['2A','3A','6A','14'] ).
verb( 'contrive', 'contrives', 'contriving', 'contrived', 'contrived', _, ['2A','6A','7A'] ).
verb( 'control', 'controls', 'controlling', 'controlled', 'controlled', tran, ['6A'] ).
verb( 'controvert', 'controverts', 'controverting', 'controverted', 'controverted', tran, ['6A'] ).
verb( 'contuse', 'contuses', 'contusing', 'contused', 'contused', tran, ['6A'] ).
verb( 'convalesce', 'convalesces', 'convalescing', 'convalesced', 'convalesced', intran, ['2A'] ).
verb( 'convene', 'convenes', 'convening', 'convened', 'convened', _, ['2A','6A'] ).
verb( 'converge', 'converges', 'converging', 'converged', 'converged', intran, ['2A','2C','3A'] ).
verb( 'converse', 'converses', 'conversing', 'conversed', 'conversed', intran, ['2A','2C','3A'] ).
verb( 'convert', 'converts', 'converting', 'converted', 'converted', tran, ['6A','14'] ).
verb( 'convey', 'conveys', 'conveying', 'conveyed', 'conveyed', tran, ['6A','14'] ).
verb( 'convict', 'convicts', 'convicting', 'convicted', 'convicted', tran, ['6A','14'] ).
verb( 'convince', 'convinces', 'convincing', 'convinced', 'convinced', tran, ['6A','11','14'] ).
verb( 'convoke', 'convokes', 'convoking', 'convoked', 'convoked', tran, ['6A'] ).
verb( 'convoy', 'convoys', 'convoying', 'convoyed', 'convoyed', tran, ['6A'] ).
verb( 'convulse', 'convulses', 'convulsing', 'convulsed', 'convulsed', tran, ['6A'] ).
verb( 'coo', 'coos', 'cooing', 'cooed', 'cooed', _, [] ).
verb( 'cook', 'cooks', 'cooking', 'cooked', 'cooked', _, ['2A','6A','12B','13B','15B'] ).
verb( 'cool', 'cools', 'cooling', 'cooled', 'cooled', _, ['2A','2C','6A'] ).
verb( 'coop', 'coops', 'cooping', 'cooped', 'cooped', tran, ['6A','15B'] ).
verb( 'cooperate', 'cooperates', 'cooperating', 'cooperated', 'cooperated', intran, ['2A','3A','4A'] ).
verb( 'coordinate', 'coordinates', 'coordinating', 'coordinated', 'coordinated', tran, ['6A'] ).
verb( 'cop', 'cops', 'copping', 'copped', 'copped', _, ['2C'] ).
verb( 'cope', 'copes', 'coping', 'coped', 'coped', intran, ['2A','3A'] ).
verb( 'copper', 'coppers', 'coppering', 'coppered', 'coppered', tran, [] ).
verb( 'copper-bottom', 'copper-bottoms', 'copper-bottoming', 'copper-bottomed', 'copper-bottomed', tran, [] ).
verb( 'copulate', 'copulates', 'copulating', 'copulated', 'copulated', intran, ['2A','3A'] ).
verb( 'copy', 'copies', 'copying', 'copied', 'copied', _, ['2A','6A','15A','15B'] ).
verb( 'copyright', 'copyrights', 'copyrighting', 'copyrighted', 'copyrighted', tran, [] ).
verb( 'cord', 'cords', 'cording', 'corded', 'corded', tran, [] ).
verb( 'cordon', 'cordons', 'cordoning', 'cordoned', 'cordoned', tran, ['15B'] ).
verb( 'core', 'cores', 'coring', 'cored', 'cored', tran, ['6A'] ).
verb( 'cork', 'corks', 'corking', 'corked', 'corked', tran, ['6A','15A'] ).
verb( 'corn', 'corns', 'corning', 'corned', 'corned', tran, [] ).
verb( 'corner', 'corners', 'cornering', 'cornered', 'cornered', _, ['2A','6A'] ).
verb( 'corral', 'corrals', 'corralling', 'corralled', 'corralled', tran, ['6A'] ).
verb( 'correct', 'corrects', 'correcting', 'corrected', 'corrected', tran, ['6A','14'] ).
verb( 'correlate', 'correlates', 'correlating', 'correlated', 'correlated', _, ['2A','3A','6A','14'] ).
verb( 'correspond', 'corresponds', 'corresponding', 'corresponded', 'corresponded', intran, ['2A','3A'] ).
verb( 'corroborate', 'corroborates', 'corroborating', 'corroborated', 'corroborated', tran, ['6A'] ).
verb( 'corrode', 'corrodes', 'corroding', 'corroded', 'corroded', _, ['2A','6A'] ).
verb( 'corrugate', 'corrugates', 'corrugating', 'corrugated', 'corrugated', _, ['2A','6A'] ).
verb( 'corrupt', 'corrupts', 'corrupting', 'corrupted', 'corrupted', _, ['2A','6A'] ).
verb( 'coruscate', 'coruscates', 'coruscating', 'coruscated', 'coruscated', intran, ['2A'] ).
verb( 'cosh', 'coshes', 'coshing', 'coshed', 'coshed', tran, [] ).
verb( 'cosset', 'cossets', 'cosseting', 'cosseted', 'cosseted', tran, [] ).
verb( 'cost', 'costs', 'costing', 'costed', 'costed', _, ['2B','6A'] ).
verb( 'cotton', 'cottons', 'cottoning', 'cottoned', 'cottoned', intran, [] ).
verb( 'couch', 'couches', 'couching', 'couched', 'couched', _, ['2A','6A','14'] ).
verb( 'cough', 'coughs', 'coughing', 'coughed', 'coughed', _, ['2A','15B'] ).
verb( 'counsel', 'counsels', 'counselling', 'counselled', 'counselled', tran, ['6A','6B','17'] ).
verb( 'count', 'counts', 'counting', 'counted', 'counted', _, ['2A','2C','3A','6A','14','16B','25'] ).
verb( 'countenance', 'countenances', 'countenancing', 'countenanced', 'countenanced', tran, ['6A'] ).
verb( 'counter', 'counters', 'countering', 'countered', 'countered', _, ['2A','3A','6A','14'] ).
verb( 'counteract', 'counteracts', 'counteracting', 'counteracted', 'counteracted', tran, ['6A'] ).
verb( 'counterattack', 'counterattacks', 'counterattacking', 'counterattacked', 'counterattacked', _, [] ).
verb( 'counterbalance', 'counterbalances', 'counterbalancing', 'counterbalanced', 'counterbalanced', tran, ['6A'] ).
verb( 'counterfeit', 'counterfeits', 'counterfeiting', 'counterfeited', 'counterfeited', tran, ['6A'] ).
verb( 'countermand', 'countermands', 'countermanding', 'countermanded', 'countermanded', tran, ['6A'] ).
verb( 'countermine', 'countermines', 'countermining', 'countermined', 'countermined', _, [] ).
verb( 'counterplot', 'counterplots', 'counterplotting', 'counterplotted', 'counterplotted', _, [] ).
verb( 'counterpoise', 'counterpoises', 'counterpoising', 'counterpoised', 'counterpoised', tran, [] ).
verb( 'countersign', 'countersigns', 'countersigning', 'countersigned', 'countersigned', tran, ['6A'] ).
verb( 'countersink', 'countersinks', 'countersinking', 'countersunk', 'countersunk', tran, [] ).
verb( 'countervail', 'countervails', 'countervailing', 'countervailed', 'countervailed', _, ['2A','6A'] ).
verb( 'couple', 'couples', 'coupling', 'coupled', 'coupled', _, ['2A','6A','14'] ).
verb( 'course', 'courses', 'coursing', 'coursed', 'coursed', _, ['2A','2C','6A'] ).
verb( 'court', 'courts', 'courting', 'courted', 'courted', _, ['2A','6A'] ).
verb( 'court-martial', 'court-martials', 'court-martialing', 'court-martialed', 'court-martialed', tran, ['6A'] ).
verb( 'covenant', 'covenants', 'covenanting', 'covenanted', 'covenanted', _, ['3A','6A','7A','9','14'] ).
verb( 'cover', 'covers', 'covering', 'covered', 'covered', tran, ['6A','15A','15B'] ).
verb( 'covet', 'covets', 'coveting', 'coveted', 'coveted', tran, ['6A'] ).
verb( 'cow', 'cows', 'cowing', 'cowed', 'cowed', tran, ['6A'] ).
verb( 'cower', 'cowers', 'cowering', 'cowered', 'cowered', intran, ['2A','2C'] ).
verb( 'cox', 'coxes', 'coxing', 'coxed', 'coxed', _, ['2A','6A'] ).
verb( 'cozen', 'cozens', 'cozening', 'cozened', 'cozened', tran, ['6A','14'] ).
verb( 'crab', 'crabs', 'crabbing', 'crabbed', 'crabbed', _, [] ).
verb( 'crack', 'cracks', 'cracking', 'cracked', 'cracked', _, ['2A','2C','6A','14'] ).
verb( 'crackle', 'crackles', 'crackling', 'crackled', 'crackled', intran, ['2A','2C'] ).
verb( 'cradle', 'cradles', 'cradling', 'cradled', 'cradled', tran, ['6A','14'] ).
verb( 'cram', 'crams', 'cramming', 'crammed', 'crammed', _, ['2A','6A','14','15B'] ).
verb( 'cramp', 'cramps', 'cramping', 'cramped', 'cramped', tran, ['6A'] ).
verb( 'crane', 'cranes', 'craning', 'craned', 'craned', _, ['2A','2C','6A','16A'] ).
verb( 'crank', 'cranks', 'cranking', 'cranked', 'cranked', tran, ['6A','15B'] ).
verb( 'crap', 'craps', 'crapping', 'crapped', 'crapped', intran, [] ).
verb( 'crash', 'crashes', 'crashing', 'crashed', 'crashed', _, ['2A','2C','6A'] ).
verb( 'crash-dive', 'crash-dives', 'crash-diving', 'crash-dived', 'crash-dived', intran, [] ).
verb( 'crash-land', 'crash-lands', 'crash-landing', 'crash-landed', 'crash-landed', _, [] ).
verb( 'crate', 'crates', 'crating', 'crated', 'crated', tran, [] ).
verb( 'crave', 'craves', 'craving', 'craved', 'craved', _, ['2A','3A'] ).
verb( 'crawl', 'crawls', 'crawling', 'crawled', 'crawled', intran, ['2A','2C'] ).
verb( 'crayon', 'crayons', 'crayoning', 'crayoned', 'crayoned', tran, [] ).
verb( 'creak', 'creaks', 'creaking', 'creaked', 'creaked', intran, [] ).
verb( 'cream', 'creams', 'creaming', 'creamed', 'creamed', tran, [] ).
verb( 'crease', 'creases', 'creasing', 'creased', 'creased', _, ['2A','6A'] ).
verb( 'create', 'creates', 'creating', 'created', 'created', tran, ['6A','23'] ).
verb( 'credit', 'credits', 'crediting', 'credited', 'credited', tran, ['6A','14'] ).
verb( 'creep', 'creeps', 'creeping', 'crept', 'crept', intran, ['2A','2B','2C'] ).
verb( 'cremate', 'cremates', 'cremating', 'cremated', 'cremated', tran, ['6A'] ).
verb( 'crepitate', 'crepitates', 'crepitating', 'crepitated', 'crepitated', intran, ['2A'] ).
verb( 'crest', 'crests', 'cresting', 'crested', 'crested', _, [] ).
verb( 'crew', 'crews', 'crewing', 'crewed', 'crewed', intran, ['3A','6A'] ).
verb( 'crib', 'cribs', 'cribbing', 'cribbed', 'cribbed', _, [] ).
verb( 'crick', 'cricks', 'cricking', 'cricked', 'cricked', tran, [] ).
verb( 'crime', 'crimes', 'criming', 'crimed', 'crimed', tran, [] ).
verb( 'crimp', 'crimps', 'crimping', 'crimped', 'crimped', tran, [] ).
verb( 'crimson', 'crimsons', 'crimsoning', 'crimsoned', 'crimsoned', _, [] ).
verb( 'cringe', 'cringes', 'cringing', 'cringed', 'cringed', intran, ['2A','2C'] ).
verb( 'crinkle', 'crinkles', 'crinkling', 'crinkled', 'crinkled', _, ['2A','2C','6A','15B'] ).
verb( 'cripple', 'cripples', 'crippling', 'crippled', 'crippled', tran, ['6A'] ).
verb( 'crisp', 'crisps', 'crisping', 'crisped', 'crisped', _, [] ).
verb( 'crisscross', 'crisscrosses', 'crisscrossing', 'crisscrossed', 'crisscrossed', _, [] ).
verb( 'criticize', 'criticizes', 'criticizing', 'criticized', 'criticized', _, ['2A','6A','14'] ).
verb( 'croak', 'croaks', 'croaking', 'croaked', 'croaked', _, ['2A','6A','15B'] ).
verb( 'crochet', 'crochets', 'crocheting', 'crocheted', 'crocheted', _, ['2A','6A'] ).
verb( 'crock', 'crocks', 'crocking', 'crocked', 'crocked', _, ['2C','15B'] ).
verb( 'crook', 'crooks', 'crooking', 'crooked', 'crooked', _, ['2A','6A'] ).
verb( 'croon', 'croons', 'crooning', 'crooned', 'crooned', _, ['2C','6A','13A','15A'] ).
verb( 'crop', 'crops', 'cropping', 'cropped', 'cropped', _, ['2A','2C','6A','14','22'] ).
verb( 'cross', 'crosses', 'crossing', 'crossed', 'crossed', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'cross-examine', 'cross-examines', 'cross-examining', 'cross-examined', 'cross-examined', tran, ['6A'] ).
verb( 'cross-fertilize', 'cross-fertilizes', 'cross-fertilizing', 'cross-fertilized', 'cross-fertilized', tran, [] ).
verb( 'cross-index', 'cross-indexes', 'cross-indexing', 'cross-indexed', 'cross-indexed', tran, [] ).
verb( 'cross-question', 'cross-questions', 'cross-questioning', 'cross-questioned', 'cross-questioned', tran, ['6A'] ).
verb( 'crossbreed', 'crossbreeds', 'crossbreeding', 'crossbred', 'crossbred', tran, ['2A','6A'] ).
verb( 'crosscheck', 'crosschecks', 'crosschecking', 'crosschecked', 'crosschecked', _, ['2A','6A'] ).
verb( 'crouch', 'crouches', 'crouching', 'crouched', 'crouched', intran, ['2A','2C','4A'] ).
verb( 'crow', 'crows', 'crowing', 'crowed', 'crowed', intran, ['3A','6A'] ).
verb( 'crowd', 'crowds', 'crowding', 'crowded', 'crowded', _, ['2B','2C','6A','14','15A','15B'] ).
verb( 'crown', 'crowns', 'crowning', 'crowned', 'crowned', tran, ['6A','14','23'] ).
verb( 'crucify', 'crucifies', 'crucifying', 'crucified', 'crucified', tran, [] ).
verb( 'cruise', 'cruises', 'cruising', 'cruised', 'cruised', intran, ['2A','2C'] ).
verb( 'crumble', 'crumbles', 'crumbling', 'crumbled', 'crumbled', _, ['2A','2C','6A'] ).
verb( 'crumple', 'crumples', 'crumpling', 'crumpled', 'crumpled', _, ['2A','2C','6A','15B'] ).
verb( 'crunch', 'crunches', 'crunching', 'crunched', 'crunched', _, ['2A','2C','6A'] ).
verb( 'crusade', 'crusades', 'crusading', 'crusaded', 'crusaded', intran, ['2A','3A'] ).
verb( 'crush', 'crushes', 'crushing', 'crushed', 'crushed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'crust', 'crusts', 'crusting', 'crusted', 'crusted', _, ['2A','2C','6A'] ).
verb( 'cry', 'cries', 'crying', 'cried', 'cried', _, ['2A','2B','2C','3A','4A','6A','9','14','15A','15B'] ).
verb( 'crystallize', 'crystallizes', 'crystallizing', 'crystallized', 'crystallized', _, ['2A','6A'] ).
verb( 'cube', 'cubes', 'cubing', 'cubed', 'cubed', tran, ['6A'] ).
verb( 'cuckold', 'cuckolds', 'cuckolding', 'cuckolded', 'cuckolded', tran, ['6A'] ).
verb( 'cuddle', 'cuddles', 'cuddling', 'cuddled', 'cuddled', _, ['2C','6A','15B'] ).
verb( 'cudgel', 'cudgels', 'cudgeling', 'cudgeled', 'cudgeled', tran, [] ).
verb( 'cuff', 'cuffs', 'cuffing', 'cuffed', 'cuffed', tran, [] ).
verb( 'cull', 'culls', 'culling', 'culled', 'culled', tran, ['6A'] ).
verb( 'culminate', 'culminates', 'culminating', 'culminated', 'culminated', tran, ['3A'] ).
verb( 'cultivate', 'cultivates', 'cultivating', 'cultivated', 'cultivated', tran, ['6A'] ).
verb( 'cumber', 'cumbers', 'cumbering', 'cumbered', 'cumbered', tran, ['6A','14'] ).
verb( 'cup', 'cups', 'cupping', 'cupped', 'cupped', tran, ['6A'] ).
verb( 'curb', 'curbs', 'curbing', 'curbed', 'curbed', tran, ['6A'] ).
verb( 'curdle', 'curdles', 'curdling', 'curdled', 'curdled', _, ['2A','6A'] ).
verb( 'cure', 'cures', 'curing', 'cured', 'cured', _, ['6A','14'] ).
verb( 'curl', 'curls', 'curling', 'curled', 'curled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'curry', 'curries', 'currying', 'curried', 'curried', tran, ['6A'] ).
verb( 'curse', 'curses', 'cursing', 'cursed', 'cursed', _, ['2A','3A','6A'] ).
verb( 'curtail', 'curtails', 'curtailing', 'curtailed', 'curtailed', tran, ['6A'] ).
verb( 'curtain', 'curtains', 'curtaining', 'curtained', 'curtained', tran, ['6A','15B'] ).
verb( 'curtsey', 'curtseys', 'curtseying', 'curtseyed', 'curtseyed', intran, [] ).
verb( 'curtsy', 'curtsies', 'curtsying', 'curtsied', 'curtsied', intran, [] ).
verb( 'curve', 'curves', 'curving', 'curved', 'curved', _, ['2A','6A'] ).
verb( 'cushion', 'cushions', 'cushioning', 'cushioned', 'cushioned', tran, ['6A','14'] ).
verb( 'cut', 'cuts', 'cutting', 'cut', 'cut', _, ['2A','2C','3A','6A','12B','13B','15A','15B','22'] ).
verb( 'cycle', 'cycles', 'cycling', 'cycled', 'cycled', intran, ['2A','2B','2C'] ).
verb( 'cyclostyle', 'cyclostyles', 'cyclostyling', 'cyclostyled', 'cyclostyled', tran, [] ).
verb( 'cypher', 'cyphers', 'cyphering', 'cyphered', 'cyphered', _, ['2A','6A'] ).
verb( 'dab', 'dabs', 'dabbing', 'dabbed', 'dabbed', _, ['2C','3A','6A','14','15A','15B'] ).
verb( 'dabble', 'dabbles', 'dabbling', 'dabbled', 'dabbled', _, ['2C','3A','6A','15A','15B'] ).
verb( 'dally', 'dallies', 'dallying', 'dallied', 'dallied', intran, ['2A','3A'] ).
verb( 'dam', 'dams', 'damming', 'dammed', 'dammed', tran, ['6A','15B'] ).
verb( 'damage', 'damages', 'damaging', 'damaged', 'damaged', tran, ['6A'] ).
verb( 'damascene', 'damascenes', 'damascening', 'damascened', 'damascened', tran, [] ).
verb( 'damn', 'damns', 'damning', 'damned', 'damned', tran, ['6A'] ).
verb( 'damp', 'damps', 'damping', 'damped', 'damped', _, ['2C','6A','15B'] ).
verb( 'dampen', 'dampens', 'dampening', 'dampened', 'dampened', _, [] ).
verb( 'dance', 'dances', 'dancing', 'danced', 'danced', _, ['2A','2C','6A','15A'] ).
verb( 'dandle', 'dandles', 'dandling', 'dandled', 'dandled', tran, ['6A','15A','15B'] ).
verb( 'dangle', 'dangles', 'dangling', 'dangled', 'dangled', _, ['2A','2C','6A','15A'] ).
verb( 'dapple', 'dapples', 'dappling', 'dappled', 'dappled', tran, ['6A'] ).
verb( 'dare', 'dares', 'daring', 'dared', 'dared', unknown, ['4A','5','6A','17'] ).
verb( 'dare', 'dares', 'daring', 'dared', 'dared', _, ['4A','5','6A','17'] ).
verb( 'darken', 'darkens', 'darkening', 'darkened', 'darkened', _, ['2A','6A'] ).
verb( 'darn', 'darns', 'darning', 'darned', 'darned', _, ['2A','6A'] ).
verb( 'dart', 'darts', 'darting', 'darted', 'darted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'dash', 'dashes', 'dashing', 'dashed', 'dashed', _, ['2C','6A','15A','15B'] ).
verb( 'date', 'dates', 'dating', 'dated', 'dated', _, ['2A','2C','3A','6A'] ).
verb( 'daub', 'daubs', 'daubing', 'daubed', 'daubed', _, ['2A','6A','14','15A','15B'] ).
verb( 'daunt', 'daunts', 'daunting', 'daunted', 'daunted', tran, ['6A'] ).
verb( 'dawdle', 'dawdles', 'dawdling', 'dawdled', 'dawdled', _, ['2A','2C','15B'] ).
verb( 'dawn', 'dawns', 'dawning', 'dawned', 'dawned', intran, ['2A','3A'] ).
verb( 'daydream', 'daydreams', 'daydreaming', 'daydreamed', 'daydreamed', intran, [] ).
verb( 'daze', 'dazes', 'dazing', 'dazed', 'dazed', tran, ['6A'] ).
verb( 'dazzle', 'dazzles', 'dazzling', 'dazzled', 'dazzled', tran, ['6A'] ).
verb( 'de-escalate', 'de-escalates', 'de-escalating', 'de-escalated', 'de-escalated', tran, ['6A'] ).
verb( 'de-ice', 'de-ices', 'de-icing', 'de-iced', 'de-iced', tran, ['6A'] ).
verb( 'deaden', 'deadens', 'deadening', 'deadened', 'deadened', tran, ['6A'] ).
verb( 'deafen', 'deafens', 'deafening', 'deafened', 'deafened', tran, ['6A'] ).
verb( 'deal', 'deals', 'dealing', 'dealt', 'dealt', _, ['2A','3A','6A','12A','13A','15B'] ).
verb( 'debag', 'debags', 'debagging', 'debagged', 'debagged', tran, ['6A'] ).
verb( 'debar', 'debars', 'debarring', 'debarred', 'debarred', tran, ['14'] ).
verb( 'debark', 'debarks', 'debarking', 'debarked', 'debarked', _, [] ).
verb( 'debase', 'debases', 'debasing', 'debased', 'debased', tran, ['6A'] ).
verb( 'debate', 'debates', 'debating', 'debated', 'debated', _, ['2A','2C','6A','8','10'] ).
verb( 'debauch', 'debauches', 'debauching', 'debauched', 'debauched', tran, ['6A'] ).
verb( 'debilitate', 'debilitates', 'debilitating', 'debilitated', 'debilitated', tran, ['6A'] ).
verb( 'debit', 'debits', 'debiting', 'debited', 'debited', tran, ['6A','14'] ).
verb( 'debouch', 'debouches', 'debouching', 'debouched', 'debouched', _, ['2A','6A'] ).
verb( 'debrief', 'debriefs', 'debriefing', 'debriefed', 'debriefed', tran, ['6A'] ).
verb( 'debug', 'debugs', 'debugging', 'debugged', 'debugged', tran, ['6A'] ).
verb( 'debunk', 'debunks', 'debunking', 'debunked', 'debunked', tran, ['6A'] ).
verb( 'decamp', 'decamps', 'decamping', 'decamped', 'decamped', intran, ['2A','3A'] ).
verb( 'decant', 'decants', 'decanting', 'decanted', 'decanted', tran, ['6A'] ).
verb( 'decapitate', 'decapitates', 'decapitating', 'decapitated', 'decapitated', tran, ['6A'] ).
verb( 'decarbonize', 'decarbonizes', 'decarbonizing', 'decarbonized', 'decarbonized', tran, [] ).
verb( 'decay', 'decays', 'decaying', 'decayed', 'decayed', intran, ['2A'] ).
verb( 'decease', 'deceases', 'deceasing', 'deceased', 'deceased', intran, [] ).
verb( 'deceive', 'deceives', 'deceiving', 'deceived', 'deceived', tran, ['6A','14'] ).
verb( 'decelerate', 'decelerates', 'decelerating', 'decelerated', 'decelerated', _, [] ).
verb( 'decentralize', 'decentralizes', 'decentralizing', 'decentralized', 'decentralized', tran, ['6A'] ).
verb( 'decide', 'decides', 'deciding', 'decided', 'decided', _, ['2A','3A','6A','7A','8','9','10','14','17'] ).
verb( 'decimalize', 'decimalizes', 'decimalizing', 'decimalized', 'decimalized', tran, ['6A'] ).
verb( 'decimate', 'decimates', 'decimating', 'decimated', 'decimated', tran, ['6A'] ).
verb( 'decipher', 'deciphers', 'deciphering', 'deciphered', 'deciphered', tran, ['6A'] ).
verb( 'deck', 'decks', 'decking', 'decked', 'decked', tran, ['6A','14','15A'] ).
verb( 'declaim', 'declaims', 'declaiming', 'declaimed', 'declaimed', _, ['2A','3A','6A'] ).
verb( 'declare', 'declares', 'declaring', 'declared', 'declared', _, ['3A','6A','9','14','25'] ).
verb( 'declassify', 'declassifies', 'declassifying', 'declassified', 'declassified', tran, ['6A'] ).
verb( 'decline', 'declines', 'declining', 'declined', 'declined', _, ['2A','2C','6A','7A'] ).
verb( 'declutch', 'declutches', 'declutching', 'declutched', 'declutched', intran, ['2A'] ).
verb( 'decode', 'decodes', 'decoding', 'decoded', 'decoded', tran, ['6A'] ).
verb( 'decoke', 'decokes', 'decoking', 'decoked', 'decoked', tran, [] ).
verb( 'decolonize', 'decolonizes', 'decolonizing', 'decolonized', 'decolonized', tran, ['6A'] ).
verb( 'decompose', 'decomposes', 'decomposing', 'decomposed', 'decomposed', _, ['2A','6A'] ).
verb( 'decompress', 'decompresses', 'decompressing', 'decompressed', 'decompressed', tran, ['6A'] ).
verb( 'decontaminate', 'decontaminates', 'decontaminating', 'decontaminated', 'decontaminated', tran, ['6A'] ).
verb( 'decontrol', 'decontrols', 'decontrolling', 'decontrolled', 'decontrolled', tran, ['6A'] ).
verb( 'decorate', 'decorates', 'decorating', 'decorated', 'decorated', tran, ['6A','14'] ).
verb( 'decoy', 'decoys', 'decoying', 'decoyed', 'decoyed', tran, ['6A','14'] ).
verb( 'decrease', 'decreases', 'decreasing', 'decreased', 'decreased', _, ['2A','6A'] ).
verb( 'decree', 'decrees', 'decreeing', 'decreed', 'decreed', tran, ['6A','9'] ).
verb( 'decry', 'decries', 'decrying', 'decried', 'decried', tran, ['6A'] ).
verb( 'dedicate', 'dedicates', 'dedicating', 'dedicated', 'dedicated', tran, ['6A','14'] ).
verb( 'deduce', 'deduces', 'deducing', 'deduced', 'deduced', tran, ['6A','9','14'] ).
verb( 'deduct', 'deducts', 'deducting', 'deducted', 'deducted', tran, ['6A','14'] ).
verb( 'deem', 'deems', 'deeming', 'deemed', 'deemed', tran, ['9','25'] ).
verb( 'deep-freeze', 'deep-freezes', 'deep-freezing', 'deep-froze', 'deep-frozen', tran, [] ).
verb( 'deepen', 'deepens', 'deepening', 'deepened', 'deepened', _, [] ).
verb( 'deface', 'defaces', 'defacing', 'defaced', 'defaced', tran, ['6A'] ).
verb( 'defame', 'defames', 'defaming', 'defamed', 'defamed', tran, ['6A'] ).
verb( 'default', 'defaults', 'defaulting', 'defaulted', 'defaulted', intran, ['2A'] ).
verb( 'defeat', 'defeats', 'defeating', 'defeated', 'defeated', tran, ['6A'] ).
verb( 'defecate', 'defecates', 'defecating', 'defecated', 'defecated', intran, ['2A'] ).
verb( 'defect', 'defects', 'defecting', 'defected', 'defected', intran, ['2A','2C','3A'] ).
verb( 'defend', 'defends', 'defending', 'defended', 'defended', tran, ['6A','14'] ).
verb( 'defer', 'defers', 'deferring', 'deferred', 'deferred', _, ['3A','6A','6C'] ).
verb( 'defile', 'defiles', 'defiling', 'defiled', 'defiled', _, ['6A'] ).
verb( 'define', 'defines', 'defining', 'defined', 'defined', tran, ['6A'] ).
verb( 'deflate', 'deflates', 'deflating', 'deflated', 'deflated', tran, ['6A'] ).
verb( 'deflect', 'deflects', 'deflecting', 'deflected', 'deflected', _, ['2A','3A','6A','14'] ).
verb( 'deflower', 'deflowers', 'deflowering', 'deflowered', 'deflowered', tran, [] ).
verb( 'defoliate', 'defoliates', 'defoliating', 'defoliated', 'defoliated', tran, ['6A'] ).
verb( 'deforest', 'deforests', 'deforesting', 'deforested', 'deforested', tran, [] ).
verb( 'deform', 'deforms', 'deforming', 'deformed', 'deformed', tran, ['6A'] ).
verb( 'defraud', 'defrauds', 'defrauding', 'defrauded', 'defrauded', tran, ['6A','14'] ).
verb( 'defray', 'defrays', 'defraying', 'defrayed', 'defrayed', tran, ['6A'] ).
verb( 'defrock', 'defrocks', 'defrocking', 'defrocked', 'defrocked', tran, [] ).
verb( 'defrost', 'defrosts', 'defrosting', 'defrosted', 'defrosted', tran, ['6A'] ).
verb( 'defuse', 'defuses', 'defusing', 'defused', 'defused', tran, ['6A'] ).
verb( 'defy', 'defies', 'defying', 'defied', 'defied', tran, ['6A','17'] ).
verb( 'degauss', 'degausses', 'degaussing', 'degaussed', 'degaussed', tran, ['6A'] ).
verb( 'degenerate', 'degenerates', 'degenerating', 'degenerated', 'degenerated', intran, ['2A','3A'] ).
verb( 'degrade', 'degrades', 'degrading', 'degraded', 'degraded', tran, ['6A'] ).
verb( 'dehorn', 'dehorns', 'dehorning', 'dehorned', 'dehorned', tran, ['6A'] ).
verb( 'dehumanize', 'dehumanizes', 'dehumanizing', 'dehumanized', 'dehumanized', tran, ['6A'] ).
verb( 'dehydrate', 'dehydrates', 'dehydrating', 'dehydrated', 'dehydrated', tran, ['6A'] ).
verb( 'deify', 'deifies', 'deifying', 'deified', 'deified', tran, ['6A'] ).
verb( 'deign', 'deigns', 'deigning', 'deigned', 'deigned', intran, ['4A'] ).
verb( 'deject', 'dejects', 'dejecting', 'dejected', 'dejected', tran, [] ).
verb( 'delay', 'delays', 'delaying', 'delayed', 'delayed', _, ['2A','2B','6A','6C'] ).
verb( 'delegate', 'delegates', 'delegating', 'delegated', 'delegated', tran, ['14','17'] ).
verb( 'delete', 'deletes', 'deleting', 'deleted', 'deleted', tran, ['6A','14'] ).
verb( 'deliberate', 'deliberates', 'deliberating', 'deliberated', 'deliberated', _, ['2A','3A','6A','8','10'] ).
verb( 'delight', 'delights', 'delighting', 'delighted', 'delighted', _, ['3A','4C','6A'] ).
verb( 'delimit', 'delimits', 'delimiting', 'delimited', 'delimited', tran, ['6A'] ).
verb( 'delimitate', 'delimitates', 'delimitating', 'delimitated', 'delimitated', tran, ['6A'] ).
verb( 'delineate', 'delineates', 'delineating', 'delineated', 'delineated', tran, ['6A'] ).
verb( 'deliver', 'delivers', 'delivering', 'delivered', 'delivered', tran, ['6A','14','15B'] ).
verb( 'delouse', 'delouses', 'delousing', 'deloused', 'deloused', tran, [] ).
verb( 'delude', 'deludes', 'deluding', 'deluded', 'deluded', tran, ['6A','14'] ).
verb( 'deluge', 'deluges', 'deluging', 'deluged', 'deluged', tran, ['6A','14'] ).
verb( 'delve', 'delves', 'delving', 'delved', 'delved', _, ['2A','3A','6A'] ).
verb( 'demagnetize', 'demagnetizes', 'demagnetizing', 'demagnetized', 'demagnetized', tran, ['6A'] ).
verb( 'demand', 'demands', 'demanding', 'demanded', 'demanded', tran, ['6A','7A','9'] ).
verb( 'demarcate', 'demarcates', 'demarcating', 'demarcated', 'demarcated', tran, ['6A'] ).
verb( 'demean', 'demeans', 'demeaning', 'demeaned', 'demeaned', tran, ['6A'] ).
verb( 'demilitarize', 'demilitarizes', 'demilitarizing', 'demilitarized', 'demilitarized', tran, ['6A'] ).
verb( 'demist', 'demists', 'demisting', 'demisted', 'demisted', tran, [] ).
verb( 'demob', 'demobs', 'demobbing', 'demobbed', 'demobbed', tran, [] ).
verb( 'demobilize', 'demobilizes', 'demobilizing', 'demobilized', 'demobilized', tran, ['6A'] ).
verb( 'democratize', 'democratizes', 'democratizing', 'democratized', 'democratized', tran, ['6A'] ).
verb( 'demolish', 'demolishes', 'demolishing', 'demolished', 'demolished', tran, ['6A'] ).
verb( 'demonetize', 'demonetizes', 'demonetizing', 'demonetized', 'demonetized', tran, [] ).
verb( 'demonstrate', 'demonstrates', 'demonstrating', 'demonstrated', 'demonstrated', _, ['2A','3A','6A','9'] ).
verb( 'demoralize', 'demoralizes', 'demoralizing', 'demoralized', 'demoralized', tran, ['6A'] ).
verb( 'demote', 'demotes', 'demoting', 'demoted', 'demoted', tran, ['6A'] ).
verb( 'demur', 'demurs', 'demurring', 'demurred', 'demurred', intran, ['2A','3A'] ).
verb( 'denationalize', 'denationalizes', 'denationalizing', 'denationalized', 'denationalized', tran, ['6A'] ).
verb( 'denigrate', 'denigrates', 'denigrating', 'denigrated', 'denigrated', tran, ['6A'] ).
verb( 'denominate', 'denominates', 'denominating', 'denominated', 'denominated', tran, ['23'] ).
verb( 'denote', 'denotes', 'denoting', 'denoted', 'denoted', tran, ['6A','9'] ).
verb( 'denounce', 'denounces', 'denouncing', 'denounced', 'denounced', tran, ['6A','16B'] ).
verb( 'dent', 'dents', 'denting', 'dented', 'dented', _, ['2A','6A'] ).
verb( 'denude', 'denudes', 'denuding', 'denuded', 'denuded', tran, ['6A','14'] ).
verb( 'deny', 'denies', 'denying', 'denied', 'denied', tran, ['6A','6C','9','12A','13A','25'] ).
verb( 'deodorize', 'deodorizes', 'deodorizing', 'deodorized', 'deodorized', tran, ['6A'] ).
verb( 'depart', 'departs', 'departing', 'departed', 'departed', intran, ['2A','3A'] ).
verb( 'depend', 'depends', 'depending', 'depended', 'depended', intran, ['3A'] ).
verb( 'depict', 'depicts', 'depicting', 'depicted', 'depicted', tran, ['6A'] ).
verb( 'deplane', 'deplanes', 'deplaning', 'deplaned', 'deplaned', intran, [] ).
verb( 'deplete', 'depletes', 'depleting', 'depleted', 'depleted', tran, ['6A','14'] ).
verb( 'deplore', 'deplores', 'deploring', 'deplored', 'deplored', tran, ['6A'] ).
verb( 'deploy', 'deploys', 'deploying', 'deployed', 'deployed', _, ['2A','6A'] ).
verb( 'depopulate', 'depopulates', 'depopulating', 'depopulated', 'depopulated', tran, ['6A'] ).
verb( 'deport', 'deports', 'deporting', 'deported', 'deported', tran, ['6A','16B'] ).
verb( 'depose', 'deposes', 'deposing', 'deposed', 'deposed', _, ['3A','6A','9'] ).
verb( 'deposit', 'deposits', 'depositing', 'deposited', 'deposited', tran, ['6A'] ).
verb( 'deprave', 'depraves', 'depraving', 'depraved', 'depraved', tran, ['6A'] ).
verb( 'deprecate', 'deprecates', 'deprecating', 'deprecated', 'deprecated', tran, ['6A','6C'] ).
verb( 'depreciate', 'depreciates', 'depreciating', 'depreciated', 'depreciated', _, ['2A','6A'] ).
verb( 'depress', 'depresses', 'depressing', 'depressed', 'depressed', tran, ['6A'] ).
verb( 'deprive', 'deprives', 'depriving', 'deprived', 'deprived', tran, ['14'] ).
verb( 'depute', 'deputes', 'deputing', 'deputed', 'deputed', tran, ['14','17'] ).
verb( 'deputize', 'deputizes', 'deputizing', 'deputized', 'deputized', intran, ['2A','3A'] ).
verb( 'derail', 'derails', 'derailing', 'derailed', 'derailed', tran, ['6A'] ).
verb( 'derange', 'deranges', 'deranging', 'deranged', 'deranged', tran, ['6A'] ).
verb( 'derate', 'derates', 'derating', 'derated', 'derated', tran, ['6A'] ).
verb( 'derequisition', 'derequisitions', 'derequisitioning', 'derequisitioned', 'derequisitioned', tran, ['6A'] ).
verb( 'derestrict', 'derestricts', 'derestricting', 'derestricted', 'derestricted', tran, ['6A'] ).
verb( 'deride', 'derides', 'deriding', 'derided', 'derided', tran, ['6A','16B'] ).
verb( 'derive', 'derives', 'deriving', 'derived', 'derived', _, ['3A','14'] ).
verb( 'derogate', 'derogates', 'derogating', 'derogated', 'derogated', intran, ['3A'] ).
verb( 'desalinate', 'desalinates', 'desalinating', 'desalinated', 'desalinated', tran, ['6A'] ).
verb( 'desalinize', 'desalinizes', 'desalinizing', 'desalinized', 'desalinized', tran, ['6A'] ).
verb( 'desalt', 'desalts', 'desalting', 'desalted', 'desalted', tran, [] ).
verb( 'descale', 'descales', 'descaling', 'descaled', 'descaled', tran, ['6A'] ).
verb( 'descant', 'descants', 'descanting', 'descanted', 'descanted', intran, ['3A'] ).
verb( 'descend', 'descends', 'descending', 'descended', 'descended', _, ['2A','2C','3A','6A'] ).
verb( 'describe', 'describes', 'describing', 'described', 'described', tran, ['6A','10','14','16B'] ).
verb( 'descry', 'descries', 'descrying', 'descried', 'descried', tran, ['6A'] ).
verb( 'desecrate', 'desecrates', 'desecrating', 'desecrated', 'desecrated', tran, ['6A'] ).
verb( 'desegregate', 'desegregates', 'desegregating', 'desegregated', 'desegregated', tran, ['6A'] ).
verb( 'desensitize', 'desensitizes', 'desensitizing', 'desensitized', 'desensitized', tran, ['6A'] ).
verb( 'desert', 'deserts', 'deserting', 'deserted', 'deserted', _, ['2A','6A'] ).
verb( 'deserve', 'deserves', 'deserving', 'deserved', 'deserved', _, ['6A','7A'] ).
verb( 'desiccate', 'desiccates', 'desiccating', 'desiccated', 'desiccated', tran, ['6A'] ).
verb( 'design', 'designs', 'designing', 'designed', 'designed', _, ['2A','2C','6A','14','16A','16B'] ).
verb( 'designate', 'designates', 'designating', 'designated', 'designated', tran, ['6A','16B','17'] ).
verb( 'desire', 'desires', 'desiring', 'desired', 'desired', tran, ['6A','7A','9','17'] ).
verb( 'desist', 'desists', 'desisting', 'desisted', 'desisted', intran, ['2A','3A'] ).
verb( 'desolate', 'desolates', 'desolating', 'desolated', 'desolated', tran, ['6A'] ).
verb( 'despair', 'despairs', 'despairing', 'despaired', 'despaired', intran, ['2A','3A'] ).
verb( 'despatch', 'despatches', 'despatching', 'despatched', 'despatched', tran, ['6A','14'] ).
verb( 'despise', 'despises', 'despising', 'despised', 'despised', tran, ['6A'] ).
verb( 'despoil', 'despoils', 'despoiling', 'despoiled', 'despoiled', tran, ['6A','14'] ).
verb( 'destine', 'destines', 'destining', 'destined', 'destined', tran, ['14','17'] ).
verb( 'destroy', 'destroys', 'destroying', 'destroyed', 'destroyed', tran, ['6A'] ).
verb( 'detach', 'detaches', 'detaching', 'detached', 'detached', tran, ['6A','14','16A'] ).
verb( 'detail', 'details', 'detailing', 'detailed', 'detailed', tran, ['6A','14','16A'] ).
verb( 'detain', 'detains', 'detaining', 'detained', 'detained', tran, ['6A','16A'] ).
verb( 'detect', 'detects', 'detecting', 'detected', 'detected', tran, ['6A'] ).
verb( 'deter', 'deters', 'deterring', 'deterred', 'deterred', tran, ['6A','14'] ).
verb( 'deteriorate', 'deteriorates', 'deteriorating', 'deteriorated', 'deteriorated', _, ['2A','6A'] ).
verb( 'determine', 'determines', 'determining', 'determined', 'determined', _, ['3A','6A','7A','8','9','10','14','17'] ).
verb( 'detest', 'detests', 'detesting', 'detested', 'detested', tran, ['6A','6C'] ).
verb( 'dethrone', 'dethrones', 'dethroning', 'dethroned', 'dethroned', tran, ['6A'] ).
verb( 'detonate', 'detonates', 'detonating', 'detonated', 'detonated', _, ['2A','6A'] ).
verb( 'detour', 'detours', 'detouring', 'detoured', 'detoured', tran, ['6A'] ).
verb( 'detract', 'detracts', 'detracting', 'detracted', 'detracted', intran, ['3A'] ).
verb( 'detrain', 'detrains', 'detraining', 'detrained', 'detrained', _, ['2A','6A'] ).
verb( 'detribalize', 'detribalizes', 'detribalizing', 'detribalized', 'detribalized', tran, ['6A'] ).
verb( 'devaluate', 'devaluates', 'devaluating', 'devaluated', 'devaluated', tran, ['6A'] ).
verb( 'devalue', 'devalues', 'devaluing', 'devalued', 'devalued', tran, ['6A'] ).
verb( 'devastate', 'devastates', 'devastating', 'devastated', 'devastated', tran, ['6A'] ).
verb( 'develop', 'develops', 'developing', 'developed', 'developed', _, ['2A','2C','3A','6A'] ).
verb( 'deviate', 'deviates', 'deviating', 'deviated', 'deviated', intran, ['3A'] ).
verb( 'devil', 'devils', 'devilling', 'devilled', 'devilled', _, ['2A','3A','6A'] ).
verb( 'devise', 'devises', 'devising', 'devised', 'devised', tran, ['6A','8','14'] ).
verb( 'devitalize', 'devitalizes', 'devitalizing', 'devitalized', 'devitalized', tran, ['6A'] ).
verb( 'devolve', 'devolves', 'devolving', 'devolved', 'devolved', _, ['3A','6A','14'] ).
verb( 'devote', 'devotes', 'devoting', 'devoted', 'devoted', tran, ['14'] ).
verb( 'devour', 'devours', 'devouring', 'devoured', 'devoured', tran, ['6A'] ).
verb( 'diagnose', 'diagnoses', 'diagnosing', 'diagnosed', 'diagnosed', tran, ['6A','16B'] ).
verb( 'dial', 'dials', 'dialling', 'dialled', 'dialled', tran, ['6A'] ).
verb( 'dibble', 'dibbles', 'dibbling', 'dibbled', 'dibbled', tran, ['15B'] ).
verb( 'dice', 'dices', 'dicing', 'diced', 'diced', _, ['2A','6A'] ).
verb( 'dicker', 'dickers', 'dickering', 'dickered', 'dickered', intran, ['2A','3A'] ).
verb( 'dictate', 'dictates', 'dictating', 'dictated', 'dictated', _, ['2A','3A','6A','14'] ).
verb( 'diddle', 'diddles', 'diddling', 'diddled', 'diddled', tran, ['6A','14'] ).
verb( 'die', 'dies', 'dying', 'died', 'died', intran, ['2A','2C','2D','3A','4C'] ).
verb( 'diet', 'diets', 'dieting', 'dieted', 'dieted', _, ['2A','6A'] ).
verb( 'differ', 'differs', 'differing', 'differed', 'differed', intran, ['2A','2C','3A'] ).
verb( 'differentiate', 'differentiates', 'differentiating', 'differentiated', 'differentiated', tran, ['3A','6A','14'] ).
verb( 'diffract', 'diffracts', 'diffracting', 'diffracted', 'diffracted', tran, ['6A'] ).
verb( 'diffuse', 'diffuses', 'diffusing', 'diffused', 'diffused', _, ['2A','6A'] ).
verb( 'dig', 'digs', 'digging', 'dug', 'dug', _, ['2C','3A','6A','15B'] ).
verb( 'digest', 'digests', 'digesting', 'digested', 'digested', _, ['2A','6A'] ).
verb( 'dignify', 'dignifies', 'dignifying', 'dignified', 'dignified', tran, ['6A','14'] ).
verb( 'digress', 'digresses', 'digressing', 'digressed', 'digressed', intran, ['2A','3A'] ).
verb( 'dike', 'dikes', 'diking', 'diked', 'diked', _, [] ).
verb( 'dilate', 'dilates', 'dilating', 'dilated', 'dilated', _, ['2A','3A','6A'] ).
verb( 'dilly-dally', 'dilly-dallies', 'dilly-dallying', 'dilly-dallied', 'dilly-dallied', intran, ['2A'] ).
verb( 'dilute', 'dilutes', 'diluting', 'diluted', 'diluted', tran, ['6A','14'] ).
verb( 'dim', 'dims', 'dimming', 'dimmed', 'dimmed', _, ['2A','6A'] ).
verb( 'diminish', 'diminishes', 'diminishing', 'diminished', 'diminished', _, ['2A','6A'] ).
verb( 'dimple', 'dimples', 'dimpling', 'dimpled', 'dimpled', _, ['2A','6A'] ).
verb( 'din', 'dins', 'dinning', 'dinned', 'dinned', _, ['2C'] ).
verb( 'dine', 'dines', 'dining', 'dined', 'dined', _, ['2A','6A'] ).
verb( 'dip', 'dips', 'dipping', 'dipped', 'dipped', _, ['2A','2C','3A','6A','14'] ).
verb( 'direct', 'directs', 'directing', 'directed', 'directed', _, ['2A','6A','9','14','17'] ).
verb( 'dirty', 'dirties', 'dirtying', 'dirtied', 'dirtied', _, ['2A','6A'] ).
verb( 'disable', 'disables', 'disabling', 'disabled', 'disabled', tran, ['6A'] ).
verb( 'disabuse', 'disabuses', 'disabusing', 'disabused', 'disabused', tran, ['6A','14'] ).
verb( 'disafforest', 'disafforests', 'disafforesting', 'disafforested', 'disafforested', tran, [] ).
verb( 'disagree', 'disagrees', 'disagreeing', 'disagreed', 'disagreed', tran, ['2A','3A'] ).
verb( 'disallow', 'disallows', 'disallowing', 'disallowed', 'disallowed', tran, ['6A'] ).
verb( 'disappear', 'disappears', 'disappearing', 'disappeared', 'disappeared', intran, ['2A'] ).
verb( 'disappoint', 'disappoints', 'disappointing', 'disappointed', 'disappointed', tran, ['6A'] ).
verb( 'disapprove', 'disapproves', 'disapproving', 'disapproved', 'disapproved', _, ['2A','3A','6A'] ).
verb( 'disarm', 'disarms', 'disarming', 'disarmed', 'disarmed', _, ['2A','6A'] ).
verb( 'disarrange', 'disarranges', 'disarranging', 'disarranged', 'disarranged', tran, ['6A'] ).
verb( 'disarray', 'disarrays', 'disarraying', 'disarrayed', 'disarrayed', tran, [] ).
verb( 'disassociate', 'disassociates', 'disassociating', 'disassociated', 'disassociated', tran, ['14'] ).
verb( 'disavow', 'disavows', 'disavowing', 'disavowed', 'disavowed', tran, ['6A'] ).
verb( 'disband', 'disbands', 'disbanding', 'disbanded', 'disbanded', _, ['2A','6A'] ).
verb( 'disbelieve', 'disbelieves', 'disbelieving', 'disbelieved', 'disbelieved', _, ['2A','3A','6A'] ).
verb( 'disbud', 'disbuds', 'disbudding', 'disbudded', 'disbudded', tran, ['6A'] ).
verb( 'disburden', 'disburdens', 'disburdening', 'disburdened', 'disburdened', tran, ['6A','14'] ).
verb( 'disburse', 'disburses', 'disbursing', 'disbursed', 'disbursed', _, ['2A','6A'] ).
verb( 'discard', 'discards', 'discarding', 'discarded', 'discarded', tran, ['6A'] ).
verb( 'discern', 'discerns', 'discerning', 'discerned', 'discerned', tran, ['6A'] ).
verb( 'discharge', 'discharges', 'discharging', 'discharged', 'discharged', _, ['6A','14'] ).
verb( 'discipline', 'disciplines', 'disciplining', 'disciplined', 'disciplined', tran, ['6A'] ).
verb( 'disclaim', 'disclaims', 'disclaiming', 'disclaimed', 'disclaimed', tran, ['6A','6C'] ).
verb( 'disclose', 'discloses', 'disclosing', 'disclosed', 'disclosed', tran, ['6A','14'] ).
verb( 'discolour', 'discolours', 'discolouring', 'discoloured', 'discoloured', _, ['2A','6A'] ).
verb( 'discomfit', 'discomfits', 'discomfiting', 'discomfited', 'discomfited', tran, ['6A'] ).
verb( 'discommode', 'discommodes', 'discommoding', 'discommoded', 'discommoded', tran, ['6A'] ).
verb( 'discompose', 'discomposes', 'discomposing', 'discomposed', 'discomposed', tran, ['6A'] ).
verb( 'disconcert', 'disconcerts', 'disconcerting', 'disconcerted', 'disconcerted', tran, ['6A'] ).
verb( 'disconnect', 'disconnects', 'disconnecting', 'disconnected', 'disconnected', tran, ['6A','14'] ).
verb( 'discontent', 'discontents', 'discontenting', 'discontented', 'discontented', tran, ['6A'] ).
verb( 'discontinue', 'discontinues', 'discontinuing', 'discontinued', 'discontinued', _, ['2A','6A','6C'] ).
verb( 'discount', 'discounts', 'discounting', 'discounted', 'discounted', tran, ['6A'] ).
verb( 'discountenance', 'discountenances', 'discountenancing', 'discountenanced', 'discountenanced', tran, ['6A'] ).
verb( 'discourage', 'discourages', 'discouraging', 'discouraged', 'discouraged', tran, ['6A','14'] ).
verb( 'discourse', 'discourses', 'discoursing', 'discoursed', 'discoursed', intran, [] ).
verb( 'discover', 'discovers', 'discovering', 'discovered', 'discovered', tran, ['6A','8','9','10','25'] ).
verb( 'discredit', 'discredits', 'discrediting', 'discredited', 'discredited', tran, ['6A'] ).
verb( 'discriminate', 'discriminates', 'discriminating', 'discriminated', 'discriminated', _, ['2A','3A','14'] ).
verb( 'discuss', 'discusses', 'discussing', 'discussed', 'discussed', tran, ['6A','8','10','14'] ).
verb( 'disdain', 'disdains', 'disdaining', 'disdained', 'disdained', tran, ['6A','6C','7A'] ).
verb( 'disembark', 'disembarks', 'disembarking', 'disembarked', 'disembarked', _, ['2A','2C','6A','14'] ).
verb( 'disembarrass', 'disembarrasses', 'disembarrassing', 'disembarrassed', 'disembarrassed', tran, ['14'] ).
verb( 'disembody', 'disembodies', 'disembodying', 'disembodied', 'disembodied', tran, ['6A'] ).
verb( 'disembowel', 'disembowels', 'disembowelling', 'disembowelled', 'disembowelled', tran, ['6A'] ).
verb( 'disenchant', 'disenchants', 'disenchanting', 'disenchanted', 'disenchanted', tran, ['6A'] ).
verb( 'disencumber', 'disencumbers', 'disencumbering', 'disencumbered', 'disencumbered', tran, ['6A','14'] ).
verb( 'disenfranchise', 'disenfranchises', 'disenfranchising', 'disenfranchised', 'disenfranchised', tran, [] ).
verb( 'disengage', 'disengages', 'disengaging', 'disengaged', 'disengaged', _, ['2A','2C','6A','14'] ).
verb( 'disentangle', 'disentangles', 'disentangling', 'disentangled', 'disentangled', _, ['2A','6A','14'] ).
verb( 'disestablish', 'disestablishes', 'disestablishing', 'disestablished', 'disestablished', tran, ['6A'] ).
verb( 'disfavour', 'disfavours', 'disfavouring', 'disfavoured', 'disfavoured', tran, ['6A'] ).
verb( 'disfigure', 'disfigures', 'disfiguring', 'disfigured', 'disfigured', tran, ['6A'] ).
verb( 'disforest', 'disforests', 'disforesting', 'disforested', 'disforested', tran, ['6A'] ).
verb( 'disfranchise', 'disfranchises', 'disfranchising', 'disfranchised', 'disfranchised', tran, ['6A'] ).
verb( 'disgorge', 'disgorges', 'disgorging', 'disgorged', 'disgorged', tran, ['6A'] ).
verb( 'disgrace', 'disgraces', 'disgracing', 'disgraced', 'disgraced', tran, ['6A'] ).
verb( 'disguise', 'disguises', 'disguising', 'disguised', 'disguised', tran, ['6A','16B'] ).
verb( 'disgust', 'disgusts', 'disgusting', 'disgusted', 'disgusted', tran, ['6A'] ).
verb( 'dish', 'dishes', 'dishing', 'dished', 'dished', tran, ['6A','15B'] ).
verb( 'dishearten', 'disheartens', 'disheartening', 'disheartened', 'disheartened', tran, ['6A'] ).
verb( 'dishonour', 'dishonours', 'dishonouring', 'dishonoured', 'dishonoured', tran, ['6A'] ).
verb( 'disillusion', 'disillusions', 'disillusioning', 'disillusioned', 'disillusioned', tran, ['6A'] ).
verb( 'disincline', 'disinclines', 'disinclining', 'disinclined', 'disinclined', tran, ['14','17'] ).
verb( 'disinfect', 'disinfects', 'disinfecting', 'disinfected', 'disinfected', tran, ['6A'] ).
verb( 'disinfest', 'disinfests', 'disinfesting', 'disinfested', 'disinfested', tran, ['6A'] ).
verb( 'disinherit', 'disinherits', 'disinheriting', 'disinherited', 'disinherited', tran, ['6A'] ).
verb( 'disintegrate', 'disintegrates', 'disintegrating', 'disintegrated', 'disintegrated', _, ['2A','6A'] ).
verb( 'disinter', 'disinters', 'disinterring', 'disinterred', 'disinterred', tran, ['6A'] ).
verb( 'disjoint', 'disjoints', 'disjointing', 'disjointed', 'disjointed', tran, ['6A'] ).
verb( 'dislike', 'dislikes', 'disliking', 'disliked', 'disliked', tran, ['6A','6C'] ).
verb( 'dislocate', 'dislocates', 'dislocating', 'dislocated', 'dislocated', tran, ['6A'] ).
verb( 'dislodge', 'dislodges', 'dislodging', 'dislodged', 'dislodged', tran, ['6A','14'] ).
verb( 'dismantle', 'dismantles', 'dismantling', 'dismantled', 'dismantled', tran, ['6A'] ).
verb( 'dismay', 'dismays', 'dismaying', 'dismayed', 'dismayed', tran, ['6A'] ).
verb( 'dismember', 'dismembers', 'dismembering', 'dismembered', 'dismembered', tran, ['6A'] ).
verb( 'dismiss', 'dismisses', 'dismissing', 'dismissed', 'dismissed', tran, ['6A','14'] ).
verb( 'dismount', 'dismounts', 'dismounting', 'dismounted', 'dismounted', _, ['2A','3A','6A'] ).
verb( 'disobey', 'disobeys', 'disobeying', 'disobeyed', 'disobeyed', tran, ['2A','6A'] ).
verb( 'disoblige', 'disobliges', 'disobliging', 'disobliged', 'disobliged', tran, ['6A'] ).
verb( 'disorder', 'disorders', 'disordering', 'disordered', 'disordered', tran, ['6A'] ).
verb( 'disorganize', 'disorganizes', 'disorganizing', 'disorganized', 'disorganized', tran, ['6A'] ).
verb( 'disorient', 'disorients', 'disorienting', 'disoriented', 'disoriented', tran, ['6A'] ).
verb( 'disorientate', 'disorientates', 'disorientating', 'disorientated', 'disorientated', tran, ['6A'] ).
verb( 'disown', 'disowns', 'disowning', 'disowned', 'disowned', tran, ['6A'] ).
verb( 'disparage', 'disparages', 'disparaging', 'disparaged', 'disparaged', tran, ['6A'] ).
verb( 'dispatch', 'dispatches', 'dispatching', 'dispatched', 'dispatched', tran, ['6A','14'] ).
verb( 'dispel', 'dispels', 'dispelling', 'dispelled', 'dispelled', tran, ['6A'] ).
verb( 'dispense', 'dispenses', 'dispensing', 'dispensed', 'dispensed', _, ['3A','6A','14'] ).
verb( 'disperse', 'disperses', 'dispersing', 'dispersed', 'dispersed', _, ['2A','6A'] ).
verb( 'dispirit', 'dispirits', 'dispiriting', 'dispirited', 'dispirited', tran, ['6A'] ).
verb( 'displace', 'displaces', 'displacing', 'displaced', 'displaced', tran, ['6A'] ).
verb( 'display', 'displays', 'displaying', 'displayed', 'displayed', tran, ['6A'] ).
verb( 'displease', 'displeases', 'displeasing', 'displeased', 'displeased', tran, ['6A'] ).
verb( 'disport', 'disports', 'disporting', 'disported', 'disported', tran, ['6A'] ).
verb( 'dispose', 'disposes', 'disposing', 'disposed', 'disposed', _, ['2A','3A','6A','17'] ).
verb( 'dispossess', 'dispossesses', 'dispossessing', 'dispossessed', 'dispossessed', tran, ['14'] ).
verb( 'disprove', 'disproves', 'disproving', 'disproved', 'disproved', tran, ['6A'] ).
verb( 'dispute', 'disputes', 'disputing', 'disputed', 'disputed', _, ['2A','3A','6A','8','10'] ).
verb( 'disqualify', 'disqualifies', 'disqualifying', 'disqualified', 'disqualified', tran, ['6A','14'] ).
verb( 'disquiet', 'disquiets', 'disquieting', 'disquieted', 'disquieted', tran, ['6A'] ).
verb( 'disregard', 'disregards', 'disregarding', 'disregarded', 'disregarded', tran, ['6A'] ).
verb( 'disrobe', 'disrobes', 'disrobing', 'disrobed', 'disrobed', _, ['2A','6A'] ).
verb( 'disrupt', 'disrupts', 'disrupting', 'disrupted', 'disrupted', tran, ['6A'] ).
verb( 'dissatisfy', 'dissatisfies', 'dissatisfying', 'dissatisfied', 'dissatisfied', tran, ['6A'] ).
verb( 'dissect', 'dissects', 'dissecting', 'dissected', 'dissected', tran, ['6A'] ).
verb( 'dissemble', 'dissembles', 'dissembling', 'dissembled', 'dissembled', _, ['2A','6A'] ).
verb( 'disseminate', 'disseminates', 'disseminating', 'disseminated', 'disseminated', tran, ['6A'] ).
verb( 'dissent', 'dissents', 'dissenting', 'dissented', 'dissented', intran, ['2A','3A'] ).
verb( 'dissever', 'dissevers', 'dissevering', 'dissevered', 'dissevered', tran, ['6A'] ).
verb( 'dissimulate', 'dissimulates', 'dissimulating', 'dissimulated', 'dissimulated', _, ['2A','6A'] ).
verb( 'dissipate', 'dissipates', 'dissipating', 'dissipated', 'dissipated', _, ['2A','6A'] ).
verb( 'dissociate', 'dissociates', 'dissociating', 'dissociated', 'dissociated', tran, ['6A','14'] ).
verb( 'dissolve', 'dissolves', 'dissolving', 'dissolved', 'dissolved', _, ['2A','2C','3A','6A','14'] ).
verb( 'dissuade', 'dissuades', 'dissuading', 'dissuaded', 'dissuaded', tran, ['6A','14'] ).
verb( 'distance', 'distances', 'distancing', 'distanced', 'distanced', tran, ['6A','14'] ).
verb( 'distemper', 'distempers', 'distempering', 'distempered', 'distempered', tran, ['6A','22'] ).
verb( 'distend', 'distends', 'distending', 'distended', 'distended', _, ['2A','6A'] ).
verb( 'distil', 'distils', 'distilling', 'distilled', 'distilled', _, ['2A','6A','14','15B'] ).
verb( 'distinguish', 'distinguishes', 'distinguishing', 'distinguished', 'distinguished', _, ['3A','6A','14'] ).
verb( 'distort', 'distorts', 'distorting', 'distorted', 'distorted', tran, ['6A'] ).
verb( 'distract', 'distracts', 'distracting', 'distracted', 'distracted', tran, ['6A','14'] ).
verb( 'distrain', 'distrains', 'distraining', 'distrained', 'distrained', intran, ['2A','3A'] ).
verb( 'distress', 'distresses', 'distressing', 'distressed', 'distressed', tran, ['6A'] ).
verb( 'distribute', 'distributes', 'distributing', 'distributed', 'distributed', tran, ['6A','14'] ).
verb( 'distrust', 'distrusts', 'distrusting', 'distrusted', 'distrusted', tran, ['6A'] ).
verb( 'disturb', 'disturbs', 'disturbing', 'disturbed', 'disturbed', tran, ['6A'] ).
verb( 'disunite', 'disunites', 'disuniting', 'disunited', 'disunited', _, ['2A','6A'] ).
verb( 'ditch', 'ditches', 'ditching', 'ditched', 'ditched', _, ['2A','6A'] ).
verb( 'dither', 'dithers', 'dithering', 'dithered', 'dithered', intran, ['2A','2C'] ).
verb( 'divagate', 'divagates', 'divagating', 'divagated', 'divagated', intran, ['2A','3A'] ).
verb( 'dive', 'dives', 'diving', 'dived', 'dived', intran, ['2A','2C'] ).
verb( 'dive-bomb', 'dive-bombs', 'dive-bombing', 'dive-bombed', 'dive-bombed', _, [] ).
verb( 'diverge', 'diverges', 'diverging', 'diverged', 'diverged', intran, ['2A','3A'] ).
verb( 'diversify', 'diversifies', 'diversifying', 'diversified', 'diversified', tran, ['6A'] ).
verb( 'divert', 'diverts', 'diverting', 'diverted', 'diverted', tran, ['6A','14'] ).
verb( 'divest', 'divests', 'divesting', 'divested', 'divested', tran, ['14'] ).
verb( 'divide', 'divides', 'dividing', 'divided', 'divided', _, ['2A','2C','6A','14','15B'] ).
verb( 'divine', 'divines', 'divining', 'divined', 'divined', _, ['2A','6A','10'] ).
verb( 'divorce', 'divorces', 'divorcing', 'divorced', 'divorced', tran, ['6A','14'] ).
verb( 'divulge', 'divulges', 'divulging', 'divulged', 'divulged', tran, ['6A','14'] ).
verb( 'dizzy', 'dizzies', 'dizzying', 'dizzied', 'dizzied', tran, [] ).
verb( 'do', 'does', 'doing', 'did', 'done', unknown, ['2A','2B','2C','3A','6A','6C','7B','12B','13B','15A','15B'] ).
verb( 'do', 'does', 'doing', 'did', 'done', _, ['2A','2B','2C','3A','6A','6C','7B','12B','13B','15A','15B'] ).
verb( 'dock', 'docks', 'docking', 'docked', 'docked', _, ['2A','6A','14'] ).
verb( 'docket', 'dockets', 'docketing', 'docketed', 'docketed', tran, ['6A'] ).
verb( 'doctor', 'doctors', 'doctoring', 'doctored', 'doctored', tran, ['6A'] ).
verb( 'document', 'documents', 'documenting', 'documented', 'documented', tran, ['6A'] ).
verb( 'dodder', 'dodders', 'doddering', 'doddered', 'doddered', intran, ['2A','2C'] ).
verb( 'dodge', 'dodges', 'dodging', 'dodged', 'dodged', _, ['2A','3A','6A'] ).
verb( 'doff', 'doffs', 'doffing', 'doffed', 'doffed', tran, ['6A'] ).
verb( 'dog', 'dogs', 'dogging', 'dogged', 'dogged', tran, ['6A'] ).
verb( 'dogmatize', 'dogmatizes', 'dogmatizing', 'dogmatized', 'dogmatized', _, ['2A','6A'] ).
verb( 'dole', 'doles', 'doling', 'doled', 'doled', tran, ['15B'] ).
verb( 'doll', 'dolls', 'dolling', 'dolled', 'dolled', _, ['2C','15B'] ).
verb( 'domesticate', 'domesticates', 'domesticating', 'domesticated', 'domesticated', tran, ['6A'] ).
verb( 'dominate', 'dominates', 'dominating', 'dominated', 'dominated', _, ['2A','3A','6A'] ).
verb( 'domineer', 'domineers', 'domineering', 'domineered', 'domineered', intran, ['2A','3A'] ).
verb( 'don', 'dons', 'donning', 'donned', 'donned', tran, ['6A'] ).
verb( 'donate', 'donates', 'donating', 'donated', 'donated', tran, ['6A','14'] ).
verb( 'doodle', 'doodles', 'doodling', 'doodled', 'doodled', intran, ['2A'] ).
verb( 'doom', 'dooms', 'dooming', 'doomed', 'doomed', tran, ['6A','14','17'] ).
verb( 'dope', 'dopes', 'doping', 'doped', 'doped', tran, ['6A'] ).
verb( 'dose', 'doses', 'dosing', 'dosed', 'dosed', tran, ['6A','14','15B'] ).
verb( 'doss', 'dosses', 'dossing', 'dossed', 'dossed', intran, ['2C'] ).
verb( 'dot', 'dots', 'dotting', 'dotted', 'dotted', tran, [] ).
verb( 'dote', 'dotes', 'doting', 'doted', 'doted', intran, ['3A'] ).
verb( 'double', 'doubles', 'doubling', 'doubled', 'doubled', _, ['2A','2C','6A','15B'] ).
verb( 'double-check', 'double-checks', 'double-checking', 'double-checked', 'double-checked', tran, [] ).
verb( 'double-cross', 'double-crosses', 'double-crossing', 'double-crossed', 'double-crossed', tran, ['6A'] ).
verb( 'double-park', 'double-parks', 'double-parking', 'double-parked', 'double-parked', _, [] ).
verb( 'doubt', 'doubts', 'doubting', 'doubted', 'doubted', tran, ['6A','9','10'] ).
verb( 'douse', 'douses', 'dousing', 'doused', 'doused', tran, ['6A'] ).
verb( 'dovetail', 'dovetails', 'dovetailing', 'dovetailed', 'dovetailed', _, ['2A','3A','6A'] ).
verb( 'dower', 'dowers', 'dowering', 'dowered', 'dowered', tran, [] ).
verb( 'down', 'downs', 'downing', 'downed', 'downed', tran, ['6A'] ).
verb( 'downgrade', 'downgrades', 'downgrading', 'downgraded', 'downgraded', tran, ['6A'] ).
verb( 'dowse', 'dowses', 'dowsing', 'dowsed', 'dowsed', tran, ['6A'] ).
verb( 'doze', 'dozes', 'dozing', 'dozed', 'dozed', intran, ['2A','2C'] ).
verb( 'draft', 'drafts', 'drafting', 'drafted', 'drafted', tran, ['6A'] ).
verb( 'drag', 'drags', 'dragging', 'dragged', 'dragged', _, ['2A','2C','6A','14','15B'] ).
verb( 'dragoon', 'dragoons', 'dragooning', 'dragooned', 'dragooned', tran, ['6A','14'] ).
verb( 'drain', 'drains', 'draining', 'drained', 'drained', _, ['2A','2C','6A','14','15B','22'] ).
verb( 'dramatize', 'dramatizes', 'dramatizing', 'dramatized', 'dramatized', tran, ['6A'] ).
verb( 'drape', 'drapes', 'draping', 'draped', 'draped', tran, ['6A','14'] ).
verb( 'drat', 'drats', 'dratting', 'dratted', 'dratted', tran, [] ).
verb( 'draught', 'draughts', 'draughting', 'draughted', 'draughted', tran, [] ).
verb( 'draw', 'draws', 'drawing', 'drew', 'drawn', _, ['2A','2B','2C','2D','3A','6A','14','15B'] ).
verb( 'drawl', 'drawls', 'drawling', 'drawled', 'drawled', _, ['2A','2C','6A','15B'] ).
verb( 'dread', 'dreads', 'dreading', 'dreaded', 'dreaded', _, ['6A','6C','7A'] ).
verb( 'dream', 'dreams', 'dreaming', 'dreamed', 'dreamed', _, ['2A','3A','6A','8','9','10','15B'] ).
verb( 'dredge', 'dredges', 'dredging', 'dredged', 'dredged', _, ['2A','3A','6A','14','15B'] ).
verb( 'drench', 'drenches', 'drenching', 'drenched', 'drenched', tran, ['6A'] ).
verb( 'dress', 'dresses', 'dressing', 'dressed', 'dressed', _, ['2A','2C','6A','15B'] ).
verb( 'dribble', 'dribbles', 'dribbling', 'dribbled', 'dribbled', _, ['2A','6A'] ).
verb( 'drift', 'drifts', 'drifting', 'drifted', 'drifted', _, ['2A','2C','6A','14','15B'] ).
verb( 'drill', 'drills', 'drilling', 'drilled', 'drilled', _, ['2A','6A','14'] ).
verb( 'drink', 'drinks', 'drinking', 'drank', 'drunk', _, ['2A','3A','6A','15B'] ).
verb( 'drip', 'drips', 'dripping', 'dripped', 'dripped', _, ['2A','2C','6A'] ).
verb( 'drip-dry', 'drip-dries', 'drip-drying', 'drip-dried', 'drip-dried', tran, [] ).
verb( 'drive', 'drives', 'driving', 'drove', 'driven', _, ['2A','2C','3A','6A','14','15B','17','22'] ).
verb( 'drivel', 'drivels', 'drivelling', 'drivelled', 'drivelled', intran, ['2A','2C'] ).
verb( 'drizzle', 'drizzles', 'drizzling', 'drizzled', 'drizzled', intran, ['2A'] ).
verb( 'drone', 'drones', 'droning', 'droned', 'droned', _, ['2C','15B'] ).
verb( 'drool', 'drools', 'drooling', 'drooled', 'drooled', intran, [] ).
verb( 'droop', 'droops', 'drooping', 'drooped', 'drooped', _, ['2A','2C','6A'] ).
verb( 'drop', 'drops', 'dropping', 'dropped', 'dropped', _, ['2A','2C','3B','6A','12A','13A','14','15A','15B'] ).
verb( 'drown', 'drowns', 'drowning', 'drowned', 'drowned', _, ['2A','6A','15B'] ).
verb( 'drowse', 'drowses', 'drowsing', 'drowsed', 'drowsed', _, ['2A','2C','15B'] ).
verb( 'drub', 'drubs', 'drubbing', 'drubbed', 'drubbed', tran, ['6A','14'] ).
verb( 'drudge', 'drudges', 'drudging', 'drudged', 'drudged', intran, ['2A','2C','3A'] ).
verb( 'drug', 'drugs', 'drugging', 'drugged', 'drugged', tran, ['6A'] ).
verb( 'drum', 'drums', 'drumming', 'drummed', 'drummed', _, ['2A','2C','3A','6A','14','15B'] ).
verb( 'dry', 'dries', 'drying', 'dried', 'dried', _, ['2A','2C','6A','15B'] ).
verb( 'dry-clean', 'dry-cleans', 'dry-cleaning', 'dry-cleaned', 'dry-cleaned', tran, [] ).
verb( 'dub', 'dubs', 'dubbing', 'dubbed', 'dubbed', tran, ['6A','22','23'] ).
verb( 'duck', 'ducks', 'ducking', 'ducked', 'ducked', _, ['2A','6A'] ).
verb( 'duel', 'duels', 'duelling', 'duelled', 'duelled', intran, [] ).
verb( 'dull', 'dulls', 'dulling', 'dulled', 'dulled', _, ['2A','6A'] ).
verb( 'dumbfound', 'dumbfounds', 'dumbfounding', 'dumbfounded', 'dumbfounded', tran, ['6A'] ).
verb( 'dump', 'dumps', 'dumping', 'dumped', 'dumped', tran, ['6A','15A'] ).
verb( 'dun', 'duns', 'dunning', 'dunned', 'dunned', tran, [] ).
verb( 'dunk', 'dunks', 'dunking', 'dunked', 'dunked', tran, ['6A','14'] ).
verb( 'dupe', 'dupes', 'duping', 'duped', 'duped', tran, ['6A'] ).
verb( 'duplicate', 'duplicates', 'duplicating', 'duplicated', 'duplicated', tran, ['6A'] ).
verb( 'dust', 'dusts', 'dusting', 'dusted', 'dusted', tran, ['6A'] ).
verb( 'dwarf', 'dwarfs', 'dwarfing', 'dwarfed', 'dwarfed', tran, ['6A'] ).
verb( 'dwell', 'dwells', 'dwelling', 'dwelt', 'dwelt', intran, ['3A'] ).
verb( 'dwindle', 'dwindles', 'dwindling', 'dwindled', 'dwindled', intran, ['2A'] ).
verb( 'dye', 'dyes', 'dying', 'dyed', 'dyed', _, ['2A','6A','22'] ).
verb( 'dyke', 'dykes', 'dyking', 'dyked', 'dyked', _, [] ).
verb( 'dynamite', 'dynamites', 'dynamiting', 'dynamited', 'dynamited', tran, ['6A'] ).
verb( 'earmark', 'earmarks', 'earmarking', 'earmarked', 'earmarked', tran, ['6A','14'] ).
verb( 'earn', 'earns', 'earning', 'earned', 'earned', tran, ['6A','12B','13B'] ).
verb( 'earth', 'earths', 'earthing', 'earthed', 'earthed', tran, ['6A','15B'] ).
verb( 'ease', 'eases', 'easing', 'eased', 'eased', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'eat', 'eats', 'eating', 'ate', 'eaten', _, ['2A','2C','3A','4A','6A','15B'] ).
verb( 'eavesdrop', 'eavesdrops', 'eavesdropping', 'eavesdropped', 'eavesdropped', intran, ['2A','3A'] ).
verb( 'ebb', 'ebbs', 'ebbing', 'ebbed', 'ebbed', intran, ['2A','2C'] ).
verb( 'echo', 'echos', 'echoing', 'echoed', 'echoed', _, ['2A','2C','6A','15B'] ).
verb( 'eclipse', 'eclipses', 'eclipsing', 'eclipsed', 'eclipsed', tran, ['6A'] ).
verb( 'economize', 'economizes', 'economizing', 'economized', 'economized', _, ['2A','3A','6A'] ).
verb( 'eddy', 'eddies', 'eddying', 'eddied', 'eddied', intran, ['2A','2C'] ).
verb( 'edge', 'edges', 'edging', 'edged', 'edged', _, ['2C','6A','14','15A','15B'] ).
verb( 'edify', 'edifies', 'edifying', 'edified', 'edified', tran, ['6A'] ).
verb( 'edit', 'edits', 'editing', 'edited', 'edited', tran, ['6A'] ).
verb( 'educate', 'educates', 'educating', 'educated', 'educated', tran, ['6A','15A','16A'] ).
verb( 'educe', 'educes', 'educing', 'educed', 'educed', tran, ['6A'] ).
verb( 'eff', 'effs', 'effing', 'effed', 'effed', intran, [] ).
verb( 'efface', 'effaces', 'effacing', 'effaced', 'effaced', tran, ['6A'] ).
verb( 'effect', 'effects', 'effecting', 'effected', 'effected', tran, ['6A'] ).
verb( 'effervesce', 'effervesces', 'effervescing', 'effervesced', 'effervesced', intran, ['2A'] ).
verb( 'egg', 'eggs', 'egging', 'egged', 'egged', tran, ['15B'] ).
verb( 'egotrip', 'egotrips', 'egotripping', 'egotripped', 'egotripped', intran, [] ).
verb( 'ejaculate', 'ejaculates', 'ejaculating', 'ejaculated', 'ejaculated', tran, ['6A'] ).
verb( 'eject', 'ejects', 'ejecting', 'ejected', 'ejected', _, ['2A','6A','14'] ).
verb( 'eke', 'ekes', 'eking', 'eked', 'eked', tran, ['15B'] ).
verb( 'elaborate', 'elaborates', 'elaborating', 'elaborated', 'elaborated', tran, ['6A'] ).
verb( 'elapse', 'elapses', 'elapsing', 'elapsed', 'elapsed', intran, ['2A'] ).
verb( 'elate', 'elates', 'elating', 'elated', 'elated', tran, ['6A'] ).
verb( 'elbow', 'elbows', 'elbowing', 'elbowed', 'elbowed', tran, ['6A','14','15B'] ).
verb( 'elect', 'elects', 'electing', 'elected', 'elected', tran, ['6A','7A','14','23','25'] ).
verb( 'electrify', 'electrifies', 'electrifying', 'electrified', 'electrified', tran, ['6A'] ).
verb( 'electrocute', 'electrocutes', 'electrocuting', 'electrocuted', 'electrocuted', tran, ['6A'] ).
verb( 'electroplate', 'electroplates', 'electroplating', 'electroplated', 'electroplated', tran, [] ).
verb( 'elevate', 'elevates', 'elevating', 'elevated', 'elevated', tran, ['6A','14'] ).
verb( 'elicit', 'elicits', 'eliciting', 'elicited', 'elicited', tran, ['6A','14'] ).
verb( 'elide', 'elides', 'eliding', 'elided', 'elided', tran, ['6A'] ).
verb( 'eliminate', 'eliminates', 'eliminating', 'eliminated', 'eliminated', tran, ['6A','14'] ).
verb( 'elongate', 'elongates', 'elongating', 'elongated', 'elongated', _, ['2A','6A'] ).
verb( 'elope', 'elopes', 'eloping', 'eloped', 'eloped', intran, ['2A','2C','3A'] ).
verb( 'elucidate', 'elucidates', 'elucidating', 'elucidated', 'elucidated', tran, ['6A'] ).
verb( 'elude', 'eludes', 'eluding', 'eluded', 'eluded', tran, ['6A'] ).
verb( 'emaciate', 'emaciates', 'emaciating', 'emaciated', 'emaciated', tran, ['6A'] ).
verb( 'emanate', 'emanates', 'emanating', 'emanated', 'emanated', intran, ['3A'] ).
verb( 'emancipate', 'emancipates', 'emancipating', 'emancipated', 'emancipated', tran, ['6A','14'] ).
verb( 'emasculate', 'emasculates', 'emasculating', 'emasculated', 'emasculated', tran, ['6A'] ).
verb( 'embalm', 'embalms', 'embalming', 'embalmed', 'embalmed', tran, ['6A'] ).
verb( 'embargo', 'embargos', 'embargoing', 'embargoed', 'embargoed', tran, ['6A'] ).
verb( 'embark', 'embarks', 'embarking', 'embarked', 'embarked', _, ['2A','2C','3A','6A'] ).
verb( 'embarrass', 'embarrasses', 'embarrassing', 'embarrassed', 'embarrassed', tran, ['6A'] ).
verb( 'embed', 'embeds', 'embedding', 'embedded', 'embedded', tran, ['6A','14'] ).
verb( 'embellish', 'embellishes', 'embellishing', 'embellished', 'embellished', tran, ['6A','14'] ).
verb( 'embezzle', 'embezzles', 'embezzling', 'embezzled', 'embezzled', tran, ['6A'] ).
verb( 'embitter', 'embitters', 'embittering', 'embittered', 'embittered', tran, ['6A'] ).
verb( 'emblazon', 'emblazons', 'emblazoning', 'emblazoned', 'emblazoned', tran, ['6A','14'] ).
verb( 'embody', 'embodies', 'embodying', 'embodied', 'embodied', tran, ['6A','14'] ).
verb( 'embolden', 'emboldens', 'emboldening', 'emboldened', 'emboldened', tran, ['6A','17'] ).
verb( 'emboss', 'embosses', 'embossing', 'embossed', 'embossed', tran, ['6A','14'] ).
verb( 'embrace', 'embraces', 'embracing', 'embraced', 'embraced', _, ['2A','6A'] ).
verb( 'embroider', 'embroiders', 'embroidering', 'embroidered', 'embroidered', _, ['2A','6A'] ).
verb( 'embroil', 'embroils', 'embroiling', 'embroiled', 'embroiled', tran, ['6A','14'] ).
verb( 'emend', 'emends', 'emending', 'emended', 'emended', tran, ['6A'] ).
verb( 'emerge', 'emerges', 'emerging', 'emerged', 'emerged', intran, ['2A','3A'] ).
verb( 'emigrate', 'emigrates', 'emigrating', 'emigrated', 'emigrated', intran, ['2A','3A'] ).
verb( 'emit', 'emits', 'emitting', 'emitted', 'emitted', tran, ['6A'] ).
verb( 'empale', 'empales', 'empaling', 'empaled', 'empaled', tran, [] ).
verb( 'empanel', 'empanels', 'empanelling', 'empanelled', 'empanelled', tran, ['6A'] ).
verb( 'emphasize', 'emphasizes', 'emphasizing', 'emphasized', 'emphasized', tran, ['6A'] ).
verb( 'emplane', 'emplanes', 'emplaning', 'emplaned', 'emplaned', _, ['2A','6A'] ).
verb( 'employ', 'employs', 'employing', 'employed', 'employed', tran, ['6A','14','16B'] ).
verb( 'empower', 'empowers', 'empowering', 'empowered', 'empowered', tran, ['17'] ).
verb( 'empty', 'empties', 'emptying', 'emptied', 'emptied', _, ['2A','2C','3A','6A','15B'] ).
verb( 'emulate', 'emulates', 'emulating', 'emulated', 'emulated', tran, ['6A'] ).
verb( 'emulsify', 'emulsifies', 'emulsifying', 'emulsified', 'emulsified', tran, ['6A'] ).
verb( 'enable', 'enables', 'enabling', 'enabled', 'enabled', tran, ['17'] ).
verb( 'enact', 'enacts', 'enacting', 'enacted', 'enacted', tran, ['6A','9'] ).
verb( 'enamel', 'enamels', 'enamelling', 'enamelled', 'enamelled', tran, [] ).
verb( 'enamour', 'enamours', 'enamouring', 'enamoured', 'enamoured', tran, ['6A'] ).
verb( 'encamp', 'encamps', 'encamping', 'encamped', 'encamped', _, ['2A','6A'] ).
verb( 'encase', 'encases', 'encasing', 'encased', 'encased', tran, ['6A','14'] ).
verb( 'enchain', 'enchains', 'enchaining', 'enchained', 'enchained', tran, ['6A'] ).
verb( 'enchant', 'enchants', 'enchanting', 'enchanted', 'enchanted', tran, ['6A'] ).
verb( 'encircle', 'encircles', 'encircling', 'encircled', 'encircled', tran, ['6A'] ).
verb( 'enclose', 'encloses', 'enclosing', 'enclosed', 'enclosed', tran, ['6A','14'] ).
verb( 'encode', 'encodes', 'encoding', 'encoded', 'encoded', tran, ['6A'] ).
verb( 'encompass', 'encompasses', 'encompassing', 'encompassed', 'encompassed', tran, ['6A'] ).
verb( 'encore', 'encores', 'encoring', 'encored', 'encored', tran, [] ).
verb( 'encounter', 'encounters', 'encountering', 'encountered', 'encountered', tran, ['6A'] ).
verb( 'encourage', 'encourages', 'encouraging', 'encouraged', 'encouraged', tran, ['6A','14','17'] ).
verb( 'encroach', 'encroaches', 'encroaching', 'encroached', 'encroached', intran, ['3A'] ).
verb( 'encrust', 'encrusts', 'encrusting', 'encrusted', 'encrusted', _, ['2A','6A','14'] ).
verb( 'encumber', 'encumbers', 'encumbering', 'encumbered', 'encumbered', tran, ['6A','14'] ).
verb( 'end', 'ends', 'ending', 'ended', 'ended', _, ['2A','2C','3A','6A','15B'] ).
verb( 'endanger', 'endangers', 'endangering', 'endangered', 'endangered', tran, ['6A'] ).
verb( 'endear', 'endears', 'endearing', 'endeared', 'endeared', tran, ['14'] ).
verb( 'endeavour', 'endeavours', 'endeavouring', 'endeavoured', 'endeavoured', intran, ['4A'] ).
verb( 'endorse', 'endorses', 'endorsing', 'endorsed', 'endorsed', tran, ['6A'] ).
verb( 'endow', 'endows', 'endowing', 'endowed', 'endowed', tran, ['6A','14'] ).
verb( 'endue', 'endues', 'enduing', 'endued', 'endued', tran, ['14'] ).
verb( 'endure', 'endures', 'enduring', 'endured', 'endured', _, ['2A','2C','6A','6D','17'] ).
verb( 'enervate', 'enervates', 'enervating', 'enervated', 'enervated', tran, ['6A'] ).
verb( 'enfeeble', 'enfeebles', 'enfeebling', 'enfeebled', 'enfeebled', tran, ['6A'] ).
verb( 'enfold', 'enfolds', 'enfolding', 'enfolded', 'enfolded', tran, ['6A','14'] ).
verb( 'enforce', 'enforces', 'enforcing', 'enforced', 'enforced', tran, ['6A','14'] ).
verb( 'enfranchise', 'enfranchises', 'enfranchising', 'enfranchised', 'enfranchised', tran, ['6A'] ).
verb( 'engage', 'engages', 'engaging', 'engaged', 'engaged', _, ['2A','3A','6A','7A','9','14','16B','17'] ).
verb( 'engender', 'engenders', 'engendering', 'engendered', 'engendered', tran, ['6A'] ).
verb( 'engineer', 'engineers', 'engineering', 'engineered', 'engineered', _, ['2A','6A'] ).
verb( 'engraft', 'engrafts', 'engrafting', 'engrafted', 'engrafted', tran, ['6A','14'] ).
verb( 'engrave', 'engraves', 'engraving', 'engraved', 'engraved', tran, ['6A','14'] ).
verb( 'engross', 'engrosses', 'engrossing', 'engrossed', 'engrossed', tran, ['6A'] ).
verb( 'engulf', 'engulfs', 'engulfing', 'engulfed', 'engulfed', tran, ['6A'] ).
verb( 'enhance', 'enhances', 'enhancing', 'enhanced', 'enhanced', tran, ['6A'] ).
verb( 'enjoin', 'enjoins', 'enjoining', 'enjoined', 'enjoined', tran, ['6A','9','14','17'] ).
verb( 'enjoy', 'enjoys', 'enjoying', 'enjoyed', 'enjoyed', tran, ['6A','6C'] ).
verb( 'enkindle', 'enkindles', 'enkindling', 'enkindled', 'enkindled', tran, ['6A'] ).
verb( 'enlarge', 'enlarges', 'enlarging', 'enlarged', 'enlarged', _, ['2A','3A','6A'] ).
verb( 'enlighten', 'enlightens', 'enlightening', 'enlightened', 'enlightened', tran, ['6A','14'] ).
verb( 'enlist', 'enlists', 'enlisting', 'enlisted', 'enlisted', _, ['2A','2C','6A','14','16B'] ).
verb( 'enliven', 'enlivens', 'enlivening', 'enlivened', 'enlivened', tran, ['6A'] ).
verb( 'enmesh', 'enmeshes', 'enmeshing', 'enmeshed', 'enmeshed', tran, ['6A','14'] ).
verb( 'ennoble', 'ennobles', 'ennobling', 'ennobled', 'ennobled', tran, ['6A'] ).
verb( 'enplane', 'enplanes', 'enplaning', 'enplaned', 'enplaned', _, [] ).
verb( 'enquire', 'enquires', 'enquiring', 'enquired', 'enquired', _, ['2A','3A','6A','8','10','14'] ).
verb( 'enrage', 'enrages', 'enraging', 'enraged', 'enraged', tran, ['6A'] ).
verb( 'enrapture', 'enraptures', 'enrapturing', 'enraptured', 'enraptured', tran, ['6A'] ).
verb( 'enrich', 'enriches', 'enriching', 'enriched', 'enriched', tran, ['6A','14'] ).
verb( 'enrol', 'enrols', 'enroling', 'enroled', 'enroled', _, ['2A','2C','6A','14','16B'] ).
verb( 'enroll', 'enrolls', 'enrolling', 'enrolled', 'enrolled', _, ['2A','2C','6A','14','16B'] ).
verb( 'ensconce', 'ensconces', 'ensconcing', 'ensconced', 'ensconced', tran, ['14'] ).
verb( 'enshrine', 'enshrines', 'enshrining', 'enshrined', 'enshrined', tran, ['6A','14'] ).
verb( 'enshroud', 'enshrouds', 'enshrouding', 'enshrouded', 'enshrouded', tran, ['6A'] ).
verb( 'enslave', 'enslaves', 'enslaving', 'enslaved', 'enslaved', tran, ['6A'] ).
verb( 'ensnare', 'ensnares', 'ensnaring', 'ensnared', 'ensnared', tran, ['6A','14'] ).
verb( 'ensue', 'ensues', 'ensuing', 'ensued', 'ensued', intran, ['2A','3A'] ).
verb( 'ensure', 'ensures', 'ensuring', 'ensured', 'ensured', _, ['3A','9','12A','13A','14'] ).
verb( 'entail', 'entails', 'entailing', 'entailed', 'entailed', tran, ['6A','14'] ).
verb( 'entangle', 'entangles', 'entangling', 'entangled', 'entangled', tran, ['6A','14','15A'] ).
verb( 'enter', 'enters', 'entering', 'entered', 'entered', _, ['2A','3A','6A','14','15B'] ).
verb( 'entertain', 'entertains', 'entertaining', 'entertained', 'entertained', tran, ['2A','6A','14'] ).
verb( 'enthral', 'enthrals', 'enthralling', 'enthralled', 'enthralled', tran, ['6A'] ).
verb( 'enthrall', 'enthralls', 'enthralling', 'enthralled', 'enthralled', tran, ['6A'] ).
verb( 'enthrone', 'enthrones', 'enthroning', 'enthroned', 'enthroned', tran, ['6A'] ).
verb( 'enthuse', 'enthuses', 'enthusing', 'enthused', 'enthused', intran, ['3A'] ).
verb( 'entice', 'entices', 'enticing', 'enticed', 'enticed', tran, ['6A','15A','17'] ).
verb( 'entitle', 'entitles', 'entitling', 'entitled', 'entitled', tran, ['14','17'] ).
verb( 'entomb', 'entombs', 'entombing', 'entombed', 'entombed', tran, ['6A'] ).
verb( 'entrain', 'entrains', 'entraining', 'entrained', 'entrained', _, ['2A','6A'] ).
verb( 'entrance', 'entrances', 'entrancing', 'entranced', 'entranced', tran, ['6A'] ).
verb( 'entrap', 'entraps', 'entrapping', 'entrapped', 'entrapped', tran, ['6A'] ).
verb( 'entreat', 'entreats', 'entreating', 'entreated', 'entreated', tran, ['6A','14','17'] ).
verb( 'entrench', 'entrenches', 'entrenching', 'entrenched', 'entrenched', tran, ['6A'] ).
verb( 'entrust', 'entrusts', 'entrusting', 'entrusted', 'entrusted', tran, ['14'] ).
verb( 'entwine', 'entwines', 'entwining', 'entwined', 'entwined', tran, ['6A','14'] ).
verb( 'enumerate', 'enumerates', 'enumerating', 'enumerated', 'enumerated', tran, ['6A'] ).
verb( 'enunciate', 'enunciates', 'enunciating', 'enunciated', 'enunciated', _, ['2A','6A'] ).
verb( 'envelop', 'envelops', 'enveloping', 'enveloped', 'enveloped', tran, ['6A','14'] ).
verb( 'envenom', 'envenoms', 'envenoming', 'envenomed', 'envenomed', tran, ['6A'] ).
verb( 'environ', 'environs', 'environing', 'environed', 'environed', tran, ['6A'] ).
verb( 'envisage', 'envisages', 'envisaging', 'envisaged', 'envisaged', tran, ['6A'] ).
verb( 'envy', 'envies', 'envying', 'envied', 'envied', tran, ['6A','12C'] ).
verb( 'enwrap', 'enwraps', 'enwrapping', 'enwrapped', 'enwrapped', tran, [] ).
verb( 'epitomize', 'epitomizes', 'epitomizing', 'epitomized', 'epitomized', tran, ['6A'] ).
verb( 'equal', 'equals', 'equalling', 'equalled', 'equalled', tran, ['6A','15A'] ).
verb( 'equalize', 'equalizes', 'equalizing', 'equalized', 'equalized', tran, ['6A'] ).
verb( 'equate', 'equates', 'equating', 'equated', 'equated', tran, ['6A','14'] ).
verb( 'equip', 'equips', 'equipping', 'equipped', 'equipped', tran, ['6A','14'] ).
verb( 'eradicate', 'eradicates', 'eradicating', 'eradicated', 'eradicated', tran, ['6A'] ).
verb( 'erase', 'erases', 'erasing', 'erased', 'erased', tran, ['6A'] ).
verb( 'erect', 'erects', 'erecting', 'erected', 'erected', tran, ['6A'] ).
verb( 'erode', 'erodes', 'eroding', 'eroded', 'eroded', tran, ['6A'] ).
verb( 'err', 'errs', 'erring', 'erred', 'erred', intran, ['2A','2C'] ).
verb( 'erupt', 'erupts', 'erupting', 'erupted', 'erupted', intran, ['2A'] ).
verb( 'escalate', 'escalates', 'escalating', 'escalated', 'escalated', _, ['2A','6A'] ).
verb( 'escape', 'escapes', 'escaping', 'escaped', 'escaped', _, ['2A','3A','6A','6C'] ).
verb( 'eschew', 'eschews', 'eschewing', 'eschewed', 'eschewed', tran, ['6A'] ).
verb( 'escort', 'escorts', 'escorting', 'escorted', 'escorted', tran, ['6A','15B'] ).
verb( 'espouse', 'espouses', 'espousing', 'espoused', 'espoused', tran, ['6A'] ).
verb( 'espy', 'espies', 'espying', 'espied', 'espied', tran, ['6A'] ).
verb( 'essay', 'essays', 'essaying', 'essayed', 'essayed', _, ['4A','6A'] ).
verb( 'establish', 'establishes', 'establishing', 'established', 'established', tran, ['6A','14','16B'] ).
verb( 'esteem', 'esteems', 'esteeming', 'esteemed', 'esteemed', tran, ['6A','25'] ).
verb( 'estimate', 'estimates', 'estimating', 'estimated', 'estimated', _, ['3A','9','14'] ).
verb( 'estrange', 'estranges', 'estranging', 'estranged', 'estranged', tran, ['6A','14'] ).
verb( 'etch', 'etches', 'etching', 'etched', 'etched', _, ['2A','6A'] ).
verb( 'eulogize', 'eulogizes', 'eulogizing', 'eulogized', 'eulogized', tran, ['6A'] ).
verb( 'evacuate', 'evacuates', 'evacuating', 'evacuated', 'evacuated', tran, ['6A','14'] ).
verb( 'evade', 'evades', 'evading', 'evaded', 'evaded', tran, ['6A','6C'] ).
verb( 'evaluate', 'evaluates', 'evaluating', 'evaluated', 'evaluated', tran, ['6A'] ).
verb( 'evaporate', 'evaporates', 'evaporating', 'evaporated', 'evaporated', _, ['2A','6A'] ).
verb( 'even', 'evens', 'evening', 'evened', 'evened', tran, ['6A','15B'] ).
verb( 'evict', 'evicts', 'evicting', 'evicted', 'evicted', tran, ['14'] ).
verb( 'evidence', 'evidences', 'evidencing', 'evidenced', 'evidenced', tran, ['6A'] ).
verb( 'evince', 'evinces', 'evincing', 'evinced', 'evinced', tran, ['6A','9'] ).
verb( 'eviscerate', 'eviscerates', 'eviscerating', 'eviscerated', 'eviscerated', tran, ['6A'] ).
verb( 'evoke', 'evokes', 'evoking', 'evoked', 'evoked', tran, ['6A'] ).
verb( 'evolve', 'evolves', 'evolving', 'evolved', 'evolved', _, ['2A','6A'] ).
verb( 'exacerbate', 'exacerbates', 'exacerbating', 'exacerbated', 'exacerbated', tran, ['6A'] ).
verb( 'exact', 'exacts', 'exacting', 'exacted', 'exacted', tran, ['6A','14'] ).
verb( 'exaggerate', 'exaggerates', 'exaggerating', 'exaggerated', 'exaggerated', _, ['2A','6A'] ).
verb( 'exalt', 'exalts', 'exalting', 'exalted', 'exalted', tran, ['6A'] ).
verb( 'examine', 'examines', 'examining', 'examined', 'examined', tran, ['6A','14'] ).
verb( 'exasperate', 'exasperates', 'exasperating', 'exasperated', 'exasperated', tran, ['6A'] ).
verb( 'excavate', 'excavates', 'excavating', 'excavated', 'excavated', tran, ['6A'] ).
verb( 'exceed', 'exceeds', 'exceeding', 'exceeded', 'exceeded', tran, ['6A'] ).
verb( 'excel', 'excels', 'excelling', 'excelled', 'excelled', _, ['2C','3A','6A','15A'] ).
verb( 'except', 'excepts', 'excepting', 'excepted', 'excepted', tran, ['6A','14'] ).
verb( 'exchange', 'exchanges', 'exchanging', 'exchanged', 'exchanged', tran, ['6A','14'] ).
verb( 'excise', 'excises', 'excising', 'excised', 'excised', tran, ['6A'] ).
verb( 'excite', 'excites', 'exciting', 'excited', 'excited', tran, ['6A','14','17'] ).
verb( 'exclaim', 'exclaims', 'exclaiming', 'exclaimed', 'exclaimed', _, ['2A','9'] ).
verb( 'exclude', 'excludes', 'excluding', 'excluded', 'excluded', tran, ['6A','14'] ).
verb( 'excogitate', 'excogitates', 'excogitating', 'excogitated', 'excogitated', tran, ['6A'] ).
verb( 'excommunicate', 'excommunicates', 'excommunicating', 'excommunicated', 'excommunicated', tran, ['6A'] ).
verb( 'excoriate', 'excoriates', 'excoriating', 'excoriated', 'excoriated', tran, ['6A'] ).
verb( 'excrete', 'excretes', 'excreting', 'excreted', 'excreted', tran, ['6A'] ).
verb( 'exculpate', 'exculpates', 'exculpating', 'exculpated', 'exculpated', tran, ['6A','14'] ).
verb( 'excuse', 'excuses', 'excusing', 'excused', 'excused', tran, ['6A','6C','12B','13B','14','19C'] ).
verb( 'execrate', 'execrates', 'execrating', 'execrated', 'execrated', tran, ['6A'] ).
verb( 'execute', 'executes', 'executing', 'executed', 'executed', tran, ['6A'] ).
verb( 'exemplify', 'exemplifies', 'exemplifying', 'exemplified', 'exemplified', tran, ['6A'] ).
verb( 'exempt', 'exempts', 'exempting', 'exempted', 'exempted', tran, ['6A','14'] ).
verb( 'exercise', 'exercises', 'exercising', 'exercised', 'exercised', _, ['2A','6A','15A'] ).
verb( 'exert', 'exerts', 'exerting', 'exerted', 'exerted', tran, ['6A','14','16A'] ).
verb( 'exhale', 'exhales', 'exhaling', 'exhaled', 'exhaled', _, ['2A','6A'] ).
verb( 'exhaust', 'exhausts', 'exhausting', 'exhausted', 'exhausted', tran, ['6A'] ).
verb( 'exhibit', 'exhibits', 'exhibiting', 'exhibited', 'exhibited', tran, ['2A','6A'] ).
verb( 'exhilarate', 'exhilarates', 'exhilarating', 'exhilarated', 'exhilarated', tran, ['6A'] ).
verb( 'exhort', 'exhorts', 'exhorting', 'exhorted', 'exhorted', tran, ['6A','14','17'] ).
verb( 'exhume', 'exhumes', 'exhuming', 'exhumed', 'exhumed', tran, ['6A'] ).
verb( 'exile', 'exiles', 'exiling', 'exiled', 'exiled', tran, ['6A','15A'] ).
verb( 'exist', 'exists', 'existing', 'existed', 'existed', intran, ['2A','2C','3A'] ).
verb( 'exit', 'exits', 'exiting', 'exited', 'exited', intran, [] ).
verb( 'exonerate', 'exonerates', 'exonerating', 'exonerated', 'exonerated', tran, ['6A','14'] ).
verb( 'exorcize', 'exorcizes', 'exorcizing', 'exorcized', 'exorcized', tran, ['6A','14'] ).
verb( 'expand', 'expands', 'expanding', 'expanded', 'expanded', _, ['2A','2C','6A','14'] ).
verb( 'expatiate', 'expatiates', 'expatiating', 'expatiated', 'expatiated', intran, ['3A'] ).
verb( 'expatriate', 'expatriates', 'expatriating', 'expatriated', 'expatriated', tran, ['6A'] ).
verb( 'expect', 'expects', 'expecting', 'expected', 'expected', tran, ['6A','7A','9','14','17'] ).
verb( 'expectorate', 'expectorates', 'expectorating', 'expectorated', 'expectorated', _, ['2A','6A'] ).
verb( 'expedite', 'expedites', 'expediting', 'expedited', 'expedited', tran, ['6A'] ).
verb( 'expel', 'expels', 'expelling', 'expelled', 'expelled', tran, ['6A','14'] ).
verb( 'expend', 'expends', 'expending', 'expended', 'expended', tran, ['6A','14'] ).
verb( 'experience', 'experiences', 'experiencing', 'experienced', 'experienced', tran, ['6A'] ).
verb( 'experiment', 'experiments', 'experimenting', 'experimented', 'experimented', intran, ['2A','2C','3A'] ).
verb( 'expiate', 'expiates', 'expiating', 'expiated', 'expiated', tran, ['6A'] ).
verb( 'expire', 'expires', 'expiring', 'expired', 'expired', intran, ['2A'] ).
verb( 'explain', 'explains', 'explaining', 'explained', 'explained', tran, ['6A','8','9','10','14','15B'] ).
verb( 'explicate', 'explicates', 'explicating', 'explicated', 'explicated', tran, ['6A'] ).
verb( 'explode', 'explodes', 'exploding', 'exploded', 'exploded', _, ['2A','2C','6A'] ).
verb( 'exploit', 'exploits', 'exploiting', 'exploited', 'exploited', tran, ['6A'] ).
verb( 'explore', 'explores', 'exploring', 'explored', 'explored', tran, ['6A'] ).
verb( 'export', 'exports', 'exporting', 'exported', 'exported', tran, ['6A'] ).
verb( 'expose', 'exposes', 'exposing', 'exposed', 'exposed', tran, ['6A','14','15A'] ).
verb( 'expostulate', 'expostulates', 'expostulating', 'expostulated', 'expostulated', intran, ['2A','3A'] ).
verb( 'expound', 'expounds', 'expounding', 'expounded', 'expounded', tran, ['6A','14'] ).
verb( 'express', 'expresses', 'expressing', 'expressed', 'expressed', tran, ['6A','10','14','15A'] ).
verb( 'expropriate', 'expropriates', 'expropriating', 'expropriated', 'expropriated', tran, ['6A','14'] ).
verb( 'expunge', 'expunges', 'expunging', 'expunged', 'expunged', tran, ['6A','14'] ).
verb( 'expurgate', 'expurgates', 'expurgating', 'expurgated', 'expurgated', tran, ['6A'] ).
verb( 'extemporize', 'extemporize', 'extemporize', 'extemporize', 'extemporize', _, ['2A','6A'] ).
verb( 'extend', 'extends', 'extending', 'extended', 'extended', _, ['2B','2C','6A','14','15A'] ).
verb( 'extenuate', 'extenuates', 'extenuating', 'extenuated', 'extenuated', tran, ['6A'] ).
verb( 'exteriorize', 'exteriorizes', 'exteriorizing', 'exteriorized', 'exteriorized', tran, [] ).
verb( 'exterminate', 'exterminates', 'exterminating', 'exterminated', 'exterminated', tran, ['6A'] ).
verb( 'externalize', 'externalizes', 'externalizing', 'externalized', 'externalized', tran, ['6A'] ).
verb( 'extinguish', 'extinguishes', 'extinguishing', 'extinguished', 'extinguished', tran, ['6A'] ).
verb( 'extirpate', 'extirpates', 'extirpating', 'extirpated', 'extirpated', tran, ['6A'] ).
verb( 'extol', 'extols', 'extolling', 'extolled', 'extolled', tran, ['6A','15A'] ).
verb( 'extort', 'extorts', 'extorting', 'extorted', 'extorted', tran, ['6A','14'] ).
verb( 'extract', 'extracts', 'extracting', 'extracted', 'extracted', tran, ['6A','14'] ).
verb( 'extradite', 'extradites', 'extraditing', 'extradited', 'extradited', tran, ['6A'] ).
verb( 'extrapolate', 'extrapolate', 'extrapolate', 'extrapolate', 'extrapolate', _, ['6A'] ).
verb( 'extricate', 'extricates', 'extricating', 'extricated', 'extricated', tran, ['6A','14'] ).
verb( 'extrude', 'extrudes', 'extruding', 'extruded', 'extruded', tran, ['6A','14'] ).
verb( 'exude', 'exudes', 'exuding', 'exuded', 'exuded', _, ['2A','2C','6A'] ).
verb( 'exult', 'exults', 'exulting', 'exulted', 'exulted', intran, ['2A','3A','4C'] ).
verb( 'eye', 'eyes', 'eying', 'eyed', 'eyed', tran, ['6A','15A'] ).
verb( 'f^ete', 'f^etes', 'f^eting', 'f^eted', 'f^eted', tran, ['6A'] ).
verb( 'fabricate', 'fabricates', 'fabricating', 'fabricated', 'fabricated', tran, ['6A'] ).
verb( 'face', 'faces', 'facing', 'faced', 'faced', _, ['2C','6A','6C','14','15B'] ).
verb( 'facilitate', 'facilitates', 'facilitating', 'facilitated', 'facilitated', tran, ['6A'] ).
verb( 'factorize', 'factorizes', 'factorizing', 'factorized', 'factorized', tran, ['6A'] ).
verb( 'fade', 'fades', 'fading', 'faded', 'faded', _, ['2A','2C','3A','6A','15B'] ).
verb( 'fag', 'fags', 'fagging', 'fagged', 'fagged', _, ['2A','2C','3A','6A','15B'] ).
verb( 'fail', 'fails', 'failing', 'failed', 'failed', _, ['2A','2C','3A','4A','6A'] ).
verb( 'faint', 'faints', 'fainting', 'fainted', 'fainted', intran, ['2A','2C'] ).
verb( 'fake', 'fakes', 'faking', 'faked', 'faked', tran, ['6A','15B'] ).
verb( 'fall', 'falls', 'falling', 'fell', 'fallen', intran, ['2A','2B','2C','2D','3A'] ).
verb( 'falsify', 'falsifies', 'falsifying', 'falsified', 'falsified', tran, ['6A'] ).
verb( 'falter', 'falters', 'faltering', 'faltered', 'faltered', _, ['2A','2C','15B'] ).
verb( 'familiarize', 'familiarizes', 'familiarizing', 'familiarized', 'familiarized', tran, ['6A','14'] ).
verb( 'famish', 'famishes', 'famishing', 'famished', 'famished', _, ['2A','3A','6A'] ).
verb( 'fan', 'fans', 'fanning', 'fanned', 'fanned', _, ['2C','6A'] ).
verb( 'fancy', 'fancies', 'fancying', 'fancied', 'fancied', tran, ['6A','6C','9','16B','19C','25'] ).
verb( 'fare', 'fares', 'faring', 'fared', 'fared', intran, ['2C'] ).
verb( 'farm', 'farms', 'farming', 'farmed', 'farmed', _, ['2A','6A','15B'] ).
verb( 'farrow', 'farrows', 'farrowing', 'farrowed', 'farrowed', intran, [] ).
verb( 'fart', 'farts', 'farting', 'farted', 'farted', intran, [] ).
verb( 'fascinate', 'fascinates', 'fascinating', 'fascinated', 'fascinated', tran, ['6A'] ).
verb( 'fashion', 'fashions', 'fashioning', 'fashioned', 'fashioned', tran, ['6A','15A'] ).
verb( 'fast', 'fasts', 'fasting', 'fasted', 'fasted', intran, ['2A','2B'] ).
verb( 'fasten', 'fastens', 'fastening', 'fastened', 'fastened', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'fat', 'fats', 'fatting', 'fatted', 'fatted', tran, [] ).
verb( 'fate', 'fates', 'fating', 'fated', 'fated', tran, ['17'] ).
verb( 'father', 'fathers', 'fathering', 'fathered', 'fathered', tran, ['6A','14'] ).
verb( 'fathom', 'fathoms', 'fathoming', 'fathomed', 'fathomed', tran, ['6A'] ).
verb( 'fatigue', 'fatigues', 'fatiguing', 'fatigued', 'fatigued', tran, ['6A'] ).
verb( 'fatten', 'fattens', 'fattening', 'fattened', 'fattened', _, ['2A','2C','6A','15B'] ).
verb( 'fault', 'faults', 'faulting', 'faulted', 'faulted', tran, ['6A'] ).
verb( 'favour', 'favours', 'favouring', 'favoured', 'favoured', tran, ['6A','14'] ).
verb( 'fawn', 'fawns', 'fawning', 'fawned', 'fawned', intran, ['2A','3A'] ).
verb( 'fear', 'fears', 'fearing', 'feared', 'feared', _, ['2A','3A','4A','6A','6C','9'] ).
verb( 'feast', 'feasts', 'feasting', 'feasted', 'feasted', _, ['2A','2B','6A','14'] ).
verb( 'feather', 'feathers', 'feathering', 'feathered', 'feathered', tran, ['6A'] ).
verb( 'featherbed', 'featherbeds', 'featherbedding', 'featherbedded', 'featherbedded', tran, [] ).
verb( 'feature', 'features', 'featuring', 'featured', 'featured', tran, ['6A'] ).
verb( 'federate', 'federates', 'federating', 'federated', 'federated', _, ['2A','6A'] ).
verb( 'fee', 'fees', 'feeing', 'feed', 'feed', tran, ['6A'] ).
verb( 'feed', 'feeds', 'feeding', 'fed', 'fed', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'feel', 'feels', 'feeling', 'felt', 'felt', _, ['2A','2C','2D','3A','6A','6C','9','10','15B','18A','19A','25'] ).
verb( 'feign', 'feigns', 'feigning', 'feigned', 'feigned', tran, ['6A','9'] ).
verb( 'feint', 'feints', 'feinting', 'feinted', 'feinted', intran, ['2A','3A'] ).
verb( 'felicitate', 'felicitates', 'felicitating', 'felicitated', 'felicitated', tran, ['6A','14'] ).
verb( 'fell', 'fells', 'felling', 'felled', 'felled', tran, ['2A','2B','2C','2D','3A','6A'] ).
verb( 'fence', 'fences', 'fencing', 'fenced', 'fenced', tran, ['2A','2C','3A','6A','15B'] ).
verb( 'fend', 'fends', 'fending', 'fended', 'fended', _, ['15B'] ).
verb( 'ferment', 'ferments', 'fermenting', 'fermented', 'fermented', _, ['2A','6A'] ).
verb( 'ferret', 'ferrets', 'ferreting', 'ferreted', 'ferreted', _, ['2A','2C','15B'] ).
verb( 'ferry', 'ferries', 'ferrying', 'ferried', 'ferried', _, ['2A','2C','6A','15A','15B'] ).
verb( 'fertilize', 'fertilizes', 'fertilizing', 'fertilized', 'fertilized', tran, ['6A'] ).
verb( 'fester', 'festers', 'festering', 'festered', 'festered', intran, ['2A'] ).
verb( 'festoon', 'festoons', 'festooning', 'festooned', 'festooned', tran, ['6A'] ).
verb( 'fetch', 'fetches', 'fetching', 'fetched', 'fetched', _, ['6A','12B','12C','13B','15A','15B'] ).
verb( 'fetter', 'fetters', 'fettering', 'fettered', 'fettered', tran, ['6A'] ).
verb( 'fib', 'fibs', 'fibbing', 'fibbed', 'fibbed', intran, ['2A'] ).
verb( 'fiddle', 'fiddles', 'fiddling', 'fiddled', 'fiddled', tran, ['2A','2C','6A'] ).
verb( 'fidget', 'fidgets', 'fidgeting', 'fidgeted', 'fidgeted', _, ['2A','2C','6A'] ).
verb( 'field', 'fields', 'fielding', 'fielded', 'fielded', _, ['2A','6A'] ).
verb( 'fight', 'fights', 'fighting', 'fought', 'fought', _, ['2A','2B','2C','3A','4A','6A','15A','15B'] ).
verb( 'figure', 'figures', 'figuring', 'figured', 'figured', _, ['2C','3A','9','15A','15B','25'] ).
verb( 'filch', 'filches', 'filching', 'filched', 'filched', tran, ['6A'] ).
verb( 'file', 'files', 'filing', 'filed', 'filed', _, ['2C','6A','15A','15B','22'] ).
verb( 'filibuster', 'filibusters', 'filibustering', 'filibustered', 'filibustered', intran, [] ).
verb( 'fill', 'fills', 'filling', 'filled', 'filled', _, ['2A','2C','6A','12B','13B','14','15B'] ).
verb( 'fillet', 'fillets', 'filleting', 'filleted', 'filleted', tran, ['6A'] ).
verb( 'film', 'films', 'filming', 'filmed', 'filmed', _, ['2A','2C','6A'] ).
verb( 'filter', 'filters', 'filtering', 'filtered', 'filtered', _, ['2A','2C','6A','14','15B'] ).
verb( 'filtrate', 'filtrates', 'filtrating', 'filtrated', 'filtrated', _, [] ).
verb( 'finalize', 'finalizes', 'finalizing', 'finalized', 'finalized', tran, ['6A'] ).
verb( 'finance', 'finances', 'financing', 'financed', 'financed', tran, ['6A'] ).
verb( 'find', 'finds', 'finding', 'found', 'found', tran, ['6A','8','9','10','12A','12B','13A','13B','15A','15B','19B','22','25'] ).
verb( 'fine', 'fines', 'fining', 'fined', 'fined', tran, ['6A','14'] ).
verb( 'finger', 'fingers', 'fingering', 'fingered', 'fingered', tran, ['6A'] ).
verb( 'finish', 'finishes', 'finishing', 'finished', 'finished', _, ['2A','2C','6A','6C','15B','24B'] ).
verb( 'fire', 'fires', 'firing', 'fired', 'fired', _, ['2A','2C','3A','6A','15B'] ).
verb( 'firm', 'firms', 'firming', 'firmed', 'firmed', _, [] ).
verb( 'fish', 'fishes', 'fishing', 'fished', 'fished', _, ['2A','2C','6A','15A','15B'] ).
verb( 'fit', 'fits', 'fitting', 'fitted', 'fitted', _, ['2A','2C','6A','14','15A','15B','16A'] ).
verb( 'fix', 'fixes', 'fixing', 'fixed', 'fixed', _, ['3A','6A','14','15A','15B'] ).
verb( 'fixate', 'fixates', 'fixating', 'fixated', 'fixated', tran, ['6A'] ).
verb( 'fizz', 'fizzes', 'fizzing', 'fizzed', 'fizzed', intran, ['2A','2C'] ).
verb( 'fizzle', 'fizzles', 'fizzling', 'fizzled', 'fizzled', intran, ['2A','2C'] ).
verb( 'flabbergast', 'flabbergasts', 'flabbergasting', 'flabbergasted', 'flabbergasted', tran, ['6A'] ).
verb( 'flag', 'flags', 'flagging', 'flagged', 'flagged', _, ['2A','6A','15B'] ).
verb( 'flagellate', 'flagellates', 'flagellating', 'flagellated', 'flagellated', tran, ['6A'] ).
verb( 'flail', 'flails', 'flailing', 'flailed', 'flailed', tran, ['6A'] ).
verb( 'flake', 'flakes', 'flaking', 'flaked', 'flaked', intran, ['2A','2C'] ).
verb( 'flame', 'flames', 'flaming', 'flamed', 'flamed', intran, ['2A','2C'] ).
verb( 'flank', 'flanks', 'flanking', 'flanked', 'flanked', tran, ['6A'] ).
verb( 'flap', 'flaps', 'flapping', 'flapped', 'flapped', _, ['2A','2C','6A','15B'] ).
verb( 'flare', 'flares', 'flaring', 'flared', 'flared', _, ['2A','2C','6A'] ).
verb( 'flash', 'flashes', 'flashing', 'flashed', 'flashed', _, ['2A','2C','6A','12C','15A'] ).
verb( 'flatten', 'flattens', 'flattening', 'flattened', 'flattened', _, ['2A','2C','6A','15A','15B'] ).
verb( 'flatter', 'flatters', 'flattering', 'flattered', 'flattered', tran, ['6A'] ).
verb( 'flaunt', 'flaunts', 'flaunting', 'flaunted', 'flaunted', _, ['2A','2C','6A'] ).
verb( 'flavour', 'flavours', 'flavouring', 'flavoured', 'flavoured', tran, ['6A'] ).
verb( 'flay', 'flays', 'flaying', 'flayed', 'flayed', tran, ['6A'] ).
verb( 'fleck', 'flecks', 'flecking', 'flecked', 'flecked', tran, ['6A'] ).
verb( 'flee', 'flees', 'fleeing', 'fled', 'flown', _, ['2A','2C','6A'] ).
verb( 'fleece', 'fleeces', 'fleecing', 'fleeced', 'fleeced', tran, ['6A','14'] ).
verb( 'flex', 'flexes', 'flexing', 'flexed', 'flexed', tran, ['6A'] ).
verb( 'flick', 'flicks', 'flicking', 'flicked', 'flicked', tran, ['6A','15A','15B','22'] ).
verb( 'flicker', 'flickers', 'flickering', 'flickered', 'flickered', intran, ['2A','2C'] ).
verb( 'flight', 'flights', 'flighting', 'flighted', 'flighted', tran, ['6A'] ).
verb( 'flinch', 'flinches', 'flinching', 'flinched', 'flinched', intran, ['2A','3A'] ).
verb( 'fling', 'flings', 'flinging', 'flung', 'flung', _, ['2C','6A','12A','13A','15A','15B','22'] ).
verb( 'flip', 'flips', 'flipping', 'flipped', 'flipped', _, ['6A','15A','15B'] ).
verb( 'flirt', 'flirts', 'flirting', 'flirted', 'flirted', intran, ['2A','3A'] ).
verb( 'flit', 'flits', 'flitting', 'flitted', 'flitted', intran, ['2C'] ).
verb( 'float', 'floats', 'floating', 'floated', 'floated', _, ['2A','2C','6A','15A','15B'] ).
verb( 'flock', 'flocks', 'flocking', 'flocked', 'flocked', intran, ['2C','4A'] ).
verb( 'flog', 'flogs', 'flogging', 'flogged', 'flogged', tran, ['6A'] ).
verb( 'flood', 'floods', 'flooding', 'flooded', 'flooded', _, ['3A','6A','14','15B','16A'] ).
verb( 'floodlight', 'floodlights', 'floodlighting', 'floodlighted', 'floodlighted', tran, [] ).
verb( 'floor', 'floors', 'flooring', 'floored', 'floored', tran, ['6A'] ).
verb( 'flop', 'flops', 'flopping', 'flopped', 'flopped', _, ['2A','2C','15A','15B'] ).
verb( 'flounce', 'flounces', 'flouncing', 'flounced', 'flounced', _, ['2C','6A'] ).
verb( 'flounder', 'flounders', 'floundering', 'floundered', 'floundered', intran, ['2A','2C'] ).
verb( 'flour', 'flours', 'flouring', 'floured', 'floured', tran, ['6A'] ).
verb( 'flourish', 'flourishes', 'flourishing', 'flourished', 'flourished', _, ['2A','6A'] ).
verb( 'flout', 'flouts', 'flouting', 'flouted', 'flouted', tran, ['6A'] ).
verb( 'flow', 'flows', 'flowing', 'flowed', 'flowed', intran, ['2A','2C'] ).
verb( 'flower', 'flowers', 'flowering', 'flowered', 'flowered', intran, ['2A','2C'] ).
verb( 'fluctuate', 'fluctuates', 'fluctuating', 'fluctuated', 'fluctuated', intran, ['2A','2C'] ).
verb( 'fluff', 'fluffs', 'fluffing', 'fluffed', 'fluffed', tran, ['6A','15B'] ).
verb( 'flummox', 'flummoxes', 'flummoxing', 'flummoxed', 'flummoxed', tran, ['6A'] ).
verb( 'flunk', 'flunks', 'flunking', 'flunked', 'flunked', _, ['2A','2C','6A','15B'] ).
verb( 'fluoridate', 'fluoridates', 'fluoridating', 'fluoridated', 'fluoridated', tran, ['6A'] ).
verb( 'fluoridize', 'fluoridizes', 'fluoridizing', 'fluoridized', 'fluoridized', tran, [] ).
verb( 'flurry', 'flurries', 'flurrying', 'flurried', 'flurried', tran, ['6A'] ).
verb( 'flush', 'flushes', 'flushing', 'flushed', 'flushed', _, ['2A','2C','2D','6A','14'] ).
verb( 'fluster', 'flusters', 'flustering', 'flustered', 'flustered', tran, ['6A'] ).
verb( 'flute', 'flutes', 'fluting', 'fluted', 'fluted', _, ['6A'] ).
verb( 'flutter', 'flutters', 'fluttering', 'fluttered', 'fluttered', _, ['2A','2C','6A','15A','15B'] ).
verb( 'fly', 'flies', 'flying', 'flew', 'flew', _, ['2A','2B','2C','2D','4A','6A','15A','15B'] ).
verb( 'fly-fish', 'fly-fishes', 'fly-fishing', 'fly-fished', 'fly-fished', intran, ['2A'] ).
verb( 'foal', 'foals', 'foaling', 'foaled', 'foaled', intran, ['2A'] ).
verb( 'foam', 'foams', 'foaming', 'foamed', 'foamed', intran, ['2A','2C'] ).
verb( 'fob', 'fobs', 'fobbing', 'fobbed', 'fobbed', tran, ['15B'] ).
verb( 'focus', 'focuses', 'focusing', 'focused', 'focused', _, ['2A','2C','6A','14'] ).
verb( 'fog', 'fogs', 'fogging', 'fogged', 'fogged', tran, [] ).
verb( 'foil', 'foils', 'foiling', 'foiled', 'foiled', tran, ['6A'] ).
verb( 'foist', 'foists', 'foisting', 'foisted', 'foisted', tran, ['14','15A'] ).
verb( 'fold', 'folds', 'folding', 'folded', 'folded', _, ['2A','2C','6A','15A','15B'] ).
verb( 'follow', 'follows', 'following', 'followed', 'followed', _, ['2A','2B','2C','6A','15B'] ).
verb( 'foment', 'foments', 'fomenting', 'fomented', 'fomented', tran, ['6A'] ).
verb( 'fondle', 'fondles', 'fondling', 'fondled', 'fondled', tran, ['6A'] ).
verb( 'fool', 'fools', 'fooling', 'fooled', 'fooled', _, ['2A','2C','6A','14','15B'] ).
verb( 'foot', 'foots', 'footing', 'footed', 'footed', _, ['6A'] ).
verb( 'footle', 'footles', 'footling', 'footled', 'footled', _, [] ).
verb( 'footslog', 'footslogs', 'footslogging', 'footslogged', 'footslogged', intran, [] ).
verb( 'forage', 'forages', 'foraging', 'foraged', 'foraged', intran, ['2A','3A'] ).
verb( 'foray', 'forays', 'foraying', 'forayed', 'forayed', intran, ['2A'] ).
verb( 'forbear', 'forbears', 'forbearing', 'forbore', 'forborne', _, ['2A','3A','6C','7A'] ).
verb( 'forbid', 'forbids', 'forbidding', 'forbad', 'forbidden', tran, ['6A','12C','17'] ).
verb( 'force', 'forces', 'forcing', 'forced', 'forced', tran, ['6A','15A','15B','17','22'] ).
verb( 'force-feed', 'force-feeds', 'force-feeding', 'force-fed', 'force-fed', tran, ['6A'] ).
verb( 'force-land', 'force-lands', 'force-landing', 'force-landed', 'force-landed', _, [] ).
verb( 'ford', 'fords', 'fording', 'forded', 'forded', tran, ['6A'] ).
verb( 'forearm', 'forearms', 'forearming', 'forearmed', 'forearmed', tran, ['6A'] ).
verb( 'forebode', 'forebodes', 'foreboding', 'foreboded', 'foreboded', tran, ['6A','9'] ).
verb( 'forecast', 'forecasts', 'forecasting', 'forecasted', 'forecasted', tran, ['6A'] ).
verb( 'foreclose', 'forecloses', 'foreclosing', 'foreclosed', 'foreclosed', _, ['2A','3A','6A'] ).
verb( 'foredoom', 'foredooms', 'foredooming', 'foredoomed', 'foredoomed', tran, ['6A','14'] ).
verb( 'foregather', 'foregathers', 'foregathering', 'foregathered', 'foregathered', intran, ['2A','2C'] ).
verb( 'forego', 'foregoes', 'foregoing', 'forewent', 'foregone', _, [] ).
verb( 'foreknow', 'foreknows', 'foreknowing', 'foreknew', 'foreknown', _, ['2A','3A','6A','8','9','10','17','18B','25'] ).
verb( 'foreordain', 'foreordains', 'foreordaining', 'foreordained', 'foreordained', tran, ['6A','14','17'] ).
verb( 'foresee', 'foresees', 'foreseeing', 'foresaw', 'foreseen', tran, ['6A','9','10'] ).
verb( 'foreshadow', 'foreshadows', 'foreshadowing', 'foreshadowed', 'foreshadowed', tran, ['6A'] ).
verb( 'foreshorten', 'foreshortens', 'foreshortening', 'foreshortened', 'foreshortened', tran, ['6A'] ).
verb( 'forestall', 'forestalls', 'forestalling', 'forestalled', 'forestalled', tran, ['6A'] ).
verb( 'foreswear', 'foreswears', 'foreswearing', 'foreswore', 'foresworn', tran, ['6A'] ).
verb( 'foretell', 'foretells', 'foretelling', 'foretold', 'foretold', tran, ['6A','9','10','12A','13A'] ).
verb( 'forewarn', 'forewarns', 'forewarning', 'forewarned', 'forewarned', tran, ['6A'] ).
verb( 'forfeit', 'forfeits', 'forfeiting', 'forfeited', 'forfeited', tran, ['6A'] ).
verb( 'forgather', 'forgathers', 'forgathering', 'forgathered', 'forgathered', intran, ['2A','2C'] ).
verb( 'forge', 'forges', 'forging', 'forged', 'forged', _, ['2C','6A'] ).
verb( 'forget', 'forgets', 'forgetting', 'forgot', 'forgotten', _, ['2A','3A','6A','6C','6D','7A','8','9','10'] ).
verb( 'forgive', 'forgives', 'forgiving', 'forgave', 'forgiven', _, ['2A','6A','12C','14'] ).
verb( 'forgo', 'forgoes', 'forgoing', 'forwent', 'forgone', tran, [] ).
verb( 'fork', 'forks', 'forking', 'forked', 'forked', _, ['2A','2C','6A','15A','15B'] ).
verb( 'form', 'forms', 'forming', 'formed', 'formed', _, ['2A','2C','6A','14','15A'] ).
verb( 'formalize', 'formalizes', 'formalizing', 'formalized', 'formalized', tran, ['6A'] ).
verb( 'formulate', 'formulates', 'formulating', 'formulated', 'formulated', tran, ['6A'] ).
verb( 'fornicate', 'fornicates', 'fornicating', 'fornicated', 'fornicated', intran, ['21'] ).
verb( 'forsake', 'forsakes', 'forsaking', 'forsook', 'forsaken', tran, ['6A'] ).
verb( 'forswear', 'forswears', 'forswearing', 'forswore', 'forsworn', tran, ['6A'] ).
verb( 'fortify', 'fortifies', 'fortifying', 'fortified', 'fortified', tran, ['6A','14'] ).
verb( 'forward', 'forwards', 'forwarding', 'forwarded', 'forwarded', tran, ['6A','12A','13A','15A'] ).
verb( 'fossilize', 'fossilizes', 'fossilizing', 'fossilized', 'fossilized', _, ['2A','6A'] ).
verb( 'foster', 'fosters', 'fostering', 'fostered', 'fostered', tran, ['6A'] ).
verb( 'foul', 'fouls', 'fouling', 'fouled', 'fouled', _, ['2A','2C','6A'] ).
verb( 'found', 'founds', 'founding', 'founded', 'founded', tran, ['6A','8','9','10','12A','12B','13A','13B','14','15A','15B','19B','22','25'] ).
verb( 'founder', 'founders', 'foundering', 'foundered', 'foundered', _, ['2A','6A'] ).
verb( 'fowl', 'fowls', 'fowling', 'fowled', 'fowled', intran, [] ).
verb( 'fox', 'foxes', 'foxing', 'foxed', 'foxed', tran, ['6A'] ).
verb( 'foxhunt', 'foxhunts', 'foxhunting', 'foxhunted', 'foxhunted', intran, [] ).
verb( 'fracture', 'fractures', 'fracturing', 'fractured', 'fractured', _, ['2A','6A'] ).
verb( 'fragment', 'fragments', 'fragmenting', 'fragmented', 'fragmented', intran, ['2A'] ).
verb( 'frame', 'frames', 'framing', 'framed', 'framed', _, ['2A','2C','6A'] ).
verb( 'frank', 'franks', 'franking', 'franked', 'franked', tran, ['6A'] ).
verb( 'fraternize', 'fraternizes', 'fraternizing', 'fraternized', 'fraternized', intran, ['2A','2C','3A'] ).
verb( 'fray', 'frays', 'fraying', 'frayed', 'frayed', _, ['2A','2C','6A'] ).
verb( 'freak', 'freaks', 'freaking', 'freaked', 'freaked', _, ['2C','15B'] ).
verb( 'freckle', 'freckles', 'freckling', 'freckled', 'freckled', _, ['2A','6A'] ).
verb( 'free', 'frees', 'freeing', 'freed', 'freed', tran, ['6A','14'] ).
verb( 'freelance', 'freelances', 'freelancing', 'freelanced', 'freelanced', intran, [] ).
verb( 'freewheel', 'freewheels', 'freewheeling', 'freewheeled', 'freewheeled', intran, ['2A','2C'] ).
verb( 'freeze', 'freezes', 'freezing', 'froze', 'frozen', _, ['2A','2C','3A','6A','15B'] ).
verb( 'freight', 'freights', 'freighting', 'freighted', 'freighted', tran, ['6A','14'] ).
verb( 'frequent', 'frequents', 'frequenting', 'frequented', 'frequented', tran, ['6A'] ).
verb( 'fresco', 'frescos', 'frescoing', 'frescoed', 'frescoed', tran, [] ).
verb( 'freshen', 'freshens', 'freshening', 'freshened', 'freshened', _, ['2A','2C','6A','15B'] ).
verb( 'fret', 'frets', 'fretting', 'fretted', 'fretted', _, ['2A','2C','3A','6A','15A'] ).
verb( 'fricassee', 'fricassees', 'fricasseeing', 'fricasseed', 'fricasseed', tran, [] ).
verb( 'fright', 'frights', 'frighting', 'frighted', 'frighted', tran, [] ).
verb( 'frighten', 'frightens', 'frightening', 'frightened', 'frightened', tran, ['6A','14','15B'] ).
verb( 'fringe', 'fringes', 'fringing', 'fringed', 'fringed', tran, ['6A'] ).
verb( 'frisk', 'frisks', 'frisking', 'frisked', 'frisked', _, ['2A','2C','6A'] ).
verb( 'fritter', 'fritters', 'frittering', 'frittered', 'frittered', tran, ['15B'] ).
verb( 'frivol', 'frivols', 'frivolling', 'frivolled', 'frivolled', _, ['2A','15B'] ).
verb( 'frizz', 'frizzes', 'frizzing', 'frizzed', 'frizzed', tran, ['6A'] ).
verb( 'frizzle', 'frizzles', 'frizzling', 'frizzled', 'frizzled', _, ['2A','2C','6A','15B'] ).
verb( 'frogmarch', 'frogmarches', 'frogmarching', 'frogmarched', 'frogmarched', tran, ['6A'] ).
verb( 'frolic', 'frolics', 'frolicking', 'frolicked', 'frolicked', intran, ['2A','2C'] ).
verb( 'front', 'fronts', 'fronting', 'fronted', 'fronted', _, ['2A','2C','6A'] ).
verb( 'frost', 'frosts', 'frosting', 'frosted', 'frosted', _, ['2A','2C','6A'] ).
verb( 'froth', 'froths', 'frothing', 'frothed', 'frothed', intran, ['2A','2C'] ).
verb( 'frown', 'frowns', 'frowning', 'frowned', 'frowned', intran, ['2A','3A'] ).
verb( 'fructify', 'fructifies', 'fructifying', 'fructified', 'fructified', _, ['2A','6A'] ).
verb( 'fruit', 'fruits', 'fruiting', 'fruited', 'fruited', intran, [] ).
verb( 'frustrate', 'frustrates', 'frustrating', 'frustrated', 'frustrated', tran, ['6A','15A'] ).
verb( 'fry', 'fries', 'frying', 'fried', 'fried', _, ['2A','6A'] ).
verb( 'fuck', 'fucks', 'fucking', 'fucked', 'fucked', _, ['2A','6A'] ).
verb( 'fuddle', 'fuddles', 'fuddling', 'fuddled', 'fuddled', tran, ['6A','15A'] ).
verb( 'fuel', 'fuels', 'fuelling', 'fuelled', 'fuelled', _, ['2A','6A'] ).
verb( 'fulfil', 'fulfils', 'fulfilling', 'fulfilled', 'fulfilled', tran, ['6A'] ).
verb( 'fulminate', 'fulminates', 'fulminating', 'fulminated', 'fulminated', intran, ['2A','3A'] ).
verb( 'fumble', 'fumbles', 'fumbling', 'fumbled', 'fumbled', _, ['2A','2C','6A'] ).
verb( 'fume', 'fumes', 'fuming', 'fumed', 'fumed', _, ['2A','2C','3A','6A'] ).
verb( 'fumigate', 'fumigates', 'fumigating', 'fumigated', 'fumigated', tran, ['6A'] ).
verb( 'function', 'functions', 'functioning', 'functioned', 'functioned', intran, ['2A','2C'] ).
verb( 'fund', 'funds', 'funding', 'funded', 'funded', tran, ['6A'] ).
verb( 'funk', 'funks', 'funking', 'funked', 'funked', _, ['2A','6A'] ).
verb( 'funnel', 'funnels', 'funnelling', 'funnelled', 'funnelled', _, ['2A','6A'] ).
verb( 'furbish', 'furbishes', 'furbishing', 'furbished', 'furbished', tran, ['6A'] ).
verb( 'furl', 'furls', 'furling', 'furled', 'furled', _, ['2A','6A'] ).
verb( 'furnish', 'furnishes', 'furnishing', 'furnished', 'furnished', tran, ['6A','14'] ).
verb( 'furrow', 'furrows', 'furrowing', 'furrowed', 'furrowed', tran, ['6A'] ).
verb( 'further', 'furthers', 'furthering', 'furthered', 'furthered', tran, ['6A'] ).
verb( 'fuse', 'fuses', 'fusing', 'fused', 'fused', _, ['2A','6A','15A','15B'] ).
verb( 'fuss', 'fusses', 'fussing', 'fussed', 'fussed', _, ['2A','2C','6A'] ).
verb( 'gabble', 'gabbles', 'gabbling', 'gabbled', 'gabbled', _, ['2A','2C','6A','15B'] ).
verb( 'gad', 'gads', 'gadding', 'gadded', 'gadded', intran, ['2C'] ).
verb( 'gag', 'gags', 'gagging', 'gagged', 'gagged', _, ['2A','6A'] ).
verb( 'gage', 'gages', 'gaging', 'gaged', 'gaged', tran, ['6A'] ).
verb( 'gain', 'gains', 'gaining', 'gained', 'gained', _, ['2A','2B','2C','3A','6A','12B','13B','14'] ).
verb( 'gainsay', 'gainsays', 'gainsaying', 'gainsaid', 'gainsaid', tran, ['6A'] ).
verb( 'gall', 'galls', 'galling', 'galled', 'galled', tran, ['6A'] ).
verb( 'gallivant', 'gallivants', 'gallivanting', 'gallivanted', 'gallivanted', intran, ['2C'] ).
verb( 'gallop', 'gallops', 'galloping', 'galloped', 'galloped', _, ['2A','2B','2C','6A'] ).
verb( 'galumph', 'galumphs', 'galumphing', 'galumphed', 'galumphed', intran, [] ).
verb( 'galvanize', 'galvanizes', 'galvanizing', 'galvanized', 'galvanized', tran, ['6A','14'] ).
verb( 'gamble', 'gambles', 'gambling', 'gambled', 'gambled', _, ['2A','2B','2C','15B'] ).
verb( 'gambol', 'gambols', 'gambolling', 'gambolled', 'gambolled', intran, ['2A','2C'] ).
verb( 'game', 'games', 'gaming', 'gamed', 'gamed', _, ['2A','2C','15B'] ).
verb( 'gang', 'gangs', 'ganging', 'ganged', 'ganged', intran, ['2C'] ).
verb( 'gangrene', 'gangrenes', 'gangrening', 'gangrened', 'gangrened', _, ['2A','6A'] ).
verb( 'gaol', 'gaols', 'gaoling', 'gaoled', 'gaoled', tran, ['6A'] ).
verb( 'gape', 'gapes', 'gaping', 'gaped', 'gaped', intran, ['2A','2C'] ).
verb( 'garage', 'garages', 'garaging', 'garaged', 'garaged', tran, ['6A'] ).
verb( 'garb', 'garbs', 'garbing', 'garbed', 'garbed', tran, ['6A'] ).
verb( 'garble', 'garbles', 'garbling', 'garbled', 'garbled', tran, ['6A'] ).
verb( 'garden', 'gardens', 'gardening', 'gardened', 'gardened', intran, ['2A'] ).
verb( 'gargle', 'gargles', 'gargling', 'gargled', 'gargled', _, ['2A','6A'] ).
verb( 'garland', 'garlands', 'garlanding', 'garlanded', 'garlanded', tran, ['6A'] ).
verb( 'garner', 'garners', 'garnering', 'garnered', 'garnered', tran, ['6A','15B'] ).
verb( 'garnish', 'garnishes', 'garnishing', 'garnished', 'garnished', tran, ['6A','14'] ).
verb( 'garotte', 'garottes', 'garotting', 'garotted', 'garotted', tran, ['6A'] ).
verb( 'garrison', 'garrisons', 'garrisoning', 'garrisoned', 'garrisoned', tran, ['6A'] ).
verb( 'garrotte', 'garrottes', 'garrotting', 'garrotted', 'garrotted', tran, ['6A'] ).
verb( 'gas', 'gasses', 'gassing', 'gassed', 'gassed', _, ['2A','2C','6A'] ).
verb( 'gash', 'gashes', 'gashing', 'gashed', 'gashed', tran, ['6A'] ).
verb( 'gasify', 'gasifies', 'gasifying', 'gasified', 'gasified', _, ['2A','6A'] ).
verb( 'gasp', 'gasps', 'gasping', 'gasped', 'gasped', _, ['2A','2C','6A','15B'] ).
verb( 'gate', 'gates', 'gating', 'gated', 'gated', tran, ['6A'] ).
verb( 'gatecrash', 'gatecrashes', 'gatecrashing', 'gatecrashed', 'gatecrashed', tran, ['6A'] ).
verb( 'gather', 'gathers', 'gathering', 'gathered', 'gathered', _, ['2A','2C','6A','9','12B','13B','15A','15B'] ).
verb( 'gauge', 'gauges', 'gauging', 'gauged', 'gauged', tran, ['6A'] ).
verb( 'gawp', 'gawps', 'gawping', 'gawped', 'gawped', intran, ['2A','3A'] ).
verb( 'gaze', 'gazes', 'gazing', 'gazed', 'gazed', intran, ['2A','2C','3A'] ).
verb( 'gazette', 'gazettes', 'gazetting', 'gazetted', 'gazetted', tran, [] ).
verb( 'gazump', 'gazumps', 'gazumping', 'gazumped', 'gazumped', _, ['2A','6A'] ).
verb( 'gear', 'gears', 'gearing', 'geared', 'geared', _, ['2A','2C','3A','14','15A'] ).
verb( 'gel', 'gels', 'gelling', 'gelled', 'gelled', intran, [] ).
verb( 'geld', 'gelds', 'gelding', 'gelded', 'gelded', tran, ['6A'] ).
verb( 'gen', 'gens', 'genning', 'genned', 'genned', tran, ['15B'] ).
verb( 'generalize', 'generalizes', 'generalizing', 'generalized', 'generalized', _, ['2A','3A','6A','14'] ).
verb( 'generate', 'generates', 'generating', 'generated', 'generated', tran, ['6A'] ).
verb( 'genuflect', 'genuflects', 'genuflecting', 'genuflected', 'genuflected', intran, ['2A'] ).
verb( 'germinate', 'germinates', 'germinating', 'germinated', 'germinated', _, ['2A','6A'] ).
verb( 'gerrymander', 'gerrymanders', 'gerrymandering', 'gerrymandered', 'gerrymandered', tran, ['6A'] ).
verb( 'gesticulate', 'gesticulates', 'gesticulating', 'gesticulated', 'gesticulated', intran, ['2A'] ).
verb( 'gesture', 'gestures', 'gesturing', 'gestured', 'gestured', intran, ['2A'] ).
verb( 'get', 'gets', 'getting', 'got', 'got', _, ['2C','2D','2E','3A','4A','6A','7A','7B','12B','13B','14','15A','15B','17','19B','22','24C'] ).
verb( 'ghost', 'ghosts', 'ghosting', 'ghosted', 'ghosted', _, [] ).
verb( 'gibber', 'gibbers', 'gibbering', 'gibbered', 'gibbered', intran, ['2A','2C'] ).
verb( 'gibbet', 'gibbets', 'gibbeting', 'gibbeted', 'gibbeted', tran, [] ).
verb( 'gibe', 'gibes', 'gibing', 'gibed', 'gibed', intran, ['2A','3A'] ).
verb( 'gift', 'gifts', 'gifting', 'gifted', 'gifted', tran, ['6A'] ).
verb( 'giggle', 'giggles', 'giggling', 'giggled', 'giggled', intran, ['2A','6A'] ).
verb( 'gild', 'gilds', 'gilding', 'gilded', 'gilded', tran, ['6A'] ).
verb( 'gin', 'gins', 'ginning', 'ginned', 'ginned', tran, ['6A'] ).
verb( 'ginger', 'gingers', 'gingering', 'gingered', 'gingered', tran, ['6A','15B'] ).
verb( 'gird', 'girds', 'girding', 'girded', 'girded', tran, ['15B'] ).
verb( 'girdle', 'girdles', 'girdling', 'girdled', 'girdled', tran, ['15A','15B'] ).
verb( 'give', 'gives', 'giving', 'gave', 'given', _, ['2A','2C','3A','6A','12A','12B','13A','15B','16A'] ).
verb( 'gladden', 'gladdens', 'gladdening', 'gladdened', 'gladdened', tran, ['6A'] ).
verb( 'glamorize', 'glamorizes', 'glamorizing', 'glamorized', 'glamorized', tran, ['6A'] ).
verb( 'glance', 'glances', 'glancing', 'glanced', 'glanced', _, ['2C','3A','15A'] ).
verb( 'glare', 'glares', 'glaring', 'glared', 'glared', _, ['2A','2C','3A','6A','14'] ).
verb( 'glass', 'glasses', 'glassing', 'glassed', 'glassed', tran, ['6A','15B'] ).
verb( 'glaze', 'glazes', 'glazing', 'glazed', 'glazed', _, ['2A','2C','6A','15B'] ).
verb( 'gleam', 'gleams', 'gleaming', 'gleamed', 'gleamed', intran, ['2A','2C'] ).
verb( 'glean', 'gleans', 'gleaning', 'gleaned', 'gleaned', _, ['2A','6A'] ).
verb( 'glide', 'glides', 'gliding', 'glided', 'glided', intran, ['2A','2C'] ).
verb( 'glimmer', 'glimmers', 'glimmering', 'glimmered', 'glimmered', intran, ['2A','2C'] ).
verb( 'glimpse', 'glimpses', 'glimpsing', 'glimpsed', 'glimpsed', tran, ['6A','19A'] ).
verb( 'glint', 'glints', 'glinting', 'glinted', 'glinted', intran, [] ).
verb( 'glissade', 'glissades', 'glissading', 'glissaded', 'glissaded', intran, [] ).
verb( 'glisten', 'glistens', 'glistening', 'glistened', 'glistened', intran, ['2A','2C'] ).
verb( 'glister', 'glisters', 'glistering', 'glistered', 'glistered', intran, [] ).
verb( 'glitter', 'glitters', 'glittering', 'glittered', 'glittered', intran, ['2A','2C'] ).
verb( 'gloat', 'gloats', 'gloating', 'gloated', 'gloated', intran, ['2A','3A'] ).
verb( 'globetrot', 'globetrots', 'globetrotting', 'globetrotted', 'globetrotted', intran, ['2A'] ).
verb( 'glorify', 'glorifies', 'glorifying', 'glorified', 'glorified', tran, ['6A'] ).
verb( 'glory', 'glories', 'glorying', 'gloried', 'gloried', intran, ['3A'] ).
verb( 'gloss', 'glosses', 'glossing', 'glossed', 'glossed', tran, ['6A','15A'] ).
verb( 'glow', 'glows', 'glowing', 'glowed', 'glowed', intran, ['2A','2C'] ).
verb( 'glower', 'glowers', 'glowering', 'glowered', 'glowered', intran, ['2A','3A'] ).
verb( 'glue', 'glues', 'gluing', 'glued', 'glued', tran, ['6A','15A','15B'] ).
verb( 'glut', 'gluts', 'glutting', 'glutted', 'glutted', tran, ['6A','14'] ).
verb( 'gnash', 'gnashes', 'gnashing', 'gnashed', 'gnashed', _, ['2A','6A'] ).
verb( 'gnaw', 'gnaws', 'gnawing', 'gnawed', 'gnawed', _, ['3A','6A','15B'] ).
verb( 'go', 'goes', 'going', 'went', 'gone', intran, ['2A','2B','2C','2D','2E','3A','4A','6A','15B'] ).
verb( 'goad', 'goads', 'goading', 'goaded', 'goaded', tran, ['6A','14','15B','17'] ).
verb( 'gobble', 'gobbles', 'gobbling', 'gobbled', 'gobbled', _, ['2A','2C','6A','15B'] ).
verb( 'goggle', 'goggles', 'goggling', 'goggled', 'goggled', intran, ['2A','3A'] ).
verb( 'golf', 'golfs', 'golfing', 'golfed', 'golfed', intran, [] ).
verb( 'gong', 'gongs', 'gonging', 'gonged', 'gonged', tran, [] ).
verb( 'goof', 'goofs', 'goofing', 'goofed', 'goofed', _, ['2A','6A'] ).
verb( 'gore', 'gores', 'goring', 'gored', 'gored', tran, ['6A'] ).
verb( 'gorge', 'gorges', 'gorging', 'gorged', 'gorged', _, ['2A','2C','6A','14'] ).
verb( 'gormandize', 'gormandizes', 'gormandizing', 'gormandized', 'gormandized', intran, [] ).
verb( 'gossip', 'gossips', 'gossiping', 'gossiped', 'gossiped', intran, ['2A','2C'] ).
verb( 'gouge', 'gouges', 'gouging', 'gouged', 'gouged', tran, ['6A','15B'] ).
verb( 'govern', 'governs', 'governing', 'governed', 'governed', _, ['2A','6A'] ).
verb( 'gown', 'gowns', 'gowning', 'gowned', 'gowned', tran, [] ).
verb( 'grab', 'grabs', 'grabbing', 'grabbed', 'grabbed', _, ['3A','6A'] ).
verb( 'grace', 'graces', 'gracing', 'graced', 'graced', tran, ['6A'] ).
verb( 'grade', 'grades', 'grading', 'graded', 'graded', tran, ['6A','15B'] ).
verb( 'graduate', 'graduates', 'graduating', 'graduated', 'graduated', _, ['2A','2C','6A'] ).
verb( 'graft', 'grafts', 'grafting', 'grafted', 'grafted', _, ['2A','6A','15A','15B'] ).
verb( 'grant', 'grants', 'granting', 'granted', 'granted', tran, ['6A','9','12A','13A','25'] ).
verb( 'granulate', 'granulates', 'granulating', 'granulated', 'granulated', _, ['2A','6A'] ).
verb( 'grapple', 'grapples', 'grappling', 'grappled', 'grappled', intran, ['2A','2C','3A'] ).
verb( 'grasp', 'grasps', 'grasping', 'grasped', 'grasped', _, ['3A','6A'] ).
verb( 'grass', 'grasses', 'grassing', 'grassed', 'grassed', _, ['2A','3A','6A','15B'] ).
verb( 'grate', 'grates', 'grating', 'grated', 'grated', _, ['2A','3A','6A','15A'] ).
verb( 'gratify', 'gratifies', 'gratifying', 'gratified', 'gratified', tran, ['6A'] ).
verb( 'grave', 'graves', 'graving', 'graved', 'graven', tran, [] ).
verb( 'gravel', 'gravels', 'gravelling', 'gravelled', 'gravelled', tran, ['6A'] ).
verb( 'gravitate', 'gravitates', 'gravitating', 'gravitated', 'gravitated', intran, ['3A'] ).
verb( 'gray', 'grays', 'graying', 'grayed', 'grayed', _, ['2A','6A'] ).
verb( 'graze', 'grazes', 'grazing', 'grazed', 'grazed', _, ['2A','2C','6A'] ).
verb( 'grease', 'greases', 'greasing', 'greased', 'greased', tran, [] ).
verb( 'greet', 'greets', 'greeting', 'greeted', 'greeted', tran, ['6A','14'] ).
verb( 'grey', 'greys', 'greying', 'greyed', 'greyed', _, ['2A','6A'] ).
verb( 'grieve', 'grieves', 'grieving', 'grieved', 'grieved', _, ['2A','2C','6A'] ).
verb( 'grill', 'grills', 'grilling', 'grilled', 'grilled', _, ['2A','2C','6A'] ).
verb( 'grimace', 'grimaces', 'grimacing', 'grimaced', 'grimaced', intran, ['2A'] ).
verb( 'grime', 'grimes', 'griming', 'grimed', 'grimed', tran, ['6A'] ).
verb( 'grin', 'grins', 'grinning', 'grinned', 'grinned', _, ['2A','2C','6A'] ).
verb( 'grind', 'grinds', 'grinding', 'ground', 'ground', _, ['2A','2C','6A','15A','15B'] ).
verb( 'grip', 'grips', 'gripping', 'gripped', 'gripped', _, ['2A','6A'] ).
verb( 'grit', 'grits', 'gritting', 'gritted', 'gritted', tran, [] ).
verb( 'grizzle', 'grizzles', 'grizzling', 'grizzled', 'grizzled', intran, [] ).
verb( 'groan', 'groans', 'groaning', 'groaned', 'groaned', _, ['2A','2C','6A','15B'] ).
verb( 'groin', 'groins', 'groining', 'groined', 'groined', tran, [] ).
verb( 'groom', 'grooms', 'grooming', 'groomed', 'groomed', tran, ['6A'] ).
verb( 'groove', 'grooves', 'grooving', 'grooved', 'grooved', tran, [] ).
verb( 'grope', 'gropes', 'groping', 'groped', 'groped', _, ['2A','2C','3A','15A'] ).
verb( 'gross', 'grosses', 'grossing', 'grossed', 'grossed', tran, ['6A'] ).
verb( 'grouch', 'grouches', 'grouching', 'grouched', 'grouched', intran, [] ).
verb( 'ground', 'grounds', 'grounding', 'grounded', 'grounded', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'group', 'groups', 'grouping', 'grouped', 'grouped', _, ['2C','6A','15A','15B'] ).
verb( 'grouse', 'grouses', 'grousing', 'groused', 'groused', intran, ['2A','2C'] ).
verb( 'grovel', 'grovels', 'grovelling', 'grovelled', 'grovelled', intran, ['2A','2C'] ).
verb( 'grow', 'grows', 'growing', 'grew', 'grown', _, ['2A','2C','2D','3A','4A','6A','12B','13B'] ).
verb( 'growl', 'growls', 'growling', 'growled', 'growled', _, ['2A','2C','6A','15B'] ).
verb( 'grub', 'grubs', 'grubbing', 'grubbed', 'grubbed', _, ['2C','6A','15B'] ).
verb( 'grudge', 'grudges', 'grudging', 'grudged', 'grudged', tran, ['6C','12A','13A'] ).
verb( 'grumble', 'grumbles', 'grumbling', 'grumbled', 'grumbled', _, ['2A','2C','3A','6A','15B'] ).
verb( 'grunt', 'grunts', 'grunting', 'grunted', 'grunted', _, ['2A','6A','15B'] ).
verb( 'guarantee', 'guarantees', 'guaranteeing', 'guaranteed', 'guaranteed', tran, ['6A','7A','9','12A','13A','25'] ).
verb( 'guard', 'guards', 'guarding', 'guarded', 'guarded', _, ['3A','6A','15A'] ).
verb( 'guess', 'guesses', 'guessing', 'guessed', 'guessed', _, ['2A','2C','3A','6A','8','9','10','25'] ).
verb( 'guffaw', 'guffaws', 'guffawing', 'guffawed', 'guffawed', intran, [] ).
verb( 'guide', 'guides', 'guiding', 'guided', 'guided', tran, ['6A','15A','15B'] ).
verb( 'guillotine', 'guillotines', 'guillotining', 'guillotined', 'guillotined', tran, ['6A'] ).
verb( 'gull', 'gulls', 'gulling', 'gulled', 'gulled', tran, ['6A','15A'] ).
verb( 'gulp', 'gulps', 'gulping', 'gulped', 'gulped', _, ['2A','6A','15B'] ).
verb( 'gum', 'gums', 'gumming', 'gummed', 'gummed', tran, ['6A','15A','15B'] ).
verb( 'gun', 'guns', 'gunning', 'gunned', 'gunned', tran, ['6A','15B'] ).
verb( 'gurgle', 'gurgles', 'gurgling', 'gurgled', 'gurgled', intran, [] ).
verb( 'gush', 'gushes', 'gushing', 'gushed', 'gushed', intran, ['2A','2C','3A'] ).
verb( 'gut', 'guts', 'gutting', 'gutted', 'gutted', tran, ['6A'] ).
verb( 'gutter', 'gutters', 'guttering', 'guttered', 'guttered', intran, ['2A'] ).
verb( 'guy', 'guys', 'guying', 'guyed', 'guyed', tran, ['6A'] ).
verb( 'guzzle', 'guzzles', 'guzzling', 'guzzled', 'guzzled', _, ['2A','6A','15B'] ).
verb( 'gybe', 'gybes', 'gybing', 'gybed', 'gybed', _, ['2A','6A'] ).
verb( 'gyp', 'gyps', 'gypping', 'gypped', 'gypped', tran, ['6A'] ).
verb( 'gyrate', 'gyrates', 'gyrating', 'gyrated', 'gyrated', intran, [] ).
verb( 'h\'m', 'h\'ms', 'h\'mming', 'h\'mmed', 'h\'mmed', intran, [] ).
verb( 'habituate', 'habituates', 'habituating', 'habituated', 'habituated', tran, ['14'] ).
verb( 'hack', 'hacks', 'hacking', 'hacked', 'hacked', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'haggle', 'haggles', 'haggling', 'haggled', 'haggled', intran, ['2A','3A'] ).
verb( 'hail', 'hails', 'hailing', 'hailed', 'hailed', _, ['2C','3A','6A','15B','16B','23'] ).
verb( 'halloo', 'halloos', 'hallooing', 'hallooed', 'hallooed', intran, [] ).
verb( 'hallow', 'hallows', 'hallowing', 'hallowed', 'hallowed', tran, ['6A'] ).
verb( 'halt', 'halts', 'halting', 'halted', 'halted', _, ['2A','2C','6A'] ).
verb( 'halve', 'halves', 'halving', 'halved', 'halved', tran, ['6A'] ).
verb( 'ham', 'hams', 'hamming', 'hammed', 'hammed', _, ['2A','6A','15B'] ).
verb( 'hammer', 'hammers', 'hammering', 'hammered', 'hammered', _, ['2A','2C','3A','6A','15B','22'] ).
verb( 'hamper', 'hampers', 'hampering', 'hampered', 'hampered', tran, ['6A'] ).
verb( 'hamstring', 'hamstrings', 'hamstringing', 'hamstrung', 'hamstrung', tran, ['6A'] ).
verb( 'hand', 'hands', 'handing', 'handed', 'handed', tran, ['12A','13A','15A','15B'] ).
verb( 'handcuff', 'handcuffs', 'handcuffing', 'handcuffed', 'handcuffed', tran, [] ).
verb( 'handicap', 'handicaps', 'handicapping', 'handicapped', 'handicapped', tran, ['6A'] ).
verb( 'handle', 'handles', 'handling', 'handled', 'handled', tran, ['6A'] ).
verb( 'hang', 'hangs', 'hanging', 'hanged', 'hanged', _, ['2A','2B','2C','3A','6A','15A','15B'] ).
verb( 'hanker', 'hankers', 'hankering', 'hankered', 'hankered', intran, ['3A'] ).
verb( 'hap', 'haps', 'happing', 'happed', 'happed', intran, [] ).
verb( 'happen', 'happens', 'happening', 'happened', 'happened', intran, ['2A','3A','4E'] ).
verb( 'harangue', 'harangues', 'haranguing', 'harangued', 'harangued', _, ['2A','6A'] ).
verb( 'harass', 'harasses', 'harassing', 'harassed', 'harassed', tran, ['6A'] ).
verb( 'harbour', 'harbours', 'harbouring', 'harboured', 'harboured', _, ['2A','6A'] ).
verb( 'harden', 'hardens', 'hardening', 'hardened', 'hardened', _, ['2A','2C','6A','15B'] ).
verb( 'hare', 'hares', 'haring', 'hared', 'hared', intran, ['2C'] ).
verb( 'hark', 'harks', 'harking', 'harked', 'harked', intran, [] ).
verb( 'harm', 'harms', 'harming', 'harmed', 'harmed', tran, ['6A'] ).
verb( 'harmonize', 'harmonizes', 'harmonizing', 'harmonized', 'harmonized', _, ['2A','2C','6A','14'] ).
verb( 'harness', 'harnesses', 'harnessing', 'harnessed', 'harnessed', tran, ['6A'] ).
verb( 'harp', 'harps', 'harping', 'harped', 'harped', intran, [] ).
verb( 'harpoon', 'harpoons', 'harpooning', 'harpooned', 'harpooned', tran, [] ).
verb( 'harrow', 'harrows', 'harrowing', 'harrowed', 'harrowed', tran, ['6A'] ).
verb( 'harry', 'harries', 'harrying', 'harried', 'harried', tran, ['6A'] ).
verb( 'harvest', 'harvests', 'harvesting', 'harvested', 'harvested', tran, ['6A'] ).
verb( 'hash', 'hashes', 'hashing', 'hashed', 'hashed', tran, ['6A','15B'] ).
verb( 'hassle', 'hassles', 'hassling', 'hassled', 'hassled', _, ['2A','3A','6A'] ).
verb( 'hasten', 'hastens', 'hastening', 'hastened', 'hastened', _, ['2A','2C','4A','6A'] ).
verb( 'hatch', 'hatches', 'hatching', 'hatched', 'hatched', _, ['2A','6A'] ).
verb( 'hate', 'hates', 'hating', 'hated', 'hated', tran, ['6A','6D','7A','17','19C'] ).
verb( 'haul', 'hauls', 'hauling', 'hauled', 'hauled', _, ['2C','3A','6A','15A','15B'] ).
verb( 'haunt', 'haunts', 'haunting', 'haunted', 'haunted', tran, ['6A'] ).
verb( 'have', 'has', 'having', 'had', 'had', unknown, ['6A','6B','7B','15B','18C','19B','24B','24C'] ).
verb( 'have', 'has', 'having', 'had', 'had', _, ['6A','6B','7B','15B','18C','19B','24B','24C'] ).
verb( 'haw', 'haws', 'hawing', 'hawed', 'hawed', intran, [] ).
verb( 'hawk', 'hawks', 'hawking', 'hawked', 'hawked', tran, ['6A','15B'] ).
verb( 'hazard', 'hazards', 'hazarding', 'hazarded', 'hazarded', tran, ['6A'] ).
verb( 'haze', 'hazes', 'hazing', 'hazed', 'hazed', tran, [] ).
verb( 'head', 'heads', 'heading', 'headed', 'headed', _, ['2C','6A','15B'] ).
verb( 'heal', 'heals', 'healing', 'healed', 'healed', _, ['2A','2C','6A'] ).
verb( 'heap', 'heaps', 'heaping', 'heaped', 'heaped', tran, ['6A','14','15A','15B'] ).
verb( 'hear', 'hears', 'hearing', 'heard', 'heard', _, ['2A','3A','6A','9','10','15A','18A','19A','24A'] ).
verb( 'hearken', 'hearkens', 'hearkening', 'hearkened', 'hearkened', intran, [] ).
verb( 'hearten', 'heartens', 'heartening', 'heartened', 'heartened', tran, ['6A'] ).
verb( 'heat', 'heats', 'heating', 'heated', 'heated', _, ['2C','6A','15B'] ).
verb( 'heave', 'heaves', 'heaving', 'heaved', 'heaved', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'heckle', 'heckles', 'heckling', 'heckled', 'heckled', tran, ['6A'] ).
verb( 'hector', 'hectors', 'hectoring', 'hectored', 'hectored', _, [] ).
verb( 'hedge', 'hedges', 'hedging', 'hedged', 'hedged', _, ['2A','6A','15A','15B'] ).
verb( 'hedgehop', 'hedgehops', 'hedgehopping', 'hedgehopped', 'hedgehopped', intran, [] ).
verb( 'heed', 'heeds', 'heeding', 'heeded', 'heeded', tran, ['6A'] ).
verb( 'heel', 'heels', 'heeling', 'heeled', 'heeled', _, ['2A','2C','6A','15B'] ).
verb( 'heighten', 'heightens', 'heightening', 'heightened', 'heightened', _, ['2A','6A'] ).
verb( 'heliograph', 'heliographs', 'heliographing', 'heliographed', 'heliographed', tran, [] ).
verb( 'help', 'helps', 'helping', 'helped', 'helped', _, ['2A','2C','6A','6C','14','15A','15B','17','18B'] ).
verb( 'hem', 'hems', 'hemming', 'hemmed', 'hemmed', _, ['6A','15B'] ).
verb( 'hemstitch', 'hemstitches', 'hemstitching', 'hemstitched', 'hemstitched', tran, [] ).
verb( 'herald', 'heralds', 'heralding', 'heralded', 'heralded', tran, ['6A'] ).
verb( 'herd', 'herds', 'herding', 'herded', 'herded', _, ['2C','15A','15B'] ).
verb( 'hesitate', 'hesitates', 'hesitating', 'hesitated', 'hesitated', intran, ['2A','3A','3B','4C'] ).
verb( 'hew', 'hews', 'hewing', 'hewed', 'hewed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'hibernate', 'hibernates', 'hibernating', 'hibernated', 'hibernated', intran, ['2A'] ).
verb( 'hiccough', 'hiccoughs', 'hiccoughing', 'hiccoughed', 'hiccoughed', intran, [] ).
verb( 'hiccup', 'hiccups', 'hiccupping', 'hiccupped', 'hiccupped', intran, [] ).
verb( 'hide', 'hides', 'hiding', 'hid', 'hidden', _, ['2A','6A','14'] ).
verb( 'hie', 'hies', 'hieing', 'hied', 'hied', intran, [] ).
verb( 'highjack', 'highjacks', 'highjacking', 'highjacked', 'highjacked', tran, ['6A'] ).
verb( 'highlight', 'highlights', 'highlighting', 'highlighted', 'highlighted', tran, [] ).
verb( 'hijack', 'hijacks', 'hijacking', 'hijacked', 'hijacked', tran, ['6A'] ).
verb( 'hike', 'hikes', 'hiking', 'hiked', 'hiked', intran, [] ).
verb( 'hinder', 'hinders', 'hindering', 'hindered', 'hindered', tran, ['6A','6C','15A'] ).
verb( 'hinge', 'hinges', 'hinging', 'hinged', 'hinged', _, ['3A','6A'] ).
verb( 'hint', 'hints', 'hinting', 'hinted', 'hinted', _, ['3A','6A','9'] ).
verb( 'hire', 'hires', 'hiring', 'hired', 'hired', tran, ['6A','15B'] ).
verb( 'hiss', 'hisses', 'hissing', 'hissed', 'hissed', _, ['2A','3A','6A','15A'] ).
verb( 'hit', 'hits', 'hitting', 'hit', 'hit', _, ['2C','3A','6A','12C','15A','15B'] ).
verb( 'hitch', 'hitches', 'hitching', 'hitched', 'hitched', _, ['2A','2C','6A','15A','15B'] ).
verb( 'hitchhike', 'hitchhikes', 'hitchhiking', 'hitchhiked', 'hitchhiked', intran, ['2A'] ).
verb( 'hive', 'hives', 'hiving', 'hived', 'hived', _, ['2C','6A'] ).
verb( 'hoard', 'hoards', 'hoarding', 'hoarded', 'hoarded', _, ['6A','15B'] ).
verb( 'hoax', 'hoaxes', 'hoaxing', 'hoaxed', 'hoaxed', tran, ['6A','14'] ).
verb( 'hobble', 'hobbles', 'hobbling', 'hobbled', 'hobbled', _, ['2A','2C','6A'] ).
verb( 'hobnob', 'hobnobs', 'hobnobbing', 'hobnobbed', 'hobnobbed', intran, ['2A','2C','3A'] ).
verb( 'hock', 'hocks', 'hocking', 'hocked', 'hocked', tran, ['6A'] ).
verb( 'hoe', 'hoes', 'hoeing', 'hoed', 'hoed', _, ['2A','6A','15B'] ).
verb( 'hog', 'hogs', 'hogging', 'hogged', 'hogged', tran, ['6A'] ).
verb( 'hoist', 'hoists', 'hoisting', 'hoisted', 'hoisted', tran, ['6A','15B'] ).
verb( 'hold', 'holds', 'holding', 'held', 'held', _, ['2A','2C','2D','3A','6A','9','14','15A','15B','22','25'] ).
verb( 'hole', 'holes', 'holing', 'holed', 'holed', _, ['2C','6A','15B'] ).
verb( 'holiday', 'holidays', 'holidaying', 'holidayed', 'holidayed', intran, [] ).
verb( 'holler', 'hollers', 'hollering', 'hollered', 'hollered', _, [] ).
verb( 'hollow', 'hollows', 'hollowing', 'hollowed', 'hollowed', tran, ['6A','15A','15B'] ).
verb( 'holystone', 'holystones', 'holystoning', 'holystoned', 'holystoned', tran, [] ).
verb( 'homogenize', 'homogenizes', 'homogenizing', 'homogenized', 'homogenized', tran, ['6A'] ).
verb( 'hone', 'hones', 'honing', 'honed', 'honed', tran, ['6A'] ).
verb( 'honeycomb', 'honeycombs', 'honeycombing', 'honeycombed', 'honeycombed', tran, ['6A'] ).
verb( 'honeymoon', 'honeymoons', 'honeymooning', 'honeymooned', 'honeymooned', intran, [] ).
verb( 'honk', 'honks', 'honking', 'honked', 'honked', intran, [] ).
verb( 'honour', 'honours', 'honouring', 'honoured', 'honoured', tran, ['6A'] ).
verb( 'hood', 'hoods', 'hooding', 'hooded', 'hooded', tran, [] ).
verb( 'hoodoo', 'hoodoos', 'hoodooing', 'hoodooed', 'hoodooed', tran, [] ).
verb( 'hoodwink', 'hoodwinks', 'hoodwinking', 'hoodwinked', 'hoodwinked', tran, ['6A','14'] ).
verb( 'hook', 'hooks', 'hooking', 'hooked', 'hooked', _, ['2C','6A','15A','15B'] ).
verb( 'hoop', 'hoops', 'hooping', 'hooped', 'hooped', tran, [] ).
verb( 'hoot', 'hoots', 'hooting', 'hooted', 'hooted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'hop', 'hops', 'hopping', 'hopped', 'hopped', _, ['2A','2C','6A'] ).
verb( 'hope', 'hopes', 'hoping', 'hoped', 'hoped', _, ['2A','3A','7A','9'] ).
verb( 'horn', 'horns', 'horning', 'horned', 'horned', intran, [] ).
verb( 'horrify', 'horrifies', 'horrifying', 'horrified', 'horrified', tran, ['6A'] ).
verb( 'horsewhip', 'horsewhips', 'horsewhipping', 'horsewhipped', 'horsewhipped', tran, [] ).
verb( 'hose', 'hoses', 'hosing', 'hosed', 'hosed', tran, ['6A','15B'] ).
verb( 'hospitalize', 'hospitalizes', 'hospitalizing', 'hospitalized', 'hospitalized', tran, [] ).
verb( 'host', 'hosts', 'hosting', 'hosted', 'hosted', tran, ['6A'] ).
verb( 'hot', 'hots', 'hotting', 'hotted', 'hotted', _, ['2C','15B'] ).
verb( 'hotfoot', 'hotfoots', 'hotfooting', 'hotfooted', 'hotfooted', intran, [] ).
verb( 'hound', 'hounds', 'hounding', 'hounded', 'hounded', tran, ['6A'] ).
verb( 'house', 'houses', 'housing', 'housed', 'housed', tran, ['6A'] ).
verb( 'hover', 'hovers', 'hovering', 'hovered', 'hovered', intran, ['2A','2C'] ).
verb( 'howl', 'howls', 'howling', 'howled', 'howled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'huddle', 'huddles', 'huddling', 'huddled', 'huddled', _, ['2C','15A','15B'] ).
verb( 'huff', 'huffs', 'huffing', 'huffed', 'huffed', intran, [] ).
verb( 'hug', 'hugs', 'hugging', 'hugged', 'hugged', tran, ['6A'] ).
verb( 'hull', 'hulls', 'hulling', 'hulled', 'hulled', tran, [] ).
verb( 'hum', 'hums', 'humming', 'hummed', 'hummed', _, ['2A','2C','6A'] ).
verb( 'humanize', 'humanizes', 'humanizing', 'humanized', 'humanized', _, ['2A','6A'] ).
verb( 'humble', 'humbles', 'humbling', 'humbled', 'humbled', tran, ['6A'] ).
verb( 'humbug', 'humbugs', 'humbugging', 'humbugged', 'humbugged', tran, ['6A','14'] ).
verb( 'humidify', 'humidifies', 'humidifying', 'humidified', 'humidified', tran, [] ).
verb( 'humiliate', 'humiliates', 'humiliating', 'humiliated', 'humiliated', tran, ['6A'] ).
verb( 'humour', 'humours', 'humouring', 'humoured', 'humoured', tran, ['6A'] ).
verb( 'hump', 'humps', 'humping', 'humped', 'humped', tran, ['6A','15B'] ).
verb( 'hunch', 'hunches', 'hunching', 'hunched', 'hunched', tran, ['6A','15B'] ).
verb( 'hunger', 'hungers', 'hungering', 'hungered', 'hungered', intran, ['2A','3A','4C'] ).
verb( 'hunt', 'hunts', 'hunting', 'hunted', 'hunted', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'hurdle', 'hurdles', 'hurdling', 'hurdled', 'hurdled', _, ['2A','15B'] ).
verb( 'hurl', 'hurls', 'hurling', 'hurled', 'hurled', tran, ['6A','15A','15B'] ).
verb( 'hurrah', 'hurrahs', 'hurrahing', 'hurrahed', 'hurrahed', intran, [] ).
verb( 'hurry', 'hurries', 'hurrying', 'hurried', 'hurried', _, ['2A','2C','6A','15A','15B'] ).
verb( 'hurt', 'hurts', 'hurting', 'hurt', 'hurt', _, ['2A','6A','6B'] ).
verb( 'hurtle', 'hurtles', 'hurtling', 'hurtled', 'hurtled', intran, ['2C'] ).
verb( 'husband', 'husbands', 'husbanding', 'husbanded', 'husbanded', tran, ['1','6A'] ).
verb( 'hush', 'hushes', 'hushing', 'hushed', 'hushed', _, ['2A','15A','15B'] ).
verb( 'husk', 'husks', 'husking', 'husked', 'husked', tran, [] ).
verb( 'hustle', 'hustles', 'hustling', 'hustled', 'hustled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'hybridize', 'hybridizes', 'hybridizing', 'hybridized', 'hybridized', _, ['2A','6A'] ).
verb( 'hydrate', 'hydrates', 'hydrating', 'hydrated', 'hydrated', _, [] ).
verb( 'hymn', 'hymns', 'hymning', 'hymned', 'hymned', tran, [] ).
verb( 'hyphen', 'hyphens', 'hyphening', 'hyphened', 'hyphened', tran, [] ).
verb( 'hyphenate', 'hyphenates', 'hyphenating', 'hyphenated', 'hyphenated', tran, [] ).
verb( 'hypnotize', 'hypnotizes', 'hypnotizing', 'hypnotized', 'hypnotized', tran, ['6A'] ).
verb( 'hypothecate', 'hypothecates', 'hypothecating', 'hypothecated', 'hypothecated', tran, [] ).
verb( 'ice', 'ices', 'icing', 'iced', 'iced', _, ['2C','6A','15B'] ).
verb( 'ice-skate', 'ice-skates', 'ice-skating', 'ice-skated', 'ice-skated', intran, [] ).
verb( 'idealize', 'idealizes', 'idealizing', 'idealized', 'idealized', tran, ['6A'] ).
verb( 'identify', 'identifies', 'identifying', 'identified', 'identified', tran, ['3A','6A','14'] ).
verb( 'idle', 'idles', 'idling', 'idled', 'idled', _, ['2A','2C','15B'] ).
verb( 'idolize', 'idolizes', 'idolizing', 'idolized', 'idolized', tran, ['6A'] ).
verb( 'ignite', 'ignites', 'igniting', 'ignited', 'ignited', _, ['2A','6A'] ).
verb( 'ignore', 'ignores', 'ignoring', 'ignored', 'ignored', tran, ['6A'] ).
verb( 'ill-treat', 'ill-treats', 'ill-treating', 'ill-treated', 'ill-treated', tran, [] ).
verb( 'ill-use', 'ill-uses', 'ill-using', 'ill-used', 'ill-used', tran, [] ).
verb( 'illume', 'illumes', 'illuming', 'illumed', 'illumed', tran, [] ).
verb( 'illuminate', 'illuminates', 'illuminating', 'illuminated', 'illuminated', tran, ['6A'] ).
verb( 'illumine', 'illumines', 'illumining', 'illumined', 'illumined', tran, ['6A'] ).
verb( 'illustrate', 'illustrates', 'illustrating', 'illustrated', 'illustrated', tran, ['6A'] ).
verb( 'image', 'images', 'imaging', 'imaged', 'imaged', tran, ['6A'] ).
verb( 'imagine', 'imagines', 'imagining', 'imagined', 'imagined', tran, ['6A','6C','9','10','16B','19A','19C','25'] ).
verb( 'imbed', 'imbeds', 'imbedding', 'imbedded', 'imbedded', tran, [] ).
verb( 'imbibe', 'imbibes', 'imbibing', 'imbibed', 'imbibed', tran, [] ).
verb( 'imbue', 'imbues', 'imbuing', 'imbued', 'imbued', tran, ['14'] ).
verb( 'imitate', 'imitates', 'imitating', 'imitated', 'imitated', tran, ['6A'] ).
verb( 'immerse', 'immerses', 'immersing', 'immersed', 'immersed', tran, ['6A','14'] ).
verb( 'immigrate', 'immigrates', 'immigrating', 'immigrated', 'immigrated', intran, ['2A','3A'] ).
verb( 'immobilize', 'immobilizes', 'immobilizing', 'immobilized', 'immobilized', tran, ['6A'] ).
verb( 'immolate', 'immolates', 'immolating', 'immolated', 'immolated', tran, ['6A','14'] ).
verb( 'immortalize', 'immortalizes', 'immortalizing', 'immortalized', 'immortalized', tran, ['6A'] ).
verb( 'immunize', 'immunizes', 'immunizing', 'immunized', 'immunized', tran, ['6A','14'] ).
verb( 'immure', 'immures', 'immuring', 'immured', 'immured', tran, ['6A'] ).
verb( 'impact', 'impacts', 'impacting', 'impacted', 'impacted', tran, [] ).
verb( 'impair', 'impairs', 'impairing', 'impaired', 'impaired', tran, ['6A'] ).
verb( 'impale', 'impales', 'impaling', 'impaled', 'impaled', tran, ['6A','15A'] ).
verb( 'impanel', 'impanels', 'impanelling', 'impanelled', 'impanelled', tran, ['6A'] ).
verb( 'impart', 'imparts', 'imparting', 'imparted', 'imparted', tran, ['6A','14'] ).
verb( 'impeach', 'impeaches', 'impeaching', 'impeached', 'impeached', tran, ['6A','14'] ).
verb( 'impede', 'impedes', 'impeding', 'impeded', 'impeded', tran, ['6A'] ).
verb( 'impel', 'impels', 'impelling', 'impelled', 'impelled', tran, ['14','17'] ).
verb( 'impend', 'impends', 'impending', 'impended', 'impended', intran, [] ).
verb( 'imperil', 'imperils', 'imperilling', 'imperilled', 'imperilled', tran, ['6A'] ).
verb( 'impersonate', 'impersonates', 'impersonating', 'impersonated', 'impersonated', tran, ['6A'] ).
verb( 'impinge', 'impinges', 'impinging', 'impinged', 'impinged', intran, ['3A'] ).
verb( 'implant', 'implants', 'implanting', 'implanted', 'implanted', tran, ['6A','14'] ).
verb( 'implement', 'implements', 'implementing', 'implemented', 'implemented', tran, ['6A'] ).
verb( 'implicate', 'implicates', 'implicating', 'implicated', 'implicated', tran, ['6A','14'] ).
verb( 'implore', 'implores', 'imploring', 'implored', 'implored', tran, ['6A','14','17'] ).
verb( 'imply', 'implies', 'implying', 'implied', 'implied', tran, ['6A','9'] ).
verb( 'import', 'imports', 'importing', 'imported', 'imported', tran, ['6A','9','14'] ).
verb( 'importune', 'importunes', 'importuning', 'importuned', 'importuned', tran, ['6A','9','14','17'] ).
verb( 'impose', 'imposes', 'imposing', 'imposed', 'imposed', _, ['3A','14'] ).
verb( 'impound', 'impounds', 'impounding', 'impounded', 'impounded', tran, ['6A'] ).
verb( 'impoverish', 'impoverishes', 'impoverishing', 'impoverished', 'impoverished', tran, ['6A'] ).
verb( 'imprecate', 'imprecates', 'imprecating', 'imprecated', 'imprecated', tran, ['14'] ).
verb( 'impregnate', 'impregnates', 'impregnating', 'impregnated', 'impregnated', tran, ['6A','14'] ).
verb( 'impress', 'impresses', 'impressing', 'impressed', 'impressed', tran, ['6A','14'] ).
verb( 'imprint', 'imprints', 'imprinting', 'imprinted', 'imprinted', tran, ['14'] ).
verb( 'imprison', 'imprisons', 'imprisoning', 'imprisoned', 'imprisoned', tran, ['6A'] ).
verb( 'improve', 'improves', 'improving', 'improved', 'improved', _, ['2A','3A','6A'] ).
verb( 'improvise', 'improvises', 'improvising', 'improvised', 'improvised', _, ['2A','6A'] ).
verb( 'impugn', 'impugns', 'impugning', 'impugned', 'impugned', tran, ['6A'] ).
verb( 'impulse-buy', 'impulse-buys', 'impulse-buying', 'impulse-bought', 'impulse-bought', _, [] ).
verb( 'impute', 'imputes', 'imputing', 'imputed', 'imputed', tran, ['14'] ).
verb( 'inactivate', 'inactivates', 'inactivating', 'inactivated', 'inactivated', tran, [] ).
verb( 'inaugurate', 'inaugurates', 'inaugurating', 'inaugurated', 'inaugurated', tran, ['6A'] ).
verb( 'incapacitate', 'incapacitates', 'incapacitating', 'incapacitated', 'incapacitated', tran, ['6A','14'] ).
verb( 'incarcerate', 'incarcerates', 'incarcerating', 'incarcerated', 'incarcerated', tran, ['6A'] ).
verb( 'incarnate', 'incarnates', 'incarnating', 'incarnated', 'incarnated', tran, ['6A'] ).
verb( 'incense', 'incenses', 'incensing', 'incensed', 'incensed', tran, ['6A'] ).
verb( 'inch', 'inches', 'inching', 'inched', 'inched', _, ['2C','15A','15B'] ).
verb( 'incinerate', 'incinerates', 'incinerating', 'incinerated', 'incinerated', tran, ['6A'] ).
verb( 'incise', 'incises', 'incising', 'incised', 'incised', tran, ['6A'] ).
verb( 'incite', 'incites', 'inciting', 'incited', 'incited', tran, ['6A','14','17'] ).
verb( 'incline', 'inclines', 'inclining', 'inclined', 'inclined', _, ['2A','3A','4C','6A','15A','17'] ).
verb( 'inclose', 'incloses', 'inclosing', 'inclosed', 'inclosed', tran, ['6A','14'] ).
verb( 'include', 'includes', 'including', 'included', 'included', tran, ['6A','6C'] ).
verb( 'incommode', 'incommodes', 'incommoding', 'incommoded', 'incommoded', tran, ['6A'] ).
verb( 'inconvenience', 'inconveniences', 'inconveniencing', 'inconvenienced', 'inconvenienced', tran, ['6A'] ).
verb( 'incorporate', 'incorporates', 'incorporating', 'incorporated', 'incorporated', _, ['2A','3A','6A','14','23'] ).
verb( 'increase', 'increases', 'increasing', 'increased', 'increased', _, ['2A','6A'] ).
verb( 'incriminate', 'incriminates', 'incriminating', 'incriminated', 'incriminated', tran, ['6A'] ).
verb( 'incubate', 'incubates', 'incubating', 'incubated', 'incubated', _, ['2A','6A'] ).
verb( 'inculcate', 'inculcates', 'inculcating', 'inculcated', 'inculcated', tran, ['6A','14'] ).
verb( 'inculpate', 'inculpates', 'inculpating', 'inculpated', 'inculpated', tran, ['6A'] ).
verb( 'incur', 'incurs', 'incurring', 'incurred', 'incurred', tran, ['6A'] ).
verb( 'indemnify', 'indemnifies', 'indemnifying', 'indemnified', 'indemnified', tran, ['6A','14'] ).
verb( 'indent', 'indents', 'indenting', 'indented', 'indented', _, ['3A','6A'] ).
verb( 'indenture', 'indentures', 'indenturing', 'indentured', 'indentured', tran, ['6A'] ).
verb( 'index', 'indexes', 'indexing', 'indexed', 'indexed', tran, ['6A'] ).
verb( 'indicate', 'indicates', 'indicating', 'indicated', 'indicated', tran, ['6A','9','14'] ).
verb( 'indict', 'indicts', 'indicting', 'indicted', 'indicted', tran, ['6A','14','16B'] ).
verb( 'indite', 'indites', 'inditing', 'indited', 'indited', tran, ['6A'] ).
verb( 'individualize', 'individualizes', 'individualizing', 'individualized', 'individualized', tran, ['6A'] ).
verb( 'indoctrinate', 'indoctrinates', 'indoctrinating', 'indoctrinated', 'indoctrinated', tran, ['6A','14'] ).
verb( 'indorse', 'indorses', 'indorsing', 'indorsed', 'indorsed', tran, ['6A'] ).
verb( 'induce', 'induces', 'inducing', 'induced', 'induced', tran, ['6A','14','17'] ).
verb( 'induct', 'inducts', 'inducting', 'inducted', 'inducted', tran, ['6A','14','16B'] ).
verb( 'indue', 'indues', 'induing', 'indued', 'indued', tran, ['14'] ).
verb( 'indulge', 'indulges', 'indulging', 'indulged', 'indulged', _, ['3A','6A'] ).
verb( 'inebriate', 'inebriates', 'inebriating', 'inebriated', 'inebriated', tran, ['6A'] ).
verb( 'infatuate', 'infatuates', 'infatuating', 'infatuated', 'infatuated', tran, [] ).
verb( 'infect', 'infects', 'infecting', 'infected', 'infected', tran, ['6A','14'] ).
verb( 'infer', 'infers', 'inferring', 'inferred', 'inferred', tran, ['6A','9','14'] ).
verb( 'infest', 'infests', 'infesting', 'infested', 'infested', tran, ['6A'] ).
verb( 'infiltrate', 'infiltrates', 'infiltrating', 'infiltrated', 'infiltrated', _, ['2A','3A','6A','14'] ).
verb( 'inflame', 'inflames', 'inflaming', 'inflamed', 'inflamed', _, ['2A','6A'] ).
verb( 'inflate', 'inflates', 'inflating', 'inflated', 'inflated', tran, ['6A','14'] ).
verb( 'inflect', 'inflects', 'inflecting', 'inflected', 'inflected', tran, ['6A'] ).
verb( 'inflict', 'inflicts', 'inflicting', 'inflicted', 'inflicted', tran, ['6A','14'] ).
verb( 'influence', 'influences', 'influencing', 'influenced', 'influenced', tran, ['6A'] ).
verb( 'inform', 'informs', 'informing', 'informed', 'informed', _, ['3A','6A','11','14','21'] ).
verb( 'infringe', 'infringes', 'infringing', 'infringed', 'infringed', _, ['3A','6A'] ).
verb( 'infuriate', 'infuriates', 'infuriating', 'infuriated', 'infuriated', tran, ['6A'] ).
verb( 'infuse', 'infuses', 'infusing', 'infused', 'infused', _, ['2A','6A','14'] ).
verb( 'ingest', 'ingests', 'ingesting', 'ingested', 'ingested', tran, ['6A'] ).
verb( 'ingraft', 'ingrafts', 'ingrafting', 'ingrafted', 'ingrafted', tran, ['6A','14'] ).
verb( 'ingratiate', 'ingratiates', 'ingratiating', 'ingratiated', 'ingratiated', tran, ['14'] ).
verb( 'inhabit', 'inhabits', 'inhabiting', 'inhabited', 'inhabited', tran, ['6A'] ).
verb( 'inhale', 'inhales', 'inhaling', 'inhaled', 'inhaled', _, ['2A','6A'] ).
verb( 'inherit', 'inherits', 'inheriting', 'inherited', 'inherited', _, ['2A','6A'] ).
verb( 'inhibit', 'inhibits', 'inhibiting', 'inhibited', 'inhibited', tran, ['6A','14'] ).
verb( 'initial', 'initials', 'initialling', 'initialled', 'initialled', tran, ['6A'] ).
verb( 'initiate', 'initiates', 'initiating', 'initiated', 'initiated', tran, ['6A','14'] ).
verb( 'inject', 'injects', 'injecting', 'injected', 'injected', tran, ['6A','14'] ).
verb( 'injure', 'injures', 'injuring', 'injured', 'injured', tran, ['6A'] ).
verb( 'ink', 'inks', 'inking', 'inked', 'inked', tran, ['6A','15B'] ).
verb( 'inlay', 'inlays', 'inlaying', 'inlaid', 'inlaid', tran, ['6A','14'] ).
verb( 'innovate', 'innovates', 'innovating', 'innovated', 'innovated', intran, ['2A'] ).
verb( 'inoculate', 'inoculates', 'inoculating', 'inoculated', 'inoculated', tran, ['6A','14'] ).
verb( 'inquire', 'inquires', 'inquiring', 'inquired', 'inquired', _, ['2A','3A','6A','8','10','14'] ).
verb( 'inscribe', 'inscribes', 'inscribing', 'inscribed', 'inscribed', tran, ['6A','15A'] ).
verb( 'inseminate', 'inseminates', 'inseminating', 'inseminated', 'inseminated', tran, ['6A'] ).
verb( 'insert', 'inserts', 'inserting', 'inserted', 'inserted', tran, ['6A','15A'] ).
verb( 'inset', 'insets', 'insetting', 'inset', 'inset', tran, [] ).
verb( 'insinuate', 'insinuates', 'insinuating', 'insinuated', 'insinuated', tran, ['6A','9','14'] ).
verb( 'insist', 'insists', 'insisting', 'insisted', 'insisted', _, ['3A','3B'] ).
verb( 'inspan', 'inspans', 'inspanning', 'inspanned', 'inspanned', tran, [] ).
verb( 'inspect', 'inspects', 'inspecting', 'inspected', 'inspected', tran, ['6A'] ).
verb( 'inspire', 'inspires', 'inspiring', 'inspired', 'inspired', tran, ['6A','14','17'] ).
verb( 'install', 'installs', 'installing', 'installed', 'installed', tran, ['6A','14'] ).
verb( 'instance', 'instances', 'instancing', 'instanced', 'instanced', tran, ['6A'] ).
verb( 'instigate', 'instigates', 'instigating', 'instigated', 'instigated', tran, ['6A','17'] ).
verb( 'instil', 'instils', 'instilling', 'instilled', 'instilled', tran, ['6A','14'] ).
verb( 'institute', 'institutes', 'instituting', 'instituted', 'instituted', tran, ['6A','14'] ).
verb( 'institutionalize', 'institutionalizes', 'institutionalizing', 'institutionalized', 'institutionalized', tran, ['6A'] ).
verb( 'instruct', 'instructs', 'instructing', 'instructed', 'instructed', tran, ['6A','11','15A','17','20','21'] ).
verb( 'insulate', 'insulates', 'insulating', 'insulated', 'insulated', tran, ['6A','14'] ).
verb( 'insult', 'insults', 'insulting', 'insulted', 'insulted', tran, ['6A'] ).
verb( 'insure', 'insures', 'insuring', 'insured', 'insured', tran, ['6A','14'] ).
verb( 'integrate', 'integrates', 'integrating', 'integrated', 'integrated', tran, ['2A','6A'] ).
verb( 'intend', 'intends', 'intending', 'intended', 'intended', tran, ['6A','6D','7A','9','14','17'] ).
verb( 'intensify', 'intensifies', 'intensifying', 'intensified', 'intensified', _, ['2A','6A'] ).
verb( 'inter', 'inters', 'interring', 'interred', 'interred', tran, [] ).
verb( 'interact', 'interacts', 'interacting', 'interacted', 'interacted', intran, [] ).
verb( 'interbreed', 'interbreeds', 'interbreeding', 'interbred', 'interbred', _, ['2A','6A'] ).
verb( 'intercede', 'intercedes', 'interceding', 'interceded', 'interceded', intran, ['3A'] ).
verb( 'intercept', 'intercepts', 'intercepting', 'intercepted', 'intercepted', tran, ['6A'] ).
verb( 'interchange', 'interchanges', 'interchanging', 'interchanged', 'interchanged', tran, ['2A','6A'] ).
verb( 'intercommunicate', 'intercommunicates', 'intercommunicating', 'intercommunicated', 'intercommunicated', intran, [] ).
verb( 'interconnect', 'interconnects', 'interconnecting', 'interconnected', 'interconnected', _, ['6A'] ).
verb( 'interdict', 'interdicts', 'interdicting', 'interdicted', 'interdicted', tran, ['6A'] ).
verb( 'interest', 'interests', 'interesting', 'interested', 'interested', tran, ['6A','14'] ).
verb( 'interfere', 'interferes', 'interfering', 'interfered', 'interfered', intran, ['2A','3A'] ).
verb( 'interject', 'interjects', 'interjecting', 'interjected', 'interjected', tran, ['6A'] ).
verb( 'interlace', 'interlaces', 'interlacing', 'interlaced', 'interlaced', _, ['2A','6A','14'] ).
verb( 'interlard', 'interlards', 'interlarding', 'interlarded', 'interlarded', tran, ['14'] ).
verb( 'interleave', 'interleaves', 'interleaving', 'interleaved', 'interleaved', tran, ['6A','14'] ).
verb( 'interlink', 'interlinks', 'interlinking', 'interlinked', 'interlinked', _, ['2A','6A'] ).
verb( 'interlock', 'interlocks', 'interlocking', 'interlocked', 'interlocked', _, ['2A','6A'] ).
verb( 'intermarry', 'intermarries', 'intermarrying', 'intermarried', 'intermarried', intran, ['2A','3A'] ).
verb( 'intermingle', 'intermingles', 'intermingling', 'intermingled', 'intermingled', _, ['6A','14'] ).
verb( 'intermix', 'intermixes', 'intermixing', 'intermixed', 'intermixed', _, [] ).
verb( 'intern', 'interns', 'interning', 'interned', 'interned', tran, ['6A'] ).
verb( 'internalize', 'internalizes', 'internalizing', 'internalized', 'internalized', tran, ['6A'] ).
verb( 'internationalize', 'internationalizes', 'internationalizing', 'internationalized', 'internationalized', tran, ['6A'] ).
verb( 'interpellate', 'interpellates', 'interpellating', 'interpellated', 'interpellated', tran, ['6A'] ).
verb( 'interpolate', 'interpolates', 'interpolating', 'interpolated', 'interpolated', tran, ['6A'] ).
verb( 'interpose', 'interposes', 'interposing', 'interposed', 'interposed', _, ['2A','3A','6A','14'] ).
verb( 'interpret', 'interprets', 'interpreting', 'interpreted', 'interpreted', _, ['2A','6A','16B'] ).
verb( 'interrelate', 'interrelates', 'interrelating', 'interrelated', 'interrelated', _, ['2A','6A'] ).
verb( 'interrogate', 'interrogates', 'interrogating', 'interrogated', 'interrogated', tran, ['6A'] ).
verb( 'interrupt', 'interrupts', 'interrupting', 'interrupted', 'interrupted', _, ['2A','6A'] ).
verb( 'intersect', 'intersects', 'intersecting', 'intersected', 'intersected', _, ['2A','6A'] ).
verb( 'intersperse', 'intersperses', 'interspersing', 'interspersed', 'interspersed', tran, ['14'] ).
verb( 'intertwine', 'intertwines', 'intertwining', 'intertwined', 'intertwined', _, ['2A','6A'] ).
verb( 'intervene', 'intervenes', 'intervening', 'intervened', 'intervened', intran, ['2A','3A'] ).
verb( 'interview', 'interviews', 'interviewing', 'interviewed', 'interviewed', tran, ['6A'] ).
verb( 'interweave', 'interweaves', 'interweaving', 'interwove', 'interwoven', tran, ['6A','14'] ).
verb( 'intimate', 'intimates', 'intimating', 'intimated', 'intimated', tran, ['6A','9','14'] ).
verb( 'intimidate', 'intimidates', 'intimidating', 'intimidated', 'intimidated', tran, ['6A','14'] ).
verb( 'intone', 'intones', 'intoning', 'intoned', 'intoned', _, ['2A','6A'] ).
verb( 'intoxicate', 'intoxicates', 'intoxicating', 'intoxicated', 'intoxicated', tran, ['6A'] ).
verb( 'intrench', 'intrenches', 'intrenching', 'intrenched', 'intrenched', tran, ['6A'] ).
verb( 'intrigue', 'intrigues', 'intriguing', 'intrigued', 'intrigued', _, ['2A','3A','6A'] ).
verb( 'introduce', 'introduces', 'introducing', 'introduced', 'introduced', tran, ['6A','14','15A'] ).
verb( 'introspect', 'introspects', 'introspecting', 'introspected', 'introspected', intran, ['2A'] ).
verb( 'introvert', 'introverts', 'introverting', 'introverted', 'introverted', tran, ['6A'] ).
verb( 'intrude', 'intrudes', 'intruding', 'intruded', 'intruded', _, ['2A','3A','14'] ).
verb( 'intrust', 'intrusts', 'intrusting', 'intrusted', 'intrusted', tran, ['14'] ).
verb( 'intuit', 'intuits', 'intuiting', 'intuited', 'intuited', _, ['2A','6A'] ).
verb( 'inundate', 'inundates', 'inundating', 'inundated', 'inundated', tran, ['6A','14'] ).
verb( 'inure', 'inures', 'inuring', 'inured', 'inured', tran, ['14'] ).
verb( 'invade', 'invades', 'invading', 'invaded', 'invaded', tran, ['6A'] ).
verb( 'invalid', 'invalids', 'invaliding', 'invalided', 'invalided', tran, ['15A'] ).
verb( 'invalidate', 'invalidates', 'invalidating', 'invalidated', 'invalidated', tran, ['6A'] ).
verb( 'inveigh', 'inveighs', 'inveighing', 'inveighed', 'inveighed', intran, ['3A'] ).
verb( 'inveigle', 'inveigles', 'inveigling', 'inveigled', 'inveigled', tran, ['14'] ).
verb( 'invent', 'invents', 'inventing', 'invented', 'invented', tran, ['6A'] ).
verb( 'invert', 'inverts', 'inverting', 'inverted', 'inverted', tran, ['6A'] ).
verb( 'invest', 'invests', 'investing', 'invested', 'invested', _, ['3A','6A','14'] ).
verb( 'investigate', 'investigates', 'investigating', 'investigated', 'investigated', tran, ['6A'] ).
verb( 'invigilate', 'invigilates', 'invigilating', 'invigilated', 'invigilated', intran, ['2A','6A'] ).
verb( 'invigorate', 'invigorates', 'invigorating', 'invigorated', 'invigorated', tran, ['6A'] ).
verb( 'invite', 'invites', 'inviting', 'invited', 'invited', tran, ['6A','15A','15B','17'] ).
verb( 'invoice', 'invoices', 'invoicing', 'invoiced', 'invoiced', tran, [] ).
verb( 'invoke', 'invokes', 'invoking', 'invoked', 'invoked', tran, ['6A','14'] ).
verb( 'involve', 'involves', 'involving', 'involved', 'involved', tran, ['6A','6B','14','19C'] ).
verb( 'ionize', 'ionizes', 'ionizing', 'ionized', 'ionized', _, ['2A','6A'] ).
verb( 'irk', 'irks', 'irking', 'irked', 'irked', tran, [] ).
verb( 'iron', 'irons', 'ironing', 'ironed', 'ironed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'irradiate', 'irradiates', 'irradiating', 'irradiated', 'irradiated', tran, ['6A'] ).
verb( 'irrigate', 'irrigates', 'irrigating', 'irrigated', 'irrigated', tran, ['6A'] ).
verb( 'irritate', 'irritates', 'irritating', 'irritated', 'irritated', tran, ['6A'] ).
verb( 'isolate', 'isolates', 'isolating', 'isolated', 'isolated', tran, ['6A','14'] ).
verb( 'issue', 'issues', 'issuing', 'issued', 'issued', _, ['2A','3A','6A','14'] ).
verb( 'italicize', 'italicizes', 'italicizing', 'italicized', 'italicized', tran, [] ).
verb( 'itch', 'itches', 'itching', 'itched', 'itched', intran, ['2A','3A','4A'] ).
verb( 'itemize', 'itemizes', 'itemizing', 'itemized', 'itemized', tran, ['6A'] ).
verb( 'iterate', 'iterates', 'iterating', 'iterated', 'iterated', tran, ['6A'] ).
verb( 'jab', 'jabs', 'jabbing', 'jabbed', 'jabbed', _, ['3A','14','15B'] ).
verb( 'jabber', 'jabbers', 'jabbering', 'jabbered', 'jabbered', _, ['2A','2C','6A','15B'] ).
verb( 'jack', 'jacks', 'jacking', 'jacked', 'jacked', tran, ['15B'] ).
verb( 'jack-knife', 'jack-knifes', 'jack-knifing', 'jack-knifed', 'jack-knifed', intran, ['2A'] ).
verb( 'jag', 'jags', 'jagging', 'jagged', 'jagged', tran, ['6A'] ).
verb( 'jail', 'jails', 'jailing', 'jailed', 'jailed', tran, ['6A'] ).
verb( 'jam', 'jams', 'jamming', 'jammed', 'jammed', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'jampack', 'jampacks', 'jampacking', 'jampacked', 'jampacked', tran, ['6A'] ).
verb( 'jangle', 'jangles', 'jangling', 'jangled', 'jangled', _, ['2A','6A'] ).
verb( 'japan', 'japans', 'japanning', 'japanned', 'japanned', tran, [] ).
verb( 'jar', 'jars', 'jarring', 'jarred', 'jarred', _, ['2A','3A','6A'] ).
verb( 'jaundice', 'jaundices', 'jaundicing', 'jaundiced', 'jaundiced', tran, [] ).
verb( 'jaunt', 'jaunts', 'jaunting', 'jaunted', 'jaunted', intran, ['2A','2C'] ).
verb( 'jaw', 'jaws', 'jawing', 'jawed', 'jawed', intran, ['2A','2C','3A'] ).
verb( 'jaywalk', 'jaywalks', 'jaywalking', 'jaywalked', 'jaywalked', intran, [] ).
verb( 'jazz', 'jazzes', 'jazzing', 'jazzed', 'jazzed', tran, ['6A','15B'] ).
verb( 'jeer', 'jeers', 'jeering', 'jeered', 'jeered', _, ['2A','3A','6A'] ).
verb( 'jell', 'jells', 'jelling', 'jelled', 'jelled', _, ['2A','6A'] ).
verb( 'jelly', 'jellies', 'jellying', 'jellied', 'jellied', _, ['2A','6A'] ).
verb( 'jeopardize', 'jeopardizes', 'jeopardizing', 'jeopardized', 'jeopardized', tran, ['6A'] ).
verb( 'jerk', 'jerks', 'jerking', 'jerked', 'jerked', _, ['2C','6A','15A','15B'] ).
verb( 'jest', 'jests', 'jesting', 'jested', 'jested', intran, ['2A','3A'] ).
verb( 'jet', 'jets', 'jetting', 'jetted', 'jetted', _, [] ).
verb( 'jettison', 'jettisons', 'jettisoning', 'jettisoned', 'jettisoned', tran, ['6A'] ).
verb( 'jewel', 'jewels', 'jewelling', 'jewelled', 'jewelled', tran, [] ).
verb( 'jib', 'jibs', 'jibbing', 'jibbed', 'jibbed', intran, ['2A','3A'] ).
verb( 'jibe', 'jibes', 'jibing', 'jibed', 'jibed', intran, ['2A','3A'] ).
verb( 'jig', 'jigs', 'jigging', 'jigged', 'jigged', _, ['2A','2C','15B'] ).
verb( 'jiggle', 'jiggles', 'jiggling', 'jiggled', 'jiggled', _, [] ).
verb( 'jilt', 'jilts', 'jilting', 'jilted', 'jilted', tran, ['6A'] ).
verb( 'jingle', 'jingles', 'jingling', 'jingled', 'jingled', _, ['2A','2C','6A','15B'] ).
verb( 'jive', 'jives', 'jiving', 'jived', 'jived', intran, [] ).
verb( 'job', 'jobs', 'jobbing', 'jobbed', 'jobbed', _, ['2A','6A','14'] ).
verb( 'jockey', 'jockeys', 'jockeying', 'jockeyed', 'jockeyed', _, ['3A','15A'] ).
verb( 'jog', 'jogs', 'jogging', 'jogged', 'jogged', _, ['2A','2C','6A','15B'] ).
verb( 'joggle', 'joggles', 'joggling', 'joggled', 'joggled', _, ['2A','6A'] ).
verb( 'join', 'joins', 'joining', 'joined', 'joined', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'joint', 'joints', 'jointing', 'jointed', 'jointed', tran, ['6A'] ).
verb( 'joke', 'jokes', 'joking', 'joked', 'joked', intran, ['2A','2C'] ).
verb( 'jolly', 'jollies', 'jollying', 'jollied', 'jollied', tran, ['6A','15A','15B'] ).
verb( 'jolt', 'jolts', 'jolting', 'jolted', 'jolted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'jostle', 'jostles', 'jostling', 'jostled', 'jostled', _, ['2C','6A'] ).
verb( 'jot', 'jots', 'jotting', 'jotted', 'jotted', tran, ['15B'] ).
verb( 'journey', 'journeys', 'journeying', 'journeyed', 'journeyed', intran, ['2A','2C'] ).
verb( 'joust', 'jousts', 'jousting', 'jousted', 'jousted', intran, [] ).
verb( 'joy', 'joys', 'joying', 'joyed', 'joyed', intran, [] ).
verb( 'judder', 'judders', 'juddering', 'juddered', 'juddered', intran, [] ).
verb( 'judge', 'judges', 'judging', 'judged', 'judged', _, ['2A','3A','6A','9','10','22','25'] ).
verb( 'jug', 'jugs', 'jugging', 'jugged', 'jugged', tran, ['6A'] ).
verb( 'juggle', 'juggles', 'juggling', 'juggled', 'juggled', _, ['2A','3A','6A','16A'] ).
verb( 'jumble', 'jumbles', 'jumbling', 'jumbled', 'jumbled', _, ['2C','15B'] ).
verb( 'jump', 'jumps', 'jumping', 'jumped', 'jumped', _, ['2A','2C','6A'] ).
verb( 'junket', 'junkets', 'junketing', 'junketed', 'junketed', intran, [] ).
verb( 'justify', 'justifies', 'justifying', 'justified', 'justified', tran, ['6A','19C'] ).
verb( 'jut', 'juts', 'jutting', 'jutted', 'jutted', intran, ['2C'] ).
verb( 'juxtapose', 'juxtaposes', 'juxtaposing', 'juxtaposed', 'juxtaposed', tran, ['6A'] ).
verb( 'keel', 'keels', 'keeling', 'keeled', 'keeled', _, ['2C','6A','15B'] ).
verb( 'keen', 'keens', 'keening', 'keened', 'keened', _, [] ).
verb( 'keep', 'keeps', 'keeping', 'kept', 'kept', _, ['2A','2C','2E','3A','6A','14','15A','15B','19B','22'] ).
verb( 'ken', 'kens', 'kenning', 'kenned', 'kenned', tran, ['6A','9'] ).
verb( 'kennel', 'kennels', 'kennelling', 'kennelled', 'kennelled', _, [] ).
verb( 'key', 'keys', 'keying', 'keyed', 'keyed', tran, ['6A'] ).
verb( 'kick', 'kicks', 'kicking', 'kicked', 'kicked', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'kid', 'kids', 'kidding', 'kidded', 'kidded', _, [] ).
verb( 'kidnap', 'kidnaps', 'kidnapping', 'kidnapped', 'kidnapped', tran, [] ).
verb( 'kill', 'kills', 'killing', 'killed', 'killed', _, ['2A','6A','15B'] ).
verb( 'kindle', 'kindles', 'kindling', 'kindled', 'kindled', _, ['2A','6A'] ).
verb( 'kink', 'kinks', 'kinking', 'kinked', 'kinked', _, [] ).
verb( 'kip', 'kips', 'kipping', 'kipped', 'kipped', intran, ['2A','2C'] ).
verb( 'kiss', 'kisses', 'kissing', 'kissed', 'kissed', _, ['2A','6A','15A','15B'] ).
verb( 'kit', 'kits', 'kitting', 'kitted', 'kitted', tran, ['15B'] ).
verb( 'knap', 'knaps', 'knapping', 'knapped', 'knapped', tran, [] ).
verb( 'knead', 'kneads', 'kneading', 'kneaded', 'kneaded', tran, ['6A'] ).
verb( 'kneel', 'kneels', 'kneeling', 'kneeled', 'kneeled', intran, ['2A','2C'] ).
verb( 'knife', 'knifes', 'knifing', 'knifed', 'knifed', tran, ['6A'] ).
verb( 'knight', 'knights', 'knighting', 'knighted', 'knighted', tran, ['6A'] ).
verb( 'knit', 'knits', 'knitting', 'knitted', 'knitted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'knock', 'knocks', 'knocking', 'knocked', 'knocked', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'knot', 'knots', 'knotting', 'knotted', 'knotted', _, ['2A','6A','15B'] ).
verb( 'know', 'knows', 'knowing', 'knew', 'known', _, ['2A','3A','6A','8','9','10','17','18B','25'] ).
verb( 'knuckle', 'knuckles', 'knuckling', 'knuckled', 'knuckled', intran, [] ).
verb( 'kotow', 'kotows', 'kotowing', 'kotowed', 'kotowed', intran, [] ).
verb( 'kowtow', 'kowtows', 'kowtowing', 'kowtowed', 'kowtowed', intran, [] ).
verb( 'label', 'labels', 'labelling', 'labelled', 'labelled', tran, ['6A'] ).
verb( 'labour', 'labours', 'labouring', 'laboured', 'laboured', _, ['2A','2C','3A','4A','6A'] ).
verb( 'lace', 'laces', 'lacing', 'laced', 'laced', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'lacerate', 'lacerates', 'lacerating', 'lacerated', 'lacerated', tran, ['6A'] ).
verb( 'lack', 'lacks', 'lacking', 'lacked', 'lacked', _, ['3A','6B'] ).
verb( 'lacquer', 'lacquers', 'lacquering', 'lacquered', 'lacquered', tran, ['6A'] ).
verb( 'ladder', 'ladders', 'laddering', 'laddered', 'laddered', intran, [] ).
verb( 'lade', 'lades', 'lading', 'laded', 'laden', tran, ['6A'] ).
verb( 'ladle', 'ladles', 'ladling', 'ladled', 'ladled', tran, ['6A','15B'] ).
verb( 'lag', 'lags', 'lagging', 'lagged', 'lagged', _, ['2A','2C','6A','14'] ).
verb( 'laicize', 'laicizes', 'laicizing', 'laicized', 'laicized', tran, ['6A'] ).
verb( 'lam', 'lams', 'lamming', 'lammed', 'lammed', _, ['3A','6A'] ).
verb( 'lamb', 'lambs', 'lambing', 'lambed', 'lambed', intran, [] ).
verb( 'lambaste', 'lambastes', 'lambasting', 'lambasted', 'lambasted', tran, [] ).
verb( 'lame', 'lames', 'laming', 'lamed', 'lamed', tran, [] ).
verb( 'lament', 'laments', 'lamenting', 'lamented', 'lamented', _, ['2A','3A','6A'] ).
verb( 'laminate', 'laminates', 'laminating', 'laminated', 'laminated', _, ['2A','6A'] ).
verb( 'lampoon', 'lampoons', 'lampooning', 'lampooned', 'lampooned', tran, ['6A'] ).
verb( 'lance', 'lances', 'lancing', 'lanced', 'lanced', tran, ['6A'] ).
verb( 'land', 'lands', 'landing', 'landed', 'landed', _, ['2A','2C','6A','12C'] ).
verb( 'landscape', 'landscapes', 'landscaping', 'landscaped', 'landscaped', tran, ['6A'] ).
verb( 'languish', 'languishes', 'languishing', 'languished', 'languished', intran, ['2A','2C'] ).
verb( 'lap', 'laps', 'lapping', 'lapped', 'lapped', _, ['2A','2C','6A','15B'] ).
verb( 'lapse', 'lapses', 'lapsing', 'lapsed', 'lapsed', intran, ['2A','3A'] ).
verb( 'lard', 'lards', 'larding', 'larded', 'larded', tran, ['6A'] ).
verb( 'lark', 'larks', 'larking', 'larked', 'larked', intran, ['2A','2C'] ).
verb( 'larn', 'larns', 'larning', 'larned', 'larned', _, [] ).
verb( 'lash', 'lashes', 'lashing', 'lashed', 'lashed', _, ['2C','6A','14','15A','15B'] ).
verb( 'lasso', 'lassos', 'lassoing', 'lassoed', 'lassoed', tran, [] ).
verb( 'last', 'lasts', 'lasting', 'lasted', 'lasted', intran, ['2A','2B','2C'] ).
verb( 'latch', 'latches', 'latching', 'latched', 'latched', _, ['2A','2C','3A','6A'] ).
verb( 'lather', 'lathers', 'lathering', 'lathered', 'lathered', _, ['2A','6A'] ).
verb( 'laud', 'lauds', 'lauding', 'lauded', 'lauded', tran, ['6A'] ).
verb( 'laugh', 'laughs', 'laughing', 'laughed', 'laughed', _, ['2A','2B','2C','3A','6B','15A','15B','22'] ).
verb( 'launch', 'launches', 'launching', 'launched', 'launched', _, ['2C','3A','6A','15A'] ).
verb( 'launder', 'launders', 'laundering', 'laundered', 'laundered', _, ['2A','6A'] ).
verb( 'lave', 'laves', 'laving', 'laved', 'laved', tran, [] ).
verb( 'lavish', 'lavishes', 'lavishing', 'lavished', 'lavished', tran, ['14'] ).
verb( 'lay', 'lays', 'laying', 'laid', 'laid', _, ['2A','2B','2C','2D','3A','6A','12C','14','15A','15B','22'] ).
verb( 'layer', 'layers', 'layering', 'layered', 'layered', tran, ['6A'] ).
verb( 'laze', 'lazes', 'lazing', 'lazed', 'lazed', _, ['2A','2C','15B'] ).
verb( 'leach', 'leaches', 'leaching', 'leached', 'leached', tran, ['6A','15B'] ).
verb( 'lead', 'leads', 'leading', 'leant', 'leant', _, ['2A','2C','3A','6A','12C','14','15A','15B','17'] ).
verb( 'leaf', 'leafs', 'leafing', 'leafed', 'leafed', intran, [] ).
verb( 'league', 'leagues', 'leaguing', 'leagued', 'leagued', _, ['2C','6A','15A','15B'] ).
verb( 'leak', 'leaks', 'leaking', 'leaked', 'leaked', _, ['2A','2C','6A','14'] ).
verb( 'lean', 'leans', 'leaning', 'leaned', 'leaned', _, ['2A','2C','3A','15A'] ).
verb( 'leap', 'leaps', 'leaping', 'leaped', 'leaped', _, ['2A','2C','3A','6A','15A'] ).
verb( 'leapfrog', 'leapfrogs', 'leapfrogging', 'leapfrogged', 'leapfrogged', tran, [] ).
verb( 'learn', 'learns', 'learning', 'learned', 'learned', _, ['2A','3A','6A','7A','8','9','10','15A','15B'] ).
verb( 'lease', 'leases', 'leasing', 'leased', 'leased', tran, ['6A'] ).
verb( 'leave', 'leaves', 'leavening', 'leavened', 'leavened', _, ['2A','2C','3A','6A','12B','13B','14','15A','15B','16A','19B','22','24B','25'] ).
verb( 'leaven', 'leavens', 'leavening', 'leavened', 'leavened', tran, ['6A'] ).
verb( 'lecture', 'lectures', 'lecturing', 'lectured', 'lectured', _, ['2A','3A','6A','14'] ).
verb( 'leer', 'leers', 'leering', 'leered', 'leered', intran, ['2A','3A'] ).
verb( 'legalize', 'legalizes', 'legalizing', 'legalized', 'legalized', tran, ['6A'] ).
verb( 'legislate', 'legislates', 'legislating', 'legislated', 'legislated', intran, ['2A','3A'] ).
verb( 'legitimatize', 'legitimatizes', 'legitimatizing', 'legitimatized', 'legitimatized', tran, [] ).
verb( 'lend', 'lends', 'lending', 'lent', 'lent', tran, ['6A','12A','13A','14'] ).
verb( 'lengthen', 'lengthens', 'lengthening', 'lengthened', 'lengthened', _, ['2A','6A'] ).
verb( 'lessen', 'lessens', 'lessening', 'lessened', 'lessened', _, ['2A','6A'] ).
verb( 'let', 'lets', 'letting', 'let', 'let', _, ['2C','6A','14','15A','15B','18B','22','24A'] ).
verb( 'levant', 'levants', 'levanting', 'levanted', 'levanted', intran, [] ).
verb( 'level', 'levels', 'levelling', 'levelled', 'levelled', _, ['2C','3A','6A','14','15A','15B'] ).
verb( 'lever', 'levers', 'levering', 'levered', 'levered', tran, ['6A','15B'] ).
verb( 'levitate', 'levitates', 'levitating', 'levitated', 'levitated', _, ['2A','6A'] ).
verb( 'levy', 'levies', 'levying', 'levied', 'levied', _, ['3A','6A','14'] ).
verb( 'liaise', 'liaises', 'liaising', 'liaised', 'liaised', intran, ['2A','3A'] ).
verb( 'libel', 'libels', 'libelling', 'libelled', 'libelled', tran, ['6A'] ).
verb( 'liberalize', 'liberalizes', 'liberalizing', 'liberalized', 'liberalized', tran, [] ).
verb( 'liberate', 'liberates', 'liberating', 'liberated', 'liberated', tran, ['6A','14'] ).
verb( 'licence', 'licences', 'licencing', 'licenced', 'licenced', tran, ['6A','17'] ).
verb( 'license', 'licenses', 'licensing', 'licensed', 'licensed', tran, ['6A','17'] ).
verb( 'lick', 'licks', 'licking', 'licked', 'licked', _, ['2A','6A','15A','15B','22'] ).
verb( 'lie', 'lies', 'lying', 'lay', 'lain', intran, ['2A','2C','2D','3A'] ).
verb( 'lie', 'lies', 'lying', 'lied', 'lied', intran, ['2A','2C','2D','3A'] ).
verb( 'lift', 'lifts', 'lifting', 'lifted', 'lifted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'light', 'lights', 'lighting', 'lighted', 'lighted', _, ['2C','3A','6A','15A','15B'] ).
verb( 'lighten', 'lightens', 'lightening', 'lightened', 'lightened', _, ['2A','6A'] ).
verb( 'lighter', 'lighters', 'lightering', 'lightered', 'lightered', tran, [] ).
verb( 'like', 'likes', 'liking', 'liked', 'liked', tran, ['6A','6D','7A','17','19B','19C','22'] ).
verb( 'liken', 'likens', 'likening', 'likened', 'likened', tran, ['14'] ).
verb( 'lilt', 'lilts', 'lilting', 'lilted', 'lilted', _, [] ).
verb( 'limber', 'limbers', 'limbering', 'limbered', 'limbered', _, ['2C','15B'] ).
verb( 'lime', 'limes', 'liming', 'limed', 'limed', tran, ['6A'] ).
verb( 'limit', 'limits', 'limiting', 'limited', 'limited', tran, ['6A','14'] ).
verb( 'limn', 'limns', 'limning', 'limned', 'limned', tran, [] ).
verb( 'limp', 'limps', 'limping', 'limped', 'limped', intran, ['2A','2C'] ).
verb( 'line', 'lines', 'lining', 'lined', 'lined', _, ['2C','6A','14','15B'] ).
verb( 'linger', 'lingers', 'lingering', 'lingered', 'lingered', intran, ['2A','2C'] ).
verb( 'link', 'links', 'linking', 'linked', 'linked', _, ['2A','2C','6A','15A','15B'] ).
verb( 'lionize', 'lionizes', 'lionizing', 'lionized', 'lionized', tran, ['6A'] ).
verb( 'lip-read', 'lip-reads', 'lip-reading', 'lip-read', 'lip-read', tran, [] ).
verb( 'liquefy', 'liquefies', 'liquefying', 'liquefied', 'liquefied', _, ['2A','6A'] ).
verb( 'liquidate', 'liquidates', 'liquidating', 'liquidated', 'liquidated', _, ['2A','6A'] ).
verb( 'liquidize', 'liquidizes', 'liquidizing', 'liquidized', 'liquidized', tran, ['6A'] ).
verb( 'lisp', 'lisps', 'lisping', 'lisped', 'lisped', _, ['2A','6A','15B'] ).
verb( 'list', 'lists', 'listing', 'listed', 'listed', _, ['2A','2C','6A'] ).
verb( 'listen', 'listens', 'listening', 'listened', 'listened', intran, ['2A','2C','3A'] ).
verb( 'lithograph', 'lithographs', 'lithographing', 'lithographed', 'lithographed', _, [] ).
verb( 'litigate', 'litigates', 'litigating', 'litigated', 'litigated', _, ['2A','6A'] ).
verb( 'litter', 'litters', 'littering', 'littered', 'littered', _, ['2A','6A','14','15A','15B'] ).
verb( 'live', 'lives', 'living', 'lived', 'lived', _, ['2A','2B','2C','2D','3A','4A','6B','15B'] ).
verb( 'liven', 'livens', 'livening', 'livened', 'livened', _, ['2C','15B'] ).
verb( 'load', 'loads', 'loading', 'loaded', 'loaded', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'loaf', 'loafs', 'loafing', 'loafed', 'loafed', _, ['2A','2C','15A','15B'] ).
verb( 'loan', 'loans', 'loaning', 'loaned', 'loaned', tran, ['6A','14'] ).
verb( 'loathe', 'loathes', 'loathing', 'loathed', 'loathed', tran, ['6A','6C'] ).
verb( 'lob', 'lobs', 'lobbing', 'lobbed', 'lobbed', _, ['2C','6A','15A'] ).
verb( 'lobby', 'lobbies', 'lobbying', 'lobbied', 'lobbied', _, ['2A','2C','6A','15A'] ).
verb( 'localize', 'localizes', 'localizing', 'localized', 'localized', tran, ['6A'] ).
verb( 'locate', 'locates', 'locating', 'located', 'located', tran, ['6A','15A'] ).
verb( 'lock', 'locks', 'locking', 'locked', 'locked', _, ['2A','2C','6A','15A','15B'] ).
verb( 'lodge', 'lodges', 'lodging', 'lodged', 'lodged', _, ['3A','6A','14','15A'] ).
verb( 'loft', 'lofts', 'lofting', 'lofted', 'lofted', tran, ['6A'] ).
verb( 'log', 'logs', 'logging', 'logged', 'logged', tran, ['6A'] ).
verb( 'loiter', 'loiters', 'loitering', 'loitered', 'loitered', _, ['2A','2C','15B'] ).
verb( 'loll', 'lolls', 'lolling', 'lolled', 'lolled', _, ['2A','2C','15B'] ).
verb( 'long', 'longs', 'longing', 'longed', 'longed', intran, ['3A','4C'] ).
verb( 'look', 'looks', 'looking', 'looked', 'looked', _, ['2A','2C','2D','3A','4A','4D','6A','8','15B'] ).
verb( 'loom', 'looms', 'looming', 'loomed', 'loomed', intran, ['2C','2D'] ).
verb( 'loop', 'loops', 'looping', 'looped', 'looped', _, ['2A','6A','15B'] ).
verb( 'loose', 'looses', 'loosing', 'loosed', 'loosed', tran, ['6A'] ).
verb( 'loosen', 'loosens', 'loosening', 'loosened', 'loosened', _, ['2A','6A','15B'] ).
verb( 'loot', 'loots', 'looting', 'looted', 'looted', _, ['2A','6A'] ).
verb( 'lop', 'lops', 'lopping', 'lopped', 'lopped', _, ['6A','15B'] ).
verb( 'lope', 'lopes', 'loping', 'loped', 'loped', intran, ['2A','2C'] ).
verb( 'lord', 'lords', 'lording', 'lorded', 'lorded', tran, [] ).
verb( 'lose', 'loses', 'losing', 'lost', 'lost', _, ['2A','2B','3A','6A','12C','15A'] ).
verb( 'lounge', 'lounges', 'lounging', 'lounged', 'lounged', intran, ['2A','2C'] ).
verb( 'lour', 'lours', 'louring', 'loured', 'loured', intran, ['2A','3A'] ).
verb( 'love', 'loves', 'loving', 'loved', 'loved', tran, ['6A','6D','7A','17'] ).
verb( 'low', 'lows', 'lowing', 'lowed', 'lowed', intran, [] ).
verb( 'lower', 'lowers', 'lowering', 'lowered', 'lowered', _, ['2A','6A','15A'] ).
verb( 'lower', 'lowers', 'lowering', 'lowered', 'lowered', intran, ['2A','3A'] ).
verb( 'lubricate', 'lubricates', 'lubricating', 'lubricated', 'lubricated', tran, ['6A'] ).
verb( 'luff', 'luffs', 'luffing', 'luffed', 'luffed', _, [] ).
verb( 'lug', 'lugs', 'lugging', 'lugged', 'lugged', tran, ['6A','15A','15B'] ).
verb( 'lull', 'lulls', 'lulling', 'lulled', 'lulled', _, ['2A','6A','15A'] ).
verb( 'lumber', 'lumbers', 'lumbering', 'lumbered', 'lumbered', _, ['2C','6A','14','15A','15B'] ).
verb( 'lump', 'lumps', 'lumping', 'lumped', 'lumped', tran, ['2A','6A','15B'] ).
verb( 'lunch', 'lunches', 'lunching', 'lunched', 'lunched', _, [] ).
verb( 'lunge', 'lunges', 'lunging', 'lunged', 'lunged', intran, ['2A','2C'] ).
verb( 'lurch', 'lurches', 'lurching', 'lurched', 'lurched', intran, ['2C'] ).
verb( 'lure', 'lures', 'luring', 'lured', 'lured', tran, ['6A','15B'] ).
verb( 'lurk', 'lurks', 'lurking', 'lurked', 'lurked', intran, ['2C'] ).
verb( 'lust', 'lusts', 'lusting', 'lusted', 'lusted', intran, ['3A'] ).
verb( 'luxuriate', 'luxuriates', 'luxuriating', 'luxuriated', 'luxuriated', intran, ['3A'] ).
verb( 'lynch', 'lynches', 'lynching', 'lynched', 'lynched', tran, ['6A'] ).
verb( 'macadamize', 'macadamizes', 'macadamizing', 'macadamized', 'macadamized', tran, [] ).
verb( 'macerate', 'macerates', 'macerating', 'macerated', 'macerated', _, ['2A','6A'] ).
verb( 'machine', 'machines', 'machining', 'machined', 'machined', tran, ['6A'] ).
verb( 'madden', 'maddens', 'maddening', 'maddened', 'maddened', tran, ['6A'] ).
verb( 'maffick', 'mafficks', 'mafficking', 'mafficked', 'mafficked', intran, [] ).
verb( 'magnetize', 'magnetizes', 'magnetizing', 'magnetized', 'magnetized', tran, ['6A'] ).
verb( 'magnify', 'magnifies', 'magnifying', 'magnified', 'magnified', tran, ['6A'] ).
verb( 'mail', 'mails', 'mailing', 'mailed', 'mailed', tran, ['6A'] ).
verb( 'maim', 'maims', 'maiming', 'maimed', 'maimed', tran, ['6A'] ).
verb( 'maintain', 'maintains', 'maintaining', 'maintained', 'maintained', tran, ['6A','9','25'] ).
verb( 'major', 'majors', 'majoring', 'majored', 'majored', intran, ['3A'] ).
verb( 'make', 'makes', 'making', 'made', 'made', _, ['2A','2C','2D','3A','6A','12A','12B','13A','13B','14','15B','16A','18B','22','23','24A','25'] ).
verb( 'malfunction', 'malfunctions', 'malfunctioning', 'malfunctioned', 'malfunctioned', intran, ['2A'] ).
verb( 'malign', 'maligns', 'maligning', 'maligned', 'maligned', tran, ['6A'] ).
verb( 'malinger', 'malingers', 'malingering', 'malingered', 'malingered', intran, ['2A'] ).
verb( 'malt', 'malts', 'malting', 'malted', 'malted', _, ['2A','6A'] ).
verb( 'maltreat', 'maltreats', 'maltreating', 'maltreated', 'maltreated', tran, ['6A'] ).
verb( 'man', 'mans', 'manning', 'manned', 'manned', tran, ['6A'] ).
verb( 'manacle', 'manacles', 'manacling', 'manacled', 'manacled', tran, ['6A'] ).
verb( 'manage', 'manages', 'managing', 'managed', 'managed', _, ['2A','2C','3A','4A','6A'] ).
verb( 'mandate', 'mandates', 'mandating', 'mandated', 'mandated', tran, ['6A'] ).
verb( 'mangle', 'mangles', 'mangling', 'mangled', 'mangled', tran, ['6A'] ).
verb( 'manhandle', 'manhandles', 'manhandling', 'manhandled', 'manhandled', tran, [] ).
verb( 'manicure', 'manicures', 'manicuring', 'manicured', 'manicured', tran, ['6A'] ).
verb( 'manifest', 'manifests', 'manifesting', 'manifested', 'manifested', tran, ['6A'] ).
verb( 'manifold', 'manifolds', 'manifolding', 'manifolded', 'manifolded', tran, ['6A'] ).
verb( 'manipulate', 'manipulates', 'manipulating', 'manipulated', 'manipulated', tran, ['6A'] ).
verb( 'manoeuvre', 'manoeuvres', 'manoeuvring', 'manoeuvred', 'manoeuvred', _, ['2A','2C','6A','15A'] ).
verb( 'mantle', 'mantles', 'mantling', 'mantled', 'mantled', _, ['2C','6A'] ).
verb( 'manufacture', 'manufactures', 'manufacturing', 'manufactured', 'manufactured', tran, ['6A'] ).
verb( 'manumit', 'manumits', 'manumitting', 'manumitted', 'manumitted', tran, [] ).
verb( 'manure', 'manures', 'manuring', 'manured', 'manured', tran, ['6A'] ).
verb( 'map', 'maps', 'mapping', 'mapped', 'mapped', tran, ['6A','15B'] ).
verb( 'mar', 'mars', 'marring', 'marred', 'marred', tran, ['6A'] ).
verb( 'maraud', 'marauds', 'marauding', 'marauded', 'marauded', intran, ['2A'] ).
verb( 'march', 'marches', 'marching', 'marched', 'marched', _, ['2A','2B','2C','3A','15A','15B'] ).
verb( 'marinade', 'marinades', 'marinading', 'marinaded', 'marinaded', tran, [] ).
verb( 'marinate', 'marinates', 'marinating', 'marinated', 'marinated', tran, [] ).
verb( 'mark', 'marks', 'marking', 'marked', 'marked', tran, ['2A','6A','8','10','15A','15B','22'] ).
verb( 'market', 'markets', 'marketing', 'marketed', 'marketed', _, ['2A','6A'] ).
verb( 'maroon', 'maroons', 'marooning', 'marooned', 'marooned', tran, ['6A'] ).
verb( 'marry', 'marries', 'marrying', 'married', 'married', _, ['2A','2D','4A','6A','15B'] ).
verb( 'marshal', 'marshals', 'marshalling', 'marshalled', 'marshalled', tran, ['6A','15A','15B'] ).
verb( 'martyr', 'martyrs', 'martyring', 'martyred', 'martyred', tran, ['6A'] ).
verb( 'marvel', 'marvels', 'marvelling', 'marvelled', 'marvelled', intran, ['3A','3B'] ).
verb( 'mash', 'mashes', 'mashing', 'mashed', 'mashed', tran, ['6A'] ).
verb( 'mask', 'masks', 'masking', 'masked', 'masked', tran, ['6A'] ).
verb( 'masquerade', 'masquerades', 'masquerading', 'masqueraded', 'masqueraded', intran, ['2A','2C'] ).
verb( 'mass', 'masses', 'massing', 'massed', 'massed', _, ['2A','6A'] ).
verb( 'mass-produce', 'mass-produces', 'mass-producing', 'mass-produced', 'mass-produced', tran, [] ).
verb( 'massacre', 'massacres', 'massacring', 'massacred', 'massacred', tran, ['6A'] ).
verb( 'massage', 'massages', 'massaging', 'massaged', 'massaged', tran, ['6A'] ).
verb( 'master', 'masters', 'mastering', 'mastered', 'mastered', tran, ['6A'] ).
verb( 'mastermind', 'masterminds', 'masterminding', 'masterminded', 'masterminded', tran, [] ).
verb( 'masticate', 'masticates', 'masticating', 'masticated', 'masticated', tran, ['6A'] ).
verb( 'masturbate', 'masturbates', 'masturbating', 'masturbated', 'masturbated', _, ['2A','6A'] ).
verb( 'mat', 'mats', 'matting', 'matted', 'matted', _, ['2A','2C','6A','15A'] ).
verb( 'match', 'matches', 'matching', 'matched', 'matched', _, ['2A','6A','12B','13B','14'] ).
verb( 'mate', 'mates', 'mating', 'mated', 'mated', _, ['2A','3A','6A','14'] ).
verb( 'materialize', 'materializes', 'materializing', 'materialized', 'materialized', intran, ['2A','6A'] ).
verb( 'matriculate', 'matriculates', 'matriculating', 'matriculated', 'matriculated', _, ['2A','2C','6A'] ).
verb( 'matter', 'matters', 'mattering', 'mattered', 'mattered', intran, ['2A','2C'] ).
verb( 'maturate', 'maturates', 'maturating', 'maturated', 'maturated', intran, ['2A'] ).
verb( 'mature', 'matures', 'maturing', 'matured', 'matured', _, ['2A','6A'] ).
verb( 'maul', 'mauls', 'mauling', 'mauled', 'mauled', tran, ['6A','15B'] ).
verb( 'maunder', 'maunders', 'maundering', 'maundered', 'maundered', intran, ['2A','2C'] ).
verb( 'maximize', 'maximizes', 'maximizing', 'maximized', 'maximized', tran, ['6A'] ).
verb( 'may', 'may', '-', '-', '-', unknown, ['5'] ).
verb( 'mean', 'means', 'meaning', 'meant', 'meant', tran, ['6A','6A','6C','7A','9','12A','13A','14','16B','17'] ).
verb( 'meander', 'meanders', 'meandering', 'meandered', 'meandered', intran, ['2A','2C'] ).
verb( 'measure', 'measures', 'measuring', 'measured', 'measured', _, ['2A','2B','6A','15A','15B'] ).
verb( 'mechanize', 'mechanizes', 'mechanizing', 'mechanized', 'mechanized', tran, ['6A'] ).
verb( 'meddle', 'meddles', 'meddling', 'meddled', 'meddled', intran, ['2A','3A'] ).
verb( 'mediate', 'mediates', 'mediating', 'mediated', 'mediated', _, ['2A','3A','6A'] ).
verb( 'medicate', 'medicates', 'medicating', 'medicated', 'medicated', tran, ['6A'] ).
verb( 'meditate', 'meditates', 'meditating', 'meditated', 'meditated', _, ['2A','3A','6A'] ).
verb( 'meet', 'meets', 'meeting', 'met', 'met', _, ['2A','2C','6A'] ).
verb( 'meliorate', 'meliorates', 'meliorating', 'meliorated', 'meliorated', _, ['2A','6A'] ).
verb( 'mellow', 'mellows', 'mellowing', 'mellowed', 'mellowed', _, ['2A','6A'] ).
verb( 'melt', 'melts', 'melting', 'melted', 'melted', _, ['2A','2C','6A','15B'] ).
verb( 'memorialize', 'memorializes', 'memorializing', 'memorialized', 'memorialized', tran, ['6A'] ).
verb( 'memorize', 'memorizes', 'memorizing', 'memorized', 'memorized', tran, ['6A'] ).
verb( 'menace', 'menaces', 'menacing', 'menaced', 'menaced', tran, ['6A'] ).
verb( 'mend', 'mends', 'mending', 'mended', 'mended', _, ['2A','6A'] ).
verb( 'menstruate', 'menstruates', 'menstruating', 'menstruated', 'menstruated', intran, ['2A'] ).
verb( 'mention', 'mentions', 'mentioning', 'mentioned', 'mentioned', tran, ['6A','6C','9','13A'] ).
verb( 'mercerize', 'mercerizes', 'mercerizing', 'mercerized', 'mercerized', tran, ['6A'] ).
verb( 'merge', 'merges', 'merging', 'merged', 'merged', _, ['2A','3A','6A','14'] ).
verb( 'merit', 'merits', 'meriting', 'merited', 'merited', tran, ['6A'] ).
verb( 'mesh', 'meshes', 'meshing', 'meshed', 'meshed', _, ['2A','3A','6A'] ).
verb( 'mesmerize', 'mesmerizes', 'mesmerizing', 'mesmerized', 'mesmerized', tran, ['6A'] ).
verb( 'mess', 'messes', 'messing', 'messed', 'messed', _, ['2C','6A','15B'] ).
verb( 'metal', 'metals', 'metalling', 'metalled', 'metalled', tran, ['6A'] ).
verb( 'metamorphose', 'metamorphoses', 'metamorphosing', 'metamorphosed', 'metamorphosed', _, ['3A','6A','14'] ).
verb( 'mete', 'metes', 'meting', 'meted', 'meted', tran, ['15B'] ).
verb( 'metricize', 'metricizes', 'metricizing', 'metricized', 'metricized', tran, [] ).
verb( 'mew', 'mews', 'mewing', 'mewed', 'mewed', intran, ['2A'] ).
verb( 'miaou', 'miaous', 'miaouing', 'miaoued', 'miaoued', intran, [] ).
verb( 'miaow', 'miaows', 'miaowing', 'miaowed', 'miaowed', intran, ['2A'] ).
verb( 'microfilm', 'microfilms', 'microfilming', 'microfilmed', 'microfilmed', tran, ['6A'] ).
verb( 'migrate', 'migrates', 'migrating', 'migrated', 'migrated', intran, ['2A','3A'] ).
verb( 'mildew', 'mildews', 'mildewing', 'mildewed', 'mildewed', _, ['2A','6A'] ).
verb( 'militate', 'militates', 'militating', 'militated', 'militated', intran, ['3A'] ).
verb( 'milk', 'milks', 'milking', 'milked', 'milked', _, ['2A','6A'] ).
verb( 'mill', 'mills', 'milling', 'milled', 'milled', _, ['2C','6A'] ).
verb( 'mime', 'mimes', 'miming', 'mimed', 'mimed', _, ['2A','6A'] ).
verb( 'mimeograph', 'mimeographs', 'mimeographing', 'mimeographed', 'mimeographed', tran, ['6A'] ).
verb( 'mimic', 'mimics', 'mimicking', 'mimicked', 'mimicked', tran, ['6A'] ).
verb( 'mince', 'minces', 'mincing', 'minced', 'minced', _, ['2A','6A'] ).
verb( 'mind', 'minds', 'minding', 'minded', 'minded', _, ['2A','6A','6C','9','19C'] ).
verb( 'mine', 'mines', 'mining', 'mined', 'mined', _, ['2A','3A','6A'] ).
verb( 'mingle', 'mingles', 'mingling', 'mingled', 'mingled', _, ['2A','2C','6A','14'] ).
verb( 'miniaturize', 'miniaturizes', 'miniaturizing', 'miniaturized', 'miniaturized', tran, ['6A'] ).
verb( 'minimize', 'minimizes', 'minimizing', 'minimized', 'minimized', tran, ['6A'] ).
verb( 'minister', 'ministers', 'ministering', 'ministered', 'ministered', intran, ['3A'] ).
verb( 'mint', 'mints', 'minting', 'minted', 'minted', tran, ['6A'] ).
verb( 'minute', 'minutes', 'minuting', 'minuted', 'minuted', tran, ['6A'] ).
verb( 'mire', 'mires', 'miring', 'mired', 'mired', _, ['2A','6A'] ).
verb( 'mirror', 'mirrors', 'mirroring', 'mirrored', 'mirrored', tran, ['6A'] ).
verb( 'misadvise', 'misadvises', 'misadvising', 'misadvised', 'misadvised', tran, ['6A'] ).
verb( 'misapply', 'misapplies', 'misapplying', 'misapplied', 'misapplied', tran, ['6A'] ).
verb( 'misapprehend', 'misapprehends', 'misapprehending', 'misapprehended', 'misapprehended', tran, ['6A'] ).
verb( 'misappropriate', 'misappropriates', 'misappropriating', 'misappropriated', 'misappropriated', tran, ['6A'] ).
verb( 'misbehave', 'misbehaves', 'misbehaving', 'misbehaved', 'misbehaved', _, ['2A','6B'] ).
verb( 'miscalculate', 'miscalculates', 'miscalculating', 'miscalculated', 'miscalculated', _, [] ).
verb( 'miscall', 'miscalls', 'miscalling', 'miscalled', 'miscalled', tran, [] ).
verb( 'miscarry', 'miscarries', 'miscarrying', 'miscarried', 'miscarried', tran, ['2A'] ).
verb( 'miscast', 'miscasts', 'miscasting', 'miscast', 'miscast', tran, ['6A'] ).
verb( 'misconceive', 'misconceives', 'misconceiving', 'misconceived', 'misconceived', _, ['3A','6A'] ).
verb( 'misconduct', 'misconducts', 'misconducting', 'misconducted', 'misconducted', tran, ['6A','6B','14'] ).
verb( 'misconstrue', 'misconstrues', 'misconstruing', 'misconstrued', 'misconstrued', tran, ['6A'] ).
verb( 'miscount', 'miscounts', 'miscounting', 'miscounted', 'miscounted', _, ['2A','6A'] ).
verb( 'misdate', 'misdates', 'misdating', 'misdated', 'misdated', tran, ['6A'] ).
verb( 'misdeal', 'misdeals', 'misdealing', 'misdealt', 'misdealt', _, ['2A','6A'] ).
verb( 'misdirect', 'misdirects', 'misdirecting', 'misdirected', 'misdirected', tran, ['6A'] ).
verb( 'misfire', 'misfires', 'misfiring', 'misfired', 'misfired', intran, ['2A'] ).
verb( 'misgive', 'misgives', 'misgiving', 'misgave', 'misgiven', tran, [] ).
verb( 'misgovern', 'misgoverns', 'misgoverning', 'misgoverned', 'misgoverned', tran, ['6A'] ).
verb( 'misguide', 'misguides', 'misguiding', 'misguided', 'misguided', tran, ['6A','14'] ).
verb( 'mishandle', 'mishandles', 'mishandling', 'mishandled', 'mishandled', tran, [] ).
verb( 'misinform', 'misinforms', 'misinforming', 'misinformed', 'misinformed', tran, ['6A'] ).
verb( 'misinterpret', 'misinterprets', 'misinterpreting', 'misinterpreted', 'misinterpreted', tran, ['6A'] ).
verb( 'misjudge', 'misjudges', 'misjudging', 'misjudged', 'misjudged', _, ['2A','6A'] ).
verb( 'mislay', 'mislays', 'mislaying', 'mislaid', 'mislaid', tran, ['6A'] ).
verb( 'mislead', 'misleads', 'misleading', 'misled', 'misled', tran, ['6A'] ).
verb( 'mismanage', 'mismanages', 'mismanaging', 'mismanaged', 'mismanaged', tran, ['6A'] ).
verb( 'misname', 'misnames', 'misnaming', 'misnamed', 'misnamed', tran, ['6A'] ).
verb( 'misplace', 'misplaces', 'misplacing', 'misplaced', 'misplaced', tran, ['6A'] ).
verb( 'misprint', 'misprints', 'misprinting', 'misprinted', 'misprinted', tran, ['6A'] ).
verb( 'mispronounce', 'mispronounces', 'mispronouncing', 'mispronounced', 'mispronounced', tran, ['6A'] ).
verb( 'misquote', 'misquotes', 'misquoting', 'misquoted', 'misquoted', tran, ['6A'] ).
verb( 'misread', 'misreads', 'misreading', 'misread', 'misread', tran, ['6A'] ).
verb( 'misrepresent', 'misrepresents', 'misrepresenting', 'misrepresented', 'misrepresented', tran, [] ).
verb( 'miss', 'misses', 'missing', 'missed', 'missed', _, ['2A','2C','6A','6B','6C','15B'] ).
verb( 'misspell', 'misspells', 'misspelling', 'misspelled', 'misspelled', tran, ['6A'] ).
verb( 'misspend', 'misspends', 'misspending', 'misspent', 'misspent', tran, ['6A'] ).
verb( 'misstate', 'misstates', 'misstating', 'misstated', 'misstated', tran, ['6A'] ).
verb( 'mist', 'mists', 'misting', 'misted', 'misted', _, ['2C','6A'] ).
verb( 'mistake', 'mistakes', 'mistaking', 'mistook', 'mistaken', _, ['6A','10','14'] ).
verb( 'mistime', 'mistimes', 'mistiming', 'mistimed', 'mistimed', tran, [] ).
verb( 'mistranslate', 'mistranslates', 'mistranslating', 'mistranslated', 'mistranslated', tran, ['6A'] ).
verb( 'mistrust', 'mistrusts', 'mistrusting', 'mistrusted', 'mistrusted', tran, ['6A'] ).
verb( 'misunderstand', 'misunderstands', 'misunderstanding', 'misunderstood', 'misunderstood', tran, ['6A'] ).
verb( 'misuse', 'misuses', 'misusing', 'misused', 'misused', tran, ['6A'] ).
verb( 'mitigate', 'mitigates', 'mitigating', 'mitigated', 'mitigated', tran, ['6A'] ).
verb( 'mix', 'mixes', 'mixing', 'mixed', 'mixed', _, ['2A','2C','3A','6A','12B','13B','14','15B'] ).
verb( 'mizzle', 'mizzles', 'mizzling', 'mizzled', 'mizzled', intran, ['2A'] ).
verb( 'moan', 'moans', 'moaning', 'moaned', 'moaned', _, ['2A','2C','15B'] ).
verb( 'mob', 'mobs', 'mobbing', 'mobbed', 'mobbed', tran, ['6A'] ).
verb( 'mobilize', 'mobilizes', 'mobilizing', 'mobilized', 'mobilized', _, ['2A','6A'] ).
verb( 'mock', 'mocks', 'mocking', 'mocked', 'mocked', _, ['3A','6A'] ).
verb( 'model', 'models', 'modelling', 'modelled', 'modelled', _, ['2A','6A','14','15A'] ).
verb( 'moderate', 'moderates', 'moderating', 'moderated', 'moderated', _, ['2A','6A'] ).
verb( 'modernize', 'modernizes', 'modernizing', 'modernized', 'modernized', tran, ['6A'] ).
verb( 'modify', 'modifies', 'modifying', 'modified', 'modified', tran, ['6A'] ).
verb( 'modulate', 'modulates', 'modulating', 'modulated', 'modulated', _, ['2C','6A'] ).
verb( 'moil', 'moils', 'moiling', 'moiled', 'moiled', intran, [] ).
verb( 'moisten', 'moistens', 'moistening', 'moistened', 'moistened', _, ['2A','6A'] ).
verb( 'molest', 'molests', 'molesting', 'molested', 'molested', tran, ['6A'] ).
verb( 'mollify', 'mollifies', 'mollifying', 'mollified', 'mollified', tran, ['6A'] ).
verb( 'mollycoddle', 'mollycoddles', 'mollycoddling', 'mollycoddled', 'mollycoddled', tran, ['6A'] ).
verb( 'monetize', 'monetizes', 'monetizing', 'monetized', 'monetized', tran, ['6A'] ).
verb( 'monitor', 'monitors', 'monitoring', 'monitored', 'monitored', _, [] ).
verb( 'monkey', 'monkeys', 'monkeying', 'monkeyed', 'monkeyed', intran, ['2C'] ).
verb( 'monopolize', 'monopolizes', 'monopolizing', 'monopolized', 'monopolized', tran, ['6A'] ).
verb( 'moo', 'moos', 'mooing', 'mooed', 'mooed', intran, [] ).
verb( 'mooch', 'mooches', 'mooching', 'mooched', 'mooched', intran, ['2C'] ).
verb( 'moon', 'moons', 'mooning', 'mooned', 'mooned', _, ['2C','15B'] ).
verb( 'moor', 'moors', 'mooring', 'moored', 'moored', tran, ['6A','15A'] ).
verb( 'moot', 'moots', 'mooting', 'mooted', 'mooted', tran, ['6A'] ).
verb( 'mop', 'mops', 'mopping', 'mopped', 'mopped', _, ['6A','15B'] ).
verb( 'mope', 'mopes', 'moping', 'moped', 'moped', intran, ['2A','2C'] ).
verb( 'moralize', 'moralizes', 'moralizing', 'moralized', 'moralized', _, ['2A','3A','6A'] ).
verb( 'mortar', 'mortars', 'mortaring', 'mortared', 'mortared', tran, ['6A'] ).
verb( 'mortgage', 'mortgages', 'mortgaging', 'mortgaged', 'mortgaged', tran, ['6A','14'] ).
verb( 'mortice', 'mortices', 'morticing', 'morticed', 'morticed', tran, ['15A','15B'] ).
verb( 'mortify', 'mortifies', 'mortifying', 'mortified', 'mortified', _, ['2A','6A'] ).
verb( 'mortise', 'mortises', 'mortising', 'mortised', 'mortised', tran, ['15A','15B'] ).
verb( 'mosey', 'moseys', 'moseying', 'moseyed', 'moseyed', intran, ['2A','2C'] ).
verb( 'mother', 'mothers', 'mothering', 'mothered', 'mothered', tran, ['6A'] ).
verb( 'mothproof', 'mothproofs', 'mothproofing', 'mothproofed', 'mothproofed', tran, [] ).
verb( 'motion', 'motions', 'motioning', 'motioned', 'motioned', _, ['3A','15A','15B','17'] ).
verb( 'motivate', 'motivates', 'motivating', 'motivated', 'motivated', tran, ['6A'] ).
verb( 'motor', 'motors', 'motoring', 'motored', 'motored', intran, ['2A','2C'] ).
verb( 'motorize', 'motorizes', 'motorizing', 'motorized', 'motorized', tran, ['6A'] ).
verb( 'mottle', 'mottles', 'mottling', 'mottled', 'mottled', tran, ['6A'] ).
verb( 'mould', 'moulds', 'moulding', 'moulded', 'moulded', _, ['2A','6A','14'] ).
verb( 'moulder', 'moulders', 'mouldering', 'mouldered', 'mouldered', intran, ['2A','2C'] ).
verb( 'moult', 'moults', 'moulting', 'moulted', 'moulted', _, ['2A','6A'] ).
verb( 'mount', 'mounts', 'mounting', 'mounted', 'mounted', _, ['2A','2C','6A'] ).
verb( 'mourn', 'mourns', 'mourning', 'mourned', 'mourned', _, ['3A','6A'] ).
verb( 'mouse', 'mouses', 'mousing', 'moused', 'moused', intran, ['2A'] ).
verb( 'mouth', 'mouths', 'mouthing', 'mouthed', 'mouthed', _, ['2A','6A'] ).
verb( 'move', 'moves', 'moving', 'moved', 'moved', _, ['2A','2C','3A','6A','9','15A','15B','17'] ).
verb( 'mow', 'mows', 'mowing', 'mowed', 'mowed', tran, ['2A','6A','15B'] ).
verb( 'mow', 'mows', 'mowing', 'mowed', 'mowed', intran, [] ).
verb( 'muck', 'mucks', 'mucking', 'mucked', 'mucked', _, ['2C','6A','15B'] ).
verb( 'mud', 'muds', 'mudding', 'mudded', 'mudded', tran, ['6A'] ).
verb( 'muddle', 'muddles', 'muddling', 'muddled', 'muddled', _, ['2C','6A','15B'] ).
verb( 'muddy', 'muddies', 'muddying', 'muddied', 'muddied', tran, ['6A'] ).
verb( 'muff', 'muffs', 'muffing', 'muffed', 'muffed', tran, ['6A'] ).
verb( 'muffle', 'muffles', 'muffling', 'muffled', 'muffled', tran, ['6A','15B'] ).
verb( 'mug', 'mugs', 'mugging', 'mugged', 'mugged', tran, ['6A','15B'] ).
verb( 'mulch', 'mulches', 'mulching', 'mulched', 'mulched', tran, [] ).
verb( 'mulct', 'mulcts', 'mulcting', 'mulcted', 'mulcted', tran, ['12C','14'] ).
verb( 'mull', 'mulls', 'mulling', 'mulled', 'mulled', tran, ['6A','15A'] ).
verb( 'multiply', 'multiplies', 'multiplying', 'multiplied', 'multiplied', _, ['2A','6A','14'] ).
verb( 'mumble', 'mumbles', 'mumbling', 'mumbled', 'mumbled', _, ['2A','2C','6A'] ).
verb( 'mummify', 'mummifies', 'mummifying', 'mummified', 'mummified', tran, ['6A'] ).
verb( 'munch', 'munches', 'munching', 'munched', 'munched', _, ['2A','2C','6A'] ).
verb( 'munition', 'munitions', 'munitioning', 'munitioned', 'munitioned', tran, ['6A'] ).
verb( 'murder', 'murders', 'murdering', 'murdered', 'murdered', tran, ['6A'] ).
verb( 'murmur', 'murmurs', 'murmuring', 'murmured', 'murmured', _, ['2A','2C','3A','6A'] ).
verb( 'muscle', 'muscles', 'muscling', 'muscled', 'muscled', intran, ['2C'] ).
verb( 'muse', 'muses', 'musing', 'mused', 'mused', intran, ['2A','3A'] ).
verb( 'mushroom', 'mushrooms', 'mushrooming', 'mushroomed', 'mushroomed', intran, ['2C'] ).
verb( 'muss', 'musses', 'mussing', 'mussed', 'mussed', tran, ['6A','15B'] ).
verb( 'must', 'must', '-', '-', '-', unknown, ['5'] ).
verb( 'muster', 'musters', 'mustering', 'mustered', 'mustered', _, ['2A','6A','15B'] ).
verb( 'mute', 'mutes', 'muting', 'muted', 'muted', tran, ['6A'] ).
verb( 'mutilate', 'mutilates', 'mutilating', 'mutilated', 'mutilated', tran, ['6A'] ).
verb( 'mutiny', 'mutinies', 'mutinying', 'mutinied', 'mutinied', intran, ['2A','3A'] ).
verb( 'mutter', 'mutters', 'muttering', 'muttered', 'muttered', _, ['2A','2C','6A','14'] ).
verb( 'muzzle', 'muzzles', 'muzzling', 'muzzled', 'muzzled', tran, ['6A'] ).
verb( 'mystify', 'mystifies', 'mystifying', 'mystified', 'mystified', tran, ['6A'] ).
verb( 'nab', 'nabs', 'nabbing', 'nabbed', 'nabbed', tran, [] ).
verb( 'nag', 'nags', 'nagging', 'nagged', 'nagged', _, ['2A','2C','3A','6A'] ).
verb( 'nail', 'nails', 'nailing', 'nailed', 'nailed', tran, ['15A','15B'] ).
verb( 'name', 'names', 'naming', 'named', 'named', tran, ['6A','14','23'] ).
verb( 'name-drop', 'name-drops', 'name-dropping', 'name-dropped', 'name-dropped', intran, ['2A'] ).
verb( 'nap', 'naps', 'napping', 'napped', 'napped', intran, [] ).
verb( 'nark', 'narks', 'narking', 'narked', 'narked', tran, [] ).
verb( 'narrate', 'narrates', 'narrating', 'narrated', 'narrated', tran, ['6A'] ).
verb( 'narrow', 'narrows', 'narrowing', 'narrowed', 'narrowed', _, ['2A','6A'] ).
verb( 'nasalize', 'nasalizes', 'nasalizing', 'nasalized', 'nasalized', tran, ['6A'] ).
verb( 'nationalize', 'nationalizes', 'nationalizing', 'nationalized', 'nationalized', tran, ['6A'] ).
verb( 'natter', 'natters', 'nattering', 'nattered', 'nattered', intran, ['2A','2C'] ).
verb( 'naturalize', 'naturalizes', 'naturalizing', 'naturalized', 'naturalized', _, ['2A','6A'] ).
verb( 'nauseate', 'nauseates', 'nauseating', 'nauseated', 'nauseated', tran, ['6A'] ).
verb( 'navigate', 'navigates', 'navigating', 'navigated', 'navigated', _, ['2A','6A'] ).
verb( 'near', 'nears', 'nearing', 'neared', 'neared', _, ['2A','6A'] ).
verb( 'necessitate', 'necessitates', 'necessitating', 'necessitated', 'necessitated', tran, ['6A','6C'] ).
verb( 'neck', 'necks', 'necking', 'necked', 'necked', intran, [] ).
verb( 'need', 'needs', 'needing', 'needed', 'needed', unknown, ['5','6A','6E','7A'] ).
verb( 'need', 'needs', 'needing', 'needed', 'needed', tran, ['5','6A','6E','7A'] ).
verb( 'needle', 'needles', 'needling', 'needled', 'needled', tran, ['6A','15A'] ).
verb( 'negate', 'negates', 'negating', 'negated', 'negated', tran, ['6A'] ).
verb( 'negative', 'negatives', 'negativing', 'negatived', 'negatived', tran, ['6A'] ).
verb( 'neglect', 'neglects', 'neglecting', 'neglected', 'neglected', tran, ['6A','6C','7A'] ).
verb( 'negotiate', 'negotiates', 'negotiating', 'negotiated', 'negotiated', _, ['2A','3A','6A','14'] ).
verb( 'neigh', 'neighs', 'neighing', 'neighed', 'neighed', intran, [] ).
verb( 'neighbour', 'neighbours', 'neighbouring', 'neighboured', 'neighboured', _, ['3A','6A'] ).
verb( 'nerve', 'nerves', 'nerving', 'nerved', 'nerved', tran, ['6A','14','16A'] ).
verb( 'nest', 'nests', 'nesting', 'nested', 'nested', intran, ['2A','2C'] ).
verb( 'nestle', 'nestles', 'nestling', 'nestled', 'nestled', _, ['2C','15A'] ).
verb( 'net', 'nets', 'netting', 'netted', 'netted', tran, ['6A'] ).
verb( 'nett', 'netts', 'netting', 'netted', 'netted', tran, ['6A'] ).
verb( 'nettle', 'nettles', 'nettling', 'nettled', 'nettled', tran, ['6A'] ).
verb( 'neuter', 'neuters', 'neutering', 'neutered', 'neutered', tran, [] ).
verb( 'neutralize', 'neutralizes', 'neutralizing', 'neutralized', 'neutralized', tran, ['6A'] ).
verb( 'nibble', 'nibbles', 'nibbling', 'nibbled', 'nibbled', _, ['2A','3A','6A'] ).
verb( 'nick', 'nicks', 'nicking', 'nicked', 'nicked', tran, [] ).
verb( 'nickel', 'nickels', 'nickelling', 'nickelled', 'nickelled', tran, [] ).
verb( 'nickname', 'nicknames', 'nicknaming', 'nicknamed', 'nicknamed', tran, ['6A','23'] ).
verb( 'niggle', 'niggles', 'niggling', 'niggled', 'niggled', intran, [] ).
verb( 'nip', 'nips', 'nipping', 'nipped', 'nipped', _, ['2A','2C','6A','15A','15B'] ).
verb( 'nobble', 'nobbles', 'nobbling', 'nobbled', 'nobbled', tran, ['6A'] ).
verb( 'nod', 'nods', 'nodding', 'nodded', 'nodded', _, ['2A','2C','3A','4A','6A','12A','13A'] ).
verb( 'noise', 'noises', 'noising', 'noised', 'noised', tran, [] ).
verb( 'nominate', 'nominates', 'nominating', 'nominated', 'nominated', tran, ['6A','14','23'] ).
verb( 'nonplus', 'nonplusses', 'nonplussing', 'nonplussed', 'nonplussed', tran, ['6A'] ).
verb( 'noose', 'nooses', 'noosing', 'noosed', 'noosed', tran, [] ).
verb( 'normalize', 'normalizes', 'normalizing', 'normalized', 'normalized', tran, [] ).
verb( 'nose', 'noses', 'nosing', 'nosed', 'nosed', _, ['2C','3A','15A','15B'] ).
verb( 'nosedive', 'nosedives', 'nosediving', 'nosedived', 'nosedived', intran, [] ).
verb( 'nosh', 'noshes', 'noshing', 'noshed', 'noshed', intran, [] ).
verb( 'notch', 'notches', 'notching', 'notched', 'notched', tran, ['6A','15B'] ).
verb( 'note', 'notes', 'noting', 'noted', 'noted', tran, ['6A','8','9','10','15B'] ).
verb( 'notice', 'notices', 'noticing', 'noticed', 'noticed', _, ['2A','6A','8','9','10','18A','19A'] ).
verb( 'notify', 'notifies', 'notifying', 'notified', 'notified', tran, ['6A','11','14'] ).
verb( 'nourish', 'nourishes', 'nourishing', 'nourished', 'nourished', tran, ['6A'] ).
verb( 'nudge', 'nudges', 'nudging', 'nudged', 'nudged', tran, ['6A'] ).
verb( 'nullify', 'nullifies', 'nullifying', 'nullified', 'nullified', tran, ['6A'] ).
verb( 'numb', 'numbs', 'numbing', 'numbed', 'numbed', tran, ['6A'] ).
verb( 'number', 'numbers', 'numbering', 'numbered', 'numbered', tran, ['2C','6A','14'] ).
verb( 'nurse', 'nurses', 'nursing', 'nursed', 'nursed', tran, ['6A'] ).
verb( 'nurture', 'nurtures', 'nurturing', 'nurtured', 'nurtured', tran, [] ).
verb( 'nut', 'nuts', 'nutting', 'nutted', 'nutted', intran, [] ).
verb( 'nuzzle', 'nuzzles', 'nuzzling', 'nuzzled', 'nuzzled', _, ['2C','6A'] ).
verb( 'obey', 'obeys', 'obeying', 'obeyed', 'obeyed', _, ['2A','6A'] ).
verb( 'obfuscate', 'obfuscates', 'obfuscating', 'obfuscated', 'obfuscated', tran, ['6A'] ).
verb( 'object', 'objects', 'objecting', 'objected', 'objected', _, ['2A','3A','9'] ).
verb( 'objurgate', 'objurgates', 'objurgating', 'objurgated', 'objurgated', tran, ['6A'] ).
verb( 'obligate', 'obligates', 'obligating', 'obligated', 'obligated', tran, ['17'] ).
verb( 'oblige', 'obliges', 'obliging', 'obliged', 'obliged', tran, ['6A','14','17'] ).
verb( 'obliterate', 'obliterates', 'obliterating', 'obliterated', 'obliterated', tran, ['6A'] ).
verb( 'obscure', 'obscures', 'obscuring', 'obscured', 'obscured', tran, ['6A'] ).
verb( 'observe', 'observes', 'observing', 'observed', 'observed', _, ['2A','6A','8','9','10','18A','19A','25'] ).
verb( 'obsess', 'obsesses', 'obsessing', 'obsessed', 'obsessed', tran, ['6A'] ).
verb( 'obstruct', 'obstructs', 'obstructing', 'obstructed', 'obstructed', tran, ['6A'] ).
verb( 'obtain', 'obtains', 'obtaining', 'obtained', 'obtained', _, ['2A','6A'] ).
verb( 'obtrude', 'obtrudes', 'obtruding', 'obtruded', 'obtruded', _, ['2A','14'] ).
verb( 'obviate', 'obviates', 'obviating', 'obviated', 'obviated', tran, ['6A'] ).
verb( 'occasion', 'occasions', 'occasioning', 'occasioned', 'occasioned', tran, ['6A','12A','13A'] ).
verb( 'occupy', 'occupies', 'occupying', 'occupied', 'occupied', tran, ['6A'] ).
verb( 'occur', 'occurs', 'occurring', 'occurred', 'occurred', intran, ['2A','2C','3A'] ).
verb( 'offend', 'offends', 'offending', 'offended', 'offended', _, ['3A','6A'] ).
verb( 'offer', 'offers', 'offering', 'offered', 'offered', _, ['2A','6A','7A','12A','13A','14','15B'] ).
verb( 'officiate', 'officiates', 'officiating', 'officiated', 'officiated', intran, ['2A','2C','3A'] ).
verb( 'offset', 'offsets', 'offsetting', 'offset', 'offset', tran, ['6A','14'] ).
verb( 'ogle', 'ogles', 'ogling', 'ogled', 'ogled', _, ['3A','6A'] ).
verb( 'oil', 'oils', 'oiling', 'oiled', 'oiled', tran, ['6A'] ).
verb( 'okay', 'okays', 'okaying', 'okayed', 'okayed', tran, ['6A'] ).
verb( 'omen', 'omens', 'omening', 'omened', 'omened', tran, ['6A'] ).
verb( 'omit', 'omits', 'omitting', 'omitted', 'omitted', tran, ['6A','6C','7A'] ).
verb( 'ooze', 'oozes', 'oozing', 'oozed', 'oozed', _, ['2C','6A'] ).
verb( 'open', 'opens', 'opening', 'opened', 'opened', _, ['2A','2C','3A','6A','12C','14','15A','15B','16A'] ).
verb( 'operate', 'operates', 'operating', 'operated', 'operated', _, ['2A','2C','3A','4A','6A'] ).
verb( 'opine', 'opines', 'opining', 'opined', 'opined', tran, ['9'] ).
verb( 'oppose', 'opposes', 'opposing', 'opposed', 'opposed', tran, ['6A','14'] ).
verb( 'oppress', 'oppresses', 'oppressing', 'oppressed', 'oppressed', tran, ['6A'] ).
verb( 'oppugn', 'oppugns', 'oppugning', 'oppugned', 'oppugned', tran, ['6A'] ).
verb( 'opt', 'opts', 'opting', 'opted', 'opted', intran, ['3A'] ).
verb( 'orate', 'orates', 'orating', 'orated', 'orated', intran, ['2A'] ).
verb( 'orbit', 'orbits', 'orbiting', 'orbited', 'orbited', _, ['2A','2C','6A'] ).
verb( 'orchestrate', 'orchestrates', 'orchestrating', 'orchestrated', 'orchestrated', tran, ['6A'] ).
verb( 'ordain', 'ordains', 'ordaining', 'ordained', 'ordained', tran, ['6A','9','23'] ).
verb( 'order', 'orders', 'ordering', 'ordered', 'ordered', tran, ['6A','9','12B','13B','15A','15B','17'] ).
verb( 'organize', 'organizes', 'organizing', 'organized', 'organized', tran, ['6A'] ).
verb( 'orient', 'orients', 'orienting', 'oriented', 'oriented', tran, [] ).
verb( 'orientate', 'orientates', 'orientating', 'orientated', 'orientated', tran, ['6A'] ).
verb( 'originate', 'originates', 'originating', 'originated', 'originated', _, ['2C','3A','6A'] ).
verb( 'ornament', 'ornaments', 'ornamenting', 'ornamented', 'ornamented', tran, ['6A','14'] ).
verb( 'orphan', 'orphans', 'orphaning', 'orphaned', 'orphaned', tran, ['6A'] ).
verb( 'oscillate', 'oscillates', 'oscillating', 'oscillated', 'oscillated', _, ['2A','6A'] ).
verb( 'ossify', 'ossifies', 'ossifying', 'ossified', 'ossified', _, ['2A','6A'] ).
verb( 'ostracize', 'ostracizes', 'ostracizing', 'ostracized', 'ostracized', tran, ['6A'] ).
verb( 'ought', 'ought', '-', '-', '-', unknown, ['7B'] ).
verb( 'oust', 'ousts', 'ousting', 'ousted', 'ousted', tran, ['6A','14'] ).
verb( 'out', 'outs', 'outing', 'outed', 'outed', tran, [] ).
verb( 'out-herod', 'out-herods', 'out-heroding', 'out-heroded', 'out-heroded', tran, ['6A'] ).
verb( 'outbalance', 'outbalances', 'outbalancing', 'outbalanced', 'outbalanced', tran, ['6A'] ).
verb( 'outbid', 'outbids', 'outbidding', 'outbid', 'outbid', tran, ['6A'] ).
verb( 'outbrave', 'outbraves', 'outbraving', 'outbraved', 'outbraved', tran, ['6A'] ).
verb( 'outclass', 'outclasses', 'outclassing', 'outclassed', 'outclassed', tran, ['6A'] ).
verb( 'outdistance', 'outdistances', 'outdistancing', 'outdistanced', 'outdistanced', tran, ['6A'] ).
verb( 'outdo', 'outdoes', 'outdoing', 'outdid', 'outdone', tran, ['6A'] ).
verb( 'outface', 'outfaces', 'outfacing', 'outfaced', 'outfaced', tran, ['6A'] ).
verb( 'outfight', 'outfights', 'outfighting', 'outfought', 'outfought', tran, ['6A'] ).
verb( 'outfit', 'outfits', 'outfitting', 'outfitted', 'outfitted', tran, [] ).
verb( 'outflank', 'outflanks', 'outflanking', 'outflanked', 'outflanked', tran, ['6A'] ).
verb( 'outfox', 'outfoxes', 'outfoxing', 'outfoxed', 'outfoxed', tran, ['6A'] ).
verb( 'outgo', 'outgoes', 'outgoing', 'outwent', 'outgone', intran, [] ).
verb( 'outgrow', 'outgrows', 'outgrowing', 'outgrew', 'outgrown', tran, ['6A'] ).
verb( 'outlast', 'outlasts', 'outlasting', 'outlasted', 'outlasted', tran, ['6A'] ).
verb( 'outlaw', 'outlaws', 'outlawing', 'outlawed', 'outlawed', tran, ['6A'] ).
verb( 'outline', 'outlines', 'outlining', 'outlined', 'outlined', tran, ['6A'] ).
verb( 'outlive', 'outlives', 'outliving', 'outlived', 'outlived', tran, ['6A'] ).
verb( 'outmanoeuvre', 'outmanoeuvres', 'outmanoeuvring', 'outmanoeuvred', 'outmanoeuvred', tran, ['6A'] ).
verb( 'outmarch', 'outmarches', 'outmarching', 'outmarched', 'outmarched', tran, ['6A'] ).
verb( 'outmatch', 'outmatches', 'outmatching', 'outmatched', 'outmatched', tran, ['6A'] ).
verb( 'outnumber', 'outnumbers', 'outnumbering', 'outnumbered', 'outnumbered', tran, ['6A'] ).
verb( 'outplay', 'outplays', 'outplaying', 'outplayed', 'outplayed', tran, ['6A'] ).
verb( 'outpoint', 'outpoints', 'outpointing', 'outpointed', 'outpointed', tran, ['6A'] ).
verb( 'outrage', 'outrages', 'outraging', 'outraged', 'outraged', tran, ['6A'] ).
verb( 'outrange', 'outranges', 'outranging', 'outranged', 'outranged', tran, ['6A'] ).
verb( 'outrank', 'outranks', 'outranking', 'outranked', 'outranked', tran, ['6A'] ).
verb( 'outride', 'outrides', 'outriding', 'outrode', 'outridden', tran, ['6A'] ).
verb( 'outrival', 'outrivals', 'outrivalling', 'outrivalled', 'outrivalled', tran, ['6A'] ).
verb( 'outrun', 'outruns', 'outrunning', 'outran', 'outrun', tran, ['6A'] ).
verb( 'outsail', 'outsails', 'outsailing', 'outsailed', 'outsailed', tran, ['6A'] ).
verb( 'outshine', 'outshines', 'outshining', 'outshone', 'outshone', tran, ['6A'] ).
verb( 'outsmart', 'outsmarts', 'outsmarting', 'outsmarted', 'outsmarted', tran, ['6A'] ).
verb( 'outspan', 'outspans', 'outspanning', 'outspanned', 'outspanned', _, ['2A','6A'] ).
verb( 'outstay', 'outstays', 'outstaying', 'outstayed', 'outstayed', tran, ['6A'] ).
verb( 'outstrip', 'outstrips', 'outstripping', 'outstripped', 'outstripped', tran, ['6A'] ).
verb( 'outvie', 'outvies', 'outvying', 'outvied', 'outvied', tran, ['6A'] ).
verb( 'outvote', 'outvotes', 'outvoting', 'outvoted', 'outvoted', tran, ['6A'] ).
verb( 'outwear', 'outwears', 'outwearing', 'outwore', 'outworn', tran, ['6A'] ).
verb( 'outweigh', 'outweighs', 'outweighing', 'outweighed', 'outweighed', tran, ['6A'] ).
verb( 'outwit', 'outwits', 'outwitting', 'outwitted', 'outwitted', tran, ['6A'] ).
verb( 'overact', 'overacts', 'overacting', 'overacted', 'overacted', _, ['2A','6A'] ).
verb( 'overarch', 'overarches', 'overarching', 'overarched', 'overarched', _, ['2A','6A'] ).
verb( 'overawe', 'overawes', 'overawing', 'overawed', 'overawed', tran, ['6A'] ).
verb( 'overbalance', 'overbalances', 'overbalancing', 'overbalanced', 'overbalanced', _, ['2A','6A'] ).
verb( 'overbear', 'overbears', 'overbearing', 'overbore', 'overborne', tran, ['6A'] ).
verb( 'overbid', 'overbids', 'overbidding', 'overbid', 'overbid', _, ['2A','6A'] ).
verb( 'overburden', 'overburdens', 'overburdening', 'overburdened', 'overburdened', tran, ['6A'] ).
verb( 'overcall', 'overcalls', 'overcalling', 'overcalled', 'overcalled', _, [] ).
verb( 'overcapitalize', 'overcapitalizes', 'overcapitalizing', 'overcapitalized', 'overcapitalized', tran, [] ).
verb( 'overcharge', 'overcharges', 'overcharging', 'overcharged', 'overcharged', _, ['2A','6A'] ).
verb( 'overclothe', 'overclothes', 'overclothing', 'overclothed', 'overclothed', tran, ['6A'] ).
verb( 'overcloud', 'overclouds', 'overclouding', 'overclouded', 'overclouded', _, ['2A','6A'] ).
verb( 'overcome', 'overcomes', 'overcoming', 'overcame', 'overcome', tran, ['6A'] ).
verb( 'overcook', 'overcooks', 'overcooking', 'overcooked', 'overcooked', tran, [] ).
verb( 'overcrop', 'overcrops', 'overcropping', 'overcropped', 'overcropped', tran, ['6A'] ).
verb( 'overcrowd', 'overcrowds', 'overcrowding', 'overcrowded', 'overcrowded', tran, ['6A'] ).
verb( 'overdo', 'overdoes', 'overdoing', 'overdid', 'overdone', tran, ['6A'] ).
verb( 'overdraw', 'overdraws', 'overdrawing', 'overdrew', 'overdrawn', _, ['2A','6A'] ).
verb( 'overdress', 'overdresses', 'overdressing', 'overdressed', 'overdressed', _, ['2A','6A'] ).
verb( 'overeat', 'overeats', 'overeating', 'overate', 'overeaten', intran, [] ).
verb( 'overemphasize', 'overemphasizes', 'overemphasizing', 'overemphasized', 'overemphasized', tran, [] ).
verb( 'overestimate', 'overestimates', 'overestimating', 'overestimated', 'overestimated', tran, [] ).
verb( 'overexert', 'overexerts', 'overexerting', 'overexerted', 'overexerted', tran, [] ).
verb( 'overexpose', 'overexposes', 'overexposing', 'overexposed', 'overexposed', tran, [] ).
verb( 'overfeed', 'overfeeds', 'overfeeding', 'overfeeded', 'overfeeded', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'overflow', 'overflows', 'overflowing', 'overflowed', 'overflowed', _, ['2A','3A','6A'] ).
verb( 'overfly', 'overflies', 'overflying', 'overflew', 'overflown', tran, ['2A','2B','2C','2D','4A','6A','15A','15B'] ).
verb( 'overhang', 'overhangs', 'overhanging', 'overhung', 'overhung', _, ['2A','6A'] ).
verb( 'overhaul', 'overhauls', 'overhauling', 'overhauled', 'overhauled', tran, ['6A'] ).
verb( 'overhear', 'overhears', 'overhearing', 'overheard', 'overheard', tran, ['6A','18A','19A'] ).
verb( 'overheat', 'overheats', 'overheating', 'overheated', 'overheated', tran, [] ).
verb( 'overindulge', 'overindulges', 'overindulging', 'overindulged', 'overindulged', _, [] ).
verb( 'overlap', 'overlaps', 'overlapping', 'overlapped', 'overlapped', _, ['2A','6A'] ).
verb( 'overlay', 'overlays', 'overlaying', 'overlaid', 'overlain', tran, [] ).
verb( 'overleap', 'overleaps', 'overleaping', 'overleaped', 'overleaped', tran, ['6A'] ).
verb( 'overlie', 'overlies', 'overlying', 'overlaid', 'overlaid', intran, ['2A','2C','2D','3A'] ).
verb( 'overload', 'overloads', 'overloading', 'overloaded', 'overloaded', tran, ['6A'] ).
verb( 'overlook', 'overlooks', 'overlooking', 'overlooked', 'overlooked', tran, ['6A'] ).
verb( 'overmaster', 'overmasters', 'overmastering', 'overmastered', 'overmastered', tran, ['6A'] ).
verb( 'overpay', 'overpays', 'overpaying', 'overpaid', 'overpaid', tran, ['6A','14'] ).
verb( 'overplay', 'overplays', 'overplaying', 'overplayed', 'overplayed', tran, [] ).
verb( 'overpower', 'overpowers', 'overpowering', 'overpowered', 'overpowered', tran, ['6A'] ).
verb( 'overpraise', 'overpraises', 'overpraising', 'overpraised', 'overpraised', tran, [] ).
verb( 'overprint', 'overprints', 'overprinting', 'overprinted', 'overprinted', tran, ['6A'] ).
verb( 'overproduce', 'overproduces', 'overproducing', 'overproduced', 'overproduced', _, [] ).
verb( 'overrate', 'overrates', 'overrating', 'overrated', 'overrated', tran, ['6A'] ).
verb( 'overreach', 'overreaches', 'overreaching', 'overreached', 'overreached', tran, ['6A'] ).
verb( 'override', 'overrides', 'overriding', 'overrode', 'overridden', tran, ['6A'] ).
verb( 'overrule', 'overrules', 'overruling', 'overruled', 'overruled', tran, ['6A'] ).
verb( 'overrun', 'overruns', 'overrunning', 'overran', 'overrun', tran, ['6A'] ).
verb( 'oversee', 'oversees', 'overseeing', 'oversaw', 'overseen', tran, ['6A'] ).
verb( 'oversew', 'oversews', 'oversewing', 'oversewed', 'oversewn', tran, ['2A','2B','2C','6A','15B'] ).
verb( 'overshadow', 'overshadows', 'overshadowing', 'overshadowed', 'overshadowed', tran, ['6A'] ).
verb( 'overshoot', 'overshoots', 'overshooting', 'overshot', 'overshot', tran, ['6A'] ).
verb( 'oversimplify', 'oversimplifies', 'oversimplifying', 'oversimplified', 'oversimplified', tran, [] ).
verb( 'oversleep', 'oversleeps', 'oversleeping', 'overslept', 'overslept', intran, ['2A'] ).
verb( 'overspend', 'overspends', 'overspending', 'overspent', 'overspent', _, ['2A','6A','14','19B'] ).
verb( 'overstate', 'overstates', 'overstating', 'overstated', 'overstated', tran, ['6A'] ).
verb( 'overstay', 'overstays', 'overstaying', 'overstayed', 'overstayed', tran, ['6A'] ).
verb( 'overstep', 'oversteps', 'overstepping', 'overstepped', 'overstepped', tran, ['6A'] ).
verb( 'overstock', 'overstocks', 'overstocking', 'overstocked', 'overstocked', tran, ['6A'] ).
verb( 'overstrain', 'overstrains', 'overstraining', 'overstrained', 'overstrained', tran, [] ).
verb( 'overtake', 'overtakes', 'overtaking', 'overtook', 'overtaken', tran, ['6A'] ).
verb( 'overtax', 'overtaxes', 'overtaxing', 'overtaxed', 'overtaxed', tran, ['6A'] ).
verb( 'overthrow', 'overthrows', 'overthrowing', 'overthrew', 'overthrown', tran, ['6A'] ).
verb( 'overtop', 'overtops', 'overtopping', 'overtopped', 'overtopped', tran, ['6A'] ).
verb( 'overtrump', 'overtrumps', 'overtrumping', 'overtrumped', 'overtrumped', tran, ['6A'] ).
verb( 'overturn', 'overturns', 'overturning', 'overturned', 'overturned', _, ['2A','6A'] ).
verb( 'overvalue', 'overvalues', 'overvaluing', 'overvalued', 'overvalued', tran, [] ).
verb( 'overwhelm', 'overwhelms', 'overwhelming', 'overwhelmed', 'overwhelmed', tran, ['6A'] ).
verb( 'overwork', 'overworks', 'overworking', 'overworked', 'overworked', _, ['2A','6A'] ).
verb( 'owe', 'owes', 'owing', 'owed', 'owed', _, ['2A','3A','6A','12A','13A','14'] ).
verb( 'own', 'owns', 'owning', 'owned', 'owned', _, ['2C','3B','6A','9'] ).
verb( 'oxidize', 'oxidizes', 'oxidizing', 'oxidized', 'oxidized', _, ['2A','6A'] ).
verb( 'oxygenate', 'oxygenates', 'oxygenating', 'oxygenated', 'oxygenated', tran, [] ).
verb( 'oxygenize', 'oxygenizes', 'oxygenizing', 'oxygenized', 'oxygenized', tran, [] ).
verb( 'pace', 'paces', 'pacing', 'paced', 'paced', _, ['2A','2C','6A','15B'] ).
verb( 'pacify', 'pacifies', 'pacifying', 'pacified', 'pacified', tran, ['6A'] ).
verb( 'pack', 'packs', 'packing', 'packed', 'packed', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'package', 'packages', 'packaging', 'packaged', 'packaged', tran, [] ).
verb( 'pad', 'pads', 'padding', 'padded', 'padded', _, ['2A','2C','6A','15B'] ).
verb( 'paddle', 'paddles', 'paddling', 'paddled', 'paddled', _, ['2A','6A'] ).
verb( 'padlock', 'padlocks', 'padlocking', 'padlocked', 'padlocked', tran, ['6A'] ).
verb( 'page', 'pages', 'paging', 'paged', 'paged', tran, ['6A'] ).
verb( 'pain', 'pains', 'paining', 'pained', 'pained', tran, ['6A'] ).
verb( 'paint', 'paints', 'painting', 'painted', 'painted', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'pair', 'pairs', 'pairing', 'paired', 'paired', _, ['2A','2C','6A','15B'] ).
verb( 'pal', 'pals', 'palling', 'palled', 'palled', intran, [] ).
verb( 'palaver', 'palavers', 'palavering', 'palavered', 'palavered', intran, ['2A'] ).
verb( 'pale', 'pales', 'paling', 'paled', 'paled', intran, ['2A','2C'] ).
verb( 'palisade', 'palisades', 'palisading', 'palisaded', 'palisaded', tran, ['6A'] ).
verb( 'pall', 'palls', 'palling', 'palled', 'palled', intran, ['2A','3A'] ).
verb( 'palliate', 'palliates', 'palliating', 'palliated', 'palliated', tran, ['6A'] ).
verb( 'palm', 'palms', 'palming', 'palmed', 'palmed', tran, ['6A','15B'] ).
verb( 'palpitate', 'palpitates', 'palpitating', 'palpitated', 'palpitated', intran, ['2A','2C'] ).
verb( 'palsy', 'palsies', 'palsying', 'palsied', 'palsied', tran, [] ).
verb( 'palter', 'palters', 'paltering', 'paltered', 'paltered', intran, ['3A'] ).
verb( 'pamper', 'pampers', 'pampering', 'pampered', 'pampered', tran, ['6A'] ).
verb( 'pan', 'pans', 'panning', 'panned', 'panned', _, ['2C','3A','6A','15B'] ).
verb( 'pander', 'panders', 'pandering', 'pandered', 'pandered', intran, ['3A'] ).
verb( 'panel', 'panels', 'panelling', 'panelled', 'panelled', tran, [] ).
verb( 'panhandle', 'panhandles', 'panhandling', 'panhandled', 'panhandled', intran, [] ).
verb( 'panic', 'panics', 'panicking', 'panicked', 'panicked', intran, [] ).
verb( 'pant', 'pants', 'panting', 'panted', 'panted', _, ['2A','2C','3A','6A','15B'] ).
verb( 'paper', 'papers', 'papering', 'papered', 'papered', tran, ['6A','15B'] ).
verb( 'parachute', 'parachutes', 'parachuting', 'parachuted', 'parachuted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'parade', 'parades', 'parading', 'paraded', 'paraded', _, ['2A','6A'] ).
verb( 'paragraph', 'paragraphs', 'paragraphing', 'paragraphed', 'paragraphed', tran, [] ).
verb( 'parallel', 'parallels', 'parallelling', 'parallelled', 'parallelled', tran, ['6A'] ).
verb( 'paralyse', 'paralyses', 'paralysing', 'paralysed', 'paralysed', tran, ['6A'] ).
verb( 'paralyze', 'paralyzes', 'paralyzing', 'paralyzed', 'paralyzed', tran, ['6A'] ).
verb( 'paraphrase', 'paraphrases', 'paraphrasing', 'paraphrased', 'paraphrased', tran, ['6A'] ).
verb( 'parboil', 'parboils', 'parboiling', 'parboiled', 'parboiled', tran, ['6A'] ).
verb( 'parcel', 'parcels', 'parcelling', 'parcelled', 'parcelled', tran, ['6A','15B'] ).
verb( 'parch', 'parches', 'parching', 'parched', 'parched', tran, ['6A'] ).
verb( 'pardon', 'pardons', 'pardoning', 'pardoned', 'pardoned', tran, ['6A','12B','13B'] ).
verb( 'pare', 'pares', 'paring', 'pared', 'pared', tran, ['6A','15B'] ).
verb( 'park', 'parks', 'parking', 'parked', 'parked', _, ['2A','6A','15A'] ).
verb( 'parley', 'parleys', 'parleying', 'parleyed', 'parleyed', intran, ['2A','3A'] ).
verb( 'parody', 'parodies', 'parodying', 'parodied', 'parodied', tran, ['6A'] ).
verb( 'parole', 'paroles', 'paroling', 'paroled', 'paroled', tran, ['6A'] ).
verb( 'parry', 'parries', 'parrying', 'parried', 'parried', tran, ['6A'] ).
verb( 'parse', 'parses', 'parsing', 'parsed', 'parsed', tran, ['6A'] ).
verb( 'part', 'parts', 'parting', 'parted', 'parted', _, ['2A','2D','3A','6A'] ).
verb( 'partake', 'partakes', 'partaking', 'partook', 'partaken', _, ['3A'] ).
verb( 'participate', 'participates', 'participating', 'participated', 'participated', intran, ['2A','3A'] ).
verb( 'particularize', 'particularizes', 'particularizing', 'particularized', 'particularized', _, ['2A','6A'] ).
verb( 'partition', 'partitions', 'partitioning', 'partitioned', 'partitioned', tran, ['6A','15B'] ).
verb( 'partner', 'partners', 'partnering', 'partnered', 'partnered', tran, ['6A','15A'] ).
verb( 'pass', 'passes', 'passing', 'passed', 'passed', _, ['2A','2C','3A','6A','12A','13A','14','15A','15B'] ).
verb( 'paste', 'pastes', 'pasting', 'pasted', 'pasted', tran, ['6A','15A','15B'] ).
verb( 'pasteurize', 'pasteurizes', 'pasteurizing', 'pasteurized', 'pasteurized', tran, ['6A'] ).
verb( 'pasture', 'pastures', 'pasturing', 'pastured', 'pastured', _, ['2A','6A'] ).
verb( 'pat', 'pats', 'patting', 'patted', 'patted', _, ['2A','6A','15A'] ).
verb( 'patch', 'patches', 'patching', 'patched', 'patched', tran, ['6A','15B'] ).
verb( 'patent', 'patents', 'patenting', 'patented', 'patented', tran, ['6A'] ).
verb( 'patrol', 'patrols', 'patrolling', 'patrolled', 'patrolled', _, ['2A','6A'] ).
verb( 'patronize', 'patronizes', 'patronizing', 'patronized', 'patronized', tran, ['6A'] ).
verb( 'patter', 'patters', 'pattering', 'pattered', 'pattered', _, ['2A','2C','6A'] ).
verb( 'pattern', 'patterns', 'patterning', 'patterned', 'patterned', tran, ['14'] ).
verb( 'pauperize', 'pauperizes', 'pauperizing', 'pauperized', 'pauperized', tran, ['6A'] ).
verb( 'pause', 'pauses', 'pausing', 'paused', 'paused', intran, ['2A','4A'] ).
verb( 'pave', 'paves', 'paving', 'paved', 'paved', tran, ['6A'] ).
verb( 'paw', 'paws', 'pawing', 'pawed', 'pawed', tran, ['6A','15B'] ).
verb( 'pawn', 'pawns', 'pawning', 'pawned', 'pawned', tran, ['6A'] ).
verb( 'pay', 'pays', 'paying', 'paid', 'paid', _, ['2A','3A','6A','12A','12B','13A','13B','14','15B'] ).
verb( 'peach', 'peaches', 'peaching', 'peached', 'peached', _, ['2A','3A','6A'] ).
verb( 'peak', 'peaks', 'peaking', 'peaked', 'peaked', intran, ['2A'] ).
verb( 'peal', 'peals', 'pealing', 'pealed', 'pealed', _, ['2A','2C','6A'] ).
verb( 'pearl', 'pearls', 'pearling', 'pearled', 'pearled', intran, [] ).
verb( 'peck', 'pecks', 'pecking', 'pecked', 'pecked', _, ['2A','2C','3A','6A'] ).
verb( 'peculate', 'peculates', 'peculating', 'peculated', 'peculated', _, [] ).
verb( 'pedal', 'pedals', 'pedalling', 'pedalled', 'pedalled', _, ['2A','2C','6A'] ).
verb( 'peddle', 'peddles', 'peddling', 'peddled', 'peddled', _, ['2A','6A'] ).
verb( 'pee', 'pees', 'peeing', 'peed', 'peed', intran, ['2A'] ).
verb( 'peek', 'peeks', 'peeking', 'peeked', 'peeked', intran, [] ).
verb( 'peel', 'peels', 'peeling', 'peeled', 'peeled', _, ['2A','2C','6A','15B'] ).
verb( 'peep', 'peeps', 'peeping', 'peeped', 'peeped', intran, ['2A','2C'] ).
verb( 'peer', 'peers', 'peering', 'peered', 'peered', intran, ['2A','3A'] ).
verb( 'peeve', 'peeves', 'peeving', 'peeved', 'peeved', tran, [] ).
verb( 'peg', 'pegs', 'pegging', 'pegged', 'pegged', _, ['2C','6A','15B'] ).
verb( 'pelt', 'pelts', 'pelting', 'pelted', 'pelted', _, ['2C','6A','14'] ).
verb( 'pen', 'pens', 'penning', 'penned', 'penned', tran, ['15A','15B'] ).
verb( 'penalize', 'penalizes', 'penalizing', 'penalized', 'penalized', tran, ['6A','14'] ).
verb( 'pencil', 'pencils', 'pencilling', 'pencilled', 'pencilled', tran, ['6A'] ).
verb( 'penetrate', 'penetrates', 'penetrating', 'penetrated', 'penetrated', _, ['3A','6A'] ).
verb( 'pension', 'pensions', 'pensioning', 'pensioned', 'pensioned', tran, ['15B'] ).
verb( 'people', 'peoples', 'peopling', 'peopled', 'peopled', tran, ['6A'] ).
verb( 'pep', 'peps', 'pepping', 'pepped', 'pepped', tran, ['15B'] ).
verb( 'pepper', 'peppers', 'peppering', 'peppered', 'peppered', tran, ['6A'] ).
verb( 'perambulate', 'perambulates', 'perambulating', 'perambulated', 'perambulated', _, ['2A','6A'] ).
verb( 'perceive', 'perceives', 'perceiving', 'perceived', 'perceived', tran, ['6A','8','9','10','18A','19A','25'] ).
verb( 'perch', 'perches', 'perching', 'perched', 'perched', _, ['2C'] ).
verb( 'percolate', 'percolates', 'percolating', 'percolated', 'percolated', _, ['2A','3A','6A'] ).
verb( 'perfect', 'perfects', 'perfecting', 'perfected', 'perfected', tran, ['6A'] ).
verb( 'perforate', 'perforates', 'perforating', 'perforated', 'perforated', tran, ['6A'] ).
verb( 'perform', 'performs', 'performing', 'performed', 'performed', _, ['2A','6A'] ).
verb( 'perfume', 'perfumes', 'perfuming', 'perfumed', 'perfumed', tran, ['6A'] ).
verb( 'peril', 'perils', 'perilling', 'perilled', 'perilled', tran, [] ).
verb( 'perish', 'perishes', 'perishing', 'perished', 'perished', _, ['2A','2C','6A'] ).
verb( 'perjure', 'perjures', 'perjuring', 'perjured', 'perjured', tran, ['6A'] ).
verb( 'perk', 'perks', 'perking', 'perked', 'perked', _, ['2C','15B'] ).
verb( 'perm', 'perms', 'perming', 'permed', 'permed', tran, [] ).
verb( 'permeate', 'permeates', 'permeating', 'permeated', 'permeated', _, ['3A','6A'] ).
verb( 'permit', 'permits', 'permitting', 'permitted', 'permitted', _, ['3A','6A','6C','17','19C'] ).
verb( 'permute', 'permutes', 'permuting', 'permuted', 'permuted', tran, ['6A'] ).
verb( 'perpetrate', 'perpetrates', 'perpetrating', 'perpetrated', 'perpetrated', tran, ['6A'] ).
verb( 'perpetuate', 'perpetuates', 'perpetuating', 'perpetuated', 'perpetuated', tran, ['6A'] ).
verb( 'perplex', 'perplexes', 'perplexing', 'perplexed', 'perplexed', tran, ['6A','14'] ).
verb( 'persecute', 'persecutes', 'persecuting', 'persecuted', 'persecuted', tran, ['6A'] ).
verb( 'persevere', 'perseveres', 'persevering', 'persevered', 'persevered', intran, ['2A','3A'] ).
verb( 'persist', 'persists', 'persisting', 'persisted', 'persisted', intran, ['2A','3A'] ).
verb( 'personalize', 'personalizes', 'personalizing', 'personalized', 'personalized', tran, ['6A'] ).
verb( 'personate', 'personates', 'personating', 'personated', 'personated', tran, ['6A'] ).
verb( 'personify', 'personifies', 'personifying', 'personified', 'personified', tran, ['6A'] ).
verb( 'perspire', 'perspires', 'perspiring', 'perspired', 'perspired', intran, ['2A'] ).
verb( 'persuade', 'persuades', 'persuading', 'persuaded', 'persuaded', tran, ['11','14','17'] ).
verb( 'pertain', 'pertains', 'pertaining', 'pertained', 'pertained', intran, ['3A'] ).
verb( 'perturb', 'perturbs', 'perturbing', 'perturbed', 'perturbed', tran, ['6A'] ).
verb( 'peruse', 'peruses', 'perusing', 'perused', 'perused', tran, ['6A'] ).
verb( 'pervade', 'pervades', 'pervading', 'pervaded', 'pervaded', tran, ['6A'] ).
verb( 'pervert', 'perverts', 'perverting', 'perverted', 'perverted', tran, ['6A'] ).
verb( 'pester', 'pesters', 'pestering', 'pestered', 'pestered', tran, ['6A','14','17'] ).
verb( 'pestle', 'pestles', 'pestling', 'pestled', 'pestled', tran, [] ).
verb( 'pet', 'pets', 'petting', 'petted', 'petted', tran, [] ).
verb( 'peter', 'peters', 'petering', 'petered', 'petered', intran, ['2C'] ).
verb( 'petition', 'petitions', 'petitioning', 'petitioned', 'petitioned', _, ['3A','6A','11','14','17'] ).
verb( 'petrify', 'petrifies', 'petrifying', 'petrified', 'petrified', _, ['2A','6A'] ).
verb( 'phase', 'phases', 'phasing', 'phased', 'phased', tran, ['6A','15B'] ).
verb( 'philander', 'philanders', 'philandering', 'philandered', 'philandered', intran, ['2A'] ).
verb( 'philosophize', 'philosophizes', 'philosophizing', 'philosophized', 'philosophized', intran, ['2A'] ).
verb( 'phone', 'phones', 'phoning', 'phoned', 'phoned', _, ['2A','4A','6A','11','12A','13A'] ).
verb( 'photocopy', 'photocopies', 'photocopying', 'photocopied', 'photocopied', tran, ['6A'] ).
verb( 'photograph', 'photographs', 'photographing', 'photographed', 'photographed', tran, ['6A'] ).
verb( 'photosensitize', 'photosensitizes', 'photosensitizing', 'photosensitized', 'photosensitized', tran, [] ).
verb( 'photostat', 'photostats', 'photostatting', 'photostatted', 'photostatted', tran, [] ).
verb( 'phrase', 'phrases', 'phrasing', 'phrased', 'phrased', tran, ['6A'] ).
verb( 'pick', 'picks', 'picking', 'picked', 'picked', _, ['3A','6A','15B'] ).
verb( 'picket', 'pickets', 'picketing', 'picketed', 'picketed', _, ['2A','6A'] ).
verb( 'pickle', 'pickles', 'pickling', 'pickled', 'pickled', tran, ['6A'] ).
verb( 'picnic', 'picnics', 'picnicking', 'picnicked', 'picnicked', intran, [] ).
verb( 'picture', 'pictures', 'picturing', 'pictured', 'pictured', tran, ['6A','14'] ).
verb( 'piddle', 'piddles', 'piddling', 'piddled', 'piddled', intran, [] ).
verb( 'piece', 'pieces', 'piecing', 'pieced', 'pieced', tran, ['6A','15A','15B'] ).
verb( 'pierce', 'pierces', 'piercing', 'pierced', 'pierced', _, ['2C','6A'] ).
verb( 'piffle', 'piffles', 'piffling', 'piffled', 'piffled', intran, [] ).
verb( 'pig', 'pigs', 'pigging', 'pigged', 'pigged', intran, [] ).
verb( 'pigeonhole', 'pigeonholes', 'pigeonholing', 'pigeonholed', 'pigeonholed', tran, [] ).
verb( 'pile', 'piles', 'piling', 'piled', 'piled', _, ['2C','6A','15A','15B'] ).
verb( 'pilfer', 'pilfers', 'pilfering', 'pilfered', 'pilfered', _, ['2A','6A'] ).
verb( 'pillage', 'pillages', 'pillaging', 'pillaged', 'pillaged', tran, [] ).
verb( 'pillow', 'pillows', 'pillowing', 'pillowed', 'pillowed', tran, ['6A'] ).
verb( 'pilot', 'pilots', 'piloting', 'piloted', 'piloted', tran, ['6A','15A'] ).
verb( 'pimp', 'pimps', 'pimping', 'pimped', 'pimped', intran, ['2A','3A'] ).
verb( 'pin', 'pins', 'pinning', 'pinned', 'pinned', tran, ['15A','15B'] ).
verb( 'pinch', 'pinches', 'pinching', 'pinched', 'pinched', _, ['2A','6A','15A','15B'] ).
verb( 'pine', 'pines', 'pining', 'pined', 'pined', intran, ['2A','2C','3A','4C'] ).
verb( 'ping', 'pings', 'pinging', 'pinged', 'pinged', intran, [] ).
verb( 'pinion', 'pinions', 'pinioning', 'pinioned', 'pinioned', tran, ['6A','15A','15B'] ).
verb( 'pink', 'pinks', 'pinking', 'pinked', 'pinked', _, ['6A','15B'] ).
verb( 'pinnacle', 'pinnacles', 'pinnacling', 'pinnacled', 'pinnacled', tran, [] ).
verb( 'pinpoint', 'pinpoints', 'pinpointing', 'pinpointed', 'pinpointed', tran, [] ).
verb( 'pioneer', 'pioneers', 'pioneering', 'pioneered', 'pioneered', _, ['2A','6A'] ).
verb( 'pip', 'pips', 'pipping', 'pipped', 'pipped', tran, ['6A'] ).
verb( 'pipe', 'pipes', 'piping', 'piped', 'piped', _, ['2A','2C','6A','15A'] ).
verb( 'pique', 'piques', 'piquing', 'piqued', 'piqued', tran, ['6A'] ).
verb( 'pirate', 'pirates', 'pirating', 'pirated', 'pirated', tran, ['6A'] ).
verb( 'pirouette', 'pirouettes', 'pirouetting', 'pirouetted', 'pirouetted', intran, [] ).
verb( 'piss', 'pisses', 'pissing', 'pissed', 'pissed', _, [] ).
verb( 'pit', 'pits', 'pitting', 'pitted', 'pitted', tran, ['6A','14'] ).
verb( 'pitch', 'pitches', 'pitching', 'pitched', 'pitched', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'pitchfork', 'pitchforks', 'pitchforking', 'pitchforked', 'pitchforked', tran, [] ).
verb( 'pity', 'pities', 'pitying', 'pitied', 'pitied', tran, ['6A'] ).
verb( 'pivot', 'pivots', 'pivoting', 'pivoted', 'pivoted', _, ['3A','6A'] ).
verb( 'placard', 'placards', 'placarding', 'placarded', 'placarded', tran, ['6A'] ).
verb( 'placate', 'placates', 'placating', 'placated', 'placated', tran, ['6A'] ).
verb( 'place', 'places', 'placing', 'placed', 'placed', tran, ['6A','15A','15B'] ).
verb( 'plagiarize', 'plagiarizes', 'plagiarizing', 'plagiarized', 'plagiarized', tran, ['6A'] ).
verb( 'plague', 'plagues', 'plaguing', 'plagued', 'plagued', tran, ['6A','14'] ).
verb( 'plain', 'plains', 'plaining', 'plained', 'plained', _, [] ).
verb( 'plait', 'plaits', 'plaiting', 'plaited', 'plaited', tran, ['6A'] ).
verb( 'plan', 'plans', 'planning', 'planned', 'planned', tran, ['6A','7A','15B'] ).
verb( 'plane', 'planes', 'planing', 'planed', 'planed', _, ['2A','2D','15B','22'] ).
verb( 'plank', 'planks', 'planking', 'planked', 'planked', tran, ['6A','15B'] ).
verb( 'plant', 'plants', 'planting', 'planted', 'planted', tran, ['6A','15A','15B'] ).
verb( 'plash', 'plashes', 'plashing', 'plashed', 'plashed', _, ['6A'] ).
verb( 'plaster', 'plasters', 'plastering', 'plastered', 'plastered', tran, ['6A','14'] ).
verb( 'plate', 'plates', 'plating', 'plated', 'plated', tran, ['6A','14'] ).
verb( 'play', 'plays', 'playing', 'played', 'played', _, ['2A','2B','2C','3A','6A','12B','12C','13B','14','15A','15B','16B'] ).
verb( 'pleach', 'pleaches', 'pleaching', 'pleached', 'pleached', tran, ['6A'] ).
verb( 'plead', 'pleads', 'pleading', 'pleaded', 'pleaded', _, ['3A','6A'] ).
verb( 'please', 'pleases', 'pleasing', 'pleased', 'pleased', _, ['2A','6A'] ).
verb( 'pleat', 'pleats', 'pleating', 'pleated', 'pleated', tran, ['6A'] ).
verb( 'pledge', 'pledges', 'pledging', 'pledged', 'pledged', tran, ['6A'] ).
verb( 'plight', 'plights', 'plighting', 'plighted', 'plighted', tran, ['6A'] ).
verb( 'plod', 'plods', 'plodding', 'plodded', 'plodded', _, ['2C'] ).
verb( 'plonk', 'plonks', 'plonking', 'plonked', 'plonked', intran, [] ).
verb( 'plop', 'plops', 'plopping', 'plopped', 'plopped', intran, [] ).
verb( 'plot', 'plots', 'plotting', 'plotted', 'plotted', _, ['2A','3A','4A','6A','8','10','15B'] ).
verb( 'plough', 'ploughs', 'ploughing', 'ploughed', 'ploughed', _, ['3A','6A','15B'] ).
verb( 'pluck', 'plucks', 'plucking', 'plucked', 'plucked', _, ['3A','6A','15B'] ).
verb( 'plug', 'plugs', 'plugging', 'plugged', 'plugged', _, ['2C','6A','15B'] ).
verb( 'plumb', 'plumbs', 'plumbing', 'plumbed', 'plumbed', tran, ['6A'] ).
verb( 'plume', 'plumes', 'pluming', 'plumed', 'plumed', tran, ['6A'] ).
verb( 'plummet', 'plummets', 'plummetting', 'plummetted', 'plummetted', intran, ['2A'] ).
verb( 'plump', 'plumps', 'plumping', 'plumped', 'plumped', _, ['2A','2C','3A','6A','15B'] ).
verb( 'plunder', 'plunders', 'plundering', 'plundered', 'plundered', _, ['2A','6A','14'] ).
verb( 'plunge', 'plunges', 'plunging', 'plunged', 'plunged', _, ['2A','2C','6A','14'] ).
verb( 'ply', 'plies', 'plying', 'plied', 'plied', _, ['2C','6A'] ).
verb( 'poach', 'poaches', 'poaching', 'poached', 'poached', _, ['2A','3A','6A'] ).
verb( 'pocket', 'pockets', 'pocketing', 'pocketed', 'pocketed', tran, ['6A'] ).
verb( 'pod', 'pods', 'podding', 'podded', 'podded', _, ['2A','2C','6A'] ).
verb( 'point', 'points', 'pointing', 'pointed', 'pointed', _, ['2A','3A','6A','14','15B'] ).
verb( 'poise', 'poises', 'poising', 'poised', 'poised', _, ['2C','6A','15A'] ).
verb( 'poison', 'poisons', 'poisoning', 'poisoned', 'poisoned', tran, ['6A'] ).
verb( 'poke', 'pokes', 'poking', 'poked', 'poked', _, ['2C','6A','15A','15B'] ).
verb( 'polarize', 'polarizes', 'polarizing', 'polarized', 'polarized', tran, ['6A'] ).
verb( 'poleax', 'poleaxes', 'poleaxing', 'poleaxed', 'poleaxed', tran, ['6A'] ).
verb( 'poleaxe', 'poleaxes', 'poleaxing', 'poleaxed', 'poleaxed', tran, ['6A'] ).
verb( 'police', 'polices', 'policing', 'policed', 'policed', tran, ['6A'] ).
verb( 'polish', 'polishes', 'polishing', 'polished', 'polished', _, ['2A','6A','15B'] ).
verb( 'politicize', 'politicizes', 'politicizing', 'politicized', 'politicized', _, ['2A','6A'] ).
verb( 'politick', 'politicks', 'politicking', 'politicked', 'politicked', intran, ['2A'] ).
verb( 'poll', 'polls', 'polling', 'polled', 'polled', _, ['2A','2C','6A'] ).
verb( 'pollard', 'pollards', 'pollarding', 'pollarded', 'pollarded', tran, ['6A'] ).
verb( 'pollinate', 'pollinates', 'pollinating', 'pollinated', 'pollinated', tran, ['6A'] ).
verb( 'pollute', 'pollutes', 'polluting', 'polluted', 'polluted', tran, ['6A'] ).
verb( 'pomade', 'pomades', 'pomading', 'pomaded', 'pomaded', tran, [] ).
verb( 'pommel', 'pommels', 'pommelling', 'pommelled', 'pommelled', tran, [] ).
verb( 'ponder', 'ponders', 'pondering', 'pondered', 'pondered', _, ['2A','3A','6A','8','10'] ).
verb( 'poniard', 'poniards', 'poniarding', 'poniarded', 'poniarded', tran, ['6A'] ).
verb( 'pontificate', 'pontificates', 'pontificating', 'pontificated', 'pontificated', intran, ['2A'] ).
verb( 'pooh-pooh', 'pooh-poohs', 'pooh-poohing', 'pooh-poohed', 'pooh-poohed', tran, ['6A'] ).
verb( 'pool', 'pools', 'pooling', 'pooled', 'pooled', tran, ['6A'] ).
verb( 'pop', 'pops', 'popping', 'popped', 'popped', _, ['2A','2C','6A','15A','15B'] ).
verb( 'popularize', 'popularizes', 'popularizing', 'popularized', 'popularized', tran, ['6A'] ).
verb( 'populate', 'populates', 'populating', 'populated', 'populated', tran, ['6A'] ).
verb( 'pore', 'pores', 'poring', 'pored', 'pored', intran, ['3A'] ).
verb( 'port', 'ports', 'porting', 'ported', 'ported', tran, ['6A'] ).
verb( 'portend', 'portends', 'portending', 'portended', 'portended', tran, ['6A'] ).
verb( 'portion', 'portions', 'portioning', 'portioned', 'portioned', tran, ['14','15B'] ).
verb( 'portray', 'portrays', 'portraying', 'portrayed', 'portrayed', tran, ['6A'] ).
verb( 'pose', 'poses', 'posing', 'posed', 'posed', _, ['2A','2C','3A','6A'] ).
verb( 'posh', 'poshes', 'poshing', 'poshed', 'poshed', tran, ['15B'] ).
verb( 'posit', 'posits', 'positing', 'posited', 'posited', tran, ['6A'] ).
verb( 'position', 'positions', 'positioning', 'positioned', 'positioned', tran, ['6A'] ).
verb( 'possess', 'possesses', 'possessing', 'possessed', 'possessed', tran, ['6A'] ).
verb( 'post', 'posts', 'posting', 'posted', 'posted', _, ['3A','6A','15A','15B'] ).
verb( 'postdate', 'postdates', 'postdating', 'postdated', 'postdated', tran, ['6A'] ).
verb( 'postmark', 'postmarks', 'postmarking', 'postmarked', 'postmarked', tran, [] ).
verb( 'postpone', 'postpones', 'postponing', 'postponed', 'postponed', tran, ['6A','6C'] ).
verb( 'postulate', 'postulates', 'postulating', 'postulated', 'postulated', tran, ['6A'] ).
verb( 'posture', 'postures', 'posturing', 'postured', 'postured', _, ['2A','6A'] ).
verb( 'pot', 'pots', 'potting', 'potted', 'potted', _, ['3A','6A','15B'] ).
verb( 'potter', 'potters', 'pottering', 'pottered', 'pottered', intran, ['2A','2C'] ).
verb( 'pouch', 'pouches', 'pouching', 'pouched', 'pouched', tran, ['2A','6A'] ).
verb( 'poultice', 'poultices', 'poulticing', 'poulticed', 'poulticed', tran, ['6A'] ).
verb( 'pounce', 'pounces', 'pouncing', 'pounced', 'pounced', intran, ['3A'] ).
verb( 'pound', 'pounds', 'pounding', 'pounded', 'pounded', _, ['2C','3A','6A','15A'] ).
verb( 'pour', 'pours', 'pouring', 'poured', 'poured', _, ['2C','6A','12B','13B','14','15A','15B'] ).
verb( 'pout', 'pouts', 'pouting', 'pouted', 'pouted', _, ['2A','6A'] ).
verb( 'powder', 'powders', 'powdering', 'powdered', 'powdered', _, ['2A','6A'] ).
verb( 'power', 'powers', 'powering', 'powered', 'powered', tran, ['6A'] ).
verb( 'power-dive', 'power-dives', 'power-diving', 'power-dived', 'power-dived', tran, [] ).
verb( 'powwow', 'powwows', 'powwowing', 'powwowed', 'powwowed', intran, [] ).
verb( 'pr_ecis', 'pr_ecises', 'pr_ecising', 'pr_ecised', 'pr_ecised', tran, [] ).
verb( 'practise', 'practises', 'practising', 'practised', 'practised', _, ['2A','2B','3A','4A','6A','6C'] ).
verb( 'praise', 'praises', 'praising', 'praised', 'praised', tran, ['6A'] ).
verb( 'prance', 'prances', 'prancing', 'pranced', 'pranced', intran, ['2A','2C'] ).
verb( 'prate', 'prates', 'prating', 'prated', 'prated', intran, ['2A','2C'] ).
verb( 'prattle', 'prattles', 'prattling', 'prattled', 'prattled', intran, ['2A','2C'] ).
verb( 'prawn', 'prawns', 'prawning', 'prawned', 'prawned', intran, [] ).
verb( 'pray', 'prays', 'praying', 'prayed', 'prayed', _, ['2A','3A','11','14','17'] ).
verb( 'pre-empt', 'pre-empts', 'pre-empting', 'pre-empted', 'pre-empted', tran, ['6A'] ).
verb( 'pre-exist', 'pre-exists', 'pre-existing', 'pre-existed', 'pre-existed', intran, ['2A'] ).
verb( 'preach', 'preaches', 'preaching', 'preached', 'preached', _, ['2A','2B','2C','3A','6A','12A','13A'] ).
verb( 'preachify', 'preachifies', 'preachifying', 'preachified', 'preachified', intran, [] ).
verb( 'prearrange', 'prearranges', 'prearranging', 'prearranged', 'prearranged', tran, [] ).
verb( 'precede', 'precedes', 'preceding', 'preceded', 'preceded', _, ['2A','6A'] ).
verb( 'precipitate', 'precipitates', 'precipitating', 'precipitated', 'precipitated', tran, ['6A','14'] ).
verb( 'preclude', 'precludes', 'precluding', 'precluded', 'precluded', tran, ['6A','6C','14'] ).
verb( 'preconceive', 'preconceives', 'preconceiving', 'preconceived', 'preconceived', tran, ['6A'] ).
verb( 'predecease', 'predeceases', 'predeceasing', 'predeceased', 'predeceased', tran, [] ).
verb( 'predestinate', 'predestinates', 'predestinating', 'predestinated', 'predestinated', tran, [] ).
verb( 'predestine', 'predestines', 'predestining', 'predestined', 'predestined', tran, ['14','17'] ).
verb( 'predetermine', 'predetermines', 'predetermining', 'predetermined', 'predetermined', tran, ['6A','17'] ).
verb( 'predicate', 'predicates', 'predicating', 'predicated', 'predicated', tran, ['6A','9','17'] ).
verb( 'predict', 'predicts', 'predicting', 'predicted', 'predicted', tran, ['6A','9','10'] ).
verb( 'predigest', 'predigests', 'predigesting', 'predigested', 'predigested', tran, [] ).
verb( 'predispose', 'predisposes', 'predisposing', 'predisposed', 'predisposed', tran, ['14','17'] ).
verb( 'predominate', 'predominates', 'predominating', 'predominated', 'predominated', intran, ['2A','3A'] ).
verb( 'preen', 'preens', 'preening', 'preened', 'preened', tran, ['6A','14'] ).
verb( 'prefabricate', 'prefabricates', 'prefabricating', 'prefabricated', 'prefabricated', tran, ['6A'] ).
verb( 'preface', 'prefaces', 'prefacing', 'prefaced', 'prefaced', tran, ['14'] ).
verb( 'prefer', 'prefers', 'preferring', 'preferred', 'preferred', tran, ['6A','6D','7A','9','14','17'] ).
verb( 'prefigure', 'prefigures', 'prefiguring', 'prefigured', 'prefigured', tran, ['6A','9','10'] ).
verb( 'prefix', 'prefixes', 'prefixing', 'prefixed', 'prefixed', tran, ['6A','14'] ).
verb( 'preheat', 'preheats', 'preheating', 'preheated', 'preheated', tran, ['6A'] ).
verb( 'prejudge', 'prejudges', 'prejudging', 'prejudged', 'prejudged', tran, ['6A'] ).
verb( 'prejudice', 'prejudices', 'prejudicing', 'prejudiced', 'prejudiced', tran, ['6A','15A'] ).
verb( 'prelude', 'preludes', 'preluding', 'preluded', 'preluded', tran, ['6A'] ).
verb( 'premeditate', 'premeditates', 'premeditating', 'premeditated', 'premeditated', tran, ['6A'] ).
verb( 'premise', 'premises', 'premising', 'premised', 'premised', tran, ['6A','9'] ).
verb( 'premiss', 'premisses', 'premissing', 'premissed', 'premissed', tran, ['6A','9'] ).
verb( 'preoccupy', 'preoccupies', 'preoccupying', 'preoccupied', 'preoccupied', tran, ['6A'] ).
verb( 'preordain', 'preordains', 'preordaining', 'preordained', 'preordained', tran, ['6A','9'] ).
verb( 'prepare', 'prepares', 'preparing', 'prepared', 'prepared', _, ['3A','6A','7A','14'] ).
verb( 'prepay', 'prepays', 'prepaying', 'prepaid', 'prepaid', tran, ['6A'] ).
verb( 'preponderate', 'preponderates', 'preponderating', 'preponderated', 'preponderated', intran, ['2A','2C'] ).
verb( 'prepossess', 'prepossesses', 'prepossessing', 'prepossessed', 'prepossessed', tran, ['6A','15A'] ).
verb( 'prerecord', 'prerecords', 'prerecording', 'prerecorded', 'prerecorded', tran, ['6A'] ).
verb( 'presage', 'presages', 'presaging', 'presaged', 'presaged', tran, ['6A'] ).
verb( 'prescribe', 'prescribes', 'prescribing', 'prescribed', 'prescribed', _, ['2A','3A','6A','8','10','14','21'] ).
verb( 'present', 'presents', 'presenting', 'presented', 'presented', tran, ['6A','14','15A'] ).
verb( 'preserve', 'preserves', 'preserving', 'preserved', 'preserved', tran, ['6A','14'] ).
verb( 'preside', 'presides', 'presiding', 'presided', 'presided', intran, ['2A','2C','3A'] ).
verb( 'press', 'presses', 'pressing', 'pressed', 'pressed', _, ['2A','2C','3A','4A','6A','14','15A','15B','17','22'] ).
verb( 'presume', 'presumes', 'presuming', 'presumed', 'presumed', _, ['3A','6A','7A','9','25'] ).
verb( 'presuppose', 'presupposes', 'presupposing', 'presupposed', 'presupposed', tran, ['6A','9'] ).
verb( 'pretend', 'pretends', 'pretending', 'pretended', 'pretended', _, ['3A','6A','7A','9'] ).
verb( 'prettify', 'prettifies', 'prettifying', 'prettified', 'prettified', tran, ['6A'] ).
verb( 'prevail', 'prevails', 'prevailing', 'prevailed', 'prevailed', intran, ['2A','3A'] ).
verb( 'prevaricate', 'prevaricates', 'prevaricating', 'prevaricated', 'prevaricated', intran, ['2A'] ).
verb( 'prevent', 'prevents', 'preventing', 'prevented', 'prevented', tran, ['6A','14','19C'] ).
verb( 'preview', 'previews', 'previewing', 'previewed', 'previewed', tran, [] ).
verb( 'prey', 'preys', 'preying', 'preyed', 'preyed', intran, ['3A'] ).
verb( 'price', 'prices', 'pricing', 'priced', 'priced', tran, ['6A'] ).
verb( 'prick', 'pricks', 'pricking', 'pricked', 'pricked', _, ['2A','6A','15B'] ).
verb( 'prickle', 'prickles', 'prickling', 'prickled', 'prickled', _, [] ).
verb( 'pride', 'prides', 'priding', 'prided', 'prided', tran, [] ).
verb( 'prim', 'prims', 'primming', 'primmed', 'primmed', tran, [] ).
verb( 'prime', 'primes', 'priming', 'primed', 'primed', tran, ['6A'] ).
verb( 'primp', 'primps', 'primping', 'primped', 'primped', tran, [] ).
verb( 'prink', 'prinks', 'prinking', 'prinked', 'prinked', tran, [] ).
verb( 'print', 'prints', 'printing', 'printed', 'printed', _, ['2A','6A','15B'] ).
verb( 'prise', 'prises', 'prising', 'prised', 'prised', tran, ['15A','15B'] ).
verb( 'prize', 'prizes', 'prizing', 'prized', 'prized', tran, ['15A','15B'] ).
verb( 'probate', 'probates', 'probating', 'probated', 'probated', tran, [] ).
verb( 'probe', 'probes', 'probing', 'probed', 'probed', tran, ['6A'] ).
verb( 'proceed', 'proceeds', 'proceeding', 'proceeded', 'proceeded', intran, ['2A','3A','4C'] ).
verb( 'process', 'processes', 'processing', 'processed', 'processed', tran, ['6A'] ).
verb( 'process', 'processes', 'processing', 'processed', 'processed', intran, [] ).
verb( 'proclaim', 'proclaims', 'proclaiming', 'proclaimed', 'proclaimed', tran, ['6A','9','23','25'] ).
verb( 'procrastinate', 'procrastinates', 'procrastinating', 'procrastinated', 'procrastinated', intran, ['2A'] ).
verb( 'procreate', 'procreates', 'procreating', 'procreated', 'procreated', tran, ['6A'] ).
verb( 'procure', 'procures', 'procuring', 'procured', 'procured', tran, ['6A','12B','13B'] ).
verb( 'prod', 'prods', 'prodding', 'prodded', 'prodded', _, ['3A','6A'] ).
verb( 'produce', 'produces', 'producing', 'produced', 'produced', _, ['2A','6A'] ).
verb( 'profane', 'profanes', 'profaning', 'profaned', 'profaned', tran, ['6A'] ).
verb( 'profess', 'professes', 'professing', 'professed', 'professed', _, ['6A','7A','9','25'] ).
verb( 'proffer', 'proffers', 'proffering', 'proffered', 'proffered', tran, ['6A','7A'] ).
verb( 'profile', 'profiles', 'profiling', 'profiled', 'profiled', tran, [] ).
verb( 'profit', 'profits', 'profiting', 'profited', 'profited', _, ['3A','6A','13A'] ).
verb( 'profiteer', 'profiteers', 'profiteering', 'profiteered', 'profiteered', intran, ['2A'] ).
verb( 'prognosticate', 'prognosticates', 'prognosticating', 'prognosticated', 'prognosticated', tran, ['6A','9'] ).
verb( 'program', 'programs', 'programming', 'programmed', 'programmed', tran, ['6A'] ).
verb( 'programme', 'programmes', 'programming', 'programmed', 'programmed', tran, ['6A'] ).
verb( 'progress', 'progresses', 'progressing', 'progressed', 'progressed', intran, ['2A','2C'] ).
verb( 'prohibit', 'prohibits', 'prohibiting', 'prohibited', 'prohibited', tran, ['6A','14'] ).
verb( 'project', 'projects', 'projecting', 'projected', 'projected', _, ['2A','2C','6A','14','15A'] ).
verb( 'prolapse', 'prolapses', 'prolapsing', 'prolapsed', 'prolapsed', intran, [] ).
verb( 'proliferate', 'proliferates', 'proliferating', 'proliferated', 'proliferated', _, ['2A','6A'] ).
verb( 'prolong', 'prolongs', 'prolonging', 'prolonged', 'prolonged', tran, ['6A'] ).
verb( 'promenade', 'promenades', 'promenading', 'promenaded', 'promenaded', _, ['2A','2C','6A','15A'] ).
verb( 'promise', 'promises', 'promising', 'promised', 'promised', _, ['2A','6A','7A','9','11','12A','13A','17'] ).
verb( 'promote', 'promotes', 'promoting', 'promoted', 'promoted', tran, ['6A','14'] ).
verb( 'prompt', 'prompts', 'prompting', 'prompted', 'prompted', tran, ['6A','17'] ).
verb( 'promulgate', 'promulgates', 'promulgating', 'promulgated', 'promulgated', tran, ['6A'] ).
verb( 'pronounce', 'pronounces', 'pronouncing', 'pronounced', 'pronounced', _, ['2A','3A','6A','9','22','25'] ).
verb( 'proof', 'proofs', 'proofing', 'proofed', 'proofed', tran, ['6A'] ).
verb( 'proofread', 'proofreads', 'proofreading', 'proofread', 'proofread', _, ['2A','6A'] ).
verb( 'prop', 'props', 'propping', 'propped', 'propped', tran, ['6A','15A','15B','22'] ).
verb( 'propagandize', 'propagandizes', 'propagandizing', 'propagandized', 'propagandized', intran, [] ).
verb( 'propagate', 'propagates', 'propagating', 'propagated', 'propagated', _, ['2A','6A'] ).
verb( 'propel', 'propels', 'propelling', 'propelled', 'propelled', tran, ['6A','15A'] ).
verb( 'prophesy', 'prophesies', 'prophesying', 'prophesied', 'prophesied', _, ['2A','2C','6A','9','10'] ).
verb( 'propitiate', 'propitiates', 'propitiating', 'propitiated', 'propitiated', tran, ['6A'] ).
verb( 'proportion', 'proportions', 'proportioning', 'proportioned', 'proportioned', tran, ['6A','14'] ).
verb( 'propose', 'proposes', 'proposing', 'proposed', 'proposed', _, ['2A','6A','6D','7A','9','14'] ).
verb( 'proposition', 'propositions', 'propositioning', 'propositioned', 'propositioned', tran, ['6A'] ).
verb( 'propound', 'propounds', 'propounding', 'propounded', 'propounded', tran, ['6A'] ).
verb( 'prorogue', 'prorogues', 'proroguing', 'prorogued', 'prorogued', tran, ['6A'] ).
verb( 'proscribe', 'proscribes', 'proscribing', 'proscribed', 'proscribed', tran, ['6A'] ).
verb( 'prosecute', 'prosecutes', 'prosecuting', 'prosecuted', 'prosecuted', tran, ['6A','14'] ).
verb( 'proselytize', 'proselytizes', 'proselytizing', 'proselytized', 'proselytized', _, ['2A','6A'] ).
verb( 'prospect', 'prospects', 'prospecting', 'prospected', 'prospected', intran, ['2A','3A'] ).
verb( 'prosper', 'prospers', 'prospering', 'prospered', 'prospered', _, ['2A','6A'] ).
verb( 'prostitute', 'prostitutes', 'prostituting', 'prostituted', 'prostituted', tran, ['6A'] ).
verb( 'prostrate', 'prostrates', 'prostrating', 'prostrated', 'prostrated', tran, ['6A'] ).
verb( 'protect', 'protects', 'protecting', 'protected', 'protected', tran, ['6A','14'] ).
verb( 'protest', 'protests', 'protesting', 'protested', 'protested', _, ['2A','3A','6A','9'] ).
verb( 'protract', 'protracts', 'protracting', 'protracted', 'protracted', tran, ['6A'] ).
verb( 'protrude', 'protrudes', 'protruding', 'protruded', 'protruded', _, ['2A','6A'] ).
verb( 'prove', 'proves', 'proving', 'proved', 'proved', _, ['4D','6A','9','14','25'] ).
verb( 'provide', 'provides', 'providing', 'provided', 'provided', _, ['3A','6A','9','14'] ).
verb( 'provision', 'provisions', 'provisioning', 'provisioned', 'provisioned', tran, ['6A'] ).
verb( 'provoke', 'provokes', 'provoking', 'provoked', 'provoked', tran, ['6A','14','17'] ).
verb( 'prowl', 'prowls', 'prowling', 'prowled', 'prowled', _, ['2A','2C','6A'] ).
verb( 'prune', 'prunes', 'pruning', 'pruned', 'pruned', tran, ['6A','14','15B'] ).
verb( 'pry', 'pries', 'prying', 'pried', 'pried', _, ['2A','2C','3A','15A','15B','22'] ).
verb( 'psychoanalyse', 'psychoanalyses', 'psychoanalysing', 'psychoanalysed', 'psychoanalysed', tran, [] ).
verb( 'psychoanalyze', 'psychoanalyzes', 'psychoanalyzing', 'psychoanalyzed', 'psychoanalyzed', tran, [] ).
verb( 'pub-crawl', 'pub-crawls', 'pub-crawling', 'pub-crawled', 'pub-crawled', intran, [] ).
verb( 'publicize', 'publicizes', 'publicizing', 'publicized', 'publicized', tran, ['6A'] ).
verb( 'publish', 'publishes', 'publishing', 'published', 'published', tran, ['6A'] ).
verb( 'pucker', 'puckers', 'puckering', 'puckered', 'puckered', _, ['2A','2C','6A','15B'] ).
verb( 'puddle', 'puddles', 'puddling', 'puddled', 'puddled', tran, [] ).
verb( 'puff', 'puffs', 'puffing', 'puffed', 'puffed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'puke', 'pukes', 'puking', 'puked', 'puked', _, [] ).
verb( 'pule', 'pules', 'puling', 'puled', 'puled', intran, ['2A'] ).
verb( 'pull', 'pulls', 'pulling', 'pulled', 'pulled', _, ['2A','2C','3A','6A','15A','15B','22'] ).
verb( 'pullulate', 'pullulates', 'pullulating', 'pullulated', 'pullulated', intran, [] ).
verb( 'pulp', 'pulps', 'pulping', 'pulped', 'pulped', _, ['2A','6A'] ).
verb( 'pulsate', 'pulsates', 'pulsating', 'pulsated', 'pulsated', _, ['2A','6A'] ).
verb( 'pulse', 'pulses', 'pulsing', 'pulsed', 'pulsed', intran, ['2C'] ).
verb( 'pulverize', 'pulverizes', 'pulverizing', 'pulverized', 'pulverized', _, ['2A','6A'] ).
verb( 'pummel', 'pummels', 'pummelling', 'pummelled', 'pummelled', tran, ['6A','15B'] ).
verb( 'pump', 'pumps', 'pumping', 'pumped', 'pumped', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'pun', 'puns', 'punning', 'punned', 'punned', intran, ['2A','3A'] ).
verb( 'punch', 'punches', 'punching', 'punched', 'punched', tran, ['6A','15A','15B'] ).
verb( 'punctuate', 'punctuates', 'punctuating', 'punctuated', 'punctuated', tran, ['6A','15A'] ).
verb( 'puncture', 'punctures', 'puncturing', 'punctured', 'punctured', _, ['2A','6A'] ).
verb( 'punish', 'punishes', 'punishing', 'punished', 'punished', tran, ['6A','14'] ).
verb( 'punt', 'punts', 'punting', 'punted', 'punted', _, ['2A','6A'] ).
verb( 'purchase', 'purchases', 'purchasing', 'purchased', 'purchased', tran, ['6A'] ).
verb( 'purge', 'purges', 'purging', 'purged', 'purged', tran, ['6A','14','15A','15B'] ).
verb( 'purify', 'purifies', 'purifying', 'purified', 'purified', tran, ['6A','14'] ).
verb( 'purl', 'purls', 'purling', 'purled', 'purled', _, [] ).
verb( 'purloin', 'purloins', 'purloining', 'purloined', 'purloined', tran, ['6A'] ).
verb( 'purport', 'purports', 'purporting', 'purported', 'purported', tran, ['6A','7A','9'] ).
verb( 'purpose', 'purposes', 'purposing', 'purposed', 'purposed', tran, ['6A','6D','7A','9'] ).
verb( 'purr', 'purrs', 'purring', 'purred', 'purred', _, ['2A','2C','6A'] ).
verb( 'purse', 'purses', 'pursing', 'pursed', 'pursed', tran, ['6A','15B'] ).
verb( 'pursue', 'pursues', 'pursuing', 'pursued', 'pursued', tran, ['6A'] ).
verb( 'purvey', 'purveys', 'purveying', 'purveyed', 'purveyed', _, ['3A','6A','14'] ).
verb( 'push', 'pushes', 'pushing', 'pushed', 'pushed', _, ['2A','2C','3A','6A','14','15A','15B','17','22'] ).
verb( 'pussyfoot', 'pussyfoots', 'pussyfooting', 'pussyfooted', 'pussyfooted', intran, ['2A','2C'] ).
verb( 'put', 'puts', 'putting', 'put', 'put', _, ['2C','6A','14','15A','15B','22'] ).
verb( 'put', 'puts', 'putting', 'putted', 'putted', _, [] ).
verb( 'putrefy', 'putrefies', 'putrefying', 'putrefied', 'putrefied', _, ['2A','6A'] ).
verb( 'putt', 'putts', 'putting', 'putted', 'putted', _, ['2A','6A'] ).
verb( 'putter', 'putters', 'puttering', 'puttered', 'puttered', _, [] ).
verb( 'putty', 'putties', 'puttying', 'puttied', 'puttied', tran, ['6A','15B'] ).
verb( 'puzzle', 'puzzles', 'puzzling', 'puzzled', 'puzzled', _, ['3A','6A','15B'] ).
verb( 'quack', 'quacks', 'quacking', 'quacked', 'quacked', intran, [] ).
verb( 'quadruple', 'quadruples', 'quadrupling', 'quadrupled', 'quadrupled', _, ['2A','6A'] ).
verb( 'quadruplicate', 'quadruplicates', 'quadruplicating', 'quadruplicated', 'quadruplicated', tran, ['6A'] ).
verb( 'quaff', 'quaffs', 'quaffing', 'quaffed', 'quaffed', _, ['2A','6A','15B'] ).
verb( 'quail', 'quails', 'quailing', 'quailed', 'quailed', intran, ['2A','3A'] ).
verb( 'quake', 'quakes', 'quaking', 'quaked', 'quaked', intran, ['2A','2C'] ).
verb( 'qualify', 'qualifies', 'qualifying', 'qualified', 'qualified', _, ['2C','3A','4A','6A','14','16B','17'] ).
verb( 'quantify', 'quantifies', 'quantifying', 'quantified', 'quantified', tran, ['6A'] ).
verb( 'quarantine', 'quarantines', 'quarantining', 'quarantined', 'quarantined', tran, ['6A'] ).
verb( 'quarrel', 'quarrels', 'quarrelling', 'quarrelled', 'quarrelled', intran, ['2A','2C','3A'] ).
verb( 'quarry', 'quarries', 'quarrying', 'quarried', 'quarried', _, ['2A','2C','6A','15A','15B'] ).
verb( 'quarter', 'quarters', 'quartering', 'quartered', 'quartered', tran, ['6A','15A'] ).
verb( 'quash', 'quashes', 'quashing', 'quashed', 'quashed', tran, ['6A'] ).
verb( 'quaver', 'quavers', 'quavering', 'quavered', 'quavered', _, ['2A','6A','15B'] ).
verb( 'queen', 'queens', 'queening', 'queened', 'queened', tran, [] ).
verb( 'queer', 'queers', 'queering', 'queered', 'queered', tran, ['6A'] ).
verb( 'quell', 'quells', 'quelling', 'quelled', 'quelled', tran, ['6A'] ).
verb( 'quench', 'quenches', 'quenching', 'quenched', 'quenched', tran, ['6A'] ).
verb( 'query', 'queries', 'querying', 'queried', 'queried', tran, ['6A','10'] ).
verb( 'quest', 'quests', 'questing', 'quested', 'quested', intran, ['3A'] ).
verb( 'question', 'questions', 'questioning', 'questioned', 'questioned', tran, ['6A','10'] ).
verb( 'queue', 'queues', 'queueing', 'queued', 'queued', intran, ['2A','2C','3A'] ).
verb( 'quibble', 'quibbles', 'quibbling', 'quibbled', 'quibbled', intran, [] ).
verb( 'quick-freeze', 'quick-freezes', 'quick-freezing', 'quick-froze', 'quick-frozen', tran, [] ).
verb( 'quicken', 'quickens', 'quickening', 'quickened', 'quickened', _, ['2A','6A'] ).
verb( 'quiet', 'quiets', 'quieting', 'quieted', 'quieted', _, [] ).
verb( 'quieten', 'quietens', 'quietening', 'quietened', 'quietened', _, ['2C','6A','15B'] ).
verb( 'quilt', 'quilts', 'quilting', 'quilted', 'quilted', tran, [] ).
verb( 'quip', 'quips', 'quipping', 'quipped', 'quipped', intran, [] ).
verb( 'quit', 'quits', 'quitting', 'quitted', 'quitted', tran, ['2A','6A','6D'] ).
verb( 'quiver', 'quivers', 'quivering', 'quivered', 'quivered', _, ['2A','6A'] ).
verb( 'quiz', 'quizzes', 'quizzing', 'quizzed', 'quizzed', tran, ['6A'] ).
verb( 'quote', 'quotes', 'quoting', 'quoted', 'quoted', tran, ['6A','13A','14'] ).
verb( 'rabbit', 'rabbits', 'rabbiting', 'rabbited', 'rabbited', intran, [] ).
verb( 'race', 'races', 'racing', 'raced', 'raced', _, ['2A','2C','3A','4A','6A','15A'] ).
verb( 'rack', 'racks', 'racking', 'racked', 'racked', tran, ['6A','15A'] ).
verb( 'racket', 'rackets', 'racketing', 'racketed', 'racketed', intran, ['2A','2C'] ).
verb( 'radiate', 'radiates', 'radiating', 'radiated', 'radiated', _, ['2A','3A','6A'] ).
verb( 'raffle', 'raffles', 'raffling', 'raffled', 'raffled', tran, ['6A','15B'] ).
verb( 'raft', 'rafts', 'rafting', 'rafted', 'rafted', _, ['2C','6A','15A','15B'] ).
verb( 'rag', 'rags', 'ragging', 'ragged', 'ragged', tran, ['6A'] ).
verb( 'rage', 'rages', 'raging', 'raged', 'raged', intran, ['2A','2C'] ).
verb( 'raid', 'raids', 'raiding', 'raided', 'raided', _, ['2A','6A'] ).
verb( 'rail', 'rails', 'railing', 'railed', 'railed', _, ['2A','3A','6A','15B'] ).
verb( 'railroad', 'railroads', 'railroading', 'railroaded', 'railroaded', tran, ['15A','15B'] ).
verb( 'rain', 'rains', 'raining', 'rained', 'rained', _, ['2C','14'] ).
verb( 'raise', 'raises', 'raising', 'raised', 'raised', tran, ['6A','15A','15B'] ).
verb( 'rake', 'rakes', 'raking', 'raked', 'raked', _, ['2A','2C','3A','6A','14','15A','15B','22'] ).
verb( 'rally', 'rallies', 'rallying', 'rallied', 'rallied', _, ['2A','2C','6A','15A'] ).
verb( 'ram', 'rams', 'ramming', 'rammed', 'rammed', tran, ['6A','15A','15B'] ).
verb( 'ramble', 'rambles', 'rambling', 'rambled', 'rambled', intran, ['2A','2C'] ).
verb( 'ramify', 'ramifies', 'ramifying', 'ramified', 'ramified', _, ['2A','6A'] ).
verb( 'ramp', 'ramps', 'ramping', 'ramped', 'ramped', tran, ['2C'] ).
verb( 'rampage', 'rampages', 'rampaging', 'rampaged', 'rampaged', intran, ['2A'] ).
verb( 'range', 'ranges', 'ranging', 'ranged', 'ranged', _, ['2C','3A','6A','15A'] ).
verb( 'rank', 'ranks', 'ranking', 'ranked', 'ranked', _, ['3A','6A','15A','16B'] ).
verb( 'rankle', 'rankles', 'rankling', 'rankled', 'rankled', intran, ['2A'] ).
verb( 'ransack', 'ransacks', 'ransacking', 'ransacked', 'ransacked', tran, ['6A','14','16A'] ).
verb( 'ransom', 'ransoms', 'ransoming', 'ransomed', 'ransomed', tran, ['6A'] ).
verb( 'rant', 'rants', 'ranting', 'ranted', 'ranted', _, ['2A','6A'] ).
verb( 'rap', 'raps', 'rapping', 'rapped', 'rapped', _, ['2A','2C','6A','15B'] ).
verb( 'rape', 'rapes', 'raping', 'raped', 'raped', tran, ['6A'] ).
verb( 'rarefy', 'rarefies', 'rarefying', 'rarefied', 'rarefied', _, ['2A','6A'] ).
verb( 'rase', 'rases', 'rasing', 'rased', 'rased', tran, ['6A'] ).
verb( 'rasp', 'rasps', 'rasping', 'rasped', 'rasped', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'rat', 'rats', 'ratting', 'ratted', 'ratted', tran, ['2A','3A'] ).
verb( 'rate', 'rates', 'rating', 'rated', 'rated', _, ['2C','2D','6A','14','15A','16B'] ).
verb( 'ratify', 'ratifies', 'ratifying', 'ratified', 'ratified', tran, ['6A'] ).
verb( 'ration', 'rations', 'rationing', 'rationed', 'rationed', tran, ['6A','15B'] ).
verb( 'rationalize', 'rationalizes', 'rationalizing', 'rationalized', 'rationalized', tran, ['6A'] ).
verb( 'rattle', 'rattles', 'rattling', 'rattled', 'rattled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'ravage', 'ravages', 'ravaging', 'ravaged', 'ravaged', _, ['6A'] ).
verb( 'rave', 'raves', 'raving', 'raved', 'raved', intran, ['2A','2C','3A','15B'] ).
verb( 'ravel', 'ravels', 'ravelling', 'ravelled', 'ravelled', _, ['2A','2C','6A','15B'] ).
verb( 'ravish', 'ravishes', 'ravishing', 'ravished', 'ravished', tran, ['6A'] ).
verb( 'ray', 'rays', 'raying', 'rayed', 'rayed', _, [] ).
verb( 'raze', 'razes', 'razing', 'razed', 'razed', tran, ['6A'] ).
verb( 'razor', 'razors', 'razoring', 'razored', 'razored', tran, [] ).
verb( 're-address', 're-addresses', 're-addressing', 're-addressed', 're-addressed', tran, ['6A'] ).
verb( 're-afforest', 're-afforests', 're-afforesting', 're-afforested', 're-afforested', tran, [] ).
verb( 're-count', 're-counts', 're-counting', 're-counted', 're-counted', tran, ['6A'] ).
verb( 're-cover', 're-covers', 're-covering', 're-covered', 're-covered', tran, ['6A'] ).
verb( 're-echo', 're-echos', 're-echoing', 're-echoed', 're-echoed', intran, ['2A'] ).
verb( 're-form', 're-forms', 're-forming', 're-formed', 're-formed', _, [] ).
verb( 're-join', 're-joins', 're-joining', 're-joined', 're-joined', tran, ['6A'] ).
verb( 'reach', 'reaches', 'reaching', 'reached', 'reached', _, ['2C','3A','6A','12B','13B','15B'] ).
verb( 'react', 'reacts', 'reacting', 'reacted', 'reacted', intran, ['2A','3A'] ).
verb( 'read', 'reads', 'reading', 'read', 'read', _, ['2A','2B','2C','6A','12A','13A','15A','15B','16B'] ).
verb( 'readjust', 'readjusts', 'readjusting', 'readjusted', 'readjusted', tran, ['3A','6A','15A'] ).
verb( 'readmit', 'readmits', 'readmitting', 'readmitted', 'readmitted', tran, ['3A','6A','6C','9','14','25'] ).
verb( 'reaffirm', 'reaffirms', 'reaffirming', 'reaffirmed', 'reaffirmed', tran, ['6A','9'] ).
verb( 'realize', 'realizes', 'realizing', 'realized', 'realized', tran, ['6A','9','10','14'] ).
verb( 'reanimate', 'reanimates', 'reanimating', 'reanimated', 'reanimated', tran, ['6A'] ).
verb( 'reap', 'reaps', 'reaping', 'reaped', 'reaped', _, ['2A','6A'] ).
verb( 'reappear', 'reappears', 'reappearing', 'reappeared', 'reappeared', intran, ['2A'] ).
verb( 'rear', 'rears', 'rearing', 'reared', 'reared', _, ['2A','2C','6A','15B'] ).
verb( 'rearm', 'rearms', 'rearming', 'rearmed', 'rearmed', _, ['2A','6A'] ).
verb( 'rearrange', 'rearranges', 'rearranging', 'rearranged', 'rearranged', tran, ['3A','4C','6A','14','15A'] ).
verb( 'reason', 'reasons', 'reasoning', 'reasoned', 'reasoned', _, ['2A','3A','6A','9','14'] ).
verb( 'reassemble', 'reassembles', 'reassembling', 'reassembled', 'reassembled', _, ['2A','6A'] ).
verb( 'reassess', 'reassesses', 'reassessing', 'reassessed', 'reassessed', tran, ['6A','14'] ).
verb( 'reassure', 'reassures', 'reassuring', 'reassured', 'reassured', tran, ['6A'] ).
verb( 'reattribute', 'reattributes', 'reattributing', 'reattributed', 'reattributed', tran, ['14'] ).
verb( 'rebel', 'rebels', 'rebelling', 'rebelled', 'rebelled', intran, ['2A','3A'] ).
verb( 'rebind', 'rebinds', 'rebinding', 'rebound', 'rebound', tran, ['6A'] ).
verb( 'rebound', 'rebounds', 'rebounding', 'rebounded', 'rebounded', intran, ['2A','3A'] ).
verb( 'rebuff', 'rebuffs', 'rebuffing', 'rebuffed', 'rebuffed', tran, ['6A'] ).
verb( 'rebuild', 'rebuilds', 'rebuilding', 'rebuilt', 'rebuilt', tran, ['6A'] ).
verb( 'rebuke', 'rebukes', 'rebuking', 'rebuked', 'rebuked', tran, ['6A','14'] ).
verb( 'rebut', 'rebuts', 'rebutting', 'rebutted', 'rebutted', tran, ['6A'] ).
verb( 'recall', 'recalls', 'recalling', 'recalled', 'recalled', tran, ['6A','6C','8','9','10','14','19C'] ).
verb( 'recant', 'recants', 'recanting', 'recanted', 'recanted', _, ['2A','6A'] ).
verb( 'recap', 'recaps', 'recapping', 'recapped', 'recapped', _, [] ).
verb( 'recap', 'recaps', 'recapping', 'recapped', 'recapped', tran, [] ).
verb( 'recapitulate', 'recapitulates', 'recapitulating', 'recapitulated', 'recapitulated', _, ['2A','6A'] ).
verb( 'recapture', 'recaptures', 'recapturing', 'recaptured', 'recaptured', tran, ['6A'] ).
verb( 'recast', 'recasts', 'recasting', 'recast', 'recast', tran, ['6A'] ).
verb( 'recede', 'recedes', 'receding', 'receded', 'receded', intran, ['2A','3A'] ).
verb( 'receipt', 'receipts', 'receipting', 'receipted', 'receipted', tran, ['6A'] ).
verb( 'receive', 'receives', 'receiving', 'received', 'received', _, ['2A','6A'] ).
verb( 'recess', 'recesses', 'recessing', 'recessed', 'recessed', tran, ['6A'] ).
verb( 'reciprocate', 'reciprocates', 'reciprocating', 'reciprocated', 'reciprocated', _, ['2A','6A'] ).
verb( 'recite', 'recites', 'reciting', 'recited', 'recited', _, ['2A','6A','15A'] ).
verb( 'reckon', 'reckons', 'reckoning', 'reckoned', 'reckoned', _, ['2A','6A','7A','9','15B','16B','25'] ).
verb( 'reclaim', 'reclaims', 'reclaiming', 'reclaimed', 'reclaimed', tran, ['6A'] ).
verb( 'recline', 'reclines', 'reclining', 'reclined', 'reclined', _, ['2A','2C','15A'] ).
verb( 'recognize', 'recognizes', 'recognizing', 'recognized', 'recognized', tran, ['6A','9','16A','25'] ).
verb( 'recoil', 'recoils', 'recoiling', 'recoiled', 'recoiled', intran, ['2A','3A'] ).
verb( 'recollect', 'recollects', 'recollecting', 'recollected', 'recollected', _, ['2A','6A','6C','8','9','10'] ).
verb( 'recommend', 'recommends', 'recommending', 'recommended', 'recommended', tran, ['6A','6C','9','12A','13A','14','16A','17'] ).
verb( 'recommit', 'recommits', 'recommitting', 'recommitted', 'recommitted', tran, ['6A','14','16A'] ).
verb( 'recompense', 'recompenses', 'recompensing', 'recompensed', 'recompensed', tran, ['6A','14'] ).
verb( 'reconcile', 'reconciles', 'reconciling', 'reconciled', 'reconciled', tran, ['6A','14'] ).
verb( 'recondition', 'reconditions', 'reconditioning', 'reconditioned', 'reconditioned', tran, ['6A'] ).
verb( 'reconnoitre', 'reconnoitres', 'reconnoitring', 'reconnoitred', 'reconnoitred', _, ['2A','6A'] ).
verb( 'reconsecrate', 'reconsecrates', 'reconsecrating', 'reconsecrated', 'reconsecrated', tran, ['6A','14','23'] ).
verb( 'reconsider', 'reconsiders', 'reconsidering', 'reconsidered', 'reconsidered', tran, ['6A','6C','8','9','10','25'] ).
verb( 'reconstruct', 'reconstructs', 'reconstructing', 'reconstructed', 'reconstructed', tran, ['6A'] ).
verb( 'reconvict', 'reconvicts', 'reconvicting', 'reconvicted', 'reconvicted', tran, ['6A','14'] ).
verb( 'record', 'records', 'recording', 'recorded', 'recorded', tran, ['6A'] ).
verb( 'recount', 'recounts', 'recounting', 'recounted', 'recounted', tran, ['6A'] ).
verb( 'recoup', 'recoups', 'recouping', 'recouped', 'recouped', tran, ['6A','14'] ).
verb( 'recover', 'recovers', 'recovering', 'recovered', 'recovered', _, ['2A','3A','6A'] ).
verb( 'recreate', 'recreates', 'recreating', 'recreated', 'recreated', tran, ['6A','23'] ).
verb( 'recriminate', 'recriminates', 'recriminating', 'recriminated', 'recriminated', intran, ['2A','3A'] ).
verb( 'recruit', 'recruits', 'recruiting', 'recruited', 'recruited', _, ['6A'] ).
verb( 'rectify', 'rectifies', 'rectifying', 'rectified', 'rectified', tran, ['6A'] ).
verb( 'recuperate', 'recuperates', 'recuperating', 'recuperated', 'recuperated', _, ['2A','6A'] ).
verb( 'recur', 'recurs', 'recurring', 'recurred', 'recurred', intran, ['2A','3A'] ).
verb( 'recurve', 'recurves', 'recurving', 'recurved', 'recurved', _, [] ).
verb( 'recycle', 'recycles', 'recycling', 'recycled', 'recycled', tran, ['6A'] ).
verb( 'redact', 'redacts', 'redacting', 'redacted', 'redacted', tran, ['6A'] ).
verb( 'redden', 'reddens', 'reddening', 'reddened', 'reddened', _, ['2A','6A'] ).
verb( 'redecorate', 'redecorates', 'redecorating', 'redecorated', 'redecorated', tran, ['6A','14'] ).
verb( 'redeem', 'redeems', 'redeeming', 'redeemed', 'redeemed', tran, ['6A','14'] ).
verb( 'redefine', 'redefines', 'redefining', 'redefined', 'redefined', tran, ['6A'] ).
verb( 'redeploy', 'redeploys', 'redeploying', 'redeployed', 'redeployed', tran, ['6A'] ).
verb( 'redesign', 'redesigns', 'redesigning', 'redesigned', 'redesigned', tran, ['2A','2C','6A','14','16A','16B'] ).
verb( 'redevelop', 'redevelops', 'redeveloping', 'redeveloped', 'redeveloped', tran, ['2A','2C','3A','6A'] ).
verb( 'rediscover', 'rediscovers', 'rediscovering', 'rediscovered', 'rediscovered', tran, ['6A','8','9','10','25'] ).
verb( 'redistribute', 'redistributes', 'redistributing', 'redistributed', 'redistributed', tran, ['6A','14'] ).
verb( 'redo', 'redoes', 'redoing', 'redid', 'redone', tran, ['6A'] ).
verb( 'redouble', 'redoubles', 'redoubling', 'redoubled', 'redoubled', _, ['2A','6A'] ).
verb( 'redound', 'redounds', 'redounding', 'redounded', 'redounded', intran, ['3A'] ).
verb( 'redress', 'redresses', 'redressing', 'redressed', 'redressed', tran, ['6A'] ).
verb( 'reduce', 'reduces', 'reducing', 'reduced', 'reduced', _, ['2A','2B','6A','14'] ).
verb( 'reduplicate', 'reduplicates', 'reduplicating', 'reduplicated', 'reduplicated', tran, ['6A'] ).
verb( 'reef', 'reefs', 'reefing', 'reefed', 'reefed', tran, ['6A'] ).
verb( 'reek', 'reeks', 'reeking', 'reeked', 'reeked', intran, ['3A'] ).
verb( 'reel', 'reels', 'reeling', 'reeled', 'reeled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'reface', 'refaces', 'refacing', 'refaced', 'refaced', tran, [] ).
verb( 'refashion', 'refashions', 'refashioning', 'refashioned', 'refashioned', tran, ['6A','15A'] ).
verb( 'refer', 'refers', 'referring', 'referred', 'referred', _, ['3A','14','15B'] ).
verb( 'referee', 'referees', 'refereeing', 'refereed', 'refereed', _, ['2A','6A'] ).
verb( 'refill', 'refills', 'refilling', 'refilled', 'refilled', tran, [] ).
verb( 'refine', 'refines', 'refining', 'refined', 'refined', _, ['2A','6A'] ).
verb( 'refit', 'refits', 'refitting', 'refitted', 'refitted', _, ['2A','6A'] ).
verb( 'reflate', 'reflates', 'reflating', 'reflated', 'reflated', tran, ['6A'] ).
verb( 'reflect', 'reflects', 'reflecting', 'reflected', 'reflected', _, ['2A','3A','6A','8','9','10','14'] ).
verb( 'refloat', 'refloats', 'refloating', 'refloated', 'refloated', _, ['2A','6A'] ).
verb( 'reforest', 'reforests', 'reforesting', 'reforested', 'reforested', tran, [] ).
verb( 'reform', 'reforms', 'reforming', 'reformed', 'reformed', _, ['2A','6A'] ).
verb( 'refract', 'refracts', 'refracting', 'refracted', 'refracted', tran, ['6A'] ).
verb( 'refrain', 'refrains', 'refraining', 'refrained', 'refrained', intran, ['2A','3A'] ).
verb( 'refresh', 'refreshes', 'refreshing', 'refreshed', 'refreshed', tran, ['6A'] ).
verb( 'refrigerate', 'refrigerates', 'refrigerating', 'refrigerated', 'refrigerated', tran, ['6A'] ).
verb( 'refuel', 'refuels', 'refuelling', 'refuelled', 'refuelled', _, ['2A','6A'] ).
verb( 'refund', 'refunds', 'refunding', 'refunded', 'refunded', tran, ['6A'] ).
verb( 'refurbish', 'refurbishes', 'refurbishing', 'refurbished', 'refurbished', tran, ['6A'] ).
verb( 'refurnish', 'refurnishes', 'refurnishing', 'refurnished', 'refurnished', tran, ['6A','14'] ).
verb( 'refuse', 'refuses', 'refusing', 'refused', 'refused', _, ['2A','6A','7A','12C'] ).
verb( 'refute', 'refutes', 'refuting', 'refuted', 'refuted', tran, ['6A'] ).
verb( 'regain', 'regains', 'regaining', 'regained', 'regained', tran, ['6A'] ).
verb( 'regale', 'regales', 'regaling', 'regaled', 'regaled', tran, ['6A','14'] ).
verb( 'regard', 'regards', 'regarding', 'regarded', 'regarded', tran, ['6A','14','16B'] ).
verb( 'regenerate', 'regenerates', 'regenerating', 'regenerated', 'regenerated', _, ['2A','6A'] ).
verb( 'regiment', 'regiments', 'regimenting', 'regimented', 'regimented', tran, ['6A'] ).
verb( 'register', 'registers', 'registering', 'registered', 'registered', _, ['2A','3A','6A','14'] ).
verb( 'regress', 'regresses', 'regressing', 'regressed', 'regressed', intran, ['2A'] ).
verb( 'regret', 'regrets', 'regretting', 'regretted', 'regretted', tran, ['6A','6D','7A','9'] ).
verb( 'regroup', 'regroups', 'regrouping', 'regrouped', 'regrouped', _, ['2A','6A'] ).
verb( 'regularize', 'regularizes', 'regularizing', 'regularized', 'regularized', tran, ['6A'] ).
verb( 'regulate', 'regulates', 'regulating', 'regulated', 'regulated', tran, ['6A'] ).
verb( 'regurgitate', 'regurgitates', 'regurgitating', 'regurgitated', 'regurgitated', _, ['2A','6A'] ).
verb( 'rehabilitate', 'rehabilitates', 'rehabilitating', 'rehabilitated', 'rehabilitated', tran, ['6A'] ).
verb( 'rehash', 'rehashes', 'rehashing', 'rehashed', 'rehashed', tran, [] ).
verb( 'rehear', 'rehears', 'rehearing', 'reheard', 'reheard', tran, ['6A'] ).
verb( 'rehearse', 'rehearses', 'rehearsing', 'rehearsed', 'rehearsed', _, ['2A','6A'] ).
verb( 'rehouse', 'rehouses', 'rehousing', 'rehoused', 'rehoused', tran, ['6A'] ).
verb( 'reign', 'reigns', 'reigning', 'reigned', 'reigned', intran, ['2A','3A'] ).
verb( 'reimburse', 'reimburses', 'reimbursing', 'reimbursed', 'reimbursed', tran, ['6A','12A','13A','14'] ).
verb( 'rein', 'reins', 'reining', 'reined', 'reined', tran, ['6A','15B'] ).
verb( 'reincarnate', 'reincarnates', 'reincarnating', 'reincarnated', 'reincarnated', tran, [] ).
verb( 'reinforce', 'reinforces', 'reinforcing', 'reinforced', 'reinforced', tran, ['6A'] ).
verb( 'reinstate', 'reinstates', 'reinstating', 'reinstated', 'reinstated', tran, ['6A','14'] ).
verb( 'reinsure', 'reinsures', 'reinsuring', 'reinsured', 'reinsured', tran, ['6A'] ).
verb( 'reintegrate', 'reintegrates', 'reintegrating', 'reintegrated', 'reintegrated', tran, ['2A','6A'] ).
verb( 'reinterpret', 'reinterprets', 'reinterpreting', 'reinterpreted', 'reinterpreted', tran, ['2A','6A','16B'] ).
verb( 'reissue', 'reissues', 'reissuing', 'reissued', 'reissued', tran, ['6A'] ).
verb( 'reiterate', 'reiterates', 'reiterating', 'reiterated', 'reiterated', tran, ['6A'] ).
verb( 'reject', 'rejects', 'rejecting', 'rejected', 'rejected', tran, ['6A'] ).
verb( 'rejig', 'rejigs', 'rejigging', 'rejigged', 'rejigged', tran, ['6A'] ).
verb( 'rejoice', 'rejoices', 'rejoicing', 'rejoiced', 'rejoiced', _, ['2A','2C','3B','4C','6A'] ).
verb( 'rejoin', 'rejoins', 'rejoining', 'rejoined', 'rejoined', tran, ['6A'] ).
verb( 'rejoin', 'rejoins', 'rejoining', 'rejoined', 'rejoined', _, ['2A','6A'] ).
verb( 'rejuvenate', 'rejuvenates', 'rejuvenating', 'rejuvenated', 'rejuvenated', _, ['2A','6A'] ).
verb( 'rekindle', 'rekindles', 'rekindling', 'rekindled', 'rekindled', _, ['2A','6A'] ).
verb( 'relapse', 'relapses', 'relapsing', 'relapsed', 'relapsed', intran, ['2A','3A'] ).
verb( 'relate', 'relates', 'relating', 'related', 'related', _, ['3A','6A','14'] ).
verb( 'relax', 'relaxes', 'relaxing', 'relaxed', 'relaxed', _, ['2A','2C','6A'] ).
verb( 'relay', 'relays', 'relaying', 'relayed', 'relayed', tran, [] ).
verb( 'relay', 'relays', 'relaying', 'relayed', 'relayed', tran, [] ).
verb( 'release', 'releases', 'releasing', 'released', 'released', tran, ['6A','14'] ).
verb( 'relegate', 'relegates', 'relegating', 'relegated', 'relegated', tran, ['14'] ).
verb( 'relent', 'relents', 'relenting', 'relented', 'relented', intran, ['2A'] ).
verb( 'relieve', 'relieves', 'relieving', 'relieved', 'relieved', tran, ['6A'] ).
verb( 'reline', 'relines', 'relining', 'relined', 'relined', tran, [] ).
verb( 'relinquish', 'relinquishes', 'relinquishing', 'relinquished', 'relinquished', tran, ['6A','14'] ).
verb( 'relish', 'relishes', 'relishing', 'relished', 'relished', tran, ['6A','6D'] ).
verb( 'relive', 'relives', 'reliving', 'relived', 'relived', tran, [] ).
verb( 'relocate', 'relocates', 'relocating', 'relocated', 'relocated', _, [] ).
verb( 'rely', 'relies', 'relying', 'relied', 'relied', intran, ['3A'] ).
verb( 'remain', 'remains', 'remaining', 'remained', 'remained', intran, ['2A','2B','2C','4A'] ).
verb( 'remake', 'remakes', 'remaking', 'remade', 'remade', tran, [] ).
verb( 'remand', 'remands', 'remanding', 'remanded', 'remanded', tran, ['6A'] ).
verb( 'remark', 'remarks', 'remarking', 'remarked', 'remarked', _, ['3A','6A','9','10'] ).
verb( 'remarry', 'remarries', 'remarrying', 'remarried', 'remarried', _, [] ).
verb( 'remedy', 'remedies', 'remedying', 'remedied', 'remedied', tran, ['6A'] ).
verb( 'remember', 'remembers', 'remembering', 'remembered', 'remembered', _, ['6A','6C','7A','8','9','10','14','16B','19C'] ).
verb( 'remilitarize', 'remilitarizes', 'remilitarizing', 'remilitarized', 'remilitarized', tran, ['6A'] ).
verb( 'remind', 'reminds', 'reminding', 'reminded', 'reminded', tran, ['6A','11','14','17','20','21'] ).
verb( 'reminisce', 'reminisces', 'reminiscing', 'reminisced', 'reminisced', intran, ['2A','3A'] ).
verb( 'remit', 'remits', 'remitting', 'remitted', 'remitted', _, ['2C','6A','12A','13A','14'] ).
verb( 'remodel', 'remodels', 'remodelling', 'remodelled', 'remodelled', tran, ['2A','6A','14','15A'] ).
verb( 'remonstrate', 'remonstrates', 'remonstrating', 'remonstrated', 'remonstrated', intran, ['2A','3A'] ).
verb( 'remould', 'remoulds', 'remoulding', 'remoulded', 'remoulded', tran, ['2A','6A','14'] ).
verb( 'remount', 'remounts', 'remounting', 'remounted', 'remounted', _, ['2A','6A'] ).
verb( 'remove', 'removes', 'removing', 'removed', 'removed', _, ['2A','2C','6A','14'] ).
verb( 'remunerate', 'remunerates', 'remunerating', 'remunerated', 'remunerated', tran, ['6A','14'] ).
verb( 'rename', 'renames', 'renaming', 'renamed', 'renamed', tran, ['6A'] ).
verb( 'rend', 'rends', 'rending', 'rent', 'rent', tran, ['6A','14','15A'] ).
verb( 'render', 'renders', 'rendering', 'rendered', 'rendered', tran, ['6A','12A','13A','14','15A','15B','22'] ).
verb( 'rendezvous', 'rendezvous', 'rendezvouing', 'rendezvoued', 'rendezvoued', intran, ['2A','2C'] ).
verb( 'renegade', 'renegades', 'renegading', 'renegaded', 'renegaded', intran, [] ).
verb( 'renege', 'reneges', 'reneging', 'reneged', 'reneged', intran, ['3A'] ).
verb( 'renegue', 'renegues', 'reneguing', 'renegued', 'renegued', intran, ['3A'] ).
verb( 'renew', 'renews', 'renewing', 'renewed', 'renewed', tran, ['6A'] ).
verb( 'renounce', 'renounces', 'renouncing', 'renounced', 'renounced', tran, ['6A'] ).
verb( 'renovate', 'renovates', 'renovating', 'renovated', 'renovated', tran, ['6A'] ).
verb( 'rent', 'rents', 'renting', 'rented', 'rented', _, ['2A','2C','6A','14','15A'] ).
verb( 'reopen', 'reopens', 'reopening', 'reopened', 'reopened', _, ['2A','6A'] ).
verb( 'reorganize', 'reorganizes', 'reorganizing', 'reorganized', 'reorganized', _, [] ).
verb( 'reorient', 'reorients', 'reorienting', 'reoriented', 'reoriented', _, [] ).
verb( 'reorientate', 'reorientates', 'reorientating', 'reorientated', 'reorientated', _, [] ).
verb( 'repaint', 'repaints', 'repainting', 'repainted', 'repainted', tran, ['2A','2C','6A','15A','15B','22'] ).
verb( 'repair', 'repairs', 'repairing', 'repaired', 'repaired', _, ['3A','6A'] ).
verb( 'repatriate', 'repatriates', 'repatriating', 'repatriated', 'repatriated', tran, ['6A'] ).
verb( 'repay', 'repays', 'repaying', 'repaid', 'repaid', _, ['2A','6A','14'] ).
verb( 'repeal', 'repeals', 'repealing', 'repealed', 'repealed', tran, ['6A'] ).
verb( 'repeat', 'repeats', 'repeating', 'repeated', 'repeated', _, ['2A','6A','9'] ).
verb( 'repel', 'repels', 'repelling', 'repelled', 'repelled', tran, ['6A'] ).
verb( 'repent', 'repents', 'repenting', 'repented', 'repented', _, ['2A','3A','6A','6D'] ).
verb( 'repine', 'repines', 'repining', 'repined', 'repined', intran, ['2A','3A'] ).
verb( 'replace', 'replaces', 'replacing', 'replaced', 'replaced', tran, ['6A','14','15A'] ).
verb( 'replant', 'replants', 'replanting', 'replanted', 'replanted', tran, ['6A','15A','15B'] ).
verb( 'replay', 'replays', 'replaying', 'replayed', 'replayed', tran, ['6A'] ).
verb( 'replenish', 'replenishes', 'replenishing', 'replenished', 'replenished', tran, ['6A','14'] ).
verb( 'reply', 'replies', 'replying', 'replied', 'replied', _, ['2A','3A','3B'] ).
verb( 'repoint', 'repoints', 'repointing', 'repointed', 'repointed', tran, ['6A'] ).
verb( 'report', 'reports', 'reporting', 'reported', 'reported', _, ['2A','3A','6A','6D','9','14','15A','15B','25'] ).
verb( 'repose', 'reposes', 'reposing', 'reposed', 'reposed', _, ['2A','2C','3A','6A','14','15A'] ).
verb( 'repot', 'repots', 'repotting', 'repotted', 'repotted', tran, [] ).
verb( 'reprehend', 'reprehends', 'reprehending', 'reprehended', 'reprehended', tran, ['6A'] ).
verb( 'represent', 'represents', 'representing', 'represented', 'represented', tran, ['6A','9','14','16A','25'] ).
verb( 'represent', 'represents', 'representing', 'represented', 'represented', tran, [] ).
verb( 'repress', 'represses', 'repressing', 'repressed', 'repressed', tran, ['6A'] ).
verb( 'reprieve', 'reprieves', 'reprieving', 'reprieved', 'reprieved', tran, ['6A'] ).
verb( 'reprimand', 'reprimands', 'reprimanding', 'reprimanded', 'reprimanded', tran, ['6A'] ).
verb( 'reprint', 'reprints', 'reprinting', 'reprinted', 'reprinted', tran, ['6A'] ).
verb( 'reproach', 'reproaches', 'reproaching', 'reproached', 'reproached', tran, ['6A','14'] ).
verb( 'reprobate', 'reprobates', 'reprobating', 'reprobated', 'reprobated', tran, ['6A'] ).
verb( 'reproduce', 'reproduces', 'reproducing', 'reproduced', 'reproduced', _, ['2A','6A'] ).
verb( 'reproof', 'reproofs', 'reproofing', 'reproofed', 'reproofed', tran, ['6A'] ).
verb( 'reprove', 'reproves', 'reproving', 'reproved', 'reproved', tran, ['6A','14'] ).
verb( 'repudiate', 'repudiates', 'repudiating', 'repudiated', 'repudiated', tran, ['6A'] ).
verb( 'repulse', 'repulses', 'repulsing', 'repulsed', 'repulsed', tran, ['6A'] ).
verb( 'repute', 'reputes', 'reputing', 'reputed', 'reputed', tran, ['25'] ).
verb( 'request', 'requests', 'requesting', 'requested', 'requested', tran, ['6A','9','17'] ).
verb( 'require', 'requires', 'requiring', 'required', 'required', tran, ['6A','6D','9','14','17'] ).
verb( 'requisition', 'requisitions', 'requisitioning', 'requisitioned', 'requisitioned', tran, ['6A','14'] ).
verb( 'requite', 'requites', 'requiting', 'requited', 'requited', tran, ['6A','14'] ).
verb( 'reread', 'rereads', 'rereading', 'reread', 'reread', tran, ['2A','2B','2C','6A','12A','13A','15A','15B','16B'] ).
verb( 'rerun', 'reruns', 'rerunning', 'reran', 'rerun', tran, [] ).
verb( 'rescind', 'rescinds', 'rescinding', 'rescinded', 'rescinded', tran, ['6A'] ).
verb( 'rescue', 'rescues', 'rescuing', 'rescued', 'rescued', tran, ['6A','14'] ).
verb( 'research', 'researches', 'researching', 'researched', 'researched', intran, ['2A','3A'] ).
verb( 'reseat', 'reseats', 'reseating', 'reseated', 'reseated', tran, ['6A'] ).
verb( 'reseed', 'reseeds', 'reseeding', 'reseeded', 'reseeded', tran, ['2A','6A'] ).
verb( 'resell', 'resells', 'reselling', 'resold', 'resold', tran, ['2A','2C','6A','12A','13A','15B'] ).
verb( 'resemble', 'resembles', 'resembling', 'resembled', 'resembled', tran, ['6B'] ).
verb( 'resent', 'resents', 'resenting', 'resented', 'resented', tran, ['6A','6C','19C'] ).
verb( 'reserve', 'reserves', 'reserving', 'reserved', 'reserved', tran, ['6A','14'] ).
verb( 'reset', 'resets', 'resetting', 'reset', 'reset', tran, ['6A'] ).
verb( 'resettle', 'resettles', 'resettling', 'resettled', 'resettled', _, ['2A','6A'] ).
verb( 'reshape', 'reshapes', 'reshaping', 'reshaped', 'reshaped', tran, ['2A','2C','6A','15A'] ).
verb( 'reshuffle', 'reshuffles', 'reshuffling', 'reshuffled', 'reshuffled', tran, ['6A'] ).
verb( 'reside', 'resides', 'residing', 'resided', 'resided', intran, ['2C','3A'] ).
verb( 'resign', 'resigns', 'resigning', 'resigned', 'resigned', _, ['2A','3A','6A','14'] ).
verb( 'resist', 'resists', 'resisting', 'resisted', 'resisted', _, ['2A','6A','6C'] ).
verb( 'resole', 'resoles', 'resoling', 'resoled', 'resoled', tran, ['6A'] ).
verb( 'resolve', 'resolves', 'resolving', 'resolved', 'resolved', _, ['3A','6A','7A','9','14'] ).
verb( 'resonate', 'resonates', 'resonating', 'resonated', 'resonated', _, [] ).
verb( 'resort', 'resorts', 'resorting', 'resorted', 'resorted', intran, ['3A'] ).
verb( 'resound', 'resounds', 'resounding', 'resounded', 'resounded', _, ['2A','2C','3A'] ).
verb( 'respect', 'respects', 'respecting', 'respected', 'respected', tran, ['6A'] ).
verb( 'respire', 'respires', 'respiring', 'respired', 'respired', intran, ['2A'] ).
verb( 'respite', 'respites', 'respiting', 'respited', 'respited', tran, ['6A'] ).
verb( 'respond', 'responds', 'responding', 'responded', 'responded', intran, ['2A','2C','3A','3B'] ).
verb( 'rest', 'rests', 'resting', 'rested', 'rested', _, ['2A','2B','2C','2D','3A','6A','14'] ).
verb( 'restart', 'restarts', 'restarting', 'restarted', 'restarted', _, ['2A','2C','3A','6A','6D','7A','15A','19B'] ).
verb( 'restate', 'restates', 'restating', 'restated', 'restated', tran, ['6A'] ).
verb( 'restock', 'restocks', 'restocking', 'restocked', 'restocked', tran, ['6A'] ).
verb( 'restore', 'restores', 'restoring', 'restored', 'restored', tran, ['6A','14'] ).
verb( 'restrain', 'restrains', 'restraining', 'restrained', 'restrained', tran, ['6A','14'] ).
verb( 'restrict', 'restricts', 'restricting', 'restricted', 'restricted', tran, ['6A','14'] ).
verb( 'restructure', 'restructures', 'restructuring', 'restructured', 'restructured', tran, ['6A'] ).
verb( 'result', 'results', 'resulting', 'resulted', 'resulted', intran, ['2A','3A'] ).
verb( 'resume', 'resumes', 'resuming', 'resumed', 'resumed', tran, ['6A','6D'] ).
verb( 'resurface', 'resurfaces', 'resurfacing', 'resurfaced', 'resurfaced', _, ['2A','6A'] ).
verb( 'resurrect', 'resurrects', 'resurrecting', 'resurrected', 'resurrected', _, ['2A','6A'] ).
verb( 'resuscitate', 'resuscitates', 'resuscitating', 'resuscitated', 'resuscitated', _, ['2A','6A'] ).
verb( 'ret', 'rets', 'retting', 'retted', 'retted', tran, ['6A'] ).
verb( 'retail', 'retails', 'retailing', 'retailed', 'retailed', _, ['3A','6A'] ).
verb( 'retain', 'retains', 'retaining', 'retained', 'retained', tran, ['6A'] ).
verb( 'retake', 'retakes', 'retaking', 'retook', 'retaken', tran, ['6A'] ).
verb( 'retaliate', 'retaliates', 'retaliating', 'retaliated', 'retaliated', intran, ['2A','3A'] ).
verb( 'retard', 'retards', 'retarding', 'retarded', 'retarded', tran, ['6A'] ).
verb( 'retch', 'retches', 'retching', 'retched', 'retched', intran, ['2A'] ).
verb( 'retell', 'retells', 'retelling', 'retold', 'retold', tran, ['6A'] ).
verb( 'rethink', 'rethinks', 'rethinking', 'rethought', 'rethought', _, ['2A','6A'] ).
verb( 'reticulate', 'reticulates', 'reticulating', 'reticulated', 'reticulated', _, ['2A','6A'] ).
verb( 'retire', 'retires', 'retiring', 'retired', 'retired', _, ['2A','2C','3A','6A'] ).
verb( 'retool', 'retools', 'retooling', 'retooled', 'retooled', tran, ['6A'] ).
verb( 'retort', 'retorts', 'retorting', 'retorted', 'retorted', _, ['2A','6A','9','14'] ).
verb( 'retouch', 'retouches', 'retouching', 'retouched', 'retouched', tran, ['6A'] ).
verb( 'retrace', 'retraces', 'retracing', 'retraced', 'retraced', tran, ['6A'] ).
verb( 'retract', 'retracts', 'retracting', 'retracted', 'retracted', _, ['2A','6A'] ).
verb( 'retransmit', 'retransmits', 'retransmitting', 'retransmitted', 'retransmitted', tran, ['6A','14'] ).
verb( 'retread', 'retreads', 'retreading', 'retreaded', 'retreaded', tran, [] ).
verb( 'retreat', 'retreats', 'retreating', 'retreated', 'retreated', intran, ['2A','2C','3A'] ).
verb( 'retrench', 'retrenches', 'retrenching', 'retrenched', 'retrenched', _, ['2A','6A'] ).
verb( 'retrieve', 'retrieves', 'retrieving', 'retrieved', 'retrieved', _, ['2A','6A','14'] ).
verb( 'retrograde', 'retrogrades', 'retrograding', 'retrograded', 'retrograded', intran, ['2A'] ).
verb( 'retrogress', 'retrogresses', 'retrogressing', 'retrogressed', 'retrogressed', intran, ['2A'] ).
verb( 'return', 'returns', 'returning', 'returned', 'returned', _, ['2A','2C','3A','4A','6A','12A','13A','15A','16A'] ).
verb( 'reunite', 'reunites', 'reuniting', 'reunited', 'reunited', _, ['2A','6A'] ).
verb( 'rev', 'revs', 'revving', 'revved', 'revved', _, ['2A','2C','6A','15B'] ).
verb( 'revalue', 'revalues', 'revaluing', 'revalued', 'revalued', tran, ['6A'] ).
verb( 'revamp', 'revamps', 'revamping', 'revamped', 'revamped', tran, ['6A'] ).
verb( 'reveal', 'reveals', 'revealing', 'revealed', 'revealed', tran, ['6A','9','14','25'] ).
verb( 'revel', 'revels', 'revelling', 'revelled', 'revelled', intran, ['2A','2B','2C','3A','15B'] ).
verb( 'revenge', 'revenges', 'revenging', 'revenged', 'revenged', tran, ['6A'] ).
verb( 'reverberate', 'reverberates', 'reverberating', 'reverberated', 'reverberated', _, ['2A','6A'] ).
verb( 'revere', 'reveres', 'revering', 'revered', 'revered', tran, ['6A'] ).
verb( 'reverence', 'reverences', 'reverencing', 'reverenced', 'reverenced', tran, ['6A'] ).
verb( 'reverse', 'reverses', 'reversing', 'reversed', 'reversed', _, ['2A','6A'] ).
verb( 'revert', 'reverts', 'reverting', 'reverted', 'reverted', intran, ['2A','3A'] ).
verb( 'review', 'reviews', 'reviewing', 'reviewed', 'reviewed', _, ['2A','2C','6A'] ).
verb( 'revile', 'reviles', 'reviling', 'reviled', 'reviled', _, ['3A','6A'] ).
verb( 'revise', 'revises', 'revising', 'revised', 'revised', tran, ['6A'] ).
verb( 'revisit', 'revisits', 'revisiting', 'revisited', 'revisited', tran, ['2C','3A','6A','14'] ).
verb( 'revitalize', 'revitalizes', 'revitalizing', 'revitalized', 'revitalized', tran, [] ).
verb( 'revive', 'revives', 'reviving', 'revived', 'revived', _, ['2A','6A'] ).
verb( 'revivify', 'revivifies', 'revivifying', 'revivified', 'revivified', tran, [] ).
verb( 'revoke', 'revokes', 'revoking', 'revoked', 'revoked', _, ['2A','6A'] ).
verb( 'revolt', 'revolts', 'revolting', 'revolted', 'revolted', _, ['2A','3A','6A'] ).
verb( 'revolutionize', 'revolutionizes', 'revolutionizing', 'revolutionized', 'revolutionized', tran, ['6A'] ).
verb( 'revolve', 'revolves', 'revolving', 'revolved', 'revolved', _, ['2A','3A','6A'] ).
verb( 'reward', 'rewards', 'rewarding', 'rewarded', 'rewarded', tran, ['6A','14'] ).
verb( 'rewire', 'rewires', 'rewiring', 'rewired', 'rewired', tran, [] ).
verb( 'reword', 'rewords', 'rewording', 'reworded', 'reworded', tran, ['6A'] ).
verb( 'rewrite', 'rewrites', 'rewriting', 'rewrote', 'rewritten', tran, ['6A'] ).
verb( 'rhapsodize', 'rhapsodizes', 'rhapsodizing', 'rhapsodized', 'rhapsodized', intran, ['2A','3A'] ).
verb( 'rhyme', 'rhymes', 'rhyming', 'rhymed', 'rhymed', _, ['2A','3A','6A'] ).
verb( 'rib', 'ribs', 'ribbing', 'ribbed', 'ribbed', tran, ['6A'] ).
verb( 'rick', 'ricks', 'ricking', 'ricked', 'ricked', tran, ['6A'] ).
verb( 'ricochet', 'ricochets', 'ricocheting', 'ricocheted', 'ricocheted', _, ['2A','6A'] ).
verb( 'rid', 'rids', 'riding', 'rid', 'rid', tran, ['14'] ).
verb( 'riddle', 'riddles', 'riddling', 'riddled', 'riddled', tran, ['6A','14'] ).
verb( 'ride', 'rides', 'riding', 'rode', 'ridden', _, ['2A','2B','2C','2D','3A','4A','6A','15A'] ).
verb( 'ridge', 'ridges', 'ridging', 'ridged', 'ridged', tran, ['6A'] ).
verb( 'ridicule', 'ridicules', 'ridiculing', 'ridiculed', 'ridiculed', tran, [] ).
verb( 'riffle', 'riffles', 'riffling', 'riffled', 'riffled', _, ['3A','6A'] ).
verb( 'rifle', 'rifles', 'rifling', 'rifled', 'rifled', tran, ['6A'] ).
verb( 'rig', 'rigs', 'rigging', 'rigged', 'rigged', _, ['2A','2C','6A','14','15B'] ).
verb( 'right', 'rights', 'righting', 'righted', 'righted', tran, ['6A'] ).
verb( 'rile', 'riles', 'riling', 'riled', 'riled', tran, ['6A'] ).
verb( 'rim', 'rims', 'rimming', 'rimmed', 'rimmed', tran, ['6A'] ).
verb( 'rime', 'rimes', 'riming', 'rimed', 'rimed', tran, ['6A'] ).
verb( 'ring', 'rings', 'ringing', 'ringed', 'ringed', _, ['2A','2B','2C','2D','3A','6A','15A','15B'] ).
verb( 'ring', 'rings', 'ringing', 'rang', 'rung', _, ['2A','2B','2C','2D','3A','6A','15A','15B'] ).
verb( 'rinse', 'rinses', 'rinsing', 'rinsed', 'rinsed', tran, ['6A','15A','15B'] ).
verb( 'riot', 'riots', 'rioting', 'rioted', 'rioted', intran, ['2A','2B','2C','3A'] ).
verb( 'rip', 'rips', 'ripping', 'ripped', 'ripped', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'ripen', 'ripens', 'ripening', 'ripened', 'ripened', _, ['2A','6A'] ).
verb( 'riposte', 'ripostes', 'riposting', 'riposted', 'riposted', intran, [] ).
verb( 'ripple', 'ripples', 'rippling', 'rippled', 'rippled', _, ['2A','6A'] ).
verb( 'rise', 'rises', 'rising', 'rose', 'risen', intran, ['2A','2B','2C','3A','4A'] ).
verb( 'risk', 'risks', 'risking', 'risked', 'risked', tran, ['6A','6D'] ).
verb( 'rival', 'rivals', 'rivalling', 'rivalled', 'rivalled', tran, ['6A','14'] ).
verb( 'rive', 'rives', 'riving', 'rived', 'rived', _, ['6A','15A','15B'] ).
verb( 'rivet', 'rivets', 'riveting', 'riveted', 'riveted', tran, ['6A','15A','15B'] ).
verb( 'roam', 'roams', 'roaming', 'roamed', 'roamed', _, ['2A','2C','6A'] ).
verb( 'roar', 'roars', 'roaring', 'roared', 'roared', _, ['2A','2C','3A','6A','15B','22'] ).
verb( 'roast', 'roasts', 'roasting', 'roasted', 'roasted', _, ['2A','6A'] ).
verb( 'rob', 'robs', 'robbing', 'robbed', 'robbed', tran, ['6A','14'] ).
verb( 'robe', 'robes', 'robing', 'robed', 'robed', _, ['2A','6A','14'] ).
verb( 'rock', 'rocks', 'rocking', 'rocked', 'rocked', _, ['2A','6A','15A'] ).
verb( 'rocket', 'rockets', 'rocketing', 'rocketed', 'rocketed', intran, ['2A'] ).
verb( 'roll', 'rolls', 'rolling', 'rolled', 'rolled', _, ['2A','2B','2C','6A','12B','13B','14','15A','15B','22'] ).
verb( 'romance', 'romances', 'romancing', 'romanced', 'romanced', intran, ['2A'] ).
verb( 'romanticize', 'romanticizes', 'romanticizing', 'romanticized', 'romanticized', _, ['2A','6A'] ).
verb( 'romp', 'romps', 'romping', 'romped', 'romped', intran, ['2A','2C'] ).
verb( 'roof', 'roofs', 'roofing', 'roofed', 'roofed', tran, ['6A','15A','15B'] ).
verb( 'rook', 'rooks', 'rooking', 'rooked', 'rooked', tran, ['6A'] ).
verb( 'room', 'rooms', 'rooming', 'roomed', 'roomed', intran, ['2C'] ).
verb( 'roost', 'roosts', 'roosting', 'roosted', 'roosted', intran, ['2A'] ).
verb( 'root', 'roots', 'rooting', 'rooted', 'rooted', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'rootle', 'rootles', 'rootling', 'rootled', 'rootled', intran, ['2C'] ).
verb( 'rope', 'ropes', 'roping', 'roped', 'roped', tran, ['15A','15B'] ).
verb( 'rosin', 'rosins', 'rosinning', 'rosinned', 'rosinned', tran, ['6A'] ).
verb( 'rot', 'rots', 'rotting', 'rotted', 'rotted', _, ['2A','2C','6A'] ).
verb( 'rotate', 'rotates', 'rotating', 'rotated', 'rotated', _, ['2A','6A'] ).
verb( 'rouge', 'rouges', 'rouging', 'rouged', 'rouged', _, ['2A','6A'] ).
verb( 'rough', 'roughs', 'roughing', 'roughed', 'roughed', tran, ['6A','15B'] ).
verb( 'rough-dry', 'rough-dries', 'rough-drying', 'rough-dried', 'rough-dried', tran, [] ).
verb( 'rough-house', 'rough-houses', 'rough-housing', 'rough-housed', 'rough-housed', _, ['2A','6A'] ).
verb( 'roughcast', 'roughcasts', 'roughcasting', 'roughcast', 'roughcast', tran, [] ).
verb( 'roughen', 'roughens', 'roughening', 'roughened', 'roughened', _, [] ).
verb( 'round', 'rounds', 'rounding', 'rounded', 'rounded', _, ['2C','3A','6A','15B'] ).
verb( 'rouse', 'rouses', 'rousing', 'roused', 'roused', _, ['2A','6A','15A'] ).
verb( 'rout', 'routs', 'routing', 'routed', 'routed', tran, ['6A','15B'] ).
verb( 'route', 'routes', 'routing', 'routed', 'routed', tran, [] ).
verb( 'rove', 'roves', 'roving', 'roved', 'roved', _, ['2A','2C','6A'] ).
verb( 'row', 'rows', 'rowing', 'rowed', 'rowed', _, ['2A','2B','2C','6A','15A','15B'] ).
verb( 'row', 'rows', 'rowing', 'rowed', 'rowed', _, ['2A','3A','6A'] ).
verb( 'rub', 'rubs', 'rubbing', 'rubbed', 'rubbed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'rubber', 'rubbers', 'rubbering', 'rubbered', 'rubbered', tran, ['6A'] ).
verb( 'rubber-stamp', 'rubber-stamps', 'rubber-stamping', 'rubber-stamped', 'rubber-stamped', tran, [] ).
verb( 'rubberize', 'rubberizes', 'rubberizing', 'rubberized', 'rubberized', tran, ['6A'] ).
verb( 'rubberneck', 'rubbernecks', 'rubbernecking', 'rubbernecked', 'rubbernecked', intran, [] ).
verb( 'rubbish', 'rubbishes', 'rubbishing', 'rubbished', 'rubbished', tran, ['6A'] ).
verb( 'ruck', 'rucks', 'rucking', 'rucked', 'rucked', _, ['2A','2C','6A','15B'] ).
verb( 'ruddle', 'ruddles', 'ruddling', 'ruddled', 'ruddled', tran, [] ).
verb( 'rue', 'rues', 'ruing', 'rued', 'rued', tran, ['6A'] ).
verb( 'ruff', 'ruffs', 'ruffing', 'ruffed', 'ruffed', _, [] ).
verb( 'ruffle', 'ruffles', 'ruffling', 'ruffled', 'ruffled', _, ['2A','6A','15B'] ).
verb( 'ruin', 'ruins', 'ruining', 'ruined', 'ruined', tran, ['6A'] ).
verb( 'rule', 'rules', 'ruling', 'ruled', 'ruled', _, ['2A','2B','2C','3A','6A','9','15A','15B','25'] ).
verb( 'rumble', 'rumbles', 'rumbling', 'rumbled', 'rumbled', _, ['2A','2C','15B'] ).
verb( 'ruminate', 'ruminates', 'ruminating', 'ruminated', 'ruminated', intran, ['2A','2B','2C'] ).
verb( 'rummage', 'rummages', 'rummaging', 'rummaged', 'rummaged', _, ['2A','2B','2C','3A','6A','15B'] ).
verb( 'rumour', 'rumours', 'rumouring', 'rumoured', 'rumoured', tran, [] ).
verb( 'rumple', 'rumples', 'rumpling', 'rumpled', 'rumpled', tran, ['6A'] ).
verb( 'run', 'runs', 'running', 'ran', 'run', _, ['2A','2B','2C','2D','2E','3A','4A','6A','14','15A','15B'] ).
verb( 'rupture', 'ruptures', 'rupturing', 'ruptured', 'ruptured', _, ['2A','6A'] ).
verb( 'rush', 'rushes', 'rushing', 'rushed', 'rushed', _, ['2A','2C','4A','6A','15A','15B'] ).
verb( 'rust', 'rusts', 'rusting', 'rusted', 'rusted', _, ['2A','2C','6A'] ).
verb( 'rusticate', 'rusticates', 'rusticating', 'rusticated', 'rusticated', _, ['2A','6A'] ).
verb( 'rustle', 'rustles', 'rustling', 'rustled', 'rustled', _, ['2A','2C','6A','15B'] ).
verb( 'rut', 'ruts', 'rutting', 'rutted', 'rutted', tran, ['2A'] ).
verb( 'sabotage', 'sabotages', 'sabotaging', 'sabotaged', 'sabotaged', tran, ['6A'] ).
verb( 'sabre', 'sabres', 'sabring', 'sabred', 'sabred', tran, [] ).
verb( 'sack', 'sacks', 'sacking', 'sacked', 'sacked', tran, ['6A'] ).
verb( 'sacrifice', 'sacrifices', 'sacrificing', 'sacrificed', 'sacrificed', _, ['3A','6A','14','16A'] ).
verb( 'sadden', 'saddens', 'saddening', 'saddened', 'saddened', _, ['2A','6A'] ).
verb( 'saddle', 'saddles', 'saddling', 'saddled', 'saddled', tran, ['6A','14'] ).
verb( 'safeguard', 'safeguards', 'safeguarding', 'safeguarded', 'safeguarded', tran, ['6A','14'] ).
verb( 'sag', 'sags', 'sagging', 'sagged', 'sagged', intran, ['2A'] ).
verb( 'sail', 'sails', 'sailing', 'sailed', 'sailed', _, ['2A','2B','2C','3A','6A'] ).
verb( 'salaam', 'salaams', 'salaaming', 'salaamed', 'salaamed', intran, [] ).
verb( 'salivate', 'salivates', 'salivating', 'salivated', 'salivated', intran, [] ).
verb( 'sallow', 'sallows', 'sallowing', 'sallowed', 'sallowed', _, [] ).
verb( 'sally', 'sallies', 'sallying', 'sallied', 'sallied', intran, ['2A','2C'] ).
verb( 'salt', 'salts', 'salting', 'salted', 'salted', tran, ['6A','15B'] ).
verb( 'salute', 'salutes', 'saluting', 'saluted', 'saluted', _, ['2A','6A'] ).
verb( 'salvage', 'salvages', 'salvaging', 'salvaged', 'salvaged', tran, ['6A'] ).
verb( 'salve', 'salves', 'salving', 'salved', 'salved', tran, ['6A'] ).
verb( 'sample', 'samples', 'sampling', 'sampled', 'sampled', tran, ['6A'] ).
verb( 'sanctify', 'sanctifies', 'sanctifying', 'sanctified', 'sanctified', tran, ['6A'] ).
verb( 'sanction', 'sanctions', 'sanctioning', 'sanctioned', 'sanctioned', tran, ['6A'] ).
verb( 'sand', 'sands', 'sanding', 'sanded', 'sanded', tran, ['6A'] ).
verb( 'sandblast', 'sandblasts', 'sandblasting', 'sandblasted', 'sandblasted', tran, ['6A'] ).
verb( 'sandpaper', 'sandpapers', 'sandpapering', 'sandpapered', 'sandpapered', tran, ['6A'] ).
verb( 'sandwich', 'sandwiches', 'sandwiching', 'sandwiched', 'sandwiched', tran, ['6A','14'] ).
verb( 'sap', 'saps', 'sapping', 'sapped', 'sapped', _, ['2A','6A'] ).
verb( 'sate', 'sates', 'sating', 'sated', 'sated', tran, ['6A'] ).
verb( 'satiate', 'satiates', 'satiating', 'satiated', 'satiated', tran, ['6A'] ).
verb( 'satirize', 'satirizes', 'satirizing', 'satirized', 'satirized', tran, ['6A'] ).
verb( 'satisfy', 'satisfies', 'satisfying', 'satisfied', 'satisfied', _, ['2A','6A','11','14'] ).
verb( 'saturate', 'saturates', 'saturating', 'saturated', 'saturated', tran, ['6A','14'] ).
verb( 'sauce', 'sauces', 'saucing', 'sauced', 'sauced', tran, ['6A'] ).
verb( 'saunter', 'saunters', 'sauntering', 'sauntered', 'sauntered', intran, ['2A','2C'] ).
verb( 'saut_e', 'saut_es', 'saut_eing', 'saut_eed', 'saut_eed', tran, [] ).
verb( 'savage', 'savages', 'savaging', 'savaged', 'savaged', tran, ['6A'] ).
verb( 'save', 'saves', 'saving', 'saved', 'saved', _, ['2A','2C','3A','6A','6D','12B','12C','13B','14','15B'] ).
verb( 'savour', 'savours', 'savouring', 'savoured', 'savoured', _, ['3A','6A'] ).
verb( 'savvy', 'savvies', 'savvying', 'savvyed', 'savvyed', intran, [] ).
verb( 'saw', 'saws', 'sawing', 'sawed', 'sawed', _, ['2A','2B','2C','4A','6A','8','9','10','15A','15B','16B','18A','19A','22','24A'] ).
verb( 'say', 'says', 'saying', 'said', 'said', _, ['6A','9','10','14'] ).
verb( 'scald', 'scalds', 'scalding', 'scalded', 'scalded', tran, ['6A'] ).
verb( 'scale', 'scales', 'scaling', 'scaled', 'scaled', _, ['2B','2C','6A','15A','15B'] ).
verb( 'scallop', 'scallops', 'scalloping', 'scalloped', 'scalloped', tran, ['6A'] ).
verb( 'scalp', 'scalps', 'scalping', 'scalped', 'scalped', tran, ['6A'] ).
verb( 'scamp', 'scamps', 'scamping', 'scamped', 'scamped', tran, ['6A'] ).
verb( 'scamper', 'scampers', 'scampering', 'scampered', 'scampered', intran, ['2A','2C'] ).
verb( 'scan', 'scans', 'scanning', 'scanned', 'scanned', _, ['2A','6A'] ).
verb( 'scandalize', 'scandalizes', 'scandalizing', 'scandalized', 'scandalized', tran, ['6A'] ).
verb( 'scant', 'scants', 'scanting', 'scanted', 'scanted', tran, ['6A'] ).
verb( 'scar', 'scars', 'scarring', 'scarred', 'scarred', _, ['2C','6A'] ).
verb( 'scare', 'scares', 'scaring', 'scared', 'scared', _, ['2A','6A','15A','15B'] ).
verb( 'scarify', 'scarifies', 'scarifying', 'scarified', 'scarified', tran, ['6A'] ).
verb( 'scarper', 'scarpers', 'scarpering', 'scarpered', 'scarpered', intran, ['2A'] ).
verb( 'scatter', 'scatters', 'scattering', 'scattered', 'scattered', _, ['2A','2C','6A','15A','15B'] ).
verb( 'scavenge', 'scavenges', 'scavenging', 'scavenged', 'scavenged', intran, ['2A','3A'] ).
verb( 'scent', 'scents', 'scenting', 'scented', 'scented', tran, ['6A'] ).
verb( 'schedule', 'schedules', 'scheduling', 'scheduled', 'scheduled', tran, ['6A','7A','14'] ).
verb( 'scheme', 'schemes', 'scheming', 'schemed', 'schemed', _, ['2A','3A','4A','6A'] ).
verb( 'school', 'schools', 'schooling', 'schooled', 'schooled', tran, ['6A','15A'] ).
verb( 'scintillate', 'scintillates', 'scintillating', 'scintillated', 'scintillated', intran, ['2A'] ).
verb( 'scoff', 'scoffs', 'scoffing', 'scoffed', 'scoffed', _, ['2A','3A'] ).
verb( 'scold', 'scolds', 'scolding', 'scolded', 'scolded', _, ['2A','6A','14'] ).
verb( 'scollop', 'scollops', 'scolloping', 'scolloped', 'scolloped', tran, [] ).
verb( 'scoop', 'scoops', 'scooping', 'scooped', 'scooped', tran, ['6A','15B'] ).
verb( 'scoot', 'scoots', 'scooting', 'scooted', 'scooted', intran, [] ).
verb( 'scorch', 'scorches', 'scorching', 'scorched', 'scorched', _, ['2A','2C','6A'] ).
verb( 'score', 'scores', 'scoring', 'scored', 'scored', _, ['2A','3A','6A','15A','15B'] ).
verb( 'scorn', 'scorns', 'scorning', 'scorned', 'scorned', tran, ['6A','6D','7A'] ).
verb( 'scotch', 'scotches', 'scotching', 'scotched', 'scotched', tran, ['6A'] ).
verb( 'scour', 'scours', 'scouring', 'scoured', 'scoured', _, ['2C','6A','15A','15B'] ).
verb( 'scourge', 'scourges', 'scourging', 'scourged', 'scourged', tran, ['6A'] ).
verb( 'scout', 'scouts', 'scouting', 'scouted', 'scouted', _, ['2C','6A'] ).
verb( 'scowl', 'scowls', 'scowling', 'scowled', 'scowled', intran, ['2A','3A'] ).
verb( 'scrabble', 'scrabbles', 'scrabbling', 'scrabbled', 'scrabbled', intran, ['2A','2C'] ).
verb( 'scrag', 'scrags', 'scragging', 'scragged', 'scragged', tran, [] ).
verb( 'scram', 'scrams', 'scramming', 'scrammed', 'scrammed', intran, [] ).
verb( 'scramble', 'scrambles', 'scrambling', 'scrambled', 'scrambled', _, ['2A','2C','3A','4A','6A'] ).
verb( 'scrap', 'scraps', 'scrapping', 'scrapped', 'scrapped', tran, ['6A'] ).
verb( 'scrape', 'scrapes', 'scraping', 'scraped', 'scraped', _, ['2C','3A','6A','14','15A','15B','22'] ).
verb( 'scratch', 'scratches', 'scratching', 'scratched', 'scratched', _, ['2A','2C','6A','15B'] ).
verb( 'scrawl', 'scrawls', 'scrawling', 'scrawled', 'scrawled', _, ['2A','2C','6A'] ).
verb( 'scream', 'screams', 'screaming', 'screamed', 'screamed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'screech', 'screeches', 'screeching', 'screeched', 'screeched', _, ['2A','2C','6A','15A','15B'] ).
verb( 'screen', 'screens', 'screening', 'screened', 'screened', _, ['2C','6A','14','15A','15B'] ).
verb( 'screw', 'screws', 'screwing', 'screwed', 'screwed', _, ['2A','6A','15A','15B'] ).
verb( 'scribble', 'scribbles', 'scribbling', 'scribbled', 'scribbled', _, ['2A','6A'] ).
verb( 'scrimmage', 'scrimmages', 'scrimmaging', 'scrimmaged', 'scrimmaged', _, [] ).
verb( 'scrimp', 'scrimps', 'scrimping', 'scrimped', 'scrimped', _, [] ).
verb( 'scrimshank', 'scrimshanks', 'scrimshanking', 'scrimshanked', 'scrimshanked', intran, [] ).
verb( 'scrounge', 'scrounges', 'scrounging', 'scrounged', 'scrounged', _, ['2A','6A'] ).
verb( 'scrub', 'scrubs', 'scrubbing', 'scrubbed', 'scrubbed', _, ['2A','2C','6A','15B','22'] ).
verb( 'scrunch', 'scrunches', 'scrunching', 'scrunched', 'scrunched', tran, [] ).
verb( 'scruple', 'scruples', 'scrupling', 'scrupled', 'scrupled', intran, ['4C'] ).
verb( 'scrutinize', 'scrutinizes', 'scrutinizing', 'scrutinized', 'scrutinized', tran, ['6A'] ).
verb( 'scud', 'scuds', 'scudding', 'scudded', 'scudded', intran, ['2A','2C'] ).
verb( 'scuff', 'scuffs', 'scuffing', 'scuffed', 'scuffed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'scuffle', 'scuffles', 'scuffling', 'scuffled', 'scuffled', intran, ['2A','2C'] ).
verb( 'scull', 'sculls', 'sculling', 'sculled', 'sculled', _, ['2A','6A'] ).
verb( 'sculpt', 'sculpts', 'sculpting', 'sculpted', 'sculpted', _, [] ).
verb( 'sculpture', 'sculptures', 'sculpturing', 'sculptured', 'sculptured', _, ['2A','6A'] ).
verb( 'scupper', 'scuppers', 'scuppering', 'scuppered', 'scuppered', tran, [] ).
verb( 'scurry', 'scurries', 'scurrying', 'scurried', 'scurried', intran, ['2A','2C','3A'] ).
verb( 'scuttle', 'scuttles', 'scuttling', 'scuttled', 'scuttled', _, ['2A','2C','6A'] ).
verb( 'scythe', 'scythes', 'scything', 'scythed', 'scythed', tran, ['6A','15A'] ).
verb( 'seal', 'seals', 'sealing', 'sealed', 'sealed', _, ['2A','6A','15A','15B'] ).
verb( 'seam', 'seams', 'seaming', 'seamed', 'seamed', tran, [] ).
verb( 'sear', 'sears', 'searing', 'seared', 'seared', tran, ['6A'] ).
verb( 'search', 'searches', 'searching', 'searched', 'searched', _, ['2A','3A','6A','14','15A','15B'] ).
verb( 'season', 'seasons', 'seasoning', 'seasoned', 'seasoned', _, ['2A','6A','14'] ).
verb( 'seat', 'seats', 'seating', 'seated', 'seated', tran, ['6A'] ).
verb( 'secede', 'secedes', 'seceding', 'seceded', 'seceded', intran, ['2A','3A'] ).
verb( 'seclude', 'secludes', 'secluding', 'secluded', 'secluded', tran, ['6A','14','15A'] ).
verb( 'second', 'seconds', 'seconding', 'seconded', 'seconded', tran, ['6A'] ).
verb( 'second', 'seconds', 'seconding', 'seconded', 'seconded', tran, ['6A','15A'] ).
verb( 'secrete', 'secretes', 'secreting', 'secreted', 'secreted', tran, ['6A'] ).
verb( 'secularize', 'secularizes', 'secularizing', 'secularized', 'secularized', tran, ['6A'] ).
verb( 'secure', 'secures', 'securing', 'secured', 'secured', tran, ['6A','12B','13B','14'] ).
verb( 'sedate', 'sedates', 'sedating', 'sedated', 'sedated', tran, ['6A'] ).
verb( 'seduce', 'seduces', 'seducing', 'seduced', 'seduced', tran, ['6A','14'] ).
verb( 'see', 'sees', 'seeing', 'saw', 'seen', _, ['2A','2B','2C','4A','6A','8','9','10','15A','16B','18A','19A','22','24A'] ).
verb( 'seed', 'seeds', 'seeding', 'seeded', 'seeded', _, ['2A','6A'] ).
verb( 'seek', 'seeks', 'seeking', 'sought', 'sought', tran, ['2A','3A','6A','7A','15A'] ).
verb( 'seem', 'seems', 'seeming', 'seemed', 'seemed', intran, ['2A','4D','4E'] ).
verb( 'seep', 'seeps', 'seeping', 'seeped', 'seeped', intran, ['2C'] ).
verb( 'seesaw', 'seesaws', 'seesawing', 'seesawed', 'seesawed', intran, [] ).
verb( 'seethe', 'seethes', 'seething', 'seethed', 'seethed', _, ['2A','3A','6A'] ).
verb( 'segment', 'segments', 'segmenting', 'segmented', 'segmented', _, [] ).
verb( 'segregate', 'segregates', 'segregating', 'segregated', 'segregated', tran, ['6A'] ).
verb( 'seine', 'seines', 'seining', 'seined', 'seined', _, [] ).
verb( 'seize', 'seizes', 'seizing', 'seized', 'seized', _, ['2A','2C','3A','6A','15A'] ).
verb( 'select', 'selects', 'selecting', 'selected', 'selected', tran, ['6A','15A','16A'] ).
verb( 'sell', 'sells', 'selling', 'sold', 'sold', _, ['2A','2C','6A','12A','13A','15B'] ).
verb( 'semaphore', 'semaphores', 'semaphoring', 'semaphored', 'semaphored', _, ['2A','6A'] ).
verb( 'send', 'sends', 'sending', 'sent', 'sent', _, ['2C','3A','6A','12A','13A','15A','15B','19B','22'] ).
verb( 'sense', 'senses', 'sensing', 'sensed', 'sensed', tran, ['6A','9'] ).
verb( 'sensitize', 'sensitizes', 'sensitizing', 'sensitized', 'sensitized', tran, ['6A'] ).
verb( 'sentence', 'sentences', 'sentencing', 'sentenced', 'sentenced', tran, ['6A','14','17'] ).
verb( 'sentimentalize', 'sentimentalizes', 'sentimentalizing', 'sentimentalized', 'sentimentalized', _, ['2A','6A'] ).
verb( 'separate', 'separates', 'separating', 'separated', 'separated', _, ['2A','6A','14','15B'] ).
verb( 'sequester', 'sequesters', 'sequestering', 'sequestered', 'sequestered', tran, ['6A'] ).
verb( 'sequestrate', 'sequestrates', 'sequestrating', 'sequestrated', 'sequestrated', tran, ['6A'] ).
verb( 'serenade', 'serenades', 'serenading', 'serenaded', 'serenaded', tran, ['6A'] ).
verb( 'serialize', 'serializes', 'serializing', 'serialized', 'serialized', tran, ['6A'] ).
verb( 'sermonize', 'sermonizes', 'sermonizing', 'sermonized', 'sermonized', _, ['2A','6A'] ).
verb( 'serve', 'serves', 'serving', 'served', 'served', _, ['2A','2B','2C','3A','4A','6A','14','15A','15B'] ).
verb( 'service', 'services', 'servicing', 'serviced', 'serviced', tran, ['6A'] ).
verb( 'set', 'sets', 'setting', 'set', 'set', _, ['2A','2C','3A','6A','12C','14','15A','16A','17','19B','22'] ).
verb( 'settle', 'settles', 'settling', 'settled', 'settled', _, ['2A','2C','3A','6A','7A','8','10','14','15A','15B'] ).
verb( 'sever', 'severs', 'severing', 'severed', 'severed', _, ['2A','6A','15A'] ).
verb( 'sew', 'sews', 'sewing', 'sewed', 'sewn', _, ['2A','2B','2C','6A','15B'] ).
verb( 'sex', 'sexes', 'sexing', 'sexed', 'sexed', tran, ['6A'] ).
verb( 'shack', 'shacks', 'shacking', 'shacked', 'shacked', intran, ['2C'] ).
verb( 'shackle', 'shackles', 'shackling', 'shackled', 'shackled', tran, ['6A'] ).
verb( 'shade', 'shades', 'shading', 'shaded', 'shaded', _, ['2C','6A','15A'] ).
verb( 'shadow', 'shadows', 'shadowing', 'shadowed', 'shadowed', tran, ['6A'] ).
verb( 'shag', 'shags', 'shagging', 'shagged', 'shagged', _, ['6A'] ).
verb( 'shake', 'shakes', 'shaking', 'shook', 'shaken', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'shall', 'shall', '-', '-', '-', unknown, [] ).
verb( 'shallow', 'shallows', 'shallowing', 'shallowed', 'shallowed', intran, [] ).
verb( 'sham', 'shams', 'shamming', 'shammed', 'shammed', _, ['2A','2D','6A'] ).
verb( 'shamble', 'shambles', 'shambling', 'shambled', 'shambled', intran, ['2A','2C'] ).
verb( 'shame', 'shames', 'shaming', 'shamed', 'shamed', tran, ['6A','14'] ).
verb( 'shampoo', 'shampoos', 'shampooing', 'shampooed', 'shampooed', tran, ['6A'] ).
verb( 'shanghai', 'shanghais', 'shanghaiing', 'shanghaied', 'shanghaied', tran, [] ).
verb( 'shape', 'shapes', 'shaping', 'shaped', 'shaped', _, ['2A','2C','6A','15A'] ).
verb( 'share', 'shares', 'sharing', 'shared', 'shared', _, ['3A','6A','14','15B'] ).
verb( 'sharpen', 'sharpens', 'sharpening', 'sharpened', 'sharpened', _, ['2A','6A'] ).
verb( 'shatter', 'shatters', 'shattering', 'shattered', 'shattered', _, ['2A','6A'] ).
verb( 'shave', 'shaves', 'shaving', 'shaved', 'shaved', _, ['2A','6A','15A','15B'] ).
verb( 'shear', 'shears', 'shearing', 'sheared', 'sheared', tran, ['6A'] ).
verb( 'sheathe', 'sheathes', 'sheathing', 'sheathed', 'sheathed', tran, ['6A'] ).
verb( 'shed', 'sheds', 'shedding', 'shed', 'shed', tran, ['6A'] ).
verb( 'sheer', 'sheers', 'sheering', 'sheered', 'sheered', intran, ['2C'] ).
verb( 'shell', 'shells', 'shelling', 'shelled', 'shelled', _, ['2C','6A','15B'] ).
verb( 'shellac', 'shellacs', 'shellacking', 'shellacked', 'shellacked', tran, [] ).
verb( 'shelter', 'shelters', 'sheltering', 'sheltered', 'sheltered', _, ['2A','2C','6A','14'] ).
verb( 'shelve', 'shelves', 'shelving', 'shelved', 'shelved', _, ['2A','2C','6A'] ).
verb( 'shepherd', 'shepherds', 'shepherding', 'shepherded', 'shepherded', tran, ['6A','15A'] ).
verb( 'shew', 'shews', 'shewing', 'shewed', 'shewed', _, ['2A','2C','6A','9','10','12A','13A','14','15A','15B','19B','20','21','24A','25'] ).
verb( 'shield', 'shields', 'shielding', 'shielded', 'shielded', tran, ['6A','15A'] ).
verb( 'shift', 'shifts', 'shifting', 'shifted', 'shifted', _, ['2A','2C','6A','14','15A'] ).
verb( 'shillyshally', 'shillyshallies', 'shillyshallying', 'shillyshallied', 'shillyshallied', intran, ['2A'] ).
verb( 'shimmer', 'shimmers', 'shimmering', 'shimmered', 'shimmered', intran, ['2A','2C'] ).
verb( 'shin', 'shins', 'shinning', 'shinned', 'shinned', intran, ['3A'] ).
verb( 'shine', 'shines', 'shining', 'shone', 'shone', _, ['2A','2C','6A'] ).
verb( 'shingle', 'shingles', 'shingling', 'shingled', 'shingled', tran, ['6A'] ).
verb( 'ship', 'ships', 'shipping', 'shipped', 'shipped', _, ['2C','6A','15A','15B'] ).
verb( 'shipwreck', 'shipwrecks', 'shipwrecking', 'shipwrecked', 'shipwrecked', intran, [] ).
verb( 'shirk', 'shirks', 'shirking', 'shirked', 'shirked', _, ['2A','6A','6D'] ).
verb( 'shit', 'shits', 'shitting', 'shitted', 'shitted', intran, ['2A','3A'] ).
verb( 'shiver', 'shivers', 'shivering', 'shivered', 'shivered', _, ['2A','2C','15A'] ).
verb( 'shoal', 'shoals', 'shoaling', 'shoaled', 'shoaled', intran, ['2A'] ).
verb( 'shock', 'shocks', 'shocking', 'shocked', 'shocked', tran, ['6A'] ).
verb( 'shoe', 'shoes', 'shoeing', 'shod', 'shod', tran, ['6A'] ).
verb( 'shoo', 'shoos', 'shooing', 'shooed', 'shooed', _, ['15B'] ).
verb( 'shoot', 'shoots', 'shooting', 'shot', 'shot', _, ['2A','2C','4A','6A','15A','15B'] ).
verb( 'shop', 'shops', 'shopping', 'shopped', 'shopped', intran, ['2A','2C'] ).
verb( 'shoplift', 'shoplifts', 'shoplifting', 'shoplifted', 'shoplifted', _, ['2A','6A'] ).
verb( 'shore', 'shores', 'shoring', 'shored', 'shored', tran, ['15B'] ).
verb( 'short', 'shorts', 'shorting', 'shorted', 'shorted', _, [] ).
verb( 'short-change', 'short-changes', 'short-changing', 'short-changed', 'short-changed', tran, [] ).
verb( 'short-circuit', 'short-circuits', 'short-circuiting', 'short-circuited', 'short-circuited', _, [] ).
verb( 'shorten', 'shortens', 'shortening', 'shortened', 'shortened', _, ['2A','6A'] ).
verb( 'shortlist', 'shortlists', 'shortlisting', 'shortlisted', 'shortlisted', tran, [] ).
verb( 'should', 'should', '-', '-', '-', unknown, [] ).
verb( 'shoulder', 'shoulders', 'shouldering', 'shouldered', 'shouldered', tran, ['6A','15A'] ).
verb( 'shout', 'shouts', 'shouting', 'shouted', 'shouted', _, ['2A','2B','2C','3A','4B','6A','15A','15B','22'] ).
verb( 'shove', 'shoves', 'shoving', 'shoved', 'shoved', _, ['2A','2C','6A','15A','15B'] ).
verb( 'shovel', 'shovels', 'shovelling', 'shovelled', 'shovelled', tran, ['6A','15A','15B'] ).
verb( 'show', 'shows', 'showing', 'showed', 'showed', _, ['2A','2C','6A','9','10','12A','13A','14','15A','15B','19B','20','21','24A','25'] ).
verb( 'shower', 'showers', 'showering', 'showered', 'showered', _, ['2C','14'] ).
verb( 'shred', 'shreds', 'shredding', 'shredded', 'shredded', tran, ['6A'] ).
verb( 'shriek', 'shrieks', 'shrieking', 'shrieked', 'shrieked', _, ['2A','2C','6A','15B'] ).
verb( 'shrill', 'shrills', 'shrilling', 'shrilled', 'shrilled', _, [] ).
verb( 'shrimp', 'shrimps', 'shrimping', 'shrimped', 'shrimped', intran, [] ).
verb( 'shrine', 'shrines', 'shrining', 'shrined', 'shrined', tran, [] ).
verb( 'shrink', 'shrinks', 'shrinking', 'shrank', 'shrunk', _, ['2A','2C','3A','6A'] ).
verb( 'shrive', 'shrives', 'shriving', 'shrived', 'shrived', tran, ['6A'] ).
verb( 'shrivel', 'shrivels', 'shrivelling', 'shrivelled', 'shrivelled', _, ['2A','2C','6A','15B'] ).
verb( 'shroud', 'shrouds', 'shrouding', 'shrouded', 'shrouded', tran, ['6A','15A'] ).
verb( 'shrug', 'shrugs', 'shrugging', 'shrugged', 'shrugged', tran, ['6A','15B'] ).
verb( 'shuck', 'shucks', 'shucking', 'shucked', 'shucked', tran, ['6A'] ).
verb( 'shudder', 'shudders', 'shuddering', 'shuddered', 'shuddered', intran, ['2A','2C','4C'] ).
verb( 'shuffle', 'shuffles', 'shuffling', 'shuffled', 'shuffled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'shun', 'shuns', 'shunning', 'shunned', 'shunned', tran, ['6A','6D'] ).
verb( 'shunt', 'shunts', 'shunting', 'shunted', 'shunted', _, ['2A','6A','15A','15B'] ).
verb( 'shush', 'shushes', 'shushing', 'shushed', 'shushed', _, [] ).
verb( 'shut', 'shuts', 'shutting', 'shut', 'shut', _, ['2A','2C','6A','15A','15B'] ).
verb( 'shutter', 'shutters', 'shuttering', 'shuttered', 'shuttered', tran, ['6A'] ).
verb( 'shuttle', 'shuttles', 'shuttling', 'shuttled', 'shuttled', _, ['2C','6A','15A'] ).
verb( 'shy', 'shies', 'shying', 'shied', 'shied', _, ['2A','3A','6A','15A'] ).
verb( 'sick', 'sicks', 'sicking', 'sicked', 'sicked', tran, [] ).
verb( 'sicken', 'sickens', 'sickening', 'sickened', 'sickened', _, ['2A','3A','4C','6A'] ).
verb( 'side', 'sides', 'siding', 'sided', 'sided', intran, ['3A'] ).
verb( 'side-slip', 'side-slips', 'side-slipping', 'side-slipped', 'side-slipped', intran, [] ).
verb( 'sidestep', 'sidesteps', 'sidestepping', 'sidestepped', 'sidestepped', _, ['2A','6A'] ).
verb( 'sidetrack', 'sidetracks', 'sidetracking', 'sidetracked', 'sidetracked', tran, ['6A'] ).
verb( 'sidle', 'sidles', 'sidling', 'sidled', 'sidled', intran, ['2C'] ).
verb( 'sieve', 'sieves', 'sieving', 'sieved', 'sieved', tran, ['6A'] ).
verb( 'sift', 'sifts', 'sifting', 'sifted', 'sifted', _, ['2C','6A','14','15A','15B'] ).
verb( 'sigh', 'sighs', 'sighing', 'sighed', 'sighed', _, ['2A','3A','6A','15B'] ).
verb( 'sight', 'sights', 'sighting', 'sighted', 'sighted', tran, ['6A'] ).
verb( 'sign', 'signs', 'signing', 'signed', 'signed', _, ['2A','2C','3A','6A','15B'] ).
verb( 'signal', 'signals', 'signalling', 'signalled', 'signalled', _, ['2A','2C','6A','9','17'] ).
verb( 'signalize', 'signalizes', 'signalizing', 'signalized', 'signalized', tran, ['6A'] ).
verb( 'signify', 'signifies', 'signifying', 'signified', 'signified', _, ['2A','2C','6A','9','15A'] ).
verb( 'signpost', 'signposts', 'signposting', 'signposted', 'signposted', tran, [] ).
verb( 'silence', 'silences', 'silencing', 'silenced', 'silenced', tran, ['6A'] ).
verb( 'silhouette', 'silhouettes', 'silhouetting', 'silhouetted', 'silhouetted', tran, [] ).
verb( 'silt', 'silts', 'silting', 'silted', 'silted', _, ['2C','15B'] ).
verb( 'silver', 'silvers', 'silvering', 'silvered', 'silvered', _, ['2A','6A'] ).
verb( 'simmer', 'simmers', 'simmering', 'simmered', 'simmered', _, ['2A','2B','2C','6A','15A'] ).
verb( 'simper', 'simpers', 'simpering', 'simpered', 'simpered', intran, ['2A'] ).
verb( 'simplify', 'simplifies', 'simplifying', 'simplified', 'simplified', tran, ['6A'] ).
verb( 'simulate', 'simulates', 'simulating', 'simulated', 'simulated', tran, ['6A'] ).
verb( 'sin', 'sins', 'sinning', 'sinned', 'sinned', intran, ['2A','3A'] ).
verb( 'sing', 'sings', 'singing', 'sang', 'sung', _, ['2A','2C','3A','6A','12B','13B','15A'] ).
verb( 'singe', 'singes', 'singeing', 'singed', 'singed', _, ['2A','6A'] ).
verb( 'single', 'singles', 'singling', 'singled', 'singled', tran, ['15B'] ).
verb( 'singularize', 'singularizes', 'singularizing', 'singularized', 'singularized', tran, [] ).
verb( 'sink', 'sinks', 'sinking', 'sank', 'sunk', _, ['2A','2C','3A','6A','14','15A'] ).
verb( 'sip', 'sips', 'sipping', 'sipped', 'sipped', _, ['2A','6A','15B'] ).
verb( 'siphon', 'siphons', 'siphoning', 'siphoned', 'siphoned', _, ['15B'] ).
verb( 'sire', 'sires', 'siring', 'sired', 'sired', tran, ['6A'] ).
verb( 'sit', 'sits', 'siting', 'sat', 'sat', _, ['2A','2C','3A','6A','15B'] ).
verb( 'site', 'sites', 'siting', 'sited', 'sited', intran, ['6A'] ).
verb( 'size', 'sizes', 'sizing', 'sized', 'sized', tran, ['6A','15B'] ).
verb( 'sizzle', 'sizzles', 'sizzling', 'sizzled', 'sizzled', intran, ['6A'] ).
verb( 'skate', 'skates', 'skating', 'skated', 'skated', intran, ['2A','2C'] ).
verb( 'skedaddle', 'skedaddles', 'skedaddling', 'skedaddled', 'skedaddled', intran, ['2A'] ).
verb( 'sketch', 'sketches', 'sketching', 'sketched', 'sketched', _, ['2A','6A','15B'] ).
verb( 'skewer', 'skewers', 'skewering', 'skewered', 'skewered', tran, ['6A'] ).
verb( 'ski', 'skis', 'skiing', 'ski\'d', 'ski\'d', intran, [] ).
verb( 'skid', 'skids', 'skidding', 'skidded', 'skidded', intran, ['2A'] ).
verb( 'skim', 'skims', 'skimming', 'skimmed', 'skimmed', _, ['2C','3A','6A','14','15A','15B'] ).
verb( 'skimp', 'skimps', 'skimping', 'skimped', 'skimped', _, ['2A','6A'] ).
verb( 'skin', 'skins', 'skinning', 'skinned', 'skinned', _, ['2C','6A','14'] ).
verb( 'skip', 'skips', 'skipping', 'skipped', 'skipped', _, ['2A','2C','6A'] ).
verb( 'skipper', 'skippers', 'skippering', 'skippered', 'skippered', tran, [] ).
verb( 'skirmish', 'skirmishes', 'skirmishing', 'skirmished', 'skirmished', intran, ['2A','2C'] ).
verb( 'skirt', 'skirts', 'skirting', 'skirted', 'skirted', _, ['2C','6A'] ).
verb( 'skitter', 'skitters', 'skittering', 'skittered', 'skittered', intran, ['2A'] ).
verb( 'skittle', 'skittles', 'skittling', 'skittled', 'skittled', tran, ['15B'] ).
verb( 'skulk', 'skulks', 'skulking', 'skulked', 'skulked', intran, ['2A','2C'] ).
verb( 'sky', 'skies', 'skying', 'skied', 'skied', tran, [] ).
verb( 'skylark', 'skylarks', 'skylarking', 'skylarked', 'skylarked', intran, [] ).
verb( 'skyrocket', 'skyrockets', 'skyrocketing', 'skyrocketed', 'skyrocketed', intran, [] ).
verb( 'slack', 'slacks', 'slacking', 'slacked', 'slacked', intran, ['2A','2C'] ).
verb( 'slacken', 'slackens', 'slackening', 'slackened', 'slackened', _, ['2A','6A'] ).
verb( 'slake', 'slakes', 'slaking', 'slaked', 'slaked', tran, ['6A'] ).
verb( 'slam', 'slams', 'slamming', 'slammed', 'slammed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'slander', 'slanders', 'slandering', 'slandered', 'slandered', tran, ['6A'] ).
verb( 'slang', 'slangs', 'slanging', 'slanged', 'slanged', tran, ['6A'] ).
verb( 'slant', 'slants', 'slanting', 'slanted', 'slanted', _, ['2A','2C','6A'] ).
verb( 'slap', 'slaps', 'slapping', 'slapped', 'slapped', tran, ['6A','15A','15B'] ).
verb( 'slash', 'slashes', 'slashing', 'slashed', 'slashed', _, ['2C','6A'] ).
verb( 'slate', 'slates', 'slating', 'slated', 'slated', tran, ['6A'] ).
verb( 'slaughter', 'slaughters', 'slaughtering', 'slaughtered', 'slaughtered', tran, ['6A'] ).
verb( 'slave', 'slaves', 'slaving', 'slaved', 'slaved', intran, ['2A','2B','2C','3A'] ).
verb( 'slaver', 'slavers', 'slavering', 'slavered', 'slavered', intran, ['2A','2C'] ).
verb( 'slay', 'slays', 'slaying', 'slew', 'slain', tran, ['6A'] ).
verb( 'sledge', 'sledges', 'sledging', 'sledged', 'sledged', _, [] ).
verb( 'sleek', 'sleeks', 'sleeking', 'sleeked', 'sleeked', tran, ['6A'] ).
verb( 'sleep', 'sleeps', 'sleeping', 'slept', 'slept', _, ['2A','2B','2C','6A'] ).
verb( 'sleet', 'sleets', 'sleeting', 'sleeted', 'sleeted', intran, ['2A'] ).
verb( 'sleigh', 'sleighs', 'sleighing', 'sleighed', 'sleighed', _, ['2A','6A'] ).
verb( 'slenderize', 'slenderizes', 'slenderizing', 'slenderized', 'slenderized', _, ['6A'] ).
verb( 'slew', 'slews', 'slewing', 'slewed', 'slewed', _, ['2C','6A','15B'] ).
verb( 'slice', 'slices', 'slicing', 'sliced', 'sliced', _, ['3A','6A','15A','15B','22'] ).
verb( 'slide', 'slides', 'sliding', 'slid', 'slid', _, ['2A','2C','3A','6A','15A'] ).
verb( 'slight', 'slights', 'slighting', 'slighted', 'slighted', tran, ['6A'] ).
verb( 'slim', 'slims', 'slimming', 'slimmed', 'slimmed', intran, [] ).
verb( 'sling', 'slings', 'slinging', 'slung', 'slung', _, ['6A','15A','15B'] ).
verb( 'slink', 'slinks', 'slinking', 'slunk', 'slunk', intran, ['2C'] ).
verb( 'slip', 'slips', 'slipping', 'slipped', 'slipped', _, ['2A','2C','6A','15A','15B'] ).
verb( 'slit', 'slits', 'slitting', 'slit', 'slit', tran, ['2A','2C','6A','15A','22'] ).
verb( 'slither', 'slithers', 'slithering', 'slithered', 'slithered', intran, ['2A','2C'] ).
verb( 'sliver', 'slivers', 'slivering', 'slivered', 'slivered', _, ['2A','6A'] ).
verb( 'slobber', 'slobbers', 'slobbering', 'slobbered', 'slobbered', _, ['2A','3A','6A'] ).
verb( 'slog', 'slogs', 'slogging', 'slogged', 'slogged', _, ['2C','3A','6A'] ).
verb( 'slop', 'slops', 'slopping', 'slopped', 'slopped', _, ['2A','2C','6A','15A'] ).
verb( 'slope', 'slopes', 'sloping', 'sloped', 'sloped', _, ['2A','2C','6A'] ).
verb( 'slosh', 'sloshes', 'sloshing', 'sloshed', 'sloshed', _, ['3A','6A','15A','15B'] ).
verb( 'slot', 'slots', 'slotting', 'slotted', 'slotted', tran, ['6A','15A'] ).
verb( 'slouch', 'slouches', 'slouching', 'slouched', 'slouched', intran, ['2A','2C'] ).
verb( 'slough', 'sloughs', 'sloughing', 'sloughed', 'sloughed', _, ['2A','6A','15B'] ).
verb( 'slow', 'slows', 'slowing', 'slowed', 'slowed', _, ['2C','15B'] ).
verb( 'slug', 'slugs', 'slugging', 'slugged', 'slugged', _, [] ).
verb( 'sluice', 'sluices', 'sluicing', 'sluiced', 'sluiced', _, ['3A','6A','15B'] ).
verb( 'slum', 'slums', 'slumming', 'slummed', 'slummed', intran, [] ).
verb( 'slumber', 'slumbers', 'slumbering', 'slumbered', 'slumbered', _, ['2A','15B'] ).
verb( 'slump', 'slumps', 'slumping', 'slumped', 'slumped', intran, ['2A','2C'] ).
verb( 'slur', 'slurs', 'slurring', 'slurred', 'slurred', _, ['3A','6A'] ).
verb( 'smack', 'smacks', 'smacking', 'smacked', 'smacked', _, ['3A','6A'] ).
verb( 'smart', 'smarts', 'smarting', 'smarted', 'smarted', intran, ['2A','2C','3A'] ).
verb( 'smarten', 'smartens', 'smartening', 'smartened', 'smartened', _, ['2C','6A','15B'] ).
verb( 'smash', 'smashes', 'smashing', 'smashed', 'smashed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'smear', 'smears', 'smearing', 'smeared', 'smeared', _, ['2A','6A','14'] ).
verb( 'smell', 'smells', 'smelling', 'smelled', 'smelled', _, ['2A','2C','2D','3A','6A','15B','19A'] ).
verb( 'smelt', 'smelts', 'smelting', 'smelted', 'smelted', tran, ['2A','2C','2D','3A','6A','15B','19A'] ).
verb( 'smile', 'smiles', 'smiling', 'smiled', 'smiled', _, ['2A','2B','3A','4B','6B'] ).
verb( 'smirch', 'smirches', 'smirching', 'smirched', 'smirched', tran, ['6A'] ).
verb( 'smirk', 'smirks', 'smirking', 'smirked', 'smirked', intran, ['2A'] ).
verb( 'smite', 'smites', 'smiting', 'smote', 'smitten', _, ['2A','2C','6A'] ).
verb( 'smoke', 'smokes', 'smoking', 'smoked', 'smoked', _, ['2A','2C','6A','15B','22'] ).
verb( 'smooth', 'smooths', 'smoothing', 'smoothed', 'smoothed', _, ['2C','6A','15B'] ).
verb( 'smother', 'smothers', 'smothering', 'smothered', 'smothered', tran, ['6A','14','15B'] ).
verb( 'smoulder', 'smoulders', 'smouldering', 'smouldered', 'smouldered', intran, ['2A','2C'] ).
verb( 'smudge', 'smudges', 'smudging', 'smudged', 'smudged', _, ['2A','6A'] ).
verb( 'smuggle', 'smuggles', 'smuggling', 'smuggled', 'smuggled', tran, ['6A','14','15A','15B'] ).
verb( 'smut', 'smuts', 'smutting', 'smutted', 'smutted', tran, ['6A'] ).
verb( 'snaffle', 'snaffles', 'snaffling', 'snaffled', 'snaffled', tran, ['6A'] ).
verb( 'snag', 'snags', 'snagging', 'snagged', 'snagged', tran, [] ).
verb( 'snake', 'snakes', 'snaking', 'snaked', 'snaked', intran, ['2C'] ).
verb( 'snap', 'snaps', 'snapping', 'snapped', 'snapped', _, ['2A','2C','3A','6A','14','15A','15B','22'] ).
verb( 'snare', 'snares', 'snaring', 'snared', 'snared', tran, ['6A'] ).
verb( 'snarl', 'snarls', 'snarling', 'snarled', 'snarled', _, ['2A','2C','3A','6A','15B'] ).
verb( 'snatch', 'snatches', 'snatching', 'snatched', 'snatched', _, ['2A','6A','15A','15B'] ).
verb( 'sneak', 'sneaks', 'sneaking', 'sneaked', 'sneaked', _, ['2A','3A','6A'] ).
verb( 'sneer', 'sneers', 'sneering', 'sneered', 'sneered', intran, ['2A','3A'] ).
verb( 'sneeze', 'sneezes', 'sneezing', 'sneezed', 'sneezed', intran, ['2A'] ).
verb( 'snick', 'snicks', 'snicking', 'snicked', 'snicked', _, [] ).
verb( 'snicker', 'snickers', 'snickering', 'snickered', 'snickered', intran, [] ).
verb( 'sniff', 'sniffs', 'sniffing', 'sniffed', 'sniffed', _, ['2A','3A','6A','15B'] ).
verb( 'sniffle', 'sniffles', 'sniffling', 'sniffled', 'sniffled', intran, [] ).
verb( 'snigger', 'sniggers', 'sniggering', 'sniggered', 'sniggered', intran, ['2A','2C'] ).
verb( 'snip', 'snips', 'snipping', 'snipped', 'snipped', _, ['3A','6A','15B'] ).
verb( 'snipe', 'snipes', 'sniping', 'sniped', 'sniped', _, ['2A','3A','6A'] ).
verb( 'snitch', 'snitches', 'snitching', 'snitched', 'snitched', _, ['2A','3A','6A'] ).
verb( 'snivel', 'snivels', 'snivelling', 'snivelled', 'snivelled', intran, ['2A'] ).
verb( 'snog', 'snogs', 'snogging', 'snogged', 'snogged', intran, ['2A'] ).
verb( 'snoop', 'snoops', 'snooping', 'snooped', 'snooped', intran, ['2A','2C','3A'] ).
verb( 'snooze', 'snoozes', 'snoozing', 'snoozed', 'snoozed', intran, ['2A'] ).
verb( 'snore', 'snores', 'snoring', 'snored', 'snored', intran, ['2A','2C'] ).
verb( 'snort', 'snorts', 'snorting', 'snorted', 'snorted', _, ['2A','2C','6A','15B'] ).
verb( 'snow', 'snows', 'snowing', 'snowed', 'snowed', _, ['2A','2B','2C'] ).
verb( 'snowball', 'snowballs', 'snowballing', 'snowballed', 'snowballed', _, [] ).
verb( 'snub', 'snubs', 'snubbing', 'snubbed', 'snubbed', tran, ['6A'] ).
verb( 'snuff', 'snuffs', 'snuffing', 'snuffed', 'snuffed', _, ['2C','6A','15B'] ).
verb( 'snuffle', 'snuffles', 'snuffling', 'snuffled', 'snuffled', intran, ['2A'] ).
verb( 'snuggle', 'snuggles', 'snuggling', 'snuggled', 'snuggled', _, ['2C','14'] ).
verb( 'soak', 'soaks', 'soaking', 'soaked', 'soaked', _, ['2A','3A','6A','14','15A','15B'] ).
verb( 'soap', 'soaps', 'soaping', 'soaped', 'soaped', tran, ['6A','15B'] ).
verb( 'soar', 'soars', 'soaring', 'soared', 'soared', intran, ['2A','2C'] ).
verb( 'sob', 'sobs', 'sobbing', 'sobbed', 'sobbed', _, ['2A','2C','15A','15B'] ).
verb( 'sober', 'sobers', 'sobering', 'sobered', 'sobered', _, ['2C','6A','15B'] ).
verb( 'socialize', 'socializes', 'socializing', 'socialized', 'socialized', tran, ['6A'] ).
verb( 'sock', 'socks', 'socking', 'socked', 'socked', tran, ['6A','15A'] ).
verb( 'sod', 'sods', 'sodding', 'sodded', 'sodded', intran, [] ).
verb( 'soft-pedal', 'soft-pedals', 'soft-pedalling', 'soft-pedalled', 'soft-pedalled', _, [] ).
verb( 'soft-soap', 'soft-soaps', 'soft-soaping', 'soft-soaped', 'soft-soaped', tran, [] ).
verb( 'soft-solder', 'soft-solders', 'soft-soldering', 'soft-soldered', 'soft-soldered', tran, [] ).
verb( 'soften', 'softens', 'softening', 'softened', 'softened', _, ['2A','6A'] ).
verb( 'softland', 'softlands', 'softlanding', 'softlanded', 'softlanded', intran, [] ).
verb( 'soil', 'soils', 'soiling', 'soiled', 'soiled', _, ['2A','2C','6A'] ).
verb( 'sojourn', 'sojourns', 'sojourning', 'sojourned', 'sojourned', intran, ['2C'] ).
verb( 'solace', 'solaces', 'solacing', 'solaced', 'solaced', tran, ['6A','15A'] ).
verb( 'solder', 'solders', 'soldering', 'soldered', 'soldered', tran, ['6A','15A','15B'] ).
verb( 'soldier', 'soldiers', 'soldiering', 'soldiered', 'soldiered', intran, ['2A'] ).
verb( 'sole', 'soles', 'soling', 'soled', 'soled', tran, ['6A'] ).
verb( 'solemnize', 'solemnizes', 'solemnizing', 'solemnized', 'solemnized', tran, ['6A'] ).
verb( 'solicit', 'solicits', 'soliciting', 'solicited', 'solicited', _, ['2A','6A','14'] ).
verb( 'solidify', 'solidifies', 'solidifying', 'solidified', 'solidified', _, ['2A','6A'] ).
verb( 'soliloquize', 'soliloquizes', 'soliloquizing', 'soliloquized', 'soliloquized', intran, ['2A'] ).
verb( 'solve', 'solves', 'solving', 'solved', 'solved', tran, ['6A'] ).
verb( 'somersault', 'somersaults', 'somersaulting', 'somersaulted', 'somersaulted', intran, ['2A'] ).
verb( 'soot', 'soots', 'sooting', 'sooted', 'sooted', tran, [] ).
verb( 'soothe', 'soothes', 'soothing', 'soothed', 'soothed', tran, ['6A'] ).
verb( 'sop', 'sops', 'sopping', 'sopped', 'sopped', tran, ['6A','15B'] ).
verb( 'sorrow', 'sorrows', 'sorrowing', 'sorrowed', 'sorrowed', intran, ['2A','2C'] ).
verb( 'sort', 'sorts', 'sorting', 'sorted', 'sorted', _, ['6A','15B'] ).
verb( 'sough', 'soughs', 'soughing', 'soughed', 'soughed', intran, ['2A'] ).
verb( 'sound', 'sounds', 'sounding', 'sounded', 'sounded', _, ['2A','2C','2D','6A','15A','15B'] ).
verb( 'soundproof', 'soundproofs', 'soundproofing', 'soundproofed', 'soundproofed', tran, ['6A'] ).
verb( 'soup', 'soups', 'souping', 'souped', 'souped', tran, ['15B'] ).
verb( 'sour', 'sours', 'souring', 'soured', 'soured', _, ['2A','6A'] ).
verb( 'souse', 'souses', 'sousing', 'soused', 'soused', tran, ['6A'] ).
verb( 'sovietize', 'sovietizes', 'sovietizing', 'sovietized', 'sovietized', tran, [] ).
verb( 'sow', 'sows', 'sowing', 'sowed', 'sowed', _, ['2A','6A','15A'] ).
verb( 'space', 'spaces', 'spacing', 'spaced', 'spaced', tran, ['6A','15B'] ).
verb( 'spade', 'spades', 'spading', 'spaded', 'spaded', tran, ['6A','15B'] ).
verb( 'span', 'spans', 'spanning', 'spanned', 'spanned', tran, ['2A','2C','6A','14','15B'] ).
verb( 'spangle', 'spangles', 'spangling', 'spangled', 'spangled', tran, [] ).
verb( 'spank', 'spanks', 'spanking', 'spanked', 'spanked', _, ['2C','6A'] ).
verb( 'spar', 'spars', 'sparring', 'sparred', 'sparred', intran, ['2A','2C'] ).
verb( 'spare', 'spares', 'sparing', 'spared', 'spared', _, ['6A','12A','12B','13A','13B'] ).
verb( 'spark', 'sparks', 'sparking', 'sparked', 'sparked', _, ['2A','15B'] ).
verb( 'sparkle', 'sparkles', 'sparkling', 'sparkled', 'sparkled', intran, ['2A','2C'] ).
verb( 'spat', 'spats', 'spatting', 'spatted', 'spatted', _, ['2A','2C','3A','6A','15B'] ).
verb( 'spatchcock', 'spatchcocks', 'spatchcocking', 'spatchcocked', 'spatchcocked', tran, ['6A','14'] ).
verb( 'spatter', 'spatters', 'spattering', 'spattered', 'spattered', _, ['2C','6A','14'] ).
verb( 'spawn', 'spawns', 'spawning', 'spawned', 'spawned', _, ['2A','6A'] ).
verb( 'spay', 'spays', 'spaying', 'spayed', 'spayed', tran, ['6A'] ).
verb( 'speak', 'speaks', 'speaking', 'spoke', 'spoken', _, ['2A','2B','2C','3A','6A','15A'] ).
verb( 'spear', 'spears', 'spearing', 'speared', 'speared', tran, ['6A'] ).
verb( 'specialize', 'specializes', 'specializing', 'specialized', 'specialized', _, ['2A','3A'] ).
verb( 'specify', 'specifies', 'specifying', 'specified', 'specified', tran, ['6A','9'] ).
verb( 'speculate', 'speculates', 'speculating', 'speculated', 'speculated', intran, ['2A','2C'] ).
verb( 'speechify', 'speechifies', 'speechifying', 'speechified', 'speechified', intran, [] ).
verb( 'speed', 'speeds', 'speeding', 'speeded', 'speeded', _, ['2A','2C','6A','14','15B'] ).
verb( 'spell', 'spells', 'spelling', 'spelled', 'spelled', _, ['2A','6A','15B'] ).
verb( 'spend', 'spends', 'spending', 'spent', 'spent', _, ['2A','6A','14','19B'] ).
verb( 'spew', 'spews', 'spewing', 'spewed', 'spewed', _, ['2A','6A','15B'] ).
verb( 'spice', 'spices', 'spicing', 'spiced', 'spiced', tran, ['6A'] ).
verb( 'spiel', 'spiels', 'spieling', 'spieled', 'spieled', _, [] ).
verb( 'spike', 'spikes', 'spiking', 'spiked', 'spiked', tran, ['6A'] ).
verb( 'spill', 'spills', 'spilling', 'spilled', 'spilled', _, ['2A','2C','6A'] ).
verb( 'spin', 'spins', 'spinning', 'span', 'spun', _, ['2A','2C','6A','14','15B'] ).
verb( 'spin-dry', 'spin-dries', 'spin-drying', 'spin-dried', 'spin-dried', tran, [] ).
verb( 'spiral', 'spirals', 'spiralling', 'spiralled', 'spiralled', intran, ['2A','2C'] ).
verb( 'spirit', 'spirits', 'spiriting', 'spirited', 'spirited', tran, ['15B'] ).
verb( 'spiritualize', 'spiritualizes', 'spiritualizing', 'spiritualized', 'spiritualized', tran, ['6A'] ).
verb( 'spirt', 'spirts', 'spirting', 'spirted', 'spirted', intran, ['2A','2C'] ).
verb( 'spit', 'spits', 'spitting', 'spat', 'spat', _, ['2A','2C','3A','6A','15B'] ).
verb( 'spite', 'spites', 'spiting', 'spited', 'spited', tran, ['6A'] ).
verb( 'splash', 'splashes', 'splashing', 'splashed', 'splashed', _, ['2A','2C','6A','14','15B'] ).
verb( 'splay', 'splays', 'splaying', 'splayed', 'splayed', _, [] ).
verb( 'splice', 'splices', 'splicing', 'spliced', 'spliced', tran, ['6A'] ).
verb( 'splinter', 'splinters', 'splintering', 'splintered', 'splintered', _, ['2C','15B'] ).
verb( 'split', 'splits', 'splitting', 'split', 'split', _, ['2A','2C','2D','3A','6A','14','15A','15B','22'] ).
verb( 'splosh', 'sploshes', 'sploshing', 'sploshed', 'sploshed', tran, [] ).
verb( 'splurge', 'splurges', 'splurging', 'splurged', 'splurged', intran, [] ).
verb( 'splutter', 'splutters', 'spluttering', 'spluttered', 'spluttered', _, ['2A','2C','6A','15B'] ).
verb( 'spoil', 'spoils', 'spoiling', 'spoiled', 'spoiled', _, ['2A','6A','14'] ).
verb( 'sponge', 'sponges', 'sponging', 'sponged', 'sponged', _, ['3A','6A','14','15B'] ).
verb( 'sponsor', 'sponsors', 'sponsoring', 'sponsored', 'sponsored', tran, ['6A'] ).
verb( 'spoof', 'spoofs', 'spoofing', 'spoofed', 'spoofed', tran, [] ).
verb( 'spoon', 'spoons', 'spooning', 'spooned', 'spooned', _, ['2A','15B'] ).
verb( 'spoonfeed', 'spoonfeeds', 'spoonfeeding', 'spoonfed', 'spoonfed', tran, [] ).
verb( 'sport', 'sports', 'sporting', 'sported', 'sported', _, ['2C','6A'] ).
verb( 'spot', 'spots', 'spotting', 'spotted', 'spotted', _, ['2A','6A'] ).
verb( 'spotlight', 'spotlights', 'spotlighting', 'spotlighted', 'spotlighted', tran, ['6A'] ).
verb( 'spout', 'spouts', 'spouting', 'spouted', 'spouted', _, ['2A','2C','6A'] ).
verb( 'sprain', 'sprains', 'spraining', 'sprained', 'sprained', tran, ['6A'] ).
verb( 'sprawl', 'sprawls', 'sprawling', 'sprawled', 'sprawled', intran, ['2A','2C'] ).
verb( 'spray', 'sprays', 'spraying', 'sprayed', 'sprayed', tran, ['6A','14'] ).
verb( 'spread', 'spreads', 'spreading', 'spread', 'spread', _, ['2A','2B','2C','6A','14','15A','15B'] ).
verb( 'spreadeagle', 'spreadeagles', 'spreadeagling', 'spreadeagled', 'spreadeagled', tran, [] ).
verb( 'spring', 'springs', 'springing', 'sprang', 'sprung', _, ['2A','2C','3A','6A','14'] ).
verb( 'spring-clean', 'spring-cleans', 'spring-cleaning', 'spring-cleaned', 'spring-cleaned', tran, [] ).
verb( 'sprinkle', 'sprinkles', 'sprinkling', 'sprinkled', 'sprinkled', tran, ['6A','14'] ).
verb( 'sprint', 'sprints', 'sprinting', 'sprinted', 'sprinted', intran, ['2A','2C'] ).
verb( 'sprout', 'sprouts', 'sprouting', 'sprouted', 'sprouted', _, ['2A','2C','6A'] ).
verb( 'spruce', 'spruces', 'sprucing', 'spruced', 'spruced', _, ['2C','6A','15B'] ).
verb( 'spue', 'spues', 'spuing', 'spued', 'spued', _, [] ).
verb( 'spur', 'spurs', 'spurring', 'spurred', 'spurred', _, ['2C','6A','15B'] ).
verb( 'spurn', 'spurns', 'spurning', 'spurned', 'spurned', tran, ['6A'] ).
verb( 'spurt', 'spurts', 'spurting', 'spurted', 'spurted', intran, ['2A','2C','3A'] ).
verb( 'sputter', 'sputters', 'sputtering', 'sputtered', 'sputtered', _, ['2A','2C'] ).
verb( 'spy', 'spies', 'spying', 'spied', 'spied', _, ['2A','3A','6A','15B','19A'] ).
verb( 'squabble', 'squabbles', 'squabbling', 'squabbled', 'squabbled', intran, ['2C'] ).
verb( 'squall', 'squalls', 'squalling', 'squalled', 'squalled', intran, ['2A'] ).
verb( 'squander', 'squanders', 'squandering', 'squandered', 'squandered', tran, ['6A'] ).
verb( 'square', 'squares', 'squaring', 'squared', 'squared', _, ['2C','3A','6A','14','15B'] ).
verb( 'squash', 'squashes', 'squashing', 'squashed', 'squashed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'squat', 'squats', 'squatting', 'squatted', 'squatted', intran, ['2A','2C','6A','15B'] ).
verb( 'squawk', 'squawks', 'squawking', 'squawked', 'squawked', intran, ['2A','2C'] ).
verb( 'squeak', 'squeaks', 'squeaking', 'squeaked', 'squeaked', _, ['2A','2C','6A','15B'] ).
verb( 'squeal', 'squeals', 'squealing', 'squealed', 'squealed', _, ['2A','2C','6A','15B'] ).
verb( 'squeegee', 'squeegees', 'squeegeeing', 'squeegeed', 'squeegeed', tran, ['6A'] ).
verb( 'squeeze', 'squeezes', 'squeezing', 'squeezed', 'squeezed', _, ['2A','2C','6A','14','15A','15B','22'] ).
verb( 'squelch', 'squelches', 'squelching', 'squelched', 'squelched', _, ['2A','2C'] ).
verb( 'squint', 'squints', 'squinting', 'squinted', 'squinted', intran, ['2A','3A'] ).
verb( 'squire', 'squires', 'squiring', 'squired', 'squired', tran, ['6A','15A'] ).
verb( 'squirm', 'squirms', 'squirming', 'squirmed', 'squirmed', intran, ['2A','2C'] ).
verb( 'squirt', 'squirts', 'squirting', 'squirted', 'squirted', _, ['2A','2C','6A','15A','15B'] ).
verb( 'stab', 'stabs', 'stabbing', 'stabbed', 'stabbed', _, ['2A','3A','6A','15A'] ).
verb( 'stabilize', 'stabilizes', 'stabilizing', 'stabilized', 'stabilized', tran, ['6A'] ).
verb( 'stable', 'stables', 'stabling', 'stabled', 'stabled', tran, [] ).
verb( 'stack', 'stacks', 'stacking', 'stacked', 'stacked', tran, ['6A','15B'] ).
verb( 'staff', 'staffs', 'staffing', 'staffed', 'staffed', tran, ['6A'] ).
verb( 'stage', 'stages', 'staging', 'staged', 'staged', _, ['2C','6A'] ).
verb( 'stagger', 'staggers', 'staggering', 'staggered', 'staggered', _, ['2A','2C','6A'] ).
verb( 'stagnate', 'stagnates', 'stagnating', 'stagnated', 'stagnated', intran, ['2A'] ).
verb( 'stain', 'stains', 'staining', 'stained', 'stained', _, ['2A','6A','22'] ).
verb( 'stake', 'stakes', 'staking', 'staked', 'staked', tran, ['6A','14','15B'] ).
verb( 'stale', 'stales', 'staling', 'staled', 'staled', intran, ['2A'] ).
verb( 'stalemate', 'stalemates', 'stalemating', 'stalemated', 'stalemated', tran, ['6A'] ).
verb( 'stalk', 'stalks', 'stalking', 'stalked', 'stalked', _, ['2C','6A'] ).
verb( 'stall', 'stalls', 'stalling', 'stalled', 'stalled', _, ['2A','6A'] ).
verb( 'stammer', 'stammers', 'stammering', 'stammered', 'stammered', _, ['2A','6A','15B'] ).
verb( 'stamp', 'stamps', 'stamping', 'stamped', 'stamped', _, ['2C','3A','6A','14','15A','15B','16B','22'] ).
verb( 'stampede', 'stampedes', 'stampeding', 'stampeded', 'stampeded', _, ['2A','6A','14'] ).
verb( 'stand', 'stands', 'standing', 'stood', 'stood', _, ['2A','2B','2C','3A','4A','6A','6C','12A','15A','15B'] ).
verb( 'standardize', 'standardizes', 'standardizing', 'standardized', 'standardized', tran, ['6A'] ).
verb( 'staple', 'staples', 'stapling', 'stapled', 'stapled', tran, ['6A'] ).
verb( 'star', 'stars', 'starring', 'starred', 'starred', _, ['3A','6A','14'] ).
verb( 'starboard', 'starboards', 'starboarding', 'starboarded', 'starboarded', tran, [] ).
verb( 'starch', 'starches', 'starching', 'starched', 'starched', tran, ['6A'] ).
verb( 'stare', 'stares', 'staring', 'stared', 'stared', _, ['2A','2B','2C','3A','15B'] ).
verb( 'start', 'starts', 'starting', 'started', 'started', _, ['2A','2C','3A','6A','6D','7A','15A','19B'] ).
verb( 'startle', 'startles', 'startling', 'startled', 'startled', tran, ['6A','15A'] ).
verb( 'starve', 'starves', 'starving', 'starved', 'starved', _, ['2A','2C','6A','15A'] ).
verb( 'stash', 'stashes', 'stashing', 'stashed', 'stashed', tran, ['15B'] ).
verb( 'state', 'states', 'stating', 'stated', 'stated', tran, ['6A','9'] ).
verb( 'station', 'stations', 'stationing', 'stationed', 'stationed', tran, ['6A','15A'] ).
verb( 'staunch', 'staunches', 'staunching', 'staunched', 'staunched', tran, ['6A'] ).
verb( 'stave', 'staves', 'staving', 'staved', 'staved', _, ['3A','15B'] ).
verb( 'stay', 'stays', 'staying', 'stayed', 'stayed', _, ['2A','2B','2C','4A','6A','15B'] ).
verb( 'steady', 'steadies', 'steadying', 'steadied', 'steadied', _, ['2A','6A'] ).
verb( 'steal', 'steals', 'stealing', 'stole', 'stolen', _, ['2A','2C','6A','14','15A'] ).
verb( 'steam', 'steams', 'steaming', 'steamed', 'steamed', _, ['2A','2C','6A','22'] ).
verb( 'steam-heat', 'steam-heats', 'steam-heating', 'steam-heated', 'steam-heated', tran, [] ).
verb( 'steamroller', 'steamrollers', 'steamrollering', 'steamrollered', 'steamrollered', tran, ['6A'] ).
verb( 'steel', 'steels', 'steeling', 'steeled', 'steeled', tran, ['6A','15A','16A'] ).
verb( 'steep', 'steeps', 'steeping', 'steeped', 'steeped', _, ['2A','6A','14'] ).
verb( 'steepen', 'steepens', 'steepening', 'steepened', 'steepened', _, ['2A','6A'] ).
verb( 'steer', 'steers', 'steering', 'steered', 'steered', _, ['2A','2C','6A'] ).
verb( 'stem', 'stems', 'stemming', 'stemmed', 'stemmed', _, ['3A','6A'] ).
verb( 'stencil', 'stencils', 'stencilling', 'stencilled', 'stencilled', tran, ['6A'] ).
verb( 'step', 'steps', 'stepping', 'stepped', 'stepped', _, ['2C','15B'] ).
verb( 'stereotype', 'stereotypes', 'stereotyping', 'stereotyped', 'stereotyped', tran, ['6A'] ).
verb( 'sterilize', 'sterilizes', 'sterilizing', 'sterilized', 'sterilized', tran, ['6A'] ).
verb( 'stew', 'stews', 'stewing', 'stewed', 'stewed', _, ['2A','2C','6A','15A'] ).
verb( 'stick', 'sticks', 'sticking', 'sticked', 'sticked', tran, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'stick', 'sticks', 'sticking', 'stuck', 'stuck', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'stiffen', 'stiffens', 'stiffening', 'stiffened', 'stiffened', _, ['2A','6A'] ).
verb( 'stifle', 'stifles', 'stifling', 'stifled', 'stifled', _, ['2A','6A'] ).
verb( 'stigmatize', 'stigmatizes', 'stigmatizing', 'stigmatized', 'stigmatized', tran, ['16B'] ).
verb( 'still', 'stills', 'stilling', 'stilled', 'stilled', tran, ['6A'] ).
verb( 'stimulate', 'stimulates', 'stimulating', 'stimulated', 'stimulated', tran, ['6A','14','17'] ).
verb( 'sting', 'stings', 'stinging', 'stang', 'stung', _, ['2A','6A','14','15A'] ).
verb( 'stink', 'stinks', 'stinking', 'stank', 'stunk', _, ['2A','3A','15B'] ).
verb( 'stint', 'stints', 'stinting', 'stinted', 'stinted', _, ['6A','14'] ).
verb( 'stipple', 'stipples', 'stippling', 'stippled', 'stippled', tran, ['6A'] ).
verb( 'stipulate', 'stipulates', 'stipulating', 'stipulated', 'stipulated', _, ['3A','6A','9'] ).
verb( 'stir', 'stirs', 'stirring', 'stirred', 'stirred', _, ['2A','2C','6A','14','15B'] ).
verb( 'stitch', 'stitches', 'stitching', 'stitched', 'stitched', _, ['2A','2C','6A','15A','15B'] ).
verb( 'stock', 'stocks', 'stocking', 'stocked', 'stocked', tran, ['6A','14'] ).
verb( 'stockade', 'stockades', 'stockading', 'stockaded', 'stockaded', tran, [] ).
verb( 'stoke', 'stokes', 'stoking', 'stoked', 'stoked', _, ['2A','2C','6A','15B'] ).
verb( 'stomach', 'stomachs', 'stomaching', 'stomached', 'stomached', tran, ['6A'] ).
verb( 'stomp', 'stomps', 'stomping', 'stomped', 'stomped', intran, ['2C'] ).
verb( 'stone', 'stones', 'stoning', 'stoned', 'stoned', tran, ['6A','15A'] ).
verb( 'stonewall', 'stonewalls', 'stonewalling', 'stonewalled', 'stonewalled', tran, [] ).
verb( 'stooge', 'stooges', 'stooging', 'stooged', 'stooged', intran, [] ).
verb( 'stoop', 'stoops', 'stooping', 'stooped', 'stooped', _, ['2A','2C','3A','4A','6A'] ).
verb( 'stop', 'stops', 'stopping', 'stopped', 'stopped', _, ['2A','2B','2C','3A','4A','6A','6B','6C','14','15A','15B'] ).
verb( 'store', 'stores', 'storing', 'stored', 'stored', tran, ['6A','15B'] ).
verb( 'storm', 'storms', 'storming', 'stormed', 'stormed', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'stow', 'stows', 'stowing', 'stowed', 'stowed', tran, ['6A','15A','15B'] ).
verb( 'straddle', 'straddles', 'straddling', 'straddled', 'straddled', _, ['2A','6A'] ).
verb( 'strafe', 'strafes', 'strafing', 'strafed', 'strafed', tran, ['6A'] ).
verb( 'straggle', 'straggles', 'straggling', 'straggled', 'straggled', intran, ['2A','2C'] ).
verb( 'straighten', 'straightens', 'straightening', 'straightened', 'straightened', _, ['2A','2C','6A','15B'] ).
verb( 'strain', 'strains', 'straining', 'strained', 'strained', _, ['2A','3A','6A','15A','15B','16A'] ).
verb( 'straiten', 'straitens', 'straitening', 'straitened', 'straitened', tran, [] ).
verb( 'strand', 'strands', 'stranding', 'stranded', 'stranded', _, ['2A','6A'] ).
verb( 'strangle', 'strangles', 'strangling', 'strangled', 'strangled', tran, ['6A'] ).
verb( 'strap', 'straps', 'strapping', 'strapped', 'strapped', tran, ['6A','15B'] ).
verb( 'stratify', 'stratifies', 'stratifying', 'stratified', 'stratified', _, ['2A','6A'] ).
verb( 'straw', 'straws', 'strawing', 'strawed', 'strawed', tran, [] ).
verb( 'stray', 'strays', 'straying', 'strayed', 'strayed', intran, ['2A','2C'] ).
verb( 'streak', 'streaks', 'streaking', 'streaked', 'streaked', _, ['2C','6A','15A'] ).
verb( 'stream', 'streams', 'streaming', 'streamed', 'streamed', intran, ['2A','2C','6A'] ).
verb( 'streamline', 'streamlines', 'streamlining', 'streamlined', 'streamlined', tran, [] ).
verb( 'strengthen', 'strengthens', 'strengthening', 'strengthened', 'strengthened', _, ['2A','6A'] ).
verb( 'stress', 'stresses', 'stressing', 'stressed', 'stressed', tran, ['6A'] ).
verb( 'stretch', 'stretches', 'stretching', 'stretched', 'stretched', _, ['2A','2B','2C','6A','15A','15B','16A','22'] ).
verb( 'strew', 'strews', 'strewing', 'strewed', 'strewed', tran, ['6A','14'] ).
verb( 'stride', 'strides', 'striding', 'strode', 'stridden', _, ['2C','6A'] ).
verb( 'stridulate', 'stridulates', 'stridulating', 'stridulated', 'stridulated', intran, ['2A'] ).
verb( 'strike', 'strikes', 'striking', 'struck', 'struck', _, ['2A','2C','2D','3A','6A','12C','14','15B','16A','22'] ).
verb( 'string', 'strings', 'stringing', 'strang', 'strung', _, ['2C','6A','15A','15B'] ).
verb( 'strip', 'strips', 'stripping', 'stripped', 'stripped', _, ['2A','2C','6A','14','15B','22'] ).
verb( 'strive', 'strives', 'striving', 'strove', 'striven', intran, ['2A','3A','4A'] ).
verb( 'stroke', 'strokes', 'stroking', 'stroked', 'stroked', tran, ['6A','15B'] ).
verb( 'stroll', 'strolls', 'strolling', 'strolled', 'strolled', intran, ['2A','2C'] ).
verb( 'strop', 'strops', 'stropping', 'stropped', 'stropped', tran, ['6A'] ).
verb( 'struggle', 'struggles', 'struggling', 'struggled', 'struggled', intran, ['2A','2B','3A','4A'] ).
verb( 'strum', 'strums', 'strumming', 'strummed', 'strummed', _, ['2A','2B','2C','3A','6A'] ).
verb( 'strut', 'struts', 'strutting', 'strutted', 'strutted', intran, ['2A','2C'] ).
verb( 'stub', 'stubs', 'stubbing', 'stubbed', 'stubbed', tran, ['6A','15B'] ).
verb( 'stucco', 'stuccos', 'stuccoing', 'stuccoed', 'stuccoed', tran, [] ).
verb( 'stud', 'studs', 'studding', 'studded', 'studded', tran, [] ).
verb( 'study', 'studies', 'studying', 'studied', 'studied', _, ['2A','2B','4A','6A','8','15A'] ).
verb( 'stuff', 'stuffs', 'stuffing', 'stuffed', 'stuffed', tran, ['2A','6A','14','15B'] ).
verb( 'stultify', 'stultifies', 'stultifying', 'stultified', 'stultified', tran, ['6A'] ).
verb( 'stumble', 'stumbles', 'stumbling', 'stumbled', 'stumbled', intran, ['2A','2C','3A'] ).
verb( 'stump', 'stumps', 'stumping', 'stumped', 'stumped', _, ['2C','6A','15B'] ).
verb( 'stun', 'stuns', 'stunning', 'stunned', 'stunned', tran, ['6A'] ).
verb( 'stunt', 'stunts', 'stunting', 'stunted', 'stunted', tran, ['6A'] ).
verb( 'stupefy', 'stupefies', 'stupefying', 'stupefied', 'stupefied', tran, ['6A'] ).
verb( 'stupify', 'stupifies', 'stupifying', 'stupified', 'stupified', tran, ['6A'] ).
verb( 'stutter', 'stutters', 'stuttering', 'stuttered', 'stuttered', _, [] ).
verb( 'style', 'styles', 'styling', 'styled', 'styled', tran, ['6A'] ).
verb( 'stylize', 'stylizes', 'stylizing', 'stylized', 'stylized', tran, [] ).
verb( 'stymie', 'stymies', 'stymying', 'stymied', 'stymied', tran, ['6A'] ).
verb( 'sub', 'subs', 'subbing', 'subbed', 'subbed', _, ['2A','3A','6A'] ).
verb( 'subcontract', 'subcontracts', 'subcontracting', 'subcontracted', 'subcontracted', _, ['2A','6A'] ).
verb( 'subdivide', 'subdivides', 'subdividing', 'subdivided', 'subdivided', _, ['2A','6A'] ).
verb( 'subdue', 'subdues', 'subduing', 'subdued', 'subdued', tran, ['6A'] ).
verb( 'subedit', 'subedits', 'subediting', 'subedited', 'subedited', tran, [] ).
verb( 'subject', 'subjects', 'subjecting', 'subjected', 'subjected', tran, ['14'] ).
verb( 'subjoin', 'subjoins', 'subjoining', 'subjoined', 'subjoined', tran, ['6A'] ).
verb( 'subjugate', 'subjugates', 'subjugating', 'subjugated', 'subjugated', tran, ['6A'] ).
verb( 'sublease', 'subleases', 'subleasing', 'subleased', 'subleased', _, ['2A','6A'] ).
verb( 'sublet', 'sublets', 'subletting', 'sublet', 'sublet', _, ['2A','6A'] ).
verb( 'sublimate', 'sublimates', 'sublimating', 'sublimated', 'sublimated', tran, ['6A'] ).
verb( 'submerge', 'submerges', 'submerging', 'submerged', 'submerged', _, ['2A','6A'] ).
verb( 'submit', 'submits', 'submitting', 'submitted', 'submitted', _, ['3A','6A','9','14'] ).
verb( 'subordinate', 'subordinates', 'subordinating', 'subordinated', 'subordinated', tran, ['6A','14'] ).
verb( 'suborn', 'suborns', 'suborning', 'suborned', 'suborned', tran, ['6A'] ).
verb( 'subpoena', 'subpoenas', 'subpoenaing', 'subpoenaed', 'subpoenaed', tran, ['6A'] ).
verb( 'subscribe', 'subscribes', 'subscribing', 'subscribed', 'subscribed', _, ['2A','3A','6A','14'] ).
verb( 'subserve', 'subserves', 'subserving', 'subserved', 'subserved', tran, ['6A'] ).
verb( 'subside', 'subsides', 'subsiding', 'subsided', 'subsided', intran, ['2A'] ).
verb( 'subsidize', 'subsidizes', 'subsidizing', 'subsidized', 'subsidized', tran, ['6A'] ).
verb( 'subsist', 'subsists', 'subsisting', 'subsisted', 'subsisted', intran, ['2A','3A'] ).
verb( 'substantiate', 'substantiates', 'substantiating', 'substantiated', 'substantiated', tran, ['6A'] ).
verb( 'substitute', 'substitutes', 'substituting', 'substituted', 'substituted', _, ['3A','6A','14'] ).
verb( 'subsume', 'subsumes', 'subsuming', 'subsumed', 'subsumed', tran, ['6A','14'] ).
verb( 'subtend', 'subtends', 'subtending', 'subtended', 'subtended', tran, [] ).
verb( 'subtract', 'subtracts', 'subtracting', 'subtracted', 'subtracted', tran, ['6A','14'] ).
verb( 'subvert', 'subverts', 'subverting', 'subverted', 'subverted', tran, ['6A'] ).
verb( 'succeed', 'succeeds', 'succeeding', 'succeeded', 'succeeded', _, ['2A','3A','6A','16B'] ).
verb( 'succour', 'succours', 'succouring', 'succoured', 'succoured', tran, ['6A'] ).
verb( 'succumb', 'succumbs', 'succumbing', 'succumbed', 'succumbed', intran, ['2A','3A'] ).
verb( 'suck', 'sucks', 'sucking', 'sucked', 'sucked', _, ['2C','6A','15A','15B','22'] ).
verb( 'suckle', 'suckles', 'suckling', 'suckled', 'suckled', tran, ['6A'] ).
verb( 'sue', 'sues', 'suing', 'sued', 'sued', _, ['3A','6A','14'] ).
verb( 'suffer', 'suffers', 'suffering', 'suffered', 'suffered', _, ['2A','3A','6A','17'] ).
verb( 'suffice', 'suffices', 'sufficing', 'sufficed', 'sufficed', _, ['2A','3A','6A'] ).
verb( 'suffocate', 'suffocates', 'suffocating', 'suffocated', 'suffocated', _, ['2A','2C','6A'] ).
verb( 'suffuse', 'suffuses', 'suffusing', 'suffused', 'suffused', tran, ['6A'] ).
verb( 'sugar', 'sugars', 'sugaring', 'sugared', 'sugared', tran, ['6A'] ).
verb( 'suggest', 'suggests', 'suggesting', 'suggested', 'suggested', tran, ['6A','6C','9','10','14'] ).
verb( 'suit', 'suits', 'suiting', 'suited', 'suited', _, ['2A','6A','6B'] ).
verb( 'sulk', 'sulks', 'sulking', 'sulked', 'sulked', intran, ['2A','2C'] ).
verb( 'sully', 'sullies', 'sullying', 'sullied', 'sullied', tran, ['6A'] ).
verb( 'sum', 'sums', 'summing', 'summed', 'summed', _, ['2C','15B'] ).
verb( 'summarize', 'summarizes', 'summarizing', 'summarized', 'summarized', tran, ['6A'] ).
verb( 'summer', 'summers', 'summering', 'summered', 'summered', intran, ['2C'] ).
verb( 'summon', 'summons', 'summoning', 'summoned', 'summoned', tran, ['6A','14','15B','17'] ).
verb( 'summons', 'summonses', 'summonsing', 'summonsed', 'summonsed', tran, ['6A','14','15B','17'] ).
verb( 'sun', 'suns', 'sunning', 'sunned', 'sunned', tran, ['6A'] ).
verb( 'sunbathe', 'sunbathes', 'sunbathing', 'sunbathed', 'sunbathed', intran, [] ).
verb( 'sunder', 'sunders', 'sundering', 'sundered', 'sundered', tran, ['6A'] ).
verb( 'sup', 'sups', 'supping', 'supped', 'supped', _, ['2A','2C','3A','6A','15B'] ).
verb( 'superannuate', 'superannuates', 'superannuating', 'superannuated', 'superannuated', tran, ['6A'] ).
verb( 'superimpose', 'superimposes', 'superimposing', 'superimposed', 'superimposed', tran, ['6A','14'] ).
verb( 'superintend', 'superintends', 'superintending', 'superintended', 'superintended', _, ['2A','6A'] ).
verb( 'supersede', 'supersedes', 'superseding', 'superseded', 'superseded', tran, ['6A'] ).
verb( 'supervene', 'supervenes', 'supervening', 'supervened', 'supervened', intran, ['2A'] ).
verb( 'supervise', 'supervises', 'supervising', 'supervised', 'supervised', _, ['2A','6A'] ).
verb( 'supplant', 'supplants', 'supplanting', 'supplanted', 'supplanted', tran, ['6A'] ).
verb( 'supplement', 'supplements', 'supplementing', 'supplemented', 'supplemented', tran, ['6A','15A'] ).
verb( 'supplicate', 'supplicates', 'supplicating', 'supplicated', 'supplicated', _, ['2C','6A','14','17'] ).
verb( 'supply', 'supplies', 'supplying', 'supplied', 'supplied', tran, ['6A','14'] ).
verb( 'support', 'supports', 'supporting', 'supported', 'supported', tran, ['6A'] ).
verb( 'suppose', 'supposes', 'supposing', 'supposed', 'supposed', tran, ['6A','9','25'] ).
verb( 'suppress', 'suppresses', 'suppressing', 'suppressed', 'suppressed', tran, ['6A'] ).
verb( 'suppurate', 'suppurates', 'suppurating', 'suppurated', 'suppurated', intran, ['2A'] ).
verb( 'surcharge', 'surcharges', 'surcharging', 'surcharged', 'surcharged', tran, ['6A','15A'] ).
verb( 'surface', 'surfaces', 'surfacing', 'surfaced', 'surfaced', _, ['2A','6A'] ).
verb( 'surfeit', 'surfeits', 'surfeiting', 'surfeited', 'surfeited', tran, ['6A','14'] ).
verb( 'surge', 'surges', 'surging', 'surged', 'surged', intran, ['2C'] ).
verb( 'surmise', 'surmises', 'surmising', 'surmised', 'surmised', _, ['2A','6A','9'] ).
verb( 'surmount', 'surmounts', 'surmounting', 'surmounted', 'surmounted', tran, ['6A'] ).
verb( 'surpass', 'surpasses', 'surpassing', 'surpassed', 'surpassed', tran, ['6A','15A'] ).
verb( 'surprise', 'surprises', 'surprising', 'surprised', 'surprised', tran, ['6A'] ).
verb( 'surrender', 'surrenders', 'surrendering', 'surrendered', 'surrendered', _, ['2A','6A','14'] ).
verb( 'surround', 'surrounds', 'surrounding', 'surrounded', 'surrounded', tran, ['6A'] ).
verb( 'surtax', 'surtaxes', 'surtaxing', 'surtaxed', 'surtaxed', tran, [] ).
verb( 'survey', 'surveys', 'surveying', 'surveyed', 'surveyed', tran, ['6A'] ).
verb( 'survive', 'survives', 'surviving', 'survived', 'survived', _, ['2A','6A'] ).
verb( 'suspect', 'suspects', 'suspecting', 'suspected', 'suspected', tran, ['6A','9','14','25'] ).
verb( 'suspend', 'suspends', 'suspending', 'suspended', 'suspended', tran, ['6A','14'] ).
verb( 'suss', 'susses', 'sussing', 'sussed', 'sussed', tran, ['15B'] ).
verb( 'sustain', 'sustains', 'sustaining', 'sustained', 'sustained', tran, ['6A'] ).
verb( 'swab', 'swabs', 'swabbing', 'swabbed', 'swabbed', tran, ['6A','15B'] ).
verb( 'swaddle', 'swaddles', 'swaddling', 'swaddled', 'swaddled', tran, ['6A'] ).
verb( 'swagger', 'swaggers', 'swaggering', 'swaggered', 'swaggered', intran, ['2A','2C'] ).
verb( 'swallow', 'swallows', 'swallowing', 'swallowed', 'swallowed', _, ['2C','6A','15B'] ).
verb( 'swamp', 'swamps', 'swamping', 'swamped', 'swamped', tran, ['6A','14'] ).
verb( 'swan', 'swans', 'swanning', 'swanned', 'swanned', intran, ['2C'] ).
verb( 'swank', 'swanks', 'swanking', 'swanked', 'swanked', intran, ['2A','2C'] ).
verb( 'swap', 'swaps', 'swapping', 'swapped', 'swapped', _, ['2A','6A','15A'] ).
verb( 'swarm', 'swarms', 'swarming', 'swarmed', 'swarmed', _, ['2A','2C','3A','6A','15B'] ).
verb( 'swat', 'swats', 'swatting', 'swatted', 'swatted', tran, [] ).
verb( 'swathe', 'swathes', 'swathing', 'swathed', 'swathed', tran, ['6A','15A'] ).
verb( 'sway', 'sways', 'swaying', 'swayed', 'swayed', _, ['2A','2C','6A','15A'] ).
verb( 'swear', 'swears', 'swearing', 'swore', 'sworn', _, ['2A','2B','2C','3A','6A','7A','9','14','15A','15B','22'] ).
verb( 'sweat', 'sweats', 'sweating', 'sweated', 'sweated', _, ['2A','2C','6A','15B'] ).
verb( 'sweep', 'sweeps', 'sweeping', 'swept', 'swept', _, ['2A','2C','6A','12A','15A','15B','22'] ).
verb( 'sweeten', 'sweetens', 'sweetening', 'sweetened', 'sweetened', _, ['2A','6A'] ).
verb( 'swell', 'swells', 'swelling', 'swelled', 'swelled', _, ['2A','2C','6A','14','15B'] ).
verb( 'swelter', 'swelters', 'sweltering', 'sweltered', 'sweltered', intran, ['2A'] ).
verb( 'swerve', 'swerves', 'swerving', 'swerved', 'swerved', _, ['2A','2C','4A','6A'] ).
verb( 'swig', 'swigs', 'swigging', 'swigged', 'swigged', _, ['2A','2C','6A','15B'] ).
verb( 'swill', 'swills', 'swilling', 'swilled', 'swilled', _, ['6A','15B'] ).
verb( 'swim', 'swims', 'swimming', 'swam', 'swum', _, ['2A','2B','2C','3A','6A'] ).
verb( 'swindle', 'swindles', 'swindling', 'swindled', 'swindled', _, ['6A','14'] ).
verb( 'swing', 'swings', 'swinging', 'swang', 'swung', _, ['2A','2B','2C','6A','6C','15A','15B','22'] ).
verb( 'swinge', 'swinges', 'swinging', 'swinged', 'swinged', tran, [] ).
verb( 'swipe', 'swipes', 'swiping', 'swiped', 'swiped', tran, ['3A','6A','15A'] ).
verb( 'swirl', 'swirls', 'swirling', 'swirled', 'swirled', _, ['2C','15B'] ).
verb( 'swish', 'swishes', 'swishing', 'swished', 'swished', _, ['2A','6A','15B'] ).
verb( 'switch', 'switches', 'switching', 'switched', 'switched', _, ['6A','15A','15B'] ).
verb( 'swivel', 'swivels', 'swivelling', 'swivelled', 'swivelled', _, ['2A','2C','6A','15B'] ).
verb( 'swob', 'swobs', 'swobbing', 'swobbed', 'swobbed', tran, [] ).
verb( 'swoon', 'swoons', 'swooning', 'swooned', 'swooned', intran, ['2A'] ).
verb( 'swoop', 'swoops', 'swooping', 'swooped', 'swooped', _, ['2A','2C','15B'] ).
verb( 'swop', 'swops', 'swopping', 'swopped', 'swopped', _, ['2A','6A','15A'] ).
verb( 'swot', 'swots', 'swotting', 'swotted', 'swotted', _, ['2A','2C','3A','15B'] ).
verb( 'syllabicate', 'syllabicates', 'syllabicating', 'syllabicated', 'syllabicated', tran, [] ).
verb( 'syllabify', 'syllabifies', 'syllabifying', 'syllabified', 'syllabified', tran, [] ).
verb( 'syllabize', 'syllabizes', 'syllabizing', 'syllabized', 'syllabized', tran, [] ).
verb( 'symbolize', 'symbolizes', 'symbolizing', 'symbolized', 'symbolized', tran, ['6A'] ).
verb( 'sympathize', 'sympathizes', 'sympathizing', 'sympathized', 'sympathized', intran, ['2A','3A'] ).
verb( 'synchronize', 'synchronizes', 'synchronizing', 'synchronized', 'synchronized', _, ['2A','6A'] ).
verb( 'syncopate', 'syncopates', 'syncopating', 'syncopated', 'syncopated', tran, ['6A'] ).
verb( 'syndicate', 'syndicates', 'syndicating', 'syndicated', 'syndicated', tran, ['6A'] ).
verb( 'synthesize', 'synthesizes', 'synthesizing', 'synthesized', 'synthesized', tran, ['6A'] ).
verb( 'syphon', 'syphons', 'syphoning', 'syphoned', 'syphoned', _, ['15B'] ).
verb( 'syringe', 'syringes', 'syringing', 'syringed', 'syringed', tran, ['6A','15B'] ).
verb( 'systematize', 'systematizes', 'systematizing', 'systematized', 'systematized', tran, ['6A'] ).
verb( 'table', 'tables', 'tabling', 'tabled', 'tabled', tran, ['6A'] ).
verb( 'taboo', 'taboos', 'tabooing', 'tabooed', 'tabooed', tran, ['6A'] ).
verb( 'tabulate', 'tabulates', 'tabulating', 'tabulated', 'tabulated', tran, ['6A'] ).
verb( 'tack', 'tacks', 'tacking', 'tacked', 'tacked', _, ['2A','2C','6A','15A','15B'] ).
verb( 'tackle', 'tackles', 'tackling', 'tackled', 'tackled', _, ['2A','6A','14'] ).
verb( 'tag', 'tags', 'tagging', 'tagged', 'tagged', _, ['2C','6A','14','15A','15B'] ).
verb( 'tail', 'tails', 'tailing', 'tailed', 'tailed', _, ['2C','3A','6A'] ).
verb( 'tailor', 'tailors', 'tailoring', 'tailored', 'tailored', tran, ['6A','15A'] ).
verb( 'taint', 'taints', 'tainting', 'tainted', 'tainted', _, ['2A','6A'] ).
verb( 'take', 'takes', 'taking', 'took', 'taken', _, ['2A','2B','2C','3A','6A','6B','12A','13A','14','15A','15B','16B','19B','22'] ).
verb( 'talk', 'talks', 'talking', 'talked', 'talked', _, ['2A','2B','2C','3A','6A','14','15B','22'] ).
verb( 'tally', 'tallies', 'tallying', 'tallied', 'tallied', intran, ['2A','3A'] ).
verb( 'tame', 'tames', 'taming', 'tamed', 'tamed', tran, ['6A'] ).
verb( 'tamp', 'tamps', 'tamping', 'tamped', 'tamped', tran, ['15B'] ).
verb( 'tamper', 'tampers', 'tampering', 'tampered', 'tampered', intran, ['3A'] ).
verb( 'tan', 'tans', 'tanning', 'tanned', 'tanned', _, ['2A','6A'] ).
verb( 'tangle', 'tangles', 'tangling', 'tangled', 'tangled', _, ['2A','2C','3A','6A','15B'] ).
verb( 'tank', 'tanks', 'tanking', 'tanked', 'tanked', intran, [] ).
verb( 'tantalize', 'tantalizes', 'tantalizing', 'tantalized', 'tantalized', tran, ['6A'] ).
verb( 'tap', 'taps', 'tapping', 'tapped', 'tapped', _, ['2A','2C','6A','14','15A','15B'] ).
verb( 'tape', 'tapes', 'taping', 'taped', 'taped', tran, ['6A'] ).
verb( 'taper', 'tapers', 'tapering', 'tapered', 'tapered', _, ['2A','2C','6A','15B'] ).
verb( 'tar', 'tars', 'tarring', 'tarred', 'tarred', tran, ['6A'] ).
verb( 'tarmac', 'tarmacs', 'tarmacking', 'tarmacked', 'tarmacked', tran, [] ).
verb( 'tarnish', 'tarnishes', 'tarnishing', 'tarnished', 'tarnished', _, ['2A','6A'] ).
verb( 'tarry', 'tarries', 'tarrying', 'tarried', 'tarried', intran, ['2A','2B','2C'] ).
verb( 'tart', 'tarts', 'tarting', 'tarted', 'tarted', tran, ['15B'] ).
verb( 'task', 'tasks', 'tasking', 'tasked', 'tasked', tran, ['6A'] ).
verb( 'taste', 'tastes', 'tasting', 'tasted', 'tasted', _, ['2A','2D','3A','6A'] ).
verb( 'tat', 'tats', 'tatting', 'tatted', 'tatted', _, [] ).
verb( 'tattle', 'tattles', 'tattling', 'tattled', 'tattled', _, ['2A'] ).
verb( 'tattoo', 'tattoos', 'tattooing', 'tattooed', 'tattooed', tran, ['6A'] ).
verb( 'taunt', 'taunts', 'taunting', 'taunted', 'taunted', tran, ['6A','14'] ).
verb( 'tax', 'taxes', 'taxing', 'taxed', 'taxed', tran, ['6A','14'] ).
verb( 'taxi', 'taxis', 'taxiing', 'taxied', 'taxied', _, ['2C','15A'] ).
verb( 'teach', 'teaches', 'teaching', 'taught', 'taught', _, ['2A','2B','2C','6A','11','12A','13A','17','20','21'] ).
verb( 'team', 'teams', 'teaming', 'teamed', 'teamed', intran, ['2C'] ).
verb( 'tear', 'tears', 'tearing', 'tore', 'torn', _, ['2A','2C','3A','6A','15A','15B','22'] ).
verb( 'tease', 'teases', 'teasing', 'teased', 'teased', tran, ['6A','15A','15B'] ).
verb( 'tee', 'tees', 'teeing', 'teed', 'teed', _, ['2A','2C','15B'] ).
verb( 'teem', 'teems', 'teeming', 'teemed', 'teemed', intran, ['2A','2C','3A'] ).
verb( 'teeter', 'teeters', 'teetering', 'teetered', 'teetered', intran, ['2C'] ).
verb( 'teethe', 'teethes', 'teething', 'teethed', 'teethed', intran, ['2A'] ).
verb( 'telecast', 'telecasts', 'telecasting', 'telecast', 'telecast', tran, [] ).
verb( 'telegraph', 'telegraphs', 'telegraphing', 'telegraphed', 'telegraphed', _, ['2A','6A','11','12A','13A'] ).
verb( 'telephone', 'telephones', 'telephoning', 'telephoned', 'telephoned', _, ['2A','4A','6A','11','12A','13A'] ).
verb( 'telescope', 'telescopes', 'telescoping', 'telescoped', 'telescoped', _, ['2A','6A'] ).
verb( 'televise', 'televises', 'televising', 'televised', 'televised', tran, ['6A'] ).
verb( 'tell', 'tells', 'telling', 'told', 'told', _, ['2A','3A','6A','8','10','11','12A','13A','14','15A','15B','17','20','21'] ).
verb( 'temper', 'tempers', 'tempering', 'tempered', 'tempered', _, ['2A','6A','15A'] ).
verb( 'temporize', 'temporizes', 'temporizing', 'temporized', 'temporized', intran, ['2A'] ).
verb( 'tempt', 'tempts', 'tempting', 'tempted', 'tempted', tran, ['6A','14','17'] ).
verb( 'tenant', 'tenants', 'tenanting', 'tenanted', 'tenanted', tran, [] ).
verb( 'tend', 'tends', 'tending', 'tended', 'tended', _, ['2C','4A','6A'] ).
verb( 'tender', 'tenders', 'tendering', 'tendered', 'tendered', _, ['2A','3A','6A','12A','13A'] ).
verb( 'tense', 'tenses', 'tensing', 'tensed', 'tensed', _, ['2A','6A'] ).
verb( 'tergiversate', 'tergiversates', 'tergiversating', 'tergiversated', 'tergiversated', intran, [] ).
verb( 'term', 'terms', 'terming', 'termed', 'termed', tran, ['23'] ).
verb( 'terminate', 'terminates', 'terminating', 'terminated', 'terminated', _, ['2A','6A','15A'] ).
verb( 'terrace', 'terraces', 'terracing', 'terraced', 'terraced', tran, ['6A'] ).
verb( 'terrify', 'terrifies', 'terrifying', 'terrified', 'terrified', tran, ['6A','15A'] ).
verb( 'terrorize', 'terrorizes', 'terrorizing', 'terrorized', 'terrorized', tran, ['6A'] ).
verb( 'test', 'tests', 'testing', 'tested', 'tested', tran, ['6A','15A'] ).
verb( 'test-drive', 'test-drives', 'test-driving', 'test-drove', 'test-driven', tran, [] ).
verb( 'testify', 'testifies', 'testifying', 'testified', 'testified', _, ['2A','3A','6A','9'] ).
verb( 'tether', 'tethers', 'tethering', 'tethered', 'tethered', tran, ['6A','15A'] ).
verb( 'thank', 'thanks', 'thanking', 'thanked', 'thanked', tran, ['6A','11','14','17'] ).
verb( 'thatch', 'thatches', 'thatching', 'thatched', 'thatched', tran, ['6A'] ).
verb( 'thaw', 'thaws', 'thawing', 'thawed', 'thawed', _, ['2A','2C','6A','15B'] ).
verb( 'theorize', 'theorizes', 'theorizing', 'theorized', 'theorized', intran, ['2A','3A'] ).
verb( 'thicken', 'thickens', 'thickening', 'thickened', 'thickened', _, [] ).
verb( 'thieve', 'thieves', 'thieving', 'thieved', 'thieved', _, ['2A','6A'] ).
verb( 'thin', 'thins', 'thinning', 'thinned', 'thinned', _, ['2A','2C','6A','15B'] ).
verb( 'think', 'thinks', 'thinking', 'thought', 'thought', _, ['2A','2B','2C','3A','6A','7A','8','9','10','15B','22','25'] ).
verb( 'thirst', 'thirsts', 'thirsting', 'thirsted', 'thirsted', tran, ['2A','3A'] ).
verb( 'thrash', 'thrashes', 'thrashing', 'thrashed', 'thrashed', _, ['2C','6A','15A','15B'] ).
verb( 'thread', 'threads', 'threading', 'threaded', 'threaded', tran, ['6A','15A'] ).
verb( 'threaten', 'threatens', 'threatening', 'threatened', 'threatened', _, ['2A','6A','14','17'] ).
verb( 'thresh', 'threshes', 'threshing', 'threshed', 'threshed', _, ['2A','6A','15A'] ).
verb( 'thrill', 'thrills', 'thrilling', 'thrilled', 'thrilled', _, ['2A','2C','6A'] ).
verb( 'thrive', 'thrives', 'thriving', 'thrived', 'thrived', intran, ['2A','3A'] ).
verb( 'throb', 'throbs', 'throbbing', 'throbbed', 'throbbed', intran, ['2A','2C'] ).
verb( 'throng', 'throngs', 'thronging', 'thronged', 'thronged', _, ['2C','4A','6A'] ).
verb( 'throttle', 'throttles', 'throttling', 'throttled', 'throttled', _, ['2C','6A','15B'] ).
verb( 'throw', 'throws', 'throwing', 'threw', 'thrown', _, ['2A','6A','12A','13A','15A','15B','22'] ).
verb( 'thrum', 'thrums', 'thrumming', 'thrummed', 'thrummed', _, ['3A','6A'] ).
verb( 'thrust', 'thrusts', 'thrusting', 'thrust', 'thrust', _, ['2A','2C','6A','15A','15B'] ).
verb( 'thud', 'thuds', 'thudding', 'thudded', 'thudded', intran, ['2C'] ).
verb( 'thumb', 'thumbs', 'thumbing', 'thumbed', 'thumbed', tran, ['6A'] ).
verb( 'thump', 'thumps', 'thumping', 'thumped', 'thumped', _, ['2A','2C','3A','6A','15A','22'] ).
verb( 'thunder', 'thunders', 'thundering', 'thundered', 'thundered', _, ['2A','2C','3A','15B'] ).
verb( 'thwack', 'thwacks', 'thwacking', 'thwacked', 'thwacked', tran, [] ).
verb( 'thwart', 'thwarts', 'thwarting', 'thwarted', 'thwarted', tran, ['6A'] ).
verb( 'tick', 'ticks', 'ticking', 'ticked', 'ticked', _, ['2A','2C','6A','10','15B'] ).
verb( 'ticket', 'tickets', 'ticketing', 'ticketed', 'ticketed', tran, ['6A'] ).
verb( 'tickle', 'tickles', 'tickling', 'tickled', 'tickled', _, ['2A','6A'] ).
verb( 'tide', 'tides', 'tiding', 'tided', 'tided', tran, ['14','15B'] ).
verb( 'tidy', 'tidies', 'tidying', 'tidied', 'tidied', _, ['2A','2C','6A','15B'] ).
verb( 'tie', 'ties', 'tying', 'tied', 'tied', _, ['2A','3A','6A','15A','15B'] ).
verb( 'tighten', 'tightens', 'tightening', 'tightened', 'tightened', _, ['2A','2C','6A','15B'] ).
verb( 'tile', 'tiles', 'tiling', 'tiled', 'tiled', tran, ['6A'] ).
verb( 'till', 'tills', 'tilling', 'tilled', 'tilled', tran, ['6A'] ).
verb( 'tilt', 'tilts', 'tilting', 'tilted', 'tilted', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'time', 'times', 'timing', 'timed', 'timed', tran, ['6A','15A'] ).
verb( 'tin', 'tins', 'tinning', 'tinned', 'tinned', tran, ['6A'] ).
verb( 'tincture', 'tinctures', 'tincturing', 'tinctured', 'tinctured', tran, [] ).
verb( 'ting', 'tings', 'tinging', 'tinged', 'tinged', _, [] ).
verb( 'tinge', 'tinges', 'tinging', 'tinged', 'tinged', tran, ['6A','14'] ).
verb( 'tingle', 'tingles', 'tingling', 'tingled', 'tingled', intran, ['2A','2C'] ).
verb( 'tinker', 'tinkers', 'tinkering', 'tinkered', 'tinkered', intran, ['2A','2C','3A'] ).
verb( 'tinkle', 'tinkles', 'tinkling', 'tinkled', 'tinkled', _, ['2A','2C','6A'] ).
verb( 'tinsel', 'tinsels', 'tinselling', 'tinselled', 'tinselled', tran, [] ).
verb( 'tint', 'tints', 'tinting', 'tinted', 'tinted', tran, ['6A','22'] ).
verb( 'tip', 'tips', 'tipping', 'tipped', 'tipped', _, ['2A','2C','6A','12C','15A','15B'] ).
verb( 'tipple', 'tipples', 'tippling', 'tippled', 'tippled', _, ['2A','6A'] ).
verb( 'tiptoe', 'tiptoes', 'tiptoeing', 'tiptoed', 'tiptoed', intran, ['2A','2C'] ).
verb( 'tire', 'tires', 'tiring', 'tired', 'tired', _, ['2A','3A','6A','15B'] ).
verb( 'titillate', 'titillates', 'titillating', 'titillated', 'titillated', tran, ['6A'] ).
verb( 'titivate', 'titivates', 'titivating', 'titivated', 'titivated', _, ['2A','6A'] ).
verb( 'titter', 'titters', 'tittering', 'tittered', 'tittered', intran, ['2A'] ).
verb( 'tittivate', 'tittivates', 'tittivating', 'tittivated', 'tittivated', _, ['2A','6A'] ).
verb( 'tittle-tattle', 'tittle-tattles', 'tittle-tattling', 'tittle-tattled', 'tittle-tattled', intran, [] ).
verb( 'toady', 'toadies', 'toadying', 'toadied', 'toadied', intran, ['2A','3A'] ).
verb( 'toast', 'toasts', 'toasting', 'toasted', 'toasted', _, ['2A','6A'] ).
verb( 'toboggan', 'toboggans', 'tobogganing', 'tobogganed', 'tobogganed', intran, ['2A','2C'] ).
verb( 'toddle', 'toddles', 'toddling', 'toddled', 'toddled', intran, ['2A','2C'] ).
verb( 'toe', 'toes', 'toeing', 'toed', 'toed', tran, ['6A'] ).
verb( 'tog', 'togs', 'togging', 'togged', 'togged', tran, ['15B'] ).
verb( 'toil', 'toils', 'toiling', 'toiled', 'toiled', intran, ['2A','2B','2C','3A','4A'] ).
verb( 'tolerate', 'tolerates', 'tolerating', 'tolerated', 'tolerated', tran, ['6A','6C'] ).
verb( 'toll', 'tolls', 'tolling', 'tolled', 'tolled', _, ['2A','6A'] ).
verb( 'tomahawk', 'tomahawks', 'tomahawking', 'tomahawked', 'tomahawked', tran, [] ).
verb( 'ton', 'tons', 'tonning', 'tonned', 'tonned', intran, ['2C'] ).
verb( 'tone', 'tones', 'toning', 'toned', 'toned', _, ['2C','6A','15B'] ).
verb( 'tonsure', 'tonsures', 'tonsuring', 'tonsured', 'tonsured', tran, [] ).
verb( 'tool', 'tools', 'tooling', 'tooled', 'tooled', tran, ['2C','6A'] ).
verb( 'toot', 'toots', 'tooting', 'tooted', 'tooted', _, ['2A','6A'] ).
verb( 'tootle', 'tootles', 'tootling', 'tootled', 'tootled', intran, [] ).
verb( 'top', 'tops', 'topping', 'topped', 'topped', tran, ['6A'] ).
verb( 'top-dress', 'top-dresses', 'top-dressing', 'top-dressed', 'top-dressed', tran, [] ).
verb( 'tope', 'topes', 'toping', 'toped', 'toped', _, ['2A','6A'] ).
verb( 'topple', 'topples', 'toppling', 'toppled', 'toppled', _, ['2A','2C','6A','15B'] ).
verb( 'torment', 'torments', 'tormenting', 'tormented', 'tormented', tran, ['6A','15A'] ).
verb( 'torpedo', 'torpedos', 'torpedoing', 'torpedoed', 'torpedoed', tran, ['6A'] ).
verb( 'torture', 'tortures', 'torturing', 'tortured', 'tortured', tran, ['6A','16A'] ).
verb( 'toss', 'tosses', 'tossing', 'tossed', 'tossed', _, ['2C','6A','12A','13A','15A','15B'] ).
verb( 'tot', 'tots', 'totting', 'totted', 'totted', _, ['2C','15B'] ).
verb( 'total', 'totals', 'totalling', 'totalled', 'totalled', _, ['2C','6A'] ).
verb( 'tote', 'totes', 'toting', 'toted', 'toted', tran, ['6A'] ).
verb( 'totter', 'totters', 'tottering', 'tottered', 'tottered', intran, ['2A','2C'] ).
verb( 'touch', 'touches', 'touching', 'touched', 'touched', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'touch-type', 'touch-types', 'touch-typing', 'touch-typed', 'touch-typed', intran, ['2A'] ).
verb( 'toughen', 'toughens', 'toughening', 'toughened', 'toughened', _, [] ).
verb( 'tour', 'tours', 'touring', 'toured', 'toured', _, ['2A','2C','6A'] ).
verb( 'tousle', 'tousles', 'tousling', 'tousled', 'tousled', tran, ['6A'] ).
verb( 'tout', 'touts', 'touting', 'touted', 'touted', intran, ['2A','3A'] ).
verb( 'tow', 'tows', 'towing', 'towed', 'towed', tran, ['6A','15A','15B'] ).
verb( 'towel', 'towels', 'towelling', 'towelled', 'towelled', tran, [] ).
verb( 'tower', 'towers', 'towering', 'towered', 'towered', intran, ['2C'] ).
verb( 'toy', 'toys', 'toying', 'toyed', 'toyed', intran, ['3A'] ).
verb( 'trace', 'traces', 'tracing', 'traced', 'traced', _, ['6A','15A','15B'] ).
verb( 'track', 'tracks', 'tracking', 'tracked', 'tracked', tran, ['6A','15A','15B'] ).
verb( 'trade', 'trades', 'trading', 'traded', 'traded', _, ['2A','2C','3A','14','15B'] ).
verb( 'traduce', 'traduces', 'traducing', 'traduced', 'traduced', tran, ['6A'] ).
verb( 'traffic', 'traffics', 'trafficking', 'trafficked', 'trafficked', intran, ['3A'] ).
verb( 'trail', 'trails', 'trailing', 'trailed', 'trailed', _, ['2A','2C','6A','15A','15B'] ).
verb( 'train', 'trains', 'training', 'trained', 'trained', _, ['2C','3A','6A','14','15A','17'] ).
verb( 'traipse', 'traipses', 'traipsing', 'traipsed', 'traipsed', intran, ['2A','2B','2C'] ).
verb( 'trammel', 'trammels', 'trammelling', 'trammelled', 'trammelled', tran, ['6A'] ).
verb( 'tramp', 'tramps', 'tramping', 'tramped', 'tramped', _, ['2A','2B','2C','6A'] ).
verb( 'trample', 'tramples', 'trampling', 'trampled', 'trampled', _, ['2C','3A','6A','15B'] ).
verb( 'tranquilize', 'tranquilizes', 'tranquilizing', 'tranquilized', 'tranquilized', tran, ['6A'] ).
verb( 'tranquillize', 'tranquillizes', 'tranquillizing', 'tranquillized', 'tranquillized', tran, ['6A'] ).
verb( 'transact', 'transacts', 'transacting', 'transacted', 'transacted', tran, ['6A','14'] ).
verb( 'transcend', 'transcends', 'transcending', 'transcended', 'transcended', tran, ['6A'] ).
verb( 'transcribe', 'transcribes', 'transcribing', 'transcribed', 'transcribed', tran, ['6A'] ).
verb( 'transfer', 'transfers', 'transferring', 'transferred', 'transferred', _, ['3A','6A','14'] ).
verb( 'transfigure', 'transfigures', 'transfiguring', 'transfigured', 'transfigured', tran, ['6A'] ).
verb( 'transfix', 'transfixes', 'transfixing', 'transfixed', 'transfixed', tran, ['6A'] ).
verb( 'transform', 'transforms', 'transforming', 'transformed', 'transformed', tran, ['6A','14'] ).
verb( 'transfuse', 'transfuses', 'transfusing', 'transfused', 'transfused', tran, ['6A'] ).
verb( 'transgress', 'transgresses', 'transgressing', 'transgressed', 'transgressed', _, ['2A','6A'] ).
verb( 'translate', 'translates', 'translating', 'translated', 'translated', tran, ['6A','14'] ).
verb( 'transliterate', 'transliterates', 'transliterating', 'transliterated', 'transliterated', tran, ['6A','14'] ).
verb( 'transmit', 'transmits', 'transmitting', 'transmitted', 'transmitted', tran, ['6A','14'] ).
verb( 'transmogrify', 'transmogrifies', 'transmogrifying', 'transmogrified', 'transmogrified', tran, ['6A'] ).
verb( 'transmute', 'transmutes', 'transmuting', 'transmuted', 'transmuted', tran, ['6A','14'] ).
verb( 'transpire', 'transpires', 'transpiring', 'transpired', 'transpired', _, ['2A','6A'] ).
verb( 'transplant', 'transplants', 'transplanting', 'transplanted', 'transplanted', _, ['2A','6A'] ).
verb( 'transport', 'transports', 'transporting', 'transported', 'transported', tran, ['6A','15A'] ).
verb( 'transpose', 'transposes', 'transposing', 'transposed', 'transposed', tran, ['6A','14'] ).
verb( 'transship', 'transships', 'transshipping', 'transshipped', 'transshipped', tran, ['6A'] ).
verb( 'trap', 'traps', 'trapping', 'trapped', 'trapped', tran, ['6A','15A'] ).
verb( 'travel', 'travels', 'travelling', 'travelled', 'travelled', _, ['2A','2B','2C','3A','4A'] ).
verb( 'traverse', 'traverses', 'traversing', 'traversed', 'traversed', tran, ['6A'] ).
verb( 'travesty', 'travesties', 'travestying', 'travestied', 'travestied', tran, ['6A'] ).
verb( 'trawl', 'trawls', 'trawling', 'trawled', 'trawled', _, ['2A','6A'] ).
verb( 'tread', 'treads', 'treading', 'trod', 'trodden', _, ['2C','3A','6A','15A','15B'] ).
verb( 'treadle', 'treadles', 'treadling', 'treadled', 'treadled', intran, ['2A'] ).
verb( 'treasure', 'treasures', 'treasuring', 'treasured', 'treasured', tran, ['6A','15B'] ).
verb( 'treat', 'treats', 'treating', 'treated', 'treated', _, ['3A','6A','14','15A','16B'] ).
verb( 'treble', 'trebles', 'trebling', 'trebled', 'trebled', _, ['2A','6A'] ).
verb( 'tree', 'trees', 'treeing', 'treed', 'treed', tran, ['6A'] ).
verb( 'trek', 'treks', 'trekking', 'trekked', 'trekked', intran, ['2A','2B','2C'] ).
verb( 'trellis', 'trellises', 'trellising', 'trellised', 'trellised', tran, ['6A'] ).
verb( 'tremble', 'trembles', 'trembling', 'trembled', 'trembled', intran, ['2A','2B','2C','4B'] ).
verb( 'trench', 'trenches', 'trenching', 'trenched', 'trenched', tran, ['6A'] ).
verb( 'trend', 'trends', 'trending', 'trended', 'trended', intran, ['2C'] ).
verb( 'trepan', 'trepans', 'trepanning', 'trepanned', 'trepanned', tran, [] ).
verb( 'trephine', 'trephines', 'trephining', 'trephined', 'trephined', tran, ['6A'] ).
verb( 'trespass', 'trespasses', 'trespassing', 'trespassed', 'trespassed', intran, ['2A','3A'] ).
verb( 'trice', 'trices', 'tricing', 'triced', 'triced', tran, ['15B'] ).
verb( 'trick', 'tricks', 'tricking', 'tricked', 'tricked', tran, ['6A','14','15B'] ).
verb( 'trickle', 'trickles', 'trickling', 'trickled', 'trickled', _, ['2A','2C','15A'] ).
verb( 'trifle', 'trifles', 'trifling', 'trifled', 'trifled', _, ['3A','15B'] ).
verb( 'trigger', 'triggers', 'triggering', 'triggered', 'triggered', tran, ['15B'] ).
verb( 'trill', 'trills', 'trilling', 'trilled', 'trilled', _, ['2A','2C','6A'] ).
verb( 'trim', 'trims', 'trimming', 'trimmed', 'trimmed', _, ['2A','6A','14','15A','22'] ).
verb( 'trip', 'trips', 'tripping', 'tripped', 'tripped', _, ['2A','2C','3A','15B'] ).
verb( 'triple', 'triples', 'tripling', 'tripled', 'tripled', _, ['2A','6A'] ).
verb( 'triplicate', 'triplicates', 'triplicating', 'triplicated', 'triplicated', tran, ['6A'] ).
verb( 'trisect', 'trisects', 'trisecting', 'trisected', 'trisected', tran, ['6A'] ).
verb( 'triumph', 'triumphs', 'triumphing', 'triumphed', 'triumphed', intran, ['2A','3A'] ).
verb( 'trivialize', 'trivializes', 'trivializing', 'trivialized', 'trivialized', tran, ['6A'] ).
verb( 'troll', 'trolls', 'trolling', 'trolled', 'trolled', _, ['2A','2C'] ).
verb( 'troop', 'troops', 'trooping', 'trooped', 'trooped', _, ['2C'] ).
verb( 'trot', 'trots', 'trotting', 'trotted', 'trotted', _, ['2A','2B','2C','15A','15B'] ).
verb( 'trouble', 'troubles', 'troubling', 'troubled', 'troubled', _, ['2A','2C','4A','6A','14','17'] ).
verb( 'trounce', 'trounces', 'trouncing', 'trounced', 'trounced', tran, ['6A'] ).
verb( 'truckle', 'truckles', 'truckling', 'truckled', 'truckled', intran, ['3A'] ).
verb( 'trudge', 'trudges', 'trudging', 'trudged', 'trudged', intran, ['2A','2B','2C'] ).
verb( 'true', 'trues', 'truing', 'trued', 'trued', tran, ['15B'] ).
verb( 'trump', 'trumps', 'trumping', 'trumped', 'trumped', _, ['2A','6A','15B'] ).
verb( 'trumpet', 'trumpets', 'trumpeting', 'trumpeted', 'trumpeted', _, ['2A','2C','6A','15B'] ).
verb( 'truncate', 'truncates', 'truncating', 'truncated', 'truncated', tran, ['6A'] ).
verb( 'trundle', 'trundles', 'trundling', 'trundled', 'trundled', _, ['2C','6A','15A','15B'] ).
verb( 'truss', 'trusses', 'trussing', 'trussed', 'trussed', tran, ['6A','15B'] ).
verb( 'trust', 'trusts', 'trusting', 'trusted', 'trusted', _, ['3A','6A','7A','9','14','15A','15B','17'] ).
verb( 'try', 'tries', 'trying', 'tried', 'tried', _, ['2A','2B','3A','6A','6B','6C','7A','10','15A','15B'] ).
verb( 'tuck', 'tucks', 'tucking', 'tucked', 'tucked', _, ['2C','3A','15A','15B'] ).
verb( 'tug', 'tugs', 'tugging', 'tugged', 'tugged', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'tumble', 'tumbles', 'tumbling', 'tumbled', 'tumbled', _, ['2A','2C','3A','6A','15A','15B'] ).
verb( 'tune', 'tunes', 'tuning', 'tuned', 'tuned', _, ['2C','6A'] ).
verb( 'tunnel', 'tunnels', 'tunnelling', 'tunnelled', 'tunnelled', _, ['2A','2C','3A','6A'] ).
verb( 'turf', 'turfs', 'turfing', 'turfed', 'turfed', tran, ['6A','15B'] ).
verb( 'turn', 'turns', 'turning', 'turned', 'turned', _, ['2A','2C','3A','4A','6A','14','15A','15B'] ).
verb( 'tussle', 'tussles', 'tussling', 'tussled', 'tussled', intran, [] ).
verb( 'tut', 'tuts', 'tutting', 'tutted', 'tutted', tran, ['6A'] ).
verb( 'tutor', 'tutors', 'tutoring', 'tutored', 'tutored', tran, ['6A','15A','16A'] ).
verb( 'twaddle', 'twaddles', 'twaddling', 'twaddled', 'twaddled', intran, ['2A'] ).
verb( 'twang', 'twangs', 'twanging', 'twanged', 'twanged', _, ['2A','6A'] ).
verb( 'tweak', 'tweaks', 'tweaking', 'tweaked', 'tweaked', tran, ['6A'] ).
verb( 'tweet', 'tweets', 'tweeting', 'tweeted', 'tweeted', intran, [] ).
verb( 'twiddle', 'twiddles', 'twiddling', 'twiddled', 'twiddled', _, ['3A','6A'] ).
verb( 'twig', 'twigs', 'twigging', 'twigged', 'twigged', _, ['2A','6A'] ).
verb( 'twin', 'twins', 'twinning', 'twinned', 'twinned', tran, ['6A','14'] ).
verb( 'twine', 'twines', 'twining', 'twined', 'twined', _, ['2A','2C','15A','15B'] ).
verb( 'twinkle', 'twinkles', 'twinkling', 'twinkled', 'twinkled', intran, ['2A','2C'] ).
verb( 'twirl', 'twirls', 'twirling', 'twirled', 'twirled', _, ['2A','2C','6A','15B','16A'] ).
verb( 'twist', 'twists', 'twisting', 'twisted', 'twisted', _, ['2A','2C','6A','15A','15B','16A'] ).
verb( 'twit', 'twits', 'twitting', 'twitted', 'twitted', tran, ['6A','14'] ).
verb( 'twitch', 'twitches', 'twitching', 'twitched', 'twitched', _, ['2A','2C','6A','15A','15B'] ).
verb( 'twitter', 'twitters', 'twittering', 'twittered', 'twittered', intran, ['15A','15B'] ).
verb( 'type', 'types', 'typing', 'typed', 'typed', _, ['2A','6A'] ).
verb( 'typecast', 'typecasts', 'typecasting', 'typecast', 'typecast', tran, ['6A'] ).
verb( 'typify', 'typifies', 'typifying', 'typified', 'typified', tran, ['6A'] ).
verb( 'tyrannize', 'tyrannizes', 'tyrannizing', 'tyrannized', 'tyrannized', _, ['3A','6A'] ).
verb( 'uglify', 'uglifies', 'uglifying', 'uglified', 'uglified', tran, [] ).
verb( 'ulcerate', 'ulcerates', 'ulcerating', 'ulcerated', 'ulcerated', _, ['2A','6A'] ).
verb( 'ululate', 'ululates', 'ululating', 'ululated', 'ululated', intran, ['2A'] ).
verb( 'umpire', 'umpires', 'umpiring', 'umpired', 'umpired', _, ['2A','6A'] ).
verb( 'unbalance', 'unbalances', 'unbalancing', 'unbalanced', 'unbalanced', tran, ['6A'] ).
verb( 'unbar', 'unbars', 'unbarring', 'unbarred', 'unbarred', tran, [] ).
verb( 'unbend', 'unbends', 'unbending', 'unbended', 'unbended', _, ['2A','6A'] ).
verb( 'unbind', 'unbinds', 'unbinding', 'unbound', 'unbound', tran, ['6A'] ).
verb( 'unblock', 'unblocks', 'unblocking', 'unblocked', 'unblocked', tran, ['6A','15B'] ).
verb( 'unbolt', 'unbolts', 'unbolting', 'unbolted', 'unbolted', _, ['2A','2C','6A','15B'] ).
verb( 'unbosom', 'unbosoms', 'unbosoming', 'unbosomed', 'unbosomed', tran, ['6A','14'] ).
verb( 'unbuckle', 'unbuckles', 'unbuckling', 'unbuckled', 'unbuckled', tran, ['6A'] ).
verb( 'unburden', 'unburdens', 'unburdening', 'unburdened', 'unburdened', tran, ['6A','15A'] ).
verb( 'unbutton', 'unbuttons', 'unbuttoning', 'unbuttoned', 'unbuttoned', tran, ['6A'] ).
verb( 'unchain', 'unchains', 'unchaining', 'unchained', 'unchained', tran, ['6A','15A','15B'] ).
verb( 'uncork', 'uncorks', 'uncorking', 'uncorked', 'uncorked', tran, ['6A'] ).
verb( 'uncouple', 'uncouples', 'uncoupling', 'uncoupled', 'uncoupled', tran, ['6A'] ).
verb( 'uncover', 'uncovers', 'uncovering', 'uncovered', 'uncovered', tran, ['2A','6A'] ).
verb( 'uncross', 'uncrosses', 'uncrossing', 'uncrossed', 'uncrossed', tran, ['6A'] ).
verb( 'uncurl', 'uncurls', 'uncurling', 'uncurled', 'uncurled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'undeceive', 'undeceives', 'undeceiving', 'undeceived', 'undeceived', tran, ['6A'] ).
verb( 'underact', 'underacts', 'underacting', 'underacted', 'underacted', _, ['2A','6A'] ).
verb( 'underbid', 'underbids', 'underbidding', 'underbid', 'underbid', tran, [] ).
verb( 'undercharge', 'undercharges', 'undercharging', 'undercharged', 'undercharged', tran, ['6A'] ).
verb( 'undercut', 'undercuts', 'undercutting', 'undercut', 'undercut', tran, ['6A'] ).
verb( 'underestimate', 'underestimates', 'underestimating', 'underestimated', 'underestimated', tran, ['6A'] ).
verb( 'underexpose', 'underexposes', 'underexposing', 'underexposed', 'underexposed', tran, ['6A'] ).
verb( 'undergo', 'undergoes', 'undergoing', 'underwent', 'undergone', tran, ['6A'] ).
verb( 'underlie', 'underlies', 'underlying', 'underlay', 'underlain', tran, ['6A'] ).
verb( 'underline', 'underlines', 'underlining', 'underlined', 'underlined', tran, ['6A'] ).
verb( 'undermine', 'undermines', 'undermining', 'undermined', 'undermined', tran, ['6A'] ).
verb( 'underpay', 'underpays', 'underpaying', 'underpaid', 'underpaid', tran, ['6A'] ).
verb( 'underpin', 'underpins', 'underpinning', 'underpinned', 'underpinned', tran, [] ).
verb( 'underquote', 'underquotes', 'underquoting', 'underquoted', 'underquoted', tran, ['6A'] ).
verb( 'underrate', 'underrates', 'underrating', 'underrated', 'underrated', tran, ['6A'] ).
verb( 'underscore', 'underscores', 'underscoring', 'underscored', 'underscored', tran, ['6A'] ).
verb( 'undersell', 'undersells', 'underselling', 'undersold', 'undersold', tran, ['6A'] ).
verb( 'undershoot', 'undershoots', 'undershooting', 'undershot', 'undershot', tran, [] ).
verb( 'undersign', 'undersigns', 'undersigning', 'undersigned', 'undersigned', tran, ['6A'] ).
verb( 'underspend', 'underspends', 'underspending', 'underspent', 'underspent', _, ['2A','6A','14','19B'] ).
verb( 'understand', 'understands', 'understanding', 'understood', 'understood', _, ['2A','6A','6C','8','9','10','17','19C'] ).
verb( 'understate', 'understates', 'understating', 'understated', 'understated', tran, ['6A'] ).
verb( 'understock', 'understocks', 'understocking', 'understocked', 'understocked', tran, ['6A'] ).
verb( 'understudy', 'understudies', 'understudying', 'understudied', 'understudied', tran, ['6A'] ).
verb( 'undertake', 'undertakes', 'undertaking', 'undertook', 'undertaken', tran, ['6A','7A','9'] ).
verb( 'undervalue', 'undervalues', 'undervaluing', 'undervalued', 'undervalued', tran, ['6A'] ).
verb( 'underwrite', 'underwrites', 'underwriting', 'underwrote', 'underwritten', tran, ['6A'] ).
verb( 'undo', 'undoes', 'undoing', 'undid', 'undone', tran, ['6A'] ).
verb( 'undock', 'undocks', 'undocking', 'undocked', 'undocked', _, ['2A','6A'] ).
verb( 'undress', 'undresses', 'undressing', 'undressed', 'undressed', _, ['2A','6A'] ).
verb( 'undulate', 'undulates', 'undulating', 'undulated', 'undulated', intran, ['2A','2C'] ).
verb( 'unearth', 'unearths', 'unearthing', 'unearthed', 'unearthed', tran, ['6A'] ).
verb( 'unfasten', 'unfastens', 'unfastening', 'unfastened', 'unfastened', _, ['2A','2C','3A','6A','14','15A','15B'] ).
verb( 'unfit', 'unfits', 'unfitting', 'unfitted', 'unfitted', tran, ['14'] ).
verb( 'unfold', 'unfolds', 'unfolding', 'unfolded', 'unfolded', _, ['2A','2C','6A'] ).
verb( 'unfrock', 'unfrocks', 'unfrocking', 'unfrocked', 'unfrocked', tran, ['6A'] ).
verb( 'unfurl', 'unfurls', 'unfurling', 'unfurled', 'unfurled', _, ['2A','6A'] ).
verb( 'unhand', 'unhands', 'unhanding', 'unhanded', 'unhanded', tran, ['6A'] ).
verb( 'unhinge', 'unhinges', 'unhinging', 'unhinged', 'unhinged', tran, ['6A'] ).
verb( 'unhitch', 'unhitches', 'unhitching', 'unhitched', 'unhitched', _, ['2A','2C','6A','15A','15B'] ).
verb( 'unhook', 'unhooks', 'unhooking', 'unhooked', 'unhooked', tran, ['6A'] ).
verb( 'unhorse', 'unhorses', 'unhorsing', 'unhorsed', 'unhorsed', tran, ['6A'] ).
verb( 'unify', 'unifies', 'unifying', 'unified', 'unified', tran, ['6A'] ).
verb( 'unite', 'unites', 'uniting', 'united', 'united', _, ['2A','3A','4A','6A'] ).
verb( 'unlearn', 'unlearns', 'unlearning', 'unlearned', 'unlearned', tran, ['6A'] ).
verb( 'unleash', 'unleashes', 'unleashing', 'unleashed', 'unleashed', tran, ['6A'] ).
verb( 'unload', 'unloads', 'unloading', 'unloaded', 'unloaded', _, ['2A','6A','14'] ).
verb( 'unlock', 'unlocks', 'unlocking', 'unlocked', 'unlocked', _, ['2A','2C','6A','15A','15B'] ).
verb( 'unloose', 'unlooses', 'unloosing', 'unloosed', 'unloosed', tran, ['6A'] ).
verb( 'unman', 'unmans', 'unmanning', 'unmanned', 'unmanned', tran, ['6A'] ).
verb( 'unmask', 'unmasks', 'unmasking', 'unmasked', 'unmasked', _, ['2A','6A'] ).
verb( 'unnerve', 'unnerves', 'unnerving', 'unnerved', 'unnerved', tran, ['6A'] ).
verb( 'unpack', 'unpacks', 'unpacking', 'unpacked', 'unpacked', _, ['2A','6A'] ).
verb( 'unpick', 'unpicks', 'unpicking', 'unpicked', 'unpicked', _, ['3A','6A','15B'] ).
verb( 'unplug', 'unplugs', 'unplugging', 'unplugged', 'unplugged', _, ['2C','6A','15B'] ).
verb( 'unravel', 'unravels', 'unravelling', 'unravelled', 'unravelled', _, ['2A','6A'] ).
verb( 'unroll', 'unrolls', 'unrolling', 'unrolled', 'unrolled', _, ['2A','6A'] ).
verb( 'unsaddle', 'unsaddles', 'unsaddling', 'unsaddled', 'unsaddled', tran, ['6A'] ).
verb( 'unsay', 'unsays', 'unsaying', 'unsaid', 'unsaid', tran, ['6A'] ).
verb( 'unscramble', 'unscrambles', 'unscrambling', 'unscrambled', 'unscrambled', tran, [] ).
verb( 'unscrew', 'unscrews', 'unscrewing', 'unscrewed', 'unscrewed', _, ['2A','6A','15A','15B'] ).
verb( 'unseat', 'unseats', 'unseating', 'unseated', 'unseated', tran, ['6A'] ).
verb( 'unsettle', 'unsettles', 'unsettling', 'unsettled', 'unsettled', tran, ['6A'] ).
verb( 'unsex', 'unsexes', 'unsexing', 'unsexed', 'unsexed', tran, ['6A'] ).
verb( 'unsheathe', 'unsheathes', 'unsheathing', 'unsheathed', 'unsheathed', tran, ['6A'] ).
verb( 'untie', 'unties', 'untying', 'untied', 'untied', _, ['2A','3A','6A','15A','15B'] ).
verb( 'unveil', 'unveils', 'unveiling', 'unveiled', 'unveiled', _, ['2A','6A'] ).
verb( 'unwind', 'unwinds', 'unwinding', 'unwound', 'unwound', _, ['2A','6A','21'] ).
verb( 'unwrap', 'unwraps', 'unwrapping', 'unwrapped', 'unwrapped', _, ['6A','14','15A','15B'] ).
verb( 'unzip', 'unzips', 'unzipping', 'unzipped', 'unzipped', tran, [] ).
verb( 'up', 'ups', 'upping', 'upped', 'upped', _, ['2A','6A'] ).
verb( 'upbraid', 'upbraids', 'upbraiding', 'upbraided', 'upbraided', tran, ['6A','14'] ).
verb( 'update', 'updates', 'updating', 'updated', 'updated', tran, ['6A'] ).
verb( 'upgrade', 'upgrades', 'upgrading', 'upgraded', 'upgraded', tran, ['6A'] ).
verb( 'uphold', 'upholds', 'upholding', 'upheld', 'upheld', tran, ['6A'] ).
verb( 'upholster', 'upholsters', 'upholstering', 'upholstered', 'upholstered', tran, ['6A'] ).
verb( 'uplift', 'uplifts', 'uplifting', 'uplifted', 'uplifted', tran, ['6A'] ).
verb( 'uproot', 'uproots', 'uprooting', 'uprooted', 'uprooted', tran, ['6A'] ).
verb( 'upset', 'upsets', 'upsetting', 'upset', 'upset', _, ['2A','6A'] ).
verb( 'upstage', 'upstages', 'upstaging', 'upstaged', 'upstaged', tran, ['6A'] ).
verb( 'urbanize', 'urbanizes', 'urbanizing', 'urbanized', 'urbanized', tran, ['6A'] ).
verb( 'urge', 'urges', 'urging', 'urged', 'urged', tran, ['6A','6D','9','14','15B','17','19C'] ).
verb( 'urinate', 'urinates', 'urinating', 'urinated', 'urinated', intran, [] ).
verb( 'use', 'uses', 'using', 'used', 'used', tran, ['6A','14','15A','15B','16A'] ).
verb( 'usher', 'ushers', 'ushering', 'ushered', 'ushered', tran, ['14','15B'] ).
verb( 'usurp', 'usurps', 'usurping', 'usurped', 'usurped', tran, ['6A'] ).
verb( 'utilize', 'utilizes', 'utilizing', 'utilized', 'utilized', tran, ['6A'] ).
verb( 'utter', 'utters', 'uttering', 'uttered', 'uttered', tran, ['6A'] ).
verb( 'vacate', 'vacates', 'vacating', 'vacated', 'vacated', tran, ['6A'] ).
verb( 'vacation', 'vacations', 'vacationing', 'vacationed', 'vacationed', intran, ['3A'] ).
verb( 'vaccinate', 'vaccinates', 'vaccinating', 'vaccinated', 'vaccinated', tran, ['6A','14'] ).
verb( 'vacillate', 'vacillates', 'vacillating', 'vacillated', 'vacillated', intran, ['2A','3A'] ).
verb( 'valet', 'valets', 'valeting', 'valeted', 'valeted', tran, ['6A'] ).
verb( 'validate', 'validates', 'validating', 'validated', 'validated', tran, ['6A'] ).
verb( 'value', 'values', 'valuing', 'valued', 'valued', tran, ['6A','15A','16B'] ).
verb( 'vamoose', 'vamooses', 'vamoosing', 'vamoosed', 'vamoosed', intran, [] ).
verb( 'vamp', 'vamps', 'vamping', 'vamped', 'vamped', _, ['2A','6A','15B'] ).
verb( 'vanish', 'vanishes', 'vanishing', 'vanished', 'vanished', intran, ['2A'] ).
verb( 'vanquish', 'vanquishes', 'vanquishing', 'vanquished', 'vanquished', tran, ['6A'] ).
verb( 'vaporize', 'vaporizes', 'vaporizing', 'vaporized', 'vaporized', _, ['2A','6A'] ).
verb( 'varnish', 'varnishes', 'varnishing', 'varnished', 'varnished', tran, ['6A'] ).
verb( 'vary', 'varies', 'varying', 'varied', 'varied', _, ['2A','6A'] ).
verb( 'vault', 'vaults', 'vaulting', 'vaulted', 'vaulted', _, ['2A','2B','2C','6A'] ).
verb( 'vaunt', 'vaunts', 'vaunting', 'vaunted', 'vaunted', _, [] ).
verb( 'veer', 'veers', 'veering', 'veered', 'veered', intran, ['2A','2C'] ).
verb( 'vegetate', 'vegetates', 'vegetating', 'vegetated', 'vegetated', intran, ['2A'] ).
verb( 'veil', 'veils', 'veiling', 'veiled', 'veiled', tran, ['6A'] ).
verb( 'vend', 'vends', 'vending', 'vended', 'vended', tran, ['6A'] ).
verb( 'veneer', 'veneers', 'veneering', 'veneered', 'veneered', tran, ['6A'] ).
verb( 'venerate', 'venerates', 'venerating', 'venerated', 'venerated', tran, ['6A'] ).
verb( 'vent', 'vents', 'venting', 'vented', 'vented', tran, ['6A','14'] ).
verb( 'ventilate', 'ventilates', 'ventilating', 'ventilated', 'ventilated', tran, ['6A'] ).
verb( 'venture', 'ventures', 'venturing', 'ventured', 'ventured', _, ['3A','6A','7A','15A','16A'] ).
verb( 'verbalize', 'verbalizes', 'verbalizing', 'verbalized', 'verbalized', tran, [] ).
verb( 'verge', 'verges', 'verging', 'verged', 'verged', intran, ['3A'] ).
verb( 'verify', 'verifies', 'verifying', 'verified', 'verified', tran, ['6A'] ).
verb( 'versify', 'versifies', 'versifying', 'versified', 'versified', _, ['2A','6A'] ).
verb( 'vest', 'vests', 'vesting', 'vested', 'vested', _, ['3A','6A','14'] ).
verb( 'vesture', 'vestures', 'vesturing', 'vestured', 'vestured', tran, [] ).
verb( 'vet', 'vets', 'vetting', 'vetted', 'vetted', tran, ['6A'] ).
verb( 'veto', 'vetos', 'vetoing', 'vetoed', 'vetoed', tran, ['6A'] ).
verb( 'vex', 'vexes', 'vexing', 'vexed', 'vexed', tran, ['6A'] ).
verb( 'vibrate', 'vibrates', 'vibrating', 'vibrated', 'vibrated', _, ['2A','2C','6A'] ).
verb( 'victimize', 'victimizes', 'victimizing', 'victimized', 'victimized', tran, ['6A'] ).
verb( 'victual', 'victuals', 'victualling', 'victualled', 'victualled', _, ['2A','6A'] ).
verb( 'videotape', 'videotapes', 'videotaping', 'videotaped', 'videotaped', tran, ['6A'] ).
verb( 'vie', 'vies', 'vying', 'vied', 'vied', intran, ['3A'] ).
verb( 'view', 'views', 'viewing', 'viewed', 'viewed', tran, ['6A'] ).
verb( 'vilify', 'vilifies', 'vilifying', 'vilified', 'vilified', tran, ['6A'] ).
verb( 'vindicate', 'vindicates', 'vindicating', 'vindicated', 'vindicated', tran, ['6A'] ).
verb( 'violate', 'violates', 'violating', 'violated', 'violated', tran, ['6A'] ).
verb( 'visa', 'visas', 'visaing', 'visaed', 'visaed', tran, ['6A'] ).
verb( 'visit', 'visits', 'visiting', 'visited', 'visited', _, ['2C','3A','6A','14'] ).
verb( 'visualize', 'visualizes', 'visualizing', 'visualized', 'visualized', tran, ['6A'] ).
verb( 'vitalize', 'vitalizes', 'vitalizing', 'vitalized', 'vitalized', tran, ['6A'] ).
verb( 'vitiate', 'vitiates', 'vitiating', 'vitiated', 'vitiated', tran, ['6A'] ).
verb( 'vitrify', 'vitrifies', 'vitrifying', 'vitrified', 'vitrified', _, ['2A','6A'] ).
verb( 'vituperate', 'vituperates', 'vituperating', 'vituperated', 'vituperated', tran, ['6A'] ).
verb( 'vivisect', 'vivisects', 'vivisecting', 'vivisected', 'vivisected', tran, ['6A'] ).
verb( 'vocalize', 'vocalizes', 'vocalizing', 'vocalized', 'vocalized', tran, [] ).
verb( 'vociferate', 'vociferates', 'vociferating', 'vociferated', 'vociferated', _, ['2A','6A'] ).
verb( 'voice', 'voices', 'voicing', 'voiced', 'voiced', tran, ['6A'] ).
verb( 'void', 'voids', 'voiding', 'voided', 'voided', tran, ['6A'] ).
verb( 'volley', 'volleys', 'volleying', 'volleyed', 'volleyed', _, ['2A','2C','6A'] ).
verb( 'volunteer', 'volunteers', 'volunteering', 'volunteered', 'volunteered', _, ['2A','3A','6A','7A'] ).
verb( 'vomit', 'vomits', 'vomiting', 'vomited', 'vomited', _, ['2A','6A','15B'] ).
verb( 'vote', 'votes', 'voting', 'voted', 'voted', _, ['3A','6A','9','12B','13B','15B','25'] ).
verb( 'vouch', 'vouches', 'vouching', 'vouched', 'vouched', intran, [] ).
verb( 'vouchsafe', 'vouchsafes', 'vouchsafing', 'vouchsafed', 'vouchsafed', tran, ['6A','7A','12C'] ).
verb( 'vow', 'vows', 'vowing', 'vowed', 'vowed', tran, ['6A','7A','9'] ).
verb( 'voyage', 'voyages', 'voyaging', 'voyaged', 'voyaged', intran, ['2A','2C'] ).
verb( 'vulcanize', 'vulcanizes', 'vulcanizing', 'vulcanized', 'vulcanized', tran, ['6A'] ).
verb( 'vulgarize', 'vulgarizes', 'vulgarizing', 'vulgarized', 'vulgarized', tran, ['6A'] ).
verb( 'wad', 'wads', 'wadding', 'wadded', 'wadded', tran, [] ).
verb( 'waddle', 'waddles', 'waddling', 'waddled', 'waddled', intran, ['2A','2C'] ).
verb( 'wade', 'wades', 'wading', 'waded', 'waded', _, ['2A','2C','6A'] ).
verb( 'waffle', 'waffles', 'waffling', 'waffled', 'waffled', intran, ['2A','2C'] ).
verb( 'waft', 'wafts', 'wafting', 'wafted', 'wafted', tran, ['6A'] ).
verb( 'wag', 'wags', 'wagging', 'wagged', 'wagged', _, ['2A','2C','6A'] ).
verb( 'wage', 'wages', 'waging', 'waged', 'waged', tran, ['6A'] ).
verb( 'wager', 'wagers', 'wagering', 'wagered', 'wagered', _, ['2A','6A','11','12C','14'] ).
verb( 'waggle', 'waggles', 'waggling', 'waggled', 'waggled', _, [] ).
verb( 'wail', 'wails', 'wailing', 'wailed', 'wailed', _, ['2A','2B','2C','6A'] ).
verb( 'wait', 'waits', 'waiting', 'waited', 'waited', _, ['2A','2B','2C','3A','4A','6A','14'] ).
verb( 'waive', 'waives', 'waiving', 'waived', 'waived', tran, ['6A'] ).
verb( 'wake', 'wakes', 'waking', 'waked', 'waked', _, ['2A','2C','4B','6A','15B'] ).
verb( 'waken', 'wakens', 'wakening', 'wakened', 'wakened', _, ['2A','6A'] ).
verb( 'walk', 'walks', 'walking', 'walked', 'walked', _, ['2A','2B','2C','6A','15A','15B'] ).
verb( 'wall', 'walls', 'walling', 'walled', 'walled', tran, ['15B'] ).
verb( 'wallop', 'wallops', 'walloping', 'walloped', 'walloped', tran, [] ).
verb( 'wallow', 'wallows', 'wallowing', 'wallowed', 'wallowed', intran, ['2A','2C'] ).
verb( 'waltz', 'waltzes', 'waltzing', 'waltzed', 'waltzed', _, ['2A','2C','15A'] ).
verb( 'wander', 'wanders', 'wandering', 'wandered', 'wandered', _, ['2A','2B','2C','6A'] ).
verb( 'wane', 'wanes', 'waning', 'waned', 'waned', intran, ['2A'] ).
verb( 'wangle', 'wangles', 'wangling', 'wangled', 'wangled', tran, ['6A'] ).
verb( 'wank', 'wanks', 'wanking', 'wanked', 'wanked', intran, ['2A'] ).
verb( 'want', 'wants', 'wanting', 'wanted', 'wanted', _, ['2A','3A','6A','6E','7A','17','19B','24A'] ).
verb( 'wanton', 'wantons', 'wantoning', 'wantoned', 'wantoned', intran, ['2A','2C'] ).
verb( 'war', 'wars', 'warring', 'warred', 'warred', intran, [] ).
verb( 'warble', 'warbles', 'warbling', 'warbled', 'warbled', _, ['2A','2C','6A'] ).
verb( 'ward', 'wards', 'warding', 'warded', 'warded', tran, ['15B'] ).
verb( 'ware', 'wares', 'waring', 'wared', 'wared', tran, ['6A'] ).
verb( 'warm', 'warms', 'warming', 'warmed', 'warmed', _, ['2A','2C','6A','15B'] ).
verb( 'warn', 'warns', 'warning', 'warned', 'warned', tran, ['6A','11','14','15B','17'] ).
verb( 'warp', 'warps', 'warping', 'warped', 'warped', _, ['2A','6A'] ).
verb( 'warrant', 'warrants', 'warranting', 'warranted', 'warranted', tran, ['6A','9','25'] ).
verb( 'wash', 'washes', 'washing', 'washed', 'washed', _, ['2A','2C','6A','15A','15B','22'] ).
verb( 'waste', 'wastes', 'wasting', 'wasted', 'wasted', _, ['2A','2C','6A','14'] ).
verb( 'watch', 'watches', 'watching', 'watched', 'watched', _, ['2A','2B','2C','3A','4A','6A','8','10','15A','18A','19A'] ).
verb( 'water', 'waters', 'watering', 'watered', 'watered', _, ['2A','6A','15B'] ).
verb( 'waterproof', 'waterproofs', 'waterproofing', 'waterproofed', 'waterproofed', tran, [] ).
verb( 'wave', 'waves', 'waving', 'waved', 'waved', _, ['2A','3A','6A','12A','13A','15A','15B','16A'] ).
verb( 'waver', 'wavers', 'wavering', 'wavered', 'wavered', intran, ['2A','2C'] ).
verb( 'wax', 'waxes', 'waxing', 'waxed', 'waxed', _, ['2A','2D','6A'] ).
verb( 'waylay', 'waylays', 'waylaying', 'waylaid', 'waylaid', tran, ['6A'] ).
verb( 'weaken', 'weakens', 'weakening', 'weakened', 'weakened', _, ['2A','6A'] ).
verb( 'wean', 'weans', 'weaning', 'weaned', 'weaned', tran, ['6A','14'] ).
verb( 'wear', 'wears', 'wearing', 'wore', 'worn', _, ['2A','2B','2C','2D','6A','15A','15B','22'] ).
verb( 'weary', 'wearies', 'wearying', 'wearied', 'wearied', _, ['2A','3A','6A','14'] ).
verb( 'weather', 'weathers', 'weathering', 'weathered', 'weathered', _, ['2A','6A'] ).
verb( 'weave', 'weaves', 'weaving', 'wove', 'woven', _, ['2A','2C','6A','15A','15B'] ).
verb( 'wed', 'weds', 'wedding', 'wedded', 'wedded', _, ['2A','6A','14'] ).
verb( 'wedge', 'wedges', 'wedging', 'wedged', 'wedged', tran, ['6A','15A','22'] ).
verb( 'wee', 'wees', 'weeing', 'weed', 'weed', intran, [] ).
verb( 'weed', 'weeds', 'weeding', 'weeded', 'weeded', _, ['2A','6A','15B'] ).
verb( 'weekend', 'weekends', 'weekending', 'weekended', 'weekended', intran, [] ).
verb( 'weep', 'weeps', 'weeping', 'wept', 'wept', _, ['2A','2B','2C','3A','4B','6A'] ).
verb( 'weigh', 'weighs', 'weighing', 'weighed', 'weighed', _, ['2B','2C','3A','6A','14','15B'] ).
verb( 'weight', 'weights', 'weighting', 'weighted', 'weighted', tran, ['6A'] ).
verb( 'welcome', 'welcomes', 'welcoming', 'welcomed', 'welcomed', tran, ['6A','15A'] ).
verb( 'weld', 'welds', 'welding', 'welded', 'welded', _, ['2A','6A','15A','15B'] ).
verb( 'well', 'wells', 'welling', 'welled', 'welled', intran, ['2C'] ).
verb( 'welsh', 'welshes', 'welshing', 'welshed', 'welshed', intran, ['3A'] ).
verb( 'welter', 'welters', 'weltering', 'weltered', 'weltered', intran, ['2C'] ).
verb( 'wench', 'wenches', 'wenching', 'wenched', 'wenched', intran, ['2A'] ).
verb( 'wend', 'wends', 'wending', 'wended', 'wended', tran, [] ).
verb( 'westernize', 'westernizes', 'westernizing', 'westernized', 'westernized', tran, ['6A'] ).
verb( 'wet', 'wets', 'wetting', 'wetted', 'wetted', tran, ['6A'] ).
verb( 'whack', 'whacks', 'whacking', 'whacked', 'whacked', tran, ['6A'] ).
verb( 'whale', 'whales', 'whaling', 'whaled', 'whaled', intran, ['2A'] ).
verb( 'whang', 'whangs', 'whanging', 'whanged', 'whanged', tran, [] ).
verb( 'wheedle', 'wheedles', 'wheedling', 'wheedled', 'wheedled', tran, ['6A','14'] ).
verb( 'wheel', 'wheels', 'wheeling', 'wheeled', 'wheeled', _, ['2A','2C','6A','15A','15B'] ).
verb( 'wheeze', 'wheezes', 'wheezing', 'wheezed', 'wheezed', _, ['2A','2B','2C','15B'] ).
verb( 'whelp', 'whelps', 'whelping', 'whelped', 'whelped', intran, [] ).
verb( 'whet', 'whets', 'whetting', 'whetted', 'whetted', tran, ['6A'] ).
verb( 'while', 'whiles', 'whiling', 'whiled', 'whiled', tran, ['15B'] ).
verb( 'whimper', 'whimpers', 'whimpering', 'whimpered', 'whimpered', _, ['2A','6A'] ).
verb( 'whine', 'whines', 'whining', 'whined', 'whined', _, ['2A','2C','4A','6A','15B'] ).
verb( 'whinny', 'whinnies', 'whinnying', 'whinnied', 'whinnied', intran, [] ).
verb( 'whip', 'whips', 'whipping', 'whipped', 'whipped', _, ['2C','6A','15A','15B'] ).
verb( 'whir', 'whirs', 'whirring', 'whirred', 'whirred', intran, ['2A','2C'] ).
verb( 'whirl', 'whirls', 'whirling', 'whirled', 'whirled', _, ['2A','2C','15A','15B'] ).
verb( 'whirr', 'whirrs', 'whirring', 'whirred', 'whirred', intran, ['2A','2C'] ).
verb( 'whisk', 'whisks', 'whisking', 'whisked', 'whisked', _, ['6A','15B'] ).
verb( 'whisper', 'whispers', 'whispering', 'whispered', 'whispered', _, ['2A','2C','3A','6A','14','15B'] ).
verb( 'whistle', 'whistles', 'whistling', 'whistled', 'whistled', _, ['2A','2C','6A','15B','16A'] ).
verb( 'whiten', 'whitens', 'whitening', 'whitened', 'whitened', _, ['2A','6A'] ).
verb( 'whitewash', 'whitewashes', 'whitewashing', 'whitewashed', 'whitewashed', tran, [] ).
verb( 'whittle', 'whittles', 'whittling', 'whittled', 'whittled', _, ['2C','3A','6A','15A','15B'] ).
verb( 'whiz', 'whizzes', 'whizzing', 'whizzed', 'whizzed', intran, ['2C'] ).
verb( 'whoop', 'whoops', 'whooping', 'whooped', 'whooped', _, [] ).
verb( 'whop', 'whops', 'whopping', 'whopped', 'whopped', tran, ['6A'] ).
verb( 'widen', 'widens', 'widening', 'widened', 'widened', _, ['2A','6A'] ).
verb( 'wield', 'wields', 'wielding', 'wielded', 'wielded', tran, ['6A'] ).
verb( 'wiggle', 'wiggles', 'wiggling', 'wiggled', 'wiggled', _, ['2A','6A'] ).
verb( 'will', 'will', '-', '-', '-', unknown, ['2A','5','6A','9','12A','13A','14','15A','17'] ).
verb( 'will', 'wills', 'willing', 'willed', 'willed', _, ['2A','5','6A','9','12A','13A','14','15A','17'] ).
verb( 'wilt', 'wilts', 'wilting', 'wilted', 'wilted', _, ['2A','5','6A','9'] ).
verb( 'win', 'wins', 'winning', 'won', 'won', _, ['2A','6A','12B','13B','15A','15B','17'] ).
verb( 'wince', 'winces', 'wincing', 'winced', 'winced', tran, ['2A','2C'] ).
verb( 'winch', 'winches', 'winching', 'winched', 'winched', tran, ['6A','15B'] ).
verb( 'wind', 'winds', 'winding', 'winded', 'winded', tran, ['6A'] ).
verb( 'wind', 'winds', 'winding', 'wound', 'wound', _, ['2A','2B','2C','6A','14','15A','15B'] ).
verb( 'wine', 'wines', 'wining', 'wined', 'wined', tran, [] ).
verb( 'wing', 'wings', 'winging', 'winged', 'winged', _, ['2C','6A','15A'] ).
verb( 'wink', 'winks', 'winking', 'winked', 'winked', _, ['2A','2C','3A','15B'] ).
verb( 'winkle', 'winkles', 'winkling', 'winkled', 'winkled', tran, ['15B'] ).
verb( 'winnow', 'winnows', 'winnowing', 'winnowed', 'winnowed', tran, ['6A','14','15A','15B'] ).
verb( 'winter', 'winters', 'wintering', 'wintered', 'wintered', intran, ['2C'] ).
verb( 'wipe', 'wipes', 'wiping', 'wiped', 'wiped', _, ['6A','15A','15B','22'] ).
verb( 'wire', 'wires', 'wiring', 'wired', 'wired', _, ['6A','11','12A','13A','15A','15B','16A'] ).
verb( 'wisecrack', 'wisecracks', 'wisecracking', 'wisecracked', 'wisecracked', intran, [] ).
verb( 'wish', 'wishes', 'wishing', 'wished', 'wished', _, ['2A','3A','6A','7A','9','12A','13A','15A','17','22'] ).
verb( 'withdraw', 'withdraws', 'withdrawing', 'withdrew', 'withdrawn', _, ['2A','2C','6A','14'] ).
verb( 'wither', 'withers', 'withering', 'withered', 'withered', _, ['2A','2C','6A','15B'] ).
verb( 'withhold', 'withholds', 'withholding', 'withheld', 'withheld', tran, ['6A','14'] ).
verb( 'withstand', 'withstands', 'withstanding', 'withstood', 'withstood', tran, ['6A'] ).
verb( 'witness', 'witnesses', 'witnessing', 'witnessed', 'witnessed', _, ['3A','6A'] ).
verb( 'wive', 'wives', 'wiving', 'wived', 'wived', _, ['2A','6A'] ).
verb( 'wobble', 'wobbles', 'wobbling', 'wobbled', 'wobbled', _, ['2A','2C','6A'] ).
verb( 'wolf', 'wolfs', 'wolfing', 'wolfed', 'wolfed', tran, ['6A','15A'] ).
verb( 'womanize', 'womanizes', 'womanizing', 'womanized', 'womanized', intran, [] ).
verb( 'wonder', 'wonders', 'wondering', 'wondered', 'wondered', _, ['2A','3A','3B','4B','8','10'] ).
verb( 'woo', 'woos', 'wooing', 'wooed', 'wooed', tran, ['6A'] ).
verb( 'word', 'words', 'wording', 'worded', 'worded', tran, ['6A'] ).
verb( 'work', 'works', 'working', 'worked', 'worked', _, ['2A','2B','2C','2D','3A','4A','6A','14','15A','15B','22'] ).
verb( 'worm', 'worms', 'worming', 'wormed', 'wormed', tran, ['6A','15A','15B'] ).
verb( 'worry', 'worries', 'worrying', 'worried', 'worried', _, ['2A','2B','2C','3A','6A','14','15A','15B','17','22'] ).
verb( 'worsen', 'worsens', 'worsening', 'worsened', 'worsened', _, ['2A','6A'] ).
verb( 'worship', 'worships', 'worshipping', 'worshipped', 'worshipped', _, ['2A','2B','6A'] ).
verb( 'worst', 'worsts', 'worsting', 'worsted', 'worsted', tran, ['6A'] ).
verb( 'wound', 'wounds', 'wounding', 'wounded', 'wounded', tran, ['6A'] ).
verb( 'wrangle', 'wrangles', 'wrangling', 'wrangled', 'wrangled', intran, ['2A','3A'] ).
verb( 'wrap', 'wraps', 'wrapping', 'wrapped', 'wrapped', _, ['6A','14','15A','15B'] ).
verb( 'wreak', 'wreaks', 'wreaking', 'wreaked', 'wreaked', tran, ['6A','14'] ).
verb( 'wreathe', 'wreathes', 'wreathing', 'wreathed', 'wreathed', _, ['2A','2C','6A','14'] ).
verb( 'wreck', 'wrecks', 'wrecking', 'wrecked', 'wrecked', tran, ['6A'] ).
verb( 'wrench', 'wrenches', 'wrenching', 'wrenched', 'wrenched', tran, ['6A','15A','22'] ).
verb( 'wrest', 'wrests', 'wresting', 'wrested', 'wrested', tran, ['14'] ).
verb( 'wrestle', 'wrestles', 'wrestling', 'wrestled', 'wrestled', intran, ['2A','2C','3A'] ).
verb( 'wrick', 'wricks', 'wricking', 'wricked', 'wricked', tran, ['6A'] ).
verb( 'wriggle', 'wriggles', 'wriggling', 'wriggled', 'wriggled', _, ['2A','2C','3A','6A','15B','22'] ).
verb( 'wring', 'wrings', 'wringing', 'wrung', 'wrung', tran, ['6A','14','15B'] ).
verb( 'wrinkle', 'wrinkles', 'wrinkling', 'wrinkled', 'wrinkled', _, ['2A','2C','6A','15B'] ).
verb( 'write', 'writes', 'writing', 'wrote', 'written', _, ['2A','2B','2C','4A','6A','12A','13A','15B'] ).
verb( 'writhe', 'writhes', 'writhing', 'writhed', 'writhed', intran, ['2A','2C'] ).
verb( 'wrong', 'wrongs', 'wronging', 'wronged', 'wronged', tran, ['6A'] ).
verb( 'yacht', 'yachts', 'yachting', 'yachted', 'yachted', intran, ['2A'] ).
verb( 'yammer', 'yammers', 'yammering', 'yammered', 'yammered', intran, [] ).
verb( 'yank', 'yanks', 'yanking', 'yanked', 'yanked', tran, ['6A','15A','15B'] ).
verb( 'yap', 'yaps', 'yapping', 'yapped', 'yapped', intran, ['2A'] ).
verb( 'yarn', 'yarns', 'yarning', 'yarned', 'yarned', intran, ['2A','2C'] ).
verb( 'yaw', 'yaws', 'yawing', 'yawed', 'yawed', intran, [] ).
verb( 'yawn', 'yawns', 'yawning', 'yawned', 'yawned', intran, ['2A','2C'] ).
verb( 'yearn', 'yearns', 'yearning', 'yearned', 'yearned', intran, ['3A','4A'] ).
verb( 'yell', 'yells', 'yelling', 'yelled', 'yelled', _, ['2A','2C','6A','15B'] ).
verb( 'yellow', 'yellows', 'yellowing', 'yellowed', 'yellowed', _, ['2A','6A'] ).
verb( 'yelp', 'yelps', 'yelping', 'yelped', 'yelped', intran, ['2A'] ).
verb( 'yen', 'yens', 'yenning', 'yenned', 'yenned', intran, [] ).
verb( 'yield', 'yields', 'yielding', 'yielded', 'yielded', _, ['2A','3A','6A','15A','15B'] ).
verb( 'yodel', 'yodels', 'yodelling', 'yodelled', 'yodelled', _, [] ).
verb( 'yoke', 'yokes', 'yoking', 'yoked', 'yoked', _, ['6A','15A'] ).
verb( 'yowl', 'yowls', 'yowling', 'yowled', 'yowled', intran, [] ).
verb( 'zap', 'zaps', 'zapping', 'zapped', 'zapped', tran, ['6A'] ).
verb( 'zero', 'zeros', 'zeroing', 'zeroed', 'zeroed', intran, ['2C'] ).
verb( 'zigzag', 'zigzags', 'zigzagging', 'zigzagged', 'zigzagged', intran, [] ).
verb( 'zip', 'zips', 'zipping', 'zipped', 'zipped', tran, ['6A','15B','22'] ).
verb( 'zone', 'zones', 'zoning', 'zoned', 'zoned', tran, ['6A'] ).
verb( 'zoom', 'zooms', 'zooming', 'zoomed', 'zoomed', intran, ['2A','2C'] ).

noun( 'a-bomb', 'a-bombs', count, _ ).
noun( 'a-level', 'a-levels', count, _ ).
noun( 'aa', '-', count, _ ).
noun( 'abc', '-', count, _ ).
noun( '-', 'abcs', count, _ ).
noun( 'ad', '-', proper, _ ).
noun( 'agm', '-', count, _ ).
noun( 'aids', '-', mass, _ ).
noun( 'awol', '-', proper, _ ).
noun( 'aachen', '-', proper, loc ).
noun( 'aarhus', '-', proper, loc ).
noun( 'abe', '-', proper, per ).
noun( 'abercarn', '-', proper, loc ).
noun( 'aberdare', '-', proper, loc ).
noun( 'aberdeen', '-', proper, loc ).
noun( 'abergavenny', '-', proper, loc ).
noun( 'abergele', '-', proper, loc ).
noun( 'abertillery', '-', proper, loc ).
noun( 'aberystwyth', '-', proper, loc ).
noun( 'abingdon', '-', proper, loc ).
noun( 'abo', 'abos', count, _ ).
noun( 'aborigine', 'aborigines', count, _ ).
noun( 'abraham', '-', proper, per ).
noun( 'accra', '-', proper, loc ).
noun( 'accrington', '-', proper, loc ).
noun( 'achilles', '-', proper, per ).
noun( 'ada', '-', proper, per ).
noun( 'adam', '-', proper, per ).
noun( 'addis ababa', '-', proper, loc ).
noun( 'addressograph', 'addressographs', count, _ ).
noun( 'adelaide', '-', proper, loc ).
noun( 'adrian', '-', proper, per ).
noun( 'adventist', 'adventists', count, _ ).
noun( 'afghan', 'afghans', both, _ ).
noun( 'afghanistan', '-', proper, loc ).
noun( 'afghanistani', 'afghanistanis', count, _ ).
noun( 'africa', '-', proper, loc ).
noun( 'african', 'africans', count, _ ).
noun( 'afrikaans', '-', mass, _ ).
noun( 'afrikaner', 'afrikaners', count, _ ).
noun( 'afro-american', 'afro-americans', count, _ ).
noun( 'afro-wig', 'afro-wigs', count, _ ).
noun( 'agatha', '-', proper, per ).
noun( 'aggie', '-', proper, per ).
noun( 'agnes', '-', proper, per ).
noun( 'agra', '-', proper, loc ).
noun( 'ahmedabad', '-', proper, loc ).
noun( 'airdrie', '-', proper, loc ).
noun( 'aireborough', '-', proper, loc ).
noun( 'airedale', 'airedales', count, _ ).
noun( 'akron', '-', proper, loc ).
noun( 'al', '-', proper, per ).
noun( 'alabama', '-', proper, loc ).
noun( 'alan', '-', proper, per ).
noun( 'alaska', '-', proper, loc ).
noun( 'albania', '-', proper, loc ).
noun( 'albanian', 'albanians', both, _ ).
noun( 'albert', '-', proper, per ).
noun( 'alberta', '-', proper, loc ).
noun( 'aldershot', '-', proper, loc ).
noun( 'aleppo', '-', proper, loc ).
noun( 'alessandria', '-', proper, loc ).
noun( 'alex', '-', proper, per ).
noun( 'alexander', '-', proper, per ).
noun( 'alexandra', '-', proper, per ).
noun( 'alexandria', '-', proper, loc ).
noun( 'alf', '-', proper, per ).
noun( 'alfred', '-', proper, per ).
noun( 'alfreton', '-', proper, loc ).
noun( 'alger', '-', proper, loc ).
noun( 'algeria', '-', proper, loc ).
noun( 'algerian', 'algerians', count, _ ).
noun( 'ali', '-', proper, per ).
noun( 'alicante', '-', proper, loc ).
noun( 'alice', '-', proper, per ).
noun( 'alison', '-', proper, per ).
noun( 'allah', '-', proper, _ ).
noun( 'allahabad', '-', proper, loc ).
noun( 'allan', '-', proper, per ).
noun( 'allen', '-', proper, per ).
noun( 'alloa', '-', proper, loc ).
noun( 'alma mater', 'alma maters', count, _ ).
noun( 'alma-ata', '-', proper, loc ).
noun( 'almeria', '-', proper, loc ).
noun( 'alnwick', '-', proper, loc ).
noun( 'alsatian', 'alsatians', count, _ ).
noun( 'alton', '-', proper, loc ).
noun( 'altrincham', '-', proper, loc ).
noun( 'alvechurch', '-', proper, loc ).
noun( 'amanda', '-', proper, per ).
noun( 'amazon', 'amazons', count, _ ).
noun( 'america', '-', proper, loc ).
noun( 'american', 'americans', count, _ ).
noun( 'americanism', 'americanisms', both, _ ).
noun( 'amesbury', '-', proper, loc ).
noun( 'amharic', '-', mass, _ ).
noun( 'amiens', '-', proper, loc ).
noun( 'ammanford', '-', proper, loc ).
noun( 'amsterdam', '-', proper, loc ).
noun( 'amy', '-', proper, per ).
noun( 'ancona', '-', proper, loc ).
noun( 'andorra', '-', proper, loc ).
noun( 'andorran', 'andorrans', count, _ ).
noun( 'andover', '-', proper, loc ).
noun( 'andrew', '-', proper, per ).
noun( 'andy', '-', proper, per ).
noun( 'angela', '-', proper, per ).
noun( 'angers', '-', proper, loc ).
noun( 'angie', '-', proper, per ).
noun( 'anglican', 'anglicans', count, _ ).
noun( 'anglo-catholic', 'anglo-catholics', count, _ ).
noun( 'anglo-indian', 'anglo-indians', count, _ ).
noun( 'anglo-saxon', 'anglo-saxons', count, _ ).
noun( 'anglomania', '-', mass, _ ).
noun( 'anglophil', 'anglophils', count, _ ).
noun( 'anglophile', 'anglophiles', count, _ ).
noun( 'anglophobe', 'anglophobes', count, _ ).
noun( 'anglophobia', '-', mass, _ ).
noun( 'angola', '-', proper, loc ).
noun( 'angolan', 'angolans', count, _ ).
noun( 'anguilla', '-', proper, loc ).
noun( 'anguillan', 'anguillans', count, _ ).
noun( 'angus', '-', proper, per ).
noun( 'anita', '-', proper, per ).
noun( 'ankara', '-', proper, loc ).
noun( 'ann', '-', proper, per ).
noun( 'annabel', '-', proper, per ).
noun( 'annan', '-', proper, loc ).
noun( 'anne', '-', proper, per ).
noun( 'annfield', '-', proper, loc ).
noun( 'annfield plain', '-', proper, loc ).
noun( 'annie', '-', proper, per ).
noun( 'anon', '-', proper, _ ).
noun( 'anshan', '-', proper, loc ).
noun( 'anthea', '-', proper, per ).
noun( 'anthony', '-', proper, per ).
noun( 'antigua', '-', proper, loc ).
noun( 'antiguan', 'antiguans', count, _ ).
noun( 'antony', '-', proper, per ).
noun( 'antrim', '-', proper, loc ).
noun( 'antwerp', '-', proper, loc ).
noun( 'apeldoorn', '-', proper, loc ).
noun( 'apocrypha', 'apocrypha', mass, _ ).
noun( 'apr', '-', proper, _ ).
noun( 'april', 'aprils', count, _ ).
noun( 'april', '-', proper, per ).
noun( 'aquarius', '-', proper, _ ).
noun( 'arab', 'arabs', count, _ ).
noun( 'arabian', 'arabians', count, _ ).
noun( 'arabic', '-', mass, _ ).
noun( 'arabist', 'arabists', count, _ ).
noun( 'arbroath', '-', proper, loc ).
noun( 'arcadian', 'arcadians', count, _ ).
noun( 'ardrossan', '-', proper, loc ).
noun( 'argentina', '-', proper, loc ).
noun( 'argentine', '-', proper, loc ).
noun( 'argentinian', 'argentinians', count, _ ).
noun( 'argonaut', 'argonauts', count, _ ).
noun( 'argus', 'arguses', count, _ ).
noun( 'aries', '-', proper, _ ).
noun( 'arizona', '-', proper, loc ).
noun( 'arkansas', '-', proper, loc ).
noun( 'armadale', '-', proper, loc ).
noun( 'armageddon', '-', proper, _ ).
noun( 'armagh', '-', proper, loc ).
noun( 'arnhem', '-', proper, loc ).
noun( 'arnold', '-', proper, per ).
noun( 'art', '-', proper, per ).
noun( 'arthur', '-', proper, per ).
noun( 'artie', '-', proper, per ).
noun( 'aryan', 'aryans', count, _ ).
noun( 'ascot', '-', proper, loc ).
noun( 'ashbourne', '-', proper, loc ).
noun( 'ashby', '-', proper, loc ).
noun( 'ashby woulds', '-', proper, loc ).
noun( 'ashby de la zouch', '-', proper, loc ).
noun( 'ashford', '-', proper, loc ).
noun( 'ashington', '-', proper, loc ).
noun( 'ashton-in-makerfield', '-', proper, loc ).
noun( 'ashton-under-lyne', '-', proper, loc ).
noun( 'asian', 'asians', count, _ ).
noun( 'asiatic', 'asiatics', count, _ ).
noun( 'askern', '-', proper, loc ).
noun( 'aspull', '-', proper, loc ).
noun( 'aston', '-', proper, loc ).
noun( 'athenian', 'athenians', count, _ ).
noun( 'athens', '-', proper, loc ).
noun( 'atherstone', '-', proper, loc ).
noun( 'athlone', '-', proper, loc ).
noun( 'atlanta', '-', proper, loc ).
noun( 'atlantic', '-', proper, _ ).
noun( 'auckland', '-', proper, loc ).
noun( 'audley', '-', proper, loc ).
noun( 'audrey', '-', proper, per ).
noun( 'aug', '-', proper, _ ).
noun( 'augsburg', '-', proper, loc ).
noun( 'august', 'augusts', count, _ ).
noun( 'august', '-', proper, _ ).
noun( 'aussie', 'aussies', count, _ ).
noun( 'australia', '-', proper, loc ).
noun( 'australian', 'australians', count, _ ).
noun( 'austria', '-', proper, loc ).
noun( 'austrian', 'austrians', count, _ ).
noun( 'ave', '-', proper, _ ).
noun( 'aveley', '-', proper, loc ).
noun( 'avignon', '-', proper, loc ).
noun( 'avon', '-', proper, loc ).
noun( 'aycliffe', '-', proper, loc ).
noun( 'aylesbury', '-', proper, loc ).
noun( 'aylesford', '-', proper, loc ).
noun( 'ayr', '-', proper, loc ).
noun( 'b ed', '-', count, _ ).
noun( 'ba', '-', count, _ ).
noun( 'bbc', '-', count, _ ).
noun( 'bc', '-', proper, _ ).
noun( 'bma', '-', count, _ ).
noun( 'bmus', '-', count, _ ).
noun( 'bst', '-', proper, _ ).
noun( 'bsc', '-', count, _ ).
noun( 'bvm', '-', count, _ ).
noun( 'babs', '-', proper, per ).
noun( 'backworth', '-', proper, loc ).
noun( 'bacup', '-', proper, loc ).
noun( 'badajoz', '-', proper, loc ).
noun( 'badalona', '-', proper, loc ).
noun( 'baghdad', '-', proper, loc ).
noun( 'bahamas', '-', proper, loc ).
noun( 'bahamian', 'bahamians', count, _ ).
noun( 'bahasa', '-', mass, _ ).
noun( 'bahrain', '-', proper, loc ).
noun( 'bahraini', 'bahrainis', count, _ ).
noun( 'bailey', '-', proper, _ ).
noun( 'baku', '-', proper, loc ).
noun( 'baldock', '-', proper, loc ).
noun( 'baltimore', '-', proper, loc ).
noun( 'banbury', '-', proper, loc ).
noun( 'bandung', '-', proper, loc ).
noun( 'bangalore', '-', proper, loc ).
noun( 'bangkok', '-', proper, loc ).
noun( 'bangladesh', '-', proper, loc ).
noun( 'bangladeshi', 'bangladeshis', count, _ ).
noun( 'bangor', '-', proper, loc ).
noun( 'bannockburn', '-', proper, loc ).
noun( 'bantry', '-', proper, loc ).
noun( 'bantu', 'bantu', count, _ ).
noun( 'baptist', 'baptists', count, _ ).
noun( 'baracaldo', '-', proper, loc ).
noun( 'barbadian', 'barbadians', count, _ ).
noun( 'barbados', '-', proper, loc ).
noun( 'barbara', '-', proper, per ).
noun( 'barcelona', '-', proper, loc ).
noun( 'bargoed', '-', proper, loc ).
noun( 'bari', '-', proper, loc ).
noun( 'barking', '-', proper, loc ).
noun( 'barnard castle', '-', proper, loc ).
noun( 'barnet', '-', proper, loc ).
noun( 'barnoldswick', '-', proper, loc ).
noun( 'barnsley', '-', proper, loc ).
noun( 'barnstaple', '-', proper, loc ).
noun( 'barranquilla', '-', proper, loc ).
noun( 'barrhead', '-', proper, loc ).
noun( 'barrow-in-furness', '-', proper, loc ).
noun( 'barry', '-', proper, per ).
noun( 'bart', '-', proper, per ).
noun( 'bart', '-', proper, per ).
noun( 'bartholomew', '-', proper, per ).
noun( 'barton-upon-humber', '-', proper, loc ).
noun( 'basel', '-', proper, loc ).
noun( 'basil', '-', proper, per ).
noun( 'basildon', '-', proper, loc ).
noun( 'basingstoke', '-', proper, loc ).
noun( 'bass', '-', mass, _ ).
noun( 'bath', '-', proper, loc ).
noun( 'bath-chair', 'bath-chairs', count, _ ).
noun( 'bathgate', '-', proper, loc ).
noun( 'batley', '-', proper, loc ).
noun( 'beaconsfield', '-', proper, loc ).
noun( 'bearsden', '-', proper, loc ).
noun( 'beatrice', '-', proper, per ).
noun( 'beaujolais', '-', mass, _ ).
noun( 'beaumaris', '-', proper, loc ).
noun( 'bebington', '-', proper, loc ).
noun( 'beccles', '-', proper, loc ).
noun( 'beddau', '-', proper, loc ).
noun( 'bedford', '-', proper, loc ).
noun( 'bedfordshire', '-', proper, loc ).
noun( 'bedlington', '-', proper, loc ).
noun( 'bedouin', 'bedouin', count, _ ).
noun( 'beduin', 'beduin', count, _ ).
noun( 'bedwas', '-', proper, loc ).
noun( 'beirut', '-', proper, loc ).
noun( 'beith', '-', proper, loc ).
noun( 'belem', '-', proper, loc ).
noun( 'belfast', '-', proper, loc ).
noun( 'belgian', 'belgians', count, _ ).
noun( 'belgium', '-', proper, loc ).
noun( 'belgrade', '-', proper, loc ).
noun( 'belinda', '-', proper, per ).
noun( 'bella', '-', proper, per ).
noun( 'belle', '-', proper, per ).
noun( 'bellshill', '-', proper, loc ).
noun( 'belo horizonte', '-', proper, loc ).
noun( 'belper', '-', proper, loc ).
noun( 'ben', '-', proper, per ).
noun( 'benedictine', 'benedictines', both, _ ).
noun( 'bengali', 'bengalis', both, _ ).
noun( 'benin', '-', proper, loc ).
noun( 'beninese', 'beninese', count, _ ).
noun( 'benjamin', '-', proper, per ).
noun( 'benny', '-', proper, per ).
noun( 'benzedrine', 'benzedrines', count, _ ).
noun( 'beograd', '-', proper, loc ).
noun( 'bergamo', '-', proper, loc ).
noun( 'bergen', '-', proper, loc ).
noun( 'berkhamsted', '-', proper, loc ).
noun( 'berkshire', '-', proper, loc ).
noun( 'berlin', '-', proper, loc ).
noun( 'bermuda', '-', proper, loc ).
noun( 'bermudan', 'bermudans', count, _ ).
noun( 'bern', '-', proper, loc ).
noun( 'bernard', '-', proper, per ).
noun( 'bernie', '-', proper, per ).
noun( 'bert', '-', proper, per ).
noun( 'bertha', '-', proper, per ).
noun( 'bertie', '-', proper, per ).
noun( 'berwick-upon-tweed', '-', proper, loc ).
noun( 'beryl', '-', proper, per ).
noun( 'bess', '-', proper, per ).
noun( 'bessie', '-', proper, per ).
noun( 'bethune', '-', proper, loc ).
noun( 'betsy', '-', proper, per ).
noun( 'betty', '-', proper, per ).
noun( 'beverley', '-', proper, loc ).
noun( 'bewdley', '-', proper, loc ).
noun( 'bexhill', '-', proper, loc ).
noun( 'bexley', '-', proper, loc ).
noun( 'bhutan', '-', proper, loc ).
noun( 'bhutani', 'bhutanis', count, _ ).
noun( 'bible', 'bibles', count, _ ).
noun( 'bicester', '-', proper, loc ).
noun( 'biddulph', '-', proper, loc ).
noun( 'bideford', '-', proper, loc ).
noun( 'bielefeld', '-', proper, loc ).
noun( 'biggleswade', '-', proper, loc ).
noun( 'bilbao', '-', proper, loc ).
noun( 'bill', '-', proper, per ).
noun( 'billericay', '-', proper, loc ).
noun( 'billington', '-', proper, loc ).
noun( 'billy', '-', proper, per ).
noun( 'bingley', '-', proper, loc ).
noun( 'birkenhead', '-', proper, loc ).
noun( 'birstall', '-', proper, loc ).
noun( 'birtley', '-', proper, loc ).
noun( 'bishop auckland', '-', proper, loc ).
noun( 'bishop\'s cleeve', '-', proper, loc ).
noun( 'bishop\'s stortford', '-', proper, loc ).
noun( 'bishopbriggs', '-', proper, loc ).
noun( 'blackburn', '-', proper, loc ).
noun( 'blackhall', '-', proper, loc ).
noun( 'blackpool', '-', proper, loc ).
noun( 'blackshirt', 'blackshirts', count, _ ).
noun( 'blackwood', '-', proper, loc ).
noun( 'blaenau-ffestiniog', '-', proper, loc ).
noun( 'blaenavon', '-', proper, loc ).
noun( 'blaengwrach', '-', proper, loc ).
noun( 'blaina', '-', proper, loc ).
noun( 'blairgowrie', '-', proper, loc ).
noun( 'blantyre', '-', proper, loc ).
noun( 'bldg', '-', proper, _ ).
noun( 'bldgs', '-', proper, _ ).
noun( 'bletchley', '-', proper, loc ).
noun( 'blidworth', '-', proper, loc ).
noun( 'blighty', '-', count, _ ).
noun( 'blvd', '-', count, _ ).
noun( 'blyth', '-', proper, loc ).
noun( 'bo\'ness', '-', proper, loc ).
noun( 'bob', '-', proper, per ).
noun( 'bobby', '-', proper, per ).
noun( 'bochum', '-', proper, loc ).
noun( 'bodmin', '-', proper, loc ).
noun( 'boer', 'boers', count, _ ).
noun( 'bognor', '-', proper, loc ).
noun( 'bognor regis', '-', proper, loc ).
noun( 'bogota', '-', proper, loc ).
noun( 'boldon', '-', proper, loc ).
noun( 'bolivia', '-', proper, loc ).
noun( 'bolivian', 'bolivians', count, _ ).
noun( 'bollington', '-', proper, loc ).
noun( 'bologna', '-', proper, loc ).
noun( 'bolshevik', 'bolsheviks', count, _ ).
noun( 'bolsover', '-', proper, loc ).
noun( 'bolton', '-', proper, loc ).
noun( 'bolton-le-sands', '-', proper, loc ).
noun( 'bolzano', '-', proper, loc ).
noun( 'bombay', '-', proper, loc ).
noun( 'bonn', '-', proper, loc ).
noun( 'bonnybridge', '-', proper, loc ).
noun( 'bonnyrigg', '-', proper, loc ).
noun( 'bootle', '-', proper, loc ).
noun( 'bordeaux', '-', mass, _ ).
noun( 'bordeaux', '-', proper, loc ).
noun( 'borders', '-', proper, loc ).
noun( 'bordon', '-', proper, loc ).
noun( 'boris', '-', proper, per ).
noun( 'boston', '-', proper, loc ).
noun( 'botswana', '-', proper, loc ).
noun( 'bottrop', '-', proper, loc ).
noun( 'bourne', '-', proper, loc ).
noun( 'bournemouth', '-', proper, loc ).
noun( 'bowburn', '-', proper, loc ).
noun( 'boxing day', 'boxing days', count, _ ).
noun( 'boxing day', '-', proper, _ ).
noun( 'bracknell', '-', proper, loc ).
noun( 'bradford', '-', proper, loc ).
noun( 'bradford-on-avon', '-', proper, loc ).
noun( 'brahmin', 'brahmins', count, _ ).
noun( 'braintree', '-', proper, loc ).
noun( 'branderburgh', '-', proper, loc ).
noun( 'brasilia', '-', proper, loc ).
noun( 'bratislava', '-', proper, loc ).
noun( 'braunschweig', '-', proper, loc ).
noun( 'braunton', '-', proper, loc ).
noun( 'brazil', '-', proper, loc ).
noun( 'brazilian', 'brazilians', count, _ ).
noun( 'breaston', '-', proper, loc ).
noun( 'brecknock', '-', proper, loc ).
noun( 'breda', '-', proper, loc ).
noun( 'breedsall', '-', proper, loc ).
noun( 'bremen', '-', proper, loc ).
noun( 'bremerhaven', '-', proper, loc ).
noun( 'bren', 'brens', count, _ ).
noun( 'bren-gun', 'bren-guns', count, _ ).
noun( 'brenda', '-', proper, per ).
noun( 'brent', '-', proper, loc ).
noun( 'brentwood', '-', proper, loc ).
noun( 'brescia', '-', proper, loc ).
noun( 'brest', '-', proper, loc ).
noun( 'brian', '-', proper, per ).
noun( 'bricket wood', '-', proper, loc ).
noun( 'bridgend', '-', proper, loc ).
noun( 'bridget', '-', proper, per ).
noun( 'bridgnorth', '-', proper, loc ).
noun( 'bridgwater', '-', proper, loc ).
noun( 'bridlington', '-', proper, loc ).
noun( 'bridport', '-', proper, loc ).
noun( 'brig', '-', proper, per ).
noun( 'brigadier', 'brigadiers', count, _ ).
noun( 'brighouse', '-', proper, loc ).
noun( 'brightlingsea', '-', proper, loc ).
noun( 'brighton', '-', proper, loc ).
noun( 'brisbane', '-', proper, loc ).
noun( 'bristol', '-', proper, loc ).
noun( 'brit', 'brits', count, _ ).
noun( 'britain', '-', proper, loc ).
noun( 'britisher', 'britishers', count, _ ).
noun( 'briton', 'britons', count, _ ).
noun( 'brixham', '-', proper, loc ).
noun( 'brno', '-', proper, loc ).
noun( 'bromley', '-', proper, loc ).
noun( 'bromsgrove', '-', proper, loc ).
noun( 'bros', '-', proper, _ ).
noun( 'broughton', '-', proper, loc ).
noun( 'broxburn', '-', proper, loc ).
noun( 'bruce', '-', proper, per ).
noun( 'bruges', '-', proper, loc ).
noun( 'brunei', '-', proper, loc ).
noun( 'bruneian', 'bruneians', count, _ ).
noun( 'brunswick', '-', proper, loc ).
noun( 'brussels', '-', proper, loc ).
noun( 'bryan', '-', proper, per ).
noun( 'bt', '-', proper, per ).
noun( 'bucarest', '-', proper, loc ).
noun( 'buckhaven', '-', proper, loc ).
noun( 'buckie', '-', proper, loc ).
noun( 'buckingham', '-', proper, loc ).
noun( 'buckinghamshire', '-', proper, loc ).
noun( 'buckley', '-', proper, loc ).
noun( 'bucksburn', '-', proper, loc ).
noun( 'budapest', '-', proper, loc ).
noun( 'buddhism', '-', mass, _ ).
noun( 'buddhist', 'buddhists', count, _ ).
noun( 'buenos aires', '-', proper, loc ).
noun( 'buffalo', '-', proper, loc ).
noun( 'bulgaria', '-', proper, loc ).
noun( 'bulgarian', 'bulgarians', count, _ ).
noun( 'bulkington', '-', proper, loc ).
noun( 'bunsen', 'bunsens', count, _ ).
noun( 'burberry', 'burberries', count, _ ).
noun( 'burgess', '-', proper, loc ).
noun( 'burgos', '-', proper, loc ).
noun( 'burgundy', '-', mass, _ ).
noun( 'burley', '-', proper, loc ).
noun( 'burma', '-', proper, loc ).
noun( 'burmese', 'burmese', both, _ ).
noun( 'burnham-on-sea', '-', proper, loc ).
noun( 'burnley', '-', proper, loc ).
noun( 'burntisland', '-', proper, loc ).
noun( 'burntwood', '-', proper, loc ).
noun( 'burry port', '-', proper, loc ).
noun( 'burscough', '-', proper, loc ).
noun( 'burton-upon-trent', '-', proper, loc ).
noun( 'burundi', '-', proper, loc ).
noun( 'burundian', 'burundians', count, _ ).
noun( 'bury', '-', proper, loc ).
noun( 'bury st. edmunds', '-', proper, loc ).
noun( 'bushman', 'bushmen', count, _ ).
noun( 'buxton', '-', proper, loc ).
noun( 'c of e', '-', count, _ ).
noun( 'c-in-c', '-', proper, per ).
noun( 'cbi', '-', count, _ ).
noun( 'cd', '-', count, _ ).
noun( 'cia', '-', count, _ ).
noun( 'cid', '-', count, _ ).
noun( 'co', '-', count, _ ).
noun( 'cod', '-', proper, _ ).
noun( 'cadiz', '-', proper, loc ).
noun( 'caen', '-', proper, loc ).
noun( 'caerleon', '-', proper, loc ).
noun( 'caernarfon', '-', proper, loc ).
noun( 'caerphilly', '-', proper, loc ).
noun( 'caesar', 'caesars', count, _ ).
noun( 'cagliari', '-', proper, loc ).
noun( 'cairo', '-', proper, loc ).
noun( 'calcutta', '-', proper, loc ).
noun( 'caldicot', '-', proper, loc ).
noun( 'cali', '-', proper, loc ).
noun( 'california', '-', proper, loc ).
noun( 'californian', 'californians', count, _ ).
noun( 'calne', '-', proper, loc ).
noun( 'calor', '-', proper, _ ).
noun( 'calvary', 'calvaries', count, _ ).
noun( 'calverton', '-', proper, loc ).
noun( 'calvinism', '-', mass, _ ).
noun( 'calvinist', 'calvinists', count, _ ).
noun( 'cambodia', '-', proper, loc ).
noun( 'cambodian', 'cambodians', count, _ ).
noun( 'cambourne', '-', proper, loc ).
noun( 'cambridge', '-', proper, loc ).
noun( 'cambridgeshire', '-', proper, loc ).
noun( 'cambuslang', '-', proper, loc ).
noun( 'camden', '-', proper, loc ).
noun( 'camembert', 'camemberts', count, _ ).
noun( 'cameroon', '-', proper, loc ).
noun( 'cameroonian', 'cameroonians', count, _ ).
noun( 'campbeltown', '-', proper, loc ).
noun( 'canada', '-', proper, loc ).
noun( 'canadian', 'canadians', count, _ ).
noun( 'canberra', '-', proper, loc ).
noun( 'cancer', '-', proper, _ ).
noun( 'cannes', '-', proper, loc ).
noun( 'cannock', '-', proper, loc ).
noun( 'cantab', '-', proper, _ ).
noun( 'canterbury', '-', proper, loc ).
noun( 'canton', '-', proper, loc ).
noun( 'canuck', 'canucks', count, _ ).
noun( 'canvey', '-', proper, loc ).
noun( 'canvey island', '-', proper, loc ).
noun( 'cape town', '-', proper, loc ).
noun( 'capitol', 'capitols', count, _ ).
noun( 'capricorn', '-', proper, _ ).
noun( 'capt', '-', proper, per ).
noun( 'caracas', '-', proper, loc ).
noun( 'cardenden', '-', proper, loc ).
noun( 'cardiff', '-', proper, loc ).
noun( 'cardigan', '-', proper, loc ).
noun( 'carl', '-', proper, per ).
noun( 'carlisle', '-', proper, loc ).
noun( 'carlow', '-', proper, loc ).
noun( 'carluke', '-', proper, loc ).
noun( 'carmarthen', '-', proper, loc ).
noun( 'carmelite', 'carmelites', count, _ ).
noun( 'carnforth', '-', proper, loc ).
noun( 'carnoustie', '-', proper, loc ).
noun( 'carol', '-', proper, per ).
noun( 'carole', '-', proper, per ).
noun( 'carolina', '-', proper, loc ).
noun( 'caroline', '-', proper, per ).
noun( 'carolyn', '-', proper, per ).
noun( 'carrie', '-', proper, per ).
noun( 'carron', '-', proper, loc ).
noun( 'cartagena', '-', proper, loc ).
noun( 'casablanca', '-', proper, loc ).
noun( 'castleford', '-', proper, loc ).
noun( 'catalan', '-', mass, _ ).
noun( 'catania', '-', proper, loc ).
noun( 'cath', '-', proper, per ).
noun( 'catherine', '-', proper, per ).
noun( 'catholic', 'catholics', count, _ ).
noun( 'catholicism', '-', mass, _ ).
noun( 'cathy', '-', proper, per ).
noun( 'caucasian', 'caucasians', count, _ ).
noun( 'cavan', '-', proper, loc ).
noun( 'cdr', '-', proper, per ).
noun( 'cdre', '-', proper, per ).
noun( 'cecil', '-', proper, per ).
noun( 'cecilia', '-', proper, per ).
noun( 'cecily', '-', proper, per ).
noun( 'cedric', '-', proper, per ).
noun( 'cefn-mawr', '-', proper, loc ).
noun( 'celia', '-', proper, per ).
noun( 'celt', 'celts', count, _ ).
noun( 'celtic', 'celtics', count, _ ).
noun( 'ceylon', '-', proper, loc ).
noun( 'chad', '-', proper, loc ).
noun( 'chadian', 'chadians', count, _ ).
noun( 'chalfont', '-', proper, loc ).
noun( 'chalfont st giles', '-', proper, loc ).
noun( 'chalfont st peter', '-', proper, loc ).
noun( 'changchun', '-', proper, loc ).
noun( 'changsha', '-', proper, loc ).
noun( 'chard', '-', proper, loc ).
noun( 'charles', '-', proper, per ).
noun( 'charleston', 'charlestons', count, _ ).
noun( 'charlie', '-', proper, per ).
noun( 'charlotte', '-', proper, per ).
noun( 'chartism', '-', mass, _ ).
noun( 'chartist', 'chartists', count, _ ).
noun( 'charybdis', '-', proper, _ ).
noun( 'chas', '-', proper, per ).
noun( 'chatham', '-', proper, loc ).
noun( 'cheadle', '-', proper, loc ).
noun( 'cheddar', '-', mass, _ ).
noun( 'chelmsford', '-', proper, loc ).
noun( 'chelsea', '-', proper, loc ).
noun( 'cheltenham', '-', proper, loc ).
noun( 'chelyabinsk', '-', proper, loc ).
noun( 'chengchow', '-', proper, loc ).
noun( 'chengtu', '-', proper, loc ).
noun( 'chepstow', '-', proper, loc ).
noun( 'chesham', '-', proper, loc ).
noun( 'cheshire', '-', proper, loc ).
noun( 'chester', '-', proper, loc ).
noun( 'chester-le-street', '-', proper, loc ).
noun( 'chesterfield', '-', proper, loc ).
noun( 'chianti', '-', mass, _ ).
noun( 'chicago', '-', proper, loc ).
noun( 'chichester', '-', proper, loc ).
noun( 'chile', '-', proper, loc ).
noun( 'chilean', 'chileans', count, _ ).
noun( 'china', '-', proper, loc ).
noun( 'chinchow', '-', proper, loc ).
noun( 'chinese', 'chinese', both, _ ).
noun( 'chippendale', 'chippendales', count, _ ).
noun( 'chippenham', '-', proper, loc ).
noun( 'chloe', '-', proper, per ).
noun( 'chopwell', '-', proper, loc ).
noun( 'chorley', '-', proper, loc ).
noun( 'chris', '-', proper, per ).
noun( 'chrissie', '-', proper, per ).
noun( 'chrissy', '-', proper, per ).
noun( 'christ', '-', proper, _ ).
noun( 'christchurch', '-', proper, loc ).
noun( 'christendom', 'christendoms', count, _ ).
noun( 'christian', 'christians', count, _ ).
noun( 'christian', '-', proper, per ).
noun( 'christianity', '-', mass, _ ).
noun( 'christina', '-', proper, per ).
noun( 'christine', '-', proper, per ).
noun( 'christmas', 'christmases', count, _ ).
noun( 'christmas', '-', proper, _ ).
noun( 'christmas-box', 'christmas-boxes', count, _ ).
noun( 'christmas-tree', 'christmas-trees', count, _ ).
noun( 'christmastide', 'christmastides', count, _ ).
noun( 'christmastime', 'christmastimes', count, _ ).
noun( 'christopher', '-', proper, per ).
noun( 'chungking', '-', proper, loc ).
noun( 'cincinnati', '-', proper, loc ).
noun( 'cinderella', 'cinderellas', count, _ ).
noun( 'cinderford', '-', proper, loc ).
noun( 'cirencester', '-', proper, loc ).
noun( 'civvy street', '-', proper, _ ).
noun( 'clackmannon', '-', proper, loc ).
noun( 'clacton', '-', proper, loc ).
noun( 'clare', '-', proper, per ).
noun( 'clarkston', '-', proper, loc ).
noun( 'clarrie', '-', proper, per ).
noun( 'classics', 'classics', mass, _ ).
noun( 'claud', '-', proper, per ).
noun( 'claude', '-', proper, per ).
noun( 'clay cross', '-', proper, loc ).
noun( 'cleator moor', '-', proper, loc ).
noun( 'cleethorpes', '-', proper, loc ).
noun( 'clem', '-', proper, per ).
noun( 'clement', '-', proper, per ).
noun( 'clermont-ferrand', '-', proper, loc ).
noun( 'clevedon', '-', proper, loc ).
noun( 'cleveland', '-', proper, loc ).
noun( 'cliff', '-', proper, per ).
noun( 'clifford', '-', proper, per ).
noun( 'clitheroe', '-', proper, loc ).
noun( 'clive', '-', proper, per ).
noun( 'clowne', '-', proper, loc ).
noun( 'clwyd', '-', proper, loc ).
noun( 'clydach', '-', proper, loc ).
noun( 'clydebank', '-', proper, loc ).
noun( 'co', '-', count, _ ).
noun( 'co-op', '-', count, _ ).
noun( 'coalville', '-', proper, loc ).
noun( 'coatbridge', '-', proper, loc ).
noun( 'cobham', '-', proper, loc ).
noun( 'coca-cola', 'coca-colas', both, _ ).
noun( 'cockermouth', '-', proper, loc ).
noun( 'codsall', '-', proper, loc ).
noun( 'coimbatore', '-', proper, loc ).
noun( 'col', '-', proper, per ).
noun( 'colchester', '-', proper, loc ).
noun( 'coleshill', '-', proper, loc ).
noun( 'colin', '-', proper, per ).
noun( 'coll', '-', proper, _ ).
noun( 'colne', '-', proper, loc ).
noun( 'cologne', '-', proper, loc ).
noun( 'colombia', '-', proper, loc ).
noun( 'colombian', 'colombians', count, _ ).
noun( 'colombo', '-', proper, loc ).
noun( 'colorado', '-', proper, loc ).
noun( 'columbia', '-', proper, loc ).
noun( 'columbus', '-', proper, loc ).
noun( 'colwyn bay', '-', proper, loc ).
noun( 'confucian', 'confucians', count, _ ).
noun( 'congleton', '-', proper, loc ).
noun( 'congo', '-', proper, loc ).
noun( 'congolese', 'congolese', count, _ ).
noun( 'connah\'s quay', '-', proper, loc ).
noun( 'connaught', '-', proper, loc ).
noun( 'connecticut', '-', proper, loc ).
noun( 'connie', '-', proper, per ).
noun( 'cons', '-', proper, _ ).
noun( 'consett', '-', proper, loc ).
noun( 'constance', '-', proper, per ).
noun( 'conwy', '-', proper, loc ).
noun( 'cookham', '-', proper, loc ).
noun( 'copenhagen', '-', proper, loc ).
noun( 'coppull', '-', proper, loc ).
noun( 'copt', 'copts', count, _ ).
noun( 'coptic', 'coptics', count, _ ).
noun( 'corby', '-', proper, loc ).
noun( 'cordoba', '-', proper, loc ).
noun( 'corinthian', 'corinthians', count, _ ).
noun( 'cork', '-', proper, loc ).
noun( 'cornwall', '-', proper, loc ).
noun( 'corp', '-', proper, _ ).
noun( 'corps diplomatique', '-', count, _ ).
noun( 'corsham', '-', proper, loc ).
noun( 'corunna', '-', proper, loc ).
noun( 'cosenza', '-', proper, loc ).
noun( 'costa rica', '-', proper, loc ).
noun( 'costa rican', 'costa ricans', count, _ ).
noun( 'coventry', '-', proper, loc ).
noun( 'cowdenbeath', '-', proper, loc ).
noun( 'cowes', '-', proper, loc ).
noun( 'cpl', '-', proper, per ).
noun( 'cramlington', '-', proper, loc ).
noun( 'crawley', '-', proper, loc ).
noun( 'creole', 'creoles', both, _ ).
noun( 'cres', '-', proper, _ ).
noun( 'creswell', '-', proper, loc ).
noun( 'crewe', '-', proper, loc ).
noun( 'crieff', '-', proper, loc ).
noun( 'croesus', '-', proper, per ).
noun( 'cromer', '-', proper, loc ).
noun( 'crook', '-', proper, loc ).
noun( 'crosby', '-', proper, loc ).
noun( 'crowborough', '-', proper, loc ).
noun( 'crowthorne', '-', proper, loc ).
noun( 'croydon', '-', proper, loc ).
noun( 'cuba', '-', proper, loc ).
noun( 'cuban', 'cubans', count, _ ).
noun( 'cudworth', '-', proper, loc ).
noun( 'cuffley', '-', proper, loc ).
noun( 'culcheth', '-', proper, loc ).
noun( 'cumberland', '-', proper, loc ).
noun( 'cumbernauld', '-', proper, loc ).
noun( 'cumbria', '-', proper, loc ).
noun( 'cumnock', '-', proper, loc ).
noun( 'cupar', '-', proper, loc ).
noun( 'cupid', '-', proper, per ).
noun( 'curitiba', '-', proper, loc ).
noun( 'currie', '-', proper, loc ).
noun( 'cwmbran', '-', proper, loc ).
noun( 'cynthia', '-', proper, per ).
noun( 'cypriot', 'cypriots', count, _ ).
noun( 'cyprus', '-', proper, loc ).
noun( 'cyril', '-', proper, per ).
noun( 'czech', 'czechs', both, _ ).
noun( 'czechoslovak', 'czechoslovaks', count, _ ).
noun( 'czechoslovakia', '-', proper, loc ).
noun( 'czechoslovakian', 'czechoslovakians', count, _ ).
noun( 'd-day', '-', proper, _ ).
noun( 'ddt', '-', mass, _ ).
noun( 'des', '-', count, _ ).
noun( 'dg', '-', proper, per ).
noun( 'diy', '-', mass, _ ).
noun( 'dj', '-', count, _ ).
noun( '-', 'djs', count, _ ).
noun( 'dlitt', '-', count, _ ).
noun( 'dm', 'dm', count, _ ).
noun( 'dna', '-', mass, _ ).
noun( 'dphil', '-', count, _ ).
noun( 'dss', '-', count, _ ).
noun( 'dsc', '-', count, _ ).
noun( 'dti', '-', count, _ ).
noun( '-', 'dts', count, _ ).
noun( 'dacca', '-', proper, loc ).
noun( 'dail eireann', '-', count, _ ).
noun( 'daisy', '-', proper, per ).
noun( 'dakar', '-', proper, loc ).
noun( 'dakota', '-', proper, loc ).
noun( 'dalkeith', '-', proper, loc ).
noun( 'dallas', '-', proper, loc ).
noun( 'dalry', '-', proper, loc ).
noun( 'dalton', '-', proper, loc ).
noun( 'damascus', '-', proper, loc ).
noun( 'damocles', '-', proper, per ).
noun( 'dan', '-', proper, per ).
noun( 'dane', 'danes', count, _ ).
noun( 'daniel', 'daniels', count, _ ).
noun( 'daniel', '-', proper, per ).
noun( 'danish', '-', mass, _ ).
noun( 'danny', '-', proper, per ).
noun( 'daphne', '-', proper, per ).
noun( 'darby', '-', proper, per ).
noun( 'darenth', '-', proper, loc ).
noun( 'darfield', '-', proper, loc ).
noun( 'darkey', 'darkeys', count, _ ).
noun( 'darkie', 'darkies', count, _ ).
noun( 'darky', 'darkies', count, _ ).
noun( 'darlington', '-', proper, loc ).
noun( 'darmstadt', '-', proper, loc ).
noun( 'dartford', '-', proper, loc ).
noun( 'dartmouth', '-', proper, loc ).
noun( 'darwen', '-', proper, loc ).
noun( 'dave', '-', proper, per ).
noun( 'daventry', '-', proper, loc ).
noun( 'davey', '-', proper, per ).
noun( 'david', '-', proper, per ).
noun( 'dawley', '-', proper, loc ).
noun( 'dawlish', '-', proper, loc ).
noun( 'dawn', '-', proper, per ).
noun( 'dayton', '-', proper, loc ).
noun( 'deal', '-', proper, loc ).
noun( 'dean', '-', proper, per ).
noun( 'debbie', '-', proper, per ).
noun( 'debby', '-', proper, per ).
noun( 'deborah', '-', proper, per ).
noun( 'dec', '-', proper, _ ).
noun( 'decalogue', 'decalogues', count, _ ).
noun( 'december', 'decembers', count, _ ).
noun( 'december', '-', proper, _ ).
noun( 'deirdre', '-', proper, per ).
noun( 'delaware', '-', proper, loc ).
noun( 'delhi', '-', proper, loc ).
noun( 'denain', '-', proper, loc ).
noun( 'denbigh', '-', proper, loc ).
noun( 'denis', '-', proper, per ).
noun( 'denise', '-', proper, per ).
noun( 'denmark', '-', proper, loc ).
noun( 'dennis', '-', proper, per ).
noun( 'denny', '-', proper, loc ).
noun( 'denver', '-', proper, loc ).
noun( 'dept', '-', proper, _ ).
noun( 'derby', '-', proper, loc ).
noun( 'derbyshire', '-', proper, loc ).
noun( 'dereham', '-', proper, loc ).
noun( 'derek', '-', proper, per ).
noun( 'des', '-', proper, per ).
noun( 'desmond', '-', proper, per ).
noun( 'dessau', '-', proper, loc ).
noun( 'detroit', '-', proper, loc ).
noun( 'deutschmark', 'deutschmarks', count, _ ).
noun( 'devizes', '-', proper, loc ).
noun( 'devon', '-', proper, loc ).
noun( 'dewsbury', '-', proper, loc ).
noun( 'di', '-', proper, per ).
noun( 'diana', '-', proper, per ).
noun( 'diaspora', 'diasporas', count, _ ).
noun( 'dick', '-', proper, per ).
noun( 'dicky', '-', proper, per ).
noun( 'dictaphone', 'dictaphones', count, _ ).
noun( 'didcot', '-', proper, loc ).
noun( 'dijon', '-', proper, loc ).
noun( 'dingle', '-', proper, loc ).
noun( 'dinnington', '-', proper, loc ).
noun( 'dip', '-', count, _ ).
noun( 'dip ed', '-', count, _ ).
noun( 'dir', '-', count, _ ).
noun( 'dives', '-', proper, per ).
noun( 'djibouti', '-', proper, loc ).
noun( 'djiboutian', 'djiboutians', count, _ ).
noun( 'dnepropetrovsk', '-', proper, loc ).
noun( 'doe', '-', count, _ ).
noun( 'dolly', '-', proper, per ).
noun( 'domesday', '-', proper, _ ).
noun( 'dominic', '-', proper, per ).
noun( 'dominica', '-', proper, loc ).
noun( 'dominican', 'dominicans', count, _ ).
noun( 'don', '-', proper, per ).
noun( 'donald', '-', proper, per ).
noun( 'doncaster', '-', proper, loc ).
noun( 'donegal', '-', proper, loc ).
noun( 'donetsk', '-', proper, loc ).
noun( 'doomsday', '-', proper, _ ).
noun( 'dora', '-', proper, per ).
noun( 'dorchester', '-', proper, loc ).
noun( 'dordrecht', '-', proper, loc ).
noun( 'doreen', '-', proper, per ).
noun( 'doris', '-', proper, per ).
noun( 'dorking', '-', proper, loc ).
noun( 'dorothy', '-', proper, per ).
noun( 'dorset', '-', proper, loc ).
noun( 'dortmund', '-', proper, loc ).
noun( 'douai', '-', proper, loc ).
noun( 'doug', '-', proper, per ).
noun( 'douglas', '-', proper, per ).
noun( 'dover', '-', proper, loc ).
noun( 'down', '-', proper, loc ).
noun( 'downing street', '-', proper, _ ).
noun( 'dr', '-', proper, per ).
noun( 'drayton', '-', proper, loc ).
noun( 'dresden', '-', proper, loc ).
noun( 'driffield', '-', proper, loc ).
noun( 'drogheda', '-', proper, loc ).
noun( 'droitwich', '-', proper, loc ).
noun( 'dronfield', '-', proper, loc ).
noun( 'druid', 'druids', count, _ ).
noun( 'dublin', '-', proper, loc ).
noun( 'dubliner', 'dubliners', count, _ ).
noun( 'duce', '-', count, _ ).
noun( 'dudley', '-', proper, loc ).
noun( 'duisburg', '-', proper, loc ).
noun( 'dukinfield', '-', proper, loc ).
noun( 'dumbarton', '-', proper, loc ).
noun( 'dumfries', '-', proper, loc ).
noun( 'dun laoghaire', '-', proper, loc ).
noun( 'duncan', '-', proper, per ).
noun( 'dundalk', '-', proper, loc ).
noun( 'dundee', '-', proper, loc ).
noun( 'dunfermline', '-', proper, loc ).
noun( 'dunkirk', '-', proper, loc ).
noun( 'dunoon', '-', proper, loc ).
noun( 'dunstable', '-', proper, loc ).
noun( 'durban', '-', proper, loc ).
noun( 'durham', '-', proper, loc ).
noun( 'durrington', '-', proper, loc ).
noun( 'dursley', '-', proper, loc ).
noun( 'dusseldorf', '-', proper, loc ).
noun( 'dutch', '-', mass, _ ).
noun( 'dutchman', 'dutchmen', count, _ ).
noun( 'dyfed', '-', proper, loc ).
noun( 'ec', '-', count, _ ).
noun( 'edp', '-', mass, _ ).
noun( 'eec', '-', count, _ ).
noun( 'eeg', '-', count, _ ).
noun( 'efta', '-', count, _ ).
noun( 'esp', '-', mass, _ ).
noun( 'eaglescliffe', '-', proper, loc ).
noun( 'ealing', '-', proper, loc ).
noun( 'earl shilton', '-', proper, loc ).
noun( 'easington', '-', proper, loc ).
noun( 'east dereham', '-', proper, loc ).
noun( 'east grinstead', '-', proper, loc ).
noun( 'east kilbride', '-', proper, loc ).
noun( 'east retford', '-', proper, loc ).
noun( 'eastbourne', '-', proper, loc ).
noun( 'easter', 'easters', count, _ ).
noun( 'easter', '-', proper, _ ).
noun( 'eastleigh', '-', proper, loc ).
noun( 'ebbw vale', '-', proper, loc ).
noun( 'eccles', '-', proper, loc ).
noun( 'ecuador', '-', proper, loc ).
noun( 'ecuadorian', 'ecuadorians', count, _ ).
noun( 'ed', '-', proper, per ).
noun( 'eddie', '-', proper, per ).
noun( 'eddy', '-', proper, per ).
noun( 'eden', '-', proper, _ ).
noun( 'edenbridge', '-', proper, loc ).
noun( 'edgar', '-', proper, per ).
noun( 'edinburgh', '-', proper, loc ).
noun( 'edith', '-', proper, per ).
noun( 'edmund', '-', proper, per ).
noun( 'edward', '-', proper, per ).
noun( 'edwardian', 'edwardians', count, _ ).
noun( 'egremont', '-', proper, loc ).
noun( 'egypt', '-', proper, loc ).
noun( 'egyptian', 'egyptians', count, _ ).
noun( 'eiche', '-', proper, loc ).
noun( 'eileen', '-', proper, per ).
noun( 'eindhoven', '-', proper, loc ).
noun( 'el dorado', '-', count, _ ).
noun( 'el salvador', '-', proper, loc ).
noun( 'elaine', '-', proper, per ).
noun( 'elastoplast', '-', mass, _ ).
noun( 'elderslie', '-', proper, loc ).
noun( 'eleanor', '-', proper, per ).
noun( 'elgin', '-', proper, loc ).
noun( 'eliza', '-', proper, per ).
noun( 'elizabeth', '-', proper, per ).
noun( 'elizabethan', 'elizabethans', count, _ ).
noun( 'elland', '-', proper, loc ).
noun( 'ellen', '-', proper, per ).
noun( 'ellesmere', '-', proper, loc ).
noun( 'ellesmere port', '-', proper, loc ).
noun( 'ellie', '-', proper, per ).
noun( 'elloughton', '-', proper, loc ).
noun( 'elsie', '-', proper, per ).
noun( 'elstree', '-', proper, loc ).
noun( 'ely', '-', proper, loc ).
noun( 'elysium', '-', proper, _ ).
noun( 'emily', '-', proper, per ).
noun( 'emma', '-', proper, per ).
noun( 'emsworth', '-', proper, loc ).
noun( 'enfield', '-', proper, loc ).
noun( 'england', '-', proper, loc ).
noun( 'english', '-', mass, _ ).
noun( 'englishman', 'englishmen', count, _ ).
noun( 'englishwoman', 'englishwomen', count, _ ).
noun( 'enoch', '-', proper, per ).
noun( 'enschede', '-', proper, loc ).
noun( 'epiphany', '-', proper, _ ).
noun( 'epping', '-', proper, loc ).
noun( 'epsom', '-', proper, loc ).
noun( 'erfurt', '-', proper, loc ).
noun( 'eric', '-', proper, per ).
noun( 'erica', '-', proper, per ).
noun( 'erin', '-', proper, loc ).
noun( 'eritrea', '-', proper, loc ).
noun( 'eritrean', 'eritreans', count, _ ).
noun( 'erlangen', '-', proper, loc ).
noun( 'ernest', '-', proper, per ).
noun( 'ernie', '-', proper, per ).
noun( 'erse', '-', mass, _ ).
noun( 'esfahan', '-', proper, loc ).
noun( 'eskimo', 'eskimos', count, _ ).
noun( 'esperanto', '-', mass, _ ).
noun( 'esq', '-', proper, per ).
noun( 'esquire', 'esquires', count, _ ).
noun( 'essen', '-', proper, loc ).
noun( 'essex', '-', proper, loc ).
noun( 'esther', '-', proper, per ).
noun( 'ethel', '-', proper, per ).
noun( 'ethiopia', '-', proper, loc ).
noun( 'ethiopian', 'ethiopians', count, _ ).
noun( 'eucharist', 'eucharists', count, _ ).
noun( 'eugene', '-', proper, per ).
noun( 'eunice', '-', proper, per ).
noun( 'eurasia', '-', proper, loc ).
noun( 'eurasian', 'eurasians', count, _ ).
noun( 'eurodollar', 'eurodollars', count, _ ).
noun( 'europe', '-', proper, loc ).
noun( 'european', 'europeans', count, _ ).
noun( 'eurovision', '-', proper, _ ).
noun( 'eva', '-', proper, per ).
noun( 'eve', '-', proper, per ).
noun( 'evelyn', '-', proper, per ).
noun( 'evesham', '-', proper, loc ).
noun( 'ewell', '-', proper, loc ).
noun( 'excellency', 'excellencies', count, _ ).
noun( 'exeter', '-', proper, loc ).
noun( 'exmouth', '-', proper, loc ).
noun( 'fa', '-', count, _ ).
noun( 'fao', '-', count, _ ).
noun( 'fbi', '-', count, _ ).
noun( 'fm', '-', mass, _ ).
noun( 'fo', '-', count, _ ).
noun( 'frs', '-', count, _ ).
noun( 'fabian', 'fabians', count, _ ).
noun( 'falkirk', '-', proper, loc ).
noun( 'falmouth', '-', proper, loc ).
noun( 'fanny', '-', proper, per ).
noun( 'farnham', '-', proper, loc ).
noun( 'farnworth', '-', proper, loc ).
noun( 'farsi', '-', mass, _ ).
noun( 'fauldhouse', '-', proper, loc ).
noun( 'faversham', '-', proper, loc ).
noun( 'fawley', '-', proper, loc ).
noun( 'featherstone', '-', proper, loc ).
noun( 'feb', '-', proper, _ ).
noun( 'february', 'februaries', count, _ ).
noun( 'february', '-', proper, _ ).
noun( 'fed', '-', count, _ ).
noun( 'felicity', '-', proper, per ).
noun( 'felix', '-', proper, per ).
noun( 'felixstowe', '-', proper, loc ).
noun( 'fermanagh', '-', proper, loc ).
noun( 'ferrara', '-', proper, loc ).
noun( 'ferryhill', '-', proper, loc ).
noun( 'fife', '-', proper, loc ).
noun( 'fiji', '-', proper, loc ).
noun( 'fijian', 'fijians', count, _ ).
noun( 'filipino', 'filipinos', count, _ ).
noun( 'finland', '-', proper, loc ).
noun( 'finn', 'finns', count, _ ).
noun( 'finnish', '-', mass, _ ).
noun( 'fiona', '-', proper, per ).
noun( 'firenze', '-', proper, loc ).
noun( 'fleet', '-', proper, loc ).
noun( 'fleet street', '-', proper, _ ).
noun( 'fleetwood', '-', proper, loc ).
noun( 'flemish', '-', mass, _ ).
noun( 'flint', '-', proper, loc ).
noun( 'flora', '-', proper, per ).
noun( 'florence', '-', proper, per ).
noun( 'florida', '-', proper, loc ).
noun( 'florrie', '-', proper, per ).
noun( 'foggia', '-', proper, loc ).
noun( 'folkestone', '-', proper, loc ).
noun( 'foochow', '-', proper, loc ).
noun( 'forfar', '-', proper, loc ).
noun( 'forli', '-', proper, loc ).
noun( 'formby', '-', proper, loc ).
noun( 'formica', '-', mass, _ ).
noun( 'formosa', '-', proper, loc ).
noun( 'fort lauderdale', '-', proper, loc ).
noun( 'fort william', '-', proper, loc ).
noun( 'fort worth', '-', proper, loc ).
noun( 'fortaleza', '-', proper, loc ).
noun( 'fowey', '-', proper, loc ).
noun( 'fr', '-', proper, per ).
noun( 'fr\"aulein', 'fr\"auleins', count, _ ).
noun( 'frampton cotterell', '-', proper, loc ).
noun( 'fran', '-', proper, per ).
noun( 'france', '-', proper, loc ).
noun( 'frances', '-', proper, per ).
noun( 'francis', '-', proper, per ).
noun( 'franciscan', 'franciscans', count, _ ).
noun( 'frank', 'franks', count, _ ).
noun( 'frank', '-', proper, per ).
noun( 'frankfurt', '-', proper, loc ).
noun( 'frankie', '-', proper, per ).
noun( 'fraserburgh', '-', proper, loc ).
noun( 'frau', 'frauen', count, _ ).
noun( 'freckleton', '-', proper, loc ).
noun( 'fred', '-', proper, per ).
noun( 'freda', '-', proper, per ).
noun( 'freddie', '-', proper, per ).
noun( 'freddy', '-', proper, per ).
noun( 'frederick', '-', proper, per ).
noun( 'frederiksberg', '-', proper, loc ).
noun( 'freemason', 'freemasons', count, _ ).
noun( 'freemasonry', '-', mass, _ ).
noun( 'freiburg', '-', proper, loc ).
noun( 'french', '-', mass, _ ).
noun( 'frenchman', 'frenchmen', count, _ ).
noun( 'frenchwoman', 'frenchwomen', count, _ ).
noun( 'freshwater', '-', proper, loc ).
noun( 'fri', '-', proper, _ ).
noun( 'friday', 'fridays', count, _ ).
noun( 'friday', '-', proper, _ ).
noun( 'frinton', '-', proper, loc ).
noun( 'frodsham', '-', proper, loc ).
noun( 'frome', '-', proper, loc ).
noun( 'furth', '-', proper, loc ).
noun( 'fushun', '-', proper, loc ).
noun( 'g-man', 'g-men', count, _ ).
noun( 'gatt', '-', count, _ ).
noun( 'gb', '-', proper, _ ).
noun( 'gcse', '-', count, _ ).
noun( '-', 'gcses', count, _ ).
noun( 'ghq', '-', proper, _ ).
noun( 'gi', '-', count, _ ).
noun( '-', 'gis', count, _ ).
noun( 'gmt', '-', proper, _ ).
noun( 'gnp', '-', count, _ ).
noun( 'gp', '-', count, _ ).
noun( '-', 'gp\'s', count, _ ).
noun( 'gabon', '-', proper, loc ).
noun( 'gabonese', 'gabonese', count, _ ).
noun( 'gael', 'gaels', count, _ ).
noun( 'gaelic', 'gaelics', count, _ ).
noun( 'gainsborough', 'gainsboroughs', count, _ ).
noun( 'gainsborough', '-', proper, loc ).
noun( 'galashiels', '-', proper, loc ).
noun( 'galloway', '-', proper, loc ).
noun( 'gallup', '-', proper, _ ).
noun( 'galway', '-', proper, loc ).
noun( 'gambia', '-', proper, loc ).
noun( 'gambian', 'gambians', count, _ ).
noun( 'gareth', '-', proper, per ).
noun( 'garforth', '-', proper, loc ).
noun( 'garrowhill', '-', proper, loc ).
noun( 'gary', '-', proper, per ).
noun( 'gateshead', '-', proper, loc ).
noun( 'gaul', 'gauls', count, _ ).
noun( 'gavin', '-', proper, per ).
noun( 'gdansk', '-', proper, loc ).
noun( 'gdn', '-', proper, _ ).
noun( 'gdns', '-', proper, _ ).
noun( 'geiger', 'geigers', count, _ ).
noun( 'gelligaer', '-', proper, loc ).
noun( 'gelsenkirchen', '-', proper, loc ).
noun( 'gemini', '-', proper, _ ).
noun( 'gen', '-', proper, per ).
noun( 'gene', '-', proper, per ).
noun( 'geneva', '-', proper, loc ).
noun( 'genoa', '-', proper, loc ).
noun( 'genova', '-', proper, loc ).
noun( 'geoff', '-', proper, per ).
noun( 'geoffrey', '-', proper, per ).
noun( 'george', '-', proper, per ).
noun( 'georgia', '-', proper, loc ).
noun( 'georgian', 'georgians', count, _ ).
noun( 'georgie', '-', proper, per ).
noun( 'gerald', '-', proper, per ).
noun( 'geraldine', '-', proper, per ).
noun( 'gerard', '-', proper, per ).
noun( 'german', 'germans', both, _ ).
noun( 'germany', '-', proper, loc ).
noun( 'gerry', '-', proper, per ).
noun( 'gertie', '-', proper, per ).
noun( 'gertrude', '-', proper, per ).
noun( 'gestapo', 'gestapos', count, _ ).
noun( 'ghana', '-', proper, loc ).
noun( 'ghanaian', 'ghanaians', count, _ ).
noun( 'ghent', '-', proper, loc ).
noun( 'gibraltar', '-', proper, loc ).
noun( 'gibraltarian', 'gibraltarians', count, _ ).
noun( 'giffnock', '-', proper, loc ).
noun( 'gilbert', '-', proper, per ).
noun( 'giles', '-', proper, per ).
noun( 'gilfach goch', '-', proper, loc ).
noun( 'gill', '-', proper, per ).
noun( 'gillian', '-', proper, per ).
noun( 'gillingham', '-', proper, loc ).
noun( 'gina', '-', proper, per ).
noun( 'girvan', '-', proper, loc ).
noun( 'gladys', '-', proper, per ).
noun( 'glamorgan', '-', proper, loc ).
noun( 'glasgow', '-', proper, loc ).
noun( 'glastonbury', '-', proper, loc ).
noun( 'glaswegian', 'glaswegians', count, _ ).
noun( 'glen', '-', proper, per ).
noun( 'glengarry', 'glengarries', count, _ ).
noun( 'glenrothes', '-', proper, loc ).
noun( 'gloria', '-', proper, per ).
noun( 'glossop', '-', proper, loc ).
noun( 'gloucester', '-', proper, loc ).
noun( 'gloucestershire', '-', proper, loc ).
noun( 'glusburn', '-', proper, loc ).
noun( 'god', '-', proper, _ ).
noun( 'godalming', '-', proper, loc ).
noun( 'godfrey', '-', proper, per ).
noun( 'godmanchester', '-', proper, loc ).
noun( 'goffs oak', '-', proper, loc ).
noun( 'golbourne', '-', proper, loc ).
noun( 'goliath', 'goliaths', count, _ ).
noun( 'goole', '-', proper, loc ).
noun( 'gordon', '-', proper, per ).
noun( 'gorgon', 'gorgons', count, _ ).
noun( 'gorgonzola', '-', mass, _ ).
noun( 'gorky', '-', proper, loc ).
noun( 'gorseinon', '-', proper, loc ).
noun( 'gosport', '-', proper, loc ).
noun( 'goteborg', '-', proper, loc ).
noun( 'goth', 'goths', count, _ ).
noun( 'gothenburg', '-', proper, loc ).
noun( 'gothic', '-', mass, _ ).
noun( 'gottingen', '-', proper, loc ).
noun( 'gourock', '-', proper, loc ).
noun( 'gov', '-', proper, per ).
noun( 'governor-general', 'governor-generals', count, _ ).
noun( 'grampian', '-', proper, loc ).
noun( 'granada', '-', proper, loc ).
noun( 'grand prix', 'grands prix', count, _ ).
noun( 'grangemouth', '-', proper, loc ).
noun( 'grantham', '-', proper, loc ).
noun( 'gravenhage', '-', proper, loc ).
noun( 'gravesend', '-', proper, loc ).
noun( 'graz', '-', proper, loc ).
noun( 'great harwood', '-', proper, loc ).
noun( 'great malvern', '-', proper, loc ).
noun( 'great shelford', '-', proper, loc ).
noun( 'great yarmouth', '-', proper, loc ).
noun( 'greece', '-', proper, loc ).
noun( 'greek', 'greeks', both, _ ).
noun( 'greenock', '-', proper, loc ).
noun( 'greenwich', '-', proper, loc ).
noun( 'greg', '-', proper, per ).
noun( 'gregory', '-', proper, per ).
noun( 'grenada', '-', proper, loc ).
noun( 'grenadian', 'grenadians', count, _ ).
noun( 'grimethorpe', '-', proper, loc ).
noun( 'grimsby', '-', proper, loc ).
noun( 'grinstead', '-', proper, loc ).
noun( 'groningen', '-', proper, loc ).
noun( 'grundyism', '-', mass, _ ).
noun( 'guadalajara', '-', proper, loc ).
noun( 'guatemala', '-', proper, loc ).
noun( 'guatemala city', '-', proper, loc ).
noun( 'guatemalan', 'guatemalans', count, _ ).
noun( 'guayaquil', '-', proper, loc ).
noun( 'guild-hall', 'guild-halls', count, _ ).
noun( 'guildford', '-', proper, loc ).
noun( 'guildhall', 'guildhalls', count, _ ).
noun( 'guinea', '-', proper, loc ).
noun( 'guinean', 'guineans', count, _ ).
noun( 'guinness', 'guinnesses', count, _ ).
noun( 'guisborough', '-', proper, loc ).
noun( 'gurkha', 'gurkhas', count, _ ).
noun( 'guy', '-', proper, per ).
noun( 'guy\'s', '-', proper, _ ).
noun( 'guyana', '-', proper, loc ).
noun( 'guyanese', 'guyanese', count, _ ).
noun( 'gwen', '-', proper, per ).
noun( 'gwendoline', '-', proper, per ).
noun( 'gwent', '-', proper, loc ).
noun( 'gwersyllt', '-', proper, loc ).
noun( 'gwynedd', '-', proper, loc ).
noun( 'gypsy', 'gypsies', count, _ ).
noun( 'h-bomb', 'h-bombs', count, _ ).
noun( 'hm', '-', proper, per ).
noun( 'hmso', '-', proper, _ ).
noun( 'hp', '-', mass, _ ).
noun( 'hq', '-', proper, _ ).
noun( 'hrh', '-', proper, per ).
noun( 'haarlem', '-', proper, loc ).
noun( 'hackney', '-', proper, loc ).
noun( 'haddington', '-', proper, loc ).
noun( 'hades', '-', proper, _ ).
noun( 'hadji', 'hadjis', count, _ ).
noun( 'hagen', '-', proper, loc ).
noun( 'hagley', '-', proper, loc ).
noun( 'hailsham', '-', proper, loc ).
noun( 'haiphong', '-', proper, loc ).
noun( 'haiti', '-', proper, loc ).
noun( 'haitian', 'haitians', count, _ ).
noun( 'hal', '-', proper, per ).
noun( 'halesowen', '-', proper, loc ).
noun( 'halifax', '-', proper, loc ).
noun( 'halle', '-', proper, loc ).
noun( 'hallowe\'en', 'hallowe\'ens', count, _ ).
noun( 'hallowe\'en', '-', proper, _ ).
noun( 'halstead', '-', proper, loc ).
noun( 'hamburg', '-', proper, loc ).
noun( 'hamhung', '-', proper, loc ).
noun( 'hamilton', '-', proper, loc ).
noun( 'hammersmith', '-', proper, loc ).
noun( 'hampreston', '-', proper, loc ).
noun( 'hampshire', '-', proper, loc ).
noun( 'hangchon', '-', proper, loc ).
noun( 'hanoi', '-', proper, loc ).
noun( 'hanover', '-', proper, loc ).
noun( 'hansard', 'hansards', count, _ ).
noun( 'harbin', '-', proper, loc ).
noun( 'haringey', '-', proper, loc ).
noun( 'harley street', '-', proper, _ ).
noun( 'harlow', '-', proper, loc ).
noun( 'harold', '-', proper, per ).
noun( 'harpenden', '-', proper, loc ).
noun( 'harriet', '-', proper, per ).
noun( 'harrogate', '-', proper, loc ).
noun( 'harrow', '-', proper, loc ).
noun( 'harry', '-', proper, per ).
noun( 'hartlepool', '-', proper, loc ).
noun( 'hartley', '-', proper, loc ).
noun( 'harvey', '-', proper, per ).
noun( 'harwich', '-', proper, loc ).
noun( 'harwood', '-', proper, loc ).
noun( 'harworth', '-', proper, loc ).
noun( 'haslemere', '-', proper, loc ).
noun( 'haslingden', '-', proper, loc ).
noun( 'hastings', '-', proper, loc ).
noun( 'hatfield', '-', proper, loc ).
noun( 'havana', 'havanas', count, _ ).
noun( 'havana', '-', proper, loc ).
noun( 'haverfordwest', '-', proper, loc ).
noun( 'haverhill', '-', proper, loc ).
noun( 'havering', '-', proper, loc ).
noun( 'hawaii', '-', proper, loc ).
noun( 'hawick', '-', proper, loc ).
noun( 'hawke\'s bay', '-', proper, loc ).
noun( 'hayling', '-', proper, loc ).
noun( 'haywards heath', '-', proper, loc ).
noun( 'hazel', '-', proper, per ).
noun( 'heanor', '-', proper, loc ).
noun( 'heather', '-', proper, per ).
noun( 'heaviside', '-', proper, _ ).
noun( 'hebden royal', '-', proper, loc ).
noun( 'hebrew', 'hebrews', both, _ ).
noun( 'hedge end', '-', proper, loc ).
noun( 'hegira', 'hegiras', count, _ ).
noun( 'heidelburg', '-', proper, loc ).
noun( 'hejira', 'hejiras', count, _ ).
noun( 'helen', '-', proper, per ).
noun( 'helensburgh', '-', proper, loc ).
noun( 'hellene', 'hellenes', count, _ ).
noun( 'helsinki', '-', proper, loc ).
noun( 'helston', '-', proper, loc ).
noun( 'hemel hempstead', '-', proper, loc ).
noun( 'hemsworth', '-', proper, loc ).
noun( 'henley-on-thames', '-', proper, loc ).
noun( 'henry', '-', proper, per ).
noun( 'herb', '-', proper, per ).
noun( 'herbert', '-', proper, per ).
noun( 'hereford', '-', proper, loc ).
noun( 'herne', '-', proper, loc ).
noun( 'herne bay', '-', proper, loc ).
noun( 'herr', 'herren', count, _ ).
noun( 'hertford', '-', proper, loc ).
noun( 'hertfordshire', '-', proper, loc ).
noun( 'hetton', '-', proper, loc ).
noun( 'hexham', '-', proper, loc ).
noun( 'heysham', '-', proper, loc ).
noun( 'heywood', '-', proper, loc ).
noun( 'high spen', '-', proper, loc ).
noun( 'high wycombe', '-', proper, loc ).
noun( 'highland', '-', proper, loc ).
noun( 'highlander', 'highlanders', count, _ ).
noun( 'hilary', '-', proper, per ).
noun( 'hilda', '-', proper, per ).
noun( 'hillingdon', '-', proper, loc ).
noun( 'hinckley', '-', proper, loc ).
noun( 'hindi', '-', mass, _ ).
noun( 'hindu', 'hindus', count, _ ).
noun( 'hinduism', '-', mass, _ ).
noun( 'hindustani', 'hindustanis', count, _ ).
noun( 'hiroshima', '-', proper, loc ).
noun( 'hitchin', '-', proper, loc ).
noun( 'hobson', '-', proper, _ ).
noun( 'hockley', '-', proper, loc ).
noun( 'hoddesdon', '-', proper, loc ).
noun( 'hofei', '-', proper, loc ).
noun( 'hogmanay', 'hogmanays', count, _ ).
noun( 'hogmanay', '-', proper, _ ).
noun( 'holland', '-', proper, loc ).
noun( 'hollander', 'hollanders', count, _ ).
noun( 'hollywood', '-', proper, _ ).
noun( 'holmfirth', '-', proper, loc ).
noun( 'holyhead', '-', proper, loc ).
noun( 'hon', '-', proper, per ).
noun( 'honduran', 'hondurans', count, _ ).
noun( 'honduras', '-', proper, loc ).
noun( 'hong kong', '-', proper, loc ).
noun( 'honiton', '-', proper, loc ).
noun( 'hoo', '-', proper, loc ).
noun( 'hoover', 'hoovers', count, _ ).
noun( 'hope', '-', proper, loc ).
noun( 'horace', '-', proper, per ).
noun( 'horley', '-', proper, loc ).
noun( 'hornsea', '-', proper, loc ).
noun( 'horsham', '-', proper, loc ).
noun( 'horsley', '-', proper, loc ).
noun( 'horwich', '-', proper, loc ).
noun( 'hosp', '-', proper, _ ).
noun( 'host', '-', count, _ ).
noun( 'houghton', '-', proper, loc ).
noun( 'hounslow', '-', proper, loc ).
noun( 'houston', '-', proper, loc ).
noun( 'hove', '-', proper, loc ).
noun( 'howard', '-', proper, per ).
noun( 'hoylake', '-', proper, loc ).
noun( 'hoyland nether', '-', proper, loc ).
noun( 'hubert', '-', proper, per ).
noun( 'hucknall', '-', proper, loc ).
noun( 'huddersfield', '-', proper, loc ).
noun( 'hugh', '-', proper, per ).
noun( 'hughie', '-', proper, per ).
noun( 'huguenot', 'huguenots', count, _ ).
noun( 'huhehot', '-', proper, loc ).
noun( 'humberside', '-', proper, loc ).
noun( 'humberston', '-', proper, loc ).
noun( 'humphrey', '-', proper, per ).
noun( 'hun', 'huns', count, _ ).
noun( 'hungarian', 'hungarians', both, _ ).
noun( 'hungary', '-', proper, loc ).
noun( 'huntingdon', '-', proper, loc ).
noun( 'hurstpierpoint', '-', proper, loc ).
noun( 'hwainan', '-', proper, loc ).
noun( 'hyde', '-', proper, loc ).
noun( 'hyderabad', '-', proper, loc ).
noun( 'hythe', '-', proper, loc ).
noun( 'iba', '-', count, _ ).
noun( 'icbm', '-', count, _ ).
noun( 'ilo', '-', count, _ ).
noun( 'imf', '-', count, _ ).
noun( 'iou', '-', count, _ ).
noun( '-', 'ious', count, _ ).
noun( 'iq', '-', mass, _ ).
noun( 'ira', '-', count, _ ).
noun( 'itv', '-', count, _ ).
noun( 'iud', '-', count, _ ).
noun( 'ian', '-', proper, per ).
noun( 'ibadan', '-', proper, loc ).
noun( 'iceland', '-', proper, loc ).
noun( 'icelander', 'icelanders', count, _ ).
noun( 'icelandic', '-', mass, _ ).
noun( 'ida', '-', proper, per ).
noun( 'idaho', '-', proper, loc ).
noun( 'ilfracombe', '-', proper, loc ).
noun( 'ilkeston', '-', proper, loc ).
noun( 'ilkley', '-', proper, loc ).
noun( 'illinois', '-', proper, loc ).
noun( 'immingham', '-', proper, loc ).
noun( 'inc', '-', proper, _ ).
noun( 'inchon', '-', proper, loc ).
noun( 'india', '-', proper, loc ).
noun( 'india-rubber', 'india-rubbers', both, _ ).
noun( 'indiaman', 'indiamen', count, _ ).
noun( 'indian', 'indians', count, _ ).
noun( 'indiana', '-', proper, loc ).
noun( 'indianapolis', '-', proper, loc ).
noun( 'indies', '-', proper, loc ).
noun( 'indonesia', '-', proper, loc ).
noun( 'indonesian', 'indonesians', both, _ ).
noun( 'indore', '-', proper, loc ).
noun( 'ingatestone', '-', proper, loc ).
noun( 'ingrid', '-', proper, per ).
noun( 'innsbruck', '-', proper, loc ).
noun( 'inst', '-', proper, _ ).
noun( 'interpol', '-', proper, _ ).
noun( 'inverkeithing', '-', proper, loc ).
noun( 'inverness', '-', proper, loc ).
noun( 'inverurie', '-', proper, loc ).
noun( 'iowa', '-', proper, loc ).
noun( 'ipswich', '-', proper, loc ).
noun( 'iran', '-', proper, loc ).
noun( 'iranian', 'iranians', both, _ ).
noun( 'iraq', '-', proper, loc ).
noun( 'iraqi', 'iraqis', count, _ ).
noun( 'ireland', '-', proper, loc ).
noun( 'irene', '-', proper, per ).
noun( 'iris', '-', proper, per ).
noun( 'irish', '-', mass, _ ).
noun( 'irishman', 'irishmen', count, _ ).
noun( 'irishwoman', 'irishwomen', count, _ ).
noun( 'irthlingborough', '-', proper, loc ).
noun( 'irvine', '-', proper, loc ).
noun( 'isaac', '-', proper, per ).
noun( 'isabel', '-', proper, per ).
noun( 'isabella', '-', proper, per ).
noun( 'islam', '-', mass, _ ).
noun( 'islington', '-', proper, loc ).
noun( 'isobel', '-', proper, per ).
noun( 'israel', '-', proper, loc ).
noun( 'israeli', 'israelis', count, _ ).
noun( 'istanbul', '-', proper, loc ).
noun( 'italian', 'italians', both, _ ).
noun( 'italy', '-', proper, loc ).
noun( 'ivan', '-', proper, per ).
noun( 'iver', '-', proper, loc ).
noun( 'ivor', '-', proper, per ).
noun( 'ivy', '-', proper, per ).
noun( 'izmir', '-', proper, loc ).
noun( 'jc', '-', proper, _ ).
noun( 'jp', '-', count, _ ).
noun( 'jabalpur', '-', proper, loc ).
noun( 'jack', '-', proper, per ).
noun( 'jackie', '-', proper, per ).
noun( 'jacksonville', '-', proper, loc ).
noun( 'jacob', '-', proper, per ).
noun( 'jacobin', 'jacobins', count, _ ).
noun( 'jacobinism', '-', mass, _ ).
noun( 'jacobite', 'jacobites', count, _ ).
noun( 'jacqueline', '-', proper, per ).
noun( 'jaipur', '-', proper, loc ).
noun( 'jakarta', '-', proper, loc ).
noun( 'jake', '-', proper, per ).
noun( 'jamaica', '-', proper, loc ).
noun( 'jamaican', 'jamaicans', count, _ ).
noun( 'james', '-', proper, per ).
noun( 'jan', '-', proper, per ).
noun( 'jan', '-', proper, _ ).
noun( 'jane', '-', proper, per ).
noun( 'janet', '-', proper, per ).
noun( 'janice', '-', proper, per ).
noun( 'janie', '-', proper, per ).
noun( 'january', 'januaries', count, _ ).
noun( 'january', '-', proper, _ ).
noun( 'janus', '-', proper, _ ).
noun( 'japan', '-', proper, loc ).
noun( 'japanese', 'japanese', both, _ ).
noun( 'jarrow', '-', proper, loc ).
noun( 'jason', '-', proper, per ).
noun( 'java', '-', proper, loc ).
noun( 'javanese', 'javanese', count, _ ).
noun( 'jean', '-', proper, per ).
noun( 'jeanie', '-', proper, per ).
noun( 'jedburgh', '-', proper, loc ).
noun( 'jeff', '-', proper, per ).
noun( 'jeffrey', '-', proper, per ).
noun( 'jehovah', '-', proper, _ ).
noun( 'jekyll-and-hyde', '-', proper, _ ).
noun( 'jennifer', '-', proper, per ).
noun( 'jenny', '-', proper, per ).
noun( 'jeremy', '-', proper, per ).
noun( 'jerome', '-', proper, per ).
noun( 'jerry', '-', proper, per ).
noun( 'jersey', '-', proper, loc ).
noun( 'jess', '-', proper, per ).
noun( 'jessica', '-', proper, per ).
noun( 'jessie', '-', proper, per ).
noun( 'jesuit', 'jesuits', count, _ ).
noun( 'jesus', '-', proper, per ).
noun( 'jew', 'jews', count, _ ).
noun( 'jewess', 'jewesses', count, _ ).
noun( 'jezebel', 'jezebels', count, _ ).
noun( 'jill', '-', proper, per ).
noun( 'jim', '-', proper, per ).
noun( 'jim crow', '-', count, _ ).
noun( 'jimmy', '-', proper, per ).
noun( 'jnr', '-', proper, _ ).
noun( 'jo', '-', proper, per ).
noun( 'joan', '-', proper, per ).
noun( 'joanie', '-', proper, per ).
noun( 'joann', '-', proper, per ).
noun( 'joanna', '-', proper, per ).
noun( 'joanne', '-', proper, per ).
noun( 'job', '-', proper, per ).
noun( 'jocelyn', '-', proper, per ).
noun( 'joe', '-', proper, per ).
noun( 'joey', '-', proper, per ).
noun( 'johannesburg', '-', proper, loc ).
noun( 'john', '-', proper, per ).
noun( 'john bull', '-', count, _ ).
noun( 'johnny', '-', proper, per ).
noun( 'johnstone', '-', proper, loc ).
noun( 'jonah', '-', proper, per ).
noun( 'jonathan', '-', proper, per ).
noun( 'jordan', '-', proper, loc ).
noun( 'jordanian', 'jordanians', count, _ ).
noun( 'joseph', '-', proper, per ).
noun( 'josephine', '-', proper, per ).
noun( 'josh', '-', proper, per ).
noun( 'joshua', '-', proper, per ).
noun( 'jove', '-', proper, _ ).
noun( 'joy', '-', proper, per ).
noun( 'joyce', '-', proper, per ).
noun( 'judaism', '-', mass, _ ).
noun( 'judas', 'judases', count, _ ).
noun( 'judas', '-', proper, per ).
noun( 'judith', '-', proper, per ).
noun( 'judy', '-', proper, per ).
noun( 'jul', '-', proper, _ ).
noun( 'julia', '-', proper, per ).
noun( 'julian', '-', proper, per ).
noun( 'julie', '-', proper, per ).
noun( 'juliet', '-', proper, per ).
noun( 'july', '-', count, _ ).
noun( 'jun', '-', proper, _ ).
noun( 'june', 'junes', count, _ ).
noun( 'june', '-', proper, per ).
noun( 'jupiter', '-', proper, _ ).
noun( 'justin', '-', proper, per ).
noun( 'kgb', '-', count, _ ).
noun( 'ko', '-', count, _ ).
noun( 'kaiser', 'kaisers', count, _ ).
noun( 'kalgan', '-', proper, loc ).
noun( 'kampuchea', '-', proper, loc ).
noun( 'kampuchean', 'kampucheans', count, _ ).
noun( 'kanpur', '-', proper, loc ).
noun( 'kansas', '-', proper, loc ).
noun( 'kansas city', '-', proper, loc ).
noun( 'kaohsiung', '-', proper, loc ).
noun( 'karachi', '-', proper, loc ).
noun( 'karaganda', '-', proper, loc ).
noun( 'karen', '-', proper, per ).
noun( 'karlsruhe', '-', proper, loc ).
noun( 'kashmir', '-', proper, loc ).
noun( 'kashmiri', 'kashmiris', count, _ ).
noun( 'kassel', '-', proper, loc ).
noun( 'kate', '-', proper, per ).
noun( 'kath', '-', proper, per ).
noun( 'katherine', '-', proper, per ).
noun( 'kathy', '-', proper, per ).
noun( 'katie', '-', proper, per ).
noun( 'katowice', '-', proper, loc ).
noun( 'kay', '-', proper, per ).
noun( 'kazan', '-', proper, loc ).
noun( 'keighley', '-', proper, loc ).
noun( 'keith', '-', proper, per ).
noun( 'kelso', '-', proper, loc ).
noun( 'kelt', 'kelts', count, _ ).
noun( 'kelty', '-', proper, loc ).
noun( 'kemsing', '-', proper, loc ).
noun( 'ken', '-', proper, per ).
noun( 'kendal', '-', proper, loc ).
noun( 'kenilworth', '-', proper, loc ).
noun( 'kenneth', '-', proper, per ).
noun( 'kenny', '-', proper, per ).
noun( 'kensington', '-', proper, loc ).
noun( 'kent', '-', proper, loc ).
noun( 'kentucky', '-', proper, loc ).
noun( 'kenya', '-', proper, loc ).
noun( 'kenyan', 'kenyans', count, _ ).
noun( 'kerry', '-', proper, loc ).
noun( 'kettering', '-', proper, loc ).
noun( 'kevin', '-', proper, per ).
noun( 'keynsham', '-', proper, loc ).
noun( 'keyworth', '-', proper, loc ).
noun( 'khalka', '-', mass, _ ).
noun( 'kharkov', '-', proper, loc ).
noun( 'khartoum', '-', proper, loc ).
noun( 'khmer', '-', mass, _ ).
noun( 'kidderminster', '-', proper, loc ).
noun( 'kidlington', '-', proper, loc ).
noun( 'kidsgrove', '-', proper, loc ).
noun( 'kiev', '-', proper, loc ).
noun( 'kilbirnie', '-', proper, loc ).
noun( 'kildare', '-', proper, loc ).
noun( 'kilkenny', '-', proper, loc ).
noun( 'killarney', '-', proper, loc ).
noun( 'kilmarnock', '-', proper, loc ).
noun( 'kilsyth', '-', proper, loc ).
noun( 'kilwinning', '-', proper, loc ).
noun( 'kimberley', '-', proper, loc ).
noun( 'king\'s lynn', '-', proper, loc ).
noun( 'kingsbury', '-', proper, loc ).
noun( 'kingsteignton', '-', proper, loc ).
noun( 'kingston', '-', proper, loc ).
noun( 'kingston upon hull', '-', proper, loc ).
noun( 'kingston upon thames', '-', proper, loc ).
noun( 'kinshasa', '-', proper, loc ).
noun( 'kippax', '-', proper, loc ).
noun( 'kirin', '-', proper, loc ).
noun( 'kirkby', '-', proper, loc ).
noun( 'kirkby in ashfield', '-', proper, loc ).
noun( 'kirkcaldy', '-', proper, loc ).
noun( 'kirkham', '-', proper, loc ).
noun( 'kirkintilloch', '-', proper, loc ).
noun( 'kirkwall', '-', proper, loc ).
noun( 'kitakyushu', '-', proper, loc ).
noun( 'kitty', '-', proper, per ).
noun( 'knaresborough', '-', proper, loc ).
noun( 'knesset', 'knessets', count, _ ).
noun( 'knottingley', '-', proper, loc ).
noun( 'knutsford', '-', proper, loc ).
noun( 'kobe', '-', proper, loc ).
noun( 'koblenz', '-', proper, loc ).
noun( 'koln', '-', proper, loc ).
noun( 'koran', 'korans', count, _ ).
noun( 'korea', '-', proper, loc ).
noun( 'korean', 'koreans', both, _ ).
noun( 'krakow', '-', proper, loc ).
noun( 'krasnoyarsk', '-', proper, loc ).
noun( 'krefeld', '-', proper, loc ).
noun( 'kremlin', 'kremlins', count, _ ).
noun( 'krivoi rog', '-', proper, loc ).
noun( 'kunming', '-', proper, loc ).
noun( 'kuwait', '-', proper, loc ).
noun( 'kuwait city', '-', proper, loc ).
noun( 'kuwaiti', 'kuwaitis', count, _ ).
noun( 'kuybyshev', '-', proper, loc ).
noun( 'kweiyang', '-', proper, loc ).
noun( 'kyoto', '-', proper, loc ).
noun( 'l-plate', 'l-plates', count, _ ).
noun( 'la', '-', proper, _ ).
noun( 'lea', '-', count, _ ).
noun( 'llb', '-', count, _ ).
noun( 'lp', '-', count, _ ).
noun( '-', 'lps', count, _ ).
noun( 'lsd', '-', mass, _ ).
noun( 'la habana', '-', proper, loc ).
noun( 'la paz', '-', proper, loc ).
noun( 'la plata', '-', proper, loc ).
noun( 'la spezia', '-', proper, loc ).
noun( 'lab', '-', proper, _ ).
noun( 'labourite', 'labourites', count, _ ).
noun( 'labrador', '-', proper, loc ).
noun( 'lady-chapel', 'lady-chapels', count, _ ).
noun( 'lagos', '-', proper, loc ).
noun( 'lahore', '-', proper, loc ).
noun( 'lakenheath', '-', proper, loc ).
noun( 'lambeth', '-', proper, loc ).
noun( 'lammas', 'lammases', count, _ ).
noun( 'lammas', '-', proper, _ ).
noun( 'lanark', '-', proper, loc ).
noun( 'lancashire', '-', proper, loc ).
noun( 'lancaster', '-', proper, loc ).
noun( 'lancastrian', 'lancastrians', count, _ ).
noun( 'lanchow', '-', proper, loc ).
noun( 'lancing', '-', proper, loc ).
noun( 'landrover', 'landrovers', count, _ ).
noun( 'lao', '-', mass, _ ).
noun( 'laos', '-', proper, loc ).
noun( 'laotian', 'laotians', both, _ ).
noun( 'largs', '-', proper, loc ).
noun( 'larkhall', '-', proper, loc ).
noun( 'larne', '-', proper, loc ).
noun( 'larry', '-', proper, per ).
noun( 'las palmas', '-', proper, loc ).
noun( 'latin', 'latins', count, _ ).
noun( 'latinist', 'latinists', count, _ ).
noun( 'laura', '-', proper, per ).
noun( 'laurence', '-', proper, per ).
noun( 'lausanne', '-', proper, loc ).
noun( 'lawrence', '-', proper, per ).
noun( 'lazarus', '-', count, _ ).
noun( 'lazarus', '-', proper, per ).
noun( 'le havre', '-', proper, loc ).
noun( 'le mans', '-', proper, loc ).
noun( 'leamington spa', '-', proper, loc ).
noun( 'leatherhead', '-', proper, loc ).
noun( 'lebanese', 'lebanese', count, _ ).
noun( 'lebanon', '-', proper, loc ).
noun( 'leeds', '-', proper, loc ).
noun( 'leek', '-', proper, loc ).
noun( 'leicester', '-', proper, loc ).
noun( 'leicestershire', '-', proper, loc ).
noun( 'leiden', '-', proper, loc ).
noun( 'leigh', '-', proper, loc ).
noun( 'leighton-linslade', '-', proper, loc ).
noun( 'leinster', '-', proper, loc ).
noun( 'leipzig', '-', proper, loc ).
noun( 'leitrim', '-', proper, loc ).
noun( 'leix', '-', proper, loc ).
noun( 'len', '-', proper, per ).
noun( 'leningrad', '-', proper, loc ).
noun( 'lenny', '-', proper, per ).
noun( 'lens', '-', proper, loc ).
noun( 'lent', 'lents', count, _ ).
noun( 'lent', '-', proper, _ ).
noun( 'leo', '-', proper, per ).
noun( 'leominster', '-', proper, loc ).
noun( 'leon', '-', proper, loc ).
noun( 'leonard', '-', proper, per ).
noun( 'lerwick', '-', proper, loc ).
noun( 'les', '-', proper, per ).
noun( 'lesley', '-', proper, per ).
noun( 'leslie', '-', proper, per ).
noun( 'lesotho', '-', proper, loc ).
noun( 'letchworth', '-', proper, loc ).
noun( 'lethe', '-', proper, _ ).
noun( 'levant', '-', proper, _ ).
noun( 'levantine', 'levantines', count, _ ).
noun( 'leven', '-', proper, loc ).
noun( 'leverkusen', '-', proper, loc ).
noun( 'lewes', '-', proper, loc ).
noun( 'lewis', '-', proper, per ).
noun( 'lewisham', '-', proper, loc ).
noun( 'leyland', '-', proper, loc ).
noun( 'lib', '-', proper, _ ).
noun( 'liberia', '-', proper, loc ).
noun( 'liberian', 'liberians', count, _ ).
noun( 'libra', '-', proper, _ ).
noun( 'libya', '-', proper, loc ).
noun( 'libyan', 'libyans', count, _ ).
noun( 'lichfield', '-', proper, loc ).
noun( 'liechtenstein', '-', proper, loc ).
noun( 'liechtensteiner', 'liechtensteiners', count, _ ).
noun( 'liege', '-', proper, loc ).
noun( 'lilian', '-', proper, per ).
noun( 'lille', '-', proper, loc ).
noun( 'lilliputian', 'lilliputians', count, _ ).
noun( 'lily', '-', proper, per ).
noun( 'lima', '-', proper, loc ).
noun( 'limerick', '-', proper, loc ).
noun( 'limoges', '-', proper, loc ).
noun( 'lincoln', '-', proper, loc ).
noun( 'lincolnshire', '-', proper, loc ).
noun( 'linda', '-', proper, per ).
noun( 'linlithgow', '-', proper, loc ).
noun( 'linwood', '-', proper, loc ).
noun( 'linz', '-', proper, loc ).
noun( 'lionel', '-', proper, per ).
noun( 'lisa', '-', proper, per ).
noun( 'lisbon', '-', proper, loc ).
noun( 'liskeard', '-', proper, loc ).
noun( 'littlehampton', '-', proper, loc ).
noun( 'liverpool', '-', proper, loc ).
noun( 'liverpudlian', 'liverpudlians', count, _ ).
noun( 'livingston', '-', proper, loc ).
noun( 'livorno', '-', proper, loc ).
noun( 'liz', '-', proper, per ).
noun( 'liza', '-', proper, per ).
noun( 'lizzy', '-', proper, per ).
noun( 'ljubljana', '-', proper, loc ).
noun( 'llandudno', '-', proper, loc ).
noun( 'llanelli', '-', proper, loc ).
noun( 'llangollen', '-', proper, loc ).
noun( 'llantrisant', '-', proper, loc ).
noun( 'llantwit major', '-', proper, loc ).
noun( 'lloyd\'s', '-', proper, _ ).
noun( 'loanhead', '-', proper, loc ).
noun( 'lochgelly', '-', proper, loc ).
noun( 'lodz', '-', proper, loc ).
noun( 'loftus', '-', proper, loc ).
noun( 'lois', '-', proper, per ).
noun( 'london', '-', proper, loc ).
noun( 'londonderry', '-', proper, loc ).
noun( 'londoner', 'londoners', count, _ ).
noun( 'long eaton', '-', proper, loc ).
noun( 'longridge', '-', proper, loc ).
noun( 'longton', '-', proper, loc ).
noun( 'lorna', '-', proper, per ).
noun( 'los angeles', '-', proper, loc ).
noun( 'lossiemouth', '-', proper, loc ).
noun( 'lothian', '-', proper, loc ).
noun( 'loughborough', '-', proper, loc ).
noun( 'louis', '-', proper, per ).
noun( 'louise', '-', proper, per ).
noun( 'louisiana', '-', proper, loc ).
noun( 'louisville', '-', proper, loc ).
noun( 'louth', '-', proper, loc ).
noun( 'lowestoft', '-', proper, loc ).
noun( 'loyang', '-', proper, loc ).
noun( 'lt', '-', proper, per ).
noun( 'ltd', '-', proper, _ ).
noun( 'lubeck', '-', proper, loc ).
noun( 'lucifer', 'lucifers', count, _ ).
noun( 'lucknow', '-', proper, loc ).
noun( 'lucy', '-', proper, per ).
noun( 'luddite', 'luddites', count, _ ).
noun( 'ludlow', '-', proper, loc ).
noun( 'ludwigshafen', '-', proper, loc ).
noun( 'luke', '-', proper, per ).
noun( 'luta', '-', proper, loc ).
noun( 'lutheran', 'lutherans', count, _ ).
noun( 'luton', '-', proper, loc ).
noun( 'luxembourg', '-', proper, loc ).
noun( 'luxemburg', '-', proper, loc ).
noun( 'luxemburger', 'luxemburgers', count, _ ).
noun( 'lvov', '-', proper, loc ).
noun( 'lyallpur', '-', proper, loc ).
noun( 'lydia', '-', proper, per ).
noun( 'lydney', '-', proper, loc ).
noun( 'lymington', '-', proper, loc ).
noun( 'lymm', '-', proper, loc ).
noun( 'lynn', '-', proper, per ).
noun( 'lynne', '-', proper, per ).
noun( 'lyons', '-', proper, loc ).
noun( 'lytham', '-', proper, loc ).
noun( 'lytham st annes', '-', proper, loc ).
noun( 'ma', '-', count, _ ).
noun( 'mb', '-', count, _ ).
noun( 'mc', '-', count, _ ).
noun( 'mcc', '-', count, _ ).
noun( 'md', '-', count, _ ).
noun( 'mi5', '-', proper, _ ).
noun( 'mi6', '-', proper, _ ).
noun( 'mp', '-', count, _ ).
noun( '-', 'mp\'s', count, _ ).
noun( 'ms', '-', count, _ ).
noun( 'msc', '-', count, _ ).
noun( 'maastricht', '-', proper, loc ).
noun( 'mabel', '-', proper, per ).
noun( 'macclesfield', '-', proper, loc ).
noun( 'macedonian', 'macedonians', both, _ ).
noun( 'mach', '-', proper, _ ).
noun( 'madagascan', 'madagascans', count, _ ).
noun( 'madagascar', '-', proper, loc ).
noun( 'madame', 'madames', count, _ ).
noun( 'maddy', '-', proper, per ).
noun( 'madeira', 'madeiras', count, _ ).
noun( 'madeleine', '-', proper, per ).
noun( 'mademoiselle', 'mademoiselles', count, _ ).
noun( 'madge', '-', proper, per ).
noun( 'madonna', 'madonnas', count, _ ).
noun( 'madras', '-', proper, loc ).
noun( 'madrid', '-', proper, loc ).
noun( 'madurai', '-', proper, loc ).
noun( 'maecenas', '-', proper, _ ).
noun( 'maesteg', '-', proper, loc ).
noun( 'mafia', 'mafias', count, _ ).
noun( 'magdeburg', '-', proper, loc ).
noun( 'maggie', '-', proper, per ).
noun( 'maghull', '-', proper, loc ).
noun( 'magnificat', 'magnificats', count, _ ).
noun( 'magyar', 'magyars', both, _ ).
noun( 'maharaja', 'maharajas', count, _ ).
noun( 'maharajah', 'maharajahs', count, _ ).
noun( 'maharanee', 'maharanees', count, _ ).
noun( 'mahatma', 'mahatmas', count, _ ).
noun( 'maidenhead', '-', proper, loc ).
noun( 'maidstone', '-', proper, loc ).
noun( 'maine', '-', proper, loc ).
noun( 'mainz', '-', proper, loc ).
noun( 'maj', '-', proper, per ).
noun( 'malaga', '-', proper, loc ).
noun( 'malawi', '-', proper, loc ).
noun( 'malawian', 'malawians', count, _ ).
noun( 'malay', 'malays', both, _ ).
noun( 'malaya', '-', proper, loc ).
noun( 'malayan', 'malayans', count, _ ).
noun( 'malaysia', '-', proper, loc ).
noun( 'malaysian', 'malaysians', count, _ ).
noun( 'malcolm', '-', proper, per ).
noun( 'maldon', '-', proper, loc ).
noun( 'mali', '-', proper, loc ).
noun( 'malian', 'malians', count, _ ).
noun( 'mallow', '-', proper, loc ).
noun( 'malmo', '-', proper, loc ).
noun( 'malta', '-', proper, loc ).
noun( 'maltby', '-', proper, loc ).
noun( 'maltese', 'maltese', both, _ ).
noun( 'malvern', '-', proper, loc ).
noun( 'mamie', '-', proper, per ).
noun( 'manchester', '-', proper, loc ).
noun( 'mancunian', 'mancunians', count, _ ).
noun( 'mandy', '-', proper, per ).
noun( 'manila', '-', proper, loc ).
noun( 'manilla', 'manillas', count, _ ).
noun( 'manitoba', '-', proper, loc ).
noun( 'mannheim', '-', proper, loc ).
noun( 'mansfield', '-', proper, loc ).
noun( 'manx', '-', mass, _ ).
noun( 'maoism', '-', mass, _ ).
noun( 'maoist', 'maoists', count, _ ).
noun( 'maori', 'maoris', count, _ ).
noun( 'mar', '-', proper, _ ).
noun( 'maracaibo', '-', proper, loc ).
noun( 'march', 'marches', count, _ ).
noun( 'march', '-', proper, loc ).
noun( 'mardi gras', 'mardi gras', count, _ ).
noun( 'margaret', '-', proper, per ).
noun( 'margate', '-', proper, loc ).
noun( 'marge', '-', proper, per ).
noun( 'margery', '-', proper, per ).
noun( 'margie', '-', proper, per ).
noun( 'margrave', 'margraves', count, _ ).
noun( 'maria', '-', proper, per ).
noun( 'marian', '-', proper, per ).
noun( 'marie', '-', proper, per ).
noun( 'marilyn', '-', proper, per ).
noun( 'marion', '-', proper, per ).
noun( 'marjorie', '-', proper, per ).
noun( 'mark', '-', proper, per ).
noun( 'market drayton', '-', proper, loc ).
noun( 'market harborough', '-', proper, loc ).
noun( 'marlborough', '-', proper, loc ).
noun( 'marlene', '-', proper, per ).
noun( 'marlow', '-', proper, loc ).
noun( 'marple', '-', proper, loc ).
noun( 'mars', '-', proper, _ ).
noun( 'marsala', '-', mass, _ ).
noun( 'marseillaise', 'marseillaises', count, _ ).
noun( 'marseilles', '-', proper, loc ).
noun( 'marske', '-', proper, loc ).
noun( 'martha', '-', proper, per ).
noun( 'martian', 'martians', count, _ ).
noun( 'martin', '-', proper, per ).
noun( 'marxism', '-', mass, _ ).
noun( 'marxist', 'marxists', count, _ ).
noun( 'mary', '-', proper, per ).
noun( 'maryland', '-', proper, loc ).
noun( 'maryport', '-', proper, loc ).
noun( 'mason-dixon', '-', proper, _ ).
noun( 'mass', 'masses', both, _ ).
noun( 'massachusetts', '-', proper, loc ).
noun( 'matlock', '-', proper, loc ).
noun( 'matt', '-', proper, per ).
noun( 'matthew', '-', proper, per ).
noun( 'maud', '-', proper, per ).
noun( 'maudie', '-', proper, per ).
noun( 'maundy thursday', 'maundy thursdays', count, _ ).
noun( 'maundy thursday', '-', proper, _ ).
noun( 'maureen', '-', proper, per ).
noun( 'maurice', '-', proper, per ).
noun( 'mauritania', '-', proper, loc ).
noun( 'mauritanian', 'mauritanians', count, _ ).
noun( 'mauritian', 'mauritians', count, _ ).
noun( 'mauritius', '-', proper, loc ).
noun( 'mavis', '-', proper, per ).
noun( 'max', '-', proper, per ).
noun( 'maxine', '-', proper, per ).
noun( 'may', 'mays', count, _ ).
noun( 'may', '-', proper, per ).
noun( 'mayfair', '-', proper, _ ).
noun( 'mayfield', '-', proper, loc ).
noun( 'mayo', '-', proper, loc ).
noun( 'mccarthyism', '-', mass, _ ).
noun( 'meath', '-', proper, loc ).
noun( 'mecca', '-', proper, loc ).
noun( 'med', '-', count, _ ).
noun( 'medan', '-', proper, loc ).
noun( 'medellin', '-', proper, loc ).
noun( 'medicare', '-', mass, _ ).
noun( 'meg', '-', proper, per ).
noun( 'melbourne', '-', proper, loc ).
noun( 'melcombe', '-', proper, loc ).
noun( 'melcombe regis', '-', proper, loc ).
noun( 'melksham', '-', proper, loc ).
noun( 'meltham', '-', proper, loc ).
noun( 'melton mowbray', '-', proper, loc ).
noun( 'memphis', '-', proper, loc ).
noun( 'meopham', '-', proper, loc ).
noun( 'mercator', '-', proper, _ ).
noun( 'mercury', '-', proper, _ ).
noun( 'merthyr tydfil', '-', proper, loc ).
noun( 'merton', '-', proper, loc ).
noun( 'messiah', 'messiahs', count, _ ).
noun( 'messina', '-', proper, loc ).
noun( 'met', '-', proper, _ ).
noun( 'methodism', '-', mass, _ ).
noun( 'methodist', 'methodists', count, _ ).
noun( 'methuselah', '-', proper, _ ).
noun( 'metro', 'metros', count, _ ).
noun( 'metz', '-', proper, loc ).
noun( 'mexican', 'mexicans', count, _ ).
noun( 'mexico', '-', proper, loc ).
noun( 'mexico city', '-', proper, loc ).
noun( 'mgr', '-', proper, per ).
noun( 'miami', '-', proper, loc ).
noun( 'michael', '-', proper, per ).
noun( 'michaelmas', 'michaelmases', count, _ ).
noun( 'michaelmas', '-', proper, _ ).
noun( 'michelle', '-', proper, per ).
noun( 'michigan', '-', proper, loc ).
noun( 'mick', '-', proper, per ).
noun( 'mickey', '-', proper, per ).
noun( 'middlesex', '-', proper, loc ).
noun( 'middleton', '-', proper, loc ).
noun( 'middlewich', '-', proper, loc ).
noun( 'midlands', '-', proper, loc ).
noun( 'midwest', '-', proper, _ ).
noun( 'mike', '-', proper, per ).
noun( 'milan', '-', proper, loc ).
noun( 'milano', '-', proper, loc ).
noun( 'mildenhall', '-', proper, loc ).
noun( 'mildred', '-', proper, per ).
noun( 'miles', '-', proper, per ).
noun( 'milford', '-', proper, loc ).
noun( 'milford haven', '-', proper, loc ).
noun( 'millicent', '-', proper, per ).
noun( 'millie', '-', proper, per ).
noun( 'millom', '-', proper, loc ).
noun( 'milly', '-', proper, per ).
noun( 'milngavie', '-', proper, loc ).
noun( 'milnrow', '-', proper, loc ).
noun( 'milton keynes', '-', proper, loc ).
noun( 'milwaukee', '-', proper, loc ).
noun( 'minehead', '-', proper, loc ).
noun( 'minneapolis', '-', proper, loc ).
noun( 'minnesota', '-', proper, loc ).
noun( 'minotaur', 'minotaurs', count, _ ).
noun( 'minsk', '-', proper, loc ).
noun( 'minster-in-sheppey', '-', proper, loc ).
noun( 'miranda', '-', proper, per ).
noun( 'miriam', '-', proper, per ).
noun( 'miss', 'misses', count, _ ).
noun( 'mississippi', '-', proper, loc ).
noun( 'missouri', '-', proper, loc ).
noun( 'mlle', '-', proper, per ).
noun( 'mme', '-', proper, per ).
noun( 'mo', '-', proper, per ).
noun( 'mod', 'mods', count, _ ).
noun( 'modena', '-', proper, loc ).
noun( 'mohammedan', 'mohammedans', count, _ ).
noun( 'moira', '-', proper, per ).
noun( 'mold', '-', proper, loc ).
noun( 'molly', '-', proper, per ).
noun( 'moloch', '-', proper, _ ).
noun( 'mon', '-', proper, _ ).
noun( 'monaco', '-', proper, loc ).
noun( 'monaghan', '-', proper, loc ).
noun( 'monchengladbach', '-', proper, loc ).
noun( 'monday', 'mondays', count, _ ).
noun( 'monday', '-', proper, _ ).
noun( 'monegasque', 'monegasques', count, _ ).
noun( 'mongol', 'mongols', both, _ ).
noun( 'mongolia', '-', proper, loc ).
noun( 'mongolian', 'mongolians', both, _ ).
noun( 'monica', '-', proper, per ).
noun( 'monifieth', '-', proper, loc ).
noun( 'monmouth', '-', proper, loc ).
noun( 'monsieur', 'messieurs', count, _ ).
noun( 'monsignor', 'monsignors', count, _ ).
noun( 'montana', '-', proper, loc ).
noun( 'monterrey', '-', proper, loc ).
noun( 'montevideo', '-', proper, loc ).
noun( 'montreal', '-', proper, loc ).
noun( 'montrose', '-', proper, loc ).
noun( 'montserrat', '-', proper, loc ).
noun( 'montserratian', 'montserratians', count, _ ).
noun( 'monza', '-', proper, loc ).
noun( 'moor', 'moors', count, _ ).
noun( 'morecambe', '-', proper, loc ).
noun( 'morley', '-', proper, loc ).
noun( 'mormon', 'mormons', count, _ ).
noun( 'mormonism', '-', mass, _ ).
noun( 'moroccan', 'moroccans', count, _ ).
noun( 'morocco', '-', proper, loc ).
noun( 'morpeth', '-', proper, loc ).
noun( 'morpheus', '-', proper, _ ).
noun( 'morse', '-', mass, _ ).
noun( 'moscow', '-', proper, loc ).
noun( 'moslem', 'moslems', count, _ ).
noun( 'mossley', '-', proper, loc ).
noun( 'motherwell', '-', proper, loc ).
noun( 'mountain ash', '-', proper, loc ).
noun( 'mountie', 'mounties', count, _ ).
noun( 'mozambican', 'mozambicans', count, _ ).
noun( 'mozambique', '-', proper, loc ).
noun( 'mr', '-', proper, per ).
noun( 'mrs', '-', proper, per ).
noun( 'ms', '-', proper, per ).
noun( 'mt', '-', proper, _ ).
noun( 'muhammad', '-', proper, _ ).
noun( 'muhammadan', 'muhammadans', count, _ ).
noun( 'muhammadanism', '-', mass, _ ).
noun( 'mukden', '-', proper, loc ).
noun( 'mulheim', '-', proper, loc ).
noun( 'mulhouse', '-', proper, loc ).
noun( 'multan', '-', proper, loc ).
noun( 'munich', '-', proper, loc ).
noun( 'munster', '-', proper, loc ).
noun( 'muriel', '-', proper, per ).
noun( 'murton', '-', proper, loc ).
noun( 'muscovite', 'muscovites', count, _ ).
noun( 'muscovy', 'muscovies', count, _ ).
noun( 'muslim', 'muslims', count, _ ).
noun( 'musselburgh', '-', proper, loc ).
noun( 'myra', '-', proper, per ).
noun( 'naafi', '-', count, _ ).
noun( 'nasa', '-', proper, _ ).
noun( 'nato', '-', proper, _ ).
noun( 'nb', '-', proper, _ ).
noun( 'nco', '-', count, _ ).
noun( 'nhs', '-', count, _ ).
noun( 'nspcc', '-', count, _ ).
noun( 'nt', '-', count, _ ).
noun( 'nagoya', '-', proper, loc ).
noun( 'nagpur', '-', proper, loc ).
noun( 'nailsea', '-', proper, loc ).
noun( 'nairn', '-', proper, loc ).
noun( 'nairobi', '-', proper, loc ).
noun( 'namibia', '-', proper, loc ).
noun( 'namibian', 'namibians', count, _ ).
noun( 'nanchang', '-', proper, loc ).
noun( 'nancy', '-', proper, per ).
noun( 'nanking', '-', proper, loc ).
noun( 'nanning', '-', proper, loc ).
noun( 'nantwich', '-', proper, loc ).
noun( 'naomi', '-', proper, per ).
noun( 'naples', '-', proper, loc ).
noun( 'napoli', '-', proper, loc ).
noun( 'narborough', '-', proper, loc ).
noun( 'nat', '-', proper, per ).
noun( 'natalie', '-', proper, per ).
noun( 'nathaniel', '-', proper, per ).
noun( 'nauru', '-', proper, loc ).
noun( 'nauruan', 'nauruans', both, _ ).
noun( 'nazi', 'nazis', count, _ ).
noun( 'nazism', '-', mass, _ ).
noun( 'neapolitan', 'neapolitans', count, _ ).
noun( 'neath', '-', proper, loc ).
noun( 'nebraska', '-', proper, loc ).
noun( 'ned', '-', proper, per ).
noun( 'neddy', '-', proper, per ).
noun( 'negress', 'negresses', count, _ ).
noun( 'negro', 'negroes', count, _ ).
noun( 'negroid', 'negroids', count, _ ).
noun( 'negus', 'neguses', count, _ ).
noun( 'neil', '-', proper, per ).
noun( 'nell', '-', proper, per ).
noun( 'nelly', '-', count, _ ).
noun( 'nelly', '-', proper, per ).
noun( 'nelson', '-', proper, loc ).
noun( 'nepal', '-', proper, loc ).
noun( 'nepalese', 'nepalese', count, _ ).
noun( 'nepali', 'nepalis', both, _ ).
noun( 'neptune', '-', proper, _ ).
noun( 'neston', '-', proper, loc ).
noun( 'nestor', '-', proper, _ ).
noun( 'netherlander', 'netherlanders', count, _ ).
noun( 'netherlands', '-', proper, loc ).
noun( 'netley', '-', proper, loc ).
noun( 'neuss', '-', proper, loc ).
noun( 'nevada', '-', proper, loc ).
noun( 'neville', '-', proper, per ).
noun( 'new cumnock', '-', proper, loc ).
noun( 'new mills', '-', proper, loc ).
noun( 'new orleans', '-', proper, loc ).
noun( 'new tredegar', '-', proper, loc ).
noun( 'new windsor', '-', proper, loc ).
noun( 'new york', '-', proper, loc ).
noun( 'newark', '-', proper, loc ).
noun( 'newarthill', '-', proper, loc ).
noun( 'newbiggin', '-', proper, loc ).
noun( 'newbury', '-', proper, loc ).
noun( 'newcastle', '-', proper, loc ).
noun( 'newcastle upon tyne', '-', proper, loc ).
noun( 'newcastle-under-lyme', '-', proper, loc ).
noun( 'newfoundland', '-', proper, loc ).
noun( 'newham', '-', proper, loc ).
noun( 'newhaven', '-', proper, loc ).
noun( 'newmains', '-', proper, loc ).
noun( 'newmarket', '-', mass, _ ).
noun( 'newmarket', '-', proper, loc ).
noun( 'newport', '-', proper, loc ).
noun( 'newport pagnell', '-', proper, loc ).
noun( 'newquay', '-', proper, loc ).
noun( 'newry', '-', proper, loc ).
noun( 'newton abbot', '-', proper, loc ).
noun( 'newton aycliffe', '-', proper, loc ).
noun( 'newton mearns', '-', proper, loc ).
noun( 'newton-le-willows', '-', proper, loc ).
noun( 'newtonian', 'newtonians', count, _ ).
noun( 'newtown', '-', proper, loc ).
noun( 'nicaragua', '-', proper, loc ).
noun( 'nicaraguan', 'nicaraguans', count, _ ).
noun( 'nice', '-', proper, loc ).
noun( 'nicholas', '-', proper, per ).
noun( 'nick', '-', proper, per ).
noun( 'nicola', '-', proper, per ).
noun( 'nicole', '-', proper, per ).
noun( 'nigel', '-', proper, per ).
noun( 'niger', '-', proper, loc ).
noun( 'nigeria', '-', proper, loc ).
noun( 'nigerian', 'nigerians', count, _ ).
noun( 'nigerien', 'nigeriens', count, _ ).
noun( 'nijmegen', '-', proper, loc ).
noun( 'nimrod', '-', proper, _ ).
noun( 'niobe', '-', proper, _ ).
noun( 'nissen', '-', proper, _ ).
noun( 'noah', '-', proper, _ ).
noun( 'nobel', '-', proper, _ ).
noun( 'noel', 'noels', count, _ ).
noun( 'noel', '-', proper, per ).
noun( 'nora', '-', proper, per ).
noun( 'nordic', 'nordics', count, _ ).
noun( 'norfolk', '-', proper, loc ).
noun( 'norman', 'normans', count, _ ).
noun( 'norman', '-', proper, per ).
noun( 'normanton', '-', proper, loc ).
noun( 'normantown', '-', proper, loc ).
noun( 'norse', '-', mass, _ ).
noun( 'north walsham', '-', proper, loc ).
noun( 'northallerton', '-', proper, loc ).
noun( 'northam', '-', proper, loc ).
noun( 'northampton', '-', proper, loc ).
noun( 'northamptonshire', '-', proper, loc ).
noun( 'northman', 'northmen', count, _ ).
noun( 'northumberland', '-', proper, loc ).
noun( 'northwich', '-', proper, loc ).
noun( 'norton', '-', proper, loc ).
noun( 'norway', '-', proper, loc ).
noun( 'norwegian', 'norwegians', both, _ ).
noun( 'norwich', '-', proper, loc ).
noun( 'nottingham', '-', proper, loc ).
noun( 'nottinghamshire', '-', proper, loc ).
noun( 'nov', '-', proper, _ ).
noun( 'nova scotia', '-', proper, loc ).
noun( 'novara', '-', proper, loc ).
noun( 'november', 'novembers', count, _ ).
noun( 'november', '-', proper, _ ).
noun( 'novokuznetsk', '-', proper, loc ).
noun( 'novosibirsk', '-', proper, loc ).
noun( 'nuneaton', '-', proper, loc ).
noun( 'nurenburg', '-', proper, loc ).
noun( 'o-level', 'o-levels', count, _ ).
noun( 'oap', '-', count, _ ).
noun( '-', 'oap\'s', count, _ ).
noun( 'oau', '-', count, _ ).
noun( 'oecd', '-', count, _ ).
noun( 'oed', '-', count, _ ).
noun( 'opec', '-', proper, _ ).
noun( 'ot', '-', count, _ ).
noun( 'oakengates', '-', proper, loc ).
noun( 'oban', '-', proper, loc ).
noun( 'oberhausen', '-', proper, loc ).
noun( 'occident', '-', proper, _ ).
noun( 'occidental', 'occidentals', count, _ ).
noun( 'ockbrook', '-', proper, loc ).
noun( 'ockendon', '-', proper, loc ).
noun( 'oct', '-', proper, _ ).
noun( 'october', 'octobers', count, _ ).
noun( 'october', '-', proper, _ ).
noun( 'odessa', '-', proper, loc ).
noun( 'oedipus', '-', proper, _ ).
noun( 'offaly', '-', proper, loc ).
noun( 'offenbach', '-', proper, loc ).
noun( 'ogmore valley', '-', proper, loc ).
noun( 'ohio', '-', proper, loc ).
noun( 'oklahoma', '-', proper, loc ).
noun( 'oklahoma city', '-', proper, loc ).
noun( 'old windsor', '-', proper, loc ).
noun( 'oldenburg', '-', proper, loc ).
noun( 'oldham', '-', proper, loc ).
noun( 'oldland', '-', proper, loc ).
noun( 'olive', '-', proper, per ).
noun( 'oliver', '-', proper, per ).
noun( 'olivia', '-', proper, per ).
noun( 'ollerton', '-', proper, loc ).
noun( 'ollie', '-', proper, per ).
noun( 'olympiad', 'olympiads', count, _ ).
noun( 'olympian', 'olympians', count, _ ).
noun( 'olympics', '-', proper, _ ).
noun( 'oman', '-', proper, loc ).
noun( 'omani', 'omanis', count, _ ).
noun( 'omsk', '-', proper, loc ).
noun( 'ongar', '-', proper, loc ).
noun( 'ontario', '-', proper, loc ).
noun( 'orangeman', 'orangemen', count, _ ).
noun( 'oregon', '-', proper, loc ).
noun( 'orkney', '-', proper, loc ).
noun( 'orleans', '-', proper, loc ).
noun( 'ormskirk', '-', proper, loc ).
noun( 'osaka', '-', proper, loc ).
noun( 'oscar', 'oscars', count, _ ).
noun( 'oscar', '-', proper, per ).
noun( 'oslo', '-', proper, loc ).
noun( 'osnabruck', '-', proper, loc ).
noun( 'ossett', '-', proper, loc ).
noun( 'oswald', '-', proper, per ).
noun( 'oswestry', '-', proper, loc ).
noun( 'otago', '-', proper, loc ).
noun( 'otley', '-', proper, loc ).
noun( 'ottawa', '-', proper, loc ).
noun( 'oviedo', '-', proper, loc ).
noun( 'oxbridge', '-', proper, _ ).
noun( 'oxford', '-', proper, loc ).
noun( 'oxfordshire', '-', proper, loc ).
noun( 'oxon', '-', proper, _ ).
noun( 'oxonian', 'oxonians', count, _ ).
noun( 'oxted', '-', proper, loc ).
noun( 'ozzie', '-', proper, per ).
noun( 'pa', '-', count, _ ).
noun( 'paye', '-', proper, _ ).
noun( 'pc', '-', proper, per ).
noun( 'pdsa', '-', count, _ ).
noun( 'pe', '-', mass, _ ).
noun( 'plc', '-', proper, _ ).
noun( 'pm', '-', count, _ ).
noun( 'po', '-', count, _ ).
noun( 'pow', '-', count, _ ).
noun( 'pr', '-', mass, _ ).
noun( 'pt', '-', mass, _ ).
noun( 'pta', '-', count, _ ).
noun( 'pto', '-', proper, _ ).
noun( 'pacific', '-', proper, _ ).
noun( 'paddy', 'paddies', count, _ ).
noun( 'paddy', '-', proper, per ).
noun( 'paddy-wagon', 'paddy-wagons', count, _ ).
noun( 'padova', '-', proper, loc ).
noun( 'padua', '-', proper, loc ).
noun( 'paisley', '-', proper, loc ).
noun( 'pakistan', '-', proper, loc ).
noun( 'pakistani', 'pakistanis', count, _ ).
noun( 'palembang', '-', proper, loc ).
noun( 'palermo', '-', proper, loc ).
noun( 'palestine', '-', proper, loc ).
noun( 'palestinian', 'palestinians', count, _ ).
noun( 'pam', '-', proper, per ).
noun( 'pamela', '-', proper, per ).
noun( 'pamplona', '-', proper, loc ).
noun( 'panama', '-', proper, loc ).
noun( 'panamanian', 'panamanians', count, _ ).
noun( 'paotow', '-', proper, loc ).
noun( 'papua', '-', proper, loc ).
noun( 'papuan', 'papuans', count, _ ).
noun( 'paraguay', '-', proper, loc ).
noun( 'paraguayan', 'paraguayans', count, _ ).
noun( 'paris', '-', proper, loc ).
noun( 'parisian', 'parisians', count, _ ).
noun( 'parma', '-', proper, loc ).
noun( 'parmesan', '-', mass, _ ).
noun( 'parsee', 'parsees', count, _ ).
noun( 'pashto', '-', mass, _ ).
noun( 'passover', 'passovers', count, _ ).
noun( 'pat', '-', proper, per ).
noun( 'patience', '-', proper, per ).
noun( 'patrai', '-', proper, loc ).
noun( 'patricia', '-', proper, per ).
noun( 'patrick', '-', proper, per ).
noun( 'patty', '-', proper, per ).
noun( 'paul', '-', proper, per ).
noun( 'paula', '-', proper, per ).
noun( 'pauline', '-', proper, per ).
noun( 'pax romana', '-', count, _ ).
noun( 'peacehaven', '-', proper, loc ).
noun( 'pearl', '-', proper, per ).
noun( 'peebles', '-', proper, loc ).
noun( 'peg', '-', proper, per ).
noun( 'peggy', '-', proper, per ).
noun( 'peking', '-', proper, loc ).
noun( 'pelton', '-', proper, loc ).
noun( 'pembroke', '-', proper, loc ).
noun( 'penarth', '-', proper, loc ).
noun( 'pendlebury', '-', proper, loc ).
noun( 'penelope', '-', proper, per ).
noun( 'penicuik', '-', proper, loc ).
noun( 'penki', '-', proper, loc ).
noun( 'pennsylvania', '-', proper, loc ).
noun( 'penny', '-', proper, per ).
noun( 'penrhyn', '-', proper, loc ).
noun( 'penrith', '-', proper, loc ).
noun( 'penryn', '-', proper, loc ).
noun( 'pentateuch', 'pentateuchs', count, _ ).
noun( 'pentecost', '-', mass, _ ).
noun( 'penzance', '-', proper, loc ).
noun( 'percy', '-', proper, per ).
noun( 'perm', '-', proper, loc ).
noun( 'pershore', '-', proper, loc ).
noun( 'persia', '-', proper, loc ).
noun( 'persian', 'persians', count, _ ).
noun( 'perth', '-', proper, loc ).
noun( 'peru', '-', proper, loc ).
noun( 'perugia', '-', proper, loc ).
noun( 'peruvian', 'peruvians', count, _ ).
noun( 'pescara', '-', proper, loc ).
noun( 'pete', '-', proper, per ).
noun( 'peter', '-', proper, per ).
noun( 'peterborough', '-', proper, loc ).
noun( 'peterhead', '-', proper, loc ).
noun( 'peterlee', '-', proper, loc ).
noun( 'petersfield', '-', proper, loc ).
noun( 'phd', '-', count, _ ).
noun( 'pharaoh', 'pharaohs', count, _ ).
noun( 'pharisee', 'pharisees', count, _ ).
noun( 'phil', '-', proper, per ).
noun( 'philadelphia', '-', proper, loc ).
noun( 'philip', '-', proper, per ).
noun( 'philippa', '-', proper, per ).
noun( 'philippine', 'philippines', count, _ ).
noun( 'philippines', '-', proper, loc ).
noun( 'philistine', 'philistines', count, _ ).
noun( 'philistinism', '-', mass, _ ).
noun( 'phnom-penh', '-', proper, loc ).
noun( 'phoebe', '-', proper, per ).
noun( 'phoenix', '-', proper, loc ).
noun( 'phyllis', '-', proper, per ).
noun( 'piacenza', '-', proper, loc ).
noun( 'pill', '-', proper, loc ).
noun( 'piraeus', '-', proper, loc ).
noun( 'pisa', '-', proper, loc ).
noun( 'pisces', '-', proper, _ ).
noun( 'pittsburgh', '-', proper, loc ).
noun( 'plimsoll', 'plimsolls', count, _ ).
noun( 'pluto', '-', proper, _ ).
noun( 'plymouth', '-', proper, loc ).
noun( 'poland', '-', proper, loc ).
noun( 'polaris', '-', count, _ ).
noun( 'polaroid', '-', mass, _ ).
noun( 'pole', 'poles', count, _ ).
noun( 'polish', '-', mass, _ ).
noun( 'polly', '-', proper, per ).
noun( 'pont-llan-fraith', '-', proper, loc ).
noun( 'pontardawe', '-', proper, loc ).
noun( 'pontardulais', '-', proper, loc ).
noun( 'pontefract', '-', proper, loc ).
noun( 'pontycymmer', '-', proper, loc ).
noun( 'pontypool', '-', proper, loc ).
noun( 'poole', '-', proper, loc ).
noun( 'poona', '-', proper, loc ).
noun( 'pope', 'popes', count, _ ).
noun( 'port glasgow', '-', proper, loc ).
noun( 'port talbot', '-', proper, loc ).
noun( 'porthcawl', '-', proper, loc ).
noun( 'porthmadog', '-', proper, loc ).
noun( 'portishead', '-', proper, loc ).
noun( 'portland', '-', proper, loc ).
noun( 'porto alegre', '-', proper, loc ).
noun( 'portsmouth', '-', proper, loc ).
noun( 'portugal', '-', proper, loc ).
noun( 'portuguese', 'portuguese', both, _ ).
noun( 'potsdam', '-', proper, loc ).
noun( 'potters bar', '-', proper, loc ).
noun( 'powys', '-', proper, loc ).
noun( 'poynton', '-', proper, loc ).
noun( 'poznan', '-', proper, loc ).
noun( 'prague', '-', proper, loc ).
noun( 'prato', '-', proper, loc ).
noun( 'pre-raphaelite', 'pre-raphaelites', count, _ ).
noun( 'pres', '-', count, _ ).
noun( 'presbyterian', 'presbyterians', count, _ ).
noun( 'presbyterianism', '-', mass, _ ).
noun( 'prescot', '-', proper, loc ).
noun( 'prestatyn', '-', proper, loc ).
noun( 'preston', '-', proper, loc ).
noun( 'prestonpans', '-', proper, loc ).
noun( 'prestwich', '-', proper, loc ).
noun( 'prestwick', '-', proper, loc ).
noun( 'pretoria', '-', proper, loc ).
noun( 'princes risborough', '-', proper, loc ).
noun( 'priscilla', '-', proper, per ).
noun( 'prof', '-', proper, per ).
noun( 'protestant', 'protestants', count, _ ).
noun( 'protestantism', '-', mass, _ ).
noun( 'providence', '-', proper, loc ).
noun( 'pru', '-', proper, per ).
noun( 'prudence', '-', proper, per ).
noun( 'prudhoe', '-', proper, loc ).
noun( 'prussian', 'prussians', count, _ ).
noun( 'pte', '-', proper, per ).
noun( 'pty', '-', proper, _ ).
noun( 'pudsey', '-', proper, loc ).
noun( 'pullman', 'pullmans', count, _ ).
noun( 'punch', 'punches', count, _ ).
noun( 'pusan', '-', proper, loc ).
noun( 'pwllheli', '-', proper, loc ).
noun( 'pyle', '-', proper, loc ).
noun( 'pyongyang', '-', proper, loc ).
noun( 'qc', '-', count, _ ).
noun( 'qed', '-', proper, _ ).
noun( 'qatar', '-', proper, loc ).
noun( 'qatari', 'qataris', count, _ ).
noun( 'quai d\'orsay', '-', count, _ ).
noun( 'quaker', 'quakers', count, _ ).
noun( 'quebec', '-', proper, loc ).
noun( 'queenborough-in-sheppey', '-', proper, loc ).
noun( 'queensbury', '-', proper, loc ).
noun( 'queensferry', '-', proper, loc ).
noun( 'queensland', '-', proper, loc ).
noun( 'quentin', '-', proper, per ).
noun( 'quinquagesima', 'quinquagesimas', count, _ ).
noun( 'quinquagesima', '-', proper, _ ).
noun( 'quito', '-', proper, loc ).
noun( 'quonset', 'quonsets', count, _ ).
noun( 'r\"ontgen', 'r\"ontgens', count, _ ).
noun( 'ra', '-', count, _ ).
noun( 'rada', '-', proper, _ ).
noun( 'raf', '-', count, _ ).
noun( 'rc', '-', proper, _ ).
noun( 'rip', '-', proper, _ ).
noun( 'rm', '-', count, _ ).
noun( 'rn', '-', count, _ ).
noun( 'rspca', '-', count, _ ).
noun( 'rsvp', '-', proper, _ ).
noun( 'rachel', '-', proper, per ).
noun( 'radcliffe', '-', proper, loc ).
noun( 'radcliffe on trent', '-', proper, loc ).
noun( 'radlett', '-', proper, loc ).
noun( 'radstock', '-', proper, loc ).
noun( 'rainford', '-', proper, loc ).
noun( 'ralph', '-', proper, per ).
noun( 'ramadan', 'ramadans', count, _ ).
noun( 'ramadan', '-', proper, _ ).
noun( 'ramsbottom', '-', proper, loc ).
noun( 'ramsgate', '-', proper, loc ).
noun( 'randolph', '-', proper, per ).
noun( 'rangoon', '-', proper, loc ).
noun( 'ravenna', '-', proper, loc ).
noun( 'rawtenstall', '-', proper, loc ).
noun( 'ray', '-', proper, per ).
noun( 'raymond', '-', proper, per ).
noun( 'rd', '-', proper, _ ).
noun( 'reading', '-', proper, loc ).
noun( 'realtor', 'realtors', count, _ ).
noun( 'rebecca', '-', proper, per ).
noun( 'recife', '-', proper, loc ).
noun( 'recklinghausen', '-', proper, loc ).
noun( 'redbridge', '-', proper, loc ).
noun( 'redditch', '-', proper, loc ).
noun( 'redruth', '-', proper, loc ).
noun( 'reg', '-', proper, per ).
noun( 'regensburg', '-', proper, loc ).
noun( 'regina', '-', proper, _ ).
noun( 'reginald', '-', proper, per ).
noun( 'reich', 'reichs', count, _ ).
noun( 'reigate', '-', proper, loc ).
noun( 'remscheid', '-', proper, loc ).
noun( 'renfrew', '-', proper, loc ).
noun( 'rennes', '-', proper, loc ).
noun( 'retford', '-', proper, loc ).
noun( 'rev', '-', proper, per ).
noun( 'revd', '-', proper, per ).
noun( 'rex', '-', proper, per ).
noun( 'reykjavik', '-', proper, loc ).
noun( 'rheims', '-', proper, loc ).
noun( 'rheydt', '-', proper, loc ).
noun( 'rhine', '-', proper, _ ).
noun( 'rhinestone', 'rhinestones', count, _ ).
noun( 'rhode', '-', proper, loc ).
noun( 'rhondda', '-', proper, loc ).
noun( 'rhosllanerchrugog', '-', proper, loc ).
noun( 'rhyl', '-', proper, loc ).
noun( 'rhymney', '-', proper, loc ).
noun( 'richard', '-', proper, per ).
noun( 'richmond', '-', proper, loc ).
noun( 'richmond upon thames', '-', proper, loc ).
noun( 'rick', '-', proper, per ).
noun( 'rickmansworth', '-', proper, loc ).
noun( 'ricky', '-', proper, per ).
noun( 'riesling', 'rieslings', count, _ ).
noun( 'riga', '-', proper, loc ).
noun( 'rijeka', '-', proper, loc ).
noun( 'rimini', '-', proper, loc ).
noun( 'ringwood', '-', proper, loc ).
noun( 'rio de janeiro', '-', proper, loc ).
noun( 'ripley', '-', proper, loc ).
noun( 'ripon', '-', proper, loc ).
noun( 'risborough', '-', proper, loc ).
noun( 'risca', '-', proper, loc ).
noun( 'rishton', '-', proper, loc ).
noun( 'rita', '-', proper, per ).
noun( 'riviera', '-', proper, _ ).
noun( 'rob', '-', proper, per ).
noun( 'robert', '-', proper, per ).
noun( 'robin', '-', proper, per ).
noun( 'rochdale', '-', proper, loc ).
noun( 'rochester', '-', proper, loc ).
noun( 'rod', '-', proper, per ).
noun( 'rodney', '-', proper, per ).
noun( 'roentgen', 'roentgens', count, _ ).
noun( 'roger', '-', proper, per ).
noun( 'romaic', '-', mass, _ ).
noun( 'roman', 'romans', count, _ ).
noun( 'romanesque', '-', mass, _ ).
noun( 'romania', '-', proper, loc ).
noun( 'romanian', 'romanians', both, _ ).
noun( 'romany', 'romanies', both, _ ).
noun( 'rome', '-', proper, loc ).
noun( 'romsey', '-', proper, loc ).
noun( 'ron', '-', proper, per ).
noun( 'ronald', '-', proper, per ).
noun( 'roneo', 'roneos', count, _ ).
noun( 'ronnie', '-', proper, per ).
noun( 'roquefort', '-', mass, _ ).
noun( 'rosalie', '-', proper, per ).
noun( 'rosalind', '-', proper, per ).
noun( 'rosamund', '-', proper, per ).
noun( 'rosario', '-', proper, loc ).
noun( 'roscommon', '-', proper, loc ).
noun( 'rose', '-', proper, per ).
noun( 'rosemary', '-', proper, per ).
noun( 'rosie', '-', proper, per ).
noun( 'roslyn', '-', proper, per ).
noun( 'ross-on-wye', '-', proper, loc ).
noun( 'rossington', '-', proper, loc ).
noun( 'rosslare', '-', proper, loc ).
noun( 'rostock', '-', proper, loc ).
noun( 'rostov-na-donu', '-', proper, loc ).
noun( 'rotarian', 'rotarians', count, _ ).
noun( 'rotherham', '-', proper, loc ).
noun( 'rothesay', '-', proper, loc ).
noun( 'rothwell', '-', proper, loc ).
noun( 'rotterdam', '-', proper, loc ).
noun( 'roubaix', '-', proper, loc ).
noun( 'rouen', '-', proper, loc ).
noun( 'roundhead', 'roundheads', count, _ ).
noun( 'rowlands gill', '-', proper, loc ).
noun( 'roy', '-', proper, per ).
noun( 'royston', '-', proper, loc ).
noun( 'rt hon', '-', proper, _ ).
noun( 'rubicon', 'rubicons', count, _ ).
noun( 'ruddington', '-', proper, loc ).
noun( 'rudolf', '-', proper, per ).
noun( 'rudy', '-', proper, per ).
noun( 'rugby', '-', proper, loc ).
noun( 'rugeley', '-', proper, loc ).
noun( 'runcorn', '-', proper, loc ).
noun( 'rupert', '-', proper, per ).
noun( 'russia', '-', proper, loc ).
noun( 'russian', 'russians', both, _ ).
noun( 'ruth', '-', proper, per ).
noun( 'rutherglen', '-', proper, loc ).
noun( 'rutland', '-', proper, loc ).
noun( 'rwanda', '-', proper, loc ).
noun( 'rwandan', 'rwandans', count, _ ).
noun( 'ryde', '-', proper, loc ).
noun( 'ryehill', '-', proper, loc ).
noun( 'salt', '-', proper, _ ).
noun( 'sos', '-', count, _ ).
noun( '-', 'soss', count, _ ).
noun( 'srn', '-', count, _ ).
noun( 'ss', '-', count, _ ).
noun( 'std', '-', proper, _ ).
noun( 'saarbrucken', '-', proper, loc ).
noun( 'sabadeli', '-', proper, loc ).
noun( 'sabah', '-', proper, loc ).
noun( 'sabahan', 'sabahans', count, _ ).
noun( 'sabbath', 'sabbaths', count, _ ).
noun( 'sacramento', '-', proper, loc ).
noun( 'saffron walden', '-', proper, loc ).
noun( 'sagittarius', '-', proper, _ ).
noun( 'saigon', '-', proper, loc ).
noun( 'salamanca', '-', proper, loc ).
noun( 'sale', '-', proper, loc ).
noun( 'salerno', '-', proper, loc ).
noun( 'salford', '-', proper, loc ).
noun( 'salisbury', '-', proper, loc ).
noun( 'sally', '-', proper, per ).
noun( 'saltash', '-', proper, loc ).
noun( 'saltburn', '-', proper, loc ).
noun( 'saltcoats', '-', proper, loc ).
noun( 'salvador', '-', proper, loc ).
noun( 'salvadorean', 'salvadoreans', count, _ ).
noun( 'salzburg', '-', proper, loc ).
noun( 'sam', '-', proper, per ).
noun( 'samantha', '-', proper, per ).
noun( 'samaritan', 'samaritans', count, _ ).
noun( 'sammy', '-', proper, per ).
noun( 'samoa', '-', proper, loc ).
noun( 'samoan', 'samoans', count, _ ).
noun( 'samuel', '-', proper, per ).
noun( 'san antonio', '-', proper, loc ).
noun( 'san bernardino', '-', proper, loc ).
noun( 'san diego', '-', proper, loc ).
noun( 'san francisco', '-', proper, loc ).
noun( 'san jose', '-', proper, loc ).
noun( 'san juan', '-', proper, loc ).
noun( 'san marinese', 'san marinese', count, _ ).
noun( 'san marino', '-', proper, loc ).
noun( 'san sebastian', '-', proper, loc ).
noun( 'sandbach', '-', proper, loc ).
noun( 'sandown', '-', proper, loc ).
noun( 'sandra', '-', proper, per ).
noun( 'sandy', '-', proper, per ).
noun( 'sango', '-', mass, _ ).
noun( 'sanskrit', '-', mass, _ ).
noun( 'santa claus', 'santa clauses', count, _ ).
noun( 'santa claus', '-', proper, _ ).
noun( 'santander', '-', proper, loc ).
noun( 'santiago', '-', proper, loc ).
noun( 'santo domingo', '-', proper, loc ).
noun( 'santos', '-', proper, loc ).
noun( 'sao paolo', '-', proper, loc ).
noun( 'sapporo', '-', proper, loc ).
noun( 'sara', '-', proper, per ).
noun( 'saracen', 'saracens', count, _ ).
noun( 'sarah', '-', proper, per ).
noun( 'sarajevo', '-', proper, loc ).
noun( 'saratov', '-', proper, loc ).
noun( 'sarawak', '-', proper, loc ).
noun( 'sarawakian', 'sarawakians', count, _ ).
noun( 'sarisbury', '-', proper, loc ).
noun( 'saskatchewan', '-', proper, loc ).
noun( 'sassari', '-', proper, loc ).
noun( 'sassenach', 'sassenachs', count, _ ).
noun( 'sat', '-', proper, _ ).
noun( 'satan', 'satans', count, _ ).
noun( 'saturday', 'saturdays', count, _ ).
noun( 'saturday', '-', proper, _ ).
noun( 'saturn', '-', proper, _ ).
noun( 'sauchie', '-', proper, loc ).
noun( 'saudi arabia', '-', proper, loc ).
noun( 'saudi arabian', 'saudi arabians', count, _ ).
noun( 'sawbridgeworth', '-', proper, loc ).
noun( 'saxon', 'saxons', count, _ ).
noun( 'scandinavian', 'scandinavians', count, _ ).
noun( 'scarborough', '-', proper, loc ).
noun( 'schwerin', '-', proper, loc ).
noun( 'sci fi', '-', mass, _ ).
noun( 'scorpio', '-', proper, _ ).
noun( 'scot', 'scots', count, _ ).
noun( 'scotch', 'scotches', both, _ ).
noun( 'scotchman', 'scotchmen', count, _ ).
noun( 'scotchwoman', 'scotchwomen', count, _ ).
noun( 'scotland', '-', proper, loc ).
noun( 'scotland yard', '-', proper, _ ).
noun( 'scotsman', 'scotsmen', count, _ ).
noun( 'scotswoman', 'scotswomen', count, _ ).
noun( 'scrooge', 'scrooges', count, _ ).
noun( 'scunthorpe', '-', proper, loc ).
noun( 'scylla', 'scyllas', count, _ ).
noun( 'seaford', '-', proper, loc ).
noun( 'seaham', '-', proper, loc ).
noun( 'sealyham', 'sealyhams', count, _ ).
noun( 'seamus', '-', proper, per ).
noun( 'sean', '-', proper, per ).
noun( 'seaton burn', '-', proper, loc ).
noun( 'seaton delaval', '-', proper, loc ).
noun( 'seattle', '-', proper, loc ).
noun( 'secretary-general', 'secretary-generals', count, _ ).
noun( 'securicor', '-', proper, _ ).
noun( 'selby', '-', proper, loc ).
noun( 'selkirk', '-', proper, loc ).
noun( 'selsey', '-', proper, loc ).
noun( 'selston', '-', proper, loc ).
noun( 'semarang', '-', proper, loc ).
noun( 'semite', 'semites', count, _ ).
noun( 'senegal', '-', proper, loc ).
noun( 'senegalese', 'senegalese', count, _ ).
noun( 'seoul', '-', proper, loc ).
noun( 'sept', '-', proper, _ ).
noun( 'september', 'septembers', count, _ ).
noun( 'september', '-', proper, _ ).
noun( 'septuagint', 'septuagints', count, _ ).
noun( 'serbo-croat', '-', mass, _ ).
noun( 'serjeant-at-arms', 'serjeants-at-arms', count, _ ).
noun( 'sesotho', '-', mass, _ ).
noun( 'setswana', '-', mass, _ ).
noun( 'seven sisters', '-', proper, loc ).
noun( 'sevenoaks', '-', proper, loc ).
noun( 'sevilla', '-', proper, loc ).
noun( 'seville', '-', proper, loc ).
noun( 'seychelles', '-', proper, loc ).
noun( 'seychellois', 'seychellois', count, _ ).
noun( 'sgt', '-', proper, per ).
noun( 'shanghai', '-', proper, loc ).
noun( 'shanklin', '-', proper, loc ).
noun( 'sharon', '-', proper, per ).
noun( 'shavian', 'shavians', count, _ ).
noun( 'sheerness', '-', proper, loc ).
noun( 'sheffield', '-', proper, loc ).
noun( 'sheila', '-', proper, per ).
noun( 'shelford', '-', proper, loc ).
noun( 'shepshed', '-', proper, loc ).
noun( 'shepton mallet', '-', proper, loc ).
noun( 'sheraton', '-', mass, _ ).
noun( 'sherbourne', '-', proper, loc ).
noun( 'sheringham', '-', proper, loc ).
noun( 'shetland', '-', proper, loc ).
noun( 'shevington', '-', proper, loc ).
noun( 'shihkiachwang', '-', proper, loc ).
noun( 'shildon', '-', proper, loc ).
noun( 'shirebrook', '-', proper, loc ).
noun( 'shiremoor', '-', proper, loc ).
noun( 'shirley', '-', proper, per ).
noun( 'shotton', '-', proper, loc ).
noun( 'shotts', '-', proper, loc ).
noun( 'shrewsbury', '-', proper, loc ).
noun( 'shropshire', '-', proper, loc ).
noun( 'shrove tuesday', 'shrove tuesdays', count, _ ).
noun( 'shrove tuesday', '-', proper, _ ).
noun( 'siam', '-', proper, loc ).
noun( 'siamese', 'siamese', count, _ ).
noun( 'sian', '-', proper, loc ).
noun( 'sicilian', 'sicilians', count, _ ).
noun( 'sid', '-', proper, per ).
noun( 'sidmouth', '-', proper, loc ).
noun( 'sidney', '-', proper, per ).
noun( 'sierra leone', '-', proper, loc ).
noun( 'sierra leonian', 'sierra leonians', count, _ ).
noun( 'sikh', 'sikhs', count, _ ).
noun( 'sileby', '-', proper, loc ).
noun( 'silsden', '-', proper, loc ).
noun( 'silvia', '-', proper, per ).
noun( 'simon', '-', proper, per ).
noun( 'singapore', '-', proper, loc ).
noun( 'singaporean', 'singaporeans', count, _ ).
noun( 'sinhala', '-', mass, _ ).
noun( 'sinhalese', 'sinhalese', count, _ ).
noun( 'sining', '-', proper, loc ).
noun( 'sinn fein', '-', proper, _ ).
noun( 'sinologist', 'sinologists', count, _ ).
noun( 'sinology', '-', mass, _ ).
noun( 'sioux', 'sioux', count, _ ).
noun( 'sir roger de coverley', '-', count, _ ).
noun( 'siracusa', '-', proper, loc ).
noun( 'siswati', '-', mass, _ ).
noun( 'sittingbourne', '-', proper, loc ).
noun( 'skegness', '-', proper, loc ).
noun( 'skelmanthorpe', '-', proper, loc ).
noun( 'skelmersdale', '-', proper, loc ).
noun( 'skewen', '-', proper, loc ).
noun( 'skipton', '-', proper, loc ).
noun( 'skopje', '-', proper, loc ).
noun( 'slav', 'slavs', count, _ ).
noun( 'sleaford', '-', proper, loc ).
noun( 'sligo', '-', proper, loc ).
noun( 'slough', '-', proper, loc ).
noun( 'slovak', '-', mass, _ ).
noun( 'slovenian', 'slovenians', both, _ ).
noun( 'snr', '-', proper, _ ).
noun( 'soc', '-', count, _ ).
noun( 'sofia', '-', proper, loc ).
noun( 'soho', '-', proper, _ ).
noun( 'sol', '-', proper, _ ).
noun( 'solicitor-general', 'solicitor-generals', count, _ ).
noun( 'solihull', '-', proper, loc ).
noun( 'solingen', '-', proper, loc ).
noun( 'solon', '-', proper, _ ).
noun( 'somali', 'somalis', both, _ ).
noun( 'somalia', '-', proper, loc ).
noun( 'somalian', 'somalians', count, _ ).
noun( 'somercotes', '-', proper, loc ).
noun( 'somerset', '-', proper, loc ).
noun( 'sonia', '-', proper, per ).
noun( 'soochow', '-', proper, loc ).
noun( 'sophia', '-', proper, per ).
noun( 'sophie', '-', proper, per ).
noun( 'sotho', 'sothos', both, _ ).
noun( 'south kirkby', '-', proper, loc ).
noun( 'south normantown', '-', proper, loc ).
noun( 'south ockendon', '-', proper, loc ).
noun( 'south shields', '-', proper, loc ).
noun( 'southampton', '-', proper, loc ).
noun( 'southend-on-sea', '-', proper, loc ).
noun( 'southport', '-', proper, loc ).
noun( 'southwark', '-', proper, loc ).
noun( 'soviet', 'soviets', count, _ ).
noun( 'spain', '-', proper, loc ).
noun( 'spalding', '-', proper, loc ).
noun( 'spaniard', 'spaniards', count, _ ).
noun( 'spanish', '-', mass, _ ).
noun( 'spartan', 'spartans', count, _ ).
noun( 'spenborough', '-', proper, loc ).
noun( 'spennymoor', '-', proper, loc ).
noun( 'split', '-', proper, loc ).
noun( 'spode', '-', mass, _ ).
noun( 'springfield', '-', proper, loc ).
noun( 'sq', '-', proper, _ ).
noun( 'sr', '-', proper, per ).
noun( 'sri lanka', '-', proper, loc ).
noun( 'sri lankan', 'sri lankans', count, _ ).
noun( 'st', '-', proper, _ ).
noun( 'st albans', '-', proper, loc ).
noun( 'st andrews', '-', proper, loc ).
noun( 'st andrews major', '-', proper, loc ).
noun( 'st austell', '-', proper, loc ).
noun( 'st blazey', '-', proper, loc ).
noun( 'st helens', '-', proper, loc ).
noun( 'st ives', '-', proper, loc ).
noun( 'st louis', '-', proper, loc ).
noun( 'st neots', '-', proper, loc ).
noun( 'st paul', '-', proper, loc ).
noun( 'st petersburg', '-', proper, loc ).
noun( 'st stephen', '-', proper, loc ).
noun( 'sta', '-', proper, _ ).
noun( 'stafford', '-', proper, loc ).
noun( 'staffordshire', '-', proper, loc ).
noun( 'stakeford', '-', proper, loc ).
noun( 'stalybridge', '-', proper, loc ).
noun( 'stamford', '-', proper, loc ).
noun( 'stan', '-', proper, per ).
noun( 'stanford', '-', proper, loc ).
noun( 'stanley', '-', proper, per ).
noun( 'statehouse', 'statehouses', count, _ ).
noun( 'stella', '-', proper, per ).
noun( 'sten', '-', proper, _ ).
noun( 'stephanie', '-', proper, per ).
noun( 'stephen', '-', proper, per ).
noun( 'steve', '-', proper, per ).
noun( 'steven', '-', proper, per ).
noun( 'stevenage', '-', proper, loc ).
noun( 'stevenston', '-', proper, loc ).
noun( 'stewart', '-', proper, per ).
noun( 'steyning', '-', proper, loc ).
noun( 'stilton', 'stiltons', both, _ ).
noun( 'stirling', '-', proper, loc ).
noun( 'stockholm', '-', proper, loc ).
noun( 'stockport', '-', proper, loc ).
noun( 'stocksbridge', '-', proper, loc ).
noun( 'stoke-on-trent', '-', proper, loc ).
noun( 'stone', '-', proper, loc ).
noun( 'stonehouse', '-', proper, loc ).
noun( 'stornoway', '-', proper, loc ).
noun( 'stotfold', '-', proper, loc ).
noun( 'stourbridge', '-', proper, loc ).
noun( 'stourport-on-severn', '-', proper, loc ).
noun( 'stowmarket', '-', proper, loc ).
noun( 'stranraer', '-', proper, loc ).
noun( 'strasbourg', '-', proper, loc ).
noun( 'stratford-on-avon', '-', proper, loc ).
noun( 'strathaven', '-', proper, loc ).
noun( 'strathclyde', '-', proper, loc ).
noun( 'street', '-', proper, loc ).
noun( 'stretford', '-', proper, loc ).
noun( 'stroud', '-', proper, loc ).
noun( 'stuart', '-', proper, per ).
noun( 'studley', '-', proper, loc ).
noun( 'stuttgart', '-', proper, loc ).
noun( 'styx', '-', proper, _ ).
noun( 'suchow', '-', proper, loc ).
noun( 'sudan', '-', proper, loc ).
noun( 'sudanese', 'sudanese', count, _ ).
noun( 'sudbury', '-', proper, loc ).
noun( 'sue', '-', proper, per ).
noun( 'suffolk', '-', proper, loc ).
noun( 'sumatra', '-', proper, loc ).
noun( 'sumatran', 'sumatrans', count, _ ).
noun( 'sun', '-', proper, _ ).
noun( 'sunday', 'sundays', count, _ ).
noun( 'sunday', '-', proper, _ ).
noun( 'sunderland', '-', proper, loc ).
noun( 'supt', '-', proper, per ).
noun( 'surabaja', '-', proper, loc ).
noun( 'surrey', '-', proper, loc ).
noun( 'susan', '-', proper, per ).
noun( 'susanna', '-', proper, per ).
noun( 'susie', '-', proper, per ).
noun( 'sussex', '-', proper, loc ).
noun( 'sutton', '-', proper, loc ).
noun( 'sutton coldfield', '-', proper, loc ).
noun( 'sutton in ashfield', '-', proper, loc ).
noun( 'sutton-at-hone', '-', proper, loc ).
noun( 'suzanne', '-', proper, per ).
noun( 'sverdlovsk', '-', proper, loc ).
noun( 'swadlincote', '-', proper, loc ).
noun( 'swahili', '-', mass, _ ).
noun( 'swanage', '-', proper, loc ).
noun( 'swanley', '-', proper, loc ).
noun( 'swansea', '-', proper, loc ).
noun( 'swazi', 'swazis', both, _ ).
noun( 'swaziland', '-', proper, loc ).
noun( 'swede', 'swedes', count, _ ).
noun( 'sweden', '-', proper, loc ).
noun( 'swedish', '-', mass, _ ).
noun( 'swindon', '-', proper, loc ).
noun( 'swinton', '-', proper, loc ).
noun( 'swiss', 'swiss', count, _ ).
noun( 'switzerland', '-', proper, loc ).
noun( 'sydney', '-', proper, loc ).
noun( 'sylvia', '-', proper, per ).
noun( 'syracuse', '-', proper, loc ).
noun( 'syria', '-', proper, loc ).
noun( 'syrian', 'syrians', count, _ ).
noun( 'syston', '-', proper, loc ).
noun( 't-junction', 't-junctions', count, _ ).
noun( 't-shirt', 't-shirts', count, _ ).
noun( 't-square', 't-squares', count, _ ).
noun( 'tb', '-', mass, _ ).
noun( 'tnt', '-', mass, _ ).
noun( 'tuc', '-', count, _ ).
noun( 'tv', '-', count, _ ).
noun( '-', 'tvs', count, _ ).
noun( 'tadley', '-', proper, loc ).
noun( 'taegu', '-', proper, loc ).
noun( 'taffy', 'taffies', count, _ ).
noun( 'tahiti', '-', proper, loc ).
noun( 'tahitian', 'tahitians', count, _ ).
noun( 'taipei', '-', proper, loc ).
noun( 'taiwan', '-', proper, loc ).
noun( 'taiwanese', 'taiwanese', count, _ ).
noun( 'taiyuan', '-', proper, loc ).
noun( 'talmud', 'talmuds', count, _ ).
noun( 'tamil', 'tamils', both, _ ).
noun( 'tammany', '-', count, _ ).
noun( 'tampere', '-', proper, loc ).
noun( 'tamworth', '-', proper, loc ).
noun( 'tangshan', '-', proper, loc ).
noun( 'tanzania', '-', proper, loc ).
noun( 'tanzanian', 'tanzanians', count, _ ).
noun( 'taranaki', '-', proper, loc ).
noun( 'taranto', '-', proper, loc ).
noun( 'tarrasa', '-', proper, loc ).
noun( 'tashkent', '-', proper, loc ).
noun( 'tasmania', '-', proper, loc ).
noun( 'taunton', '-', proper, loc ).
noun( 'taurus', '-', proper, _ ).
noun( 'taverham', '-', proper, loc ).
noun( 'tavistock', '-', proper, loc ).
noun( 'tayside', '-', proper, loc ).
noun( 'tbilisi', '-', proper, loc ).
noun( 'te deum', 'te deums', count, _ ).
noun( 'tech', '-', count, _ ).
noun( 'technicolor', '-', mass, _ ).
noun( 'ted', 'teds', count, _ ).
noun( 'ted', '-', proper, per ).
noun( 'teddy', '-', proper, per ).
noun( 'teddy boy', 'teddy boys', count, _ ).
noun( 'teesside', '-', proper, loc ).
noun( 'teheran', '-', proper, loc ).
noun( 'teignmouth', '-', proper, loc ).
noun( 'tel-aviv', '-', proper, loc ).
noun( 'telford', '-', proper, loc ).
noun( 'telstar', '-', proper, _ ).
noun( 'tenby', '-', proper, loc ).
noun( 'tennessee', '-', proper, loc ).
noun( 'tenterden', '-', proper, loc ).
noun( 'terence', '-', proper, per ).
noun( 'teresa', '-', proper, per ).
noun( 'termi', '-', proper, loc ).
noun( 'terr', '-', proper, _ ).
noun( 'terry', '-', proper, per ).
noun( 'tess', '-', proper, per ).
noun( 'tessa', '-', proper, per ).
noun( 'teuton', 'teutons', count, _ ).
noun( 'tewkesbury', '-', proper, loc ).
noun( 'texan', 'texans', count, _ ).
noun( 'texas', '-', proper, loc ).
noun( 'thai', 'thais', both, _ ).
noun( 'thailand', '-', proper, loc ).
noun( 'thame', '-', proper, loc ).
noun( 'thanksgiving', 'thanksgivings', count, _ ).
noun( 'thatcham', '-', proper, loc ).
noun( 'theo', '-', proper, per ).
noun( 'theodore', '-', proper, per ).
noun( 'theresa', '-', proper, per ).
noun( 'thespian', 'thespians', count, _ ).
noun( 'thessaloniki', '-', proper, loc ).
noun( 'thetford', '-', proper, loc ).
noun( 'thionville', '-', proper, loc ).
noun( 'thirsk', '-', proper, loc ).
noun( 'thomas', '-', proper, per ).
noun( 'thornbury', '-', proper, loc ).
noun( 'thorne', '-', proper, loc ).
noun( 'thurcroft', '-', proper, loc ).
noun( 'thurmaston', '-', proper, loc ).
noun( 'thurrock', '-', proper, loc ).
noun( 'thurs', '-', proper, _ ).
noun( 'thursday', 'thursdays', count, _ ).
noun( 'thursday', '-', proper, _ ).
noun( 'thurso', '-', proper, loc ).
noun( 'tibet', '-', proper, loc ).
noun( 'tibetan', 'tibetans', both, _ ).
noun( 'tidworth', '-', proper, loc ).
noun( 'tientsin', '-', proper, loc ).
noun( 'tilburg', '-', proper, loc ).
noun( 'tilbury', '-', proper, loc ).
noun( 'tim', '-', proper, per ).
noun( 'timmy', '-', proper, per ).
noun( 'timothy', '-', proper, per ).
noun( 'tina', '-', proper, per ).
noun( 'tipperary', '-', proper, loc ).
noun( 'tiverton', '-', proper, loc ).
noun( 'tobago', '-', proper, loc ).
noun( 'tobagonian', 'tobagonians', count, _ ).
noun( 'toby', '-', proper, per ).
noun( 'todmorden', '-', proper, loc ).
noun( 'togo', '-', proper, loc ).
noun( 'togolese', 'togolese', count, _ ).
noun( 'tokay', '-', mass, _ ).
noun( 'tokyo', '-', proper, loc ).
noun( 'tom', '-', proper, per ).
noun( 'tommy', '-', proper, per ).
noun( 'tonbridge', '-', proper, loc ).
noun( 'tonga', '-', proper, loc ).
noun( 'tongan', 'tongans', both, _ ).
noun( 'tony', '-', proper, per ).
noun( 'tonyrefail', '-', proper, loc ).
noun( 'torbay', '-', proper, loc ).
noun( 'torino', '-', proper, loc ).
noun( 'toronto', '-', proper, loc ).
noun( 'tory', 'tories', count, _ ).
noun( 'totnes', '-', proper, loc ).
noun( 'toulon', '-', proper, loc ).
noun( 'toulouse', '-', proper, loc ).
noun( 'tours', '-', proper, loc ).
noun( 'tower hamlets', '-', proper, loc ).
noun( 'tracy', '-', proper, per ).
noun( 'tralee', '-', proper, loc ).
noun( 'tranent', '-', proper, loc ).
noun( 'trappist', 'trappists', count, _ ).
noun( 'tredegar', '-', proper, loc ).
noun( 'trevor', '-', proper, per ).
noun( 'trieste', '-', proper, loc ).
noun( 'tring', '-', proper, loc ).
noun( 'trinidad', '-', proper, loc ).
noun( 'trinidadian', 'trinidadians', count, _ ).
noun( 'trojan', 'trojans', count, _ ).
noun( 'troon', '-', proper, loc ).
noun( 'trowbridge', '-', proper, loc ).
noun( 'truro', '-', proper, loc ).
noun( 'tsar', 'tsars', count, _ ).
noun( 'tsarina', 'tsarinas', count, _ ).
noun( 'tsinan', '-', proper, loc ).
noun( 'tsingtao', '-', proper, loc ).
noun( 'tsitsihar', '-', proper, loc ).
noun( 'tswana', 'tswanas', both, _ ).
noun( 'tues', '-', proper, _ ).
noun( 'tuesday', 'tuesdays', count, _ ).
noun( 'tuesday', '-', proper, _ ).
noun( 'tullibody', '-', proper, loc ).
noun( 'tunbridge wells', '-', proper, loc ).
noun( 'tunis', '-', proper, loc ).
noun( 'tunisia', '-', proper, loc ).
noun( 'tunisian', 'tunisians', count, _ ).
noun( 'turin', '-', proper, loc ).
noun( 'turk', 'turks', count, _ ).
noun( 'turkey', '-', proper, loc ).
noun( 'turkish', '-', mass, _ ).
noun( 'turku', '-', proper, loc ).
noun( 'twelfth-night', 'twelfth-nights', count, _ ).
noun( 'twelfth-night', '-', proper, _ ).
noun( 'tyne', '-', proper, loc ).
noun( 'tynemouth', '-', proper, loc ).
noun( 'tyrone', '-', proper, loc ).
noun( 'tzupo', '-', proper, loc ).
noun( 'u-boat', 'u-boats', count, _ ).
noun( 'u-turn', 'u-turns', count, _ ).
noun( 'udi', '-', proper, _ ).
noun( 'ufo', '-', count, _ ).
noun( '-', 'ufo\'s', count, _ ).
noun( 'uk', '-', count, _ ).
noun( 'un', '-', count, _ ).
noun( 'unesco', '-', proper, _ ).
noun( 'unicef', '-', proper, _ ).
noun( 'us', '-', count, _ ).
noun( 'usa', '-', count, _ ).
noun( 'usaf', '-', count, _ ).
noun( 'usn', '-', count, _ ).
noun( 'uss', '-', count, _ ).
noun( 'ussr', '-', count, _ ).
noun( 'uddingston', '-', proper, loc ).
noun( 'udine', '-', proper, loc ).
noun( 'ufa', '-', proper, loc ).
noun( 'uganda', '-', proper, loc ).
noun( 'ugandan', 'ugandans', count, _ ).
noun( 'ulster', '-', proper, loc ).
noun( 'ulverston', '-', proper, loc ).
noun( 'unitarian', 'unitarians', count, _ ).
noun( 'unitarianism', '-', mass, _ ).
noun( 'univ', '-', proper, _ ).
noun( 'up holland', '-', proper, loc ).
noun( 'upton', '-', proper, loc ).
noun( 'uranus', '-', proper, _ ).
noun( 'urdu', '-', mass, _ ).
noun( 'ursula', '-', proper, per ).
noun( 'uruguay', '-', proper, loc ).
noun( 'uruguayan', 'uruguayans', count, _ ).
noun( 'urumchi', '-', proper, loc ).
noun( 'ushaw moor', '-', proper, loc ).
noun( 'utah', '-', proper, loc ).
noun( 'utopia', 'utopias', count, _ ).
noun( 'utrecht', '-', proper, loc ).
noun( 'uttoxeter', '-', proper, loc ).
noun( 'v and a', '-', count, _ ).
noun( 'vat', '-', mass, _ ).
noun( 'vc', '-', count, _ ).
noun( 'vd', '-', mass, _ ).
noun( 'vhf', '-', proper, _ ).
noun( 'vip', '-', count, _ ).
noun( '-', 'vips', count, _ ).
noun( 'vp', '-', count, _ ).
noun( 'vso', '-', proper, _ ).
noun( 'valencia', '-', proper, loc ).
noun( 'valenciennes', '-', proper, loc ).
noun( 'valladolid', '-', proper, loc ).
noun( 'vancouver', '-', proper, loc ).
noun( 'vandal', 'vandals', count, _ ).
noun( 'vanessa', '-', proper, per ).
noun( 'varanasi', '-', proper, loc ).
noun( 'vasteras', '-', proper, loc ).
noun( 'vatican', 'vaticans', count, _ ).
noun( 'venezia', '-', proper, loc ).
noun( 'venezuela', '-', proper, loc ).
noun( 'venezuelan', 'venezuelans', count, _ ).
noun( 'venice', '-', proper, loc ).
noun( 'ventnor', '-', proper, loc ).
noun( 'venus', '-', proper, _ ).
noun( 'vera', '-', proper, per ).
noun( 'vermont', '-', proper, loc ).
noun( 'vernon', '-', proper, per ).
noun( 'verona', '-', proper, loc ).
noun( 'veronica', '-', proper, per ).
noun( 'vic', '-', proper, per ).
noun( 'vicenza', '-', proper, loc ).
noun( 'vicky', '-', proper, per ).
noun( 'victor', '-', proper, per ).
noun( 'victoria', '-', proper, per ).
noun( 'victorian', 'victorians', count, _ ).
noun( 'victoriana', '-', mass, _ ).
noun( 'vienna', '-', proper, loc ).
noun( 'vietnam', '-', proper, loc ).
noun( 'vietnamese', 'vietnamese', both, _ ).
noun( 'viewpark', '-', proper, loc ).
noun( 'vigo', '-', proper, loc ).
noun( 'viking', 'vikings', count, _ ).
noun( 'vince', '-', proper, per ).
noun( 'vincent', '-', proper, per ).
noun( 'viola', '-', proper, per ).
noun( 'violet', '-', proper, per ).
noun( 'virginia', '-', mass, _ ).
noun( 'virginia', '-', proper, per ).
noun( 'virginia water', '-', proper, loc ).
noun( 'virgo', '-', proper, _ ).
noun( 'vitoria', '-', proper, loc ).
noun( 'viv', '-', proper, per ).
noun( 'vivian', '-', proper, per ).
noun( 'vivien', '-', proper, per ).
noun( 'vivienne', '-', proper, per ).
noun( 'volgograd', '-', proper, loc ).
noun( 'voronezh', '-', proper, loc ).
noun( 'vulgate', 'vulgates', count, _ ).
noun( 'wc', '-', count, _ ).
noun( '-', 'wcs', count, _ ).
noun( 'who', '-', count, _ ).
noun( 'wi', '-', count, _ ).
noun( 'wrac', '-', count, _ ).
noun( 'wraf', '-', count, _ ).
noun( 'wrns', '-', count, _ ).
noun( 'wakefield', '-', proper, loc ).
noun( 'wales', '-', proper, loc ).
noun( 'wall street', '-', proper, _ ).
noun( 'wallace', '-', proper, per ).
noun( 'wallasey', '-', proper, loc ).
noun( 'wallingford', '-', proper, loc ).
noun( 'wallsend', '-', proper, loc ).
noun( 'wally', '-', proper, per ).
noun( 'walsall', '-', proper, loc ).
noun( 'walsham', '-', proper, loc ).
noun( 'walter', '-', proper, per ).
noun( 'waltham forest', '-', proper, loc ).
noun( 'walton', '-', proper, loc ).
noun( 'wandsworth', '-', proper, loc ).
noun( 'wantage', '-', proper, loc ).
noun( 'ware', '-', proper, loc ).
noun( 'warley', '-', proper, loc ).
noun( 'warminster', '-', proper, loc ).
noun( 'warrington', '-', proper, loc ).
noun( 'warsaw', '-', proper, loc ).
noun( 'warsop', '-', proper, loc ).
noun( 'warwick', '-', proper, loc ).
noun( 'warwickshire', '-', proper, loc ).
noun( 'washington', '-', proper, loc ).
noun( 'washington dc', '-', proper, loc ).
noun( 'waterford', '-', proper, loc ).
noun( 'waterloo', '-', proper, _ ).
noun( 'watford', '-', proper, loc ).
noun( 'wayne', '-', proper, per ).
noun( 'wear', '-', proper, loc ).
noun( 'wed', '-', proper, _ ).
noun( 'wednesday', 'wednesdays', count, _ ).
noun( 'wednesday', '-', proper, _ ).
noun( 'wellingborough', '-', proper, loc ).
noun( 'wellington', '-', proper, loc ).
noun( 'wells', '-', proper, loc ).
noun( 'welsh', '-', mass, _ ).
noun( 'welshman', 'welshmen', count, _ ).
noun( 'welshpool', '-', proper, loc ).
noun( 'welwyn', '-', proper, loc ).
noun( 'welwyn garden city', '-', proper, loc ).
noun( 'wendover', '-', proper, loc ).
noun( 'wendy', '-', proper, per ).
noun( 'wesleyan', 'wesleyans', count, _ ).
noun( 'west bromwich', '-', proper, loc ).
noun( 'westbury', '-', proper, loc ).
noun( 'westhoughton', '-', proper, loc ).
noun( 'westmeath', '-', proper, loc ).
noun( 'westminster', '-', proper, loc ).
noun( 'westmoreland', '-', proper, loc ).
noun( 'weston-super-mare', '-', proper, loc ).
noun( 'wetherby', '-', proper, loc ).
noun( 'wexford', '-', proper, loc ).
noun( 'weymouth', '-', proper, loc ).
noun( 'whaley bridge', '-', proper, loc ).
noun( 'wheatley hill', '-', proper, loc ).
noun( 'whig', 'whigs', count, _ ).
noun( 'whit', 'whits', count, _ ).
noun( 'whitburn', '-', proper, loc ).
noun( 'whitby', '-', proper, loc ).
noun( 'whitchurch', '-', proper, loc ).
noun( 'whitehall', '-', proper, _ ).
noun( 'whitehaven', '-', proper, loc ).
noun( 'whitley bay', '-', proper, loc ).
noun( 'whitstable', '-', proper, loc ).
noun( 'whitsun', 'whitsuns', count, _ ).
noun( 'whitsun', '-', proper, _ ).
noun( 'whitsuntide', 'whitsuntides', count, _ ).
noun( 'whitsuntide', '-', proper, _ ).
noun( 'whittle-le-woods', '-', proper, loc ).
noun( 'whittlesey', '-', proper, loc ).
noun( 'whitworth', '-', proper, loc ).
noun( 'wick', '-', proper, loc ).
noun( 'wickford', '-', proper, loc ).
noun( 'wicklow', '-', proper, loc ).
noun( 'widnes', '-', proper, loc ).
noun( 'wiesbaden', '-', proper, loc ).
noun( 'wigan', '-', proper, loc ).
noun( 'wight', '-', proper, loc ).
noun( 'wilf', '-', proper, per ).
noun( 'wilfrid', '-', proper, per ).
noun( 'wilhelmshaven', '-', proper, loc ).
noun( 'will', '-', proper, per ).
noun( 'william', '-', proper, per ).
noun( 'willie', '-', proper, per ).
noun( 'willington', '-', proper, loc ).
noun( 'willy', '-', proper, per ).
noun( 'wilmslow', '-', proper, loc ).
noun( 'wilton', 'wiltons', count, _ ).
noun( 'wiltshire', '-', proper, loc ).
noun( 'wimbourne', '-', proper, loc ).
noun( 'wimbourne minster', '-', proper, loc ).
noun( 'winchester', '-', proper, loc ).
noun( 'windermere', '-', proper, loc ).
noun( 'windsor', '-', proper, loc ).
noun( 'wingate', '-', proper, loc ).
noun( 'winifred', '-', proper, per ).
noun( 'winnie', '-', proper, per ).
noun( 'winnipeg', '-', proper, loc ).
noun( 'winsford', '-', proper, loc ).
noun( 'wisbech', '-', proper, loc ).
noun( 'wisconsin', '-', proper, loc ).
noun( 'witham', '-', proper, loc ).
noun( 'withernsea', '-', proper, loc ).
noun( 'witney', '-', proper, loc ).
noun( 'woburn', '-', proper, loc ).
noun( 'wokingham', '-', proper, loc ).
noun( 'wolverhampton', '-', proper, loc ).
noun( 'wolverton', '-', proper, loc ).
noun( 'wombourne', '-', proper, loc ).
noun( 'woodbridge', '-', proper, loc ).
noun( 'wootton bassett', '-', proper, loc ).
noun( 'worcester', '-', proper, loc ).
noun( 'workington', '-', proper, loc ).
noun( 'worksop', '-', proper, loc ).
noun( 'worthing', '-', proper, loc ).
noun( 'wrexham', '-', proper, loc ).
noun( 'writtle', '-', proper, loc ).
noun( 'wroclaw', '-', proper, loc ).
noun( 'wroughton', '-', proper, loc ).
noun( 'wuhan', '-', proper, loc ).
noun( 'wuppertal', '-', proper, loc ).
noun( 'wurzburg', '-', proper, loc ).
noun( 'wusih', '-', proper, loc ).
noun( 'wycombe', '-', proper, loc ).
noun( 'wymondham', '-', proper, loc ).
noun( 'wyoming', '-', proper, loc ).
noun( 'wythall', '-', proper, loc ).
noun( 'x-ray', 'x-rays', count, _ ).
noun( 'xerox', 'xeroxes', count, _ ).
noun( 'xmas', 'xmases', count, _ ).
noun( 'xmas', '-', proper, _ ).
noun( 'yha', '-', count, _ ).
noun( 'ymca', '-', count, _ ).
noun( 'ywca', '-', count, _ ).
noun( 'yank', 'yanks', count, _ ).
noun( 'yankee', 'yankees', count, _ ).
noun( 'yarmouth', '-', proper, loc ).
noun( 'yaroslavi', '-', proper, loc ).
noun( 'yate', '-', proper, loc ).
noun( 'yateley', '-', proper, loc ).
noun( 'yemen', '-', proper, loc ).
noun( 'yemeni', 'yemenis', count, _ ).
noun( 'yeovil', '-', proper, loc ).
noun( 'yerevan', '-', proper, loc ).
noun( 'yiddish', '-', mass, _ ).
noun( 'yokohama', '-', proper, loc ).
noun( 'york', '-', proper, loc ).
noun( 'yorkley', '-', proper, loc ).
noun( 'yorkshire', '-', proper, loc ).
noun( 'ystradgynlais', '-', proper, loc ).
noun( 'yugoslav', 'yugoslavs', count, _ ).
noun( 'yugoslavia', '-', proper, loc ).
noun( 'yugoslavian', 'yugoslavians', count, _ ).
noun( 'yukon', '-', proper, loc ).
noun( 'yvonne', '-', proper, per ).
noun( 'zagreb', '-', proper, loc ).
noun( 'zaire', '-', proper, loc ).
noun( 'zairean', 'zaireans', count, _ ).
noun( 'zambia', '-', proper, loc ).
noun( 'zambian', 'zambians', count, _ ).
noun( 'zaporozhye', '-', proper, loc ).
noun( 'zaragoza', '-', proper, loc ).
noun( 'zealand', '-', proper, loc ).
noun( 'zealander', 'zealanders', count, _ ).
noun( 'zen', '-', mass, _ ).
noun( 'zimbabwe', '-', proper, loc ).
noun( 'zimbabwean', 'zimbabweans', count, _ ).
noun( 'zion', 'zions', count, _ ).
noun( 'zionism', '-', mass, _ ).
noun( 'zionist', 'zionists', count, _ ).
noun( 'zoe', '-', proper, per ).
noun( 'zurich', '-', proper, loc ).
noun( '_eclair', '_eclairs', count, _ ).
noun( '_eclat', '-', mass, _ ).
noun( '_elan', '-', mass, _ ).
noun( '_elite', '_elites', count, _ ).
noun( '_emigr_e', '_emigr_es', count, _ ).
noun( '_ep_ee', '_ep_ees', count, _ ).
noun( 'a', '-', count, _ ).
noun( 'abacus', 'abacuses', count, _ ).
noun( 'abandon', '-', mass, _ ).
noun( 'abandonment', '-', mass, _ ).
noun( 'abasement', '-', mass, _ ).
noun( 'abatement', '-', mass, _ ).
noun( 'abattoir', 'abattoirs', count, _ ).
noun( 'abb_e', 'abb_es', count, _ ).
noun( 'abbess', 'abbesses', count, _ ).
noun( 'abbey', 'abbeys', count, _ ).
noun( 'abbot', 'abbots', count, _ ).
noun( 'abbreviation', 'abbreviations', both, _ ).
noun( 'abdication', 'abdications', both, _ ).
noun( 'abdomen', 'abdomens', count, _ ).
noun( 'abduction', 'abductions', count, _ ).
noun( 'aberration', 'aberrations', both, _ ).
noun( 'abeyance', '-', mass, _ ).
noun( 'abhorrence', '-', mass, _ ).
noun( 'ability', 'abilities', both, _ ).
noun( 'abjection', 'abjections', count, _ ).
noun( 'abjuration', 'abjurations', both, _ ).
noun( 'ablative', 'ablatives', count, _ ).
noun( 'ablaut', 'ablauts', count, _ ).
noun( 'ablution', 'ablutions', count, _ ).
noun( 'abnegation', '-', mass, _ ).
noun( 'abnormality', 'abnormalities', both, _ ).
noun( 'abode', 'abodes', count, _ ).
noun( 'abolition', '-', mass, _ ).
noun( 'abolitionist', 'abolitionists', count, _ ).
noun( 'abomination', 'abominations', both, _ ).
noun( 'aboriginal', 'aboriginals', count, _ ).
noun( 'aborigine', 'aborigines', count, _ ).
noun( 'abortion', 'abortions', both, _ ).
noun( 'abortionist', 'abortionists', count, _ ).
noun( 'about-face', '-', count, _ ).
noun( 'abracadabra', '-', mass, _ ).
noun( 'abrasion', 'abrasions', both, _ ).
noun( 'abrasive', 'abrasives', both, _ ).
noun( 'abridgement', 'abridgements', both, _ ).
noun( 'abridgment', 'abridgments', both, _ ).
noun( 'abrogation', 'abrogations', count, _ ).
noun( 'abruptness', '-', mass, _ ).
noun( 'abscess', 'abscesses', count, _ ).
noun( 'absence', 'absences', both, _ ).
noun( 'absent-mindedness', '-', mass, _ ).
noun( 'absentee', 'absentees', count, _ ).
noun( 'absenteeism', '-', mass, _ ).
noun( 'absinth', '-', mass, _ ).
noun( 'absinthe', '-', mass, _ ).
noun( 'absolution', '-', mass, _ ).
noun( 'absolutism', '-', mass, _ ).
noun( 'absorbent', 'absorbents', count, _ ).
noun( 'absorption', '-', mass, _ ).
noun( 'abstainer', 'abstainers', count, _ ).
noun( 'abstemiousness', '-', mass, _ ).
noun( 'abstention', 'abstentions', both, _ ).
noun( 'abstinence', '-', mass, _ ).
noun( 'abstract', 'abstracts', count, _ ).
noun( 'abstraction', 'abstractions', both, _ ).
noun( 'abstruseness', '-', mass, _ ).
noun( 'absurdity', 'absurdities', both, _ ).
noun( 'abundance', '-', mass, _ ).
noun( 'abuse', 'abuses', both, _ ).
noun( 'abutment', 'abutments', count, _ ).
noun( 'abysm', 'abysms', count, _ ).
noun( 'abyss', 'abysses', count, _ ).
noun( 'acacia', 'acacias', count, _ ).
noun( 'academic', 'academics', count, _ ).
noun( 'academician', 'academicians', count, _ ).
noun( 'academy', 'academies', count, _ ).
noun( 'accelerando', 'accelerandos', count, _ ).
noun( 'acceleration', '-', mass, _ ).
noun( 'accelerator', 'accelerators', count, _ ).
noun( 'accent', 'accents', both, _ ).
noun( 'accentuation', 'accentuations', count, _ ).
noun( 'acceptability', '-', mass, _ ).
noun( 'acceptance', '-', mass, _ ).
noun( 'acceptation', 'acceptations', count, _ ).
noun( 'access', '-', mass, _ ).
noun( 'accessary', 'accessaries', count, _ ).
noun( 'accessibility', '-', mass, _ ).
noun( 'accession', 'accessions', both, _ ).
noun( 'accessory', 'accessories', count, _ ).
noun( 'accidence', '-', mass, _ ).
noun( 'accident', 'accidents', both, _ ).
noun( 'acclaim', '-', mass, _ ).
noun( 'acclamation', '-', mass, _ ).
noun( 'acclimation', '-', mass, _ ).
noun( 'acclimatization', '-', mass, _ ).
noun( 'acclivity', 'acclivities', count, _ ).
noun( 'accolade', 'accolades', count, _ ).
noun( 'accommodation', 'accommodations', both, _ ).
noun( 'accompaniment', 'accompaniments', count, _ ).
noun( 'accompanist', 'accompanists', count, _ ).
noun( 'accomplice', 'accomplices', count, _ ).
noun( 'accomplishment', 'accomplishments', both, _ ).
noun( 'accord', 'accords', both, _ ).
noun( 'accordance', 'accordances', count, _ ).
noun( 'accordion', 'accordions', count, _ ).
noun( 'accouchement', 'accouchements', count, _ ).
noun( 'account', 'accounts', both, _ ).
noun( 'accountancy', '-', mass, _ ).
noun( 'accountant', 'accountants', count, _ ).
noun( 'accretion', 'accretions', both, _ ).
noun( 'accumulation', 'accumulations', both, _ ).
noun( 'accumulator', 'accumulators', count, _ ).
noun( 'accuracy', 'accuracies', both, _ ).
noun( 'accusation', 'accusations', both, _ ).
noun( 'accusative', 'accusatives', count, _ ).
noun( 'accuser', 'accusers', count, _ ).
noun( 'ace', 'aces', count, _ ).
noun( 'acerbity', 'acerbities', both, _ ).
noun( 'acetate', 'acetates', count, _ ).
noun( 'acetylene', '-', mass, _ ).
noun( 'ache', 'aches', count, _ ).
noun( 'achievement', 'achievements', both, _ ).
noun( 'acid', 'acids', both, _ ).
noun( 'acidity', '-', mass, _ ).
noun( 'ack-ack', '-', mass, _ ).
noun( 'acknowledgement', 'acknowledgements', count, _ ).
noun( 'acme', '-', count, _ ).
noun( 'acne', '-', mass, _ ).
noun( 'acolyte', 'acolytes', count, _ ).
noun( 'aconite', 'aconites', count, _ ).
noun( 'acorn', 'acorns', count, _ ).
noun( 'acorn-cup', 'acorn-cups', count, _ ).
noun( 'acoustic', 'acoustics', count, _ ).
noun( 'acoustics', 'acoustics', mass, _ ).
noun( 'acquaintance', 'acquaintances', both, _ ).
noun( 'acquaintanceship', 'acquaintanceships', count, _ ).
noun( 'acquiescence', 'acquiescences', count, _ ).
noun( 'acquirement', 'acquirements', both, _ ).
noun( 'acquisition', 'acquisitions', both, _ ).
noun( 'acquittal', 'acquittals', both, _ ).
noun( 'acre', 'acres', count, _ ).
noun( 'acreage', '-', mass, _ ).
noun( 'acrimony', '-', mass, _ ).
noun( 'acrobat', 'acrobats', count, _ ).
noun( 'acrobatics', 'acrobatics', mass, _ ).
noun( 'acronym', 'acronyms', count, _ ).
noun( 'acropolis', 'acropolises', count, _ ).
noun( 'acrostic', 'acrostics', count, _ ).
noun( 'acrylic', 'acrylics', count, _ ).
noun( 'act', 'acts', count, _ ).
noun( 'acting', '-', mass, _ ).
noun( 'actinism', '-', mass, _ ).
noun( 'action', 'actions', both, _ ).
noun( 'activation', 'activations', count, _ ).
noun( 'activist', 'activists', count, _ ).
noun( 'activity', 'activities', both, _ ).
noun( 'actor', 'actors', count, _ ).
noun( 'actress', 'actresses', count, _ ).
noun( 'actuality', 'actualities', both, _ ).
noun( 'actuary', 'actuaries', count, _ ).
noun( 'acuity', '-', mass, _ ).
noun( 'acumen', '-', mass, _ ).
noun( 'acupuncture', '-', mass, _ ).
noun( 'acuteness', '-', mass, _ ).
noun( 'ad', 'ads', count, _ ).
noun( 'ad-man', 'ad-men', count, _ ).
noun( 'adage', 'adages', count, _ ).
noun( 'adagio', 'adagios', count, _ ).
noun( 'adamant', 'adamants', count, _ ).
noun( 'adaptability', '-', mass, _ ).
noun( 'adaptation', 'adaptations', both, _ ).
noun( 'adapter', 'adapters', count, _ ).
noun( 'adaptor', 'adaptors', count, _ ).
noun( 'addendum', 'addenda', count, _ ).
noun( 'adder', 'adders', count, _ ).
noun( 'addict', 'addicts', count, _ ).
noun( 'addiction', 'addictions', both, _ ).
noun( 'adding-machine', 'adding-machines', count, _ ).
noun( 'addition', 'additions', both, _ ).
noun( 'additive', 'additives', count, _ ).
noun( 'addle-head', 'addle-heads', count, _ ).
noun( 'address', 'addresses', both, _ ).
noun( 'addressee', 'addressees', count, _ ).
noun( 'adept', 'adepts', count, _ ).
noun( 'adequacy', '-', mass, _ ).
noun( 'adherence', 'adherences', count, _ ).
noun( 'adherent', 'adherents', count, _ ).
noun( 'adhesion', 'adhesions', both, _ ).
noun( 'adhesive', 'adhesives', both, _ ).
noun( 'adieu', 'adieus', count, _ ).
noun( 'adjective', 'adjectives', count, _ ).
noun( 'adjournment', 'adjournments', count, _ ).
noun( 'adjudication', 'adjudications', count, _ ).
noun( 'adjudicator', 'adjudicators', count, _ ).
noun( 'adjunct', 'adjuncts', count, _ ).
noun( 'adjuration', 'adjurations', both, _ ).
noun( 'adjuster', 'adjusters', count, _ ).
noun( 'adjustment', 'adjustments', both, _ ).
noun( 'adjutant', 'adjutants', count, _ ).
noun( 'admass', '-', mass, _ ).
noun( 'administration', 'administrations', both, _ ).
noun( 'administrator', 'administrators', count, _ ).
noun( 'admiral', 'admirals', count, _ ).
noun( 'admiralty', 'admiralties', count, _ ).
noun( 'admiration', '-', mass, _ ).
noun( 'admirer', 'admirers', count, _ ).
noun( 'admissibility', '-', mass, _ ).
noun( 'admission', 'admissions', both, _ ).
noun( 'admittance', '-', mass, _ ).
noun( 'admixture', 'admixtures', count, _ ).
noun( 'admonition', 'admonitions', both, _ ).
noun( 'ado', '-', mass, _ ).
noun( 'adobe', '-', mass, _ ).
noun( 'adolescence', '-', mass, _ ).
noun( 'adolescent', 'adolescents', count, _ ).
noun( 'adoption', 'adoptions', count, _ ).
noun( 'adoration', '-', mass, _ ).
noun( 'adorer', 'adorers', count, _ ).
noun( 'adornment', 'adornments', both, _ ).
noun( 'adrenalin', '-', mass, _ ).
noun( 'adroitness', '-', mass, _ ).
noun( 'adulation', '-', mass, _ ).
noun( 'adult', 'adults', count, _ ).
noun( 'adulterant', 'adulterants', count, _ ).
noun( 'adulteration', 'adulterations', count, _ ).
noun( 'adulterer', 'adulterers', count, _ ).
noun( 'adulteress', 'adulteresses', count, _ ).
noun( 'adultery', 'adulteries', both, _ ).
noun( 'adulthood', 'adulthoods', both, _ ).
noun( 'advance', 'advances', both, _ ).
noun( 'advancement', '-', mass, _ ).
noun( 'advantage', 'advantages', both, _ ).
noun( 'advent', 'advents', count, _ ).
noun( 'adventure', 'adventures', both, _ ).
noun( 'adventurer', 'adventurers', count, _ ).
noun( 'adventuress', 'adventuresses', count, _ ).
noun( 'adverb', 'adverbs', count, _ ).
noun( 'adversary', 'adversaries', count, _ ).
noun( 'adversity', 'adversities', both, _ ).
noun( 'advert', 'adverts', count, _ ).
noun( 'advertisement', 'advertisements', both, _ ).
noun( 'advertiser', 'advertisers', count, _ ).
noun( 'advice', 'advices', both, _ ).
noun( 'advisability', '-', mass, _ ).
noun( 'adviser', 'advisers', count, _ ).
noun( 'advocacy', '-', mass, _ ).
noun( 'advocate', 'advocates', count, _ ).
noun( 'advowson', 'advowsons', count, _ ).
noun( 'adz', 'adzes', count, _ ).
noun( 'adze', 'adzes', count, _ ).
noun( 'aegis', 'aegises', count, _ ).
noun( 'aeon', 'aeons', count, _ ).
noun( 'aeration', 'aerations', count, _ ).
noun( 'aerial', 'aerials', count, _ ).
noun( 'aerie', 'aeries', count, _ ).
noun( 'aerobatics', 'aerobatics', mass, _ ).
noun( 'aerodrome', 'aerodromes', count, _ ).
noun( 'aerodynamics', 'aerodynamics', mass, _ ).
noun( 'aeronaut', 'aeronauts', count, _ ).
noun( 'aeronautics', 'aeronautics', mass, _ ).
noun( 'aeroplane', 'aeroplanes', count, _ ).
noun( 'aerosol', 'aerosols', both, _ ).
noun( 'aerospace', '-', mass, _ ).
noun( 'aertex', '-', mass, _ ).
noun( 'aery', 'aeries', count, _ ).
noun( 'aesthete', 'aesthetes', count, _ ).
noun( 'aesthetic', '-', mass, _ ).
noun( 'aesthetics', 'aesthetics', mass, _ ).
noun( 'aether', '-', mass, _ ).
noun( 'aetiology', 'aetiologies', count, _ ).
noun( 'affability', '-', mass, _ ).
noun( 'affair', 'affairs', count, _ ).
noun( 'affectation', 'affectations', both, _ ).
noun( 'affection', 'affections', both, _ ).
noun( 'affidavit', 'affidavits', count, _ ).
noun( 'affiliation', 'affiliations', both, _ ).
noun( 'affinity', 'affinities', both, _ ).
noun( 'affirmation', 'affirmations', both, _ ).
noun( 'affirmative', 'affirmatives', count, _ ).
noun( 'affix', 'affixes', count, _ ).
noun( 'afflatus', '-', mass, _ ).
noun( 'affliction', 'afflictions', both, _ ).
noun( 'affluence', '-', mass, _ ).
noun( 'affluent', 'affluents', count, _ ).
noun( 'afforestation', 'afforestations', count, _ ).
noun( 'affray', 'affrays', count, _ ).
noun( 'affront', 'affronts', count, _ ).
noun( 'aftercare', '-', mass, _ ).
noun( 'afterdamp', '-', mass, _ ).
noun( 'aftereffect', 'aftereffects', count, _ ).
noun( 'afterglow', '-', count, _ ).
noun( 'aftermath', '', count, _ ).
noun( 'afternoon', 'afternoons', both, _ ).
noun( 'afterthought', 'afterthoughts', both, _ ).
noun( 'agar-agar', '-', mass, _ ).
noun( 'agate', 'agates', count, _ ).
noun( 'agave', 'agaves', count, _ ).
noun( 'age', 'ages', both, _ ).
noun( 'age-bracket', 'age-brackets', count, _ ).
noun( 'age-group', 'age-groups', count, _ ).
noun( 'ageing', '-', mass, _ ).
noun( 'agency', 'agencies', both, _ ).
noun( 'agenda', 'agendas', count, _ ).
noun( 'agent', 'agents', count, _ ).
noun( 'agent provocateur', 'agents provocateurs', count, _ ).
noun( 'agglomeration', 'agglomerations', both, _ ).
noun( 'aggrandizement', 'aggrandizements', count, _ ).
noun( 'aggravation', 'aggravations', both, _ ).
noun( 'aggregate', 'aggregates', count, _ ).
noun( 'aggregation', 'aggregations', both, _ ).
noun( 'aggression', 'aggressions', both, _ ).
noun( 'aggressiveness', '-', mass, _ ).
noun( 'aggressor', 'aggressors', count, _ ).
noun( 'aggro', '-', mass, _ ).
noun( 'agility', '-', mass, _ ).
noun( 'aging', '-', mass, _ ).
noun( 'agitation', 'agitations', both, _ ).
noun( 'agitator', 'agitators', count, _ ).
noun( 'agnail', '-', mass, _ ).
noun( 'agnostic', 'agnostics', count, _ ).
noun( 'agnosticism', '-', mass, _ ).
noun( 'agony', 'agonies', both, _ ).
noun( 'agora', 'agoras', count, _ ).
noun( 'agoraphobia', '-', mass, _ ).
noun( 'agreement', 'agreements', both, _ ).
noun( 'agriculture', '-', mass, _ ).
noun( 'ague', 'agues', count, _ ).
noun( 'aid', 'aids', both, _ ).
noun( 'aide-de-camp', 'aides-de-camp', count, _ ).
noun( 'aide-m_emoire', 'aide-m_emoires', count, _ ).
noun( 'aigret', 'aigrets', count, _ ).
noun( 'aigrette', 'aigrettes', count, _ ).
noun( 'aileron', 'ailerons', count, _ ).
noun( 'ailment', 'ailments', count, _ ).
noun( 'aim', 'aims', both, _ ).
noun( 'air', 'airs', both, _ ).
noun( 'air-bladder', 'air-bladders', count, _ ).
noun( 'air-conditioning', '-', mass, _ ).
noun( 'air-pump', 'air-pumps', count, _ ).
noun( 'air-raid', 'air-raids', count, _ ).
noun( 'air-shaft', 'air-shafts', count, _ ).
noun( 'air-sickness', '-', mass, _ ).
noun( 'airbed', 'airbeds', count, _ ).
noun( 'airbrake', 'airbrakes', count, _ ).
noun( 'aircraft', 'aircraft', count, _ ).
noun( 'aircraftman', 'aircraftmen', count, _ ).
noun( 'aircrew', 'aircrews', count, _ ).
noun( 'airdrome', 'airdromes', count, _ ).
noun( 'airfield', 'airfields', count, _ ).
noun( 'airflow', 'airflows', count, _ ).
noun( 'airframe', 'airframes', count, _ ).
noun( 'airing', 'airings', count, _ ).
noun( 'airing-cupboard', 'airing-cupboards', count, _ ).
noun( 'airline', 'airlines', count, _ ).
noun( 'airliner', 'airliners', count, _ ).
noun( 'airmail', '-', mass, _ ).
noun( 'airman', 'airmen', count, _ ).
noun( 'airplane', 'airplanes', count, _ ).
noun( 'airport', 'airports', count, _ ).
noun( 'airscrew', 'airscrews', count, _ ).
noun( 'airship', 'airships', count, _ ).
noun( 'airstrip', 'airstrips', count, _ ).
noun( 'airway', 'airways', count, _ ).
noun( 'airwoman', 'airwomen', count, _ ).
noun( 'airworthiness', '-', mass, _ ).
noun( 'aisle', 'aisles', count, _ ).
noun( 'aitch', 'aitches', count, _ ).
noun( 'aitch-bone', 'aitch-bones', count, _ ).
noun( 'alabaster', '-', mass, _ ).
noun( 'alacrity', '-', mass, _ ).
noun( 'alarm', 'alarms', both, _ ).
noun( 'alarm-clock', 'alarm-clocks', count, _ ).
noun( 'alarmist', 'alarmists', count, _ ).
noun( 'alb', 'albs', count, _ ).
noun( 'albatross', 'albatrosses', count, _ ).
noun( 'albino', 'albinos', count, _ ).
noun( 'album', 'albums', count, _ ).
noun( 'albumen', '-', mass, _ ).
noun( 'alchemist', 'alchemists', count, _ ).
noun( 'alchemy', '-', mass, _ ).
noun( 'alcohol', 'alcohols', both, _ ).
noun( 'alcoholic', 'alcoholics', count, _ ).
noun( 'alcoholism', '-', mass, _ ).
noun( 'alcove', 'alcoves', count, _ ).
noun( 'alder', 'alders', count, _ ).
noun( 'alderman', 'aldermen', count, _ ).
noun( 'ale', 'ales', both, _ ).
noun( 'ale-house', 'ale-houses', count, _ ).
noun( 'alert', 'alerts', count, _ ).
noun( 'alertness', '-', mass, _ ).
noun( 'alexandrine', 'alexandrines', count, _ ).
noun( 'alexia', '-', mass, _ ).
noun( 'alexic', 'alexics', count, _ ).
noun( 'alfalfa', '-', mass, _ ).
noun( 'alga', 'algae', count, _ ).
noun( 'algebra', 'algebras', both, _ ).
noun( 'alias', 'aliases', count, _ ).
noun( 'alibi', 'alibis', count, _ ).
noun( 'alien', 'aliens', count, _ ).
noun( 'alienation', '-', mass, _ ).
noun( 'alienist', 'alienists', count, _ ).
noun( 'alignment', 'alignments', both, _ ).
noun( 'alimony', '-', mass, _ ).
noun( 'alkali', 'alkalis', both, _ ).
noun( 'all', '-', count, _ ).
noun( 'all-rounder', 'all-rounders', count, _ ).
noun( 'allegation', 'allegations', both, _ ).
noun( 'allegiance', '-', mass, _ ).
noun( 'allegory', 'allegories', count, _ ).
noun( 'allegretto', 'allegrettos', count, _ ).
noun( 'allegro', 'allegros', count, _ ).
noun( 'allergen', 'allergens', count, _ ).
noun( 'allergy', 'allergies', count, _ ).
noun( 'alleviation', 'alleviations', count, _ ).
noun( 'alley', 'alleys', count, _ ).
noun( 'alleyway', 'alleyways', count, _ ).
noun( 'alliance', 'alliances', both, _ ).
noun( 'alligator', 'alligators', count, _ ).
noun( 'alliteration', '-', mass, _ ).
noun( 'allocation', 'allocations', both, _ ).
noun( 'allotment', 'allotments', both, _ ).
noun( 'allowance', 'allowances', count, _ ).
noun( 'alloy', 'alloys', both, _ ).
noun( 'allspice', '-', mass, _ ).
noun( 'allure', 'allures', both, _ ).
noun( 'allurement', 'allurements', both, _ ).
noun( 'allusion', 'allusions', count, _ ).
noun( 'ally', 'allies', count, _ ).
noun( 'almanac', 'almanacs', count, _ ).
noun( 'almighty', '-', count, _ ).
noun( 'almond', 'almonds', count, _ ).
noun( 'almoner', 'almoners', count, _ ).
noun( 'alms-box', 'alms-boxes', count, _ ).
noun( 'alms-giving', '-', mass, _ ).
noun( 'alms-house', 'alms-houses', count, _ ).
noun( 'aloe', 'aloes', count, _ ).
noun( 'aloofness', '-', mass, _ ).
noun( 'alp', 'alps', count, _ ).
noun( 'alpaca', 'alpacas', both, _ ).
noun( 'alpenstock', 'alpenstocks', count, _ ).
noun( 'alpha', 'alphas', count, _ ).
noun( 'alphabet', 'alphabets', count, _ ).
noun( 'alpinist', 'alpinists', count, _ ).
noun( 'alsatian', 'alsatians', count, _ ).
noun( 'also-ran', 'also-rans', count, _ ).
noun( 'altar', 'altars', count, _ ).
noun( 'altar-piece', 'altar-pieces', count, _ ).
noun( 'alter ego', 'alter egos', count, _ ).
noun( 'alteration', 'alterations', both, _ ).
noun( 'altercation', 'altercations', both, _ ).
noun( 'alternation', 'alternations', count, _ ).
noun( 'alternative', 'alternatives', count, _ ).
noun( 'altimeter', 'altimeters', count, _ ).
noun( 'altitude', 'altitudes', count, _ ).
noun( 'alto', 'altos', count, _ ).
noun( 'altruism', 'altruisms', both, _ ).
noun( 'altruist', 'altruists', count, _ ).
noun( 'alum', '-', mass, _ ).
noun( 'aluminium', '-', mass, _ ).
noun( 'alumna', 'alumnae', count, _ ).
noun( 'alumnus', 'alumni', count, _ ).
noun( 'alveolar', 'alveolars', count, _ ).
noun( 'am', '-', proper, _ ).
noun( 'amah', 'amahs', count, _ ).
noun( 'amalgam', 'amalgams', count, _ ).
noun( 'amalgamation', 'amalgamations', both, _ ).
noun( 'amanuensis', 'amanuenses', count, _ ).
noun( 'amaryllis', 'amaryllises', count, _ ).
noun( 'amateur', 'amateurs', count, _ ).
noun( 'amateurism', '-', mass, _ ).
noun( 'amazement', '-', mass, _ ).
noun( 'ambassador', 'ambassadors', count, _ ).
noun( 'ambassadress', 'ambassadresses', count, _ ).
noun( 'amber', '-', mass, _ ).
noun( 'ambergris', '-', mass, _ ).
noun( 'ambience', '-', count, _ ).
noun( 'ambiguity', 'ambiguities', both, _ ).
noun( 'ambit', 'ambits', count, _ ).
noun( 'ambition', 'ambitions', both, _ ).
noun( 'ambivalence', '-', mass, _ ).
noun( 'amble', 'ambles', count, _ ).
noun( 'ambrosia', '-', mass, _ ).
noun( 'ambulance', 'ambulances', count, _ ).
noun( 'ambuscade', 'ambuscades', count, _ ).
noun( 'ambush', 'ambushes', both, _ ).
noun( 'ameba', 'amebas', count, _ ).
noun( 'ameer', 'ameers', count, _ ).
noun( 'amelioration', 'ameliorations', count, _ ).
noun( 'amendment', 'amendments', both, _ ).
noun( 'amenity', 'amenities', count, _ ).
noun( 'amethyst', 'amethysts', count, _ ).
noun( 'amiability', '-', mass, _ ).
noun( 'amicability', '-', mass, _ ).
noun( 'amir', 'amirs', count, _ ).
noun( 'amity', '-', mass, _ ).
noun( 'ammeter', 'ammeters', count, _ ).
noun( 'ammonia', '-', mass, _ ).
noun( 'ammonite', 'ammonites', count, _ ).
noun( 'ammunition', '-', mass, _ ).
noun( 'amnesia', '-', mass, _ ).
noun( 'amnesty', 'amnesties', count, _ ).
noun( 'amoeba', 'amoebas', count, _ ).
noun( 'amortization', 'amortizations', count, _ ).
noun( 'amount', 'amounts', count, _ ).
noun( 'amour', 'amours', count, _ ).
noun( 'amour-propre', '-', mass, _ ).
noun( 'amp', 'amps', count, _ ).
noun( 'ampere', 'amperes', count, _ ).
noun( 'amphetamine', 'amphetamines', both, _ ).
noun( 'amphibian', 'amphibians', count, _ ).
noun( 'amphitheatre', 'amphitheatres', count, _ ).
noun( 'amphora', 'amphoras', count, _ ).
noun( 'amplification', 'amplifications', count, _ ).
noun( 'amplifier', 'amplifiers', count, _ ).
noun( 'amplitude', '-', mass, _ ).
noun( 'ampoule', 'ampoules', count, _ ).
noun( 'amputation', 'amputations', count, _ ).
noun( 'amulet', 'amulets', count, _ ).
noun( 'amusement', 'amusements', both, _ ).
noun( 'anachronism', 'anachronisms', count, _ ).
noun( 'anaconda', 'anacondas', count, _ ).
noun( 'anaemia', '-', mass, _ ).
noun( 'anaesthesia', '-', mass, _ ).
noun( 'anaesthetic', 'anaesthetics', count, _ ).
noun( 'anaesthetist', 'anaesthetists', count, _ ).
noun( 'anagram', 'anagrams', count, _ ).
noun( 'analgesia', '-', mass, _ ).
noun( 'analgesic', 'analgesics', count, _ ).
noun( 'analog', 'analogs', count, _ ).
noun( 'analogue', 'analogues', count, _ ).
noun( 'analogy', 'analogies', both, _ ).
noun( 'analysis', 'analyses', both, _ ).
noun( 'analyst', 'analysts', count, _ ).
noun( 'anapaest', 'anapaests', count, _ ).
noun( 'anarchism', '-', mass, _ ).
noun( 'anarchist', 'anarchists', count, _ ).
noun( 'anarchy', '-', mass, _ ).
noun( 'anathema', 'anathemas', count, _ ).
noun( 'anatomist', 'anatomists', count, _ ).
noun( 'anatomy', '-', mass, _ ).
noun( 'ancestor', 'ancestors', count, _ ).
noun( 'ancestress', 'ancestresses', count, _ ).
noun( 'ancestry', 'ancestries', count, _ ).
noun( 'anchor', 'anchors', count, _ ).
noun( 'anchorage', 'anchorages', count, _ ).
noun( 'anchorite', 'anchorites', count, _ ).
noun( 'anchorman', 'anchormen', count, _ ).
noun( 'anchovy', 'anchovies', count, _ ).
noun( 'andante', 'andantes', count, _ ).
noun( 'andiron', 'andirons', count, _ ).
noun( 'anecdote', 'anecdotes', count, _ ).
noun( 'anemometer', 'anemometers', count, _ ).
noun( 'anemone', 'anemones', count, _ ).
noun( 'aneroid', 'aneroids', count, _ ).
noun( 'anesthetic', 'anesthetics', count, _ ).
noun( 'anesthetist', 'anesthetists', count, _ ).
noun( 'angel', 'angels', count, _ ).
noun( 'angelica', '-', mass, _ ).
noun( 'angelus', 'angeluses', count, _ ).
noun( 'anger', '-', mass, _ ).
noun( 'angina', '-', mass, _ ).
noun( 'angle', 'angles', count, _ ).
noun( 'angle-dozer', 'angle-dozers', count, _ ).
noun( 'angle-iron', 'angle-irons', count, _ ).
noun( 'angler', 'anglers', count, _ ).
noun( 'anglicism', 'anglicisms', count, _ ).
noun( 'angling', '-', mass, _ ).
noun( 'angora', 'angoras', both, _ ).
noun( 'angostura', '-', mass, _ ).
noun( 'angst', '-', mass, _ ).
noun( 'anguish', '-', mass, _ ).
noun( 'angularity', 'angularities', both, _ ).
noun( 'aniline', 'anilines', both, _ ).
noun( 'animadversion', 'animadversions', count, _ ).
noun( 'animal', 'animals', count, _ ).
noun( 'animalcule', 'animalcules', count, _ ).
noun( 'animation', '-', mass, _ ).
noun( 'animism', '-', mass, _ ).
noun( 'animosity', 'animosities', both, _ ).
noun( 'animus', '-', mass, _ ).
noun( 'anise', 'anises', count, _ ).
noun( 'aniseed', '-', mass, _ ).
noun( 'ankle', 'ankles', count, _ ).
noun( 'anklet', 'anklets', count, _ ).
noun( 'anna', 'annas', count, _ ).
noun( 'annalist', 'annalists', count, _ ).
noun( 'annex', 'annexes', count, _ ).
noun( 'annexation', '-', mass, _ ).
noun( 'annexe', 'annexes', count, _ ).
noun( 'annihilation', '-', mass, _ ).
noun( 'anniversary', 'anniversaries', count, _ ).
noun( 'annotation', 'annotations', both, _ ).
noun( 'announcement', 'announcements', count, _ ).
noun( 'announcer', 'announcers', count, _ ).
noun( 'annoyance', 'annoyances', both, _ ).
noun( 'annual', 'annuals', count, _ ).
noun( 'annuitant', 'annuitants', count, _ ).
noun( 'annuity', 'annuities', count, _ ).
noun( 'annulment', 'annulments', count, _ ).
noun( 'annunciation', 'annunciations', count, _ ).
noun( 'anode', 'anodes', count, _ ).
noun( 'anodyne', 'anodynes', both, _ ).
noun( 'anointment', 'anointments', count, _ ).
noun( 'anomaly', 'anomalies', count, _ ).
noun( 'anonymity', '-', mass, _ ).
noun( 'anopheles', 'anopheles', count, _ ).
noun( 'anorak', 'anoraks', count, _ ).
noun( 'answer', 'answers', count, _ ).
noun( 'ant', 'ants', count, _ ).
noun( 'ant-eater', 'ant-eaters', count, _ ).
noun( 'ant-hill', 'ant-hills', count, _ ).
noun( 'antagonism', 'antagonisms', both, _ ).
noun( 'antagonist', 'antagonists', count, _ ).
noun( 'ante', 'antes', count, _ ).
noun( 'antecedence', 'antecedences', count, _ ).
noun( 'antecedent', 'antecedents', count, _ ).
noun( 'antechamber', 'antechambers', count, _ ).
noun( 'antediluvian', 'antediluvians', count, _ ).
noun( 'antelope', 'antelopes', count, _ ).
noun( 'antenna', 'antennae', count, _ ).
noun( 'anteroom', 'anterooms', count, _ ).
noun( 'anthem', 'anthems', count, _ ).
noun( 'anther', 'anthers', count, _ ).
noun( 'anthology', 'anthologies', count, _ ).
noun( 'anthracite', '-', mass, _ ).
noun( 'anthrax', '-', mass, _ ).
noun( 'anthropoid', 'anthropoids', count, _ ).
noun( 'anthropologist', 'anthropologists', count, _ ).
noun( 'anthropology', '-', mass, _ ).
noun( 'anti-semite', 'anti-semites', count, _ ).
noun( 'anti-semitism', '-', mass, _ ).
noun( 'anti-hero', 'anti-heroes', count, _ ).
noun( 'antibiotic', 'antibiotics', count, _ ).
noun( 'antibody', 'antibodies', count, _ ).
noun( 'antic', 'antics', count, _ ).
noun( 'anticipation', 'anticipations', both, _ ).
noun( 'anticlimax', 'anticlimaxes', count, _ ).
noun( 'anticyclone', 'anticyclones', count, _ ).
noun( 'antidote', 'antidotes', count, _ ).
noun( 'antifreeze', '-', mass, _ ).
noun( 'antiknock', '-', mass, _ ).
noun( 'antilogarithm', 'antilogarithms', count, _ ).
noun( 'antimacassar', 'antimacassars', count, _ ).
noun( 'antimony', '-', mass, _ ).
noun( 'antipathy', 'antipathies', both, _ ).
noun( 'antiquarian', 'antiquarians', count, _ ).
noun( 'antiquary', 'antiquaries', count, _ ).
noun( 'antique', 'antiques', count, _ ).
noun( 'antiquity', 'antiquities', both, _ ).
noun( 'antirrhinum', 'antirrhinums', count, _ ).
noun( 'antiseptic', 'antiseptics', count, _ ).
noun( 'antithesis', 'antitheses', both, _ ).
noun( 'antitoxin', 'antitoxins', count, _ ).
noun( 'antitrade', 'antitrades', count, _ ).
noun( 'antler', 'antlers', count, _ ).
noun( 'antonym', 'antonyms', count, _ ).
noun( 'anus', 'anuses', count, _ ).
noun( 'anvil', 'anvils', count, _ ).
noun( 'anxiety', 'anxieties', both, _ ).
noun( 'anybody', '-', count, _ ).
noun( 'anyone', '-', count, _ ).
noun( 'anything', '-', count, _ ).
noun( 'aorta', 'aortas', count, _ ).
noun( 'apache', 'apaches', count, _ ).
noun( 'apanage', '-', mass, _ ).
noun( 'apartheid', '-', mass, _ ).
noun( 'apartment', 'apartments', count, _ ).
noun( 'apathy', '-', mass, _ ).
noun( 'ape', 'apes', count, _ ).
noun( 'aperient', 'aperients', count, _ ).
noun( 'aperitif', 'aperitifs', count, _ ).
noun( 'aperture', 'apertures', count, _ ).
noun( 'apex', 'apexes', count, _ ).
noun( 'aphasia', '-', mass, _ ).
noun( 'aphid', 'aphids', count, _ ).
noun( 'aphis', 'aphides', count, _ ).
noun( 'aphorism', 'aphorisms', count, _ ).
noun( 'aphrodisiac', 'aphrodisiacs', both, _ ).
noun( 'apiarist', 'apiarists', count, _ ).
noun( 'apiary', 'apiaries', count, _ ).
noun( 'apiculture', 'apicultures', count, _ ).
noun( 'aplomb', '-', mass, _ ).
noun( 'apocalypse', 'apocalypses', count, _ ).
noun( 'apogee', 'apogees', count, _ ).
noun( 'apologetics', 'apologetics', mass, _ ).
noun( 'apologist', 'apologists', count, _ ).
noun( 'apology', 'apologies', count, _ ).
noun( 'apophthegm', 'apophthegms', count, _ ).
noun( 'apoplexy', '-', mass, _ ).
noun( 'apostasy', 'apostasies', both, _ ).
noun( 'apostate', 'apostates', count, _ ).
noun( 'apostle', 'apostles', count, _ ).
noun( 'apostrophe', 'apostrophes', count, _ ).
noun( 'apothecary', 'apothecaries', count, _ ).
noun( 'apothegm', 'apothegms', count, _ ).
noun( 'apotheosis', 'apotheoses', count, _ ).
noun( 'appanage', '-', mass, _ ).
noun( 'apparatus', 'apparatuses', both, _ ).
noun( 'apparel', '-', mass, _ ).
noun( 'apparition', 'apparitions', count, _ ).
noun( 'appeal', 'appeals', both, _ ).
noun( 'appearance', 'appearances', count, _ ).
noun( 'appeasement', '-', mass, _ ).
noun( 'appellant', 'appellants', count, _ ).
noun( 'appellation', 'appellations', count, _ ).
noun( 'appendage', 'appendages', count, _ ).
noun( 'appendectomy', 'appendectomies', count, _ ).
noun( 'appendicitis', '-', mass, _ ).
noun( 'appendix', 'appendixes', count, _ ).
noun( 'appetite', 'appetites', both, _ ).
noun( 'appetizer', 'appetizers', count, _ ).
noun( 'applause', '-', mass, _ ).
noun( 'apple', 'apples', both, _ ).
noun( 'applejack', '-', mass, _ ).
noun( 'appliance', 'appliances', count, _ ).
noun( 'applicability', '-', mass, _ ).
noun( 'applicant', 'applicants', count, _ ).
noun( 'application', 'applications', both, _ ).
noun( 'appliqu_e', '-', mass, _ ).
noun( 'appointee', 'appointees', count, _ ).
noun( 'appointment', 'appointments', both, _ ).
noun( 'apportionment', '-', mass, _ ).
noun( 'apposition', '-', mass, _ ).
noun( 'appraisal', 'appraisals', count, _ ).
noun( 'appreciation', 'appreciations', both, _ ).
noun( 'apprehension', 'apprehensions', both, _ ).
noun( 'apprentice', 'apprentices', count, _ ).
noun( 'apprenticeship', 'apprenticeships', count, _ ).
noun( 'appro', '-', mass, _ ).
noun( 'approach', 'approaches', both, _ ).
noun( 'approbation', '-', mass, _ ).
noun( 'appropriation', 'appropriations', both, _ ).
noun( 'approval', '-', mass, _ ).
noun( 'approx', '-', proper, _ ).
noun( 'approximation', 'approximations', both, _ ).
noun( 'appurtenance', 'appurtenances', count, _ ).
noun( 'apricot', 'apricots', count, _ ).
noun( 'apron', 'aprons', count, _ ).
noun( 'apse', 'apses', count, _ ).
noun( 'aptitude', 'aptitudes', both, _ ).
noun( 'aptness', '-', mass, _ ).
noun( 'aqualung', 'aqualungs', count, _ ).
noun( 'aquamarine', 'aquamarines', both, _ ).
noun( 'aquanaut', 'aquanauts', count, _ ).
noun( 'aquaplane', 'aquaplanes', count, _ ).
noun( 'aquarium', 'aquariums', count, _ ).
noun( 'aquatint', 'aquatints', both, _ ).
noun( 'aquavit', '-', mass, _ ).
noun( 'aqueduct', 'aqueducts', count, _ ).
noun( 'ar^ete', 'ar^etes', count, _ ).
noun( 'arabesque', 'arabesques', count, _ ).
noun( 'arachnid', 'arachnids', count, _ ).
noun( 'arbiter', 'arbiters', count, _ ).
noun( 'arbitrament', 'arbitraments', both, _ ).
noun( 'arbitration', 'arbitrations', both, _ ).
noun( 'arbitrator', 'arbitrators', count, _ ).
noun( 'arbour', 'arbours', count, _ ).
noun( 'arc', 'arcs', count, _ ).
noun( 'arc-lamp', 'arc-lamps', count, _ ).
noun( 'arc-light', 'arc-lights', count, _ ).
noun( 'arcade', 'arcades', count, _ ).
noun( 'arch', 'arches', count, _ ).
noun( 'archaeologist', 'archaeologists', count, _ ).
noun( 'archaeology', '-', mass, _ ).
noun( 'archaism', 'archaisms', both, _ ).
noun( 'archangel', 'archangels', count, _ ).
noun( 'archbishop', 'archbishops', count, _ ).
noun( 'archbishopric', 'archbishoprics', count, _ ).
noun( 'archdeacon', 'archdeacons', count, _ ).
noun( 'archdeaconry', 'archdeaconries', count, _ ).
noun( 'archdiocese', 'archdioceses', count, _ ).
noun( 'archduke', 'archdukes', count, _ ).
noun( 'archeology', '-', mass, _ ).
noun( 'archer', 'archers', count, _ ).
noun( 'archery', '-', mass, _ ).
noun( 'archetype', 'archetypes', count, _ ).
noun( 'archimandrite', 'archimandrites', count, _ ).
noun( 'archipelago', 'archipelagos', count, _ ).
noun( 'architect', 'architects', count, _ ).
noun( 'architecture', '-', mass, _ ).
noun( 'archivist', 'archivists', count, _ ).
noun( 'archway', 'archways', count, _ ).
noun( 'ardour', 'ardours', both, _ ).
noun( 'are', 'ares', count, _ ).
noun( 'area', 'areas', both, _ ).
noun( 'areca', 'arecas', count, _ ).
noun( 'arena', 'arenas', count, _ ).
noun( 'argent', 'argents', count, _ ).
noun( 'argon', '-', mass, _ ).
noun( 'argosy', 'argosies', count, _ ).
noun( 'argot', '-', mass, _ ).
noun( 'argument', 'arguments', both, _ ).
noun( 'argumentation', '-', mass, _ ).
noun( 'aria', 'arias', count, _ ).
noun( 'aridity', '-', mass, _ ).
noun( 'aristocracy', 'aristocracies', both, _ ).
noun( 'aristocrat', 'aristocrats', count, _ ).
noun( 'arithmetic', '-', mass, _ ).
noun( 'arithmetician', 'arithmeticians', count, _ ).
noun( 'ark', 'arks', count, _ ).
noun( 'arm', 'arms', count, _ ).
noun( 'arm-hole', 'arm-holes', count, _ ).
noun( 'armada', 'armadas', count, _ ).
noun( 'armadillo', 'armadillos', count, _ ).
noun( 'armament', 'armaments', both, _ ).
noun( 'armature', 'armatures', count, _ ).
noun( 'armband', 'armbands', count, _ ).
noun( 'armchair', 'armchairs', count, _ ).
noun( 'armful', 'armfuls', count, _ ).
noun( 'armistice', 'armistices', count, _ ).
noun( 'armlet', 'armlets', count, _ ).
noun( 'armoire', 'armoires', count, _ ).
noun( 'armour', '-', mass, _ ).
noun( 'armour-plate', 'armour-plates', count, _ ).
noun( 'armourer', 'armourers', count, _ ).
noun( 'armoury', 'armouries', count, _ ).
noun( 'armpit', 'armpits', count, _ ).
noun( 'arms-race', '-', count, _ ).
noun( 'arms-runner', 'arms-runners', count, _ ).
noun( 'army', 'armies', count, _ ).
noun( 'arnica', '-', mass, _ ).
noun( 'aroma', 'aromas', count, _ ).
noun( 'arpeggio', 'arpeggios', count, _ ).
noun( 'arquebus', 'arquebuses', count, _ ).
noun( 'arr', '-', proper, _ ).
noun( 'arrack', '-', mass, _ ).
noun( 'arraignment', 'arraignments', count, _ ).
noun( 'arrangement', 'arrangements', both, _ ).
noun( 'arras', 'arrases', count, _ ).
noun( 'array', 'arrays', count, _ ).
noun( 'arrest', 'arrests', count, _ ).
noun( 'arrester', 'arresters', count, _ ).
noun( 'arri`ere pens_ee', 'arri`ere pens_ees', count, _ ).
noun( 'arrival', 'arrivals', both, _ ).
noun( 'arrogance', '-', mass, _ ).
noun( 'arrow', 'arrows', count, _ ).
noun( 'arrowhead', 'arrowheads', count, _ ).
noun( 'arrowroot', '-', mass, _ ).
noun( 'arse', 'arses', count, _ ).
noun( 'arsehole', 'arseholes', count, _ ).
noun( 'arsenal', 'arsenals', count, _ ).
noun( 'arsenic', '-', mass, _ ).
noun( 'arson', '-', mass, _ ).
noun( 'art', 'arts', both, _ ).
noun( 'artefact', 'artefacts', count, _ ).
noun( 'arteriosclerosis', '-', mass, _ ).
noun( 'artery', 'arteries', count, _ ).
noun( 'artfulness', '-', mass, _ ).
noun( 'arthritis', '-', mass, _ ).
noun( 'artichoke', 'artichokes', count, _ ).
noun( 'article', 'articles', count, _ ).
noun( 'articulation', '-', mass, _ ).
noun( 'artifact', 'artifacts', count, _ ).
noun( 'artifice', 'artifices', both, _ ).
noun( 'artificer', 'artificers', count, _ ).
noun( 'artillery', '-', mass, _ ).
noun( 'artisan', 'artisans', count, _ ).
noun( 'artist', 'artists', count, _ ).
noun( 'artiste', 'artistes', count, _ ).
noun( 'artistry', '-', mass, _ ).
noun( 'artlessness', '-', mass, _ ).
noun( 'arum', 'arums', count, _ ).
noun( 'asap', '-', proper, _ ).
noun( 'asbestos', '-', mass, _ ).
noun( 'ascendancy', '-', mass, _ ).
noun( 'ascendant', 'ascendants', count, _ ).
noun( 'ascendency', '-', mass, _ ).
noun( 'ascendent', 'ascendents', count, _ ).
noun( 'ascension', 'ascensions', count, _ ).
noun( 'ascent', 'ascents', count, _ ).
noun( 'ascetic', 'ascetics', count, _ ).
noun( 'asceticism', '-', mass, _ ).
noun( 'ascription', 'ascriptions', count, _ ).
noun( 'asdic', 'asdics', count, _ ).
noun( 'asepsis', '-', mass, _ ).
noun( 'asexuality', '-', mass, _ ).
noun( 'ash', 'ashes', both, _ ).
noun( 'ash-bin', 'ash-bins', count, _ ).
noun( 'ash-can', 'ash-cans', count, _ ).
noun( 'ash-key', 'ash-keys', count, _ ).
noun( 'ash-pan', 'ash-pans', count, _ ).
noun( 'ashtray', 'ashtrays', count, _ ).
noun( 'aside', 'asides', count, _ ).
noun( 'asking', '-', count, _ ).
noun( 'asp', 'asps', count, _ ).
noun( 'asparagus', '-', mass, _ ).
noun( 'aspect', 'aspects', count, _ ).
noun( 'aspen', 'aspens', count, _ ).
noun( 'asperity', 'asperities', both, _ ).
noun( 'aspersion', 'aspersions', count, _ ).
noun( 'asphalt', '-', mass, _ ).
noun( 'asphodel', 'asphodels', count, _ ).
noun( 'asphyxia', '-', mass, _ ).
noun( 'asphyxiation', '-', mass, _ ).
noun( 'aspic', '-', mass, _ ).
noun( 'aspidistra', 'aspidistras', count, _ ).
noun( 'aspirant', 'aspirants', count, _ ).
noun( 'aspirate', 'aspirates', count, _ ).
noun( 'aspiration', 'aspirations', both, _ ).
noun( 'aspirin', 'aspirins', both, _ ).
noun( 'ass', 'asses', count, _ ).
noun( 'assagai', 'assagais', count, _ ).
noun( 'assailant', 'assailants', count, _ ).
noun( 'assassin', 'assassins', count, _ ).
noun( 'assassination', 'assassinations', both, _ ).
noun( 'assault', 'assaults', count, _ ).
noun( 'assay', 'assays', count, _ ).
noun( 'assegai', 'assegais', count, _ ).
noun( 'assemblage', 'assemblages', both, _ ).
noun( 'assembly', 'assemblies', count, _ ).
noun( 'assent', 'assents', count, _ ).
noun( 'assertion', 'assertions', both, _ ).
noun( 'assessment', 'assessments', both, _ ).
noun( 'assessor', 'assessors', count, _ ).
noun( 'asset', 'assets', count, _ ).
noun( 'asseveration', 'asseverations', count, _ ).
noun( 'asshole', 'assholes', count, _ ).
noun( 'assiduity', 'assiduities', both, _ ).
noun( 'assignation', 'assignations', count, _ ).
noun( 'assignment', 'assignments', both, _ ).
noun( 'assimilation', '-', mass, _ ).
noun( 'assistance', '-', mass, _ ).
noun( 'assistant', 'assistants', count, _ ).
noun( 'assize', '-', mass, _ ).
noun( 'assoc', '-', count, _ ).
noun( 'associate', 'associates', count, _ ).
noun( 'association', 'associations', both, _ ).
noun( 'assonance', 'assonances', count, _ ).
noun( 'assortment', 'assortments', count, _ ).
noun( 'asst', '-', count, _ ).
noun( 'assumption', 'assumptions', count, _ ).
noun( 'assurance', 'assurances', both, _ ).
noun( 'assuredness', '-', mass, _ ).
noun( 'aster', 'asters', count, _ ).
noun( 'asterisk', 'asterisks', count, _ ).
noun( 'asteroid', 'asteroids', count, _ ).
noun( 'asthma', '-', mass, _ ).
noun( 'astigmatism', '-', mass, _ ).
noun( 'astonishment', '-', mass, _ ).
noun( 'astrakhan', '-', mass, _ ).
noun( 'astringency', '-', mass, _ ).
noun( 'astringent', 'astringents', count, _ ).
noun( 'astrodome', 'astrodomes', count, _ ).
noun( 'astrolabe', 'astrolabes', count, _ ).
noun( 'astrologer', 'astrologers', count, _ ).
noun( 'astrology', '-', mass, _ ).
noun( 'astronaut', 'astronauts', count, _ ).
noun( 'astronautics', 'astronautics', mass, _ ).
noun( 'astronomer', 'astronomers', count, _ ).
noun( 'astronomy', '-', mass, _ ).
noun( 'astrophysics', 'astrophysics', mass, _ ).
noun( 'astuteness', '-', mass, _ ).
noun( 'asylum', 'asylums', both, _ ).
noun( 'asymmetry', '-', mass, _ ).
noun( 'asymptote', 'asymptotes', count, _ ).
noun( 'at-home', 'at-homes', count, _ ).
noun( 'atabrine', '-', mass, _ ).
noun( 'atavism', 'atavisms', count, _ ).
noun( 'atelier', 'ateliers', count, _ ).
noun( 'atheism', '-', mass, _ ).
noun( 'atheist', 'atheists', count, _ ).
noun( 'athlete', 'athletes', count, _ ).
noun( 'athletics', 'athletics', mass, _ ).
noun( 'atlas', 'atlases', count, _ ).
noun( 'atmosphere', 'atmospheres', both, _ ).
noun( 'atmospherics', 'atmospherics', mass, _ ).
noun( 'atoll', 'atolls', count, _ ).
noun( 'atom', 'atoms', count, _ ).
noun( 'atomizer', 'atomizers', count, _ ).
noun( 'atonality', '-', mass, _ ).
noun( 'atonement', '-', mass, _ ).
noun( 'atrocity', 'atrocities', both, _ ).
noun( 'atrophy', '-', mass, _ ).
noun( 'attach_e', 'attach_es', count, _ ).
noun( 'attachment', 'attachments', both, _ ).
noun( 'attack', 'attacks', both, _ ).
noun( 'attacker', 'attackers', count, _ ).
noun( 'attainder', 'attainders', count, _ ).
noun( 'attainment', 'attainments', both, _ ).
noun( 'attar', '-', mass, _ ).
noun( 'attempt', 'attempts', count, _ ).
noun( 'attendance', 'attendances', both, _ ).
noun( 'attendant', 'attendants', count, _ ).
noun( 'attention', 'attentions', both, _ ).
noun( 'attentiveness', '-', mass, _ ).
noun( 'attenuation', '-', mass, _ ).
noun( 'attic', 'attics', count, _ ).
noun( 'attire', '-', mass, _ ).
noun( 'attitude', 'attitudes', count, _ ).
noun( 'attorney', 'attorneys', count, _ ).
noun( 'attraction', 'attractions', both, _ ).
noun( 'attribute', 'attributes', count, _ ).
noun( 'attribution', 'attributions', both, _ ).
noun( 'attrition', '-', mass, _ ).
noun( 'au pair', 'au pairs', count, _ ).
noun( 'aubergine', 'aubergines', count, _ ).
noun( 'aubrietia', 'aubrietias', count, _ ).
noun( 'auction', 'auctions', both, _ ).
noun( 'auctioneer', 'auctioneers', count, _ ).
noun( 'audacity', '-', mass, _ ).
noun( 'audibility', '-', mass, _ ).
noun( 'audience', 'audiences', count, _ ).
noun( 'audit', 'audits', count, _ ).
noun( 'audition', 'auditions', both, _ ).
noun( 'auditor', 'auditors', count, _ ).
noun( 'auditorium', 'auditoriums', count, _ ).
noun( 'auger', 'augers', count, _ ).
noun( 'aught', '-', mass, _ ).
noun( 'augmentation', 'augmentations', both, _ ).
noun( 'augur', 'augurs', count, _ ).
noun( 'augury', 'auguries', count, _ ).
noun( 'auk', 'auks', count, _ ).
noun( 'auld lang syne', '-', count, _ ).
noun( 'aunt', 'aunts', count, _ ).
noun( 'auntie', 'aunties', count, _ ).
noun( 'aunty', 'aunties', count, _ ).
noun( 'aura', 'auras', count, _ ).
noun( 'aureole', 'aureoles', count, _ ).
noun( 'auricle', 'auricles', count, _ ).
noun( 'aurora', 'auroras', count, _ ).
noun( 'austerity', 'austerities', both, _ ).
noun( 'autarchy', 'autarchies', both, _ ).
noun( 'autarky', '-', mass, _ ).
noun( 'authentication', '-', mass, _ ).
noun( 'authenticity', '-', mass, _ ).
noun( 'author', 'authors', count, _ ).
noun( 'authoress', 'authoresses', count, _ ).
noun( 'authoritarian', 'authoritarians', count, _ ).
noun( 'authoritarianism', '-', mass, _ ).
noun( 'authority', 'authorities', both, _ ).
noun( 'authorization', '-', mass, _ ).
noun( 'authorship', '-', mass, _ ).
noun( 'autism', '-', mass, _ ).
noun( 'auto', 'autos', count, _ ).
noun( 'auto-changer', 'auto-changers', count, _ ).
noun( 'auto-da-f_e', '-', count, _ ).
noun( 'autobahn', 'autobahns', count, _ ).
noun( 'autobiography', 'autobiographies', both, _ ).
noun( 'autocracy', 'autocracies', both, _ ).
noun( 'autocrat', 'autocrats', count, _ ).
noun( 'autogiro', 'autogiros', count, _ ).
noun( 'autograph', 'autographs', count, _ ).
noun( 'autogyro', 'autogyros', count, _ ).
noun( 'automat', 'automats', count, _ ).
noun( 'automatic', 'automatics', count, _ ).
noun( 'automation', '-', mass, _ ).
noun( 'automaton', 'automatons', count, _ ).
noun( 'automobile', 'automobiles', count, _ ).
noun( 'autonomy', 'autonomies', both, _ ).
noun( 'autopsy', 'autopsies', count, _ ).
noun( 'autostrada', 'autostradas', count, _ ).
noun( 'autumn', 'autumns', both, _ ).
noun( 'auxiliary', 'auxiliaries', count, _ ).
noun( 'avail', '-', mass, _ ).
noun( 'availability', '-', mass, _ ).
noun( 'avalanche', 'avalanches', count, _ ).
noun( 'avant-garde', '-', count, _ ).
noun( 'avarice', '-', mass, _ ).
noun( 'avatar', 'avatars', count, _ ).
noun( 'avenger', 'avengers', count, _ ).
noun( 'avenue', 'avenues', count, _ ).
noun( 'average', 'averages', both, _ ).
noun( 'aversion', 'aversions', both, _ ).
noun( 'aviary', 'aviaries', count, _ ).
noun( 'aviation', '-', mass, _ ).
noun( 'aviator', 'aviators', count, _ ).
noun( 'avidity', '-', mass, _ ).
noun( 'avocado', 'avocados', count, _ ).
noun( 'avocation', 'avocations', count, _ ).
noun( 'avoidance', '-', mass, _ ).
noun( 'avoirdupois', '-', mass, _ ).
noun( 'avowal', 'avowals', both, _ ).
noun( 'awakening', 'awakenings', count, _ ).
noun( 'award', 'awards', count, _ ).
noun( 'awareness', '-', mass, _ ).
noun( 'awe', '-', mass, _ ).
noun( 'awkwardness', '-', mass, _ ).
noun( 'awl', 'awls', count, _ ).
noun( 'awning', 'awnings', count, _ ).
noun( 'ax', 'axes', count, _ ).
noun( 'axe', 'axes', count, _ ).
noun( 'axiom', 'axioms', count, _ ).
noun( 'axis', 'axes', count, _ ).
noun( 'axle', 'axles', count, _ ).
noun( 'ayah', 'ayahs', count, _ ).
noun( 'azalea', 'azaleas', count, _ ).
noun( 'azimuth', 'azimuths', count, _ ).
noun( 'azure', 'azures', count, _ ).
noun( 'b', '-', count, _ ).
noun( 'b^ete noire', 'b^etes noires', count, _ ).
noun( 'baa', 'baas', count, _ ).
noun( 'baa-lamb', 'baa-lambs', count, _ ).
noun( 'baas', '-', count, _ ).
noun( 'babble', '-', mass, _ ).
noun( 'babbler', 'babblers', count, _ ).
noun( 'babe', 'babes', count, _ ).
noun( 'babel', 'babels', count, _ ).
noun( 'baboo', 'baboos', count, _ ).
noun( 'baboon', 'baboons', count, _ ).
noun( 'babu', 'babus', count, _ ).
noun( 'baby', 'babies', count, _ ).
noun( 'baby-farmer', 'baby-farmers', count, _ ).
noun( 'baby-minder', 'baby-minders', count, _ ).
noun( 'baby-talk', '-', mass, _ ).
noun( 'babyhood', '-', count, _ ).
noun( 'babysitter', 'babysitters', count, _ ).
noun( 'babysitting', '-', mass, _ ).
noun( 'baccalaureate', 'baccalaureates', count, _ ).
noun( 'baccarat', '-', mass, _ ).
noun( 'bacchanal', 'bacchanals', count, _ ).
noun( 'baccy', '-', mass, _ ).
noun( 'bachelor', 'bachelors', count, _ ).
noun( 'bacillus', 'bacilli', count, _ ).
noun( 'back', 'backs', count, _ ).
noun( 'back-down', 'back-downs', count, _ ).
noun( 'back-formation', 'back-formations', both, _ ).
noun( 'back-up', 'back-ups', count, _ ).
noun( 'backache', 'backaches', both, _ ).
noun( 'backband', 'backbands', count, _ ).
noun( 'backbench', 'backbenches', count, _ ).
noun( 'backbencher', 'backbenchers', count, _ ).
noun( 'backbiter', 'backbiters', count, _ ).
noun( 'backboard', 'backboards', count, _ ).
noun( 'backbone', 'backbones', both, _ ).
noun( 'backchat', '-', mass, _ ).
noun( 'backcloth', 'backcloths', count, _ ).
noun( 'backdoor', 'backdoors', count, _ ).
noun( 'backdrop', 'backdrops', count, _ ).
noun( 'backer', 'backers', count, _ ).
noun( 'backfire', 'backfires', count, _ ).
noun( 'backgammon', '-', mass, _ ).
noun( 'background', 'backgrounds', count, _ ).
noun( 'backing', 'backings', both, _ ).
noun( 'backlash', '-', mass, _ ).
noun( 'backlog', 'backlogs', count, _ ).
noun( 'backroom', 'backrooms', count, _ ).
noun( 'backscratcher', 'backscratchers', count, _ ).
noun( 'backseat', 'backseats', count, _ ).
noun( 'backsheesh', 'backsheesh', both, _ ).
noun( 'backside', 'backsides', count, _ ).
noun( 'backstroke', 'backstrokes', both, _ ).
noun( 'backsword', 'backswords', count, _ ).
noun( 'backtalk', '-', mass, _ ).
noun( 'backwash', '-', mass, _ ).
noun( 'backwater', 'backwaters', count, _ ).
noun( 'backwoodsman', 'backwoodsmen', count, _ ).
noun( 'bacon', '-', mass, _ ).
noun( 'bacteriologist', 'bacteriologists', count, _ ).
noun( 'bacteriology', '-', mass, _ ).
noun( 'bacterium', 'bacteria', count, _ ).
noun( 'bad', '-', mass, _ ).
noun( 'badge', 'badges', count, _ ).
noun( 'badger', 'badgers', count, _ ).
noun( 'badinage', '-', mass, _ ).
noun( 'badminton', '-', mass, _ ).
noun( 'badness', '-', mass, _ ).
noun( 'baffle', 'baffles', count, _ ).
noun( 'bag', 'bags', count, _ ).
noun( 'bagatelle', 'bagatelles', both, _ ).
noun( 'baggage', '-', mass, _ ).
noun( 'bagnio', 'bagnios', count, _ ).
noun( 'bagpipe', 'bagpipes', count, _ ).
noun( 'bail', 'bails', both, _ ).
noun( 'bailee', 'bailees', count, _ ).
noun( 'bailey', 'baileys', count, _ ).
noun( 'bailiff', 'bailiffs', count, _ ).
noun( 'bailment', 'bailments', count, _ ).
noun( 'bailor', 'bailors', count, _ ).
noun( 'bairn', 'bairns', count, _ ).
noun( 'bait', 'baits', count, _ ).
noun( 'baize', '-', mass, _ ).
noun( 'bakelite', '-', mass, _ ).
noun( 'baker', 'bakers', count, _ ).
noun( 'bakery', 'bakeries', count, _ ).
noun( 'baking-powder', 'baking-powders', count, _ ).
noun( 'baksheesh', 'baksheesh', both, _ ).
noun( 'balaclava', 'balaclavas', count, _ ).
noun( 'balalaika', 'balalaikas', count, _ ).
noun( 'balance', 'balances', both, _ ).
noun( 'balance-sheet', 'balance-sheets', count, _ ).
noun( 'balance-wheel', 'balance-wheels', count, _ ).
noun( 'balcony', 'balconies', count, _ ).
noun( 'bald-head', 'bald-heads', count, _ ).
noun( 'bald-pate', 'bald-pates', count, _ ).
noun( 'balderdash', '-', mass, _ ).
noun( 'baldness', '-', mass, _ ).
noun( 'baldric', 'baldrics', count, _ ).
noun( 'bale', 'bales', count, _ ).
noun( 'balk', 'balks', count, _ ).
noun( 'ball', 'balls', count, _ ).
noun( 'ball-cartridge', 'ball-cartridges', count, _ ).
noun( 'ball-dress', 'ball-dresses', count, _ ).
noun( 'ballad', 'ballads', count, _ ).
noun( 'ballade', 'ballades', count, _ ).
noun( 'ballast', '-', mass, _ ).
noun( 'ballbearing', 'ballbearings', count, _ ).
noun( 'ballcock', 'ballcocks', count, _ ).
noun( 'ballerina', 'ballerinas', count, _ ).
noun( 'ballet', 'ballets', both, _ ).
noun( 'ballet-dancer', 'ballet-dancers', count, _ ).
noun( 'ballet-skirt', 'ballet-skirts', count, _ ).
noun( 'ballistics', 'ballistics', mass, _ ).
noun( 'ballock', 'ballocks', count, _ ).
noun( 'balloon', 'balloons', count, _ ).
noun( 'balloonist', 'balloonists', count, _ ).
noun( 'ballot', 'ballots', both, _ ).
noun( 'ballot-box', 'ballot-boxes', count, _ ).
noun( 'ballpen', 'ballpens', count, _ ).
noun( 'ballpoint', 'ballpoints', count, _ ).
noun( 'ballpoint-pen', 'ballpoint-pens', count, _ ).
noun( 'ballroom', 'ballrooms', count, _ ).
noun( 'balls-up', 'balls-ups', count, _ ).
noun( 'ballyhoo', '-', mass, _ ).
noun( 'balm', '-', mass, _ ).
noun( 'baloney', '-', mass, _ ).
noun( 'balsa', 'balsas', both, _ ).
noun( 'balsam', 'balsams', count, _ ).
noun( 'baluster', 'balusters', count, _ ).
noun( 'balustrade', 'balustrades', count, _ ).
noun( 'bambino', 'bambinos', count, _ ).
noun( 'bamboo', 'bamboos', both, _ ).
noun( 'ban', 'bans', count, _ ).
noun( 'banality', 'banalities', both, _ ).
noun( 'banana', 'bananas', count, _ ).
noun( 'band', 'bands', count, _ ).
noun( 'band-saw', 'band-saws', count, _ ).
noun( 'bandage', 'bandages', count, _ ).
noun( 'bandanna', 'bandannas', count, _ ).
noun( 'bandbox', 'bandboxes', count, _ ).
noun( 'bandeau', 'bandeaux', count, _ ).
noun( 'bandit', 'bandits', count, _ ).
noun( 'banditry', '-', mass, _ ).
noun( 'bandleader', 'bandleaders', count, _ ).
noun( 'bandmaster', 'bandmasters', count, _ ).
noun( 'bandoleer', 'bandoleers', count, _ ).
noun( 'bandolier', 'bandoliers', count, _ ).
noun( 'bandsman', 'bandsmen', count, _ ).
noun( 'bandstand', 'bandstands', count, _ ).
noun( 'bandwagon', 'bandwagons', count, _ ).
noun( 'bane', '-', mass, _ ).
noun( 'bang', 'bangs', both, _ ).
noun( 'banger', 'bangers', count, _ ).
noun( 'bangle', 'bangles', count, _ ).
noun( 'banian', 'banians', count, _ ).
noun( 'banian-tree', 'banian-trees', count, _ ).
noun( 'banishment', '-', mass, _ ).
noun( 'banister', 'banisters', count, _ ).
noun( 'banjo', 'banjos', count, _ ).
noun( 'bank', 'banks', count, _ ).
noun( 'bank-bill', 'bank-bills', count, _ ).
noun( 'bank-book', 'bank-books', count, _ ).
noun( 'bank-draft', 'bank-drafts', count, _ ).
noun( 'bank-rate', 'bank-rates', count, _ ).
noun( 'banker', 'bankers', count, _ ).
noun( 'banking', '-', mass, _ ).
noun( 'banknote', 'banknotes', count, _ ).
noun( 'bankroll', 'bankrolls', count, _ ).
noun( 'bankrupt', 'bankrupts', count, _ ).
noun( 'bankruptcy', 'bankruptcies', both, _ ).
noun( 'banner', 'banners', count, _ ).
noun( 'banning-order', 'banning-orders', count, _ ).
noun( 'bannister', 'bannisters', count, _ ).
noun( 'bannock', 'bannocks', count, _ ).
noun( 'banquet', 'banquets', count, _ ).
noun( 'banshee', 'banshees', count, _ ).
noun( 'bantam', 'bantams', count, _ ).
noun( 'banter', '-', mass, _ ).
noun( 'banting', '-', mass, _ ).
noun( 'banyan', 'banyans', count, _ ).
noun( 'baobab', 'baobabs', count, _ ).
noun( 'baptism', 'baptisms', both, _ ).
noun( 'bar', 'bars', count, _ ).
noun( 'barb', 'barbs', count, _ ).
noun( 'barbarian', 'barbarians', count, _ ).
noun( 'barbarism', 'barbarisms', both, _ ).
noun( 'barbarity', 'barbarities', both, _ ).
noun( 'barbecue', 'barbecues', count, _ ).
noun( 'barbel', 'barbels', count, _ ).
noun( 'barber', 'barbers', count, _ ).
noun( 'barbican', 'barbicans', count, _ ).
noun( 'barbitone', '-', mass, _ ).
noun( 'barbiturate', 'barbiturates', both, _ ).
noun( 'barcarole', 'barcaroles', count, _ ).
noun( 'barcarolle', 'barcarolles', count, _ ).
noun( 'bard', 'bards', count, _ ).
noun( 'bardolatry', '-', mass, _ ).
noun( 'bareness', '-', mass, _ ).
noun( 'bargain', 'bargains', count, _ ).
noun( 'barge', 'barges', count, _ ).
noun( 'bargee', 'bargees', count, _ ).
noun( 'bargepole', 'bargepoles', count, _ ).
noun( 'baritone', 'baritones', count, _ ).
noun( 'barium', '-', mass, _ ).
noun( 'bark', 'barks', both, _ ).
noun( 'barker', 'barkers', count, _ ).
noun( 'barley', '-', mass, _ ).
noun( 'barley-sugar', 'barley-sugars', both, _ ).
noun( 'barley-water', '-', mass, _ ).
noun( 'barleycorn', 'barleycorns', both, _ ).
noun( 'barm', '-', mass, _ ).
noun( 'barmaid', 'barmaids', count, _ ).
noun( 'barman', 'barmen', count, _ ).
noun( 'barn', 'barns', count, _ ).
noun( 'barn-door', 'barn-doors', count, _ ).
noun( 'barnacle', 'barnacles', count, _ ).
noun( 'barnstormer', 'barnstormers', count, _ ).
noun( 'barnyard', 'barnyards', count, _ ).
noun( 'barometer', 'barometers', count, _ ).
noun( 'baron', 'barons', count, _ ).
noun( 'baronage', 'baronages', count, _ ).
noun( 'baroness', 'baronesses', count, _ ).
noun( 'baronet', 'baronets', count, _ ).
noun( 'baronetcy', 'baronetcies', count, _ ).
noun( 'barony', 'baronies', count, _ ).
noun( 'baroque', '-', count, _ ).
noun( 'barouche', 'barouches', count, _ ).
noun( 'barque', 'barques', count, _ ).
noun( 'barrack', 'barracks', count, _ ).
noun( 'barracking', '-', mass, _ ).
noun( 'barracuda', 'barracudas', count, _ ).
noun( 'barrage', 'barrages', count, _ ).
noun( 'barrel', 'barrels', count, _ ).
noun( 'barrenness', '-', mass, _ ).
noun( 'barricade', 'barricades', count, _ ).
noun( 'barrier', 'barriers', count, _ ).
noun( 'barrister', 'barristers', count, _ ).
noun( 'barrow', 'barrows', count, _ ).
noun( 'barrow-boy', 'barrow-boys', count, _ ).
noun( 'barrow-man', 'barrow-men', count, _ ).
noun( 'bartender', 'bartenders', count, _ ).
noun( 'barter', '-', mass, _ ).
noun( 'barterer', 'barterers', count, _ ).
noun( 'bas-relief', 'bas-reliefs', both, _ ).
noun( 'basalt', '-', mass, _ ).
noun( 'bascule', 'bascules', count, _ ).
noun( 'base', 'bases', count, _ ).
noun( 'baseball', 'baseballs', both, _ ).
noun( 'baseboard', 'baseboards', count, _ ).
noun( 'basement', 'basements', count, _ ).
noun( 'bash', 'bashes', count, _ ).
noun( 'basil', '-', mass, _ ).
noun( 'basilica', 'basilicas', count, _ ).
noun( 'basilisk', 'basilisks', count, _ ).
noun( 'basin', 'basins', count, _ ).
noun( 'basis', 'bases', count, _ ).
noun( 'basket', 'baskets', count, _ ).
noun( 'basketball', 'basketballs', both, _ ).
noun( 'basketry', '-', mass, _ ).
noun( 'bass', 'bass', count, _ ).
noun( 'bass', 'basses', count, _ ).
noun( 'bassinet', 'bassinets', count, _ ).
noun( 'bassoon', 'bassoons', count, _ ).
noun( 'bast', '-', mass, _ ).
noun( 'bastard', 'bastards', count, _ ).
noun( 'bastardy', 'bastardies', count, _ ).
noun( 'bastinado', '-', count, _ ).
noun( 'bastion', 'bastions', count, _ ).
noun( 'bat', 'bats', count, _ ).
noun( 'batch', 'batches', count, _ ).
noun( 'bath', 'baths', count, _ ).
noun( 'bathe', 'bathes', count, _ ).
noun( 'bather', 'bathers', count, _ ).
noun( 'bathing', 'bathings', count, _ ).
noun( 'bathing-cap', 'bathing-caps', count, _ ).
noun( 'bathing-costume', 'bathing-costumes', count, _ ).
noun( 'bathing-machine', 'bathing-machines', count, _ ).
noun( 'bathing-suit', 'bathing-suits', count, _ ).
noun( 'bathos', '-', mass, _ ).
noun( 'bathrobe', 'bathrobes', count, _ ).
noun( 'bathroom', 'bathrooms', count, _ ).
noun( 'bathtub', 'bathtubs', count, _ ).
noun( 'bathysphere', 'bathyspheres', count, _ ).
noun( 'batik', 'batiks', both, _ ).
noun( 'batiste', '-', mass, _ ).
noun( 'batman', 'batmen', count, _ ).
noun( 'baton', 'batons', count, _ ).
noun( 'batsman', 'batsmen', count, _ ).
noun( 'battalion', 'battalions', count, _ ).
noun( 'batten', 'battens', count, _ ).
noun( 'batter', '-', mass, _ ).
noun( 'battery', 'batteries', count, _ ).
noun( 'batting', '-', mass, _ ).
noun( 'battle', 'battles', both, _ ).
noun( 'battle-axe', 'battle-axes', count, _ ).
noun( 'battle-cruiser', 'battle-cruisers', count, _ ).
noun( 'battledore', 'battledores', count, _ ).
noun( 'battledress', '-', count, _ ).
noun( 'battlefield', 'battlefields', count, _ ).
noun( 'battleground', 'battlegrounds', count, _ ).
noun( 'battleship', 'battleships', count, _ ).
noun( 'battue', 'battues', count, _ ).
noun( 'bauble', 'baubles', count, _ ).
noun( 'baulk', 'baulks', count, _ ).
noun( 'bauxite', '-', mass, _ ).
noun( 'bawbee', 'bawbees', count, _ ).
noun( 'bawd', 'bawds', count, _ ).
noun( 'bawdy', '-', mass, _ ).
noun( 'bay', 'bays', count, _ ).
noun( 'bay-wreath', 'bay-wreaths', count, _ ).
noun( 'bayonet', 'bayonets', count, _ ).
noun( 'bayou', 'bayous', count, _ ).
noun( 'bazaar', 'bazaars', count, _ ).
noun( 'bazooka', 'bazookas', count, _ ).
noun( 'beach', 'beaches', count, _ ).
noun( 'beachcomber', 'beachcombers', count, _ ).
noun( 'beachhead', 'beachheads', count, _ ).
noun( 'beachwear', '-', mass, _ ).
noun( 'beacon', 'beacons', count, _ ).
noun( 'beacon-fire', 'beacon-fires', count, _ ).
noun( 'beacon-light', 'beacon-lights', count, _ ).
noun( 'bead', 'beads', count, _ ).
noun( 'beading', '-', mass, _ ).
noun( 'beadle', 'beadles', count, _ ).
noun( 'beagle', 'beagles', count, _ ).
noun( 'beagling', '-', mass, _ ).
noun( 'beak', 'beaks', count, _ ).
noun( 'beaker', 'beakers', count, _ ).
noun( 'beam', 'beams', count, _ ).
noun( 'bean', 'beans', count, _ ).
noun( 'beanfeast', 'beanfeasts', count, _ ).
noun( 'beano', 'beanos', count, _ ).
noun( 'beanstalk', 'beanstalks', count, _ ).
noun( 'bear', 'bears', count, _ ).
noun( 'beard', 'beards', count, _ ).
noun( 'bearer', 'bearers', count, _ ).
noun( 'bearing', 'bearings', both, _ ).
noun( 'bearskin', 'bearskins', count, _ ).
noun( 'beast', 'beasts', count, _ ).
noun( 'beastliness', '-', mass, _ ).
noun( 'beat', 'beats', count, _ ).
noun( 'beater', 'beaters', count, _ ).
noun( 'beatification', 'beatifications', count, _ ).
noun( 'beating', 'beatings', count, _ ).
noun( 'beatitude', 'beatitudes', both, _ ).
noun( 'beatnik', 'beatniks', count, _ ).
noun( 'beau', 'beaux', count, _ ).
noun( 'beautician', 'beauticians', count, _ ).
noun( 'beauty', 'beauties', both, _ ).
noun( 'beauty-parlour', 'beauty-parlours', count, _ ).
noun( 'beauty-salon', 'beauty-salons', count, _ ).
noun( 'beauty-sleep', '-', mass, _ ).
noun( 'beauty-spot', 'beauty-spots', count, _ ).
noun( 'beaver', 'beavers', both, _ ).
noun( 'beck', 'becks', count, _ ).
noun( 'bed', 'beds', count, _ ).
noun( 'bedbug', 'bedbugs', count, _ ).
noun( 'bedding', '-', mass, _ ).
noun( 'bedevilment', 'bedevilments', count, _ ).
noun( 'bedfellow', 'bedfellows', count, _ ).
noun( 'bedlam', 'bedlams', count, _ ).
noun( 'bedpan', 'bedpans', count, _ ).
noun( 'bedpost', 'bedposts', count, _ ).
noun( 'bedrock', 'bedrocks', count, _ ).
noun( 'bedroll', 'bedrolls', count, _ ).
noun( 'bedroom', 'bedrooms', count, _ ).
noun( 'bedside', 'bedsides', count, _ ).
noun( 'bedsit', 'bedsits', count, _ ).
noun( 'bedsitter', 'bedsitters', count, _ ).
noun( 'bedsitting-room', 'bedsitting-rooms', count, _ ).
noun( 'bedsore', 'bedsores', count, _ ).
noun( 'bedspread', 'bedspreads', count, _ ).
noun( 'bedstead', 'bedsteads', count, _ ).
noun( 'bedtime', 'bedtimes', count, _ ).
noun( 'bee', 'bees', count, _ ).
noun( 'beech', 'beeches', both, _ ).
noun( 'beef', '-', both, _ ).
noun( 'beefeater', 'beefeaters', count, _ ).
noun( 'beefsteak', 'beefsteaks', count, _ ).
noun( 'beehive', 'beehives', count, _ ).
noun( 'beep', 'beeps', count, _ ).
noun( 'beer', 'beers', both, _ ).
noun( 'beeswax', '-', mass, _ ).
noun( 'beet', 'beets', count, _ ).
noun( 'beetle', 'beetles', count, _ ).
noun( 'beetroot', 'beetroots', count, _ ).
noun( 'begetter', 'begetters', count, _ ).
noun( 'beggar', 'beggars', count, _ ).
noun( 'beggarman', 'beggarmen', count, _ ).
noun( 'beggarwoman', 'beggarwomen', count, _ ).
noun( 'beggary', '-', mass, _ ).
noun( 'beginner', 'beginners', count, _ ).
noun( 'beginning', 'beginnings', count, _ ).
noun( 'begonia', 'begonias', count, _ ).
noun( 'begum', 'begums', count, _ ).
noun( 'behalf', 'behalves', count, _ ).
noun( 'behaviour', '-', mass, _ ).
noun( 'behaviourism', '-', mass, _ ).
noun( 'behaviourist', 'behaviourists', count, _ ).
noun( 'behest', 'behests', count, _ ).
noun( 'behind', 'behinds', count, _ ).
noun( 'beholder', 'beholders', count, _ ).
noun( 'beige', '-', mass, _ ).
noun( 'being', 'beings', both, _ ).
noun( 'belay', 'belays', count, _ ).
noun( 'belaying-pin', 'belaying-pins', count, _ ).
noun( 'belch', 'belches', count, _ ).
noun( 'beldam', 'beldams', count, _ ).
noun( 'beldame', 'beldames', count, _ ).
noun( 'belfry', 'belfries', count, _ ).
noun( 'belief', 'beliefs', both, _ ).
noun( 'believer', 'believers', count, _ ).
noun( 'believing', '-', mass, _ ).
noun( 'bell', 'bells', count, _ ).
noun( 'bell-buoy', 'bell-buoys', count, _ ).
noun( 'bell-flower', 'bell-flowers', count, _ ).
noun( 'bell-founder', 'bell-founders', count, _ ).
noun( 'bell-foundry', 'bell-foundries', count, _ ).
noun( 'bell-metal', '-', mass, _ ).
noun( 'bell-push', 'bell-pushes', count, _ ).
noun( 'bell-ringer', 'bell-ringers', count, _ ).
noun( 'bell-tent', 'bell-tents', count, _ ).
noun( 'belladonna', 'belladonnas', both, _ ).
noun( 'bellboy', 'bellboys', count, _ ).
noun( 'belle', 'belles', count, _ ).
noun( 'bellhop', 'bellhops', count, _ ).
noun( 'belligerency', '-', mass, _ ).
noun( 'belligerent', 'belligerents', count, _ ).
noun( 'bellwether', 'bellwethers', count, _ ).
noun( 'belly', 'bellies', count, _ ).
noun( 'bellyache', 'bellyaches', count, _ ).
noun( 'bellyflop', 'bellyflops', count, _ ).
noun( 'bellyful', 'bellyfuls', count, _ ).
noun( 'bellylaugh', 'bellylaughs', count, _ ).
noun( 'beloved', 'beloveds', count, _ ).
noun( 'belt', 'belts', count, _ ).
noun( 'belting', 'beltings', count, _ ).
noun( 'ben', 'bens', count, _ ).
noun( 'bench', 'benches', count, _ ).
noun( 'bend', 'bends', count, _ ).
noun( 'benedick', 'benedicks', count, _ ).
noun( 'benediction', 'benedictions', count, _ ).
noun( 'benefaction', 'benefactions', both, _ ).
noun( 'benefactor', 'benefactors', count, _ ).
noun( 'benefactress', 'benefactresses', count, _ ).
noun( 'benefice', 'benefices', count, _ ).
noun( 'beneficence', '-', mass, _ ).
noun( 'beneficiary', 'beneficiaries', count, _ ).
noun( 'benefit', 'benefits', both, _ ).
noun( 'benevolence', '-', mass, _ ).
noun( 'benignity', 'benignities', both, _ ).
noun( 'benison', 'benisons', count, _ ).
noun( 'bent', 'bents', count, _ ).
noun( 'benzene', '-', mass, _ ).
noun( 'benzine', '-', mass, _ ).
noun( 'benzol', '-', mass, _ ).
noun( 'bequest', 'bequests', both, _ ).
noun( 'bereavement', 'bereavements', both, _ ).
noun( 'beret', 'berets', count, _ ).
noun( 'berg', 'bergs', count, _ ).
noun( 'beri-beri', '-', mass, _ ).
noun( 'berry', 'berries', count, _ ).
noun( 'berth', 'berths', count, _ ).
noun( 'beryl', 'beryls', count, _ ).
noun( 'besieger', 'besiegers', count, _ ).
noun( 'besom', 'besoms', count, _ ).
noun( 'best-seller', 'best-sellers', count, _ ).
noun( 'bestiality', 'bestialities', both, _ ).
noun( 'bestiary', 'bestiaries', count, _ ).
noun( 'bestowal', 'bestowals', count, _ ).
noun( 'bet', 'bets', count, _ ).
noun( 'beta', 'betas', count, _ ).
noun( 'betel', 'betels', count, _ ).
noun( 'betel-nut', 'betel-nuts', count, _ ).
noun( 'bethel', 'bethels', count, _ ).
noun( 'betrayal', 'betrayals', both, _ ).
noun( 'betrayer', 'betrayers', count, _ ).
noun( 'betrothal', 'betrothals', count, _ ).
noun( 'betrothed', 'betrotheds', count, _ ).
noun( 'better', 'betters', count, _ ).
noun( 'betterment', '-', mass, _ ).
noun( 'bettor', 'bettors', count, _ ).
noun( 'bevel', 'bevels', count, _ ).
noun( 'beverage', 'beverages', count, _ ).
noun( 'bevy', 'bevies', count, _ ).
noun( 'bewilderment', '-', mass, _ ).
noun( 'bey', 'beys', count, _ ).
noun( 'bezique', '-', mass, _ ).
noun( 'bhang', 'bhangs', count, _ ).
noun( 'bias', 'biases', count, _ ).
noun( 'bib', 'bibs', count, _ ).
noun( 'bibliographer', 'bibliographers', count, _ ).
noun( 'bibliography', 'bibliographies', both, _ ).
noun( 'bibliophile', 'bibliophiles', count, _ ).
noun( 'bicarbonate', '-', mass, _ ).
noun( 'bicentenary', 'bicentenaries', count, _ ).
noun( 'bicentennial', 'bicentennials', count, _ ).
noun( 'biceps', 'biceps', count, _ ).
noun( 'bicycle', 'bicycles', count, _ ).
noun( 'bid', 'bids', count, _ ).
noun( 'bidder', 'bidders', count, _ ).
noun( 'bidding', '-', mass, _ ).
noun( 'bidet', 'bidets', count, _ ).
noun( 'biennial', 'biennials', count, _ ).
noun( 'bier', 'biers', count, _ ).
noun( 'biff', 'biffs', count, _ ).
noun( 'bifurcation', 'bifurcations', count, _ ).
noun( 'bigamist', 'bigamists', count, _ ).
noun( 'bigamy', '-', mass, _ ).
noun( 'bight', 'bights', count, _ ).
noun( 'bigot', 'bigots', count, _ ).
noun( 'bigotry', 'bigotries', both, _ ).
noun( 'bigwig', 'bigwigs', count, _ ).
noun( 'bijou', '-', count, _ ).
noun( 'bike', 'bikes', count, _ ).
noun( 'bikini', 'bikinis', count, _ ).
noun( 'bilabial', 'bilabials', count, _ ).
noun( 'bilateralism', 'bilateralisms', both, _ ).
noun( 'bilberry', 'bilberries', count, _ ).
noun( 'bile', '-', mass, _ ).
noun( 'bile-duct', 'bile-ducts', count, _ ).
noun( 'bilge', '-', mass, _ ).
noun( 'bilge-water', '-', mass, _ ).
noun( 'bilharzia', '-', mass, _ ).
noun( 'bilingual', 'bilinguals', count, _ ).
noun( 'biliousness', '-', mass, _ ).
noun( 'bill', 'bills', count, _ ).
noun( 'bill-poster', 'bill-posters', count, _ ).
noun( 'bill-sticker', 'bill-stickers', count, _ ).
noun( 'billboard', 'billboards', count, _ ).
noun( 'billet', 'billets', count, _ ).
noun( 'billet-doux', 'billets-doux', count, _ ).
noun( 'billfold', 'billfolds', count, _ ).
noun( 'billhook', 'billhooks', count, _ ).
noun( 'billiard-marker', 'billiard-markers', count, _ ).
noun( 'billiard-player', 'billiard-players', count, _ ).
noun( 'billiard-room', 'billiard-rooms', count, _ ).
noun( 'billiard-table', 'billiard-tables', count, _ ).
noun( 'billiards', 'billiards', mass, _ ).
noun( 'billingsgate', '-', mass, _ ).
noun( 'billion', 'billions', count, _ ).
noun( 'billionth', 'billionths', count, _ ).
noun( 'billow', 'billows', count, _ ).
noun( 'billy', 'billies', count, _ ).
noun( 'billy-goat', 'billy-goats', count, _ ).
noun( 'billy-ho', '-', count, _ ).
noun( 'billy-o', '-', mass, _ ).
noun( 'biltong', '-', mass, _ ).
noun( 'bimetallism', '-', mass, _ ).
noun( 'bin', 'bins', count, _ ).
noun( 'bind', 'binds', both, _ ).
noun( 'binder', 'binders', count, _ ).
noun( 'bindery', 'binderies', count, _ ).
noun( 'bindweed', '-', mass, _ ).
noun( 'bine', 'bines', count, _ ).
noun( 'binge', 'binges', count, _ ).
noun( 'bingo', '-', mass, _ ).
noun( 'binnacle', 'binnacles', count, _ ).
noun( 'biochemistry', '-', mass, _ ).
noun( 'biograph', 'biographs', count, _ ).
noun( 'biographer', 'biographers', count, _ ).
noun( 'biography', 'biographies', both, _ ).
noun( 'biologist', 'biologists', count, _ ).
noun( 'biology', '-', mass, _ ).
noun( 'bioscope', 'bioscopes', count, _ ).
noun( 'biped', 'bipeds', count, _ ).
noun( 'biplane', 'biplanes', count, _ ).
noun( 'birch', 'birches', both, _ ).
noun( 'birch-rod', 'birch-rods', count, _ ).
noun( 'bird', 'birds', count, _ ).
noun( 'bird-fancier', 'bird-fanciers', count, _ ).
noun( 'birdcage', 'birdcages', count, _ ).
noun( 'birdlime', 'birdlimes', count, _ ).
noun( 'birdnesting', '-', mass, _ ).
noun( 'birdwatcher', 'birdwatchers', count, _ ).
noun( 'biretta', 'birettas', count, _ ).
noun( 'biro', 'biros', count, _ ).
noun( 'birth', 'births', both, _ ).
noun( 'birth-control', '-', mass, _ ).
noun( 'birthday', 'birthdays', count, _ ).
noun( 'birthmark', 'birthmarks', count, _ ).
noun( 'birthplace', 'birthplaces', count, _ ).
noun( 'birthrate', 'birthrates', count, _ ).
noun( 'birthright', 'birthrights', count, _ ).
noun( 'biscuit', 'biscuits', count, _ ).
noun( 'bisection', '-', mass, _ ).
noun( 'bisexual', 'bisexuals', count, _ ).
noun( 'bisexuality', '-', mass, _ ).
noun( 'bishop', 'bishops', count, _ ).
noun( 'bishopric', 'bishoprics', count, _ ).
noun( 'bismuth', '-', mass, _ ).
noun( 'bison', 'bison', count, _ ).
noun( 'bistro', 'bistros', count, _ ).
noun( 'bit', 'bits', count, _ ).
noun( 'bitch', 'bitches', count, _ ).
noun( 'bite', 'bites', both, _ ).
noun( 'bitter', 'bitters', both, _ ).
noun( 'bittern', 'bitterns', count, _ ).
noun( 'bitterness', '-', mass, _ ).
noun( 'bitumen', '-', mass, _ ).
noun( 'bivalve', 'bivalves', count, _ ).
noun( 'bivouac', 'bivouacs', count, _ ).
noun( 'biz', '-', mass, _ ).
noun( 'blabbermouth', 'blabbermouths', count, _ ).
noun( 'black', 'blacks', both, _ ).
noun( 'black-beetle', 'black-beetles', count, _ ).
noun( 'black-lead', '-', mass, _ ).
noun( 'blackamoor', 'blackamoors', count, _ ).
noun( 'blackberry', 'blackberries', count, _ ).
noun( 'blackbird', 'blackbirds', count, _ ).
noun( 'blackboard', 'blackboards', count, _ ).
noun( 'blackcurrant', 'blackcurrants', count, _ ).
noun( 'blackguard', 'blackguards', count, _ ).
noun( 'blackhead', 'blackheads', count, _ ).
noun( 'blacking', '-', mass, _ ).
noun( 'blackleg', 'blacklegs', count, _ ).
noun( 'blacklist', 'blacklists', count, _ ).
noun( 'blackmail', '-', mass, _ ).
noun( 'blackmailer', 'blackmailers', count, _ ).
noun( 'blackness', '-', mass, _ ).
noun( 'blackout', 'blackouts', count, _ ).
noun( 'blacksmith', 'blacksmiths', count, _ ).
noun( 'blackthorn', 'blackthorns', count, _ ).
noun( 'bladder', 'bladders', count, _ ).
noun( 'blade', 'blades', count, _ ).
noun( 'blaeberry', 'blaeberries', count, _ ).
noun( 'blah', '-', mass, _ ).
noun( 'blame', '-', mass, _ ).
noun( 'blancmange', 'blancmanges', both, _ ).
noun( 'blandishment', 'blandishments', count, _ ).
noun( 'blandness', '-', mass, _ ).
noun( 'blank', 'blanks', both, _ ).
noun( 'blanket', 'blankets', count, _ ).
noun( 'blare', '-', mass, _ ).
noun( 'blarney', '-', mass, _ ).
noun( 'blasphemer', 'blasphemers', count, _ ).
noun( 'blasphemy', 'blasphemies', both, _ ).
noun( 'blast', 'blasts', both, _ ).
noun( 'blast-furnace', 'blast-furnaces', count, _ ).
noun( 'blast-off', 'blast-offs', count, _ ).
noun( 'blather', 'blathers', both, _ ).
noun( 'blaze', 'blazes', count, _ ).
noun( 'blazer', 'blazers', count, _ ).
noun( 'blazon', 'blazons', count, _ ).
noun( 'blazonry', 'blazonries', count, _ ).
noun( 'bleach', 'bleaches', count, _ ).
noun( 'bleaching-powder', 'bleaching-powders', count, _ ).
noun( 'bleat', 'bleats', count, _ ).
noun( 'bleep', 'bleeps', count, _ ).
noun( 'blemish', 'blemishes', both, _ ).
noun( 'blend', 'blends', count, _ ).
noun( 'blessedness', '-', mass, _ ).
noun( 'blessing', 'blessings', count, _ ).
noun( 'blether', '-', mass, _ ).
noun( 'blight', 'blights', both, _ ).
noun( 'blighter', 'blighters', count, _ ).
noun( 'blimp', 'blimps', count, _ ).
noun( 'blind', 'blinds', count, _ ).
noun( 'blindfold', 'blindfolds', count, _ ).
noun( 'blindman\'s buff', '-', mass, _ ).
noun( 'blindness', '-', mass, _ ).
noun( 'blink', 'blinks', count, _ ).
noun( 'blip', 'blips', count, _ ).
noun( 'bliss', '-', mass, _ ).
noun( 'blister', 'blisters', count, _ ).
noun( 'blitz', 'blitzes', count, _ ).
noun( 'blizzard', 'blizzards', count, _ ).
noun( 'bloater', 'bloaters', count, _ ).
noun( 'blob', 'blobs', count, _ ).
noun( 'bloc', 'blocs', count, _ ).
noun( 'block', 'blocks', count, _ ).
noun( 'blockade', 'blockades', count, _ ).
noun( 'blockade-runner', 'blockade-runners', count, _ ).
noun( 'blockage', 'blockages', count, _ ).
noun( 'blockbuster', 'blockbusters', count, _ ).
noun( 'blockhead', 'blockheads', count, _ ).
noun( 'blockhouse', 'blockhouses', count, _ ).
noun( 'bloke', 'blokes', count, _ ).
noun( 'blond', 'blonds', count, _ ).
noun( 'blonde', 'blondes', count, _ ).
noun( 'blood', 'bloods', both, _ ).
noun( 'blood-bath', 'blood-baths', count, _ ).
noun( 'blood-donor', 'blood-donors', count, _ ).
noun( 'blood-group', 'blood-groups', count, _ ).
noun( 'blood-heat', '-', mass, _ ).
noun( 'blood-letting', 'blood-lettings', both, _ ).
noun( 'blood-money', '-', mass, _ ).
noun( 'blood-poisoning', '-', mass, _ ).
noun( 'blood-pressure', 'blood-pressures', both, _ ).
noun( 'blood-relation', 'blood-relations', count, _ ).
noun( 'blood-transfusion', 'blood-transfusions', count, _ ).
noun( 'blood-type', 'blood-types', count, _ ).
noun( 'blood-vessel', 'blood-vessels', count, _ ).
noun( 'bloodhound', 'bloodhounds', count, _ ).
noun( 'bloodlust', '-', mass, _ ).
noun( 'bloodshed', 'bloodsheds', count, _ ).
noun( 'bloodstain', 'bloodstains', count, _ ).
noun( 'bloodstock', 'bloodstocks', count, _ ).
noun( 'bloodsucker', 'bloodsuckers', count, _ ).
noun( 'bloodthirstiness', '-', mass, _ ).
noun( 'bloom', 'blooms', both, _ ).
noun( 'bloomer', 'bloomers', count, _ ).
noun( 'blossom', 'blossoms', both, _ ).
noun( 'blot', 'blots', count, _ ).
noun( 'blotch', 'blotches', count, _ ).
noun( 'blotter', 'blotters', count, _ ).
noun( 'blotting-paper', 'blotting-papers', count, _ ).
noun( 'blouse', 'blouses', count, _ ).
noun( 'blow', 'blows', count, _ ).
noun( 'blow-up', 'blow-ups', count, _ ).
noun( 'blowback', 'blowbacks', count, _ ).
noun( 'blower', 'blowers', count, _ ).
noun( 'blowfly', 'blowflies', count, _ ).
noun( 'blowhole', 'blowholes', count, _ ).
noun( 'blowing-up', 'blowing-ups', count, _ ).
noun( 'blowlamp', 'blowlamps', count, _ ).
noun( 'blowout', 'blowouts', count, _ ).
noun( 'blowpipe', 'blowpipes', count, _ ).
noun( 'blowtorch', 'blowtorches', count, _ ).
noun( 'blubber', '-', mass, _ ).
noun( 'bludgeon', 'bludgeons', count, _ ).
noun( 'blue', 'blues', both, _ ).
noun( 'blue-jacket', 'blue-jackets', count, _ ).
noun( 'bluebell', 'bluebells', count, _ ).
noun( 'bluebottle', 'bluebottles', count, _ ).
noun( 'blueprint', 'blueprints', count, _ ).
noun( 'bluestocking', 'bluestockings', count, _ ).
noun( 'bluff', 'bluffs', both, _ ).
noun( 'bluffer', 'bluffers', count, _ ).
noun( 'bluffness', '-', mass, _ ).
noun( 'blunder', 'blunders', count, _ ).
noun( 'blunderbuss', 'blunderbusses', count, _ ).
noun( 'blunderer', 'blunderers', count, _ ).
noun( 'bluntness', '-', mass, _ ).
noun( 'blur', 'blurs', count, _ ).
noun( 'blurb', 'blurbs', both, _ ).
noun( 'blush', 'blushes', count, _ ).
noun( 'bluster', 'blusters', both, _ ).
noun( 'bo\'sn', 'bo\'sns', count, _ ).
noun( 'bo\'sun', 'bo\'suns', count, _ ).
noun( 'boa', 'boas', count, _ ).
noun( 'boa-constrictor', 'boa-constrictors', count, _ ).
noun( 'boar', 'boars', count, _ ).
noun( 'board', 'boards', count, _ ).
noun( 'boarder', 'boarders', count, _ ).
noun( 'boarding', '-', mass, _ ).
noun( 'boarding-card', 'boarding-cards', count, _ ).
noun( 'boarding-house', 'boarding-houses', count, _ ).
noun( 'boarding-school', 'boarding-schools', count, _ ).
noun( 'boardroom', 'boardrooms', count, _ ).
noun( 'boardwalk', 'boardwalks', count, _ ).
noun( 'boast', 'boasts', count, _ ).
noun( 'boaster', 'boasters', count, _ ).
noun( 'boat', 'boats', count, _ ).
noun( 'boat-hook', 'boat-hooks', count, _ ).
noun( 'boat-house', 'boat-houses', count, _ ).
noun( 'boat-race', 'boat-races', count, _ ).
noun( 'boat-train', 'boat-trains', count, _ ).
noun( 'boater', 'boaters', count, _ ).
noun( 'boatman', 'boatmen', count, _ ).
noun( 'boatswain', 'boatswains', count, _ ).
noun( 'bob', 'bob', count, _ ).
noun( 'bobbin', 'bobbins', count, _ ).
noun( 'bobby', 'bobbies', count, _ ).
noun( 'bobby-soxer', 'bobby-soxers', count, _ ).
noun( 'bobolink', 'bobolinks', count, _ ).
noun( 'bobsled', 'bobsleds', count, _ ).
noun( 'bobsleigh', 'bobsleighs', count, _ ).
noun( 'bobtail', 'bobtails', count, _ ).
noun( 'bodice', 'bodices', count, _ ).
noun( 'boding', '-', mass, _ ).
noun( 'bodkin', 'bodkins', count, _ ).
noun( 'body', 'bodies', both, _ ).
noun( 'body-servant', 'body-servants', count, _ ).
noun( 'body-snatcher', 'body-snatchers', count, _ ).
noun( 'bodyguard', 'bodyguards', count, _ ).
noun( 'bodywork', 'bodyworks', count, _ ).
noun( 'boffin', 'boffins', count, _ ).
noun( 'bog', 'bogs', count, _ ).
noun( 'bogey', 'bogeys', count, _ ).
noun( 'bogeyman', 'bogeymen', count, _ ).
noun( 'bogie', 'bogies', count, _ ).
noun( 'bogy', 'bogies', count, _ ).
noun( 'bohemian', 'bohemians', count, _ ).
noun( 'boil', 'boils', count, _ ).
noun( 'boiler', 'boilers', count, _ ).
noun( 'boilersuit', 'boilersuits', count, _ ).
noun( 'boiling-point', 'boiling-points', count, _ ).
noun( 'boisterousness', '-', mass, _ ).
noun( 'boldness', '-', mass, _ ).
noun( 'bole', 'boles', count, _ ).
noun( 'bolero', 'boleros', count, _ ).
noun( 'boll', 'bolls', count, _ ).
noun( 'bollard', 'bollards', count, _ ).
noun( 'bollock', 'bollocks', count, _ ).
noun( 'boloney', '-', mass, _ ).
noun( 'bolster', 'bolsters', count, _ ).
noun( 'bolt', 'bolts', count, _ ).
noun( 'bolt-hole', 'bolt-holes', count, _ ).
noun( 'bomb', 'bombs', count, _ ).
noun( 'bomb-sight', 'bomb-sights', count, _ ).
noun( 'bomb-site', 'bomb-sites', count, _ ).
noun( 'bombardier', 'bombardiers', count, _ ).
noun( 'bombardment', 'bombardments', both, _ ).
noun( 'bombast', '-', mass, _ ).
noun( 'bomber', 'bombers', count, _ ).
noun( 'bombshell', 'bombshells', count, _ ).
noun( 'bon mot', 'bons mots', count, _ ).
noun( 'bonanza', 'bonanzas', count, _ ).
noun( 'bonbon', 'bonbons', count, _ ).
noun( 'bond', 'bonds', count, _ ).
noun( 'bond-holder', 'bond-holders', count, _ ).
noun( 'bondage', '-', mass, _ ).
noun( 'bone', 'bones', both, _ ).
noun( 'bone-head', 'bone-heads', count, _ ).
noun( 'bone-setter', 'bone-setters', count, _ ).
noun( 'bonemeal', 'bonemeals', count, _ ).
noun( 'boner', 'boners', count, _ ).
noun( 'boneshaker', 'boneshakers', count, _ ).
noun( 'bonfire', 'bonfires', count, _ ).
noun( 'bongo', 'bongos', count, _ ).
noun( 'bonhomie', '-', mass, _ ).
noun( 'bonito', 'bonitos', count, _ ).
noun( 'bonnet', 'bonnets', count, _ ).
noun( 'bonus', 'bonuses', count, _ ).
noun( 'boo', 'boos', count, _ ).
noun( 'boob', 'boobs', count, _ ).
noun( 'booby', 'boobies', count, _ ).
noun( 'booby-trap', 'booby-traps', count, _ ).
noun( 'boogie', 'boogies', both, _ ).
noun( 'boogie-woogie', 'boogie-woogies', both, _ ).
noun( 'book', 'books', count, _ ).
noun( 'book-end', 'book-ends', count, _ ).
noun( 'book-keeper', 'book-keepers', count, _ ).
noun( 'bookcase', 'bookcases', count, _ ).
noun( 'bookclub', 'bookclubs', count, _ ).
noun( 'bookie', 'bookies', count, _ ).
noun( 'booking', 'bookings', count, _ ).
noun( 'bookishness', '-', mass, _ ).
noun( 'bookkeeper', 'bookkeepers', count, _ ).
noun( 'bookkeeping', '-', mass, _ ).
noun( 'booklet', 'booklets', count, _ ).
noun( 'bookmaker', 'bookmakers', count, _ ).
noun( 'bookmark', 'bookmarks', count, _ ).
noun( 'bookmarker', 'bookmarkers', count, _ ).
noun( 'bookmobile', 'bookmobiles', count, _ ).
noun( 'bookseller', 'booksellers', count, _ ).
noun( 'bookshop', 'bookshops', count, _ ).
noun( 'bookstall', 'bookstalls', count, _ ).
noun( 'bookworm', 'bookworms', count, _ ).
noun( 'boom', 'booms', count, _ ).
noun( 'boomerang', 'boomerangs', count, _ ).
noun( 'boon', 'boons', count, _ ).
noun( 'boor', 'boors', count, _ ).
noun( 'boorishness', '-', mass, _ ).
noun( 'boost', 'boosts', count, _ ).
noun( 'booster', 'boosters', count, _ ).
noun( 'boot', 'boots', count, _ ).
noun( 'bootee', 'bootees', count, _ ).
noun( 'booth', 'booths', count, _ ).
noun( 'bootlace', 'bootlaces', count, _ ).
noun( 'bootlegger', 'bootleggers', count, _ ).
noun( 'booty', '-', mass, _ ).
noun( 'booze', '-', mass, _ ).
noun( 'booze-up', 'booze-ups', count, _ ).
noun( 'boozer', 'boozers', count, _ ).
noun( 'bopeep', '-', mass, _ ).
noun( 'borage', '-', mass, _ ).
noun( 'borax', '-', mass, _ ).
noun( 'border', 'borders', count, _ ).
noun( 'borderer', 'borderers', count, _ ).
noun( 'borderland', 'borderlands', count, _ ).
noun( 'borderline', 'borderlines', count, _ ).
noun( 'bore', 'bores', count, _ ).
noun( 'bore-hole', 'bore-holes', count, _ ).
noun( 'boredom', '-', mass, _ ).
noun( 'borer', 'borers', count, _ ).
noun( 'boron', '-', mass, _ ).
noun( 'borough', 'boroughs', count, _ ).
noun( 'borrower', 'borrowers', count, _ ).
noun( 'borsch', '-', mass, _ ).
noun( 'borstal', 'borstals', count, _ ).
noun( 'bortsch', '-', mass, _ ).
noun( 'borzoi', 'borzois', count, _ ).
noun( 'bosh', '-', mass, _ ).
noun( 'bosom', 'bosoms', count, _ ).
noun( 'boss', 'bosses', count, _ ).
noun( 'botanist', 'botanists', count, _ ).
noun( 'botany', '-', mass, _ ).
noun( 'botch', 'botches', count, _ ).
noun( 'botcher', 'botchers', count, _ ).
noun( 'bother', '-', mass, _ ).
noun( 'bottle', 'bottles', both, _ ).
noun( 'bottleneck', 'bottlenecks', count, _ ).
noun( 'bottom', 'bottoms', count, _ ).
noun( 'botulism', '-', mass, _ ).
noun( 'boudoir', 'boudoirs', count, _ ).
noun( 'bougainvillea', 'bougainvilleas', count, _ ).
noun( 'bough', 'boughs', count, _ ).
noun( 'bouillon', '-', mass, _ ).
noun( 'boulder', 'boulders', count, _ ).
noun( 'boulevard', 'boulevards', count, _ ).
noun( 'bounce', 'bounces', both, _ ).
noun( 'bound', 'bounds', count, _ ).
noun( 'boundary', 'boundaries', count, _ ).
noun( 'bounder', 'bounders', count, _ ).
noun( 'bounty', 'bounties', both, _ ).
noun( 'bouquet', 'bouquets', count, _ ).
noun( 'bourbon', 'bourbons', both, _ ).
noun( 'bourgeois', 'bourgeois', count, _ ).
noun( 'bourgeoisie', 'bourgeoisies', count, _ ).
noun( 'bourn', 'bourns', count, _ ).
noun( 'bourne', 'bournes', count, _ ).
noun( 'bourse', 'bourses', count, _ ).
noun( 'bout', 'bouts', count, _ ).
noun( 'boutique', 'boutiques', count, _ ).
noun( 'bovril', '-', mass, _ ).
noun( 'bow', 'bows', count, _ ).
noun( 'bow', 'bows', count, _ ).
noun( 'bow-wow', 'bow-wows', count, _ ).
noun( 'bowel', 'bowels', count, _ ).
noun( 'bower', 'bowers', count, _ ).
noun( 'bowie knife', 'bowie knives', count, _ ).
noun( 'bowing', '-', mass, _ ).
noun( 'bowl', 'bowls', count, _ ).
noun( 'bowler', 'bowlers', count, _ ).
noun( 'bowline', 'bowlines', count, _ ).
noun( 'bowling-green', 'bowling-greens', count, _ ).
noun( 'bowls', 'bowls', mass, _ ).
noun( 'bowman', 'bowmen', count, _ ).
noun( 'bowsprit', 'bowsprits', count, _ ).
noun( 'box', 'boxes', both, _ ).
noun( 'box-kite', 'box-kites', count, _ ).
noun( 'box-number', 'box-numbers', count, _ ).
noun( 'box-office', 'box-offices', count, _ ).
noun( 'boxer', 'boxers', count, _ ).
noun( 'boxful', 'boxfuls', count, _ ).
noun( 'boxing', '-', mass, _ ).
noun( 'boxing-glove', 'boxing-gloves', count, _ ).
noun( 'boxing-match', 'boxing-matches', count, _ ).
noun( 'boxwood', '-', mass, _ ).
noun( 'boy', 'boys', count, _ ).
noun( 'boycott', 'boycotts', count, _ ).
noun( 'boyfriend', 'boyfriends', count, _ ).
noun( 'boyhood', '-', mass, _ ).
noun( 'bra', 'bras', count, _ ).
noun( 'brace', 'brace', count, _ ).
noun( 'bracelet', 'bracelets', count, _ ).
noun( 'bracken', '-', mass, _ ).
noun( 'bracket', 'brackets', count, _ ).
noun( 'bract', 'bracts', count, _ ).
noun( 'brad', 'brads', count, _ ).
noun( 'bradawl', 'bradawls', count, _ ).
noun( 'brae', 'braes', count, _ ).
noun( 'braggart', 'braggarts', count, _ ).
noun( 'bragging', '-', mass, _ ).
noun( 'braid', 'braids', both, _ ).
noun( 'braille', '-', mass, _ ).
noun( 'brain', 'brains', count, _ ).
noun( 'brain-fag', '-', mass, _ ).
noun( 'brain-teaser', 'brain-teasers', count, _ ).
noun( 'brainchild', 'brainchildren', count, _ ).
noun( 'brainstorm', 'brainstorms', count, _ ).
noun( 'brainwashing', '-', mass, _ ).
noun( 'brainwave', 'brainwaves', count, _ ).
noun( 'brake', 'brakes', count, _ ).
noun( 'brakeman', 'brakemen', count, _ ).
noun( 'bramble', 'brambles', count, _ ).
noun( 'bran', '-', mass, _ ).
noun( 'branch', 'branches', count, _ ).
noun( 'brand', 'brands', count, _ ).
noun( 'branding-iron', 'branding-irons', count, _ ).
noun( 'brandy', 'brandies', both, _ ).
noun( 'brandy-ball', 'brandy-balls', count, _ ).
noun( 'brandy-snap', 'brandy-snaps', count, _ ).
noun( 'brass', 'brasses', both, _ ).
noun( 'brassard', 'brassards', count, _ ).
noun( 'brasserie', 'brasseries', count, _ ).
noun( 'brassi`ere', 'brassi`eres', count, _ ).
noun( 'brassiere', 'brassieres', count, _ ).
noun( 'brat', 'brats', count, _ ).
noun( 'bravado', 'bravados', both, _ ).
noun( 'brave', 'braves', count, _ ).
noun( 'bravery', '-', mass, _ ).
noun( 'bravo', 'bravos', count, _ ).
noun( 'brawl', 'brawls', count, _ ).
noun( 'brawler', 'brawlers', count, _ ).
noun( 'brawn', '-', mass, _ ).
noun( 'bray', 'brays', count, _ ).
noun( 'brazier', 'braziers', count, _ ).
noun( 'breach', 'breaches', count, _ ).
noun( 'bread', '-', mass, _ ).
noun( 'breadcrumb', 'breadcrumbs', count, _ ).
noun( 'breadfruit', 'breadfruit', count, _ ).
noun( 'breadline', 'breadlines', count, _ ).
noun( 'breadth', 'breadths', both, _ ).
noun( 'breadwinner', 'breadwinners', count, _ ).
noun( 'break', 'breaks', both, _ ).
noun( 'break-in', 'break-ins', count, _ ).
noun( 'break-up', 'break-ups', count, _ ).
noun( 'breakage', 'breakages', count, _ ).
noun( 'breakaway', 'breakaways', count, _ ).
noun( 'breakaways', 'breakawayss', count, _ ).
noun( 'breakdown', 'breakdowns', count, _ ).
noun( 'breaker', 'breakers', count, _ ).
noun( 'breakfast', 'breakfasts', count, _ ).
noun( 'breakthrough', 'breakthroughs', count, _ ).
noun( 'breakwater', 'breakwaters', count, _ ).
noun( 'bream', 'bream', count, _ ).
noun( 'breast', 'breasts', count, _ ).
noun( 'breast-plate', 'breast-plates', count, _ ).
noun( 'breaststroke', '-', mass, _ ).
noun( 'breastwork', 'breastworks', count, _ ).
noun( 'breath', 'breaths', both, _ ).
noun( 'breathalyser', 'breathalysers', count, _ ).
noun( 'breather', 'breathers', count, _ ).
noun( 'breathing', '-', mass, _ ).
noun( 'breathing-space', 'breathing-spaces', count, _ ).
noun( 'breech', 'breeches', count, _ ).
noun( 'breech-block', 'breech-blocks', count, _ ).
noun( 'breeches-buoy', 'breeches-buoys', count, _ ).
noun( 'breed', 'breeds', count, _ ).
noun( 'breeder', 'breeders', count, _ ).
noun( 'breeding', '-', mass, _ ).
noun( 'breeze', 'breezes', both, _ ).
noun( 'breeziness', '-', mass, _ ).
noun( 'breve', 'breves', count, _ ).
noun( 'brevet', 'brevets', count, _ ).
noun( 'breviary', 'breviaries', count, _ ).
noun( 'brevity', '-', mass, _ ).
noun( 'brew', 'brews', count, _ ).
noun( 'brewer', 'brewers', count, _ ).
noun( 'brewery', 'breweries', count, _ ).
noun( 'briar', 'briars', both, _ ).
noun( 'bribe', 'bribes', count, _ ).
noun( 'bribery', '-', mass, _ ).
noun( 'bric-a-brac', '-', mass, _ ).
noun( 'brick', 'bricks', both, _ ).
noun( 'brick-field', 'brick-fields', count, _ ).
noun( 'brickbat', 'brickbats', count, _ ).
noun( 'brickkiln', 'brickkilns', count, _ ).
noun( 'bricklayer', 'bricklayers', count, _ ).
noun( 'brickwork', 'brickworks', count, _ ).
noun( 'bridal', 'bridals', count, _ ).
noun( 'bride', 'brides', count, _ ).
noun( 'bridecake', 'bridecakes', count, _ ).
noun( 'bridegroom', 'bridegrooms', count, _ ).
noun( 'bridesmaid', 'bridesmaids', count, _ ).
noun( 'bridge', 'bridges', both, _ ).
noun( 'bridgehead', 'bridgeheads', count, _ ).
noun( 'bridle', 'bridles', count, _ ).
noun( 'bridle-path', 'bridle-paths', count, _ ).
noun( 'bridle-road', 'bridle-roads', count, _ ).
noun( 'brief', 'briefs', count, _ ).
noun( 'briefcase', 'briefcases', count, _ ).
noun( 'briefing', 'briefings', count, _ ).
noun( 'brier', 'briers', count, _ ).
noun( 'brig', 'brigs', count, _ ).
noun( 'brigade', 'brigades', count, _ ).
noun( 'brigand', 'brigands', count, _ ).
noun( 'brigantine', 'brigantines', count, _ ).
noun( 'brightness', '-', mass, _ ).
noun( 'brill', 'brill', count, _ ).
noun( 'brilliance', '-', mass, _ ).
noun( 'brilliancy', '-', mass, _ ).
noun( 'brilliantine', '-', mass, _ ).
noun( 'brim', 'brims', count, _ ).
noun( 'brimstone', '-', mass, _ ).
noun( 'brine', 'brines', both, _ ).
noun( 'brink', 'brinks', count, _ ).
noun( 'brinkmanship', '-', mass, _ ).
noun( 'brioche', 'brioches', count, _ ).
noun( 'briquet', 'briquets', count, _ ).
noun( 'briquette', 'briquettes', count, _ ).
noun( 'brisket', '-', mass, _ ).
noun( 'bristle', 'bristles', count, _ ).
noun( 'broad', 'broads', count, _ ).
noun( 'broad-mindedness', '-', mass, _ ).
noun( 'broadcast', 'broadcasts', count, _ ).
noun( 'broadcasting', '-', mass, _ ).
noun( 'broadcloth', '-', mass, _ ).
noun( 'broadness', '-', mass, _ ).
noun( 'broadsheet', 'broadsheets', count, _ ).
noun( 'broadside', 'broadsides', count, _ ).
noun( 'brocade', 'brocades', both, _ ).
noun( 'broccoli', '-', mass, _ ).
noun( 'brochure', 'brochures', count, _ ).
noun( 'brogue', 'brogues', count, _ ).
noun( 'broiler', 'broilers', count, _ ).
noun( 'broker', 'brokers', count, _ ).
noun( 'brokerage', '-', mass, _ ).
noun( 'brolly', 'brollies', count, _ ).
noun( 'bromide', 'bromides', both, _ ).
noun( 'bromine', '-', mass, _ ).
noun( 'bronchitis', '-', mass, _ ).
noun( 'bronchus', 'bronchi', count, _ ).
noun( 'bronco', 'broncos', count, _ ).
noun( 'bronze', 'bronzes', both, _ ).
noun( 'brooch', 'brooches', count, _ ).
noun( 'brood', 'broods', count, _ ).
noun( 'brood-hen', 'brood-hens', count, _ ).
noun( 'brood-mare', 'brood-mares', count, _ ).
noun( 'brook', 'brooks', count, _ ).
noun( 'broom', 'brooms', both, _ ).
noun( 'broomstick', 'broomsticks', count, _ ).
noun( 'broth', '-', mass, _ ).
noun( 'brothel', 'brothels', count, _ ).
noun( 'brother', 'brothers', count, _ ).
noun( 'brother-in-law', 'brothers-in-law', count, _ ).
noun( 'brotherhood', 'brotherhoods', both, _ ).
noun( 'brougham', 'broughams', count, _ ).
noun( 'brouhaha', 'brouhahas', count, _ ).
noun( 'brow', 'brows', count, _ ).
noun( 'brownie', 'brownies', count, _ ).
noun( 'brownstone', 'brownstones', both, _ ).
noun( 'browse', 'browses', count, _ ).
noun( 'bruin', 'bruins', count, _ ).
noun( 'bruise', 'bruises', count, _ ).
noun( 'bruiser', 'bruisers', count, _ ).
noun( 'brunch', 'brunches', count, _ ).
noun( 'brunette', 'brunettes', count, _ ).
noun( 'brunt', 'brunts', count, _ ).
noun( 'brush', 'brushes', both, _ ).
noun( 'brush-off', 'brush-offs', count, _ ).
noun( 'brush-up', 'brush-ups', count, _ ).
noun( 'brushwood', '-', mass, _ ).
noun( 'brushwork', 'brushworks', count, _ ).
noun( 'brusqueness', '-', mass, _ ).
noun( 'brutality', 'brutalities', both, _ ).
noun( 'brute', 'brutes', count, _ ).
noun( 'bubble', 'bubbles', count, _ ).
noun( 'bubbly', '-', mass, _ ).
noun( 'buccaneer', 'buccaneers', count, _ ).
noun( 'buck', 'bucks', count, _ ).
noun( 'bucket', 'buckets', count, _ ).
noun( 'bucketful', 'bucketfuls', count, _ ).
noun( 'buckle', 'buckles', count, _ ).
noun( 'buckler', 'bucklers', count, _ ).
noun( 'buckram', '-', mass, _ ).
noun( 'buckshot', '-', mass, _ ).
noun( 'buckskin', '-', mass, _ ).
noun( 'bucktooth', 'buckteeth', count, _ ).
noun( 'buckwheat', '-', mass, _ ).
noun( 'bucolics', 'bucolics', mass, _ ).
noun( 'bud', 'buds', count, _ ).
noun( 'buddy', 'buddies', count, _ ).
noun( 'budgerigar', 'budgerigars', count, _ ).
noun( 'budget', 'budgets', count, _ ).
noun( 'budgie', 'budgies', count, _ ).
noun( 'buff', 'buffs', both, _ ).
noun( 'buffalo', 'buffalo', count, _ ).
noun( 'buffer', 'buffers', count, _ ).
noun( 'buffet', 'buffets', count, _ ).
noun( 'buffet', 'buffets', count, _ ).
noun( 'buffoon', 'buffoons', count, _ ).
noun( 'buffoonery', '-', mass, _ ).
noun( 'bug', 'bugs', count, _ ).
noun( 'bug-hunter', 'bug-hunters', count, _ ).
noun( 'bugaboo', 'bugaboos', count, _ ).
noun( 'bugbear', 'bugbears', count, _ ).
noun( 'bugger', 'buggers', count, _ ).
noun( 'bugger-all', '-', mass, _ ).
noun( 'buggery', '-', mass, _ ).
noun( 'buggy', 'buggies', count, _ ).
noun( 'bugle', 'bugles', count, _ ).
noun( 'bugler', 'buglers', count, _ ).
noun( 'buhl', '-', mass, _ ).
noun( 'build', '-', mass, _ ).
noun( 'build-up', 'build-ups', count, _ ).
noun( 'builder', 'builders', count, _ ).
noun( 'building', 'buildings', both, _ ).
noun( 'building-society', 'building-societies', count, _ ).
noun( 'bulb', 'bulbs', count, _ ).
noun( 'bulbul', 'bulbuls', count, _ ).
noun( 'bulge', 'bulges', count, _ ).
noun( 'bulk', '-', mass, _ ).
noun( 'bulkhead', 'bulkheads', count, _ ).
noun( 'bull', 'bulls', count, _ ).
noun( 'bull\'s-eye', 'bull\'s-eyes', count, _ ).
noun( 'bull-neck', 'bull-necks', count, _ ).
noun( 'bull-terrier', 'bull-terriers', count, _ ).
noun( 'bulldog', 'bulldogs', count, _ ).
noun( 'bulldozer', 'bulldozers', count, _ ).
noun( 'bullet', 'bullets', count, _ ).
noun( 'bulletin', 'bulletins', count, _ ).
noun( 'bullfight', 'bullfights', count, _ ).
noun( 'bullfighter', 'bullfighters', count, _ ).
noun( 'bullfinch', 'bullfinches', count, _ ).
noun( 'bullfrog', 'bullfrogs', count, _ ).
noun( 'bullion', '-', mass, _ ).
noun( 'bullock', 'bullocks', count, _ ).
noun( 'bullring', 'bullrings', count, _ ).
noun( 'bullshit', '-', mass, _ ).
noun( 'bully', 'bullies', both, _ ).
noun( 'bulrush', 'bulrushes', count, _ ).
noun( 'bulwark', 'bulwarks', count, _ ).
noun( 'bum', 'bums', count, _ ).
noun( 'bumblebee', 'bumblebees', count, _ ).
noun( 'bumboat', 'bumboats', count, _ ).
noun( 'bump', 'bumps', count, _ ).
noun( 'bumper', 'bumpers', count, _ ).
noun( 'bumpkin', 'bumpkins', count, _ ).
noun( 'bumptiousness', '-', mass, _ ).
noun( 'bun', 'buns', count, _ ).
noun( 'buna', '-', mass, _ ).
noun( 'bunch', 'bunches', count, _ ).
noun( 'bundle', 'bundles', count, _ ).
noun( 'bung', 'bungs', count, _ ).
noun( 'bung-hole', 'bung-holes', count, _ ).
noun( 'bungalow', 'bungalows', count, _ ).
noun( 'bungle', 'bungles', count, _ ).
noun( 'bungler', 'bunglers', count, _ ).
noun( 'bunion', 'bunions', count, _ ).
noun( 'bunk', 'bunks', both, _ ).
noun( 'bunker', 'bunkers', count, _ ).
noun( 'bunkum', '-', mass, _ ).
noun( 'bunny', 'bunnies', count, _ ).
noun( 'bunting', '-', mass, _ ).
noun( 'buoy', 'buoys', count, _ ).
noun( 'buoyancy', '-', mass, _ ).
noun( 'bur', 'burs', count, _ ).
noun( 'burden', 'burdens', both, _ ).
noun( 'burdock', 'burdocks', count, _ ).
noun( 'bureau', 'bureaux', count, _ ).
noun( 'bureaucracy', 'bureaucracies', both, _ ).
noun( 'bureaucrat', 'bureaucrats', count, _ ).
noun( 'burette', 'burettes', count, _ ).
noun( 'burg', 'burgs', count, _ ).
noun( 'burgess', 'burgesses', count, _ ).
noun( 'burgh', 'burghs', count, _ ).
noun( 'burgher', 'burghers', count, _ ).
noun( 'burglar', 'burglars', count, _ ).
noun( 'burglar-alarm', 'burglar-alarms', count, _ ).
noun( 'burglary', 'burglaries', both, _ ).
noun( 'burgomaster', 'burgomasters', count, _ ).
noun( 'burial', 'burials', both, _ ).
noun( 'burial-ground', 'burial-grounds', count, _ ).
noun( 'burlap', '-', mass, _ ).
noun( 'burlesque', 'burlesques', both, _ ).
noun( 'burn', 'burns', count, _ ).
noun( 'burn-up', 'burn-ups', count, _ ).
noun( 'burner', 'burners', count, _ ).
noun( 'burnouse', 'burnouses', count, _ ).
noun( 'burp', 'burps', count, _ ).
noun( 'burr', 'burrs', count, _ ).
noun( 'burr-drill', 'burr-drills', count, _ ).
noun( 'burrow', 'burrows', count, _ ).
noun( 'bursar', 'bursars', count, _ ).
noun( 'bursary', 'bursaries', count, _ ).
noun( 'burst', 'bursts', count, _ ).
noun( 'burthen', 'burthens', count, _ ).
noun( 'burton', 'burtons', count, _ ).
noun( 'burying-ground', 'burying-grounds', count, _ ).
noun( 'bus', 'buses', count, _ ).
noun( 'busby', 'busbies', count, _ ).
noun( 'bush', 'bushes', both, _ ).
noun( 'bushel', 'bushels', count, _ ).
noun( 'business', 'businesses', both, _ ).
noun( 'businessman', 'businessmen', count, _ ).
noun( 'busker', 'buskers', count, _ ).
noun( 'busman', 'busmen', count, _ ).
noun( 'bust', 'busts', count, _ ).
noun( 'bust-up', 'bust-ups', count, _ ).
noun( 'bustard', 'bustards', count, _ ).
noun( 'buster', 'busters', count, _ ).
noun( 'bustle', 'bustles', both, _ ).
noun( 'busybody', 'busybodies', count, _ ).
noun( 'butane', '-', mass, _ ).
noun( 'butcher', 'butchers', count, _ ).
noun( 'butchery', '-', mass, _ ).
noun( 'butler', 'butlers', count, _ ).
noun( 'butt', 'butts', count, _ ).
noun( 'butter', '-', mass, _ ).
noun( 'butterbean', 'butterbeans', count, _ ).
noun( 'buttercup', 'buttercups', count, _ ).
noun( 'butterfingers', '-', count, _ ).
noun( 'butterfly', 'butterflies', count, _ ).
noun( 'buttermilk', '-', mass, _ ).
noun( 'butterscotch', '-', mass, _ ).
noun( 'buttery', 'butteries', count, _ ).
noun( 'buttock', 'buttocks', count, _ ).
noun( 'button', 'buttons', count, _ ).
noun( 'buttonhole', 'buttonholes', count, _ ).
noun( 'buttonhook', 'buttonhooks', count, _ ).
noun( 'buttonwood', 'buttonwoods', both, _ ).
noun( 'buttress', 'buttresses', count, _ ).
noun( 'buy', 'buys', count, _ ).
noun( 'buyer', 'buyers', count, _ ).
noun( 'buzz', 'buzzes', count, _ ).
noun( 'buzzard', 'buzzards', count, _ ).
noun( 'buzzer', 'buzzers', count, _ ).
noun( 'by-election', 'by-elections', count, _ ).
noun( 'bye', 'byes', count, _ ).
noun( 'bye-bye', 'bye-byes', count, _ ).
noun( 'bye-law', 'bye-laws', count, _ ).
noun( 'bylaw', 'bylaws', count, _ ).
noun( 'bypass', 'bypasses', count, _ ).
noun( 'bypath', 'bypaths', count, _ ).
noun( 'byplay', '-', mass, _ ).
noun( 'byproduct', 'byproducts', count, _ ).
noun( 'byroad', 'byroads', count, _ ).
noun( 'bystander', 'bystanders', count, _ ).
noun( 'byway', 'byways', count, _ ).
noun( 'byword', 'bywords', count, _ ).
noun( 'c', '-', count, _ ).
noun( 'ca\'canny', '-', count, _ ).
noun( 'cab', 'cabs', count, _ ).
noun( 'cab-rank', 'cab-ranks', count, _ ).
noun( 'cabal', 'cabals', count, _ ).
noun( 'cabaret', 'cabarets', count, _ ).
noun( 'cabbage', 'cabbages', both, _ ).
noun( 'cabby', 'cabbies', count, _ ).
noun( 'caber', 'cabers', count, _ ).
noun( 'cabin', 'cabins', count, _ ).
noun( 'cabinet', 'cabinets', count, _ ).
noun( 'cabinet-maker', 'cabinet-makers', count, _ ).
noun( 'cable', 'cables', both, _ ).
noun( 'cable\'s-length', 'cable\'s-lengths', count, _ ).
noun( 'cable-car', 'cable-cars', count, _ ).
noun( 'cable-length', 'cable-lengths', count, _ ).
noun( 'cable-railway', 'cable-railways', count, _ ).
noun( 'cablegram', 'cablegrams', count, _ ).
noun( 'cabman', 'cabmen', count, _ ).
noun( 'caboodle', 'caboodles', count, _ ).
noun( 'caboose', 'cabooses', count, _ ).
noun( 'cabstand', 'cabstands', count, _ ).
noun( 'cacao', 'cacaos', count, _ ).
noun( 'cacao-bean', 'cacao-beans', count, _ ).
noun( 'cacao-tree', 'cacao-trees', count, _ ).
noun( 'cache', 'caches', count, _ ).
noun( 'cachet', 'cachets', count, _ ).
noun( 'cachou', 'cachous', count, _ ).
noun( 'cackle', 'cackles', both, _ ).
noun( 'cackler', 'cacklers', count, _ ).
noun( 'cacophony', 'cacophonies', count, _ ).
noun( 'cactus', 'cactuses', count, _ ).
noun( 'cad', 'cads', count, _ ).
noun( 'cadaver', 'cadavers', count, _ ).
noun( 'caddie', 'caddies', count, _ ).
noun( 'caddy', 'caddies', count, _ ).
noun( 'cadence', 'cadences', count, _ ).
noun( 'cadenza', 'cadenzas', count, _ ).
noun( 'cadet', 'cadets', count, _ ).
noun( 'cadger', 'cadgers', count, _ ).
noun( 'cadmium', '-', mass, _ ).
noun( 'cadre', 'cadres', count, _ ).
noun( 'caesura', 'caesuras', count, _ ).
noun( 'caf_e', 'caf_es', count, _ ).
noun( 'cafe-au-lait', 'cafe-au-laits', count, _ ).
noun( 'cafeteria', 'cafeterias', count, _ ).
noun( 'caff', 'caffs', count, _ ).
noun( 'caffeine', '-', mass, _ ).
noun( 'caftan', 'caftans', count, _ ).
noun( 'cage', 'cages', count, _ ).
noun( 'cagoule', 'cagoules', count, _ ).
noun( 'caiman', 'caimans', count, _ ).
noun( 'cairn', 'cairns', count, _ ).
noun( 'caisson', 'caissons', count, _ ).
noun( 'caitiff', 'caitiffs', count, _ ).
noun( 'cajolery', '-', mass, _ ).
noun( 'cake', 'cakes', both, _ ).
noun( 'calabash', 'calabashes', count, _ ).
noun( 'calamity', 'calamities', count, _ ).
noun( 'calcination', 'calcinations', count, _ ).
noun( 'calcium', '-', mass, _ ).
noun( 'calculation', 'calculations', both, _ ).
noun( 'calculator', 'calculators', count, _ ).
noun( 'calculus', 'calculuses', both, _ ).
noun( 'caldron', 'caldrons', count, _ ).
noun( 'calendar', 'calendars', count, _ ).
noun( 'calender', 'calenders', count, _ ).
noun( 'calf', 'calves', both, _ ).
noun( 'calf-love', '-', mass, _ ).
noun( 'calibration', 'calibrations', both, _ ).
noun( 'calibre', 'calibres', both, _ ).
noun( 'calico', '-', mass, _ ).
noun( 'calif', 'califs', count, _ ).
noun( 'caliph', 'caliphs', count, _ ).
noun( 'caliphate', 'caliphates', count, _ ).
noun( 'calisthenics', 'calisthenics', mass, _ ).
noun( 'calk', 'calks', count, _ ).
noun( 'call', 'calls', both, _ ).
noun( 'call-box', 'call-boxes', count, _ ).
noun( 'call-girl', 'call-girls', count, _ ).
noun( 'call-over', 'call-overs', count, _ ).
noun( 'call-up', 'call-ups', count, _ ).
noun( 'caller', 'callers', count, _ ).
noun( 'calligraphy', '-', mass, _ ).
noun( 'calling', 'callings', count, _ ).
noun( 'calliope', 'calliopes', count, _ ).
noun( 'callisthenics', 'callisthenics', mass, _ ).
noun( 'callosity', 'callosities', count, _ ).
noun( 'callousness', '-', mass, _ ).
noun( 'callowness', '-', mass, _ ).
noun( 'callus', 'calluses', count, _ ).
noun( 'calm', 'calms', count, _ ).
noun( 'calmness', '-', mass, _ ).
noun( 'calomel', '-', mass, _ ).
noun( 'calorie', 'calories', count, _ ).
noun( 'calumny', 'calumnies', both, _ ).
noun( 'calypso', 'calypsos', count, _ ).
noun( 'calyx', 'calyxes', count, _ ).
noun( 'cam', 'cams', count, _ ).
noun( 'camaraderie', '-', mass, _ ).
noun( 'camber', 'cambers', count, _ ).
noun( 'cambric', '-', mass, _ ).
noun( 'camel', 'camels', count, _ ).
noun( 'camel-hair', '-', mass, _ ).
noun( 'camellia', 'camellias', count, _ ).
noun( 'cameo', 'cameos', count, _ ).
noun( 'camera', 'cameras', count, _ ).
noun( 'cameraman', 'cameramen', count, _ ).
noun( 'camion', 'camions', count, _ ).
noun( 'camomile', '-', mass, _ ).
noun( 'camouflage', '-', mass, _ ).
noun( 'camp', 'camps', count, _ ).
noun( 'camp-bed', 'camp-beds', count, _ ).
noun( 'camp-chair', 'camp-chairs', count, _ ).
noun( 'camp-fire', 'camp-fires', count, _ ).
noun( 'camp-follower', 'camp-followers', count, _ ).
noun( 'camp-stool', 'camp-stools', count, _ ).
noun( 'campaign', 'campaigns', count, _ ).
noun( 'campaigner', 'campaigners', count, _ ).
noun( 'campanile', 'campaniles', count, _ ).
noun( 'campanula', 'campanulas', count, _ ).
noun( 'camper', 'campers', count, _ ).
noun( 'camphor', '-', mass, _ ).
noun( 'camping', '-', mass, _ ).
noun( 'campion', '-', mass, _ ).
noun( 'campus', 'campuses', count, _ ).
noun( 'camshaft', 'camshafts', count, _ ).
noun( 'can', 'cans', count, _ ).
noun( 'canal', 'canals', count, _ ).
noun( 'canalization', 'canalizations', count, _ ).
noun( 'canap_e', 'canap_es', count, _ ).
noun( 'canard', 'canards', count, _ ).
noun( 'canary', 'canaries', count, _ ).
noun( 'canary-bird', 'canary-birds', count, _ ).
noun( 'canary-wine', 'canary-wines', count, _ ).
noun( 'canasta', 'canastas', count, _ ).
noun( 'cancan', 'cancans', count, _ ).
noun( 'cancellation', 'cancellations', both, _ ).
noun( 'cancer', 'cancers', both, _ ).
noun( 'candelabrum', 'candelabra', count, _ ).
noun( 'candidate', 'candidates', count, _ ).
noun( 'candidature', 'candidatures', count, _ ).
noun( 'candle', 'candles', count, _ ).
noun( 'candle-power', '-', mass, _ ).
noun( 'candlelight', '-', mass, _ ).
noun( 'candlestick', 'candlesticks', count, _ ).
noun( 'candlewick', '-', mass, _ ).
noun( 'candour', '-', mass, _ ).
noun( 'candy', 'candies', both, _ ).
noun( 'candy-floss', '-', mass, _ ).
noun( 'candytuft', 'candytufts', count, _ ).
noun( 'cane', 'canes', both, _ ).
noun( 'canister', 'canisters', count, _ ).
noun( 'canker', '-', mass, _ ).
noun( 'canna', 'cannas', count, _ ).
noun( 'cannabis', '-', mass, _ ).
noun( 'cannery', 'canneries', count, _ ).
noun( 'cannibal', 'cannibals', count, _ ).
noun( 'cannibalism', '-', mass, _ ).
noun( 'cannon', 'cannons', count, _ ).
noun( 'cannon-fodder', '-', mass, _ ).
noun( 'cannonade', 'cannonades', count, _ ).
noun( 'canoe', 'canoes', count, _ ).
noun( 'canoeist', 'canoeists', count, _ ).
noun( 'canon', 'canons', count, _ ).
noun( 'canonization', 'canonizations', count, _ ).
noun( 'canopy', 'canopies', count, _ ).
noun( 'cant', 'cants', both, _ ).
noun( 'cantaloup', 'cantaloups', count, _ ).
noun( 'cantaloupe', 'cantaloupes', count, _ ).
noun( 'cantata', 'cantatas', count, _ ).
noun( 'canteen', 'canteens', count, _ ).
noun( 'canter', 'canters', count, _ ).
noun( 'canticle', 'canticles', count, _ ).
noun( 'cantilever', 'cantilevers', count, _ ).
noun( 'canto', 'cantos', count, _ ).
noun( 'canton', 'cantons', count, _ ).
noun( 'cantonment', 'cantonments', count, _ ).
noun( 'cantor', 'cantors', count, _ ).
noun( 'canvas', 'canvases', both, _ ).
noun( 'canvass', 'canvasses', count, _ ).
noun( 'canyon', 'canyons', count, _ ).
noun( 'cap', 'caps', count, _ ).
noun( 'capability', 'capabilities', both, _ ).
noun( 'capacity', 'capacities', both, _ ).
noun( 'caparison', 'caparisons', count, _ ).
noun( 'cape', 'capes', count, _ ).
noun( 'caper', 'capers', count, _ ).
noun( 'capillary', 'capillaries', count, _ ).
noun( 'capital', 'capitals', both, _ ).
noun( 'capitalism', '-', mass, _ ).
noun( 'capitalist', 'capitalists', count, _ ).
noun( 'capitalization', 'capitalizations', count, _ ).
noun( 'capitation', 'capitations', count, _ ).
noun( 'capitulation', '-', mass, _ ).
noun( 'capon', 'capons', count, _ ).
noun( 'caprice', 'caprices', count, _ ).
noun( 'capriciousness', '-', mass, _ ).
noun( 'capsicum', 'capsicums', count, _ ).
noun( 'capstan', 'capstans', count, _ ).
noun( 'capsule', 'capsules', count, _ ).
noun( 'captain', 'captains', count, _ ).
noun( 'caption', 'captions', count, _ ).
noun( 'captive', 'captives', count, _ ).
noun( 'captivity', '-', mass, _ ).
noun( 'captor', 'captors', count, _ ).
noun( 'capture', 'captures', both, _ ).
noun( 'car', 'cars', count, _ ).
noun( 'car-ferry', 'car-ferries', count, _ ).
noun( 'carafe', 'carafes', count, _ ).
noun( 'caramel', 'caramels', both, _ ).
noun( 'carapace', 'carapaces', count, _ ).
noun( 'carat', 'carats', count, _ ).
noun( 'caravan', 'caravans', count, _ ).
noun( 'caravanning', '-', mass, _ ).
noun( 'caravansary', 'caravansaries', count, _ ).
noun( 'caravanserai', 'caravanserais', count, _ ).
noun( 'caraway', 'caraways', count, _ ).
noun( 'carbide', 'carbides', count, _ ).
noun( 'carbine', 'carbines', count, _ ).
noun( 'carbohydrate', 'carbohydrates', both, _ ).
noun( 'carbon', 'carbons', both, _ ).
noun( 'carbon-paper', '-', mass, _ ).
noun( 'carbonization', 'carbonizations', both, _ ).
noun( 'carborundum', '-', mass, _ ).
noun( 'carboy', 'carboys', count, _ ).
noun( 'carbuncle', 'carbuncles', count, _ ).
noun( 'carburettor', 'carburettors', count, _ ).
noun( 'carcase', 'carcases', count, _ ).
noun( 'carcass', 'carcasses', count, _ ).
noun( 'card', 'cards', count, _ ).
noun( 'card-sharper', 'card-sharpers', count, _ ).
noun( 'cardamom', '-', mass, _ ).
noun( 'cardboard', '-', mass, _ ).
noun( 'cardigan', 'cardigans', count, _ ).
noun( 'cardinal', 'cardinals', count, _ ).
noun( 'care', 'cares', both, _ ).
noun( 'career', 'careers', both, _ ).
noun( 'careerist', 'careerists', count, _ ).
noun( 'carefulness', '-', mass, _ ).
noun( 'carelessness', '-', mass, _ ).
noun( 'caress', 'caresses', count, _ ).
noun( 'caret', 'carets', count, _ ).
noun( 'caretaker', 'caretakers', count, _ ).
noun( 'cargo', 'cargoes', both, _ ).
noun( 'caribou', 'caribou', count, _ ).
noun( 'caricature', 'caricatures', both, _ ).
noun( 'caricaturist', 'caricaturists', count, _ ).
noun( 'caries', '-', mass, _ ).
noun( 'carillon', 'carillons', count, _ ).
noun( 'carmine', 'carmines', both, _ ).
noun( 'carnage', '-', mass, _ ).
noun( 'carnation', 'carnations', count, _ ).
noun( 'carnival', 'carnivals', both, _ ).
noun( 'carnivore', 'carnivores', count, _ ).
noun( 'carol', 'carols', count, _ ).
noun( 'caroller', 'carollers', count, _ ).
noun( 'carousal', 'carousals', count, _ ).
noun( 'carousel', 'carousels', count, _ ).
noun( 'carp', 'carp', count, _ ).
noun( 'carpal', 'carpals', count, _ ).
noun( 'carpenter', 'carpenters', count, _ ).
noun( 'carpentry', '-', mass, _ ).
noun( 'carpet', 'carpets', count, _ ).
noun( 'carpet-beater', 'carpet-beaters', count, _ ).
noun( 'carpet-knight', 'carpet-knights', count, _ ).
noun( 'carpet-sweeper', 'carpet-sweepers', count, _ ).
noun( 'carpetbag', 'carpetbags', count, _ ).
noun( 'carpetbagger', 'carpetbaggers', count, _ ).
noun( 'carport', 'carports', count, _ ).
noun( 'carriage', 'carriages', both, _ ).
noun( 'carriageway', 'carriageways', count, _ ).
noun( 'carrier', 'carriers', count, _ ).
noun( 'carrier-bag', 'carrier-bags', count, _ ).
noun( 'carrier-pigeon', 'carrier-pigeons', count, _ ).
noun( 'carrion', '-', mass, _ ).
noun( 'carrion-crow', 'carrion-crows', count, _ ).
noun( 'carrot', 'carrots', count, _ ).
noun( 'carry', 'carries', count, _ ).
noun( 'carrycot', 'carrycots', count, _ ).
noun( 'carsickness', '-', mass, _ ).
noun( 'cart', 'carts', count, _ ).
noun( 'cart-track', 'cart-tracks', count, _ ).
noun( 'cartage', '-', mass, _ ).
noun( 'carte blanche', '-', count, _ ).
noun( 'cartel', 'cartels', count, _ ).
noun( 'carter', 'carters', count, _ ).
noun( 'carthorse', 'carthorses', count, _ ).
noun( 'cartilage', 'cartilages', both, _ ).
noun( 'cartload', 'cartloads', count, _ ).
noun( 'cartographer', 'cartographers', count, _ ).
noun( 'cartography', '-', mass, _ ).
noun( 'carton', 'cartons', count, _ ).
noun( 'cartoon', 'cartoons', count, _ ).
noun( 'cartoonist', 'cartoonists', count, _ ).
noun( 'cartridge', 'cartridges', count, _ ).
noun( 'cartridge-belt', 'cartridge-belts', count, _ ).
noun( 'cartridge-paper', 'cartridge-papers', count, _ ).
noun( 'cartroad', 'cartroads', count, _ ).
noun( 'cartwheel', 'cartwheels', count, _ ).
noun( 'carver', 'carvers', count, _ ).
noun( 'carving', 'carvings', both, _ ).
noun( 'carving-fork', 'carving-forks', count, _ ).
noun( 'carving-knife', 'carving-knives', count, _ ).
noun( 'caryatid', 'caryatids', count, _ ).
noun( 'cascade', 'cascades', count, _ ).
noun( 'case', 'cases', count, _ ).
noun( 'case-history', 'case-histories', count, _ ).
noun( 'case-law', 'case-laws', count, _ ).
noun( 'casebook', 'casebooks', count, _ ).
noun( 'casein', '-', mass, _ ).
noun( 'casement', 'casements', count, _ ).
noun( 'casework', 'caseworks', count, _ ).
noun( 'cash', '-', mass, _ ).
noun( 'cashew', 'cashews', count, _ ).
noun( 'cashier', 'cashiers', count, _ ).
noun( 'cashmere', '-', mass, _ ).
noun( 'casing', 'casings', count, _ ).
noun( 'casino', 'casinos', count, _ ).
noun( 'cask', 'casks', count, _ ).
noun( 'casket', 'caskets', count, _ ).
noun( 'cassava', '-', mass, _ ).
noun( 'casserole', 'casseroles', count, _ ).
noun( 'cassette', 'cassettes', count, _ ).
noun( 'cassock', 'cassocks', count, _ ).
noun( 'cassowary', 'cassowaries', count, _ ).
noun( 'cast', 'casts', count, _ ).
noun( 'castaway', 'castaways', count, _ ).
noun( 'caste', 'castes', both, _ ).
noun( 'caster', 'casters', count, _ ).
noun( 'castigation', 'castigations', both, _ ).
noun( 'casting', 'castings', count, _ ).
noun( 'castle', 'castles', count, _ ).
noun( 'castor', 'castors', count, _ ).
noun( 'castor oil', '-', mass, _ ).
noun( 'castration', 'castrations', count, _ ).
noun( 'casualty', 'casualties', count, _ ).
noun( 'casuist', 'casuists', count, _ ).
noun( 'casuistry', 'casuistries', both, _ ).
noun( 'casus belli', '-', count, _ ).
noun( 'cat', 'cats', count, _ ).
noun( 'cat-nap', 'cat-naps', count, _ ).
noun( 'cat-o\'-nine-tails', '-', count, _ ).
noun( 'cat-sleep', 'cat-sleeps', count, _ ).
noun( 'cataclysm', 'cataclysms', count, _ ).
noun( 'catafalque', 'catafalques', count, _ ).
noun( 'catalepsy', '-', mass, _ ).
noun( 'cataleptic', 'cataleptics', count, _ ).
noun( 'catalogue', 'catalogues', count, _ ).
noun( 'catalpa', 'catalpas', count, _ ).
noun( 'catalysis', '-', mass, _ ).
noun( 'catalyst', 'catalysts', count, _ ).
noun( 'catamaran', 'catamarans', count, _ ).
noun( 'catapult', 'catapults', count, _ ).
noun( 'cataract', 'cataracts', count, _ ).
noun( 'catarrh', '-', mass, _ ).
noun( 'catastrophe', 'catastrophes', count, _ ).
noun( 'catcall', 'catcalls', count, _ ).
noun( 'catch', 'catches', count, _ ).
noun( 'catch-crop', 'catch-crops', count, _ ).
noun( 'catcher', 'catchers', count, _ ).
noun( 'catchment', 'catchments', count, _ ).
noun( 'catchment-area', 'catchment-areas', count, _ ).
noun( 'catchment-basin', 'catchment-basins', count, _ ).
noun( 'catchup', 'catchups', count, _ ).
noun( 'catchword', 'catchwords', count, _ ).
noun( 'catechism', 'catechisms', both, _ ).
noun( 'category', 'categories', count, _ ).
noun( 'caterer', 'caterers', count, _ ).
noun( 'caterpillar', 'caterpillars', count, _ ).
noun( 'caterwaul', 'caterwauls', count, _ ).
noun( 'catfish', 'catfish', count, _ ).
noun( 'catgut', '-', mass, _ ).
noun( 'catharsis', 'catharses', both, _ ).
noun( 'cathartic', 'cathartics', both, _ ).
noun( 'cathedral', 'cathedrals', count, _ ).
noun( 'cathode', 'cathodes', count, _ ).
noun( 'catholicity', '-', mass, _ ).
noun( 'catkin', 'catkins', count, _ ).
noun( 'catsup', 'catsups', both, _ ).
noun( 'cattiness', '-', mass, _ ).
noun( 'cattle-cake', '-', mass, _ ).
noun( 'cattleman', 'cattlemen', count, _ ).
noun( 'catwalk', 'catwalks', count, _ ).
noun( 'caucus', 'caucuses', count, _ ).
noun( 'caul', 'cauls', count, _ ).
noun( 'cauldron', 'cauldrons', count, _ ).
noun( 'cauliflower', 'cauliflowers', both, _ ).
noun( 'causality', '-', mass, _ ).
noun( 'causation', '-', mass, _ ).
noun( 'cause', 'causes', both, _ ).
noun( 'causerie', 'causeries', count, _ ).
noun( 'causeway', 'causeways', count, _ ).
noun( 'caution', 'cautions', both, _ ).
noun( 'cavalcade', 'cavalcades', count, _ ).
noun( 'cavalier', 'cavaliers', count, _ ).
noun( 'cavalry', 'cavalries', count, _ ).
noun( 'cavalryman', 'cavalrymen', count, _ ).
noun( 'cave', 'caves', count, _ ).
noun( 'cave-dweller', 'cave-dwellers', count, _ ).
noun( 'cave-in', 'cave-ins', count, _ ).
noun( 'caveat', 'caveats', count, _ ).
noun( 'caveman', 'cavemen', count, _ ).
noun( 'cavern', 'caverns', count, _ ).
noun( 'caviar', '-', mass, _ ).
noun( 'caviare', '-', mass, _ ).
noun( 'cavity', 'cavities', count, _ ).
noun( 'caw', 'caws', count, _ ).
noun( 'cayenne', '-', mass, _ ).
noun( 'cayenne pepper', '-', mass, _ ).
noun( 'cayman', 'caymans', count, _ ).
noun( 'ca~non', 'ca~nons', count, _ ).
noun( 'cc', 'cc', count, _ ).
noun( 'cease', 'ceases', count, _ ).
noun( 'cease-fire', 'cease-fires', count, _ ).
noun( 'cedar', 'cedars', both, _ ).
noun( 'cedilla', 'cedillas', count, _ ).
noun( 'ceiling', 'ceilings', count, _ ).
noun( 'celandine', 'celandines', count, _ ).
noun( 'celebrant', 'celebrants', count, _ ).
noun( 'celebration', 'celebrations', both, _ ).
noun( 'celebrity', 'celebrities', both, _ ).
noun( 'celerity', '-', mass, _ ).
noun( 'celery', '-', mass, _ ).
noun( 'celibacy', '-', mass, _ ).
noun( 'celibate', 'celibates', count, _ ).
noun( 'cell', 'cells', count, _ ).
noun( 'cellar', 'cellars', count, _ ).
noun( 'cellarage', 'cellarages', both, _ ).
noun( 'cellist', 'cellists', count, _ ).
noun( 'cello', 'cellos', count, _ ).
noun( 'cellophane', '-', mass, _ ).
noun( 'celluloid', '-', mass, _ ).
noun( 'cellulose', '-', mass, _ ).
noun( 'cement', '-', mass, _ ).
noun( 'cement-mixer', 'cement-mixers', count, _ ).
noun( 'cemetery', 'cemeteries', count, _ ).
noun( 'cenotaph', 'cenotaphs', count, _ ).
noun( 'censer', 'censers', count, _ ).
noun( 'censor', 'censors', count, _ ).
noun( 'censorship', 'censorships', count, _ ).
noun( 'censure', 'censures', both, _ ).
noun( 'census', 'censuses', count, _ ).
noun( 'cent', 'cents', count, _ ).
noun( 'centaur', 'centaurs', count, _ ).
noun( 'centenarian', 'centenarians', count, _ ).
noun( 'centenary', 'centenaries', count, _ ).
noun( 'centennial', 'centennials', count, _ ).
noun( 'centime', 'centimes', count, _ ).
noun( 'centimetre', 'centimetres', count, _ ).
noun( 'centipede', 'centipedes', count, _ ).
noun( 'central', 'centrals', count, _ ).
noun( 'centralization', 'centralizations', both, _ ).
noun( 'centre', 'centres', count, _ ).
noun( 'centre-bit', 'centre-bits', count, _ ).
noun( 'centre-board', 'centre-boards', count, _ ).
noun( 'centrepiece', 'centrepieces', count, _ ).
noun( 'centrifuge', 'centrifuges', count, _ ).
noun( 'centurion', 'centurions', count, _ ).
noun( 'century', 'centuries', count, _ ).
noun( 'ceramics', 'ceramics', mass, _ ).
noun( 'cereal', 'cereals', both, _ ).
noun( 'cerebration', '-', mass, _ ).
noun( 'ceremonial', 'ceremonials', both, _ ).
noun( 'ceremony', 'ceremonies', both, _ ).
noun( 'cerise', 'cerises', both, _ ).
noun( 'cert', 'certs', count, _ ).
noun( 'certainty', 'certainties', both, _ ).
noun( 'certificate', 'certificates', count, _ ).
noun( 'certification', 'certifications', both, _ ).
noun( 'certitude', '-', mass, _ ).
noun( 'cervix', 'cervixes', count, _ ).
noun( 'cessation', '-', mass, _ ).
noun( 'cession', 'cessions', both, _ ).
noun( 'cesspit', 'cesspits', count, _ ).
noun( 'cesspool', 'cesspools', count, _ ).
noun( 'cf', '-', proper, _ ).
noun( 'ch^ateau', 'ch^ateaux', count, _ ).
noun( 'chafe', 'chafes', count, _ ).
noun( 'chaff', '-', mass, _ ).
noun( 'chaffinch', 'chaffinches', count, _ ).
noun( 'chafing dish', 'chafing dishes', count, _ ).
noun( 'chagrin', '-', mass, _ ).
noun( 'chain', 'chains', count, _ ).
noun( 'chain-armour', '-', mass, _ ).
noun( 'chain-gang', 'chain-gangs', count, _ ).
noun( 'chain-letter', 'chain-letters', count, _ ).
noun( 'chain-mail', '-', mass, _ ).
noun( 'chain-smoker', 'chain-smokers', count, _ ).
noun( 'chain-stitch', 'chain-stitches', count, _ ).
noun( 'chain-store', 'chain-stores', count, _ ).
noun( 'chair', 'chairs', count, _ ).
noun( 'chair-lift', 'chair-lifts', count, _ ).
noun( 'chairman', 'chairmen', count, _ ).
noun( 'chairmanship', '-', mass, _ ).
noun( 'chaise', 'chaises', count, _ ).
noun( 'chaise longue', 'chaise longues', count, _ ).
noun( 'chalet', 'chalets', count, _ ).
noun( 'chalice', 'chalices', count, _ ).
noun( 'chalk', 'chalks', both, _ ).
noun( 'chalkpit', 'chalkpits', count, _ ).
noun( 'challenge', 'challenges', count, _ ).
noun( 'challenger', 'challengers', count, _ ).
noun( 'chamber', 'chambers', count, _ ).
noun( 'chamberlain', 'chamberlains', count, _ ).
noun( 'chambermaid', 'chambermaids', count, _ ).
noun( 'chamberpot', 'chamberpots', count, _ ).
noun( 'chameleon', 'chameleons', count, _ ).
noun( 'chammy-leather', 'chammy-leathers', count, _ ).
noun( 'chamois', 'chamoises', count, _ ).
noun( 'chamois-leather', 'chamois-leathers', count, _ ).
noun( 'champ', 'champs', count, _ ).
noun( 'champagne', 'champagnes', count, _ ).
noun( 'champion', 'champions', count, _ ).
noun( 'championship', 'championships', both, _ ).
noun( 'chance', 'chances', both, _ ).
noun( 'chancel', 'chancels', count, _ ).
noun( 'chancellery', 'chancelleries', count, _ ).
noun( 'chancellor', 'chancellors', count, _ ).
noun( 'chancery', 'chanceries', count, _ ).
noun( 'chandelier', 'chandeliers', count, _ ).
noun( 'chandler', 'chandlers', count, _ ).
noun( 'change', 'changes', both, _ ).
noun( 'changeableness', '-', mass, _ ).
noun( 'changeling', 'changelings', count, _ ).
noun( 'changeover', 'changeovers', count, _ ).
noun( 'channel', 'channels', count, _ ).
noun( 'chant', 'chants', count, _ ).
noun( 'chaos', '-', mass, _ ).
noun( 'chap', 'chaps', count, _ ).
noun( 'chapel', 'chapels', count, _ ).
noun( 'chapelgoer', 'chapelgoers', count, _ ).
noun( 'chaperon', 'chaperons', count, _ ).
noun( 'chaplain', 'chaplains', count, _ ).
noun( 'chaplaincy', 'chaplaincies', count, _ ).
noun( 'chaplet', 'chaplets', count, _ ).
noun( 'chapman', 'chapmen', count, _ ).
noun( 'chapter', 'chapters', count, _ ).
noun( 'chapterhouse', 'chapterhouses', count, _ ).
noun( 'char', 'chars', both, _ ).
noun( 'char`abanc', 'char`abancs', count, _ ).
noun( 'charabanc', 'charabancs', count, _ ).
noun( 'character', 'characters', both, _ ).
noun( 'characteristic', 'characteristics', count, _ ).
noun( 'characterization', '-', mass, _ ).
noun( 'charade', 'charades', count, _ ).
noun( 'charcoal', '-', mass, _ ).
noun( 'charcoal-burner', 'charcoal-burners', count, _ ).
noun( 'chard', 'chards', count, _ ).
noun( 'charg_e d\'affaires', 'charg_es d\'affaires', count, _ ).
noun( 'charge', 'charges', both, _ ).
noun( 'charge-account', 'charge-accounts', count, _ ).
noun( 'charge-sheet', 'charge-sheets', count, _ ).
noun( 'charger', 'chargers', count, _ ).
noun( 'chariot', 'chariots', count, _ ).
noun( 'charioteer', 'charioteers', count, _ ).
noun( 'charisma', 'charismas', count, _ ).
noun( 'charity', 'charities', both, _ ).
noun( 'charivari', '-', mass, _ ).
noun( 'charlady', 'charladies', count, _ ).
noun( 'charlatan', 'charlatans', count, _ ).
noun( 'charlock', '-', mass, _ ).
noun( 'charm', 'charms', both, _ ).
noun( 'charmer', 'charmers', count, _ ).
noun( 'charnel house', 'charnel houses', count, _ ).
noun( 'chart', 'charts', count, _ ).
noun( 'charter', 'charters', count, _ ).
noun( 'charter-party', 'charter-parties', count, _ ).
noun( 'chartreuse', '-', mass, _ ).
noun( 'charwoman', 'charwomen', count, _ ).
noun( 'chase', 'chases', count, _ ).
noun( 'chaser', 'chasers', count, _ ).
noun( 'chasm', 'chasms', count, _ ).
noun( 'chassis', 'chassis', count, _ ).
noun( 'chastisement', '-', mass, _ ).
noun( 'chastity', '-', mass, _ ).
noun( 'chasuble', 'chasubles', count, _ ).
noun( 'chat', 'chats', count, _ ).
noun( 'chatelaine', 'chatelaines', count, _ ).
noun( 'chattel', 'chattels', count, _ ).
noun( 'chatter', '-', mass, _ ).
noun( 'chatterbox', 'chatterboxes', count, _ ).
noun( 'chauffeur', 'chauffeurs', count, _ ).
noun( 'chauffeuse', 'chauffeuses', count, _ ).
noun( 'chauvinism', '-', mass, _ ).
noun( 'chauvinist', 'chauvinists', count, _ ).
noun( 'chaw', 'chaws', count, _ ).
noun( 'chaw-bacon', 'chaw-bacons', count, _ ).
noun( 'cheapness', '-', mass, _ ).
noun( 'cheat', 'cheats', count, _ ).
noun( 'check', 'checks', both, _ ).
noun( 'checkbook', 'checkbooks', count, _ ).
noun( 'checker', 'checkers', count, _ ).
noun( 'checkers', 'checkers', mass, _ ).
noun( 'checklist', 'checklists', count, _ ).
noun( 'checkmate', 'checkmates', count, _ ).
noun( 'checkout', 'checkouts', count, _ ).
noun( 'checkpoint', 'checkpoints', count, _ ).
noun( 'checkroom', 'checkrooms', count, _ ).
noun( 'checkup', 'checkups', count, _ ).
noun( 'cheek', 'cheeks', both, _ ).
noun( 'cheekbone', 'cheekbones', count, _ ).
noun( 'cheep', 'cheeps', count, _ ).
noun( 'cheer', 'cheers', both, _ ).
noun( 'cheerfulness', '-', mass, _ ).
noun( 'cheering', '-', mass, _ ).
noun( 'cheerleader', 'cheerleaders', count, _ ).
noun( 'cheerlessness', '-', mass, _ ).
noun( 'cheese', 'cheeses', both, _ ).
noun( 'cheesecake', 'cheesecakes', both, _ ).
noun( 'cheesecloth', 'cheesecloths', both, _ ).
noun( 'cheetah', 'cheetahs', count, _ ).
noun( 'chef', 'chefs', count, _ ).
noun( 'chef-d\'oeuvre', 'chefs-d\'oeuvre', count, _ ).
noun( 'chemical', 'chemicals', count, _ ).
noun( 'chemise', 'chemises', count, _ ).
noun( 'chemist', 'chemists', count, _ ).
noun( 'chemistry', '-', mass, _ ).
noun( 'chemotherapy', '-', mass, _ ).
noun( 'chenille', '-', mass, _ ).
noun( 'cheque', 'cheques', count, _ ).
noun( 'chequebook', 'chequebooks', count, _ ).
noun( 'cheroot', 'cheroots', count, _ ).
noun( 'cherry', 'cherries', count, _ ).
noun( 'cherub', 'cherubs', count, _ ).
noun( 'chervil', '-', mass, _ ).
noun( 'chess', '-', mass, _ ).
noun( 'chessboard', 'chessboards', count, _ ).
noun( 'chessman', 'chessmen', count, _ ).
noun( 'chest', 'chests', count, _ ).
noun( 'chesterfield', 'chesterfields', count, _ ).
noun( 'chestnut', 'chestnuts', both, _ ).
noun( 'cheval glass', 'cheval glasses', count, _ ).
noun( 'chevron', 'chevrons', count, _ ).
noun( 'chew', 'chews', count, _ ).
noun( 'chewing-gum', '-', mass, _ ).
noun( 'chiaroscuro', '-', mass, _ ).
noun( 'chic', '-', mass, _ ).
noun( 'chicanery', 'chicaneries', both, _ ).
noun( 'chick', 'chicks', count, _ ).
noun( 'chicken', 'chickens', both, _ ).
noun( 'chicken-run', 'chicken-runs', count, _ ).
noun( 'chickenfeed', '-', mass, _ ).
noun( 'chickenpox', '-', mass, _ ).
noun( 'chickpea', 'chickpeas', count, _ ).
noun( 'chickweed', '-', mass, _ ).
noun( 'chicle', '-', mass, _ ).
noun( 'chicory', '-', mass, _ ).
noun( 'chief', 'chiefs', count, _ ).
noun( 'chieftain', 'chieftains', count, _ ).
noun( 'chieftaincy', 'chieftaincies', count, _ ).
noun( 'chiffon', '-', mass, _ ).
noun( 'chiffonier', 'chiffoniers', count, _ ).
noun( 'chignon', 'chignons', count, _ ).
noun( 'chilblain', 'chilblains', count, _ ).
noun( 'child', 'children', count, _ ).
noun( 'child\'s-play', '-', mass, _ ).
noun( 'child-bearing', '-', mass, _ ).
noun( 'childbirth', '-', mass, _ ).
noun( 'childhood', '-', mass, _ ).
noun( 'chill', 'chills', count, _ ).
noun( 'chilli', 'chillies', count, _ ).
noun( 'chilly', 'chillies', count, _ ).
noun( 'chimaera', 'chimaeras', count, _ ).
noun( 'chime', 'chimes', count, _ ).
noun( 'chimera', 'chimeras', count, _ ).
noun( 'chimney', 'chimneys', count, _ ).
noun( 'chimney-sweep', 'chimney-sweeps', count, _ ).
noun( 'chimneybreast', 'chimneybreasts', count, _ ).
noun( 'chimneypiece', 'chimneypieces', count, _ ).
noun( 'chimneypot', 'chimneypots', count, _ ).
noun( 'chimneystack', 'chimneystacks', count, _ ).
noun( 'chimneysweep', 'chimneysweeps', count, _ ).
noun( 'chimneysweeper', 'chimneysweepers', count, _ ).
noun( 'chimp', 'chimps', count, _ ).
noun( 'chimpanzee', 'chimpanzees', count, _ ).
noun( 'chin', 'chins', count, _ ).
noun( 'chin-strap', 'chin-straps', count, _ ).
noun( 'chin-wagging', '-', mass, _ ).
noun( 'china', '-', mass, _ ).
noun( 'china-closet', 'china-closets', count, _ ).
noun( 'chinaware', '-', mass, _ ).
noun( 'chinchilla', 'chinchillas', both, _ ).
noun( 'chine', 'chines', count, _ ).
noun( 'chink', 'chinks', count, _ ).
noun( 'chintz', '-', mass, _ ).
noun( 'chip', 'chips', count, _ ).
noun( 'chipboard', '-', mass, _ ).
noun( 'chipmunk', 'chipmunks', count, _ ).
noun( 'chiropodist', 'chiropodists', count, _ ).
noun( 'chiropody', '-', mass, _ ).
noun( 'chiropractor', 'chiropractors', count, _ ).
noun( 'chirp', 'chirps', count, _ ).
noun( 'chirpiness', '-', mass, _ ).
noun( 'chirrup', 'chirrups', count, _ ).
noun( 'chisel', 'chisels', count, _ ).
noun( 'chiseller', 'chisellers', count, _ ).
noun( 'chit', 'chits', count, _ ).
noun( 'chit-chat', '-', mass, _ ).
noun( 'chivalry', '-', mass, _ ).
noun( 'chive', 'chives', both, _ ).
noun( 'chloride', '-', mass, _ ).
noun( 'chlorination', '-', mass, _ ).
noun( 'chlorine', '-', mass, _ ).
noun( 'chloroform', '-', mass, _ ).
noun( 'chlorophyll', '-', mass, _ ).
noun( 'choc', 'chocs', count, _ ).
noun( 'choc-ice', 'choc-ices', count, _ ).
noun( 'chock', 'chocks', count, _ ).
noun( 'chocolate', 'chocolates', both, _ ).
noun( 'choice', 'choices', both, _ ).
noun( 'choir', 'choirs', count, _ ).
noun( 'choir-school', 'choir-schools', count, _ ).
noun( 'choirboy', 'choirboys', count, _ ).
noun( 'choke', 'chokes', count, _ ).
noun( 'choke-damp', '-', mass, _ ).
noun( 'choker', 'chokers', count, _ ).
noun( 'chokey', 'chokeys', count, _ ).
noun( 'choky', 'chokies', count, _ ).
noun( 'choler', 'cholers', count, _ ).
noun( 'cholera', '-', mass, _ ).
noun( 'chop', 'chops', count, _ ).
noun( 'chop suey', '-', mass, _ ).
noun( 'chop-house', 'chop-houses', count, _ ).
noun( 'chopper', 'choppers', count, _ ).
noun( 'chorale', 'chorales', count, _ ).
noun( 'chord', 'chords', count, _ ).
noun( 'chore', 'chores', count, _ ).
noun( 'choreographer', 'choreographers', count, _ ).
noun( 'choreography', '-', mass, _ ).
noun( 'chorister', 'choristers', count, _ ).
noun( 'chortle', 'chortles', count, _ ).
noun( 'chorus', 'choruses', count, _ ).
noun( 'chorus-girl', 'chorus-girls', count, _ ).
noun( 'chow', 'chows', count, _ ).
noun( 'chowder', 'chowders', count, _ ).
noun( 'christening', 'christenings', count, _ ).
noun( 'chrome', '-', mass, _ ).
noun( 'chromium', '-', mass, _ ).
noun( 'chromosome', 'chromosomes', count, _ ).
noun( 'chronicle', 'chronicles', count, _ ).
noun( 'chronicler', 'chroniclers', count, _ ).
noun( 'chronology', 'chronologies', both, _ ).
noun( 'chronometer', 'chronometers', count, _ ).
noun( 'chrysalis', 'chrysalises', count, _ ).
noun( 'chrysanthemum', 'chrysanthemums', count, _ ).
noun( 'chuck', 'chucks', count, _ ).
noun( 'chucker-out', 'chuckers-out', count, _ ).
noun( 'chuckle', 'chuckles', count, _ ).
noun( 'chug', 'chugs', count, _ ).
noun( 'chukker', 'chukkers', count, _ ).
noun( 'chum', 'chums', count, _ ).
noun( 'chump', 'chumps', count, _ ).
noun( 'chunk', 'chunks', count, _ ).
noun( 'church', 'churches', both, _ ).
noun( 'churchgoer', 'churchgoers', count, _ ).
noun( 'churchman', 'churchmen', count, _ ).
noun( 'churchwarden', 'churchwardens', count, _ ).
noun( 'churchyard', 'churchyards', count, _ ).
noun( 'churl', 'churls', count, _ ).
noun( 'churn', 'churns', count, _ ).
noun( 'chute', 'chutes', count, _ ).
noun( 'chutney', 'chutneys', both, _ ).
noun( 'cicada', 'cicadas', count, _ ).
noun( 'cicala', 'cicalas', count, _ ).
noun( 'cicatrice', 'cicatrices', count, _ ).
noun( 'cicatrix', 'cicatrices', count, _ ).
noun( 'cicerone', '-', count, _ ).
noun( 'cider', 'ciders', both, _ ).
noun( 'ciderpress', 'ciderpresses', count, _ ).
noun( 'cif', '-', proper, _ ).
noun( 'cigar', 'cigars', count, _ ).
noun( 'cigarette', 'cigarettes', count, _ ).
noun( 'cigarette-case', 'cigarette-cases', count, _ ).
noun( 'cigarette-holder', 'cigarette-holders', count, _ ).
noun( 'cigarette-paper', 'cigarette-papers', count, _ ).
noun( 'cinch', '-', count, _ ).
noun( 'cinchona', 'cinchonas', count, _ ).
noun( 'cincture', 'cinctures', count, _ ).
noun( 'cinder', 'cinders', count, _ ).
noun( 'cinder-track', 'cinder-tracks', count, _ ).
noun( 'cine-camera', 'cine-cameras', count, _ ).
noun( 'cine-film', 'cine-films', count, _ ).
noun( 'cine-projector', 'cine-projectors', count, _ ).
noun( 'cinema', 'cinemas', count, _ ).
noun( 'cinematography', '-', mass, _ ).
noun( 'cinnamon', '-', mass, _ ).
noun( 'cinquefoil', 'cinquefoils', count, _ ).
noun( 'cipher', 'ciphers', count, _ ).
noun( 'circle', 'circles', count, _ ).
noun( 'circlet', 'circlets', count, _ ).
noun( 'circuit', 'circuits', count, _ ).
noun( 'circular', 'circulars', count, _ ).
noun( 'circularity', '-', mass, _ ).
noun( 'circulation', 'circulations', both, _ ).
noun( 'circumcision', 'circumcisions', count, _ ).
noun( 'circumference', 'circumferences', count, _ ).
noun( 'circumflex', 'circumflexes', count, _ ).
noun( 'circumlocution', 'circumlocutions', both, _ ).
noun( 'circumnavigation', 'circumnavigations', count, _ ).
noun( 'circumscription', 'circumscriptions', both, _ ).
noun( 'circumspection', '-', mass, _ ).
noun( 'circumstance', 'circumstances', count, _ ).
noun( 'circumvention', 'circumventions', count, _ ).
noun( 'circus', 'circuses', count, _ ).
noun( 'cirrhosis', '-', mass, _ ).
noun( 'cirrus', '-', mass, _ ).
noun( 'cissy', 'cissies', count, _ ).
noun( 'cistern', 'cisterns', count, _ ).
noun( 'citadel', 'citadels', count, _ ).
noun( 'citation', 'citations', both, _ ).
noun( 'citizen', 'citizens', count, _ ).
noun( 'citizenship', 'citizenships', count, _ ).
noun( 'citron', 'citrons', count, _ ).
noun( 'citrus', 'citruses', count, _ ).
noun( 'city', 'cities', count, _ ).
noun( 'civet', 'civets', both, _ ).
noun( 'civet-cat', 'civet-cats', count, _ ).
noun( 'civics', 'civics', mass, _ ).
noun( 'civilian', 'civilians', count, _ ).
noun( 'civility', 'civilities', both, _ ).
noun( 'civilization', 'civilizations', both, _ ).
noun( 'clack', 'clacks', count, _ ).
noun( 'claim', 'claims', both, _ ).
noun( 'claimant', 'claimants', count, _ ).
noun( 'clairvoyance', '-', mass, _ ).
noun( 'clairvoyant', 'clairvoyants', count, _ ).
noun( 'clam', 'clams', count, _ ).
noun( 'clambake', 'clambakes', count, _ ).
noun( 'clamber', 'clambers', count, _ ).
noun( 'clamour', 'clamours', both, _ ).
noun( 'clamp', 'clamps', count, _ ).
noun( 'clamp-down', 'clamp-downs', count, _ ).
noun( 'clan', 'clans', count, _ ).
noun( 'clang', 'clangs', count, _ ).
noun( 'clanger', 'clangers', count, _ ).
noun( 'clangour', '-', mass, _ ).
noun( 'clank', 'clanks', count, _ ).
noun( 'clansman', 'clansmen', count, _ ).
noun( 'clap', 'claps', both, _ ).
noun( 'clapboard', 'clapboards', count, _ ).
noun( 'clapper', 'clappers', count, _ ).
noun( 'clapperboard', 'clapperboards', count, _ ).
noun( 'claptrap', '-', mass, _ ).
noun( 'claque', 'claques', count, _ ).
noun( 'claret', 'clarets', both, _ ).
noun( 'clarification', '-', mass, _ ).
noun( 'clarinet', 'clarinets', count, _ ).
noun( 'clarinetist', 'clarinetists', count, _ ).
noun( 'clarinettist', 'clarinettists', count, _ ).
noun( 'clarion', 'clarions', count, _ ).
noun( 'clarity', '-', mass, _ ).
noun( 'clash', 'clashes', count, _ ).
noun( 'clasp', 'clasps', count, _ ).
noun( 'clasp-knife', 'clasp-knives', count, _ ).
noun( 'class', 'classes', both, _ ).
noun( 'class-feeling', '-', mass, _ ).
noun( 'class-fellow', 'class-fellows', count, _ ).
noun( 'class-list', 'class-lists', count, _ ).
noun( 'class-warfare', '-', mass, _ ).
noun( 'classic', 'classics', count, _ ).
noun( 'classicism', '-', mass, _ ).
noun( 'classicist', 'classicists', count, _ ).
noun( 'classics', 'classics', mass, _ ).
noun( 'classification', 'classifications', both, _ ).
noun( 'classmate', 'classmates', count, _ ).
noun( 'classroom', 'classrooms', count, _ ).
noun( 'clatter', '-', count, _ ).
noun( 'clause', 'clauses', count, _ ).
noun( 'claustrophobia', '-', mass, _ ).
noun( 'clavichord', 'clavichords', count, _ ).
noun( 'clavicle', 'clavicles', count, _ ).
noun( 'claw', 'claws', count, _ ).
noun( 'clawback', '-', mass, _ ).
noun( 'clawhammer', 'clawhammers', count, _ ).
noun( 'clay', '-', mass, _ ).
noun( 'clean', 'cleans', count, _ ).
noun( 'clean-up', 'clean-ups', count, _ ).
noun( 'cleaner', 'cleaners', count, _ ).
noun( 'cleanliness', '-', mass, _ ).
noun( 'cleanser', 'cleansers', both, _ ).
noun( 'clear', 'clears', count, _ ).
noun( 'clearance', 'clearances', both, _ ).
noun( 'clearing', 'clearings', count, _ ).
noun( 'clearing-house', 'clearing-houses', count, _ ).
noun( 'clearness', '-', mass, _ ).
noun( 'clearway', 'clearways', count, _ ).
noun( 'cleat', 'cleats', count, _ ).
noun( 'cleavage', 'cleavages', count, _ ).
noun( 'cleaver', 'cleavers', count, _ ).
noun( 'clef', 'clefs', count, _ ).
noun( 'cleft', 'clefts', count, _ ).
noun( 'clematis', '-', mass, _ ).
noun( 'clemency', '-', mass, _ ).
noun( 'clerestory', 'clerestories', count, _ ).
noun( 'clergy', 'clergies', count, _ ).
noun( 'clergyman', 'clergymen', count, _ ).
noun( 'cleric', 'clerics', count, _ ).
noun( 'clerihew', 'clerihews', count, _ ).
noun( 'clerk', 'clerks', count, _ ).
noun( 'cleverness', '-', mass, _ ).
noun( 'clew', 'clews', count, _ ).
noun( 'clich_e', 'clich_es', count, _ ).
noun( 'click', 'clicks', count, _ ).
noun( 'client', 'clients', count, _ ).
noun( 'clientele', 'clienteles', count, _ ).
noun( 'cliff', 'cliffs', count, _ ).
noun( 'cliff-hanger', 'cliff-hangers', count, _ ).
noun( 'climacteric', 'climacterics', count, _ ).
noun( 'climate', 'climates', count, _ ).
noun( 'climatology', '-', mass, _ ).
noun( 'climax', 'climaxes', count, _ ).
noun( 'climb', 'climbs', count, _ ).
noun( 'climb-down', 'climb-downs', count, _ ).
noun( 'climber', 'climbers', count, _ ).
noun( 'clime', 'climes', count, _ ).
noun( 'clinch', 'clinches', count, _ ).
noun( 'clincher', 'clinchers', count, _ ).
noun( 'clinic', 'clinics', count, _ ).
noun( 'clink', 'clinks', count, _ ).
noun( 'clinker', 'clinkers', both, _ ).
noun( 'clip', 'clips', count, _ ).
noun( 'clip-joint', 'clip-joints', count, _ ).
noun( 'clipper', 'clippers', count, _ ).
noun( 'clipping', 'clippings', count, _ ).
noun( 'clique', 'cliques', count, _ ).
noun( 'clitoris', 'clitorises', count, _ ).
noun( 'cloak', 'cloaks', count, _ ).
noun( 'cloakroom', 'cloakrooms', count, _ ).
noun( 'clobber', 'clobbers', count, _ ).
noun( 'cloche', 'cloches', count, _ ).
noun( 'clock', 'clocks', count, _ ).
noun( 'clock-dial', 'clock-dials', count, _ ).
noun( 'clock-face', 'clock-faces', count, _ ).
noun( 'clock-golf', '-', mass, _ ).
noun( 'clock-tower', 'clock-towers', count, _ ).
noun( 'clock-watching', '-', mass, _ ).
noun( 'clockwork', 'clockworks', count, _ ).
noun( 'clod', 'clods', count, _ ).
noun( 'clodhopper', 'clodhoppers', count, _ ).
noun( 'clog', 'clogs', count, _ ).
noun( 'clog-dance', 'clog-dances', count, _ ).
noun( 'cloisonn_e', '-', mass, _ ).
noun( 'cloister', 'cloisters', count, _ ).
noun( 'clone', 'clones', count, _ ).
noun( 'close', 'closes', count, _ ).
noun( 'close', '-', count, _ ).
noun( 'close-down', 'close-downs', count, _ ).
noun( 'close-up', 'close-ups', count, _ ).
noun( 'closeness', '-', mass, _ ).
noun( 'closet', 'closets', count, _ ).
noun( 'closure', 'closures', both, _ ).
noun( 'clot', 'clots', count, _ ).
noun( 'cloth', 'cloths', both, _ ).
noun( 'clothes-basket', 'clothes-baskets', count, _ ).
noun( 'clothes-hanger', 'clothes-hangers', count, _ ).
noun( 'clothes-moth', 'clothes-moths', count, _ ).
noun( 'clothes-peg', 'clothes-pegs', count, _ ).
noun( 'clothes-pin', 'clothes-pins', count, _ ).
noun( 'clotheshorse', 'clotheshorses', count, _ ).
noun( 'clothesline', 'clotheslines', count, _ ).
noun( 'clothier', 'clothiers', count, _ ).
noun( 'clothing', '-', mass, _ ).
noun( 'cloud', 'clouds', both, _ ).
noun( 'cloud-bank', 'cloud-banks', count, _ ).
noun( 'cloud-cuckoo-land', '-', count, _ ).
noun( 'cloudburst', 'cloudbursts', count, _ ).
noun( 'clout', 'clouts', count, _ ).
noun( 'clove', 'cloves', count, _ ).
noun( 'clove hitch', 'clove hitches', count, _ ).
noun( 'clover', '-', mass, _ ).
noun( 'cloverleaf', 'cloverleaves', count, _ ).
noun( 'clown', 'clowns', count, _ ).
noun( 'club', 'clubs', count, _ ).
noun( 'clubfoot', 'clubfeet', count, _ ).
noun( 'clubhouse', 'clubhouses', count, _ ).
noun( 'cluck', 'clucks', count, _ ).
noun( 'clue', 'clues', count, _ ).
noun( 'clump', 'clumps', count, _ ).
noun( 'clumsiness', '-', mass, _ ).
noun( 'clunk', 'clunks', count, _ ).
noun( 'cluster', 'clusters', count, _ ).
noun( 'clutch', 'clutches', count, _ ).
noun( 'clutter', 'clutters', both, _ ).
noun( 'cm', 'cm', count, _ ).
noun( 'co-ed', 'co-eds', count, _ ).
noun( 'co-op', 'co-ops', count, _ ).
noun( 'co-respondent', 'co-respondents', count, _ ).
noun( 'co-star', 'co-stars', count, _ ).
noun( 'coach', 'coaches', count, _ ).
noun( 'coach-builder', 'coach-builders', count, _ ).
noun( 'coachman', 'coachmen', count, _ ).
noun( 'coagulation', 'coagulations', both, _ ).
noun( 'coal', 'coals', both, _ ).
noun( 'coal-gas', '-', mass, _ ).
noun( 'coal-hole', 'coal-holes', count, _ ).
noun( 'coal-house', 'coal-houses', count, _ ).
noun( 'coal-scuttle', 'coal-scuttles', count, _ ).
noun( 'coal-seam', 'coal-seams', count, _ ).
noun( 'coal-tar', '-', mass, _ ).
noun( 'coalescence', 'coalescences', both, _ ).
noun( 'coalface', 'coalfaces', count, _ ).
noun( 'coalfield', 'coalfields', count, _ ).
noun( 'coaling-station', 'coaling-stations', count, _ ).
noun( 'coalition', 'coalitions', both, _ ).
noun( 'coalman', 'coalmen', count, _ ).
noun( 'coalmine', 'coalmines', count, _ ).
noun( 'coalpit', 'coalpits', count, _ ).
noun( 'coaming', 'coamings', count, _ ).
noun( 'coarseness', '-', mass, _ ).
noun( 'coast', 'coasts', count, _ ).
noun( 'coaster', 'coasters', count, _ ).
noun( 'coastguard', 'coastguards', count, _ ).
noun( 'coastline', 'coastlines', count, _ ).
noun( 'coat', 'coats', count, _ ).
noun( 'coat-hanger', 'coat-hangers', count, _ ).
noun( 'coatee', 'coatees', count, _ ).
noun( 'coating', 'coatings', both, _ ).
noun( 'coaxing', 'coaxings', both, _ ).
noun( 'cob', 'cobs', count, _ ).
noun( 'cob-nut', 'cob-nuts', count, _ ).
noun( 'cobalt', '-', mass, _ ).
noun( 'cobber', 'cobbers', count, _ ).
noun( 'cobble', 'cobbles', count, _ ).
noun( 'cobbler', 'cobblers', count, _ ).
noun( 'cobblestone', 'cobblestones', count, _ ).
noun( 'cobra', 'cobras', count, _ ).
noun( 'cobweb', 'cobwebs', count, _ ).
noun( 'cocaine', '-', mass, _ ).
noun( 'cochineal', '-', mass, _ ).
noun( 'cochlea', 'cochleas', count, _ ).
noun( 'cock', 'cocks', count, _ ).
noun( 'cock-a-doodle-doo', 'cock-a-doodle-doos', count, _ ).
noun( 'cock-crow', 'cock-crows', count, _ ).
noun( 'cockade', 'cockades', count, _ ).
noun( 'cockatoo', 'cockatoos', count, _ ).
noun( 'cockchafer', 'cockchafers', count, _ ).
noun( 'cocker', 'cockers', count, _ ).
noun( 'cockerel', 'cockerels', count, _ ).
noun( 'cockfighting', '-', mass, _ ).
noun( 'cockhorse', 'cockhorses', count, _ ).
noun( 'cockle', 'cockles', count, _ ).
noun( 'cockleshell', 'cockleshells', count, _ ).
noun( 'cockney', 'cockneys', count, _ ).
noun( 'cockpit', 'cockpits', count, _ ).
noun( 'cockroach', 'cockroaches', count, _ ).
noun( 'cockscomb', 'cockscombs', count, _ ).
noun( 'cocktail', 'cocktails', count, _ ).
noun( 'cockup', 'cockups', count, _ ).
noun( 'coco', 'cocos', count, _ ).
noun( 'cocoa', '-', mass, _ ).
noun( 'coconut', 'coconuts', count, _ ).
noun( 'cocoon', 'cocoons', count, _ ).
noun( 'cocotte', 'cocottes', count, _ ).
noun( 'cod', 'cods', both, _ ).
noun( 'cod-liver oil', '-', mass, _ ).
noun( 'coda', 'codas', count, _ ).
noun( 'code', 'codes', both, _ ).
noun( 'codeine', '-', mass, _ ).
noun( 'codex', 'codices', count, _ ).
noun( 'codfish', 'codfish', count, _ ).
noun( 'codger', 'codgers', count, _ ).
noun( 'codicil', 'codicils', count, _ ).
noun( 'codification', 'codifications', count, _ ).
noun( 'codling', 'codlings', count, _ ).
noun( 'codpiece', 'codpieces', count, _ ).
noun( 'coeducation', '-', mass, _ ).
noun( 'coefficient', 'coefficients', count, _ ).
noun( 'coercion', '-', mass, _ ).
noun( 'coeval', 'coevals', count, _ ).
noun( 'coexistence', '-', mass, _ ).
noun( 'coffee', 'coffees', both, _ ).
noun( 'coffee-house', 'coffee-houses', count, _ ).
noun( 'coffee-mill', 'coffee-mills', count, _ ).
noun( 'coffee-stall', 'coffee-stalls', count, _ ).
noun( 'coffer', 'coffers', count, _ ).
noun( 'coffer-dam', 'coffer-dams', count, _ ).
noun( 'coffin', 'coffins', count, _ ).
noun( 'cog', 'cogs', count, _ ).
noun( 'cogency', '-', mass, _ ).
noun( 'cogitation', '-', mass, _ ).
noun( 'cognac', '-', mass, _ ).
noun( 'cognate', 'cognates', count, _ ).
noun( 'cognition', '-', mass, _ ).
noun( 'cognizance', '-', mass, _ ).
noun( 'cognomen', 'cognomens', count, _ ).
noun( 'cogwheel', 'cogwheels', count, _ ).
noun( 'cohabitation', 'cohabitations', both, _ ).
noun( 'coherence', '-', mass, _ ).
noun( 'coherency', '-', mass, _ ).
noun( 'cohesion', '-', mass, _ ).
noun( 'cohort', 'cohorts', count, _ ).
noun( 'coif', 'coifs', count, _ ).
noun( 'coiffeur', 'coiffeurs', count, _ ).
noun( 'coiffure', 'coiffures', count, _ ).
noun( 'coign', 'coigns', count, _ ).
noun( 'coil', 'coils', count, _ ).
noun( 'coin', 'coins', both, _ ).
noun( 'coinage', 'coinages', both, _ ).
noun( 'coincidence', 'coincidences', both, _ ).
noun( 'coiner', 'coiners', count, _ ).
noun( 'coir', '-', mass, _ ).
noun( 'coition', '-', mass, _ ).
noun( 'coitus', '-', mass, _ ).
noun( 'coke', 'cokes', both, _ ).
noun( 'col', 'cols', count, _ ).
noun( 'cola', 'colas', count, _ ).
noun( 'colander', 'colanders', count, _ ).
noun( 'cold', 'colds', both, _ ).
noun( 'coldness', '-', mass, _ ).
noun( 'coleslaw', '-', mass, _ ).
noun( 'colic', '-', mass, _ ).
noun( 'colitis', '-', mass, _ ).
noun( 'collaboration', '-', mass, _ ).
noun( 'collaborationist', 'collaborationists', count, _ ).
noun( 'collaborator', 'collaborators', count, _ ).
noun( 'collage', 'collages', both, _ ).
noun( 'collapse', 'collapses', count, _ ).
noun( 'collar', 'collars', count, _ ).
noun( 'collarbone', 'collarbones', count, _ ).
noun( 'collateral', '-', mass, _ ).
noun( 'collation', 'collations', count, _ ).
noun( 'colleague', 'colleagues', count, _ ).
noun( 'collect', 'collects', count, _ ).
noun( 'collection', 'collections', both, _ ).
noun( 'collectivization', 'collectivizations', count, _ ).
noun( 'collector', 'collectors', count, _ ).
noun( 'colleen', 'colleens', count, _ ).
noun( 'college', 'colleges', both, _ ).
noun( 'collie', 'collies', count, _ ).
noun( 'collier', 'colliers', count, _ ).
noun( 'colliery', 'collieries', count, _ ).
noun( 'collision', 'collisions', both, _ ).
noun( 'collocation', 'collocations', both, _ ).
noun( 'colloquialism', 'colloquialisms', count, _ ).
noun( 'colloquy', 'colloquies', both, _ ).
noun( 'collusion', '-', mass, _ ).
noun( 'colon', 'colons', count, _ ).
noun( 'colonel', 'colonels', count, _ ).
noun( 'colonial', 'colonials', count, _ ).
noun( 'colonialism', '-', mass, _ ).
noun( 'colonialist', 'colonialists', count, _ ).
noun( 'colonist', 'colonists', count, _ ).
noun( 'colonization', '-', mass, _ ).
noun( 'colonizer', 'colonizers', count, _ ).
noun( 'colonnade', 'colonnades', count, _ ).
noun( 'colony', 'colonies', count, _ ).
noun( 'coloratura', '-', mass, _ ).
noun( 'colossus', 'colossi', count, _ ).
noun( 'colour', 'colours', both, _ ).
noun( 'colour-bar', 'colour-bars', count, _ ).
noun( 'colour-wash', 'colour-washes', count, _ ).
noun( 'colouring', '-', mass, _ ).
noun( 'colt', 'colts', count, _ ).
noun( 'columbine', 'columbines', count, _ ).
noun( 'column', 'columns', count, _ ).
noun( 'columnist', 'columnists', count, _ ).
noun( 'coma', 'comas', count, _ ).
noun( 'comb', 'combs', count, _ ).
noun( 'comb-out', 'comb-outs', count, _ ).
noun( 'combat', 'combats', count, _ ).
noun( 'combatant', 'combatants', count, _ ).
noun( 'combination', 'combinations', both, _ ).
noun( 'combination-lock', 'combination-locks', count, _ ).
noun( 'combine', 'combines', count, _ ).
noun( 'combustible', 'combustibles', count, _ ).
noun( 'combustion', '-', mass, _ ).
noun( 'come-on', 'come-ons', count, _ ).
noun( 'comeback', 'comebacks', count, _ ).
noun( 'comedian', 'comedians', count, _ ).
noun( 'comedienne', 'comediennes', count, _ ).
noun( 'comedown', 'comedowns', count, _ ).
noun( 'comedy', 'comedies', both, _ ).
noun( 'comeliness', '-', mass, _ ).
noun( 'comer', 'comers', count, _ ).
noun( 'comestible', 'comestibles', count, _ ).
noun( 'comet', 'comets', count, _ ).
noun( 'comfit', 'comfits', count, _ ).
noun( 'comfort', 'comforts', both, _ ).
noun( 'comforter', 'comforters', count, _ ).
noun( 'comfrey', '-', mass, _ ).
noun( 'comic', 'comics', count, _ ).
noun( 'coming', 'comings', count, _ ).
noun( 'comity', '-', mass, _ ).
noun( 'comma', 'commas', count, _ ).
noun( 'command', 'commands', both, _ ).
noun( 'commandant', 'commandants', count, _ ).
noun( 'commander', 'commanders', count, _ ).
noun( 'commandment', 'commandments', count, _ ).
noun( 'commando', 'commandos', count, _ ).
noun( 'commemoration', 'commemorations', both, _ ).
noun( 'commencement', 'commencements', count, _ ).
noun( 'commendation', 'commendations', both, _ ).
noun( 'comment', 'comments', both, _ ).
noun( 'commentary', 'commentaries', count, _ ).
noun( 'commentator', 'commentators', count, _ ).
noun( 'commerce', '-', mass, _ ).
noun( 'commercial', 'commercials', count, _ ).
noun( 'commercialism', '-', mass, _ ).
noun( 'commination', 'comminations', both, _ ).
noun( 'commiseration', 'commiserations', both, _ ).
noun( 'commissar', 'commissars', count, _ ).
noun( 'commissariat', 'commissariats', count, _ ).
noun( 'commissary', 'commissaries', count, _ ).
noun( 'commission', 'commissions', both, _ ).
noun( 'commissionaire', 'commissionaires', count, _ ).
noun( 'commissioner', 'commissioners', count, _ ).
noun( 'commital', 'commitals', count, _ ).
noun( 'commitment', 'commitments', both, _ ).
noun( 'committee', 'committees', count, _ ).
noun( 'commode', 'commodes', count, _ ).
noun( 'commodity', 'commodities', count, _ ).
noun( 'commodore', 'commodores', count, _ ).
noun( 'common', 'commons', count, _ ).
noun( 'common-room', 'common-rooms', count, _ ).
noun( 'commonalty', 'commonalties', count, _ ).
noun( 'commoner', 'commoners', count, _ ).
noun( 'commonplace', 'commonplaces', count, _ ).
noun( 'commonwealth', 'commonwealths', count, _ ).
noun( 'commotion', 'commotions', both, _ ).
noun( 'commune', 'communes', count, _ ).
noun( 'communicant', 'communicants', count, _ ).
noun( 'communication', 'communications', both, _ ).
noun( 'communion', 'communions', both, _ ).
noun( 'communiqu_e', 'communiqu_es', count, _ ).
noun( 'communism', '-', mass, _ ).
noun( 'communist', 'communists', count, _ ).
noun( 'community', 'communities', both, _ ).
noun( 'commutation', 'commutations', both, _ ).
noun( 'commutator', 'commutators', count, _ ).
noun( 'commuter', 'commuters', count, _ ).
noun( 'comp`ere', 'comp`eres', count, _ ).
noun( 'compact', 'compacts', count, _ ).
noun( 'compactness', '-', mass, _ ).
noun( 'companion', 'companions', count, _ ).
noun( 'companionship', '-', mass, _ ).
noun( 'companionway', 'companionways', count, _ ).
noun( 'company', 'companies', both, _ ).
noun( 'comparability', '-', mass, _ ).
noun( 'comparative', 'comparatives', count, _ ).
noun( 'compare', 'compares', count, _ ).
noun( 'comparison', 'comparisons', both, _ ).
noun( 'compartment', 'compartments', count, _ ).
noun( 'compass', 'compasses', count, _ ).
noun( 'compassion', '-', mass, _ ).
noun( 'compatibility', '-', mass, _ ).
noun( 'compatriot', 'compatriots', count, _ ).
noun( 'compeer', 'compeers', count, _ ).
noun( 'compendium', 'compendiums', count, _ ).
noun( 'compensation', 'compensations', both, _ ).
noun( 'competence', '-', mass, _ ).
noun( 'competition', 'competitions', both, _ ).
noun( 'competitiveness', '-', mass, _ ).
noun( 'competitor', 'competitors', count, _ ).
noun( 'compilation', 'compilations', both, _ ).
noun( 'compiler', 'compilers', count, _ ).
noun( 'complacence', '-', mass, _ ).
noun( 'complacency', '-', mass, _ ).
noun( 'complainant', 'complainants', count, _ ).
noun( 'complaint', 'complaints', both, _ ).
noun( 'complaisance', '-', mass, _ ).
noun( 'complement', 'complements', count, _ ).
noun( 'completeness', '-', mass, _ ).
noun( 'completion', '-', mass, _ ).
noun( 'complex', 'complexes', count, _ ).
noun( 'complexion', 'complexions', count, _ ).
noun( 'complexity', 'complexities', both, _ ).
noun( 'compliance', '-', mass, _ ).
noun( 'complication', 'complications', count, _ ).
noun( 'complicity', '-', mass, _ ).
noun( 'compliment', 'compliments', count, _ ).
noun( 'complin', 'complins', count, _ ).
noun( 'compline', 'complines', count, _ ).
noun( 'component', 'components', count, _ ).
noun( 'comportment', 'comportments', count, _ ).
noun( 'composer', 'composers', count, _ ).
noun( 'composition', 'compositions', both, _ ).
noun( 'compositor', 'compositors', count, _ ).
noun( 'compost', '-', mass, _ ).
noun( 'composure', '-', mass, _ ).
noun( 'compote', 'compotes', both, _ ).
noun( 'compound', 'compounds', count, _ ).
noun( 'comprehensibility', '-', mass, _ ).
noun( 'comprehension', 'comprehensions', both, _ ).
noun( 'comprehensiveness', '-', mass, _ ).
noun( 'compress', 'compresses', count, _ ).
noun( 'compression', '-', mass, _ ).
noun( 'compressor', 'compressors', count, _ ).
noun( 'compromise', 'compromises', both, _ ).
noun( 'comptroller', 'comptrollers', count, _ ).
noun( 'compulsion', '-', mass, _ ).
noun( 'compunction', '-', mass, _ ).
noun( 'computation', 'computations', both, _ ).
noun( 'computer', 'computers', count, _ ).
noun( 'comrade', 'comrades', count, _ ).
noun( 'comradeship', 'comradeships', count, _ ).
noun( 'con', 'cons', count, _ ).
noun( 'con-man', 'con-men', count, _ ).
noun( 'concatenation', 'concatenations', both, _ ).
noun( 'concavity', 'concavities', both, _ ).
noun( 'concealment', '-', mass, _ ).
noun( 'conceit', 'conceits', both, _ ).
noun( 'concentrate', 'concentrates', count, _ ).
noun( 'concentration', 'concentrations', both, _ ).
noun( 'concept', 'concepts', count, _ ).
noun( 'conception', 'conceptions', both, _ ).
noun( 'concern', 'concerns', both, _ ).
noun( 'concert', 'concerts', both, _ ).
noun( 'concert-hall', 'concert-halls', count, _ ).
noun( 'concertina', 'concertinas', count, _ ).
noun( 'concerto', 'concertos', count, _ ).
noun( 'concession', 'concessions', both, _ ).
noun( 'concessionaire', 'concessionaires', count, _ ).
noun( 'conch', 'conches', count, _ ).
noun( 'conchology', '-', mass, _ ).
noun( 'concierge', 'concierges', count, _ ).
noun( 'conciliation', '-', mass, _ ).
noun( 'conciseness', '-', mass, _ ).
noun( 'conclave', 'conclaves', count, _ ).
noun( 'conclusion', 'conclusions', count, _ ).
noun( 'concoction', 'concoctions', both, _ ).
noun( 'concomitant', 'concomitants', count, _ ).
noun( 'concord', 'concords', both, _ ).
noun( 'concordance', 'concordances', both, _ ).
noun( 'concordat', 'concordats', count, _ ).
noun( 'concourse', 'concourses', count, _ ).
noun( 'concrete', '-', mass, _ ).
noun( 'concretion', 'concretions', both, _ ).
noun( 'concubine', 'concubines', count, _ ).
noun( 'concupiscence', '-', mass, _ ).
noun( 'concurrence', 'concurrences', both, _ ).
noun( 'concussion', 'concussions', both, _ ).
noun( 'condemnation', 'condemnations', both, _ ).
noun( 'condensation', 'condensations', both, _ ).
noun( 'condenser', 'condensers', count, _ ).
noun( 'condescension', 'condescensions', both, _ ).
noun( 'condiment', 'condiments', both, _ ).
noun( 'condition', 'conditions', count, _ ).
noun( 'conditioner', 'conditioners', count, _ ).
noun( 'condolence', 'condolences', count, _ ).
noun( 'condominium', 'condominiums', count, _ ).
noun( 'condonation', 'condonations', count, _ ).
noun( 'condor', 'condors', count, _ ).
noun( 'conduct', '-', mass, _ ).
noun( 'conduction', '-', mass, _ ).
noun( 'conductivity', 'conductivities', both, _ ).
noun( 'conductor', 'conductors', count, _ ).
noun( 'conductress', 'conductresses', count, _ ).
noun( 'conduit', 'conduits', count, _ ).
noun( 'cone', 'cones', count, _ ).
noun( 'coney', 'coneys', count, _ ).
noun( 'confab', 'confabs', count, _ ).
noun( 'confabulation', 'confabulations', count, _ ).
noun( 'confection', 'confections', both, _ ).
noun( 'confectioner', 'confectioners', count, _ ).
noun( 'confectionery', 'confectioneries', both, _ ).
noun( 'confederacy', 'confederacies', count, _ ).
noun( 'confederate', 'confederates', count, _ ).
noun( 'confederation', 'confederations', both, _ ).
noun( 'conference', 'conferences', both, _ ).
noun( 'conferment', 'conferments', count, _ ).
noun( 'confession', 'confessions', both, _ ).
noun( 'confessional', 'confessionals', count, _ ).
noun( 'confessor', 'confessors', count, _ ).
noun( 'confetti', 'confetti', mass, _ ).
noun( 'confidant', 'confidants', count, _ ).
noun( 'confidence', 'confidences', both, _ ).
noun( 'confidentiality', 'confidentialities', both, _ ).
noun( 'configuration', 'configurations', count, _ ).
noun( 'confinement', 'confinements', both, _ ).
noun( 'confirmation', 'confirmations', both, _ ).
noun( 'confiscation', 'confiscations', both, _ ).
noun( 'conflagration', 'conflagrations', count, _ ).
noun( 'conflict', 'conflicts', both, _ ).
noun( 'confluence', 'confluences', count, _ ).
noun( 'conformation', 'conformations', count, _ ).
noun( 'conformist', 'conformists', count, _ ).
noun( 'conformity', '-', mass, _ ).
noun( 'confr`ere', 'confr`eres', count, _ ).
noun( 'confrontation', 'confrontations', both, _ ).
noun( 'confusion', '-', mass, _ ).
noun( 'confutation', 'confutations', count, _ ).
noun( 'cong_e', 'cong_es', count, _ ).
noun( 'conger', 'congers', count, _ ).
noun( 'conger-eel', 'conger-eels', count, _ ).
noun( 'congestion', '-', mass, _ ).
noun( 'conglomerate', 'conglomerates', count, _ ).
noun( 'conglomeration', 'conglomerations', both, _ ).
noun( 'congratulation', 'congratulations', count, _ ).
noun( 'congregation', 'congregations', both, _ ).
noun( 'congress', 'congresses', count, _ ).
noun( 'congressman', 'congressmen', count, _ ).
noun( 'congresswoman', 'congresswomen', count, _ ).
noun( 'conifer', 'conifers', count, _ ).
noun( 'conjecture', 'conjectures', both, _ ).
noun( 'conjugation', 'conjugations', both, _ ).
noun( 'conjunction', 'conjunctions', both, _ ).
noun( 'conjunctive', 'conjunctives', count, _ ).
noun( 'conjuncture', 'conjunctures', count, _ ).
noun( 'conjuration', 'conjurations', count, _ ).
noun( 'conjurer', 'conjurers', count, _ ).
noun( 'conjuror', 'conjurors', count, _ ).
noun( 'conk', 'conks', count, _ ).
noun( 'conker', 'conkers', count, _ ).
noun( 'connection', 'connections', both, _ ).
noun( 'connective', 'connectives', count, _ ).
noun( 'connexion', 'connexions', both, _ ).
noun( 'connivance', '-', mass, _ ).
noun( 'connoisseur', 'connoisseurs', count, _ ).
noun( 'connotation', 'connotations', count, _ ).
noun( 'conqueror', 'conquerors', count, _ ).
noun( 'conquest', 'conquests', both, _ ).
noun( 'conquistador', 'conquistadors', count, _ ).
noun( 'consanguinity', '-', mass, _ ).
noun( 'conscience', 'consciences', both, _ ).
noun( 'conscientiousness', '-', mass, _ ).
noun( 'consciousness', '-', mass, _ ).
noun( 'conscript', 'conscripts', count, _ ).
noun( 'conscription', '-', mass, _ ).
noun( 'consecration', 'consecrations', both, _ ).
noun( 'consensus', '-', both, _ ).
noun( 'consent', '-', mass, _ ).
noun( 'consequence', 'consequences', both, _ ).
noun( 'conservancy', 'conservancies', both, _ ).
noun( 'conservation', '-', mass, _ ).
noun( 'conservatism', '-', mass, _ ).
noun( 'conservative', 'conservatives', count, _ ).
noun( 'conservatoire', 'conservatoires', count, _ ).
noun( 'conservatory', 'conservatories', count, _ ).
noun( 'conserve', 'conserves', count, _ ).
noun( 'considerateness', '-', mass, _ ).
noun( 'consideration', 'considerations', both, _ ).
noun( 'consignee', 'consignees', count, _ ).
noun( 'consigner', 'consigners', count, _ ).
noun( 'consignment', 'consignments', both, _ ).
noun( 'consignor', 'consignors', count, _ ).
noun( 'consistence', '-', mass, _ ).
noun( 'consistency', 'consistencies', both, _ ).
noun( 'consistory', 'consistories', count, _ ).
noun( 'consolation', 'consolations', both, _ ).
noun( 'console', 'consoles', count, _ ).
noun( 'consolidation', 'consolidations', both, _ ).
noun( 'consomm_e', 'consomm_es', both, _ ).
noun( 'consonance', '-', mass, _ ).
noun( 'consonant', 'consonants', count, _ ).
noun( 'consort', 'consorts', count, _ ).
noun( 'consortium', 'consortia', count, _ ).
noun( 'conspectus', 'conspectuses', count, _ ).
noun( 'conspicuousness', '-', mass, _ ).
noun( 'conspiracy', 'conspiracies', both, _ ).
noun( 'conspirator', 'conspirators', count, _ ).
noun( 'constable', 'constables', count, _ ).
noun( 'constabulary', 'constabularies', count, _ ).
noun( 'constancy', '-', mass, _ ).
noun( 'constant', 'constants', count, _ ).
noun( 'constellation', 'constellations', count, _ ).
noun( 'consternation', '-', mass, _ ).
noun( 'constipation', '-', mass, _ ).
noun( 'constituency', 'constituencies', count, _ ).
noun( 'constituent', 'constituents', count, _ ).
noun( 'constitution', 'constitutions', both, _ ).
noun( 'constitutional', 'constitutionals', count, _ ).
noun( 'constitutionalism', '-', mass, _ ).
noun( 'constitutionalist', 'constitutionalists', count, _ ).
noun( 'constraint', 'constraints', both, _ ).
noun( 'constriction', 'constrictions', both, _ ).
noun( 'construction', 'constructions', both, _ ).
noun( 'constructor', 'constructors', count, _ ).
noun( 'consubstantiation', '-', mass, _ ).
noun( 'consul', 'consuls', count, _ ).
noun( 'consulate', 'consulates', count, _ ).
noun( 'consulship', 'consulships', count, _ ).
noun( 'consultant', 'consultants', count, _ ).
noun( 'consultation', 'consultations', both, _ ).
noun( 'consumer', 'consumers', count, _ ).
noun( 'consummation', 'consummations', both, _ ).
noun( 'consumption', '-', mass, _ ).
noun( 'consumptive', 'consumptives', count, _ ).
noun( 'cont', '-', proper, _ ).
noun( 'contact', 'contacts', both, _ ).
noun( 'contagion', 'contagions', both, _ ).
noun( 'container', 'containers', count, _ ).
noun( 'containment', '-', mass, _ ).
noun( 'contaminant', 'contaminants', count, _ ).
noun( 'contamination', 'contaminations', both, _ ).
noun( 'contemplation', '-', mass, _ ).
noun( 'contemporary', 'contemporaries', count, _ ).
noun( 'contempt', '-', mass, _ ).
noun( 'contender', 'contenders', count, _ ).
noun( 'content', 'contents', count, _ ).
noun( 'content', '-', mass, _ ).
noun( 'contention', 'contentions', both, _ ).
noun( 'contentment', '-', mass, _ ).
noun( 'contest', 'contests', count, _ ).
noun( 'contestant', 'contestants', count, _ ).
noun( 'context', 'contexts', both, _ ).
noun( 'contiguity', '-', mass, _ ).
noun( 'continence', '-', mass, _ ).
noun( 'continent', 'continents', count, _ ).
noun( 'contingency', 'contingencies', both, _ ).
noun( 'contingent', 'contingents', count, _ ).
noun( 'continuance', '-', count, _ ).
noun( 'continuation', 'continuations', both, _ ).
noun( 'continuity', '-', mass, _ ).
noun( 'contortion', 'contortions', both, _ ).
noun( 'contortionist', 'contortionists', count, _ ).
noun( 'contour', 'contours', count, _ ).
noun( 'contraband', '-', mass, _ ).
noun( 'contrabass', 'contrabasses', count, _ ).
noun( 'contraception', '-', mass, _ ).
noun( 'contraceptive', 'contraceptives', count, _ ).
noun( 'contract', 'contracts', both, _ ).
noun( 'contraction', 'contractions', both, _ ).
noun( 'contractor', 'contractors', count, _ ).
noun( 'contradiction', 'contradictions', both, _ ).
noun( 'contradistinction', 'contradistinctions', count, _ ).
noun( 'contralto', 'contraltos', count, _ ).
noun( 'contraption', 'contraptions', count, _ ).
noun( 'contrariety', 'contrarieties', both, _ ).
noun( 'contrariness', '-', mass, _ ).
noun( 'contrary', 'contraries', both, _ ).
noun( 'contrast', 'contrasts', both, _ ).
noun( 'contravention', 'contraventions', both, _ ).
noun( 'contretemps', 'contretemps', count, _ ).
noun( 'contribution', 'contributions', both, _ ).
noun( 'contributor', 'contributors', count, _ ).
noun( 'contrition', '-', mass, _ ).
noun( 'contrivance', 'contrivances', both, _ ).
noun( 'contriver', 'contrivers', count, _ ).
noun( 'control', 'controls', both, _ ).
noun( 'controller', 'controllers', count, _ ).
noun( 'controversialist', 'controversialists', count, _ ).
noun( 'controversy', 'controversies', both, _ ).
noun( 'contumacy', 'contumacies', both, _ ).
noun( 'contumely', 'contumelies', both, _ ).
noun( 'contusion', 'contusions', count, _ ).
noun( 'conundrum', 'conundrums', count, _ ).
noun( 'conurbation', 'conurbations', count, _ ).
noun( 'convalescence', '-', mass, _ ).
noun( 'convalescent', 'convalescents', count, _ ).
noun( 'convection', '-', mass, _ ).
noun( 'convector', 'convectors', count, _ ).
noun( 'convener', 'conveners', count, _ ).
noun( 'convenience', 'conveniences', both, _ ).
noun( 'convent', 'convents', count, _ ).
noun( 'conventicle', 'conventicles', count, _ ).
noun( 'convention', 'conventions', both, _ ).
noun( 'conventionality', 'conventionalities', both, _ ).
noun( 'convergence', 'convergences', count, _ ).
noun( 'conversation', 'conversations', both, _ ).
noun( 'conversationalist', 'conversationalists', count, _ ).
noun( 'converse', '-', both, _ ).
noun( 'conversion', 'conversions', both, _ ).
noun( 'convert', 'converts', count, _ ).
noun( 'converter', 'converters', count, _ ).
noun( 'convertibility', '-', mass, _ ).
noun( 'convertible', 'convertibles', count, _ ).
noun( 'convexity', '-', mass, _ ).
noun( 'conveyance', 'conveyances', both, _ ).
noun( 'conveyancer', 'conveyancers', count, _ ).
noun( 'conveyer', 'conveyers', count, _ ).
noun( 'conveyer-belt', 'conveyer-belts', count, _ ).
noun( 'conveyor', 'conveyors', count, _ ).
noun( 'convict', 'convicts', count, _ ).
noun( 'conviction', 'convictions', both, _ ).
noun( 'conviviality', 'convivialities', both, _ ).
noun( 'convocation', 'convocations', both, _ ).
noun( 'convolution', 'convolutions', count, _ ).
noun( 'convolvulus', 'convolvuluses', count, _ ).
noun( 'convoy', 'convoys', both, _ ).
noun( 'convulsion', 'convulsions', count, _ ).
noun( 'cony', 'conies', count, _ ).
noun( 'coo', 'coos', count, _ ).
noun( 'cook', 'cooks', count, _ ).
noun( 'cookbook', 'cookbooks', count, _ ).
noun( 'cooker', 'cookers', count, _ ).
noun( 'cookery', '-', mass, _ ).
noun( 'cookery-book', 'cookery-books', count, _ ).
noun( 'cookhouse', 'cookhouses', count, _ ).
noun( 'cookie', 'cookies', count, _ ).
noun( 'cooking', '-', mass, _ ).
noun( 'cooky', 'cookies', count, _ ).
noun( 'cool', '-', mass, _ ).
noun( 'coolant', 'coolants', both, _ ).
noun( 'cooler', 'coolers', count, _ ).
noun( 'coolie', 'coolies', count, _ ).
noun( 'cooling-tower', 'cooling-towers', count, _ ).
noun( 'coolness', '-', mass, _ ).
noun( 'coon', 'coons', count, _ ).
noun( 'coop', 'coops', count, _ ).
noun( 'cooper', 'coopers', count, _ ).
noun( 'cooperation', '-', mass, _ ).
noun( 'cooperative', 'cooperatives', count, _ ).
noun( 'cooperator', 'cooperators', count, _ ).
noun( 'coordinate', 'coordinates', count, _ ).
noun( 'coordination', 'coordinations', both, _ ).
noun( 'coordinator', 'coordinators', count, _ ).
noun( 'coot', 'coots', count, _ ).
noun( 'cop', 'cops', count, _ ).
noun( 'cop-out', 'cop-outs', count, _ ).
noun( 'copartner', 'copartners', count, _ ).
noun( 'copartnership', 'copartnerships', count, _ ).
noun( 'cope', 'copes', count, _ ).
noun( 'copeck', 'copecks', count, _ ).
noun( 'coping', 'copings', count, _ ).
noun( 'coping-stone', 'coping-stones', count, _ ).
noun( 'copper', 'coppers', both, _ ).
noun( 'copperhead', 'copperheads', count, _ ).
noun( 'copperplate', '-', mass, _ ).
noun( 'coppersmith', 'coppersmiths', count, _ ).
noun( 'coppice', 'coppices', count, _ ).
noun( 'copra', '-', mass, _ ).
noun( 'copse', 'copses', count, _ ).
noun( 'copula', 'copulas', count, _ ).
noun( 'copulation', 'copulations', count, _ ).
noun( 'copulative', 'copulatives', count, _ ).
noun( 'copy', 'copies', both, _ ).
noun( 'copybook', 'copybooks', count, _ ).
noun( 'copycat', 'copycats', count, _ ).
noun( 'copyhold', '-', mass, _ ).
noun( 'copyholder', 'copyholders', count, _ ).
noun( 'copyist', 'copyists', count, _ ).
noun( 'copyright', 'copyrights', both, _ ).
noun( 'copywriter', 'copywriters', count, _ ).
noun( 'coquetry', 'coquetries', both, _ ).
noun( 'coquette', 'coquettes', count, _ ).
noun( 'cor anglais', 'cor anglais', count, _ ).
noun( 'coracle', 'coracles', count, _ ).
noun( 'coral', 'corals', both, _ ).
noun( 'coral-reef', 'coral-reefs', count, _ ).
noun( 'corbel', 'corbels', count, _ ).
noun( 'cord', 'cords', both, _ ).
noun( 'cordage', '-', mass, _ ).
noun( 'cordial', 'cordials', both, _ ).
noun( 'cordiality', 'cordialities', both, _ ).
noun( 'cordite', '-', mass, _ ).
noun( 'cordon', 'cordons', count, _ ).
noun( 'corduroy', '-', mass, _ ).
noun( 'core', 'cores', count, _ ).
noun( 'coreligionist', 'coreligionists', count, _ ).
noun( 'corgi', 'corgis', count, _ ).
noun( 'cork', 'corks', both, _ ).
noun( 'corkage', '-', mass, _ ).
noun( 'corker', 'corkers', count, _ ).
noun( 'corkscrew', 'corkscrews', count, _ ).
noun( 'corm', 'corms', count, _ ).
noun( 'cormorant', 'cormorants', count, _ ).
noun( 'corn', 'corns', both, _ ).
noun( 'corn-exchange', 'corn-exchanges', count, _ ).
noun( 'corncob', 'corncobs', count, _ ).
noun( 'corncrake', 'corncrakes', count, _ ).
noun( 'cornea', 'corneas', count, _ ).
noun( 'cornelian', 'cornelians', count, _ ).
noun( 'corner', 'corners', count, _ ).
noun( 'corner-kick', 'corner-kicks', count, _ ).
noun( 'cornerstone', 'cornerstones', count, _ ).
noun( 'cornet', 'cornets', count, _ ).
noun( 'cornflake', 'cornflakes', count, _ ).
noun( 'cornflour', '-', mass, _ ).
noun( 'cornflower', 'cornflowers', count, _ ).
noun( 'cornice', 'cornices', count, _ ).
noun( 'cornpone', '-', mass, _ ).
noun( 'cornstarch', '-', mass, _ ).
noun( 'cornucopia', 'cornucopias', count, _ ).
noun( 'corolla', 'corollas', count, _ ).
noun( 'corollary', 'corollaries', count, _ ).
noun( 'corona', 'coronas', count, _ ).
noun( 'coronary', 'coronaries', count, _ ).
noun( 'coronation', 'coronations', count, _ ).
noun( 'coroner', 'coroners', count, _ ).
noun( 'coronet', 'coronets', count, _ ).
noun( 'corporal', 'corporals', count, _ ).
noun( 'corporation', 'corporations', count, _ ).
noun( 'corps', '-', count, _ ).
noun( 'corps de ballet', '-', count, _ ).
noun( 'corpse', 'corpses', count, _ ).
noun( 'corpulence', '-', mass, _ ).
noun( 'corpus', 'corpora', count, _ ).
noun( 'corpuscle', 'corpuscles', count, _ ).
noun( 'corral', 'corrals', count, _ ).
noun( 'correction', 'corrections', both, _ ).
noun( 'correctitude', '-', mass, _ ).
noun( 'corrective', 'correctives', count, _ ).
noun( 'correctness', '-', mass, _ ).
noun( 'correlation', 'correlations', count, _ ).
noun( 'correlative', 'correlatives', count, _ ).
noun( 'correspondence', 'correspondences', both, _ ).
noun( 'correspondent', 'correspondents', count, _ ).
noun( 'corridor', 'corridors', count, _ ).
noun( 'corrie', 'corries', count, _ ).
noun( 'corrigendum', 'corrigenda', count, _ ).
noun( 'corroboration', '-', mass, _ ).
noun( 'corrosion', '-', mass, _ ).
noun( 'corrosive', 'corrosives', count, _ ).
noun( 'corrugation', 'corrugations', both, _ ).
noun( 'corruptibility', '-', mass, _ ).
noun( 'corruption', '-', mass, _ ).
noun( 'corruptness', '-', mass, _ ).
noun( 'corsage', 'corsages', count, _ ).
noun( 'corsair', 'corsairs', count, _ ).
noun( 'corse', 'corses', count, _ ).
noun( 'corselet', 'corselets', count, _ ).
noun( 'corset', 'corsets', count, _ ).
noun( 'corslet', 'corslets', count, _ ).
noun( 'cort`ege', 'cort`eges', count, _ ).
noun( 'cortege', 'corteges', count, _ ).
noun( 'cortex', 'cortices', count, _ ).
noun( 'cortisone', '-', mass, _ ).
noun( 'corundum', '-', mass, _ ).
noun( 'coruscation', 'coruscations', count, _ ).
noun( 'corv_ee', 'corv_ees', count, _ ).
noun( 'corvette', 'corvettes', count, _ ).
noun( 'cos', 'cos', count, _ ).
noun( 'cosh', 'coshes', count, _ ).
noun( 'cosignatory', 'cosignatories', count, _ ).
noun( 'cosine', 'cosines', count, _ ).
noun( 'cosiness', '-', mass, _ ).
noun( 'cosmetic', 'cosmetics', count, _ ).
noun( 'cosmetician', 'cosmeticians', count, _ ).
noun( 'cosmogony', 'cosmogonies', count, _ ).
noun( 'cosmonaut', 'cosmonauts', count, _ ).
noun( 'cosmopolitan', 'cosmopolitans', count, _ ).
noun( 'cosmos', '-', count, _ ).
noun( 'cost', 'costs', both, _ ).
noun( 'costermonger', 'costermongers', count, _ ).
noun( 'costing', 'costings', both, _ ).
noun( 'costliness', '-', mass, _ ).
noun( 'costume', 'costumes', both, _ ).
noun( 'costumier', 'costumiers', count, _ ).
noun( 'cosy', 'cosies', count, _ ).
noun( 'cot', 'cots', count, _ ).
noun( 'cote', 'cotes', count, _ ).
noun( 'cotenant', 'cotenants', count, _ ).
noun( 'coterie', 'coteries', count, _ ).
noun( 'cotilion', 'cotilions', count, _ ).
noun( 'cotillion', 'cotillions', count, _ ).
noun( 'cottage', 'cottages', count, _ ).
noun( 'cottar', 'cottars', count, _ ).
noun( 'cotter', 'cotters', count, _ ).
noun( 'cotton', '-', mass, _ ).
noun( 'cotton-cake', '-', mass, _ ).
noun( 'cotton-plant', 'cotton-plants', count, _ ).
noun( 'cotton-wool', '-', mass, _ ).
noun( 'cottontail', 'cottontails', count, _ ).
noun( 'cotyledon', 'cotyledons', count, _ ).
noun( 'couch', 'couches', both, _ ).
noun( 'couch-grass', '-', mass, _ ).
noun( 'couchette', 'couchettes', count, _ ).
noun( 'cougar', 'cougars', count, _ ).
noun( 'cough', 'coughs', count, _ ).
noun( 'coulter', 'coulters', count, _ ).
noun( 'council', 'councils', count, _ ).
noun( 'council-board', 'council-boards', count, _ ).
noun( 'council-chamber', 'council-chambers', count, _ ).
noun( 'councillor', 'councillors', count, _ ).
noun( 'counsel', '-', mass, _ ).
noun( 'counsellor', 'counsellors', count, _ ).
noun( 'count', 'counts', both, _ ).
noun( 'countdown', 'countdowns', count, _ ).
noun( 'countenance', 'countenances', both, _ ).
noun( 'counter', 'counters', count, _ ).
noun( 'counter-example', 'counter-examples', count, _ ).
noun( 'counter-revolution', 'counter-revolutions', both, _ ).
noun( 'counter-revolutionary', 'counter-revolutionaries', count, _ ).
noun( 'counteraction', 'counteractions', count, _ ).
noun( 'counterattack', 'counterattacks', count, _ ).
noun( 'counterattraction', 'counterattractions', count, _ ).
noun( 'counterbalance', 'counterbalances', count, _ ).
noun( 'counterblast', 'counterblasts', count, _ ).
noun( 'counterclaim', 'counterclaims', count, _ ).
noun( 'counterespionage', '-', mass, _ ).
noun( 'counterfeit', 'counterfeits', count, _ ).
noun( 'counterfeiter', 'counterfeiters', count, _ ).
noun( 'counterfoil', 'counterfoils', count, _ ).
noun( 'counterintelligence', '-', mass, _ ).
noun( 'counterirritant', 'counterirritants', count, _ ).
noun( 'countermine', 'countermines', count, _ ).
noun( 'counteroffer', 'counteroffers', count, _ ).
noun( 'counterpane', 'counterpanes', count, _ ).
noun( 'counterpart', 'counterparts', count, _ ).
noun( 'counterplot', 'counterplots', count, _ ).
noun( 'counterpoint', 'counterpoints', both, _ ).
noun( 'counterpoise', 'counterpoises', both, _ ).
noun( 'countersign', 'countersigns', count, _ ).
noun( 'countertenor', 'countertenors', count, _ ).
noun( 'countess', 'countesses', count, _ ).
noun( 'counting-house', 'counting-houses', count, _ ).
noun( 'country', 'countries', both, _ ).
noun( 'country-house', 'country-houses', count, _ ).
noun( 'country-seat', 'country-seats', count, _ ).
noun( 'countryman', 'countrymen', count, _ ).
noun( 'countryside', '-', mass, _ ).
noun( 'countrywoman', 'countrywomen', count, _ ).
noun( 'county', 'counties', count, _ ).
noun( 'coup', 'coups', count, _ ).
noun( 'coup d\'etat', 'coups d\'etat', count, _ ).
noun( 'coup de grace', '-', count, _ ).
noun( 'coup_e', 'coup_es', count, _ ).
noun( 'couple', 'couples', count, _ ).
noun( 'couplet', 'couplets', count, _ ).
noun( 'coupling', 'couplings', both, _ ).
noun( 'coupon', 'coupons', count, _ ).
noun( 'courage', '-', mass, _ ).
noun( 'courgette', 'courgettes', count, _ ).
noun( 'courier', 'couriers', count, _ ).
noun( 'course', 'courses', both, _ ).
noun( 'courser', 'coursers', count, _ ).
noun( 'coursing', 'coursings', count, _ ).
noun( 'court', 'courts', both, _ ).
noun( 'court-card', 'court-cards', count, _ ).
noun( 'court-martial', 'courts-martial', count, _ ).
noun( 'courtesan', 'courtesans', count, _ ).
noun( 'courtesy', 'courtesies', both, _ ).
noun( 'courtier', 'courtiers', count, _ ).
noun( 'courtliness', '-', mass, _ ).
noun( 'courtroom', 'courtrooms', count, _ ).
noun( 'courtship', 'courtships', both, _ ).
noun( 'courtyard', 'courtyards', count, _ ).
noun( 'cousin', 'cousins', count, _ ).
noun( 'cove', 'coves', count, _ ).
noun( 'coven', 'covens', count, _ ).
noun( 'covenant', 'covenants', count, _ ).
noun( 'cover', 'covers', both, _ ).
noun( 'cover-up', 'cover-ups', count, _ ).
noun( 'coverage', '-', mass, _ ).
noun( 'covering', 'coverings', count, _ ).
noun( 'coverlet', 'coverlets', count, _ ).
noun( 'covert', 'coverts', count, _ ).
noun( 'covetousness', '-', mass, _ ).
noun( 'covey', 'coveys', count, _ ).
noun( 'cow', 'cows', count, _ ).
noun( 'coward', 'cowards', count, _ ).
noun( 'cowardice', '-', mass, _ ).
noun( 'cowbell', 'cowbells', count, _ ).
noun( 'cowboy', 'cowboys', count, _ ).
noun( 'cowcatcher', 'cowcatchers', count, _ ).
noun( 'cowhand', 'cowhands', count, _ ).
noun( 'cowherd', 'cowherds', count, _ ).
noun( 'cowhide', 'cowhides', both, _ ).
noun( 'cowhouse', 'cowhouses', count, _ ).
noun( 'cowl', 'cowls', count, _ ).
noun( 'cowling', 'cowlings', count, _ ).
noun( 'cowman', 'cowmen', count, _ ).
noun( 'cowpox', 'cowpoxes', count, _ ).
noun( 'cowrie', 'cowries', count, _ ).
noun( 'cowshed', 'cowsheds', count, _ ).
noun( 'cowskin', 'cowskins', count, _ ).
noun( 'cowslip', 'cowslips', count, _ ).
noun( 'cox', 'coxes', count, _ ).
noun( 'coxcomb', 'coxcombs', count, _ ).
noun( 'coxswain', 'coxswains', count, _ ).
noun( 'coyness', '-', mass, _ ).
noun( 'coyote', 'coyotes', count, _ ).
noun( 'coypu', 'coypus', count, _ ).
noun( 'cp', '-', proper, _ ).
noun( 'cr^epe', '-', mass, _ ).
noun( 'cr`eche', 'cr`eches', count, _ ).
noun( 'cr`eme de menthe', '-', mass, _ ).
noun( 'crab', 'crabs', both, _ ).
noun( 'crab-apple', 'crab-apples', count, _ ).
noun( 'crack', 'cracks', count, _ ).
noun( 'crack-down', 'crack-downs', count, _ ).
noun( 'crack-up', 'crack-ups', count, _ ).
noun( 'cracker', 'crackers', count, _ ).
noun( 'crackle', '-', mass, _ ).
noun( 'crackle-china', '-', mass, _ ).
noun( 'crackleware', '-', mass, _ ).
noun( 'crackling', '-', mass, _ ).
noun( 'crackpot', 'crackpots', count, _ ).
noun( 'cracksman', 'cracksmen', count, _ ).
noun( 'cradle', 'cradles', count, _ ).
noun( 'craft', 'crafts', both, _ ).
noun( 'craftiness', '-', mass, _ ).
noun( 'craftsman', 'craftsmen', count, _ ).
noun( 'craftsmanship', '-', mass, _ ).
noun( 'crag', 'crags', count, _ ).
noun( 'cragsman', 'cragsmen', count, _ ).
noun( 'crake', 'crakes', count, _ ).
noun( 'crammer', 'crammers', count, _ ).
noun( 'cramp', 'cramps', both, _ ).
noun( 'cramp-iron', 'cramp-irons', count, _ ).
noun( 'crampon', 'crampons', count, _ ).
noun( 'cranberry', 'cranberries', count, _ ).
noun( 'crane', 'cranes', count, _ ).
noun( 'crane-fly', 'crane-flies', count, _ ).
noun( 'cranium', 'craniums', count, _ ).
noun( 'crank', 'cranks', count, _ ).
noun( 'crankshaft', 'crankshafts', count, _ ).
noun( 'cranny', 'crannies', count, _ ).
noun( 'crap', 'craps', both, _ ).
noun( 'crap-shooting', 'crap-shootings', count, _ ).
noun( 'crape', '-', mass, _ ).
noun( 'craps', 'craps', mass, _ ).
noun( 'crash', 'crashes', both, _ ).
noun( 'crash-dive', 'crash-dives', count, _ ).
noun( 'crash-helmet', 'crash-helmets', count, _ ).
noun( 'crash-landing', 'crash-landings', count, _ ).
noun( 'crate', 'crates', count, _ ).
noun( 'crater', 'craters', count, _ ).
noun( 'cravat', 'cravats', count, _ ).
noun( 'craven', 'cravens', count, _ ).
noun( 'craving', 'cravings', count, _ ).
noun( 'crawfish', 'crawfishes', count, _ ).
noun( 'crawl', '-', count, _ ).
noun( 'crawler', 'crawlers', count, _ ).
noun( 'crayfish', 'crayfishes', count, _ ).
noun( 'crayon', 'crayons', count, _ ).
noun( 'craze', 'crazes', count, _ ).
noun( 'craziness', '-', mass, _ ).
noun( 'creak', 'creaks', count, _ ).
noun( 'cream', 'creams', both, _ ).
noun( 'creamery', 'creameries', count, _ ).
noun( 'crease', 'creases', count, _ ).
noun( 'creation', 'creations', both, _ ).
noun( 'creativeness', '-', mass, _ ).
noun( 'creator', 'creators', count, _ ).
noun( 'creature', 'creatures', count, _ ).
noun( 'credence', '-', mass, _ ).
noun( 'credibility', '-', mass, _ ).
noun( 'credit', 'credits', both, _ ).
noun( 'credit-side', 'credit-sides', count, _ ).
noun( 'credit-worthiness', '-', mass, _ ).
noun( 'creditor', 'creditors', count, _ ).
noun( 'credo', 'credos', count, _ ).
noun( 'credulity', 'credulities', both, _ ).
noun( 'creed', 'creeds', count, _ ).
noun( 'creek', 'creeks', count, _ ).
noun( 'creel', 'creels', count, _ ).
noun( 'creep', 'creeps', count, _ ).
noun( 'creeper', 'creepers', count, _ ).
noun( 'cremation', 'cremations', both, _ ).
noun( 'crematorium', 'crematoriums', count, _ ).
noun( 'crematory', 'crematories', count, _ ).
noun( 'creosote', '-', mass, _ ).
noun( 'crepe', '-', mass, _ ).
noun( 'crepitation', 'crepitations', count, _ ).
noun( 'crescendo', 'crescendos', count, _ ).
noun( 'crescent', 'crescents', count, _ ).
noun( 'cress', '-', mass, _ ).
noun( 'crest', 'crests', count, _ ).
noun( 'cretin', 'cretins', count, _ ).
noun( 'cretonne', '-', mass, _ ).
noun( 'crevasse', 'crevasses', count, _ ).
noun( 'crevice', 'crevices', count, _ ).
noun( 'crew', 'crews', count, _ ).
noun( 'crew-cut', 'crew-cuts', count, _ ).
noun( 'crew-neck', 'crew-necks', count, _ ).
noun( 'crib', 'cribs', count, _ ).
noun( 'cribbage', '-', mass, _ ).
noun( 'cribbage-board', 'cribbage-boards', count, _ ).
noun( 'crick', 'cricks', count, _ ).
noun( 'cricket', 'crickets', both, _ ).
noun( 'cricketer', 'cricketers', count, _ ).
noun( 'crier', 'criers', count, _ ).
noun( 'crime', 'crimes', both, _ ).
noun( 'criminal', 'criminals', count, _ ).
noun( 'criminology', '-', mass, _ ).
noun( 'crimson', 'crimsons', both, _ ).
noun( 'crinkle', 'crinkles', count, _ ).
noun( 'crinoline', 'crinolines', both, _ ).
noun( 'cripple', 'cripples', count, _ ).
noun( 'crisis', 'crises', count, _ ).
noun( 'crisp', 'crisps', count, _ ).
noun( 'crispness', '-', mass, _ ).
noun( 'criterion', 'criterions', count, _ ).
noun( 'critic', 'critics', count, _ ).
noun( 'criticism', 'criticisms', both, _ ).
noun( 'critique', 'critiques', count, _ ).
noun( 'croak', 'croaks', count, _ ).
noun( 'crochet', '-', mass, _ ).
noun( 'crochet-hook', 'crochet-hooks', count, _ ).
noun( 'crock', 'crocks', count, _ ).
noun( 'crockery', '-', mass, _ ).
noun( 'crocodile', 'crocodiles', count, _ ).
noun( 'crocus', 'crocuses', count, _ ).
noun( 'croft', 'crofts', count, _ ).
noun( 'crofter', 'crofters', count, _ ).
noun( 'cromlech', 'cromlechs', count, _ ).
noun( 'crone', 'crones', count, _ ).
noun( 'crony', 'cronies', count, _ ).
noun( 'crook', 'crooks', count, _ ).
noun( 'crookedness', '-', mass, _ ).
noun( 'crooner', 'crooners', count, _ ).
noun( 'crop', 'crops', count, _ ).
noun( 'crop-dusting', '-', mass, _ ).
noun( 'cropper', 'croppers', count, _ ).
noun( 'croquet', '-', mass, _ ).
noun( 'croquette', 'croquettes', count, _ ).
noun( 'crore', 'crores', count, _ ).
noun( 'crosier', 'crosiers', count, _ ).
noun( 'cross', 'crosses', count, _ ).
noun( 'cross-bench', 'cross-benches', count, _ ).
noun( 'cross-bencher', 'cross-benchers', count, _ ).
noun( 'cross-division', 'cross-divisions', both, _ ).
noun( 'cross-examination', 'cross-examinations', count, _ ).
noun( 'cross-examiner', 'cross-examiners', count, _ ).
noun( 'cross-fertilization', 'cross-fertilizations', count, _ ).
noun( 'cross-heading', 'cross-headings', count, _ ).
noun( 'cross-index', 'cross-indexes', count, _ ).
noun( 'cross-reference', 'cross-references', count, _ ).
noun( 'cross-section', 'cross-sections', count, _ ).
noun( 'cross-stitch', 'cross-stitches', both, _ ).
noun( 'crossbar', 'crossbars', count, _ ).
noun( 'crossbeam', 'crossbeams', count, _ ).
noun( 'crossbow', 'crossbows', count, _ ).
noun( 'crossbreed', 'crossbreeds', count, _ ).
noun( 'crosscheck', 'crosschecks', count, _ ).
noun( 'crosscurrent', 'crosscurrents', count, _ ).
noun( 'crosscut', 'crosscuts', count, _ ).
noun( 'crosse', 'crosses', count, _ ).
noun( 'crossfire', 'crossfires', both, _ ).
noun( 'crossing', 'crossings', both, _ ).
noun( 'crossness', '-', mass, _ ).
noun( 'crosspatch', 'crosspatches', count, _ ).
noun( 'crosspiece', 'crosspieces', count, _ ).
noun( 'crossroad', 'crossroads', count, _ ).
noun( 'crosstalk', '-', mass, _ ).
noun( 'crosswalk', 'crosswalks', count, _ ).
noun( 'crosswind', 'crosswinds', count, _ ).
noun( 'crossword', 'crosswords', count, _ ).
noun( 'crotch', 'crotches', count, _ ).
noun( 'crotchet', 'crotchets', count, _ ).
noun( 'crouch', 'crouches', count, _ ).
noun( 'croup', 'croups', both, _ ).
noun( 'croupier', 'croupiers', count, _ ).
noun( 'crow', 'crows', count, _ ).
noun( 'crow\'s-nest', 'crow\'s-nests', count, _ ).
noun( 'crowbar', 'crowbars', count, _ ).
noun( 'crowd', 'crowds', count, _ ).
noun( 'crown', 'crowns', count, _ ).
noun( 'crown-land', 'crown-lands', count, _ ).
noun( 'crozier', 'croziers', count, _ ).
noun( 'crucible', 'crucibles', count, _ ).
noun( 'crucifix', 'crucifixes', count, _ ).
noun( 'crucifixion', 'crucifixions', both, _ ).
noun( 'crudeness', '-', mass, _ ).
noun( 'crudity', 'crudities', both, _ ).
noun( 'cruelty', 'cruelties', both, _ ).
noun( 'cruet', 'cruets', count, _ ).
noun( 'cruet-stand', 'cruet-stands', count, _ ).
noun( 'cruise', 'cruises', count, _ ).
noun( 'cruiser', 'cruisers', count, _ ).
noun( 'crumb', 'crumbs', both, _ ).
noun( 'crumpet', 'crumpets', count, _ ).
noun( 'crunch', 'crunches', count, _ ).
noun( 'crupper', 'cruppers', count, _ ).
noun( 'crusade', 'crusades', count, _ ).
noun( 'crusader', 'crusaders', count, _ ).
noun( 'cruse', 'cruses', count, _ ).
noun( 'crush', '-', mass, _ ).
noun( 'crust', 'crusts', both, _ ).
noun( 'crustacean', 'crustaceans', count, _ ).
noun( 'crutch', 'crutches', count, _ ).
noun( 'crux', 'cruxes', count, _ ).
noun( 'cruzeiro', 'cruzeiros', count, _ ).
noun( 'cry', 'cries', count, _ ).
noun( 'crybaby', 'crybabies', count, _ ).
noun( 'crypt', 'crypts', count, _ ).
noun( 'cryptogram', 'cryptograms', count, _ ).
noun( 'crystal', 'crystals', both, _ ).
noun( 'crystal-gazing', '-', mass, _ ).
noun( 'crystallization', 'crystallizations', both, _ ).
noun( 'crystallography', '-', mass, _ ).
noun( 'cub', 'cubs', count, _ ).
noun( 'cubbyhole', 'cubbyholes', count, _ ).
noun( 'cube', 'cubes', count, _ ).
noun( 'cubicle', 'cubicles', count, _ ).
noun( 'cubism', '-', mass, _ ).
noun( 'cubist', 'cubists', count, _ ).
noun( 'cubit', 'cubits', count, _ ).
noun( 'cuckold', 'cuckolds', count, _ ).
noun( 'cuckoo', 'cuckoos', count, _ ).
noun( 'cuckoo-clock', 'cuckoo-clocks', count, _ ).
noun( 'cucumber', 'cucumbers', both, _ ).
noun( 'cud', '-', mass, _ ).
noun( 'cuddle', 'cuddles', count, _ ).
noun( 'cudgel', 'cudgels', count, _ ).
noun( 'cue', 'cues', count, _ ).
noun( 'cuff', 'cuffs', count, _ ).
noun( 'cuirass', 'cuirasses', count, _ ).
noun( 'cuirassier', 'cuirassiers', count, _ ).
noun( 'cuisine', '-', mass, _ ).
noun( 'cul-de-sac', 'cul-de-sacs', count, _ ).
noun( 'cull', 'culls', count, _ ).
noun( 'cullender', 'cullenders', count, _ ).
noun( 'culmination', 'culminations', count, _ ).
noun( 'culpability', '-', mass, _ ).
noun( 'culprit', 'culprits', count, _ ).
noun( 'cult', 'cults', count, _ ).
noun( 'cultivation', '-', mass, _ ).
noun( 'cultivator', 'cultivators', count, _ ).
noun( 'culture', 'cultures', both, _ ).
noun( 'culvert', 'culverts', count, _ ).
noun( 'cummerbund', 'cummerbunds', count, _ ).
noun( 'cumulus', 'cumuli', count, _ ).
noun( 'cunning', '-', mass, _ ).
noun( 'cunt', 'cunts', count, _ ).
noun( 'cup', 'cups', count, _ ).
noun( 'cup-bearer', 'cup-bearers', count, _ ).
noun( 'cup-final', 'cup-finals', count, _ ).
noun( 'cup-tie', 'cup-ties', count, _ ).
noun( 'cupboard', 'cupboards', count, _ ).
noun( 'cupboard-love', '-', mass, _ ).
noun( 'cupful', 'cupfuls', count, _ ).
noun( 'cupidity', '-', mass, _ ).
noun( 'cupola', 'cupolas', count, _ ).
noun( 'cuppa', 'cuppas', count, _ ).
noun( 'cupping', '-', mass, _ ).
noun( 'cupro-nickel', '-', mass, _ ).
noun( 'cur', 'curs', count, _ ).
noun( 'cur_e', 'cur_es', count, _ ).
noun( 'cura<cao', '-', mass, _ ).
noun( 'cura<coa', '-', mass, _ ).
noun( 'curability', '-', mass, _ ).
noun( 'curacy', 'curacies', count, _ ).
noun( 'curate', 'curates', count, _ ).
noun( 'curator', 'curators', count, _ ).
noun( 'curb', 'curbs', count, _ ).
noun( 'curd', 'curds', both, _ ).
noun( 'cure', 'cures', count, _ ).
noun( 'cure-all', 'cure-alls', count, _ ).
noun( 'curfew', 'curfews', count, _ ).
noun( 'curio', 'curios', count, _ ).
noun( 'curiosity', 'curiosities', both, _ ).
noun( 'curl', 'curls', both, _ ).
noun( 'curler', 'curlers', count, _ ).
noun( 'curlew', 'curlews', count, _ ).
noun( 'curling', '-', mass, _ ).
noun( 'curmudgeon', 'curmudgeons', count, _ ).
noun( 'currant', 'currants', count, _ ).
noun( 'currency', 'currencies', both, _ ).
noun( 'current', 'currents', count, _ ).
noun( 'curriculum', 'curriculums', count, _ ).
noun( 'curriculum vitae', '-', count, _ ).
noun( 'curry', 'curries', both, _ ).
noun( 'curry-powder', 'curry-powders', count, _ ).
noun( 'curse', 'curses', count, _ ).
noun( 'curtailment', 'curtailments', both, _ ).
noun( 'curtain', 'curtains', count, _ ).
noun( 'curtain-call', 'curtain-calls', count, _ ).
noun( 'curtain-lecture', 'curtain-lectures', count, _ ).
noun( 'curtain-raiser', 'curtain-raisers', count, _ ).
noun( 'curtness', '-', mass, _ ).
noun( 'curtsey', 'curtseys', count, _ ).
noun( 'curtsy', 'curtsies', count, _ ).
noun( 'curvature', '-', mass, _ ).
noun( 'curve', 'curves', count, _ ).
noun( 'cushion', 'cushions', count, _ ).
noun( 'cusp', 'cusps', count, _ ).
noun( 'cuspidor', 'cuspidors', count, _ ).
noun( 'cuss', 'cusses', count, _ ).
noun( 'cussedness', '-', mass, _ ).
noun( 'custard', 'custards', both, _ ).
noun( 'custodian', 'custodians', count, _ ).
noun( 'custody', '-', mass, _ ).
noun( 'custom', 'customs', both, _ ).
noun( 'customer', 'customers', count, _ ).
noun( 'cut', 'cuts', count, _ ).
noun( 'cut-out', 'cut-outs', count, _ ).
noun( 'cut-throat', 'cut-throats', count, _ ).
noun( 'cutback', 'cutbacks', count, _ ).
noun( 'cuteness', '-', mass, _ ).
noun( 'cuticle', 'cuticles', count, _ ).
noun( 'cutlass', 'cutlasses', count, _ ).
noun( 'cutler', 'cutlers', count, _ ).
noun( 'cutlery', '-', mass, _ ).
noun( 'cutlet', 'cutlets', count, _ ).
noun( 'cutpurse', 'cutpurses', count, _ ).
noun( 'cutter', 'cutters', count, _ ).
noun( 'cutting', 'cuttings', both, _ ).
noun( 'cutting-room', 'cutting-rooms', count, _ ).
noun( 'cuttlefish', 'cuttlefishes', both, _ ).
noun( 'cutworm', 'cutworms', count, _ ).
noun( 'cwt', 'cwt', count, _ ).
noun( 'cyanide', '-', mass, _ ).
noun( 'cybernetics', 'cybernetics', mass, _ ).
noun( 'cyclamen', 'cyclamens', count, _ ).
noun( 'cycle', 'cycles', count, _ ).
noun( 'cyclist', 'cyclists', count, _ ).
noun( 'cyclone', 'cyclones', count, _ ).
noun( 'cyclopaedia', 'cyclopaedias', count, _ ).
noun( 'cyclostyle', 'cyclostyles', count, _ ).
noun( 'cyclotron', 'cyclotrons', count, _ ).
noun( 'cyder', 'cyders', both, _ ).
noun( 'cygnet', 'cygnets', count, _ ).
noun( 'cylinder', 'cylinders', count, _ ).
noun( 'cymbal', 'cymbals', count, _ ).
noun( 'cynic', 'cynics', count, _ ).
noun( 'cynicism', 'cynicisms', both, _ ).
noun( 'cynosure', 'cynosures', count, _ ).
noun( 'cypher', 'cyphers', count, _ ).
noun( 'cypress', 'cypresses', count, _ ).
noun( 'cyst', 'cysts', count, _ ).
noun( 'czar', 'czars', count, _ ).
noun( 'czarina', 'czarinas', count, _ ).
noun( 'd', '-', count, _ ).
noun( 'd_eb^acle', 'd_eb^acles', count, _ ).
noun( 'd_ebris', '-', mass, _ ).
noun( 'd_ebut', 'd_ebuts', count, _ ).
noun( 'd_ebutante', 'd_ebutantes', count, _ ).
noun( 'd_ecor', '-', count, _ ).
noun( 'd_emarche', 'd_emarches', count, _ ).
noun( 'd_enouement', 'd_enouements', count, _ ).
noun( 'd_eshabill_e', '-', mass, _ ).
noun( 'd_etente', '-', mass, _ ).
noun( 'dab', 'dabs', count, _ ).
noun( 'dabbler', 'dabblers', count, _ ).
noun( 'dace', 'dace', count, _ ).
noun( 'dacha', 'dachas', count, _ ).
noun( 'dachshund', 'dachshunds', count, _ ).
noun( 'dacoit', 'dacoits', count, _ ).
noun( 'dacoity', 'dacoities', count, _ ).
noun( 'dactyl', 'dactyls', count, _ ).
noun( 'dad', 'dads', count, _ ).
noun( 'daddy', 'daddies', count, _ ).
noun( 'daddy-longlegs', 'daddy-longlegs', count, _ ).
noun( 'dado', 'dados', count, _ ).
noun( 'daemon', 'daemons', count, _ ).
noun( 'daffodil', 'daffodils', count, _ ).
noun( 'dagger', 'daggers', count, _ ).
noun( 'dago', 'dagos', count, _ ).
noun( 'daguerreotype', 'daguerreotypes', count, _ ).
noun( 'dahlia', 'dahlias', count, _ ).
noun( 'daily', 'dailies', count, _ ).
noun( 'daintiness', '-', mass, _ ).
noun( 'dainty', 'dainties', count, _ ).
noun( 'dairy', 'dairies', count, _ ).
noun( 'dairy-farm', 'dairy-farms', count, _ ).
noun( 'dairy-farming', '-', mass, _ ).
noun( 'dairying', '-', mass, _ ).
noun( 'dairymaid', 'dairymaids', count, _ ).
noun( 'dairyman', 'dairymen', count, _ ).
noun( 'dais', 'daises', count, _ ).
noun( 'daisy', 'daisies', count, _ ).
noun( 'dale', 'dales', count, _ ).
noun( 'dalesman', 'dalesmen', count, _ ).
noun( 'dalliance', '-', mass, _ ).
noun( 'dalmatian', 'dalmatians', count, _ ).
noun( 'dam', 'dams', count, _ ).
noun( 'damage', 'damages', both, _ ).
noun( 'damask', '-', mass, _ ).
noun( 'dame', 'dames', count, _ ).
noun( 'damn', '-', count, _ ).
noun( 'damnation', '-', mass, _ ).
noun( 'damp', '-', mass, _ ).
noun( 'damper', 'dampers', count, _ ).
noun( 'dampness', '-', mass, _ ).
noun( 'damsel', 'damsels', count, _ ).
noun( 'damson', 'damsons', count, _ ).
noun( 'dance', 'dances', count, _ ).
noun( 'dance-band', 'dance-bands', count, _ ).
noun( 'dance-hall', 'dance-halls', count, _ ).
noun( 'dance-orchestra', 'dance-orchestras', count, _ ).
noun( 'dancer', 'dancers', count, _ ).
noun( 'dancing', '-', mass, _ ).
noun( 'dandelion', 'dandelions', count, _ ).
noun( 'dander', '-', count, _ ).
noun( 'dandruff', '-', mass, _ ).
noun( 'dandy', 'dandies', count, _ ).
noun( 'danger', 'dangers', both, _ ).
noun( 'danse macabre', '-', count, _ ).
noun( 'daphne', 'daphnes', count, _ ).
noun( 'dapple-grey', 'dapple-greys', count, _ ).
noun( 'dare', 'dares', count, _ ).
noun( 'daredevil', 'daredevils', count, _ ).
noun( 'daring', '-', mass, _ ).
noun( 'dark', '-', mass, _ ).
noun( 'darkness', '-', mass, _ ).
noun( 'darkroom', 'darkrooms', count, _ ).
noun( 'darling', 'darlings', count, _ ).
noun( 'darn', 'darns', count, _ ).
noun( 'darning', '-', mass, _ ).
noun( 'darning-needle', 'darning-needles', count, _ ).
noun( 'dart', 'darts', count, _ ).
noun( 'dash', 'dashes', both, _ ).
noun( 'dashboard', 'dashboards', count, _ ).
noun( 'dastard', 'dastards', count, _ ).
noun( 'data', 'data', both, _ ).
noun( 'date', 'dates', both, _ ).
noun( 'dateline', 'datelines', count, _ ).
noun( 'dative', 'datives', count, _ ).
noun( 'datum', 'data', count, _ ).
noun( 'daub', 'daubs', both, _ ).
noun( 'dauber', 'daubers', count, _ ).
noun( 'daughter', 'daughters', count, _ ).
noun( 'daughter-in-law', 'daughters-in-law', count, _ ).
noun( 'dauphin', 'dauphins', count, _ ).
noun( 'davenport', 'davenports', count, _ ).
noun( 'davit', 'davits', count, _ ).
noun( 'daw', 'daws', count, _ ).
noun( 'dawdler', 'dawdlers', count, _ ).
noun( 'dawn', 'dawns', both, _ ).
noun( 'day', 'days', both, _ ).
noun( 'day-boarder', 'day-boarders', count, _ ).
noun( 'day-labourer', 'day-labourers', count, _ ).
noun( 'day-return', 'day-returns', count, _ ).
noun( 'day-school', 'day-schools', count, _ ).
noun( 'daybook', 'daybooks', count, _ ).
noun( 'dayboy', 'dayboys', count, _ ).
noun( 'daybreak', 'daybreaks', count, _ ).
noun( 'daydream', 'daydreams', count, _ ).
noun( 'daygirl', 'daygirls', count, _ ).
noun( 'daylight', '-', mass, _ ).
noun( 'daylight-saving', '-', mass, _ ).
noun( 'dayspring', 'daysprings', count, _ ).
noun( 'daytime', 'daytimes', count, _ ).
noun( 'daze', '-', count, _ ).
noun( 'dazzle', '-', mass, _ ).
noun( 'de-escalation', 'de-escalations', count, _ ).
noun( 'deacon', 'deacons', count, _ ).
noun( 'deaconess', 'deaconesses', count, _ ).
noun( 'deadline', 'deadlines', count, _ ).
noun( 'deadlock', 'deadlocks', both, _ ).
noun( 'deaf mute', 'deaf mutes', count, _ ).
noun( 'deaf-aid', 'deaf-aids', count, _ ).
noun( 'deafness', '-', mass, _ ).
noun( 'deal', 'deals', count, _ ).
noun( 'dealer', 'dealers', count, _ ).
noun( 'dealing', 'dealings', both, _ ).
noun( 'dean', 'deans', count, _ ).
noun( 'deanery', 'deaneries', count, _ ).
noun( 'dear', 'dears', count, _ ).
noun( 'dearest', '-', count, _ ).
noun( 'dearie', 'dearies', count, _ ).
noun( 'dearness', '-', mass, _ ).
noun( 'dearth', 'dearths', count, _ ).
noun( 'deary', 'dearies', count, _ ).
noun( 'death', 'deaths', both, _ ).
noun( 'death-mask', 'death-masks', count, _ ).
noun( 'death-rate', 'death-rates', count, _ ).
noun( 'death-roll', 'death-rolls', count, _ ).
noun( 'death-warrant', 'death-warrants', count, _ ).
noun( 'deathbed', 'deathbeds', count, _ ).
noun( 'deathblow', 'deathblows', count, _ ).
noun( 'deathtrap', 'deathtraps', count, _ ).
noun( 'deb', 'debs', count, _ ).
noun( 'debarkation', 'debarkations', count, _ ).
noun( 'debasement', 'debasements', count, _ ).
noun( 'debate', 'debates', both, _ ).
noun( 'debater', 'debaters', count, _ ).
noun( 'debauch', 'debauches', count, _ ).
noun( 'debauchee', 'debauchees', count, _ ).
noun( 'debauchery', 'debaucheries', both, _ ).
noun( 'debenture', 'debentures', count, _ ).
noun( 'debility', '-', mass, _ ).
noun( 'debit', 'debits', count, _ ).
noun( 'debit-side', 'debit-sides', count, _ ).
noun( 'debris', '-', mass, _ ).
noun( 'debt', 'debts', both, _ ).
noun( 'debtor', 'debtors', count, _ ).
noun( 'debut', 'debuts', count, _ ).
noun( 'debutante', 'debutantes', count, _ ).
noun( 'decade', 'decades', count, _ ).
noun( 'decadence', '-', mass, _ ).
noun( 'decadent', 'decadents', count, _ ).
noun( 'decanter', 'decanters', count, _ ).
noun( 'decapitation', 'decapitations', count, _ ).
noun( 'decasyllable', 'decasyllables', count, _ ).
noun( 'decay', '-', mass, _ ).
noun( 'decease', '-', mass, _ ).
noun( 'deceit', 'deceits', both, _ ).
noun( 'deceitfulness', '-', mass, _ ).
noun( 'deceiver', 'deceivers', count, _ ).
noun( 'decency', 'decencies', both, _ ).
noun( 'decentralization', '-', mass, _ ).
noun( 'deception', 'deceptions', both, _ ).
noun( 'decibel', 'decibels', count, _ ).
noun( 'decimalization', '-', mass, _ ).
noun( 'decision', 'decisions', both, _ ).
noun( 'deck', 'decks', count, _ ).
noun( 'decker', 'deckers', count, _ ).
noun( 'declamation', 'declamations', both, _ ).
noun( 'declaration', 'declarations', both, _ ).
noun( 'declassification', 'declassifications', both, _ ).
noun( 'declension', 'declensions', both, _ ).
noun( 'declination', 'declinations', count, _ ).
noun( 'decline', 'declines', count, _ ).
noun( 'declivity', 'declivities', count, _ ).
noun( 'decoder', 'decoders', count, _ ).
noun( 'decolonization', '-', mass, _ ).
noun( 'decomposition', 'decompositions', both, _ ).
noun( 'decompression', 'decompressions', count, _ ).
noun( 'decontamination', '-', mass, _ ).
noun( 'decoration', 'decorations', both, _ ).
noun( 'decorator', 'decorators', count, _ ).
noun( 'decorum', '-', mass, _ ).
noun( 'decoy', 'decoys', count, _ ).
noun( 'decrease', 'decreases', both, _ ).
noun( 'decree', 'decrees', count, _ ).
noun( 'decree nisi', '-', count, _ ).
noun( 'decrepitude', '-', mass, _ ).
noun( 'dedication', 'dedications', both, _ ).
noun( 'deduction', 'deductions', both, _ ).
noun( 'deed', 'deeds', count, _ ).
noun( 'deed-box', 'deed-boxes', count, _ ).
noun( 'deedpoll', 'deedpolls', both, _ ).
noun( 'deep', 'deeps', count, _ ).
noun( 'deep-freeze', 'deep-freezes', count, _ ).
noun( 'deepness', '-', mass, _ ).
noun( 'deer', 'deer', count, _ ).
noun( 'deerskin', 'deerskins', count, _ ).
noun( 'deerstalker', 'deerstalkers', count, _ ).
noun( 'deerstalking', '-', mass, _ ).
noun( 'defacement', 'defacements', both, _ ).
noun( 'defalcation', 'defalcations', both, _ ).
noun( 'defamation', '-', mass, _ ).
noun( 'default', '-', mass, _ ).
noun( 'defaulter', 'defaulters', count, _ ).
noun( 'defeat', 'defeats', both, _ ).
noun( 'defeatism', '-', mass, _ ).
noun( 'defeatist', 'defeatists', count, _ ).
noun( 'defecation', 'defecations', both, _ ).
noun( 'defect', 'defects', count, _ ).
noun( 'defection', 'defections', both, _ ).
noun( 'defectiveness', '-', mass, _ ).
noun( 'defector', 'defectors', count, _ ).
noun( 'defence', 'defences', both, _ ).
noun( 'defencelessness', '-', mass, _ ).
noun( 'defendant', 'defendants', count, _ ).
noun( 'defender', 'defenders', count, _ ).
noun( 'defensive', '-', count, _ ).
noun( 'deference', '-', mass, _ ).
noun( 'deferment', 'deferments', count, _ ).
noun( 'defiance', '-', mass, _ ).
noun( 'deficiency', 'deficiencies', both, _ ).
noun( 'deficit', 'deficits', count, _ ).
noun( 'defile', 'defiles', count, _ ).
noun( 'defilement', '-', mass, _ ).
noun( 'definition', 'definitions', both, _ ).
noun( 'deflation', '-', mass, _ ).
noun( 'deflection', 'deflections', count, _ ).
noun( 'defoliant', 'defoliants', count, _ ).
noun( 'defoliation', 'defoliations', count, _ ).
noun( 'deformity', 'deformities', both, _ ).
noun( 'defrayal', 'defrayals', count, _ ).
noun( 'defrayment', 'defrayments', count, _ ).
noun( 'defroster', 'defrosters', count, _ ).
noun( 'deftness', '-', mass, _ ).
noun( 'degeneracy', '-', mass, _ ).
noun( 'degenerate', 'degenerates', count, _ ).
noun( 'degeneration', '-', mass, _ ).
noun( 'degradation', 'degradations', both, _ ).
noun( 'degree', 'degrees', both, _ ).
noun( 'deification', '-', mass, _ ).
noun( 'deism', '-', mass, _ ).
noun( 'deist', 'deists', count, _ ).
noun( 'deity', 'deities', both, _ ).
noun( 'dejection', '-', mass, _ ).
noun( 'dekko', '-', count, _ ).
noun( 'delay', 'delays', both, _ ).
noun( 'delayed-action', '-', mass, _ ).
noun( 'delectation', '-', mass, _ ).
noun( 'delegacy', 'delegacies', count, _ ).
noun( 'delegate', 'delegates', count, _ ).
noun( 'delegation', 'delegations', both, _ ).
noun( 'deletion', 'deletions', both, _ ).
noun( 'delf', '-', mass, _ ).
noun( 'delft', '-', mass, _ ).
noun( 'deliberation', 'deliberations', both, _ ).
noun( 'delicacy', 'delicacies', both, _ ).
noun( 'delicatessen', 'delicatessens', both, _ ).
noun( 'delight', 'delights', both, _ ).
noun( 'delimitation', 'delimitations', both, _ ).
noun( 'delineation', 'delineations', both, _ ).
noun( 'delinquency', 'delinquencies', both, _ ).
noun( 'delinquent', 'delinquents', count, _ ).
noun( 'delirium', '-', mass, _ ).
noun( 'delirium tremens', '-', count, _ ).
noun( 'deliverance', 'deliverances', both, _ ).
noun( 'deliverer', 'deliverers', count, _ ).
noun( 'delivery', 'deliveries', both, _ ).
noun( 'dell', 'dells', count, _ ).
noun( 'delphinium', 'delphiniums', count, _ ).
noun( 'delta', 'deltas', count, _ ).
noun( 'deluge', 'deluges', count, _ ).
noun( 'delusion', 'delusions', both, _ ).
noun( 'demagnetization', '-', mass, _ ).
noun( 'demagogue', 'demagogues', count, _ ).
noun( 'demagogy', '-', mass, _ ).
noun( 'demand', 'demands', both, _ ).
noun( 'demarcation', '-', mass, _ ).
noun( 'demeanour', '-', mass, _ ).
noun( 'demerara', '-', mass, _ ).
noun( 'demerit', 'demerits', count, _ ).
noun( 'demesne', 'demesnes', both, _ ).
noun( 'demigod', 'demigods', count, _ ).
noun( 'demijohn', 'demijohns', count, _ ).
noun( 'demimondaine', 'demimondaines', count, _ ).
noun( 'demimonde', '-', count, _ ).
noun( 'demise', '-', count, _ ).
noun( 'demister', 'demisters', count, _ ).
noun( 'demo', 'demos', count, _ ).
noun( 'demobilization', '-', mass, _ ).
noun( 'democracy', 'democracies', both, _ ).
noun( 'democrat', 'democrats', count, _ ).
noun( 'democratization', '-', mass, _ ).
noun( 'demography', '-', mass, _ ).
noun( 'demolition', 'demolitions', both, _ ).
noun( 'demon', 'demons', count, _ ).
noun( 'demonetization', 'demonetizations', both, _ ).
noun( 'demoniac', 'demoniacs', count, _ ).
noun( 'demonstrability', '-', mass, _ ).
noun( 'demonstration', 'demonstrations', both, _ ).
noun( 'demonstrator', 'demonstrators', count, _ ).
noun( 'demoralization', '-', mass, _ ).
noun( 'demotion', 'demotions', both, _ ).
noun( 'demur', '-', mass, _ ).
noun( 'demureness', '-', mass, _ ).
noun( 'den', 'dens', count, _ ).
noun( 'denationalization', '-', mass, _ ).
noun( 'denial', 'denials', both, _ ).
noun( 'denier', 'denier', count, _ ).
noun( 'denigration', '-', mass, _ ).
noun( 'denim', '-', mass, _ ).
noun( 'denizen', 'denizens', count, _ ).
noun( 'denomination', 'denominations', count, _ ).
noun( 'denominator', 'denominators', count, _ ).
noun( 'denseness', '-', mass, _ ).
noun( 'density', 'densities', both, _ ).
noun( 'dent', 'dents', count, _ ).
noun( 'dentifrice', '-', mass, _ ).
noun( 'dentist', 'dentists', count, _ ).
noun( 'dentistry', '-', mass, _ ).
noun( 'denture', 'dentures', count, _ ).
noun( 'denudation', '-', mass, _ ).
noun( 'denunciation', 'denunciations', both, _ ).
noun( 'deodar', 'deodars', count, _ ).
noun( 'deodorant', 'deodorants', count, _ ).
noun( 'dep', '-', proper, _ ).
noun( 'departed', 'departed', count, _ ).
noun( 'department', 'departments', count, _ ).
noun( 'departure', 'departures', both, _ ).
noun( 'dependant', 'dependants', count, _ ).
noun( 'dependence', '-', mass, _ ).
noun( 'dependency', 'dependencies', count, _ ).
noun( 'dependent', 'dependents', count, _ ).
noun( 'depiction', 'depictions', count, _ ).
noun( 'depilatory', 'depilatories', both, _ ).
noun( 'depletion', '-', mass, _ ).
noun( 'deployment', 'deployments', count, _ ).
noun( 'deponent', 'deponents', count, _ ).
noun( 'depopulation', '-', mass, _ ).
noun( 'deportation', 'deportations', both, _ ).
noun( 'deportee', 'deportees', count, _ ).
noun( 'deportment', '-', mass, _ ).
noun( 'deposit', 'deposits', count, _ ).
noun( 'deposition', 'depositions', both, _ ).
noun( 'depositor', 'depositors', count, _ ).
noun( 'depository', 'depositories', count, _ ).
noun( 'depot', 'depots', count, _ ).
noun( 'depravity', 'depravities', both, _ ).
noun( 'deprecation', 'deprecations', count, _ ).
noun( 'depreciation', '-', mass, _ ).
noun( 'depredation', 'depredations', count, _ ).
noun( 'depression', 'depressions', both, _ ).
noun( 'depressive', 'depressives', count, _ ).
noun( 'deprivation', 'deprivations', both, _ ).
noun( 'depth', 'depths', both, _ ).
noun( 'depth-bomb', 'depth-bombs', count, _ ).
noun( 'depth-charge', 'depth-charges', count, _ ).
noun( 'deputation', 'deputations', count, _ ).
noun( 'deputy', 'deputies', count, _ ).
noun( 'derailment', 'derailments', count, _ ).
noun( 'derangement', 'derangements', count, _ ).
noun( 'derby', 'derbies', count, _ ).
noun( 'derby', 'derbies', count, _ ).
noun( 'dereliction', '-', mass, _ ).
noun( 'derision', '-', mass, _ ).
noun( 'derivation', 'derivations', both, _ ).
noun( 'derivative', 'derivatives', count, _ ).
noun( 'dermatitis', '-', mass, _ ).
noun( 'dermatologist', 'dermatologists', count, _ ).
noun( 'dermatology', '-', mass, _ ).
noun( 'derogation', '-', mass, _ ).
noun( 'derrick', 'derricks', count, _ ).
noun( 'derring-do', '-', mass, _ ).
noun( 'derv', '-', mass, _ ).
noun( 'dervish', 'dervishes', count, _ ).
noun( 'desalination', '-', mass, _ ).
noun( 'desalinization', '-', mass, _ ).
noun( 'descant', 'descants', count, _ ).
noun( 'descendant', 'descendants', count, _ ).
noun( 'descent', 'descents', both, _ ).
noun( 'description', 'descriptions', both, _ ).
noun( 'desecration', '-', mass, _ ).
noun( 'desegregation', '-', mass, _ ).
noun( 'desensitization', '-', mass, _ ).
noun( 'desert', 'deserts', both, _ ).
noun( 'deserter', 'deserters', count, _ ).
noun( 'desertion', 'desertions', both, _ ).
noun( 'desiccant', 'desiccants', count, _ ).
noun( 'desideratum', 'desiderata', count, _ ).
noun( 'design', 'designs', both, _ ).
noun( 'designation', 'designations', both, _ ).
noun( 'designer', 'designers', count, _ ).
noun( 'designing', '-', mass, _ ).
noun( 'desirability', '-', mass, _ ).
noun( 'desire', 'desires', both, _ ).
noun( 'desk', 'desks', count, _ ).
noun( 'desolation', '-', mass, _ ).
noun( 'despair', '-', mass, _ ).
noun( 'despatch', 'despatches', both, _ ).
noun( 'desperado', 'desperadoes', count, _ ).
noun( 'desperation', '-', mass, _ ).
noun( 'despite', '-', mass, _ ).
noun( 'despondency', '-', mass, _ ).
noun( 'despot', 'despots', count, _ ).
noun( 'despotism', 'despotisms', both, _ ).
noun( 'dessert', 'desserts', count, _ ).
noun( 'dessertspoon', 'dessertspoons', count, _ ).
noun( 'dessertspoonful', 'dessertspoonfuls', count, _ ).
noun( 'destination', 'destinations', count, _ ).
noun( 'destiny', 'destinies', both, _ ).
noun( 'destitution', '-', mass, _ ).
noun( 'destroyer', 'destroyers', count, _ ).
noun( 'destructibility', '-', mass, _ ).
noun( 'destruction', '-', mass, _ ).
noun( 'destructiveness', '-', mass, _ ).
noun( 'desuetude', '-', mass, _ ).
noun( 'detachment', 'detachments', both, _ ).
noun( 'detail', 'details', both, _ ).
noun( 'detainee', 'detainees', count, _ ).
noun( 'detection', '-', mass, _ ).
noun( 'detective', 'detectives', count, _ ).
noun( 'detector', 'detectors', count, _ ).
noun( 'detention', 'detentions', both, _ ).
noun( 'detergent', 'detergents', count, _ ).
noun( 'deterioration', 'deteriorations', both, _ ).
noun( 'determinant', 'determinants', count, _ ).
noun( 'determination', '-', mass, _ ).
noun( 'determinative', 'determinatives', count, _ ).
noun( 'determiner', 'determiners', count, _ ).
noun( 'deterrent', 'deterrents', count, _ ).
noun( 'detestation', 'detestations', both, _ ).
noun( 'dethronement', 'dethronements', count, _ ).
noun( 'detonation', 'detonations', count, _ ).
noun( 'detonator', 'detonators', count, _ ).
noun( 'detour', 'detours', count, _ ).
noun( 'detraction', 'detractions', both, _ ).
noun( 'detractor', 'detractors', count, _ ).
noun( 'detribalization', '-', mass, _ ).
noun( 'detriment', '-', mass, _ ).
noun( 'detritus', '-', mass, _ ).
noun( 'deuce', 'deuces', count, _ ).
noun( 'devaluation', 'devaluations', both, _ ).
noun( 'devastation', '-', mass, _ ).
noun( 'developer', 'developers', count, _ ).
noun( 'development', 'developments', both, _ ).
noun( 'deviant', 'deviants', count, _ ).
noun( 'deviation', 'deviations', both, _ ).
noun( 'deviationism', '-', mass, _ ).
noun( 'deviationist', 'deviationists', count, _ ).
noun( 'device', 'devices', count, _ ).
noun( 'devil', 'devils', count, _ ).
noun( 'devilment', 'devilments', both, _ ).
noun( 'devilry', 'devilries', both, _ ).
noun( 'deviousness', '-', mass, _ ).
noun( 'devitalization', '-', mass, _ ).
noun( 'devolution', '-', mass, _ ).
noun( 'devotee', 'devotees', count, _ ).
noun( 'devotion', 'devotions', both, _ ).
noun( 'devoutness', '-', mass, _ ).
noun( 'dew', '-', mass, _ ).
noun( 'dewlap', 'dewlaps', count, _ ).
noun( 'dexterity', '-', mass, _ ).
noun( 'dextrose', '-', mass, _ ).
noun( 'dhoti', 'dhotis', count, _ ).
noun( 'dhow', 'dhows', count, _ ).
noun( 'diabetes', '-', mass, _ ).
noun( 'diabetic', 'diabetics', count, _ ).
noun( 'diacritic', 'diacritics', count, _ ).
noun( 'diadem', 'diadems', count, _ ).
noun( 'diaeresis', 'diaereses', count, _ ).
noun( 'diagnosis', 'diagnoses', both, _ ).
noun( 'diagonal', 'diagonals', count, _ ).
noun( 'diagram', 'diagrams', count, _ ).
noun( 'dial', 'dials', count, _ ).
noun( 'dialect', 'dialects', both, _ ).
noun( 'dialectic', 'dialectics', count, _ ).
noun( 'dialectician', 'dialecticians', count, _ ).
noun( 'dialogue', 'dialogues', both, _ ).
noun( 'diameter', 'diameters', count, _ ).
noun( 'diamond', 'diamonds', count, _ ).
noun( 'diaper', 'diapers', count, _ ).
noun( 'diaphragm', 'diaphragms', count, _ ).
noun( 'diarchy', 'diarchies', count, _ ).
noun( 'diarist', 'diarists', count, _ ).
noun( 'diarrhea', '-', mass, _ ).
noun( 'diarrhoea', '-', mass, _ ).
noun( 'diary', 'diaries', count, _ ).
noun( 'diatribe', 'diatribes', count, _ ).
noun( 'dibber', 'dibbers', count, _ ).
noun( 'dibble', 'dibbles', count, _ ).
noun( 'dice', 'dice', count, _ ).
noun( 'dice-box', 'dice-boxes', count, _ ).
noun( 'dichotomy', 'dichotomies', count, _ ).
noun( 'dickens', '-', count, _ ).
noun( 'dickey', 'dickeys', count, _ ).
noun( 'dicky', 'dickies', count, _ ).
noun( 'dicky-seat', 'dicky-seats', count, _ ).
noun( 'dickybird', 'dickybirds', count, _ ).
noun( 'dictate', 'dictates', count, _ ).
noun( 'dictation', 'dictations', both, _ ).
noun( 'dictator', 'dictators', count, _ ).
noun( 'dictatorship', 'dictatorships', both, _ ).
noun( 'diction', '-', mass, _ ).
noun( 'dictionary', 'dictionaries', count, _ ).
noun( 'dictum', 'dictums', count, _ ).
noun( 'die', 'dies', count, _ ).
noun( 'die-hard', 'die-hards', count, _ ).
noun( 'dieresis', 'diereses', count, _ ).
noun( 'diesel', 'diesels', both, _ ).
noun( 'diet', 'diets', count, _ ).
noun( 'dietetics', 'dietetics', mass, _ ).
noun( 'dietician', 'dieticians', count, _ ).
noun( 'dietitian', 'dietitians', count, _ ).
noun( 'difference', 'differences', both, _ ).
noun( 'differential', 'differentials', count, _ ).
noun( 'differentiation', 'differentiations', both, _ ).
noun( 'difficulty', 'difficulties', both, _ ).
noun( 'diffidence', '-', mass, _ ).
noun( 'diffraction', 'diffractions', count, _ ).
noun( 'diffuseness', '-', mass, _ ).
noun( 'diffusion', '-', mass, _ ).
noun( 'dig', 'digs', count, _ ).
noun( 'digest', 'digests', count, _ ).
noun( 'digestibility', '-', mass, _ ).
noun( 'digestion', 'digestions', both, _ ).
noun( 'digger', 'diggers', count, _ ).
noun( 'digging', 'diggings', both, _ ).
noun( 'digit', 'digits', count, _ ).
noun( 'dignitary', 'dignitaries', count, _ ).
noun( 'dignity', 'dignities', both, _ ).
noun( 'digraph', 'digraphs', count, _ ).
noun( 'digression', 'digressions', both, _ ).
noun( 'dike', 'dikes', count, _ ).
noun( 'dilapidation', '-', mass, _ ).
noun( 'dilation', '-', mass, _ ).
noun( 'dilemma', 'dilemmas', count, _ ).
noun( 'dilettante', 'dilettantes', count, _ ).
noun( 'diligence', '-', mass, _ ).
noun( 'dill', '-', mass, _ ).
noun( 'dilution', 'dilutions', both, _ ).
noun( 'dime', 'dimes', count, _ ).
noun( 'dimension', 'dimensions', both, _ ).
noun( 'diminuendo', 'diminuendos', count, _ ).
noun( 'diminution', 'diminutions', both, _ ).
noun( 'diminutive', 'diminutives', count, _ ).
noun( 'dimity', '-', mass, _ ).
noun( 'dimness', '-', mass, _ ).
noun( 'dimple', 'dimples', count, _ ).
noun( 'din', '-', mass, _ ).
noun( 'dinar', 'dinars', count, _ ).
noun( 'diner', 'diners', count, _ ).
noun( 'ding-dong', 'ding-dongs', count, _ ).
noun( 'dinghy', 'dinghies', count, _ ).
noun( 'dinginess', '-', mass, _ ).
noun( 'dingle', 'dingles', count, _ ).
noun( 'dining-car', 'dining-cars', count, _ ).
noun( 'dining-room', 'dining-rooms', count, _ ).
noun( 'dining-table', 'dining-tables', count, _ ).
noun( 'dinner', 'dinners', count, _ ).
noun( 'dinner-jacket', 'dinner-jackets', count, _ ).
noun( 'dinner-party', 'dinner-parties', count, _ ).
noun( 'dinner-service', 'dinner-services', count, _ ).
noun( 'dinner-set', 'dinner-sets', count, _ ).
noun( 'dinosaur', 'dinosaurs', count, _ ).
noun( 'dint', 'dints', count, _ ).
noun( 'diocesan', 'diocesans', count, _ ).
noun( 'diocese', 'dioceses', count, _ ).
noun( 'dioxide', 'dioxides', count, _ ).
noun( 'dip', 'dips', both, _ ).
noun( 'diphtheria', '-', mass, _ ).
noun( 'diphthong', 'diphthongs', count, _ ).
noun( 'diploma', 'diplomas', count, _ ).
noun( 'diplomacy', '-', mass, _ ).
noun( 'diplomat', 'diplomats', count, _ ).
noun( 'diplomatist', 'diplomatists', count, _ ).
noun( 'dipper', 'dippers', count, _ ).
noun( 'dipsomania', '-', mass, _ ).
noun( 'dipsomaniac', 'dipsomaniacs', count, _ ).
noun( 'dipstick', 'dipsticks', count, _ ).
noun( 'diptych', 'diptychs', count, _ ).
noun( 'direction', 'directions', both, _ ).
noun( 'direction-finder', 'direction-finders', count, _ ).
noun( 'directive', 'directives', count, _ ).
noun( 'directness', '-', mass, _ ).
noun( 'director', 'directors', count, _ ).
noun( 'directorate', 'directorates', count, _ ).
noun( 'directorship', 'directorships', count, _ ).
noun( 'directory', 'directories', count, _ ).
noun( 'dirge', 'dirges', count, _ ).
noun( 'dirigible', 'dirigibles', count, _ ).
noun( 'dirk', 'dirks', count, _ ).
noun( 'dirndl', 'dirndls', count, _ ).
noun( 'dirt', '-', mass, _ ).
noun( 'dirt-track', 'dirt-tracks', count, _ ).
noun( 'disability', 'disabilities', both, _ ).
noun( 'disablement', 'disablements', both, _ ).
noun( 'disadvantage', 'disadvantages', both, _ ).
noun( 'disaffection', '-', mass, _ ).
noun( 'disagreeableness', '-', mass, _ ).
noun( 'disagreement', 'disagreements', both, _ ).
noun( 'disappearance', 'disappearances', count, _ ).
noun( 'disappointment', 'disappointments', both, _ ).
noun( 'disapprobation', '-', mass, _ ).
noun( 'disapproval', '-', mass, _ ).
noun( 'disarmament', '-', mass, _ ).
noun( 'disarrangement', 'disarrangements', count, _ ).
noun( 'disarray', '-', mass, _ ).
noun( 'disaster', 'disasters', both, _ ).
noun( 'disavowal', 'disavowals', count, _ ).
noun( 'disbandment', 'disbandments', count, _ ).
noun( 'disbelief', '-', mass, _ ).
noun( 'disbursement', 'disbursements', both, _ ).
noun( 'disc', 'discs', count, _ ).
noun( 'discard', 'discards', count, _ ).
noun( 'discernment', '-', mass, _ ).
noun( 'discharge', 'discharges', both, _ ).
noun( 'disciple', 'disciples', count, _ ).
noun( 'disciplinarian', 'disciplinarians', count, _ ).
noun( 'discipline', 'disciplines', both, _ ).
noun( 'disclaimer', 'disclaimers', count, _ ).
noun( 'disclosure', 'disclosures', both, _ ).
noun( 'disco', 'discos', count, _ ).
noun( 'discolouration', 'discolourations', both, _ ).
noun( 'discomfiture', '-', mass, _ ).
noun( 'discomfort', 'discomforts', both, _ ).
noun( 'discomposure', '-', mass, _ ).
noun( 'discontent', 'discontents', both, _ ).
noun( 'discontinuance', 'discontinuances', count, _ ).
noun( 'discontinuity', 'discontinuities', both, _ ).
noun( 'discord', 'discords', both, _ ).
noun( 'discordance', '-', mass, _ ).
noun( 'discotheque', 'discotheques', count, _ ).
noun( 'discount', 'discounts', both, _ ).
noun( 'discouragement', 'discouragements', both, _ ).
noun( 'discourse', 'discourses', count, _ ).
noun( 'discourtesy', 'discourtesies', both, _ ).
noun( 'discoverer', 'discoverers', count, _ ).
noun( 'discovery', 'discoveries', both, _ ).
noun( 'discredit', '-', mass, _ ).
noun( 'discrepancy', 'discrepancies', both, _ ).
noun( 'discreteness', '-', mass, _ ).
noun( 'discretion', '-', mass, _ ).
noun( 'discrimination', '-', mass, _ ).
noun( 'discursiveness', '-', mass, _ ).
noun( 'discus', 'discuses', count, _ ).
noun( 'discussion', 'discussions', both, _ ).
noun( 'disdain', '-', mass, _ ).
noun( 'disease', 'diseases', both, _ ).
noun( 'disembarkation', 'disembarkations', count, _ ).
noun( 'disembarrassment', 'disembarrassments', count, _ ).
noun( 'disenchantment', 'disenchantments', count, _ ).
noun( 'disengagement', 'disengagements', count, _ ).
noun( 'disentanglement', 'disentanglements', count, _ ).
noun( 'disequilibrium', '-', mass, _ ).
noun( 'disestablishment', '-', mass, _ ).
noun( 'disfavour', '-', mass, _ ).
noun( 'disfigurement', 'disfigurements', both, _ ).
noun( 'disfranchisement', '-', mass, _ ).
noun( 'disgrace', '-', mass, _ ).
noun( 'disguise', 'disguises', both, _ ).
noun( 'disgust', '-', mass, _ ).
noun( 'dish', 'dishes', count, _ ).
noun( 'dishabille', '-', mass, _ ).
noun( 'disharmony', '-', mass, _ ).
noun( 'dishcloth', 'dishcloths', count, _ ).
noun( 'dishful', 'dishfuls', count, _ ).
noun( 'dishonesty', 'dishonesties', both, _ ).
noun( 'dishonour', '-', mass, _ ).
noun( 'dishwasher', 'dishwashers', count, _ ).
noun( 'dishwater', '-', mass, _ ).
noun( 'disillusion', '-', mass, _ ).
noun( 'disillusionment', 'disillusionments', both, _ ).
noun( 'disincentive', 'disincentives', count, _ ).
noun( 'disinclination', 'disinclinations', both, _ ).
noun( 'disinfectant', 'disinfectants', count, _ ).
noun( 'disinfection', '-', mass, _ ).
noun( 'disinfestation', '-', mass, _ ).
noun( 'disinflation', '-', mass, _ ).
noun( 'disingenuousness', '-', mass, _ ).
noun( 'disinheritance', '-', mass, _ ).
noun( 'disintegration', 'disintegrations', both, _ ).
noun( 'disinterestedness', '-', mass, _ ).
noun( 'disinterment', 'disinterments', count, _ ).
noun( 'disjointedness', '-', mass, _ ).
noun( 'disk', 'disks', count, _ ).
noun( 'dislike', 'dislikes', both, _ ).
noun( 'dislocation', 'dislocations', both, _ ).
noun( 'dislodgement', 'dislodgements', count, _ ).
noun( 'disloyalty', 'disloyalties', both, _ ).
noun( 'dismantlement', '-', mass, _ ).
noun( 'dismay', '-', mass, _ ).
noun( 'dismemberment', 'dismemberments', count, _ ).
noun( 'dismissal', 'dismissals', both, _ ).
noun( 'disobedience', '-', mass, _ ).
noun( 'disorder', 'disorders', both, _ ).
noun( 'disorganization', '-', mass, _ ).
noun( 'disparagement', 'disparagements', count, _ ).
noun( 'disparity', 'disparities', both, _ ).
noun( 'dispassionateness', '-', mass, _ ).
noun( 'dispatch', 'dispatches', both, _ ).
noun( 'dispatch-box', 'dispatch-boxes', count, _ ).
noun( 'dispatch-rider', 'dispatch-riders', count, _ ).
noun( 'dispensary', 'dispensaries', count, _ ).
noun( 'dispensation', 'dispensations', both, _ ).
noun( 'dispenser', 'dispensers', count, _ ).
noun( 'dispersal', 'dispersals', both, _ ).
noun( 'dispersion', 'dispersions', count, _ ).
noun( 'displacement', 'displacements', both, _ ).
noun( 'display', 'displays', both, _ ).
noun( 'displeasure', '-', mass, _ ).
noun( 'disposal', '-', mass, _ ).
noun( 'disposition', 'dispositions', count, _ ).
noun( 'dispossession', 'dispossessions', count, _ ).
noun( 'disproof', 'disproofs', both, _ ).
noun( 'disproportion', '-', mass, _ ).
noun( 'disputant', 'disputants', count, _ ).
noun( 'disputation', 'disputations', both, _ ).
noun( 'dispute', 'disputes', both, _ ).
noun( 'disqualification', 'disqualifications', both, _ ).
noun( 'disquiet', '-', mass, _ ).
noun( 'disquietude', '-', mass, _ ).
noun( 'disquisition', 'disquisitions', count, _ ).
noun( 'disregard', '-', mass, _ ).
noun( 'disrepair', '-', mass, _ ).
noun( 'disrepute', '-', mass, _ ).
noun( 'disrespect', '-', mass, _ ).
noun( 'disruption', 'disruptions', both, _ ).
noun( 'dissatisfaction', '-', mass, _ ).
noun( 'dissection', 'dissections', both, _ ).
noun( 'dissembler', 'dissemblers', count, _ ).
noun( 'dissemination', '-', mass, _ ).
noun( 'dissension', 'dissensions', both, _ ).
noun( 'dissent', '-', mass, _ ).
noun( 'dissenter', 'dissenters', count, _ ).
noun( 'dissertation', 'dissertations', count, _ ).
noun( 'disservice', 'disservices', both, _ ).
noun( 'dissidence', '-', mass, _ ).
noun( 'dissident', 'dissidents', count, _ ).
noun( 'dissimilarity', 'dissimilarities', both, _ ).
noun( 'dissimilitude', '-', mass, _ ).
noun( 'dissimulation', 'dissimulations', count, _ ).
noun( 'dissipation', '-', mass, _ ).
noun( 'dissociation', '-', mass, _ ).
noun( 'dissolubility', '-', mass, _ ).
noun( 'dissolution', 'dissolutions', both, _ ).
noun( 'dissonance', 'dissonances', both, _ ).
noun( 'dissuasion', '-', mass, _ ).
noun( 'dissyllable', 'dissyllables', count, _ ).
noun( 'distaff', 'distaffs', count, _ ).
noun( 'distance', 'distances', both, _ ).
noun( 'distaste', 'distastes', both, _ ).
noun( 'distastefulness', '-', mass, _ ).
noun( 'distemper', '-', mass, _ ).
noun( 'distension', 'distensions', count, _ ).
noun( 'distillation', 'distillations', both, _ ).
noun( 'distiller', 'distillers', count, _ ).
noun( 'distillery', 'distilleries', count, _ ).
noun( 'distinction', 'distinctions', both, _ ).
noun( 'distinctiveness', '-', mass, _ ).
noun( 'distinctness', '-', mass, _ ).
noun( 'distortion', 'distortions', both, _ ).
noun( 'distraction', 'distractions', both, _ ).
noun( 'distraint', 'distraints', count, _ ).
noun( 'distress', '-', mass, _ ).
noun( 'distribution', 'distributions', both, _ ).
noun( 'distributor', 'distributors', count, _ ).
noun( 'district', 'districts', count, _ ).
noun( 'distrust', '-', both, _ ).
noun( 'distrustfulness', '-', mass, _ ).
noun( 'disturbance', 'disturbances', both, _ ).
noun( 'disunion', '-', mass, _ ).
noun( 'disunity', '-', mass, _ ).
noun( 'disuse', '-', mass, _ ).
noun( 'disyllable', 'disyllables', count, _ ).
noun( 'ditch', 'ditches', count, _ ).
noun( 'dither', 'dithers', count, _ ).
noun( 'ditto', '-', count, _ ).
noun( 'ditty', 'ditties', count, _ ).
noun( 'divagation', 'divagations', both, _ ).
noun( 'divan', 'divans', count, _ ).
noun( 'divan-bed', 'divan-beds', count, _ ).
noun( 'dive', 'dives', count, _ ).
noun( 'dive-bomber', 'dive-bombers', count, _ ).
noun( 'diver', 'divers', count, _ ).
noun( 'divergence', 'divergences', both, _ ).
noun( 'divergency', 'divergencies', both, _ ).
noun( 'diversification', '-', mass, _ ).
noun( 'diversion', 'diversions', both, _ ).
noun( 'diversionist', 'diversionists', count, _ ).
noun( 'diversity', '-', mass, _ ).
noun( 'divide', 'divides', count, _ ).
noun( 'dividend', 'dividends', count, _ ).
noun( 'dividend-warrant', 'dividend-warrants', count, _ ).
noun( 'divination', 'divinations', both, _ ).
noun( 'divine', 'divines', count, _ ).
noun( 'diviner', 'diviners', count, _ ).
noun( 'diving-bell', 'diving-bells', count, _ ).
noun( 'diving-board', 'diving-boards', count, _ ).
noun( 'diving-dress', '-', mass, _ ).
noun( 'diving-suit', 'diving-suits', count, _ ).
noun( 'divinity', 'divinities', both, _ ).
noun( 'division', 'divisions', both, _ ).
noun( 'divisor', 'divisors', count, _ ).
noun( 'divorce', 'divorces', both, _ ).
noun( 'divorcee', 'divorcees', count, _ ).
noun( 'divot', 'divots', count, _ ).
noun( 'divulgence', 'divulgences', count, _ ).
noun( 'divvy', 'divvies', count, _ ).
noun( 'dixie', 'dixies', count, _ ).
noun( 'dizziness', '-', mass, _ ).
noun( 'djinn', 'djinns', count, _ ).
noun( 'do', '-', count, _ ).
noun( 'do', 'dos', count, _ ).
noun( 'do-gooder', 'do-gooders', count, _ ).
noun( 'dobbin', 'dobbins', count, _ ).
noun( 'docility', '-', mass, _ ).
noun( 'dock', 'docks', count, _ ).
noun( 'docker', 'dockers', count, _ ).
noun( 'docket', 'dockets', count, _ ).
noun( 'dockyard', 'dockyards', count, _ ).
noun( 'doctor', 'doctors', count, _ ).
noun( 'doctorate', 'doctorates', count, _ ).
noun( 'doctrinaire', 'doctrinaires', count, _ ).
noun( 'doctrine', 'doctrines', both, _ ).
noun( 'document', 'documents', count, _ ).
noun( 'documentation', '-', mass, _ ).
noun( 'dodderer', 'dodderers', count, _ ).
noun( 'dodge', 'dodges', count, _ ).
noun( 'dodgem', 'dodgems', count, _ ).
noun( 'dodger', 'dodgers', count, _ ).
noun( 'dodo', 'dodos', count, _ ).
noun( 'doe', 'does', count, _ ).
noun( 'doer', 'doers', count, _ ).
noun( 'dog', 'dogs', count, _ ).
noun( 'dog\'s-tooth', '-', mass, _ ).
noun( 'dog-biscuit', 'dog-biscuits', count, _ ).
noun( 'dog-cart', 'dog-carts', count, _ ).
noun( 'dog-collar', 'dog-collars', count, _ ).
noun( 'doge', 'doges', count, _ ).
noun( 'dogfish', 'dogfish', count, _ ).
noun( 'doggedness', '-', mass, _ ).
noun( 'doggerel', '-', mass, _ ).
noun( 'doggie', 'doggies', count, _ ).
noun( 'doggy', 'doggies', count, _ ).
noun( 'doghouse', 'doghouses', count, _ ).
noun( 'dogma', 'dogmas', both, _ ).
noun( 'dogmatism', '-', mass, _ ).
noun( 'dogsbody', 'dogsbodies', count, _ ).
noun( 'dogtooth', '-', mass, _ ).
noun( 'dogtrot', '-', count, _ ).
noun( 'dogwatch', 'dogwatches', count, _ ).
noun( 'dogwood', 'dogwoods', count, _ ).
noun( 'doh', '-', count, _ ).
noun( 'doily', 'doilies', count, _ ).
noun( 'dole', 'doles', both, _ ).
noun( 'doll', 'dolls', count, _ ).
noun( 'dollar', 'dollars', count, _ ).
noun( 'dollop', 'dollops', count, _ ).
noun( 'dolly', 'dollies', count, _ ).
noun( 'dolmen', 'dolmens', count, _ ).
noun( 'dolour', 'dolours', count, _ ).
noun( 'dolphin', 'dolphins', count, _ ).
noun( 'dolt', 'dolts', count, _ ).
noun( 'domain', 'domains', count, _ ).
noun( 'dome', 'domes', count, _ ).
noun( 'domestication', '-', mass, _ ).
noun( 'domesticity', '-', mass, _ ).
noun( 'domicile', 'domiciles', count, _ ).
noun( 'dominance', '-', mass, _ ).
noun( 'domination', '-', mass, _ ).
noun( 'dominie', 'dominies', count, _ ).
noun( 'dominion', 'dominions', both, _ ).
noun( 'domino', 'dominos', count, _ ).
noun( 'don', 'dons', count, _ ).
noun( 'don\'t-know', 'don\'t-knows', count, _ ).
noun( 'donation', 'donations', both, _ ).
noun( 'donjon', 'donjons', count, _ ).
noun( 'donkey', 'donkeys', count, _ ).
noun( 'donkey-jacket', 'donkey-jackets', count, _ ).
noun( 'donkey-work', '-', mass, _ ).
noun( 'donor', 'donors', count, _ ).
noun( 'doodlebug', 'doodlebugs', count, _ ).
noun( 'doom', 'dooms', count, _ ).
noun( 'door', 'doors', count, _ ).
noun( 'doorbell', 'doorbells', count, _ ).
noun( 'doorcase', 'doorcases', count, _ ).
noun( 'doorframe', 'doorframes', count, _ ).
noun( 'doorhandle', 'doorhandles', count, _ ).
noun( 'doorkeeper', 'doorkeepers', count, _ ).
noun( 'doorknob', 'doorknobs', count, _ ).
noun( 'doorknocker', 'doorknockers', count, _ ).
noun( 'doorman', 'doormen', count, _ ).
noun( 'doormat', 'doormats', count, _ ).
noun( 'doornail', 'doornails', count, _ ).
noun( 'doorplate', 'doorplates', count, _ ).
noun( 'doorpost', 'doorposts', count, _ ).
noun( 'doorstep', 'doorsteps', count, _ ).
noun( 'doorstopper', 'doorstoppers', count, _ ).
noun( 'doorway', 'doorways', count, _ ).
noun( 'dope', 'dopes', both, _ ).
noun( 'dormer', 'dormers', count, _ ).
noun( 'dormer-window', 'dormer-windows', count, _ ).
noun( 'dormitory', 'dormitories', count, _ ).
noun( 'dormouse', 'dormice', count, _ ).
noun( 'dory', 'dories', count, _ ).
noun( 'dosage', 'dosages', both, _ ).
noun( 'dose', 'doses', count, _ ).
noun( 'doss-house', 'doss-houses', count, _ ).
noun( 'dosser', 'dossers', count, _ ).
noun( 'dossier', 'dossiers', count, _ ).
noun( 'dot', 'dots', count, _ ).
noun( 'dotage', '-', mass, _ ).
noun( 'dotard', 'dotards', count, _ ).
noun( 'dottle', 'dottles', count, _ ).
noun( 'double', 'doubles', count, _ ).
noun( 'double-bass', 'double-basses', count, _ ).
noun( 'double-cross', 'double-crosses', count, _ ).
noun( 'double-dealer', 'double-dealers', count, _ ).
noun( 'double-dealing', '-', mass, _ ).
noun( 'double-decker', 'double-deckers', count, _ ).
noun( 'double-dutch', '-', mass, _ ).
noun( 'double-entry', 'double-entries', count, _ ).
noun( 'double-first', 'double-firsts', count, _ ).
noun( 'double-spacing', '-', mass, _ ).
noun( 'double-talk', '-', mass, _ ).
noun( 'double-think', '-', mass, _ ).
noun( 'doublet', 'doublets', count, _ ).
noun( 'doubloon', 'doubloons', count, _ ).
noun( 'doubt', 'doubts', both, _ ).
noun( 'douche', 'douches', count, _ ).
noun( 'dough', '-', mass, _ ).
noun( 'doughnut', 'doughnuts', count, _ ).
noun( 'dove', 'doves', count, _ ).
noun( 'dovecote', 'dovecotes', count, _ ).
noun( 'dovetail', 'dovetails', count, _ ).
noun( 'dowager', 'dowagers', count, _ ).
noun( 'dowdiness', '-', mass, _ ).
noun( 'dowel', 'dowels', count, _ ).
noun( 'dower', 'dowers', count, _ ).
noun( 'down', 'downs', both, _ ).
noun( 'down-and-out', 'down-and-outs', count, _ ).
noun( 'downbeat', 'downbeats', count, _ ).
noun( 'downfall', '-', count, _ ).
noun( 'downpour', 'downpours', count, _ ).
noun( 'downrightness', '-', mass, _ ).
noun( 'dowry', 'dowries', count, _ ).
noun( 'dowser', 'dowsers', count, _ ).
noun( 'dowsing', '-', mass, _ ).
noun( 'doxology', 'doxologies', count, _ ).
noun( 'doyen', 'doyens', count, _ ).
noun( 'doyley', 'doyleys', count, _ ).
noun( 'doyly', 'doylies', count, _ ).
noun( 'doz', 'doz', count, _ ).
noun( 'doze', 'dozes', count, _ ).
noun( 'dozen', 'dozens', count, _ ).
noun( 'drabness', '-', mass, _ ).
noun( 'drachm', 'drachms', count, _ ).
noun( 'drachma', 'drachmas', count, _ ).
noun( 'draft', 'drafts', count, _ ).
noun( 'draftee', 'draftees', count, _ ).
noun( 'drafting', 'draftings', count, _ ).
noun( 'draftsman', 'draftsmen', count, _ ).
noun( 'drag', '-', mass, _ ).
noun( 'dragnet', 'dragnets', count, _ ).
noun( 'dragoman', 'dragomans', count, _ ).
noun( 'dragon', 'dragons', count, _ ).
noun( 'dragonfly', 'dragonflies', count, _ ).
noun( 'dragoon', 'dragoons', count, _ ).
noun( 'drain', 'drains', count, _ ).
noun( 'drainage', '-', mass, _ ).
noun( 'drainage-basin', 'drainage-basins', count, _ ).
noun( 'draining-board', 'draining-boards', count, _ ).
noun( 'drainpipe', 'drainpipes', count, _ ).
noun( 'drake', 'drakes', count, _ ).
noun( 'dram', 'drams', count, _ ).
noun( 'drama', 'dramas', both, _ ).
noun( 'dramatics', 'dramatics', mass, _ ).
noun( 'dramatist', 'dramatists', count, _ ).
noun( 'dramatization', 'dramatizations', both, _ ).
noun( 'drape', 'drapes', count, _ ).
noun( 'draper', 'drapers', count, _ ).
noun( 'drapery', 'draperies', both, _ ).
noun( 'draught', 'draughts', both, _ ).
noun( 'draught-horse', 'draught-horses', count, _ ).
noun( 'draughts', 'draughts', mass, _ ).
noun( 'draughtsman', 'draughtsmen', count, _ ).
noun( 'draw', 'draws', count, _ ).
noun( 'drawback', 'drawbacks', both, _ ).
noun( 'drawbridge', 'drawbridges', count, _ ).
noun( 'drawer', 'drawers', count, _ ).
noun( 'drawing', 'drawings', both, _ ).
noun( 'drawing-board', 'drawing-boards', count, _ ).
noun( 'drawing-pin', 'drawing-pins', count, _ ).
noun( 'drawing-room', 'drawing-rooms', count, _ ).
noun( 'drawl', 'drawls', count, _ ).
noun( 'dray', 'drays', count, _ ).
noun( 'dread', '-', mass, _ ).
noun( 'dreadfulness', '-', mass, _ ).
noun( 'dreadnought', 'dreadnoughts', count, _ ).
noun( 'dream', 'dreams', count, _ ).
noun( 'dreamer', 'dreamers', count, _ ).
noun( 'dreamland', 'dreamlands', count, _ ).
noun( 'dreamworld', 'dreamworlds', count, _ ).
noun( 'dreariness', '-', mass, _ ).
noun( 'dredge', 'dredges', count, _ ).
noun( 'dredger', 'dredgers', count, _ ).
noun( 'drenching', 'drenchings', count, _ ).
noun( 'dress', 'dresses', both, _ ).
noun( 'dress-hanger', 'dress-hangers', count, _ ).
noun( 'dressage', '-', mass, _ ).
noun( 'dresser', 'dressers', count, _ ).
noun( 'dressing', 'dressings', both, _ ).
noun( 'dressing-case', 'dressing-cases', count, _ ).
noun( 'dressing-down', '-', count, _ ).
noun( 'dressing-gown', 'dressing-gowns', count, _ ).
noun( 'dressing-table', 'dressing-tables', count, _ ).
noun( 'dressmaker', 'dressmakers', count, _ ).
noun( 'dressmaking', '-', mass, _ ).
noun( 'dribbler', 'dribblers', count, _ ).
noun( 'driblet', 'driblets', count, _ ).
noun( 'drier', 'driers', count, _ ).
noun( 'drift', 'drifts', both, _ ).
noun( 'drift-ice', '-', mass, _ ).
noun( 'drift-net', 'drift-nets', count, _ ).
noun( 'drift-wood', '-', mass, _ ).
noun( 'driftage', '-', mass, _ ).
noun( 'drifter', 'drifters', count, _ ).
noun( 'drill', 'drills', both, _ ).
noun( 'drink', 'drinks', both, _ ).
noun( 'drinker', 'drinkers', count, _ ).
noun( 'drinking', '-', mass, _ ).
noun( 'drinking-bout', 'drinking-bouts', count, _ ).
noun( 'drinking-fountain', 'drinking-fountains', count, _ ).
noun( 'drinking-song', 'drinking-songs', count, _ ).
noun( 'drinking-water', '-', mass, _ ).
noun( 'drip', 'drips', count, _ ).
noun( 'dripping', '-', mass, _ ).
noun( 'dripping-pan', 'dripping-pans', count, _ ).
noun( 'drive', 'drives', both, _ ).
noun( 'drive-in', 'drive-ins', count, _ ).
noun( 'drivel', '-', mass, _ ).
noun( 'driveller', 'drivellers', count, _ ).
noun( 'driver', 'drivers', count, _ ).
noun( 'driveway', 'driveways', count, _ ).
noun( 'driving-belt', 'driving-belts', count, _ ).
noun( 'driving-wheel', 'driving-wheels', count, _ ).
noun( 'drizzle', '-', mass, _ ).
noun( 'drogue', 'drogues', count, _ ).
noun( 'drollery', 'drolleries', both, _ ).
noun( 'dromedary', 'dromedaries', count, _ ).
noun( 'drone', 'drones', both, _ ).
noun( 'droop', '-', count, _ ).
noun( 'drop', 'drops', count, _ ).
noun( 'drop-curtain', 'drop-curtains', count, _ ).
noun( 'drop-kick', 'drop-kicks', count, _ ).
noun( 'dropout', 'dropouts', count, _ ).
noun( 'dropping-zone', 'dropping-zones', count, _ ).
noun( 'dropsy', '-', mass, _ ).
noun( 'droshky', 'droshkies', count, _ ).
noun( 'dross', '-', mass, _ ).
noun( 'drought', 'droughts', both, _ ).
noun( 'drove', 'droves', count, _ ).
noun( 'drover', 'drovers', count, _ ).
noun( 'drowse', '-', count, _ ).
noun( 'drowsiness', '-', mass, _ ).
noun( 'drubbing', '-', count, _ ).
noun( 'drudge', 'drudges', count, _ ).
noun( 'drudgery', '-', mass, _ ).
noun( 'drug', 'drugs', count, _ ).
noun( 'drugget', 'druggets', both, _ ).
noun( 'druggist', 'druggists', count, _ ).
noun( 'drugstore', 'drugstores', count, _ ).
noun( 'druid', 'druids', count, _ ).
noun( 'drum', 'drums', count, _ ).
noun( 'drum-major', 'drum-majors', count, _ ).
noun( 'drum-majorette', 'drum-majorettes', count, _ ).
noun( 'drumfire', '-', mass, _ ).
noun( 'drummer', 'drummers', count, _ ).
noun( 'drumstick', 'drumsticks', count, _ ).
noun( 'drunk', 'drunks', count, _ ).
noun( 'drunkard', 'drunkards', count, _ ).
noun( 'drunkenness', '-', mass, _ ).
noun( 'drupe', 'drupes', count, _ ).
noun( 'dry-cleaner', 'dry-cleaners', count, _ ).
noun( 'dry-cleaning', '-', mass, _ ).
noun( 'dry-walling', '-', mass, _ ).
noun( 'dryad', 'dryads', count, _ ).
noun( 'dryer', 'dryers', count, _ ).
noun( 'dryness', '-', mass, _ ).
noun( 'dubbin', '-', mass, _ ).
noun( 'dubiety', 'dubieties', both, _ ).
noun( 'dubiousness', '-', mass, _ ).
noun( 'ducat', 'ducats', count, _ ).
noun( 'duchess', 'duchesses', count, _ ).
noun( 'duchy', 'duchies', count, _ ).
noun( 'duck', 'duck', both, _ ).
noun( 'ducking', 'duckings', count, _ ).
noun( 'ducking-stool', 'ducking-stools', count, _ ).
noun( 'duckling', 'ducklings', count, _ ).
noun( 'duckweed', '-', mass, _ ).
noun( 'ducky', 'duckies', count, _ ).
noun( 'duct', 'ducts', count, _ ).
noun( 'ductility', '-', mass, _ ).
noun( 'dud', 'duds', count, _ ).
noun( 'dude', 'dudes', count, _ ).
noun( 'dudgeon', '-', mass, _ ).
noun( 'due', 'dues', count, _ ).
noun( 'duel', 'duels', count, _ ).
noun( 'duelist', 'duelists', count, _ ).
noun( 'duellist', 'duellists', count, _ ).
noun( 'duenna', 'duennas', count, _ ).
noun( 'duet', 'duets', count, _ ).
noun( 'duffel', '-', mass, _ ).
noun( 'duffer', 'duffers', count, _ ).
noun( 'duffle', '-', mass, _ ).
noun( 'dug', 'dugs', count, _ ).
noun( 'dugong', 'dugongs', count, _ ).
noun( 'dugout', 'dugouts', count, _ ).
noun( 'duke', 'dukes', count, _ ).
noun( 'dukedom', 'dukedoms', count, _ ).
noun( 'dulcimer', 'dulcimers', count, _ ).
noun( 'dullard', 'dullards', count, _ ).
noun( 'dullness', '-', mass, _ ).
noun( 'dumbbell', 'dumbbells', count, _ ).
noun( 'dumbness', '-', mass, _ ).
noun( 'dumbwaiter', 'dumbwaiters', count, _ ).
noun( 'dumdum', 'dumdums', count, _ ).
noun( 'dummy', 'dummies', count, _ ).
noun( 'dump', 'dumps', count, _ ).
noun( 'dumper', 'dumpers', count, _ ).
noun( 'dumpling', 'dumplings', count, _ ).
noun( 'dun', 'duns', count, _ ).
noun( 'dunce', 'dunces', count, _ ).
noun( 'dunderhead', 'dunderheads', count, _ ).
noun( 'dune', 'dunes', count, _ ).
noun( 'dung', '-', mass, _ ).
noun( 'dungeon', 'dungeons', count, _ ).
noun( 'dunghill', 'dunghills', count, _ ).
noun( 'duodenum', 'duodenums', count, _ ).
noun( 'duologue', 'duologues', count, _ ).
noun( 'dupe', 'dupes', count, _ ).
noun( 'dupl', '-', proper, _ ).
noun( 'duplicate', 'duplicates', count, _ ).
noun( 'duplication', 'duplications', both, _ ).
noun( 'duplicator', 'duplicators', count, _ ).
noun( 'duplicity', '-', mass, _ ).
noun( 'durability', '-', mass, _ ).
noun( 'durable', 'durables', count, _ ).
noun( 'durance', '-', mass, _ ).
noun( 'duration', 'durations', both, _ ).
noun( 'durbar', 'durbars', count, _ ).
noun( 'duress', '-', mass, _ ).
noun( 'dusk', '-', mass, _ ).
noun( 'dust', '-', mass, _ ).
noun( 'dust-bowl', 'dust-bowls', count, _ ).
noun( 'dust-coat', 'dust-coats', count, _ ).
noun( 'dust-jacket', 'dust-jackets', count, _ ).
noun( 'dust-sheet', 'dust-sheets', count, _ ).
noun( 'dust-up', 'dust-ups', count, _ ).
noun( 'dust-wrapper', 'dust-wrappers', count, _ ).
noun( 'dustbin', 'dustbins', count, _ ).
noun( 'dustcart', 'dustcarts', count, _ ).
noun( 'duster', 'dusters', count, _ ).
noun( 'dustman', 'dustmen', count, _ ).
noun( 'dustpan', 'dustpans', count, _ ).
noun( 'duty', 'duties', both, _ ).
noun( 'duvet', 'duvets', count, _ ).
noun( 'dwarf', 'dwarfs', count, _ ).
noun( 'dweller', 'dwellers', count, _ ).
noun( 'dwelling', 'dwellings', count, _ ).
noun( 'dwelling-house', 'dwelling-houses', count, _ ).
noun( 'dyarchy', 'dyarchies', count, _ ).
noun( 'dye', 'dyes', both, _ ).
noun( 'dye-works', 'dye-works', count, _ ).
noun( 'dyer', 'dyers', count, _ ).
noun( 'dyestuff', 'dyestuffs', count, _ ).
noun( 'dyke', 'dykes', count, _ ).
noun( 'dynamic', 'dynamics', count, _ ).
noun( 'dynamism', '-', mass, _ ).
noun( 'dynamite', '-', mass, _ ).
noun( 'dynamo', 'dynamos', count, _ ).
noun( 'dynast', 'dynasts', count, _ ).
noun( 'dynasty', 'dynasties', count, _ ).
noun( 'dyne', 'dynes', count, _ ).
noun( 'dysentery', '-', mass, _ ).
noun( 'dyslexia', '-', mass, _ ).
noun( 'dyspepsia', '-', mass, _ ).
noun( 'dyspeptic', 'dyspeptics', count, _ ).
noun( 'e', '-', count, _ ).
noun( 'eagerness', '-', mass, _ ).
noun( 'eagle', 'eagles', count, _ ).
noun( 'eaglet', 'eaglets', count, _ ).
noun( 'ear', 'ears', count, _ ).
noun( 'ear-trumpet', 'ear-trumpets', count, _ ).
noun( 'earache', 'earaches', both, _ ).
noun( 'eardrop', 'eardrops', count, _ ).
noun( 'eardrum', 'eardrums', count, _ ).
noun( 'earful', 'earfuls', count, _ ).
noun( 'earl', 'earls', count, _ ).
noun( 'earldom', 'earldoms', count, _ ).
noun( 'earmark', 'earmarks', count, _ ).
noun( 'earnest', '-', mass, _ ).
noun( 'earnest-money', '-', mass, _ ).
noun( 'earnestness', '-', mass, _ ).
noun( 'earphone', 'earphones', count, _ ).
noun( 'earpiece', 'earpieces', count, _ ).
noun( 'earring', 'earrings', count, _ ).
noun( 'earshot', '-', mass, _ ).
noun( 'earth', '-', mass, _ ).
noun( 'earth-closet', 'earth-closets', count, _ ).
noun( 'earthenware', '-', mass, _ ).
noun( 'earthnut', 'earthnuts', count, _ ).
noun( 'earthquake', 'earthquakes', count, _ ).
noun( 'earthwork', 'earthworks', count, _ ).
noun( 'earthworm', 'earthworms', count, _ ).
noun( 'earwax', '-', mass, _ ).
noun( 'earwig', 'earwigs', count, _ ).
noun( 'ease', '-', mass, _ ).
noun( 'easel', 'easels', count, _ ).
noun( 'east', '-', mass, _ ).
noun( 'eatable', 'eatables', count, _ ).
noun( 'eater', 'eaters', count, _ ).
noun( 'eating-apple', 'eating-apples', count, _ ).
noun( 'eating-house', 'eating-houses', count, _ ).
noun( 'eau de cologne', '-', mass, _ ).
noun( 'eau-de-vie', '-', mass, _ ).
noun( 'eavesdropper', 'eavesdroppers', count, _ ).
noun( 'ebb', 'ebbs', count, _ ).
noun( 'ebbtide', 'ebbtides', count, _ ).
noun( 'ebonite', '-', mass, _ ).
noun( 'ebony', '-', mass, _ ).
noun( 'ebullience', '-', mass, _ ).
noun( 'eccentric', 'eccentrics', count, _ ).
noun( 'eccentricity', 'eccentricities', both, _ ).
noun( 'ecclesiastic', 'ecclesiastics', count, _ ).
noun( 'echelon', 'echelons', count, _ ).
noun( 'echo', 'echoes', both, _ ).
noun( 'echo-sounder', 'echo-sounders', count, _ ).
noun( 'echo-sounding', 'echo-soundings', count, _ ).
noun( 'eclecticism', '-', mass, _ ).
noun( 'eclipse', 'eclipses', count, _ ).
noun( 'ecliptic', '-', count, _ ).
noun( 'ecologist', 'ecologists', count, _ ).
noun( 'ecology', '-', mass, _ ).
noun( 'economics', 'economics', mass, _ ).
noun( 'economist', 'economists', count, _ ).
noun( 'economy', 'economies', both, _ ).
noun( 'ecosystem', 'ecosystems', count, _ ).
noun( 'ecstasy', 'ecstasies', both, _ ).
noun( 'ectoplasm', '-', mass, _ ).
noun( 'eczema', '-', mass, _ ).
noun( 'ed', '-', proper, _ ).
noun( 'eddy', 'eddies', count, _ ).
noun( 'edelweiss', '-', mass, _ ).
noun( 'edge', 'edges', count, _ ).
noun( 'edging', 'edgings', both, _ ).
noun( 'edibility', '-', mass, _ ).
noun( 'edible', 'edibles', count, _ ).
noun( 'edict', 'edicts', count, _ ).
noun( 'edification', '-', mass, _ ).
noun( 'edifice', 'edifices', count, _ ).
noun( 'edition', 'editions', count, _ ).
noun( 'editor', 'editors', count, _ ).
noun( 'editorial', 'editorials', count, _ ).
noun( 'education', '-', mass, _ ).
noun( 'educationalist', 'educationalists', count, _ ).
noun( 'educationist', 'educationists', count, _ ).
noun( 'educator', 'educators', count, _ ).
noun( 'eel', 'eels', count, _ ).
noun( 'eeriness', '-', mass, _ ).
noun( 'effacement', '-', mass, _ ).
noun( 'effect', 'effects', both, _ ).
noun( 'effectiveness', '-', mass, _ ).
noun( 'effectuality', '-', mass, _ ).
noun( 'effectualness', '-', mass, _ ).
noun( 'effeminacy', '-', mass, _ ).
noun( 'effendi', 'effendis', count, _ ).
noun( 'effervescence', '-', mass, _ ).
noun( 'effeteness', '-', mass, _ ).
noun( 'efficacy', '-', mass, _ ).
noun( 'efficiency', '-', mass, _ ).
noun( 'effigy', 'effigies', count, _ ).
noun( 'efflorescence', '-', mass, _ ).
noun( 'effluent', 'effluents', both, _ ).
noun( 'efflux', 'effluxes', both, _ ).
noun( 'effort', 'efforts', both, _ ).
noun( 'effrontery', 'effronteries', both, _ ).
noun( 'effulgence', '-', mass, _ ).
noun( 'effusion', 'effusions', both, _ ).
noun( 'effusiveness', '-', mass, _ ).
noun( 'eft', 'efts', count, _ ).
noun( 'eg', '-', proper, _ ).
noun( 'egalitarian', 'egalitarians', count, _ ).
noun( 'egalitarianism', '-', mass, _ ).
noun( 'egg', 'eggs', both, _ ).
noun( 'egg-beater', 'egg-beaters', count, _ ).
noun( 'egg-cup', 'egg-cups', count, _ ).
noun( 'egg-whisk', 'egg-whisks', count, _ ).
noun( 'egghead', 'eggheads', count, _ ).
noun( 'eggplant', 'eggplants', both, _ ).
noun( 'eggshake', 'eggshakes', count, _ ).
noun( 'eggshell', 'eggshells', count, _ ).
noun( 'eglantine', '-', mass, _ ).
noun( 'ego', 'egos', count, _ ).
noun( 'egoism', '-', mass, _ ).
noun( 'egoist', 'egoists', count, _ ).
noun( 'egotism', '-', mass, _ ).
noun( 'egotist', 'egotists', count, _ ).
noun( 'egotrip', 'egotrips', count, _ ).
noun( 'egress', 'egresses', both, _ ).
noun( 'egret', 'egrets', count, _ ).
noun( 'eiderdown', 'eiderdowns', count, _ ).
noun( 'eight', 'eights', count, _ ).
noun( 'eighteen', 'eighteens', count, _ ).
noun( 'eighteenth', 'eighteenths', count, _ ).
noun( 'eighth', 'eighths', count, _ ).
noun( 'eightieth', 'eightieths', count, _ ).
noun( 'eightpence', 'eightpences', count, _ ).
noun( 'eightsome', 'eightsomes', count, _ ).
noun( 'eighty', 'eighties', count, _ ).
noun( 'eisteddfod', 'eisteddfods', count, _ ).
noun( 'ejaculation', 'ejaculations', count, _ ).
noun( 'ejection', 'ejections', count, _ ).
noun( 'ejector', 'ejectors', count, _ ).
noun( 'ejector-seat', 'ejector-seats', count, _ ).
noun( 'elaborateness', '-', mass, _ ).
noun( 'elaboration', 'elaborations', both, _ ).
noun( 'eland', 'elands', count, _ ).
noun( 'elastic', '-', mass, _ ).
noun( 'elasticity', 'elasticities', both, _ ).
noun( 'elation', '-', mass, _ ).
noun( 'elbow', 'elbows', count, _ ).
noun( 'elder', 'elders', count, _ ).
noun( 'election', 'elections', both, _ ).
noun( 'electioneering', '-', mass, _ ).
noun( 'elector', 'electors', count, _ ).
noun( 'electorate', 'electorates', count, _ ).
noun( 'electrician', 'electricians', count, _ ).
noun( 'electricity', '-', mass, _ ).
noun( 'electrification', '-', mass, _ ).
noun( 'electrocardiogram', 'electrocardiograms', count, _ ).
noun( 'electrocardiograph', 'electrocardiographs', count, _ ).
noun( 'electrochemistry', '-', mass, _ ).
noun( 'electrocution', 'electrocutions', both, _ ).
noun( 'electrode', 'electrodes', count, _ ).
noun( 'electrolysis', '-', mass, _ ).
noun( 'electrolyte', 'electrolytes', count, _ ).
noun( 'electromagnet', 'electromagnets', count, _ ).
noun( 'electromagnetism', '-', mass, _ ).
noun( 'electron', 'electrons', count, _ ).
noun( 'electronics', 'electronics', mass, _ ).
noun( 'electroplate', '-', mass, _ ).
noun( 'elegance', '-', mass, _ ).
noun( 'elegy', 'elegies', count, _ ).
noun( 'element', 'elements', count, _ ).
noun( 'elephant', 'elephants', count, _ ).
noun( 'elephantiasis', '-', mass, _ ).
noun( 'elevation', 'elevations', both, _ ).
noun( 'elevator', 'elevators', count, _ ).
noun( 'eleven', 'elevens', count, _ ).
noun( 'eleventh', 'elevenths', count, _ ).
noun( 'elf', 'elves', count, _ ).
noun( 'elicitation', 'elicitations', count, _ ).
noun( 'eligibility', '-', mass, _ ).
noun( 'elimination', 'eliminations', both, _ ).
noun( 'elision', 'elisions', both, _ ).
noun( 'elitism', '-', mass, _ ).
noun( 'elitist', 'elitists', count, _ ).
noun( 'elixir', 'elixirs', count, _ ).
noun( 'elk', 'elks', count, _ ).
noun( 'ellipse', 'ellipses', count, _ ).
noun( 'ellipsis', 'ellipses', both, _ ).
noun( 'elm', 'elms', both, _ ).
noun( 'elocution', '-', mass, _ ).
noun( 'elocutionist', 'elocutionists', count, _ ).
noun( 'elongation', 'elongations', both, _ ).
noun( 'elopement', 'elopements', count, _ ).
noun( 'eloquence', '-', mass, _ ).
noun( 'elucidation', 'elucidations', count, _ ).
noun( 'elver', 'elvers', count, _ ).
noun( 'emaciation', '-', mass, _ ).
noun( 'emanation', 'emanations', both, _ ).
noun( 'emancipation', '-', mass, _ ).
noun( 'emasculation', 'emasculations', both, _ ).
noun( 'embalmment', 'embalmments', count, _ ).
noun( 'embankment', 'embankments', count, _ ).
noun( 'embargo', 'embargoes', count, _ ).
noun( 'embarkation', 'embarkations', both, _ ).
noun( 'embarrassment', 'embarrassments', both, _ ).
noun( 'embassy', 'embassies', count, _ ).
noun( 'embellishment', 'embellishments', both, _ ).
noun( 'ember', 'embers', count, _ ).
noun( 'embezzlement', 'embezzlements', both, _ ).
noun( 'embitterment', 'embitterments', count, _ ).
noun( 'emblem', 'emblems', count, _ ).
noun( 'embodiment', 'embodiments', count, _ ).
noun( 'embonpoint', '-', mass, _ ).
noun( 'embrace', 'embraces', count, _ ).
noun( 'embrasure', 'embrasures', count, _ ).
noun( 'embrocation', '-', mass, _ ).
noun( 'embroidery', '-', mass, _ ).
noun( 'embryo', 'embryos', count, _ ).
noun( 'emeer', 'emeers', count, _ ).
noun( 'emendation', 'emendations', both, _ ).
noun( 'emerald', 'emeralds', count, _ ).
noun( 'emergence', '-', mass, _ ).
noun( 'emergency', 'emergencies', both, _ ).
noun( 'emery', '-', mass, _ ).
noun( 'emetic', 'emetics', count, _ ).
noun( 'emigrant', 'emigrants', count, _ ).
noun( 'emigration', 'emigrations', both, _ ).
noun( 'eminence', 'eminences', both, _ ).
noun( 'emir', 'emirs', count, _ ).
noun( 'emirate', 'emirates', count, _ ).
noun( 'emissary', 'emissaries', count, _ ).
noun( 'emission', 'emissions', both, _ ).
noun( 'emolument', 'emoluments', count, _ ).
noun( 'emotion', 'emotions', both, _ ).
noun( 'empathy', '-', mass, _ ).
noun( 'emperor', 'emperors', count, _ ).
noun( 'emphasis', 'emphases', both, _ ).
noun( 'empire', 'empires', both, _ ).
noun( 'empiricism', '-', mass, _ ).
noun( 'empiricist', 'empiricists', count, _ ).
noun( 'emplacement', 'emplacements', count, _ ).
noun( 'employ', 'employs', count, _ ).
noun( 'employee', 'employees', count, _ ).
noun( 'employer', 'employers', count, _ ).
noun( 'employment', '-', mass, _ ).
noun( 'emporium', 'emporiums', count, _ ).
noun( 'empress', 'empresses', count, _ ).
noun( 'emptiness', '-', mass, _ ).
noun( 'empty', 'empties', count, _ ).
noun( 'empyrean', '-', count, _ ).
noun( 'emu', 'emus', count, _ ).
noun( 'emulation', '-', mass, _ ).
noun( 'emulsion', 'emulsions', both, _ ).
noun( 'enactment', 'enactments', both, _ ).
noun( 'enamel', '-', mass, _ ).
noun( 'enc', '-', proper, _ ).
noun( 'encampment', 'encampments', count, _ ).
noun( 'encephalitis', '-', mass, _ ).
noun( 'enchanter', 'enchanters', count, _ ).
noun( 'enchantment', 'enchantments', both, _ ).
noun( 'enchantress', 'enchantresses', count, _ ).
noun( 'encirclement', 'encirclements', count, _ ).
noun( 'enclave', 'enclaves', count, _ ).
noun( 'enclosure', 'enclosures', both, _ ).
noun( 'encomium', 'encomiums', count, _ ).
noun( 'encore', 'encores', count, _ ).
noun( 'encounter', 'encounters', count, _ ).
noun( 'encouragement', 'encouragements', both, _ ).
noun( 'encroachment', 'encroachments', both, _ ).
noun( 'encumbrance', 'encumbrances', count, _ ).
noun( 'encyclical', 'encyclicals', count, _ ).
noun( 'encyclopaedia', 'encyclopaedias', count, _ ).
noun( 'encyclopedia', 'encyclopedias', count, _ ).
noun( 'end', 'ends', count, _ ).
noun( 'end-all', '-', count, _ ).
noun( 'endearment', 'endearments', both, _ ).
noun( 'endeavour', 'endeavours', count, _ ).
noun( 'endemic', 'endemics', count, _ ).
noun( 'ending', 'endings', count, _ ).
noun( 'endive', 'endives', both, _ ).
noun( 'endorsement', 'endorsements', both, _ ).
noun( 'endowment', 'endowments', both, _ ).
noun( 'endurance', '-', mass, _ ).
noun( 'enema', 'enemas', count, _ ).
noun( 'enemy', 'enemies', count, _ ).
noun( 'energy', 'energies', both, _ ).
noun( 'enfant terrible', 'enfants terribles', count, _ ).
noun( 'enforcement', '-', mass, _ ).
noun( 'enfranchisement', 'enfranchisements', count, _ ).
noun( 'engagement', 'engagements', both, _ ).
noun( 'engine', 'engines', count, _ ).
noun( 'engine-driver', 'engine-drivers', count, _ ).
noun( 'engineer', 'engineers', count, _ ).
noun( 'engineering', '-', mass, _ ).
noun( 'engraver', 'engravers', count, _ ).
noun( 'engraving', 'engravings', both, _ ).
noun( 'enhancement', 'enhancements', both, _ ).
noun( 'enigma', 'enigmas', count, _ ).
noun( 'enjoyment', 'enjoyments', both, _ ).
noun( 'enlargement', 'enlargements', both, _ ).
noun( 'enlightenment', '-', mass, _ ).
noun( 'enlistment', 'enlistments', both, _ ).
noun( 'enmity', 'enmities', both, _ ).
noun( 'ennoblement', 'ennoblements', count, _ ).
noun( 'ennui', 'ennuis', both, _ ).
noun( 'enormity', 'enormities', both, _ ).
noun( 'enormousness', '-', mass, _ ).
noun( 'enough', 'enough', mass, _ ).
noun( 'enquirer', 'enquirers', count, _ ).
noun( 'enquiry', 'enquiries', both, _ ).
noun( 'enrichment', '-', mass, _ ).
noun( 'enrolment', 'enrolments', both, _ ).
noun( 'ensemble', 'ensembles', count, _ ).
noun( 'ensign', 'ensigns', count, _ ).
noun( 'ensilage', '-', mass, _ ).
noun( 'enslavement', 'enslavements', count, _ ).
noun( 'entail', 'entails', both, _ ).
noun( 'entanglement', 'entanglements', both, _ ).
noun( 'entente', 'ententes', count, _ ).
noun( 'entente cordiale', '-', count, _ ).
noun( 'enteritis', '-', mass, _ ).
noun( 'enterprise', 'enterprises', both, _ ).
noun( 'entertainer', 'entertainers', count, _ ).
noun( 'entertainment', 'entertainments', both, _ ).
noun( 'enthronement', 'enthronements', count, _ ).
noun( 'enthusiasm', '-', mass, _ ).
noun( 'enthusiast', 'enthusiasts', count, _ ).
noun( 'enticement', 'enticements', both, _ ).
noun( 'entirety', '-', mass, _ ).
noun( 'entitlement', 'entitlements', count, _ ).
noun( 'entity', 'entities', both, _ ).
noun( 'entomologist', 'entomologists', count, _ ).
noun( 'entomology', '-', mass, _ ).
noun( 'entourage', '-', count, _ ).
noun( 'entr\'acte', 'entr\'actes', count, _ ).
noun( 'entr_ee', 'entr_ees', both, _ ).
noun( 'entrance', 'entrances', both, _ ).
noun( 'entrance-fee', 'entrance-fees', count, _ ).
noun( 'entrance-money', '-', mass, _ ).
noun( 'entrant', 'entrants', count, _ ).
noun( 'entreaty', 'entreaties', both, _ ).
noun( 'entrenchment', 'entrenchments', count, _ ).
noun( 'entrepot', 'entrepots', count, _ ).
noun( 'entrepreneur', 'entrepreneurs', count, _ ).
noun( 'entry', 'entries', both, _ ).
noun( 'enumeration', 'enumerations', both, _ ).
noun( 'enunciation', 'enunciations', count, _ ).
noun( 'envelope', 'envelopes', count, _ ).
noun( 'envelopment', 'envelopments', count, _ ).
noun( 'environment', 'environments', both, _ ).
noun( 'envoi', 'envois', count, _ ).
noun( 'envoy', 'envoys', count, _ ).
noun( 'envy', '-', mass, _ ).
noun( 'enzyme', 'enzymes', count, _ ).
noun( 'eon', 'eons', count, _ ).
noun( 'epaulet', 'epaulets', count, _ ).
noun( 'epaulette', 'epaulettes', count, _ ).
noun( 'epic', 'epics', count, _ ).
noun( 'epicentre', 'epicentres', count, _ ).
noun( 'epicure', 'epicures', count, _ ).
noun( 'epicurean', 'epicureans', count, _ ).
noun( 'epidemic', 'epidemics', count, _ ).
noun( 'epidemiologist', 'epidemiologists', count, _ ).
noun( 'epidemiology', '-', mass, _ ).
noun( 'epidermis', '-', mass, _ ).
noun( 'epidiascope', 'epidiascopes', count, _ ).
noun( 'epiglottis', 'epiglottises', count, _ ).
noun( 'epigram', 'epigrams', count, _ ).
noun( 'epilepsy', '-', mass, _ ).
noun( 'epileptic', 'epileptics', count, _ ).
noun( 'epilogue', 'epilogues', count, _ ).
noun( 'episcopalian', 'episcopalians', count, _ ).
noun( 'episode', 'episodes', count, _ ).
noun( 'epistle', 'epistles', count, _ ).
noun( 'epitaph', 'epitaphs', count, _ ).
noun( 'epithet', 'epithets', count, _ ).
noun( 'epitome', 'epitomes', count, _ ).
noun( 'epoch', 'epochs', count, _ ).
noun( 'equal', 'equals', count, _ ).
noun( 'equalitarian', 'equalitarians', count, _ ).
noun( 'equality', '-', mass, _ ).
noun( 'equalization', 'equalizations', count, _ ).
noun( 'equalizer', 'equalizers', count, _ ).
noun( 'equanimity', '-', mass, _ ).
noun( 'equation', 'equations', both, _ ).
noun( 'equator', 'equators', count, _ ).
noun( 'equerry', 'equerries', count, _ ).
noun( 'equestrian', 'equestrians', count, _ ).
noun( 'equilibrium', '-', mass, _ ).
noun( 'equinox', 'equinoxes', count, _ ).
noun( 'equipage', 'equipages', count, _ ).
noun( 'equipment', '-', mass, _ ).
noun( 'equipoise', 'equipoises', both, _ ).
noun( 'equity', 'equities', both, _ ).
noun( 'equivalence', 'equivalences', both, _ ).
noun( 'equivalent', 'equivalents', count, _ ).
noun( 'equivocation', 'equivocations', both, _ ).
noun( 'era', 'eras', count, _ ).
noun( 'eradication', 'eradications', count, _ ).
noun( 'eraser', 'erasers', count, _ ).
noun( 'erasure', 'erasures', both, _ ).
noun( 'erection', 'erections', both, _ ).
noun( 'erectness', '-', mass, _ ).
noun( 'eremite', 'eremites', count, _ ).
noun( 'erg', 'ergs', count, _ ).
noun( 'ergonomics', 'ergonomics', mass, _ ).
noun( 'ermine', '-', mass, _ ).
noun( 'erosion', '-', mass, _ ).
noun( 'eroticism', '-', mass, _ ).
noun( 'errand', 'errands', count, _ ).
noun( 'erratum', 'errata', count, _ ).
noun( 'error', 'errors', both, _ ).
noun( 'eructation', 'eructations', both, _ ).
noun( 'erudition', '-', mass, _ ).
noun( 'eruption', 'eruptions', both, _ ).
noun( 'erysipelas', '-', mass, _ ).
noun( 'escalation', 'escalations', both, _ ).
noun( 'escalator', 'escalators', count, _ ).
noun( 'escalope', 'escalopes', count, _ ).
noun( 'escapade', 'escapades', count, _ ).
noun( 'escape', 'escapes', both, _ ).
noun( 'escapee', 'escapees', count, _ ).
noun( 'escapement', 'escapements', count, _ ).
noun( 'escapism', '-', mass, _ ).
noun( 'escapist', 'escapists', count, _ ).
noun( 'escapologist', 'escapologists', count, _ ).
noun( 'escarpment', 'escarpments', count, _ ).
noun( 'eschatology', '-', mass, _ ).
noun( 'escort', 'escorts', count, _ ).
noun( 'escritoire', 'escritoires', count, _ ).
noun( 'escudo', 'escudos', count, _ ).
noun( 'escutcheon', 'escutcheons', count, _ ).
noun( 'esophagus', 'esophaguses', count, _ ).
noun( 'espalier', 'espaliers', count, _ ).
noun( 'espionage', '-', mass, _ ).
noun( 'esplanade', 'esplanades', count, _ ).
noun( 'espousal', 'espousals', count, _ ).
noun( 'espresso', 'espressos', count, _ ).
noun( 'esprit', '-', mass, _ ).
noun( 'esprit de corps', '-', count, _ ).
noun( 'essay', 'essays', count, _ ).
noun( 'essayist', 'essayists', count, _ ).
noun( 'essence', 'essences', both, _ ).
noun( 'essential', 'essentials', count, _ ).
noun( 'establishment', 'establishments', both, _ ).
noun( 'estaminet', 'estaminets', count, _ ).
noun( 'estate', 'estates', both, _ ).
noun( 'esteem', '-', mass, _ ).
noun( 'esthete', 'esthetes', count, _ ).
noun( 'esthetic', '-', mass, _ ).
noun( 'esthetics', 'esthetics', mass, _ ).
noun( 'estimate', 'estimates', count, _ ).
noun( 'estimation', '-', mass, _ ).
noun( 'estrangement', 'estrangements', both, _ ).
noun( 'estuary', 'estuaries', count, _ ).
noun( 'et al', '-', proper, _ ).
noun( 'et seq', '-', proper, _ ).
noun( 'eta', '-', count, _ ).
noun( 'etc', '-', proper, _ ).
noun( 'etcher', 'etchers', count, _ ).
noun( 'etching', 'etchings', both, _ ).
noun( 'etd', '-', count, _ ).
noun( 'eternity', 'eternities', both, _ ).
noun( 'ether', '-', mass, _ ).
noun( 'ethic', '-', count, _ ).
noun( 'ethics', 'ethics', mass, _ ).
noun( 'ethnographer', 'ethnographers', count, _ ).
noun( 'ethnography', '-', mass, _ ).
noun( 'ethnologist', 'ethnologists', count, _ ).
noun( 'ethnology', '-', mass, _ ).
noun( 'ethos', '-', count, _ ).
noun( 'ethyl', 'ethyls', both, _ ).
noun( 'etiology', 'etiologies', count, _ ).
noun( 'etiquette', '-', mass, _ ).
noun( 'etymologist', 'etymologists', count, _ ).
noun( 'etymology', 'etymologies', both, _ ).
noun( 'eucalyptus', 'eucalyptuses', count, _ ).
noun( 'eugenics', 'eugenics', mass, _ ).
noun( 'eulogist', 'eulogists', count, _ ).
noun( 'eulogy', 'eulogies', both, _ ).
noun( 'eunuch', 'eunuchs', count, _ ).
noun( 'euphemism', 'euphemisms', both, _ ).
noun( 'euphonium', 'euphoniums', count, _ ).
noun( 'euphony', 'euphonies', both, _ ).
noun( 'euphoria', '-', mass, _ ).
noun( 'euphuism', 'euphuisms', count, _ ).
noun( 'eurhythmics', 'eurhythmics', mass, _ ).
noun( 'eurythmics', 'eurythmics', mass, _ ).
noun( 'euthanasia', '-', mass, _ ).
noun( 'evacuation', 'evacuations', both, _ ).
noun( 'evacuee', 'evacuees', count, _ ).
noun( 'evaluation', 'evaluations', count, _ ).
noun( 'evanescence', '-', mass, _ ).
noun( 'evangelicalism', '-', mass, _ ).
noun( 'evangelism', '-', mass, _ ).
noun( 'evangelist', 'evangelists', count, _ ).
noun( 'evaporation', 'evaporations', both, _ ).
noun( 'evasion', 'evasions', both, _ ).
noun( 'evasiveness', '-', mass, _ ).
noun( 'eve', 'eves', count, _ ).
noun( 'even', 'evens', count, _ ).
noun( 'evening', 'evenings', both, _ ).
noun( 'evenness', '-', mass, _ ).
noun( 'evensong', 'evensongs', count, _ ).
noun( 'event', 'events', count, _ ).
noun( 'eventide', '-', mass, _ ).
noun( 'eventuality', 'eventualities', count, _ ).
noun( 'evergreen', 'evergreens', count, _ ).
noun( 'eviction', 'evictions', both, _ ).
noun( 'evidence', '-', mass, _ ).
noun( 'evil', 'evils', both, _ ).
noun( 'evil-doer', 'evil-doers', count, _ ).
noun( 'evocation', 'evocations', count, _ ).
noun( 'evolution', 'evolutions', both, _ ).
noun( 'ewe', 'ewes', count, _ ).
noun( 'ewer', 'ewers', count, _ ).
noun( 'ex-serviceman', 'ex-servicemen', count, _ ).
noun( 'exacerbation', 'exacerbations', count, _ ).
noun( 'exaction', 'exactions', both, _ ).
noun( 'exactitude', '-', mass, _ ).
noun( 'exactness', '-', mass, _ ).
noun( 'exaggeration', 'exaggerations', both, _ ).
noun( 'exaltation', '-', mass, _ ).
noun( 'exam', 'exams', count, _ ).
noun( 'examination', 'examinations', both, _ ).
noun( 'examiner', 'examiners', count, _ ).
noun( 'example', 'examples', both, _ ).
noun( 'exasperation', '-', mass, _ ).
noun( 'excavation', 'excavations', both, _ ).
noun( 'excavator', 'excavators', count, _ ).
noun( 'excellence', 'excellences', both, _ ).
noun( 'excelsior', '-', mass, _ ).
noun( 'exception', 'exceptions', both, _ ).
noun( 'excerpt', 'excerpts', count, _ ).
noun( 'excess', 'excesses', both, _ ).
noun( 'exchange', 'exchanges', both, _ ).
noun( 'exchanger', 'exchangers', count, _ ).
noun( 'exchequer', 'exchequers', count, _ ).
noun( 'excise', '-', mass, _ ).
noun( 'exciseman', 'excisemen', count, _ ).
noun( 'excision', 'excisions', both, _ ).
noun( 'excitability', '-', mass, _ ).
noun( 'excitement', 'excitements', both, _ ).
noun( 'excl', '-', proper, _ ).
noun( 'exclamation', 'exclamations', both, _ ).
noun( 'exclusion', '-', mass, _ ).
noun( 'excogitation', 'excogitations', both, _ ).
noun( 'excommunication', 'excommunications', both, _ ).
noun( 'excoriation', 'excoriations', count, _ ).
noun( 'excrement', '-', mass, _ ).
noun( 'excrescence', 'excrescences', count, _ ).
noun( 'excretion', 'excretions', both, _ ).
noun( 'excursion', 'excursions', count, _ ).
noun( 'excursionist', 'excursionists', count, _ ).
noun( 'excuse', 'excuses', count, _ ).
noun( 'execration', 'execrations', count, _ ).
noun( 'executant', 'executants', count, _ ).
noun( 'execution', 'executions', both, _ ).
noun( 'executioner', 'executioners', count, _ ).
noun( 'executive', 'executives', count, _ ).
noun( 'executor', 'executors', count, _ ).
noun( 'executrix', 'executrixes', count, _ ).
noun( 'exegesis', '-', mass, _ ).
noun( 'exemplification', 'exemplifications', both, _ ).
noun( 'exemption', 'exemptions', both, _ ).
noun( 'exercise', 'exercises', both, _ ).
noun( 'exertion', 'exertions', both, _ ).
noun( 'exhalation', 'exhalations', both, _ ).
noun( 'exhaust', 'exhausts', both, _ ).
noun( 'exhaust-pipe', 'exhaust-pipes', count, _ ).
noun( 'exhaustion', '-', mass, _ ).
noun( 'exhibit', 'exhibits', count, _ ).
noun( 'exhibition', 'exhibitions', count, _ ).
noun( 'exhibitioner', 'exhibitioners', count, _ ).
noun( 'exhibitionism', '-', mass, _ ).
noun( 'exhibitionist', 'exhibitionists', count, _ ).
noun( 'exhibitor', 'exhibitors', count, _ ).
noun( 'exhilaration', '-', mass, _ ).
noun( 'exhortation', 'exhortations', both, _ ).
noun( 'exhumation', 'exhumations', both, _ ).
noun( 'exigency', 'exigencies', count, _ ).
noun( 'exile', 'exiles', both, _ ).
noun( 'existence', 'existences', both, _ ).
noun( 'existentialism', '-', mass, _ ).
noun( 'existentialist', 'existentialists', count, _ ).
noun( 'exit', 'exits', count, _ ).
noun( 'exodus', 'exoduses', count, _ ).
noun( 'exoneration', 'exonerations', count, _ ).
noun( 'exorbitance', '-', mass, _ ).
noun( 'expanse', 'expanses', count, _ ).
noun( 'expansion', '-', mass, _ ).
noun( 'expansiveness', '-', mass, _ ).
noun( 'expatriate', 'expatriates', count, _ ).
noun( 'expectancy', 'expectancies', both, _ ).
noun( 'expectation', 'expectations', both, _ ).
noun( 'expectorant', 'expectorants', count, _ ).
noun( 'expedience', '-', mass, _ ).
noun( 'expediency', '-', mass, _ ).
noun( 'expedient', 'expedients', count, _ ).
noun( 'expedition', 'expeditions', both, _ ).
noun( 'expenditure', 'expenditures', both, _ ).
noun( 'expense', 'expenses', both, _ ).
noun( 'experience', 'experiences', both, _ ).
noun( 'experiment', 'experiments', both, _ ).
noun( 'experimentation', '-', mass, _ ).
noun( 'experimenter', 'experimenters', count, _ ).
noun( 'expert', 'experts', count, _ ).
noun( 'expertise', '-', mass, _ ).
noun( 'expertness', '-', mass, _ ).
noun( 'expiation', '-', mass, _ ).
noun( 'expiration', '-', mass, _ ).
noun( 'expiry', 'expiries', count, _ ).
noun( 'explanation', 'explanations', both, _ ).
noun( 'expletive', 'expletives', count, _ ).
noun( 'explicitness', '-', mass, _ ).
noun( 'exploit', 'exploits', count, _ ).
noun( 'exploitation', '-', mass, _ ).
noun( 'exploration', 'explorations', both, _ ).
noun( 'explorer', 'explorers', count, _ ).
noun( 'explosion', 'explosions', count, _ ).
noun( 'explosive', 'explosives', count, _ ).
noun( 'expo', 'expos', count, _ ).
noun( 'exponent', 'exponents', count, _ ).
noun( 'exponential', 'exponentials', count, _ ).
noun( 'export', 'exports', both, _ ).
noun( 'exportation', '-', mass, _ ).
noun( 'exporter', 'exporters', count, _ ).
noun( 'expos_e', 'expos_es', count, _ ).
noun( 'exposition', 'expositions', both, _ ).
noun( 'expostulation', 'expostulations', both, _ ).
noun( 'exposure', 'exposures', both, _ ).
noun( 'express', 'expresses', both, _ ).
noun( 'expression', 'expressions', both, _ ).
noun( 'expressionism', '-', mass, _ ).
noun( 'expressionist', 'expressionists', count, _ ).
noun( 'expressway', 'expressways', count, _ ).
noun( 'expropriation', '-', mass, _ ).
noun( 'expulsion', 'expulsions', both, _ ).
noun( 'expurgation', 'expurgations', count, _ ).
noun( 'exquisiteness', '-', mass, _ ).
noun( 'extension', 'extensions', both, _ ).
noun( 'extent', '-', mass, _ ).
noun( 'extenuation', 'extenuations', both, _ ).
noun( 'exterior', 'exteriors', count, _ ).
noun( 'extermination', 'exterminations', count, _ ).
noun( 'external', 'externals', count, _ ).
noun( 'extinction', '-', mass, _ ).
noun( 'extinguisher', 'extinguishers', count, _ ).
noun( 'extirpation', '-', mass, _ ).
noun( 'extortion', 'extortions', both, _ ).
noun( 'extra', 'extras', count, _ ).
noun( 'extract', 'extracts', both, _ ).
noun( 'extraction', 'extractions', both, _ ).
noun( 'extradition', 'extraditions', both, _ ).
noun( 'extrapolation', '-', mass, _ ).
noun( 'extravagance', 'extravagances', both, _ ).
noun( 'extravaganza', 'extravaganzas', count, _ ).
noun( 'extreme', 'extremes', count, _ ).
noun( 'extremist', 'extremists', count, _ ).
noun( 'extremity', 'extremities', count, _ ).
noun( 'extrication', '-', mass, _ ).
noun( 'extroversion', '-', mass, _ ).
noun( 'extrovert', 'extroverts', count, _ ).
noun( 'extrusion', 'extrusions', both, _ ).
noun( 'exuberance', '-', mass, _ ).
noun( 'exultation', '-', mass, _ ).
noun( 'eye', 'eyes', count, _ ).
noun( 'eye-opener', 'eye-openers', count, _ ).
noun( 'eye-shadow', '-', mass, _ ).
noun( 'eyeball', 'eyeballs', count, _ ).
noun( 'eyebath', 'eyebaths', count, _ ).
noun( 'eyebrow', 'eyebrows', count, _ ).
noun( 'eyecup', 'eyecups', count, _ ).
noun( 'eyeful', 'eyefuls', count, _ ).
noun( 'eyeglass', 'eyeglasses', count, _ ).
noun( 'eyelash', 'eyelashes', count, _ ).
noun( 'eyelet', 'eyelets', count, _ ).
noun( 'eyelid', 'eyelids', count, _ ).
noun( 'eyepiece', 'eyepieces', count, _ ).
noun( 'eyeshot', '-', mass, _ ).
noun( 'eyesight', '-', mass, _ ).
noun( 'eyesore', 'eyesores', count, _ ).
noun( 'eyestrain', '-', mass, _ ).
noun( 'eyetooth', 'eyeteeth', count, _ ).
noun( 'eyewash', '-', mass, _ ).
noun( 'eyewitness', 'eyewitnesses', count, _ ).
noun( 'eyrie', 'eyries', count, _ ).
noun( 'eyry', 'eyries', count, _ ).
noun( 'f', '-', count, _ ).
noun( 'f\"uhrer', 'f\"uhrers', count, _ ).
noun( 'f^ete', 'f^etes', count, _ ).
noun( 'fa', '-', count, _ ).
noun( 'fa<cade', 'fa<cades', count, _ ).
noun( 'fable', 'fables', both, _ ).
noun( 'fabric', 'fabrics', both, _ ).
noun( 'fabrication', 'fabrications', both, _ ).
noun( 'face', 'faces', count, _ ).
noun( 'face-ache', '-', mass, _ ).
noun( 'face-card', 'face-cards', count, _ ).
noun( 'face-cloth', 'face-cloths', count, _ ).
noun( 'face-cream', 'face-creams', both, _ ).
noun( 'face-lift', 'face-lifts', count, _ ).
noun( 'face-lifting', 'face-liftings', count, _ ).
noun( 'face-pack', 'face-packs', count, _ ).
noun( 'face-powder', 'face-powders', both, _ ).
noun( 'face-saver', 'face-savers', count, _ ).
noun( 'face-saving', '-', mass, _ ).
noun( 'facer', 'facers', count, _ ).
noun( 'facet', 'facets', count, _ ).
noun( 'facetiousness', '-', mass, _ ).
noun( 'facia', 'facias', count, _ ).
noun( 'facial', 'facials', count, _ ).
noun( 'facility', 'facilities', both, _ ).
noun( 'facing', 'facings', count, _ ).
noun( 'facsimile', 'facsimiles', count, _ ).
noun( 'fact', 'facts', both, _ ).
noun( 'faction', 'factions', both, _ ).
noun( 'factor', 'factors', count, _ ).
noun( 'factory', 'factories', count, _ ).
noun( 'factotum', 'factotums', count, _ ).
noun( 'faculty', 'faculties', count, _ ).
noun( 'fad', 'fads', count, _ ).
noun( 'faerie', 'faeries', count, _ ).
noun( 'faery', 'faeries', count, _ ).
noun( 'fag', 'fags', both, _ ).
noun( 'fag-end', 'fag-ends', count, _ ).
noun( 'faggot', 'faggots', count, _ ).
noun( 'faience', '-', mass, _ ).
noun( 'fail', '-', mass, _ ).
noun( 'failing', 'failings', count, _ ).
noun( 'failure', 'failures', both, _ ).
noun( 'faint', 'faints', count, _ ).
noun( 'faintness', '-', mass, _ ).
noun( 'fair', 'fairs', count, _ ).
noun( 'fairground', 'fairgrounds', count, _ ).
noun( 'fairness', '-', mass, _ ).
noun( 'fairway', 'fairways', count, _ ).
noun( 'fairy', 'fairies', count, _ ).
noun( 'fairyland', 'fairylands', count, _ ).
noun( 'fairytale', 'fairytales', count, _ ).
noun( 'fait accompli', 'faits accomplis', count, _ ).
noun( 'faith', 'faiths', both, _ ).
noun( 'faith-healing', '-', mass, _ ).
noun( 'faithfulness', '-', mass, _ ).
noun( 'faithlessness', '-', mass, _ ).
noun( 'fake', 'fakes', count, _ ).
noun( 'fakir', 'fakirs', count, _ ).
noun( 'falcon', 'falcons', count, _ ).
noun( 'falconry', '-', mass, _ ).
noun( 'fall', 'falls', count, _ ).
noun( 'fallacy', 'fallacies', both, _ ).
noun( 'fallibility', '-', mass, _ ).
noun( 'fallout', '-', mass, _ ).
noun( 'fallow', '-', mass, _ ).
noun( 'fallow-deer', 'fallow-deer', count, _ ).
noun( 'falsehood', 'falsehoods', both, _ ).
noun( 'falseness', '-', mass, _ ).
noun( 'falsetto', 'falsettos', count, _ ).
noun( 'falsification', 'falsifications', both, _ ).
noun( 'falsity', 'falsities', both, _ ).
noun( 'fame', '-', mass, _ ).
noun( 'familiar', 'familiars', count, _ ).
noun( 'familiarity', 'familiarities', both, _ ).
noun( 'family', 'families', both, _ ).
noun( 'famine', 'famines', both, _ ).
noun( 'fan', 'fans', count, _ ).
noun( 'fan-belt', 'fan-belts', count, _ ).
noun( 'fanatic', 'fanatics', count, _ ).
noun( 'fanaticism', 'fanaticisms', both, _ ).
noun( 'fancier', 'fanciers', count, _ ).
noun( 'fancy', 'fancies', both, _ ).
noun( 'fandango', 'fandangos', count, _ ).
noun( 'fanfare', 'fanfares', count, _ ).
noun( 'fang', 'fangs', count, _ ).
noun( 'fanlight', 'fanlights', count, _ ).
noun( 'fanny', 'fannies', count, _ ).
noun( 'fantan', '-', mass, _ ).
noun( 'fantasia', 'fantasias', count, _ ).
noun( 'fantasy', 'fantasies', both, _ ).
noun( 'farce', 'farces', both, _ ).
noun( 'fare', 'fares', both, _ ).
noun( 'fare-stage', 'fare-stages', count, _ ).
noun( 'farewell', 'farewells', count, _ ).
noun( 'farm', 'farms', count, _ ).
noun( 'farmer', 'farmers', count, _ ).
noun( 'farmhand', 'farmhands', count, _ ).
noun( 'farmhouse', 'farmhouses', count, _ ).
noun( 'farmstead', 'farmsteads', count, _ ).
noun( 'farmyard', 'farmyards', count, _ ).
noun( 'farrago', 'farragos', count, _ ).
noun( 'farrier', 'farriers', count, _ ).
noun( 'farrow', 'farrows', count, _ ).
noun( 'fart', 'farts', count, _ ).
noun( 'farthing', 'farthings', count, _ ).
noun( 'fascia', 'fascias', count, _ ).
noun( 'fascination', 'fascinations', both, _ ).
noun( 'fascism', '-', mass, _ ).
noun( 'fascist', 'fascists', count, _ ).
noun( 'fashion', 'fashions', both, _ ).
noun( 'fast', 'fasts', count, _ ).
noun( 'fastener', 'fasteners', count, _ ).
noun( 'fastening', 'fastenings', count, _ ).
noun( 'fastidiousness', '-', mass, _ ).
noun( 'fastness', 'fastnesses', both, _ ).
noun( 'fat', 'fats', both, _ ).
noun( 'fatalism', '-', mass, _ ).
noun( 'fatalist', 'fatalists', count, _ ).
noun( 'fatality', 'fatalities', both, _ ).
noun( 'fate', 'fates', both, _ ).
noun( 'fathead', 'fatheads', count, _ ).
noun( 'father', 'fathers', count, _ ).
noun( 'father-in-law', 'fathers-in-law', count, _ ).
noun( 'fatherhood', '-', mass, _ ).
noun( 'fatherland', 'fatherlands', count, _ ).
noun( 'fathom', 'fathoms', count, _ ).
noun( 'fatigue', 'fatigues', both, _ ).
noun( 'fatigue-party', 'fatigue-parties', count, _ ).
noun( 'fatness', '-', mass, _ ).
noun( 'fatuity', 'fatuities', both, _ ).
noun( 'fatuousness', '-', mass, _ ).
noun( 'faucet', 'faucets', count, _ ).
noun( 'fault', 'faults', count, _ ).
noun( 'fault-finder', 'fault-finders', count, _ ).
noun( 'fault-finding', '-', mass, _ ).
noun( 'faun', 'fauns', count, _ ).
noun( 'faux pas', 'faux pas', count, _ ).
noun( 'favour', 'favours', both, _ ).
noun( 'favourite', 'favourites', count, _ ).
noun( 'favouritism', '-', mass, _ ).
noun( 'fawn', 'fawns', count, _ ).
noun( 'fealty', 'fealties', count, _ ).
noun( 'fear', 'fears', both, _ ).
noun( 'fearfulness', '-', mass, _ ).
noun( 'fearlessness', '-', mass, _ ).
noun( 'feasibility', '-', mass, _ ).
noun( 'feast', 'feasts', count, _ ).
noun( 'feast-day', 'feast-days', count, _ ).
noun( 'feat', 'feats', count, _ ).
noun( 'feather', 'feathers', count, _ ).
noun( 'feather-boa', 'feather-boas', count, _ ).
noun( 'featherbed', 'featherbeds', count, _ ).
noun( 'featherweight', 'featherweights', count, _ ).
noun( 'feature', 'features', count, _ ).
noun( 'fecklessness', '-', mass, _ ).
noun( 'fecundity', '-', mass, _ ).
noun( 'federalism', '-', mass, _ ).
noun( 'federalist', 'federalists', count, _ ).
noun( 'federation', 'federations', both, _ ).
noun( 'fee', 'fees', both, _ ).
noun( 'feebleness', '-', mass, _ ).
noun( 'feed', 'feeds', both, _ ).
noun( 'feedback', '-', mass, _ ).
noun( 'feeder', 'feeders', count, _ ).
noun( 'feeding-bottle', 'feeding-bottles', count, _ ).
noun( 'feel', '-', count, _ ).
noun( 'feeler', 'feelers', count, _ ).
noun( 'feeling', 'feelings', both, _ ).
noun( 'feint', 'feints', count, _ ).
noun( 'feldspar', '-', mass, _ ).
noun( 'felicitation', 'felicitations', count, _ ).
noun( 'felicity', 'felicities', both, _ ).
noun( 'fellah', '-', count, _ ).
noun( 'fellow', 'fellows', count, _ ).
noun( 'fellow-feeling', '-', mass, _ ).
noun( 'fellow-traveller', 'fellow-travellers', count, _ ).
noun( 'fellowship', 'fellowships', both, _ ).
noun( 'felon', 'felons', count, _ ).
noun( 'felony', 'felonies', both, _ ).
noun( 'felspar', '-', mass, _ ).
noun( 'felt', '-', mass, _ ).
noun( 'felucca', 'feluccas', count, _ ).
noun( 'fem', '-', proper, _ ).
noun( 'female', 'females', count, _ ).
noun( 'femininity', '-', mass, _ ).
noun( 'feminism', '-', mass, _ ).
noun( 'feminist', 'feminists', count, _ ).
noun( 'femur', 'femurs', count, _ ).
noun( 'fen', 'fens', count, _ ).
noun( 'fence', 'fences', count, _ ).
noun( 'fencer', 'fencers', count, _ ).
noun( 'fencing', '-', mass, _ ).
noun( 'fender', 'fenders', count, _ ).
noun( 'fennel', '-', mass, _ ).
noun( 'feoff', 'feoffs', count, _ ).
noun( 'ferment', 'ferments', count, _ ).
noun( 'fermentation', '-', mass, _ ).
noun( 'fern', 'ferns', count, _ ).
noun( 'ferocity', 'ferocities', both, _ ).
noun( 'ferret', 'ferrets', count, _ ).
noun( 'ferroconcrete', '-', mass, _ ).
noun( 'ferrule', 'ferrules', count, _ ).
noun( 'ferry', 'ferries', count, _ ).
noun( 'ferryboat', 'ferryboats', count, _ ).
noun( 'ferryman', 'ferrymen', count, _ ).
noun( 'fertility', '-', mass, _ ).
noun( 'fertilization', '-', mass, _ ).
noun( 'fertilizer', 'fertilizers', both, _ ).
noun( 'ferule', 'ferules', count, _ ).
noun( 'fervency', '-', mass, _ ).
noun( 'fervour', '-', mass, _ ).
noun( 'festival', 'festivals', count, _ ).
noun( 'festivity', 'festivities', both, _ ).
noun( 'festoon', 'festoons', count, _ ).
noun( 'fete-day', 'fete-days', count, _ ).
noun( 'fetish', 'fetishes', count, _ ).
noun( 'fetlock', 'fetlocks', count, _ ).
noun( 'fetter', 'fetters', count, _ ).
noun( 'fettle', '-', mass, _ ).
noun( 'feud', 'feuds', count, _ ).
noun( 'feudalism', '-', mass, _ ).
noun( 'feudatory', 'feudatories', count, _ ).
noun( 'fever', 'fevers', both, _ ).
noun( 'fewness', '-', mass, _ ).
noun( 'fez', 'fezes', count, _ ).
noun( 'fianc_e', 'fianc_es', count, _ ).
noun( 'fianc_ee', 'fianc_ees', count, _ ).
noun( 'fiasco', 'fiascos', count, _ ).
noun( 'fiat', 'fiats', count, _ ).
noun( 'fib', 'fibs', count, _ ).
noun( 'fibber', 'fibbers', count, _ ).
noun( 'fibbing', '-', mass, _ ).
noun( 'fibre', 'fibres', both, _ ).
noun( 'fibreboard', '-', mass, _ ).
noun( 'fibreglass', '-', mass, _ ).
noun( 'fibrositis', '-', mass, _ ).
noun( 'fibula', 'fibulas', count, _ ).
noun( 'fickleness', '-', mass, _ ).
noun( 'fiction', 'fictions', both, _ ).
noun( 'fiddle', 'fiddles', count, _ ).
noun( 'fiddler', 'fiddlers', count, _ ).
noun( 'fiddlestick', 'fiddlesticks', count, _ ).
noun( 'fidelity', '-', mass, _ ).
noun( 'fidget', 'fidgets', count, _ ).
noun( 'fief', 'fiefs', count, _ ).
noun( 'field', 'fields', count, _ ).
noun( 'field-hospital', 'field-hospitals', count, _ ).
noun( 'field-officer', 'field-officers', count, _ ).
noun( 'fielder', 'fielders', count, _ ).
noun( 'fieldsman', 'fieldsmen', count, _ ).
noun( 'fieldwork', 'fieldworks', both, _ ).
noun( 'fiend', 'fiends', count, _ ).
noun( 'fierceness', '-', mass, _ ).
noun( 'fieriness', '-', mass, _ ).
noun( 'fiesta', 'fiestas', count, _ ).
noun( 'fife', 'fifes', count, _ ).
noun( 'fifteen', 'fifteens', count, _ ).
noun( 'fifteenth', 'fifteenths', count, _ ).
noun( 'fifth', 'fifths', count, _ ).
noun( 'fiftieth', 'fiftieths', count, _ ).
noun( 'fifty', 'fifties', count, _ ).
noun( 'fig', 'figs', count, _ ).
noun( 'fig', '-', count, _ ).
noun( 'fig-leaf', 'fig-leaves', count, _ ).
noun( 'fight', 'fights', both, _ ).
noun( 'fighter', 'fighters', count, _ ).
noun( 'fighting', '-', mass, _ ).
noun( 'figment', 'figments', count, _ ).
noun( 'figure', 'figures', count, _ ).
noun( 'figurehead', 'figureheads', count, _ ).
noun( 'filament', 'filaments', count, _ ).
noun( 'filature', 'filatures', count, _ ).
noun( 'filbert', 'filberts', count, _ ).
noun( 'file', 'files', count, _ ).
noun( 'filibuster', 'filibusters', count, _ ).
noun( 'filigree', '-', mass, _ ).
noun( 'fill', 'fills', both, _ ).
noun( 'fillet', 'fillets', count, _ ).
noun( 'filling', 'fillings', both, _ ).
noun( 'fillip', 'fillips', count, _ ).
noun( 'filly', 'fillies', count, _ ).
noun( 'film', 'films', both, _ ).
noun( 'film-star', 'film-stars', count, _ ).
noun( 'filter', 'filters', count, _ ).
noun( 'filth', '-', mass, _ ).
noun( 'filthiness', '-', mass, _ ).
noun( 'filtrate', 'filtrates', count, _ ).
noun( 'filtration', '-', mass, _ ).
noun( 'fin', 'fins', count, _ ).
noun( 'final', 'finals', count, _ ).
noun( 'finale', 'finales', count, _ ).
noun( 'finalist', 'finalists', count, _ ).
noun( 'finality', '-', mass, _ ).
noun( 'finance', 'finances', both, _ ).
noun( 'financier', 'financiers', count, _ ).
noun( 'finch', 'finches', count, _ ).
noun( 'find', 'finds', count, _ ).
noun( 'finder', 'finders', count, _ ).
noun( 'finding', 'findings', count, _ ).
noun( 'fine', 'fines', count, _ ).
noun( 'fineness', '-', mass, _ ).
noun( 'finery', '-', mass, _ ).
noun( 'finesse', 'finesses', both, _ ).
noun( 'finger', 'fingers', count, _ ).
noun( 'finger-alphabet', 'finger-alphabets', count, _ ).
noun( 'finger-bowl', 'finger-bowls', count, _ ).
noun( 'finger-plate', 'finger-plates', count, _ ).
noun( 'finger-post', 'finger-posts', count, _ ).
noun( 'fingerboard', 'fingerboards', count, _ ).
noun( 'fingermark', 'fingermarks', count, _ ).
noun( 'fingernail', 'fingernails', count, _ ).
noun( 'fingerprint', 'fingerprints', count, _ ).
noun( 'fingerstall', 'fingerstalls', count, _ ).
noun( 'fingertip', 'fingertips', count, _ ).
noun( 'finis', '-', count, _ ).
noun( 'finish', 'finishes', both, _ ).
noun( 'finnan', '-', mass, _ ).
noun( 'finnan haddie', '-', mass, _ ).
noun( 'finnan haddock', '-', mass, _ ).
noun( 'fiord', 'fiords', count, _ ).
noun( 'fir', 'firs', both, _ ).
noun( 'fir-cone', 'fir-cones', count, _ ).
noun( 'fire', 'fires', both, _ ).
noun( 'fire-alarm', 'fire-alarms', count, _ ).
noun( 'fire-brigade', 'fire-brigades', count, _ ).
noun( 'fire-control', '-', mass, _ ).
noun( 'fire-eater', 'fire-eaters', count, _ ).
noun( 'fire-engine', 'fire-engines', count, _ ).
noun( 'fire-escape', 'fire-escapes', count, _ ).
noun( 'fire-extinguisher', 'fire-extinguishers', count, _ ).
noun( 'fire-fighter', 'fire-fighters', count, _ ).
noun( 'fire-hose', 'fire-hoses', count, _ ).
noun( 'fire-power', '-', mass, _ ).
noun( 'fire-raising', '-', mass, _ ).
noun( 'fire-walker', 'fire-walkers', count, _ ).
noun( 'fire-walking', '-', mass, _ ).
noun( 'fire-watcher', 'fire-watchers', count, _ ).
noun( 'fire-watching', '-', mass, _ ).
noun( 'firearm', 'firearms', count, _ ).
noun( 'fireball', 'fireballs', count, _ ).
noun( 'firebird', 'firebirds', count, _ ).
noun( 'firebomb', 'firebombs', count, _ ).
noun( 'firebox', 'fireboxes', count, _ ).
noun( 'firebrand', 'firebrands', count, _ ).
noun( 'firebreak', 'firebreaks', count, _ ).
noun( 'firebrick', 'firebricks', count, _ ).
noun( 'firebug', 'firebugs', count, _ ).
noun( 'fireclay', '-', mass, _ ).
noun( 'firecracker', 'firecrackers', count, _ ).
noun( 'firedamp', '-', mass, _ ).
noun( 'firedog', 'firedogs', count, _ ).
noun( 'firefly', 'fireflies', count, _ ).
noun( 'fireguard', 'fireguards', count, _ ).
noun( 'firelight', 'firelights', count, _ ).
noun( 'firelighter', 'firelighters', count, _ ).
noun( 'fireman', 'firemen', count, _ ).
noun( 'fireplace', 'fireplaces', count, _ ).
noun( 'fireside', 'firesides', count, _ ).
noun( 'firestone', '-', mass, _ ).
noun( 'firewater', '-', mass, _ ).
noun( 'firewood', '-', mass, _ ).
noun( 'firework', 'fireworks', count, _ ).
noun( 'firing-line', 'firing-lines', count, _ ).
noun( 'firing-party', 'firing-parties', count, _ ).
noun( 'firing-squad', 'firing-squads', count, _ ).
noun( 'firkin', 'firkins', count, _ ).
noun( 'firm', 'firms', count, _ ).
noun( 'firmament', 'firmaments', count, _ ).
noun( 'firmness', '-', mass, _ ).
noun( 'first', 'firsts', count, _ ).
noun( 'first-nighter', 'first-nighters', count, _ ).
noun( 'firstborn', 'firstborns', count, _ ).
noun( 'firth', 'firths', count, _ ).
noun( 'fish', 'fish', both, _ ).
noun( 'fish-hook', 'fish-hooks', count, _ ).
noun( 'fish-knife', 'fish-knives', count, _ ).
noun( 'fish-slice', 'fish-slices', count, _ ).
noun( 'fishball', 'fishballs', count, _ ).
noun( 'fishbone', 'fishbones', count, _ ).
noun( 'fishcake', 'fishcakes', count, _ ).
noun( 'fisher', 'fishers', count, _ ).
noun( 'fisherman', 'fishermen', count, _ ).
noun( 'fishery', 'fisheries', count, _ ).
noun( 'fishing', '-', mass, _ ).
noun( 'fishing-line', 'fishing-lines', count, _ ).
noun( 'fishing-rod', 'fishing-rods', count, _ ).
noun( 'fishing-tackle', '-', mass, _ ).
noun( 'fishmonger', 'fishmongers', count, _ ).
noun( 'fishpaste', 'fishpastes', both, _ ).
noun( 'fishplate', 'fishplates', count, _ ).
noun( 'fishwife', 'fishwives', count, _ ).
noun( 'fission', '-', mass, _ ).
noun( 'fissure', 'fissures', count, _ ).
noun( 'fist', 'fists', count, _ ).
noun( 'fistula', 'fistulas', count, _ ).
noun( 'fit', 'fits', count, _ ).
noun( 'fitment', 'fitments', count, _ ).
noun( 'fitness', '-', mass, _ ).
noun( 'fitter', 'fitters', count, _ ).
noun( 'fitting', 'fittings', count, _ ).
noun( 'five', 'fives', count, _ ).
noun( 'fivepence', 'fivepences', count, _ ).
noun( 'fiver', 'fivers', count, _ ).
noun( 'fives', 'fives', mass, _ ).
noun( 'fix', 'fixes', count, _ ).
noun( 'fixation', 'fixations', both, _ ).
noun( 'fixative', 'fixatives', count, _ ).
noun( 'fixture', 'fixtures', count, _ ).
noun( 'fizz', '-', mass, _ ).
noun( 'fjord', 'fjords', count, _ ).
noun( 'flabbiness', '-', mass, _ ).
noun( 'flaccidity', '-', mass, _ ).
noun( 'flag', 'flags', count, _ ).
noun( 'flag-captain', 'flag-captains', count, _ ).
noun( 'flag-day', 'flag-days', count, _ ).
noun( 'flagellant', 'flagellants', count, _ ).
noun( 'flagellation', 'flagellations', count, _ ).
noun( 'flageolet', 'flageolets', count, _ ).
noun( 'flagon', 'flagons', count, _ ).
noun( 'flagpole', 'flagpoles', count, _ ).
noun( 'flagship', 'flagships', count, _ ).
noun( 'flagstaff', 'flagstaffs', count, _ ).
noun( 'flagstone', 'flagstones', count, _ ).
noun( 'flail', 'flails', count, _ ).
noun( 'flair', 'flairs', both, _ ).
noun( 'flak', '-', mass, _ ).
noun( 'flake', 'flakes', count, _ ).
noun( 'flakiness', '-', mass, _ ).
noun( 'flambeau', 'flambeaus', count, _ ).
noun( 'flamboyance', '-', mass, _ ).
noun( 'flame', 'flames', both, _ ).
noun( 'flamethrower', 'flamethrowers', count, _ ).
noun( 'flamingo', 'flamingos', count, _ ).
noun( 'flan', 'flans', count, _ ).
noun( 'flange', 'flanges', count, _ ).
noun( 'flank', 'flanks', count, _ ).
noun( 'flannel', 'flannels', both, _ ).
noun( 'flannelette', '-', mass, _ ).
noun( 'flap', 'flaps', count, _ ).
noun( 'flapjack', 'flapjacks', both, _ ).
noun( 'flapper', 'flappers', count, _ ).
noun( 'flare', 'flares', both, _ ).
noun( 'flare-path', 'flare-paths', count, _ ).
noun( 'flare-up', 'flare-ups', count, _ ).
noun( 'flash', 'flashes', count, _ ).
noun( 'flashback', 'flashbacks', count, _ ).
noun( 'flashbulb', 'flashbulbs', count, _ ).
noun( 'flashgun', 'flashguns', count, _ ).
noun( 'flashlight', 'flashlights', count, _ ).
noun( 'flashpoint', 'flashpoints', count, _ ).
noun( 'flask', 'flasks', count, _ ).
noun( 'flat', 'flats', count, _ ).
noun( 'flat-car', 'flat-cars', count, _ ).
noun( 'flat-iron', 'flat-irons', count, _ ).
noun( 'flatfish', 'flatfish', count, _ ).
noun( 'flatlet', 'flatlets', count, _ ).
noun( 'flatness', '-', mass, _ ).
noun( 'flatterer', 'flatterers', count, _ ).
noun( 'flattery', 'flatteries', both, _ ).
noun( 'flattop', 'flattops', count, _ ).
noun( 'flatulence', '-', mass, _ ).
noun( 'flautist', 'flautists', count, _ ).
noun( 'flavour', 'flavours', both, _ ).
noun( 'flavouring', 'flavourings', both, _ ).
noun( 'flaw', 'flaws', count, _ ).
noun( 'flax', '-', mass, _ ).
noun( 'flea', 'fleas', count, _ ).
noun( 'flea-bite', 'flea-bites', count, _ ).
noun( 'fleapit', 'fleapits', count, _ ).
noun( 'fleck', 'flecks', count, _ ).
noun( 'fledgeling', 'fledgelings', count, _ ).
noun( 'fledgling', 'fledglings', count, _ ).
noun( 'fleece', 'fleeces', both, _ ).
noun( 'fleet', 'fleets', count, _ ).
noun( 'fleetness', '-', mass, _ ).
noun( 'flesh', '-', mass, _ ).
noun( 'flesh-wound', 'flesh-wounds', count, _ ).
noun( 'fleur-de-lis', 'fleurs-de-lis', count, _ ).
noun( 'fleur-de-lys', 'fleurs-de-lys', count, _ ).
noun( 'flex', 'flexes', both, _ ).
noun( 'flexibility', '-', mass, _ ).
noun( 'flibbertigibbet', 'flibbertigibbets', count, _ ).
noun( 'flick', 'flicks', count, _ ).
noun( 'flick-knife', 'flick-knives', count, _ ).
noun( 'flicker', 'flickers', count, _ ).
noun( 'flier', 'fliers', count, _ ).
noun( 'flight', 'flights', both, _ ).
noun( 'flimsiness', '-', mass, _ ).
noun( 'flimsy', '-', mass, _ ).
noun( 'fling', 'flings', count, _ ).
noun( 'flint', 'flints', both, _ ).
noun( 'flintstone', '-', mass, _ ).
noun( 'flip', 'flips', count, _ ).
noun( 'flippancy', 'flippancies', both, _ ).
noun( 'flipper', 'flippers', count, _ ).
noun( 'flirt', 'flirts', count, _ ).
noun( 'flirtation', 'flirtations', both, _ ).
noun( 'flit', 'flits', count, _ ).
noun( 'float', 'floats', count, _ ).
noun( 'floatation', 'floatations', both, _ ).
noun( 'flock', 'flocks', count, _ ).
noun( 'floe', 'floes', count, _ ).
noun( 'flogging', 'floggings', both, _ ).
noun( 'flood', 'floods', count, _ ).
noun( 'flood-tide', 'flood-tides', count, _ ).
noun( 'floodgate', 'floodgates', count, _ ).
noun( 'floor', 'floors', count, _ ).
noun( 'floor-walker', 'floor-walkers', count, _ ).
noun( 'floorboard', 'floorboards', count, _ ).
noun( 'flooring', '-', mass, _ ).
noun( 'floozie', 'floozies', count, _ ).
noun( 'floozy', 'floozies', count, _ ).
noun( 'flop', 'flops', count, _ ).
noun( 'floriculture', '-', mass, _ ).
noun( 'florin', 'florins', count, _ ).
noun( 'florist', 'florists', count, _ ).
noun( 'floss', '-', mass, _ ).
noun( 'flotation', 'flotations', both, _ ).
noun( 'flotilla', 'flotillas', count, _ ).
noun( 'flotsam', '-', mass, _ ).
noun( 'flounce', 'flounces', count, _ ).
noun( 'flounder', 'flounders', count, _ ).
noun( 'flour', '-', mass, _ ).
noun( 'flourish', 'flourishes', count, _ ).
noun( 'flow', '-', count, _ ).
noun( 'flower', 'flowers', count, _ ).
noun( 'flower-girl', 'flower-girls', count, _ ).
noun( 'flowerbed', 'flowerbeds', count, _ ).
noun( 'flowerpot', 'flowerpots', count, _ ).
noun( 'flu', '-', mass, _ ).
noun( 'fluctuation', 'fluctuations', both, _ ).
noun( 'flue', 'flues', count, _ ).
noun( 'fluency', '-', mass, _ ).
noun( 'fluff', '-', mass, _ ).
noun( 'fluid', 'fluids', both, _ ).
noun( 'fluidity', 'fluidities', both, _ ).
noun( 'fluke', 'flukes', count, _ ).
noun( 'flume', 'flumes', count, _ ).
noun( 'flunkey', 'flunkeys', count, _ ).
noun( 'flunky', 'flunkies', count, _ ).
noun( 'fluorescence', 'fluorescences', both, _ ).
noun( 'fluoridation', 'fluoridations', count, _ ).
noun( 'fluoride', 'fluorides', both, _ ).
noun( 'fluoridization', 'fluoridizations', count, _ ).
noun( 'fluorine', '-', mass, _ ).
noun( 'flurry', 'flurries', count, _ ).
noun( 'flush', 'flushes', both, _ ).
noun( 'fluster', 'flusters', count, _ ).
noun( 'flute', 'flutes', count, _ ).
noun( 'fluting', '-', mass, _ ).
noun( 'flutist', 'flutists', count, _ ).
noun( 'flutter', 'flutters', both, _ ).
noun( 'flux', 'fluxes', both, _ ).
noun( 'fly', 'flies', count, _ ).
noun( 'fly-fishing', '-', mass, _ ).
noun( 'fly-swat', 'fly-swats', count, _ ).
noun( 'fly-swatter', 'fly-swatters', count, _ ).
noun( 'flycatcher', 'flycatchers', count, _ ).
noun( 'flyer', 'flyers', count, _ ).
noun( 'flying-bomb', 'flying-bombs', count, _ ).
noun( 'flying-fish', 'flying-fish', count, _ ).
noun( 'flying-fox', 'flying-foxes', count, _ ).
noun( 'flying-squad', 'flying-squads', count, _ ).
noun( 'flyleaf', 'flyleaves', count, _ ).
noun( 'flyover', 'flyovers', count, _ ).
noun( 'flypaper', 'flypapers', count, _ ).
noun( 'flypast', 'flypasts', count, _ ).
noun( 'flytrap', 'flytraps', count, _ ).
noun( 'flyweight', 'flyweights', count, _ ).
noun( 'flywheel', 'flywheels', count, _ ).
noun( 'fo\'c\'sle', 'fo\'c\'sles', count, _ ).
noun( 'foal', 'foals', count, _ ).
noun( 'foam', 'foams', both, _ ).
noun( 'foam-rubber', '-', mass, _ ).
noun( 'focus', 'focuses', count, _ ).
noun( 'fodder', '-', mass, _ ).
noun( 'foe', 'foes', count, _ ).
noun( 'foetus', 'foetuses', count, _ ).
noun( 'fog', 'fogs', both, _ ).
noun( 'fogbank', 'fogbanks', count, _ ).
noun( 'fogey', 'fogeys', count, _ ).
noun( 'foghorn', 'foghorns', count, _ ).
noun( 'foglamp', 'foglamps', count, _ ).
noun( 'fogsignal', 'fogsignals', count, _ ).
noun( 'foible', 'foibles', count, _ ).
noun( 'foil', 'foils', both, _ ).
noun( 'fold', 'folds', count, _ ).
noun( 'folder', 'folders', count, _ ).
noun( 'foliage', '-', mass, _ ).
noun( 'folio', 'folios', count, _ ).
noun( 'folk', 'folks', count, _ ).
noun( 'folk-dance', 'folk-dances', count, _ ).
noun( 'folklore', '-', mass, _ ).
noun( 'folksong', 'folksongs', count, _ ).
noun( 'folktale', 'folktales', count, _ ).
noun( 'follow-on', 'follow-ons', count, _ ).
noun( 'follow-through', 'follow-throughs', count, _ ).
noun( 'follow-up', 'follow-ups', count, _ ).
noun( 'follower', 'followers', count, _ ).
noun( 'following', 'followings', count, _ ).
noun( 'folly', 'follies', both, _ ).
noun( 'fomentation', 'fomentations', both, _ ).
noun( 'fondant', 'fondants', count, _ ).
noun( 'fondness', '-', mass, _ ).
noun( 'font', 'fonts', count, _ ).
noun( 'food', 'foods', both, _ ).
noun( 'foodstuff', 'foodstuffs', count, _ ).
noun( 'fool', 'fools', count, _ ).
noun( 'foolery', '-', mass, _ ).
noun( 'foolhardiness', '-', mass, _ ).
noun( 'foolishness', '-', mass, _ ).
noun( 'foolscap', '-', mass, _ ).
noun( 'foot', 'feet', count, _ ).
noun( 'foot-and-mouth', '-', mass, _ ).
noun( 'foot-bath', 'foot-baths', count, _ ).
noun( 'foot-pound', 'foot-pounds', count, _ ).
noun( 'foot-race', 'foot-races', count, _ ).
noun( 'foot-rot', '-', mass, _ ).
noun( 'footage', '-', mass, _ ).
noun( 'football', 'footballs', both, _ ).
noun( 'footballer', 'footballers', count, _ ).
noun( 'footboard', 'footboards', count, _ ).
noun( 'footbridge', 'footbridges', count, _ ).
noun( 'footer', 'footers', count, _ ).
noun( 'footfall', 'footfalls', count, _ ).
noun( 'footfault', 'footfaults', count, _ ).
noun( 'foothold', 'footholds', count, _ ).
noun( 'footing', '-', count, _ ).
noun( 'footman', 'footmen', count, _ ).
noun( 'footmark', 'footmarks', count, _ ).
noun( 'footnote', 'footnotes', count, _ ).
noun( 'footpath', 'footpaths', count, _ ).
noun( 'footplate', 'footplates', count, _ ).
noun( 'footprint', 'footprints', count, _ ).
noun( 'footslogger', 'footsloggers', count, _ ).
noun( 'footstep', 'footsteps', count, _ ).
noun( 'footstool', 'footstools', count, _ ).
noun( 'footwear', '-', mass, _ ).
noun( 'footwork', '-', mass, _ ).
noun( 'fop', 'fops', count, _ ).
noun( 'forage', '-', mass, _ ).
noun( 'foray', 'forays', count, _ ).
noun( 'forbear', 'forbears', count, _ ).
noun( 'forbearance', '-', mass, _ ).
noun( 'force', 'forces', both, _ ).
noun( 'force majeure', '-', mass, _ ).
noun( 'forcefulness', '-', mass, _ ).
noun( 'forcemeat', '-', mass, _ ).
noun( 'ford', 'fords', count, _ ).
noun( 'fore', '-', mass, _ ).
noun( 'forearm', 'forearms', count, _ ).
noun( 'foreboding', 'forebodings', both, _ ).
noun( 'forecast', 'forecasts', count, _ ).
noun( 'forecaster', 'forecasters', count, _ ).
noun( 'forecastle', 'forecastles', count, _ ).
noun( 'foreclosure', 'foreclosures', both, _ ).
noun( 'forecourt', 'forecourts', count, _ ).
noun( 'forefather', 'forefathers', count, _ ).
noun( 'forefinger', 'forefingers', count, _ ).
noun( 'forefoot', 'forefeet', count, _ ).
noun( 'forefront', 'forefronts', count, _ ).
noun( 'foreground', 'foregrounds', count, _ ).
noun( 'forehead', 'foreheads', count, _ ).
noun( 'foreigner', 'foreigners', count, _ ).
noun( 'foreknowledge', '-', mass, _ ).
noun( 'foreland', 'forelands', count, _ ).
noun( 'foreleg', 'forelegs', count, _ ).
noun( 'forelock', 'forelocks', count, _ ).
noun( 'foreman', 'foremen', count, _ ).
noun( 'foremast', 'foremasts', count, _ ).
noun( 'forename', 'forenames', count, _ ).
noun( 'forenoon', 'forenoons', count, _ ).
noun( 'forerunner', 'forerunners', count, _ ).
noun( 'foresail', 'foresails', count, _ ).
noun( 'foreshore', 'foreshores', count, _ ).
noun( 'foresight', '-', mass, _ ).
noun( 'foreskin', 'foreskins', count, _ ).
noun( 'forest', 'forests', both, _ ).
noun( 'forester', 'foresters', count, _ ).
noun( 'forestry', '-', mass, _ ).
noun( 'foretaste', 'foretastes', count, _ ).
noun( 'forethought', '-', mass, _ ).
noun( 'foretop', 'foretops', count, _ ).
noun( 'forewoman', 'forewomen', count, _ ).
noun( 'foreword', 'forewords', count, _ ).
noun( 'forfeit', 'forfeits', count, _ ).
noun( 'forfeiture', '-', mass, _ ).
noun( 'forge', 'forges', count, _ ).
noun( 'forger', 'forgers', count, _ ).
noun( 'forgery', 'forgeries', both, _ ).
noun( 'forget-me-not', 'forget-me-nots', count, _ ).
noun( 'forgetfulness', '-', mass, _ ).
noun( 'forging', 'forgings', count, _ ).
noun( 'forgiveness', '-', mass, _ ).
noun( 'fork', 'forks', count, _ ).
noun( 'forlornness', '-', mass, _ ).
noun( 'form', 'forms', both, _ ).
noun( 'formaldehyde', '-', mass, _ ).
noun( 'formalin', '-', mass, _ ).
noun( 'formalism', '-', mass, _ ).
noun( 'formality', 'formalities', both, _ ).
noun( 'format', 'formats', count, _ ).
noun( 'formation', 'formations', both, _ ).
noun( 'formula', 'formulas', count, _ ).
noun( 'formulation', 'formulations', both, _ ).
noun( 'fornication', '-', mass, _ ).
noun( 'forsythia', '-', mass, _ ).
noun( 'fort', 'forts', count, _ ).
noun( 'forte', 'fortes', count, _ ).
noun( 'fortieth', 'fortieths', count, _ ).
noun( 'fortification', 'fortifications', both, _ ).
noun( 'fortitude', '-', mass, _ ).
noun( 'fortnight', 'fortnights', count, _ ).
noun( 'fortress', 'fortresses', count, _ ).
noun( 'fortune', 'fortunes', both, _ ).
noun( 'forty', 'forties', count, _ ).
noun( 'forum', 'forums', count, _ ).
noun( 'forward', 'forwards', count, _ ).
noun( 'forwardness', '-', mass, _ ).
noun( 'fosse', 'fosses', count, _ ).
noun( 'fossil', 'fossils', count, _ ).
noun( 'fossilization', 'fossilizations', both, _ ).
noun( 'foster-brother', 'foster-brothers', count, _ ).
noun( 'foster-child', 'foster-children', count, _ ).
noun( 'foster-father', 'foster-fathers', count, _ ).
noun( 'foster-mother', 'foster-mothers', count, _ ).
noun( 'foster-parent', 'foster-parents', count, _ ).
noun( 'foster-sister', 'foster-sisters', count, _ ).
noun( 'foul', 'fouls', both, _ ).
noun( 'foulness', '-', mass, _ ).
noun( 'foundation', 'foundations', both, _ ).
noun( 'foundation-stone', 'foundation-stones', count, _ ).
noun( 'founder', 'founders', count, _ ).
noun( 'foundling', 'foundlings', count, _ ).
noun( 'foundress', 'foundresses', count, _ ).
noun( 'foundry', 'foundries', count, _ ).
noun( 'fount', 'founts', count, _ ).
noun( 'fountain', 'fountains', count, _ ).
noun( 'fountain-head', 'fountain-heads', count, _ ).
noun( 'fountain-pen', 'fountain-pens', count, _ ).
noun( 'four', 'fours', count, _ ).
noun( 'four-in-hand', 'four-in-hands', count, _ ).
noun( 'four-poster', 'four-posters', count, _ ).
noun( 'four-pounder', 'four-pounders', count, _ ).
noun( 'four-wheeler', 'four-wheelers', count, _ ).
noun( 'fourpence', 'fourpences', count, _ ).
noun( 'fourscore', '-', count, _ ).
noun( 'foursome', 'foursomes', count, _ ).
noun( 'fourteen', 'fourteens', count, _ ).
noun( 'fourteenth', 'fourteenths', count, _ ).
noun( 'fourth', 'fourths', count, _ ).
noun( 'fowl', 'fowls', both, _ ).
noun( 'fowl-run', 'fowl-runs', count, _ ).
noun( 'fowler', 'fowlers', count, _ ).
noun( 'fowlingpiece', 'fowlingpieces', count, _ ).
noun( 'fowlpest', '-', mass, _ ).
noun( 'fox', 'foxes', count, _ ).
noun( 'fox-terrier', 'fox-terriers', count, _ ).
noun( 'foxglove', 'foxgloves', count, _ ).
noun( 'foxhole', 'foxholes', count, _ ).
noun( 'foxhound', 'foxhounds', count, _ ).
noun( 'foxhunt', 'foxhunts', count, _ ).
noun( 'foxhunter', 'foxhunters', count, _ ).
noun( 'foxtrot', 'foxtrots', count, _ ).
noun( 'foyer', 'foyers', count, _ ).
noun( 'fracas', '-', count, _ ).
noun( 'fraction', 'fractions', count, _ ).
noun( 'fractiousness', '-', mass, _ ).
noun( 'fracture', 'fractures', both, _ ).
noun( 'fragility', '-', mass, _ ).
noun( 'fragment', 'fragments', count, _ ).
noun( 'fragmentation', 'fragmentations', both, _ ).
noun( 'fragrance', 'fragrances', both, _ ).
noun( 'frailty', 'frailties', both, _ ).
noun( 'frame', 'frames', count, _ ).
noun( 'frame-up', 'frame-ups', count, _ ).
noun( 'framework', 'frameworks', count, _ ).
noun( 'franc', 'francs', count, _ ).
noun( 'franchise', 'franchises', both, _ ).
noun( 'frankfurter', 'frankfurters', count, _ ).
noun( 'frankincense', '-', mass, _ ).
noun( 'franking-machine', 'franking-machines', count, _ ).
noun( 'franklin', 'franklins', count, _ ).
noun( 'frankness', '-', mass, _ ).
noun( 'fraternity', 'fraternities', both, _ ).
noun( 'fraternization', '-', mass, _ ).
noun( 'fratricide', 'fratricides', both, _ ).
noun( 'fraud', 'frauds', both, _ ).
noun( 'fray', 'frays', count, _ ).
noun( 'frazzle', 'frazzles', count, _ ).
noun( 'freak', 'freaks', count, _ ).
noun( 'freak-out', 'freak-outs', count, _ ).
noun( 'freakishness', '-', mass, _ ).
noun( 'freckle', 'freckles', count, _ ).
noun( 'free-for-all', 'free-for-alls', count, _ ).
noun( 'free-list', 'free-lists', count, _ ).
noun( 'free-liver', 'free-livers', count, _ ).
noun( 'free-living', '-', mass, _ ).
noun( 'free-thinker', 'free-thinkers', count, _ ).
noun( 'free-thought', '-', mass, _ ).
noun( 'free-trader', 'free-traders', count, _ ).
noun( 'freebooter', 'freebooters', count, _ ).
noun( 'freedman', 'freedmen', count, _ ).
noun( 'freedom', 'freedoms', both, _ ).
noun( 'freehold', 'freeholds', count, _ ).
noun( 'freeholder', 'freeholders', count, _ ).
noun( 'freelance', 'freelances', count, _ ).
noun( 'freeman', 'freemen', count, _ ).
noun( 'freesia', 'freesias', count, _ ).
noun( 'freestone', '-', mass, _ ).
noun( 'freestyle', '-', mass, _ ).
noun( 'freeway', 'freeways', count, _ ).
noun( 'freeze', 'freezes', count, _ ).
noun( 'freezer', 'freezers', count, _ ).
noun( 'freezing-mixture', 'freezing-mixtures', count, _ ).
noun( 'freezing-point', 'freezing-points', count, _ ).
noun( 'freight', '-', mass, _ ).
noun( 'freight-train', 'freight-trains', count, _ ).
noun( 'freighter', 'freighters', count, _ ).
noun( 'freightliner', 'freightliners', count, _ ).
noun( 'frenzy', '-', mass, _ ).
noun( 'frequency', 'frequencies', both, _ ).
noun( 'fresco', 'frescos', both, _ ).
noun( 'fresher', 'freshers', count, _ ).
noun( 'freshman', 'freshmen', count, _ ).
noun( 'freshness', '-', mass, _ ).
noun( 'fret', 'frets', count, _ ).
noun( 'fretsaw', 'fretsaws', count, _ ).
noun( 'fretwork', '-', mass, _ ).
noun( 'friability', '-', mass, _ ).
noun( 'friar', 'friars', count, _ ).
noun( 'fricassee', 'fricassees', both, _ ).
noun( 'fricative', 'fricatives', count, _ ).
noun( 'friction', 'frictions', both, _ ).
noun( 'fridge', 'fridges', count, _ ).
noun( 'friend', 'friends', count, _ ).
noun( 'friendlessness', '-', mass, _ ).
noun( 'friendliness', '-', mass, _ ).
noun( 'friendship', 'friendships', both, _ ).
noun( 'frier', 'friers', count, _ ).
noun( 'frieze', 'friezes', count, _ ).
noun( 'frigate', 'frigates', count, _ ).
noun( 'fright', 'frights', both, _ ).
noun( 'frightfulness', '-', mass, _ ).
noun( 'frigidity', '-', mass, _ ).
noun( 'frill', 'frills', count, _ ).
noun( 'fringe', 'fringes', count, _ ).
noun( 'frippery', 'fripperies', both, _ ).
noun( 'frisson', 'frissons', count, _ ).
noun( 'fritter', 'fritters', count, _ ).
noun( 'frivolity', 'frivolities', both, _ ).
noun( 'frock', 'frocks', count, _ ).
noun( 'frock-coat', 'frock-coats', count, _ ).
noun( 'frog', 'frogs', count, _ ).
noun( 'frogman', 'frogmen', count, _ ).
noun( 'frolic', 'frolics', count, _ ).
noun( 'frond', 'fronds', count, _ ).
noun( 'front', 'fronts', both, _ ).
noun( 'front-bench', 'front-benches', count, _ ).
noun( 'front-bencher', 'front-benchers', count, _ ).
noun( 'frontage', 'frontages', count, _ ).
noun( 'frontier', 'frontiers', count, _ ).
noun( 'frontiersman', 'frontiersmen', count, _ ).
noun( 'frontispiece', 'frontispieces', count, _ ).
noun( 'frost', 'frosts', both, _ ).
noun( 'frostbite', '-', mass, _ ).
noun( 'frostiness', '-', mass, _ ).
noun( 'frosting', '-', mass, _ ).
noun( 'froth', '-', mass, _ ).
noun( 'frothiness', '-', mass, _ ).
noun( 'frown', 'frowns', count, _ ).
noun( 'fructification', '-', mass, _ ).
noun( 'frugality', 'frugalities', both, _ ).
noun( 'fruit', 'fruits', both, _ ).
noun( 'fruit-fly', 'fruit-flies', count, _ ).
noun( 'fruitcake', 'fruitcakes', count, _ ).
noun( 'fruiterer', 'fruiterers', count, _ ).
noun( 'fruitfulness', '-', mass, _ ).
noun( 'fruition', '-', mass, _ ).
noun( 'fruitlessness', '-', mass, _ ).
noun( 'frump', 'frumps', count, _ ).
noun( 'frustration', 'frustrations', both, _ ).
noun( 'fry', 'fry', count, _ ).
noun( 'fry-pan', 'fry-pans', count, _ ).
noun( 'fryer', 'fryers', count, _ ).
noun( 'frying-pan', 'frying-pans', count, _ ).
noun( 'ft', 'ft', count, _ ).
noun( 'fuchsia', 'fuchsias', count, _ ).
noun( 'fuck-all', '-', mass, _ ).
noun( 'fucker', 'fuckers', count, _ ).
noun( 'fuddy-duddy', 'fuddy-duddies', count, _ ).
noun( 'fudge', 'fudges', both, _ ).
noun( 'fuel', 'fuels', both, _ ).
noun( 'fug', 'fugs', count, _ ).
noun( 'fugitive', 'fugitives', count, _ ).
noun( 'fugue', 'fugues', count, _ ).
noun( 'fulcrum', 'fulcrums', count, _ ).
noun( 'fulfilment', 'fulfilments', count, _ ).
noun( 'fullback', 'fullbacks', count, _ ).
noun( 'fuller', 'fullers', count, _ ).
noun( 'fullness', '-', mass, _ ).
noun( 'fulmar', 'fulmars', count, _ ).
noun( 'fulmination', 'fulminations', both, _ ).
noun( 'fulsomeness', '-', mass, _ ).
noun( 'fumbler', 'fumblers', count, _ ).
noun( 'fume', 'fumes', count, _ ).
noun( 'fumigation', 'fumigations', count, _ ).
noun( 'fun', '-', mass, _ ).
noun( 'function', 'functions', count, _ ).
noun( 'functionalism', '-', mass, _ ).
noun( 'functionalist', 'functionalists', count, _ ).
noun( 'functionary', 'functionaries', count, _ ).
noun( 'fund', 'funds', count, _ ).
noun( 'fundamental', 'fundamentals', count, _ ).
noun( 'fundamentalism', '-', mass, _ ).
noun( 'fundamentalist', 'fundamentalists', count, _ ).
noun( 'funeral', 'funerals', count, _ ).
noun( 'funfair', 'funfairs', count, _ ).
noun( 'fungicide', 'fungicides', both, _ ).
noun( 'fungus', 'funguses', both, _ ).
noun( 'funicular', 'funiculars', count, _ ).
noun( 'funk', 'funks', count, _ ).
noun( 'funnel', 'funnels', count, _ ).
noun( 'funniness', '-', mass, _ ).
noun( 'funny-bone', 'funny-bones', count, _ ).
noun( 'fur', 'furs', both, _ ).
noun( 'furbelow', 'furbelows', count, _ ).
noun( 'furlong', 'furlongs', count, _ ).
noun( 'furlough', 'furloughs', both, _ ).
noun( 'furnace', 'furnaces', count, _ ).
noun( 'furniture', '-', mass, _ ).
noun( 'furore', 'furores', count, _ ).
noun( 'furrier', 'furriers', count, _ ).
noun( 'furrow', 'furrows', count, _ ).
noun( 'furtherance', '-', mass, _ ).
noun( 'furtiveness', '-', mass, _ ).
noun( 'fury', 'furies', both, _ ).
noun( 'furze', '-', mass, _ ).
noun( 'fuse', 'fuses', count, _ ).
noun( 'fuselage', 'fuselages', count, _ ).
noun( 'fusilier', 'fusiliers', count, _ ).
noun( 'fusillade', 'fusillades', count, _ ).
noun( 'fusion', 'fusions', both, _ ).
noun( 'fuss', 'fusses', both, _ ).
noun( 'fussiness', '-', mass, _ ).
noun( 'fusspot', 'fusspots', count, _ ).
noun( 'fustian', '-', mass, _ ).
noun( 'futility', 'futilities', both, _ ).
noun( 'future', 'futures', both, _ ).
noun( 'futurism', '-', mass, _ ).
noun( 'futurist', 'futurists', count, _ ).
noun( 'futurity', 'futurities', both, _ ).
noun( 'fuze', 'fuzes', count, _ ).
noun( 'fuzz', '-', mass, _ ).
noun( 'fwd', '-', proper, _ ).
noun( 'g', '-', count, _ ).
noun( 'g^ateau', 'g^ateaus', count, _ ).
noun( 'gab', '-', mass, _ ).
noun( 'gabardine', '-', mass, _ ).
noun( 'gabble', '-', mass, _ ).
noun( 'gaberdine', '-', mass, _ ).
noun( 'gable', 'gables', count, _ ).
noun( 'gadabout', 'gadabouts', count, _ ).
noun( 'gadfly', 'gadflies', count, _ ).
noun( 'gadget', 'gadgets', count, _ ).
noun( 'gadgetry', '-', mass, _ ).
noun( 'gaff', 'gaffs', count, _ ).
noun( 'gaffe', 'gaffes', count, _ ).
noun( 'gaffer', 'gaffers', count, _ ).
noun( 'gag', 'gags', count, _ ).
noun( 'gage', 'gages', count, _ ).
noun( 'gaggle', 'gaggles', count, _ ).
noun( 'gaiety', '-', mass, _ ).
noun( 'gain', 'gains', both, _ ).
noun( 'gait', 'gaits', count, _ ).
noun( 'gaiter', 'gaiters', count, _ ).
noun( 'gal', 'gals', count, _ ).
noun( 'gala', 'galas', count, _ ).
noun( 'galantine', 'galantines', count, _ ).
noun( 'galaxy', 'galaxies', count, _ ).
noun( 'gale', 'gales', count, _ ).
noun( 'gall', 'galls', both, _ ).
noun( 'gallant', 'gallants', count, _ ).
noun( 'gallantry', 'gallantries', both, _ ).
noun( 'galleon', 'galleons', count, _ ).
noun( 'gallery', 'galleries', count, _ ).
noun( 'galley', 'galleys', count, _ ).
noun( 'galley-proof', 'galley-proofs', count, _ ).
noun( 'galley-slave', 'galley-slaves', count, _ ).
noun( 'gallicism', 'gallicisms', count, _ ).
noun( 'gallon', 'gallons', count, _ ).
noun( 'gallop', 'gallops', count, _ ).
noun( 'gallows-bird', 'gallows-birds', count, _ ).
noun( 'gallstone', 'gallstones', count, _ ).
noun( 'galosh', 'galoshes', count, _ ).
noun( 'galvanism', '-', mass, _ ).
noun( 'gambit', 'gambits', count, _ ).
noun( 'gamble', 'gambles', count, _ ).
noun( 'gambler', 'gamblers', count, _ ).
noun( 'gambling', '-', mass, _ ).
noun( 'gambling-den', 'gambling-dens', count, _ ).
noun( 'gamboge', '-', mass, _ ).
noun( 'gambol', 'gambols', count, _ ).
noun( 'game', 'games', both, _ ).
noun( 'game-bag', 'game-bags', count, _ ).
noun( 'game-bird', 'game-birds', count, _ ).
noun( 'game-licence', 'game-licences', count, _ ).
noun( 'gamecock', 'gamecocks', count, _ ).
noun( 'gamekeeper', 'gamekeepers', count, _ ).
noun( 'games-master', 'games-masters', count, _ ).
noun( 'games-mistress', 'games-mistresses', count, _ ).
noun( 'gamesmanship', '-', mass, _ ).
noun( 'gaming-house', 'gaming-houses', count, _ ).
noun( 'gaming-table', 'gaming-tables', count, _ ).
noun( 'gamma', 'gammas', count, _ ).
noun( 'gammon', 'gammons', both, _ ).
noun( 'gamp', 'gamps', count, _ ).
noun( 'gamut', 'gamuts', count, _ ).
noun( 'gander', 'ganders', count, _ ).
noun( 'gang', 'gangs', count, _ ).
noun( 'ganger', 'gangers', count, _ ).
noun( 'ganglion', 'ganglions', count, _ ).
noun( 'gangplank', 'gangplanks', count, _ ).
noun( 'gangrene', '-', mass, _ ).
noun( 'gangster', 'gangsters', count, _ ).
noun( 'gangway', 'gangways', count, _ ).
noun( 'gannet', 'gannets', count, _ ).
noun( 'gantry', 'gantries', count, _ ).
noun( 'gaol', 'gaols', both, _ ).
noun( 'gaolbird', 'gaolbirds', count, _ ).
noun( 'gaolbreak', 'gaolbreaks', count, _ ).
noun( 'gaoler', 'gaolers', count, _ ).
noun( 'gap', 'gaps', count, _ ).
noun( 'gape', 'gapes', count, _ ).
noun( 'garage', 'garages', count, _ ).
noun( 'garb', '-', mass, _ ).
noun( 'garbage', '-', mass, _ ).
noun( 'garbage-can', 'garbage-cans', count, _ ).
noun( 'garden', 'gardens', both, _ ).
noun( 'garden-truck', 'garden-trucks', count, _ ).
noun( 'gardener', 'gardeners', count, _ ).
noun( 'gardenia', 'gardenias', count, _ ).
noun( 'gardening', '-', mass, _ ).
noun( 'gargle', 'gargles', count, _ ).
noun( 'gargoyle', 'gargoyles', count, _ ).
noun( 'garland', 'garlands', count, _ ).
noun( 'garlic', '-', mass, _ ).
noun( 'garment', 'garments', count, _ ).
noun( 'garner', 'garners', count, _ ).
noun( 'garnet', 'garnets', count, _ ).
noun( 'garnish', 'garnishes', count, _ ).
noun( 'garotte', 'garottes', count, _ ).
noun( 'garret', 'garrets', count, _ ).
noun( 'garrison', 'garrisons', count, _ ).
noun( 'garrotte', 'garrottes', count, _ ).
noun( 'garrulity', '-', mass, _ ).
noun( 'garter', 'garters', count, _ ).
noun( 'gas', 'gases', both, _ ).
noun( 'gas-bracket', 'gas-brackets', count, _ ).
noun( 'gas-cooker', 'gas-cookers', count, _ ).
noun( 'gas-engine', 'gas-engines', count, _ ).
noun( 'gas-fitter', 'gas-fitters', count, _ ).
noun( 'gas-helmet', 'gas-helmets', count, _ ).
noun( 'gas-holder', 'gas-holders', count, _ ).
noun( 'gas-mask', 'gas-masks', count, _ ).
noun( 'gas-meter', 'gas-meters', count, _ ).
noun( 'gas-oven', 'gas-ovens', count, _ ).
noun( 'gas-ring', 'gas-rings', count, _ ).
noun( 'gas-station', 'gas-stations', count, _ ).
noun( 'gas-stove', 'gas-stoves', count, _ ).
noun( 'gasbag', 'gasbags', count, _ ).
noun( 'gash', 'gashes', count, _ ).
noun( 'gasification', 'gasifications', both, _ ).
noun( 'gasket', 'gaskets', count, _ ).
noun( 'gaslight', '-', mass, _ ).
noun( 'gasmask', 'gasmasks', count, _ ).
noun( 'gasolene', '-', mass, _ ).
noun( 'gasoline', '-', mass, _ ).
noun( 'gasometer', 'gasometers', count, _ ).
noun( 'gasp', 'gasps', count, _ ).
noun( 'gastritis', '-', mass, _ ).
noun( 'gastronomy', '-', mass, _ ).
noun( 'gasworks', 'gasworks', count, _ ).
noun( 'gate', 'gates', count, _ ).
noun( 'gatecrasher', 'gatecrashers', count, _ ).
noun( 'gatehouse', 'gatehouses', count, _ ).
noun( 'gatepost', 'gateposts', count, _ ).
noun( 'gateway', 'gateways', count, _ ).
noun( 'gatherer', 'gatherers', count, _ ).
noun( 'gathering', 'gatherings', count, _ ).
noun( 'gaucherie', 'gaucheries', both, _ ).
noun( 'gaucho', 'gauchos', count, _ ).
noun( 'gaud', 'gauds', count, _ ).
noun( 'gaudy', 'gaudies', count, _ ).
noun( 'gauge', 'gauges', both, _ ).
noun( 'gauntlet', 'gauntlets', count, _ ).
noun( 'gauntness', '-', mass, _ ).
noun( 'gauze', '-', mass, _ ).
noun( 'gavel', 'gavels', count, _ ).
noun( 'gavotte', 'gavottes', count, _ ).
noun( 'gawk', 'gawks', count, _ ).
noun( 'gawkiness', '-', mass, _ ).
noun( 'gay', 'gays', count, _ ).
noun( 'gayness', '-', mass, _ ).
noun( 'gaze', '-', count, _ ).
noun( 'gazelle', 'gazelles', count, _ ).
noun( 'gazette', 'gazettes', count, _ ).
noun( 'gazetteer', 'gazetteers', count, _ ).
noun( 'gear', 'gears', both, _ ).
noun( 'gear-case', 'gear-cases', count, _ ).
noun( 'gearbox', 'gearboxes', count, _ ).
noun( 'gearshift', 'gearshifts', count, _ ).
noun( 'gecko', 'geckos', count, _ ).
noun( 'gee-gee', 'gee-gees', count, _ ).
noun( 'geezer', 'geezers', count, _ ).
noun( 'geisha', 'geishas', count, _ ).
noun( 'gel', 'gels', count, _ ).
noun( 'gelatin', '-', mass, _ ).
noun( 'gelatine', '-', mass, _ ).
noun( 'gelding', 'geldings', count, _ ).
noun( 'gelignite', '-', mass, _ ).
noun( 'gem', 'gems', count, _ ).
noun( 'gen', '-', mass, _ ).
noun( 'gendarme', 'gendarmes', count, _ ).
noun( 'gendarmerie', 'gendarmeries', count, _ ).
noun( 'gender', 'genders', count, _ ).
noun( 'gene', 'genes', count, _ ).
noun( 'genealogist', 'genealogists', count, _ ).
noun( 'genealogy', 'genealogies', both, _ ).
noun( 'general', 'generals', count, _ ).
noun( 'generalissimo', 'generalissimos', count, _ ).
noun( 'generality', 'generalities', both, _ ).
noun( 'generalization', 'generalizations', both, _ ).
noun( 'generation', 'generations', both, _ ).
noun( 'generator', 'generators', count, _ ).
noun( 'generosity', 'generosities', both, _ ).
noun( 'genesis', 'geneses', count, _ ).
noun( 'geneticist', 'geneticists', count, _ ).
noun( 'genetics', 'genetics', mass, _ ).
noun( 'geniality', 'genialities', both, _ ).
noun( 'genie', 'genies', count, _ ).
noun( 'genius', 'geniuses', both, _ ).
noun( 'genius loci', '-', count, _ ).
noun( 'genocide', '-', mass, _ ).
noun( 'genre', 'genres', count, _ ).
noun( 'genre-painting', 'genre-paintings', count, _ ).
noun( 'gent', 'gents', count, _ ).
noun( 'gentian', 'gentians', count, _ ).
noun( 'gentile', 'gentiles', count, _ ).
noun( 'gentility', '-', mass, _ ).
noun( 'gentleman', 'gentlemen', count, _ ).
noun( 'gentleman-at-arms', 'gentlemen-at-arms', count, _ ).
noun( 'gentleness', '-', mass, _ ).
noun( 'gentlewoman', 'gentlewomen', count, _ ).
noun( 'gentry', 'gentry', count, _ ).
noun( 'genuflection', 'genuflections', count, _ ).
noun( 'genuflexion', 'genuflexions', count, _ ).
noun( 'genuineness', '-', mass, _ ).
noun( 'genus', 'geni', count, _ ).
noun( 'geographer', 'geographers', count, _ ).
noun( 'geography', '-', mass, _ ).
noun( 'geologist', 'geologists', count, _ ).
noun( 'geology', '-', mass, _ ).
noun( 'geometry', '-', mass, _ ).
noun( 'geophysics', 'geophysics', mass, _ ).
noun( 'geopolitics', 'geopolitics', mass, _ ).
noun( 'georgette', '-', mass, _ ).
noun( 'geranium', 'geraniums', count, _ ).
noun( 'geriatrician', 'geriatricians', count, _ ).
noun( 'geriatrics', 'geriatrics', mass, _ ).
noun( 'germ', 'germs', count, _ ).
noun( 'germicide', 'germicides', count, _ ).
noun( 'germination', '-', mass, _ ).
noun( 'gerontology', '-', mass, _ ).
noun( 'gerrymander', 'gerrymanders', count, _ ).
noun( 'gerund', 'gerunds', count, _ ).
noun( 'gestation', 'gestations', count, _ ).
noun( 'gesticulation', 'gesticulations', both, _ ).
noun( 'gesture', 'gestures', both, _ ).
noun( 'get-together', 'get-togethers', count, _ ).
noun( 'get-up', 'get-ups', count, _ ).
noun( 'getaway', 'getaways', count, _ ).
noun( 'geum', 'geums', count, _ ).
noun( 'geyser', 'geysers', count, _ ).
noun( 'gharry', 'gharries', count, _ ).
noun( 'ghat', 'ghats', count, _ ).
noun( 'ghee', '-', mass, _ ).
noun( 'gherkin', 'gherkins', count, _ ).
noun( 'ghetto', 'ghettos', count, _ ).
noun( 'ghost', 'ghosts', count, _ ).
noun( 'ghost-writer', 'ghost-writers', count, _ ).
noun( 'ghostliness', '-', mass, _ ).
noun( 'ghoul', 'ghouls', count, _ ).
noun( 'giant', 'giants', count, _ ).
noun( 'giantess', 'giantesses', count, _ ).
noun( 'gibberish', '-', mass, _ ).
noun( 'gibbet', 'gibbets', count, _ ).
noun( 'gibbon', 'gibbons', count, _ ).
noun( 'gibe', 'gibes', count, _ ).
noun( 'giddiness', '-', mass, _ ).
noun( 'gift', 'gifts', both, _ ).
noun( 'gig', 'gigs', count, _ ).
noun( 'giggle', 'giggles', count, _ ).
noun( 'gigolo', 'gigolos', count, _ ).
noun( 'gild', 'gilds', count, _ ).
noun( 'gilder', 'gilders', count, _ ).
noun( 'gilding', '-', mass, _ ).
noun( 'gill', 'gills', count, _ ).
noun( 'gill', 'gills', count, _ ).
noun( 'gillie', 'gillies', count, _ ).
noun( 'gilt', '-', mass, _ ).
noun( 'gimlet', 'gimlets', count, _ ).
noun( 'gimmick', 'gimmicks', count, _ ).
noun( 'gin', 'gins', both, _ ).
noun( 'ginger', '-', mass, _ ).
noun( 'gingerbread', 'gingerbreads', both, _ ).
noun( 'gingham', '-', mass, _ ).
noun( 'gingko', 'gingkos', count, _ ).
noun( 'ginseng', '-', mass, _ ).
noun( 'gipsy', 'gipsies', count, _ ).
noun( 'giraffe', 'giraffes', count, _ ).
noun( 'girder', 'girders', count, _ ).
noun( 'girdle', 'girdles', count, _ ).
noun( 'girl', 'girls', count, _ ).
noun( 'girlfriend', 'girlfriends', count, _ ).
noun( 'girlhood', '-', mass, _ ).
noun( 'girlishness', '-', mass, _ ).
noun( 'giro', '-', mass, _ ).
noun( 'girth', 'girths', count, _ ).
noun( 'gist', 'gists', count, _ ).
noun( 'give', '-', mass, _ ).
noun( 'giveaway', 'giveaways', count, _ ).
noun( 'giver', 'givers', count, _ ).
noun( 'gizzard', 'gizzards', count, _ ).
noun( 'glacier', 'glaciers', count, _ ).
noun( 'glade', 'glades', count, _ ).
noun( 'gladiator', 'gladiators', count, _ ).
noun( 'gladiolus', 'gladioluses', count, _ ).
noun( 'gladness', '-', mass, _ ).
noun( 'glamorization', '-', mass, _ ).
noun( 'glamour', '-', mass, _ ).
noun( 'glance', 'glances', count, _ ).
noun( 'gland', 'glands', count, _ ).
noun( 'glanders', '-', mass, _ ).
noun( 'glare', 'glares', both, _ ).
noun( 'glass', 'glasses', both, _ ).
noun( 'glass-blower', 'glass-blowers', count, _ ).
noun( 'glass-cutter', 'glass-cutters', count, _ ).
noun( 'glass-wool', '-', mass, _ ).
noun( 'glassful', 'glassfuls', count, _ ).
noun( 'glasshouse', 'glasshouses', count, _ ).
noun( 'glassware', '-', mass, _ ).
noun( 'glassworks', 'glassworks', count, _ ).
noun( 'glaucoma', '-', mass, _ ).
noun( 'glaze', 'glazes', both, _ ).
noun( 'glazier', 'glaziers', count, _ ).
noun( 'gleam', 'gleams', count, _ ).
noun( 'gleaner', 'gleaners', count, _ ).
noun( 'glebe', 'glebes', count, _ ).
noun( 'glee', 'glees', both, _ ).
noun( 'glen', 'glens', count, _ ).
noun( 'glibness', '-', mass, _ ).
noun( 'glide', 'glides', count, _ ).
noun( 'glider', 'gliders', count, _ ).
noun( 'gliding', '-', mass, _ ).
noun( 'glimmer', 'glimmers', count, _ ).
noun( 'glimpse', 'glimpses', count, _ ).
noun( 'glint', 'glints', count, _ ).
noun( 'glissade', 'glissades', count, _ ).
noun( 'glister', '-', count, _ ).
noun( 'glitter', '-', mass, _ ).
noun( 'gloaming', '-', count, _ ).
noun( 'globe', 'globes', count, _ ).
noun( 'globetrotter', 'globetrotters', count, _ ).
noun( 'globule', 'globules', count, _ ).
noun( 'glockenspiel', 'glockenspiels', count, _ ).
noun( 'gloom', 'glooms', both, _ ).
noun( 'glorification', '-', mass, _ ).
noun( 'glory', 'glories', both, _ ).
noun( 'glory-hole', 'glory-holes', count, _ ).
noun( 'gloss', 'glosses', count, _ ).
noun( 'glossary', 'glossaries', count, _ ).
noun( 'glossiness', '-', mass, _ ).
noun( 'glottis', 'glottises', count, _ ).
noun( 'glove', 'gloves', count, _ ).
noun( 'glove-compartment', 'glove-compartments', count, _ ).
noun( 'glow', '-', count, _ ).
noun( 'glow-worm', 'glow-worms', count, _ ).
noun( 'glucose', '-', mass, _ ).
noun( 'glue', 'glues', both, _ ).
noun( 'glumness', '-', mass, _ ).
noun( 'glut', 'gluts', count, _ ).
noun( 'gluten', '-', mass, _ ).
noun( 'glutton', 'gluttons', count, _ ).
noun( 'gluttony', '-', mass, _ ).
noun( 'glycerine', '-', mass, _ ).
noun( 'gnat', 'gnats', count, _ ).
noun( 'gnome', 'gnomes', count, _ ).
noun( 'gnu', 'gnus', count, _ ).
noun( 'go', 'goes', count, _ ).
noun( 'go-ahead', '-', count, _ ).
noun( 'go-between', 'go-betweens', count, _ ).
noun( 'go-by', '-', count, _ ).
noun( 'go-cart', 'go-carts', count, _ ).
noun( 'go-getter', 'go-getters', count, _ ).
noun( 'go-kart', 'go-karts', count, _ ).
noun( 'go-slow', 'go-slows', count, _ ).
noun( 'goad', 'goads', count, _ ).
noun( 'goal', 'goals', count, _ ).
noun( 'goal-kick', 'goal-kicks', count, _ ).
noun( 'goal-line', 'goal-lines', count, _ ).
noun( 'goalie', 'goalies', count, _ ).
noun( 'goalkeeper', 'goalkeepers', count, _ ).
noun( 'goalmouth', 'goalmouths', count, _ ).
noun( 'goat', 'goats', count, _ ).
noun( 'goatee', 'goatees', count, _ ).
noun( 'goatherd', 'goatherds', count, _ ).
noun( 'goatskin', 'goatskins', both, _ ).
noun( 'gob', 'gobs', count, _ ).
noun( 'gobbet', 'gobbets', count, _ ).
noun( 'gobble', 'gobbles', count, _ ).
noun( 'gobbledygook', '-', mass, _ ).
noun( 'gobbler', 'gobblers', count, _ ).
noun( 'goblet', 'goblets', count, _ ).
noun( 'goblin', 'goblins', count, _ ).
noun( 'god', 'gods', count, _ ).
noun( 'godchild', 'godchildren', count, _ ).
noun( 'goddaughter', 'goddaughters', count, _ ).
noun( 'goddess', 'goddesses', count, _ ).
noun( 'godfather', 'godfathers', count, _ ).
noun( 'godhead', '-', mass, _ ).
noun( 'godlessness', '-', mass, _ ).
noun( 'godliness', '-', mass, _ ).
noun( 'godmother', 'godmothers', count, _ ).
noun( 'godown', 'godowns', count, _ ).
noun( 'godparent', 'godparents', count, _ ).
noun( 'godsend', 'godsends', count, _ ).
noun( 'godson', 'godsons', count, _ ).
noun( 'godspeed', '-', mass, _ ).
noun( 'goggle-box', 'goggle-boxes', count, _ ).
noun( 'going', 'goings', both, _ ).
noun( 'going-over', 'going-overs', count, _ ).
noun( 'goitre', 'goitres', count, _ ).
noun( 'gold', '-', mass, _ ).
noun( 'gold-beater', 'gold-beaters', count, _ ).
noun( 'gold-digger', 'gold-diggers', count, _ ).
noun( 'gold-dust', '-', mass, _ ).
noun( 'gold-foil', '-', mass, _ ).
noun( 'gold-leaf', '-', mass, _ ).
noun( 'gold-plate', '-', mass, _ ).
noun( 'gold-rush', 'gold-rushes', count, _ ).
noun( 'goldfield', 'goldfields', count, _ ).
noun( 'goldfinch', 'goldfinches', count, _ ).
noun( 'goldfish', 'goldfish', count, _ ).
noun( 'goldmine', 'goldmines', count, _ ).
noun( 'goldsmith', 'goldsmiths', count, _ ).
noun( 'golf', '-', mass, _ ).
noun( 'golf-ball', 'golf-balls', count, _ ).
noun( 'golf-club', 'golf-clubs', count, _ ).
noun( 'golf-course', 'golf-courses', count, _ ).
noun( 'golf-links', 'golf-links', count, _ ).
noun( 'golfer', 'golfers', count, _ ).
noun( 'golliwog', 'golliwogs', count, _ ).
noun( 'golosh', 'goloshes', count, _ ).
noun( 'gondola', 'gondolas', count, _ ).
noun( 'gondolier', 'gondoliers', count, _ ).
noun( 'goner', 'goners', count, _ ).
noun( 'gong', 'gongs', count, _ ).
noun( 'gonorrhea', '-', mass, _ ).
noun( 'gonorrhoea', '-', mass, _ ).
noun( 'goo', '-', mass, _ ).
noun( 'good', '-', mass, _ ).
noun( 'good-fellowship', '-', mass, _ ).
noun( 'good-for-naught', 'good-for-naughts', count, _ ).
noun( 'good-for-nothing', 'good-for-nothings', count, _ ).
noun( 'good-neighbourliness', '-', mass, _ ).
noun( 'goodbye', 'goodbyes', count, _ ).
noun( 'goodness', '-', mass, _ ).
noun( 'goodwill', '-', mass, _ ).
noun( 'goody', 'goodies', count, _ ).
noun( 'goody-goody', 'goody-goodies', count, _ ).
noun( 'goof', 'goofs', count, _ ).
noun( 'googly', 'googlies', count, _ ).
noun( 'goon', 'goons', count, _ ).
noun( 'goose', 'geese', count, _ ).
noun( 'goose-flesh', '-', mass, _ ).
noun( 'goose-step', 'goose-steps', count, _ ).
noun( 'gooseberry', 'gooseberries', count, _ ).
noun( 'gopher', 'gophers', count, _ ).
noun( 'gore', '-', mass, _ ).
noun( 'gorge', 'gorges', count, _ ).
noun( 'gorilla', 'gorillas', count, _ ).
noun( 'gorse', '-', mass, _ ).
noun( 'gosling', 'goslings', count, _ ).
noun( 'gospel', 'gospels', both, _ ).
noun( 'gossamer', 'gossamers', both, _ ).
noun( 'gossip', 'gossips', both, _ ).
noun( 'gouache', '-', mass, _ ).
noun( 'gouge', 'gouges', count, _ ).
noun( 'goulash', 'goulashes', both, _ ).
noun( 'gourd', 'gourds', count, _ ).
noun( 'gourmand', 'gourmands', count, _ ).
noun( 'gourmet', 'gourmets', count, _ ).
noun( 'gout', '-', mass, _ ).
noun( 'governance', '-', mass, _ ).
noun( 'governess', 'governesses', count, _ ).
noun( 'government', 'governments', both, _ ).
noun( 'governor', 'governors', count, _ ).
noun( 'govt', '-', count, _ ).
noun( 'gown', 'gowns', count, _ ).
noun( 'grab', 'grabs', count, _ ).
noun( 'grabber', 'grabbers', count, _ ).
noun( 'grace', 'graces', both, _ ).
noun( 'graciousness', '-', mass, _ ).
noun( 'gradation', 'gradations', both, _ ).
noun( 'grade', 'grades', count, _ ).
noun( 'gradient', 'gradients', count, _ ).
noun( 'gradualness', '-', mass, _ ).
noun( 'graduate', 'graduates', count, _ ).
noun( 'graduation', 'graduations', count, _ ).
noun( 'graffito', 'graffiti', count, _ ).
noun( 'graft', 'grafts', both, _ ).
noun( 'grail', 'grails', count, _ ).
noun( 'grain', 'grains', both, _ ).
noun( 'gram', 'grams', count, _ ).
noun( 'grammar', 'grammars', both, _ ).
noun( 'grammarian', 'grammarians', count, _ ).
noun( 'gramme', 'grammes', count, _ ).
noun( 'gramophone', 'gramophones', count, _ ).
noun( 'grampus', 'grampuses', count, _ ).
noun( 'granary', 'granaries', count, _ ).
noun( 'grandad', 'grandads', count, _ ).
noun( 'grandaunt', 'grandaunts', count, _ ).
noun( 'grandchild', 'grandchildren', count, _ ).
noun( 'granddad', 'granddads', count, _ ).
noun( 'granddaughter', 'granddaughters', count, _ ).
noun( 'grandee', 'grandees', count, _ ).
noun( 'grandeur', '-', mass, _ ).
noun( 'grandfather', 'grandfathers', count, _ ).
noun( 'grandiloquence', '-', mass, _ ).
noun( 'grandma', 'grandmas', count, _ ).
noun( 'grandmother', 'grandmothers', count, _ ).
noun( 'grandnephew', 'grandnephews', count, _ ).
noun( 'grandniece', 'grandnieces', count, _ ).
noun( 'grandpa', 'grandpas', count, _ ).
noun( 'grandparent', 'grandparents', count, _ ).
noun( 'grandson', 'grandsons', count, _ ).
noun( 'grandstand', 'grandstands', count, _ ).
noun( 'granduncle', 'granduncles', count, _ ).
noun( 'grange', 'granges', count, _ ).
noun( 'granite', '-', mass, _ ).
noun( 'grannie', 'grannies', count, _ ).
noun( 'granny', 'grannies', count, _ ).
noun( 'grant', 'grants', count, _ ).
noun( 'granularity', '-', mass, _ ).
noun( 'granule', 'granules', count, _ ).
noun( 'grape', 'grapes', count, _ ).
noun( 'grape-sugar', '-', mass, _ ).
noun( 'grapefruit', 'grapefruit', both, _ ).
noun( 'grapeshot', '-', mass, _ ).
noun( 'grapevine', 'grapevines', count, _ ).
noun( 'graph', 'graphs', count, _ ).
noun( 'graphics', 'graphics', mass, _ ).
noun( 'graphite', '-', mass, _ ).
noun( 'grapnel', 'grapnels', count, _ ).
noun( 'grappling-iron', 'grappling-irons', count, _ ).
noun( 'grasp', 'grasps', count, _ ).
noun( 'grass', 'grasses', both, _ ).
noun( 'grasshopper', 'grasshoppers', count, _ ).
noun( 'grassland', 'grasslands', count, _ ).
noun( 'grate', 'grates', count, _ ).
noun( 'gratefulness', '-', mass, _ ).
noun( 'grater', 'graters', count, _ ).
noun( 'gratification', 'gratifications', both, _ ).
noun( 'grating', 'gratings', count, _ ).
noun( 'gratitude', '-', mass, _ ).
noun( 'gratuity', 'gratuities', count, _ ).
noun( 'grave', 'graves', count, _ ).
noun( 'grave', 'graves', count, _ ).
noun( 'gravel', '-', mass, _ ).
noun( 'gravestone', 'gravestones', count, _ ).
noun( 'graveyard', 'graveyards', count, _ ).
noun( 'graving dock', 'graving docks', count, _ ).
noun( 'gravitation', '-', mass, _ ).
noun( 'gravity', '-', mass, _ ).
noun( 'gravure', 'gravures', count, _ ).
noun( 'gravy', '-', mass, _ ).
noun( 'gravy-boat', 'gravy-boats', count, _ ).
noun( 'gray', 'grays', both, _ ).
noun( 'graze', 'grazes', count, _ ).
noun( 'grazier', 'graziers', count, _ ).
noun( 'grazing-land', 'grazing-lands', count, _ ).
noun( 'grease', '-', mass, _ ).
noun( 'grease-gun', 'grease-guns', count, _ ).
noun( 'greasepaint', '-', mass, _ ).
noun( 'greaser', 'greasers', count, _ ).
noun( 'greasiness', '-', mass, _ ).
noun( 'greatcoat', 'greatcoats', count, _ ).
noun( 'greatness', '-', mass, _ ).
noun( 'grebe', 'grebes', count, _ ).
noun( 'greed', '-', mass, _ ).
noun( 'greediness', '-', mass, _ ).
noun( 'green', 'greens', both, _ ).
noun( 'greenback', 'greenbacks', count, _ ).
noun( 'greenery', '-', mass, _ ).
noun( 'greenfly', '-', mass, _ ).
noun( 'greengage', 'greengages', count, _ ).
noun( 'greengrocer', 'greengrocers', count, _ ).
noun( 'greengrocery', 'greengroceries', count, _ ).
noun( 'greenhorn', 'greenhorns', count, _ ).
noun( 'greenhouse', 'greenhouses', count, _ ).
noun( 'greenishness', '-', mass, _ ).
noun( 'greensward', '-', mass, _ ).
noun( 'greenwood', 'greenwoods', count, _ ).
noun( 'greeting', 'greetings', count, _ ).
noun( 'gregariousness', '-', mass, _ ).
noun( 'gremlin', 'gremlins', count, _ ).
noun( 'grenade', 'grenades', count, _ ).
noun( 'grenadier', 'grenadiers', count, _ ).
noun( 'grey', 'greys', both, _ ).
noun( 'greybeard', 'greybeards', count, _ ).
noun( 'greyhound', 'greyhounds', count, _ ).
noun( 'grid', 'grids', count, _ ).
noun( 'griddle', 'griddles', count, _ ).
noun( 'gridiron', 'gridirons', count, _ ).
noun( 'grief', 'griefs', both, _ ).
noun( 'grievance', 'grievances', count, _ ).
noun( 'griffin', 'griffins', count, _ ).
noun( 'griffon', 'griffons', count, _ ).
noun( 'grill', 'grills', count, _ ).
noun( 'grille', 'grilles', count, _ ).
noun( 'grillroom', 'grillrooms', count, _ ).
noun( 'grimace', 'grimaces', count, _ ).
noun( 'grime', '-', mass, _ ).
noun( 'grimness', '-', mass, _ ).
noun( 'grin', 'grins', count, _ ).
noun( 'grind', 'grinds', count, _ ).
noun( 'grinder', 'grinders', count, _ ).
noun( 'grindstone', 'grindstones', count, _ ).
noun( 'grip', 'grips', count, _ ).
noun( 'grippe', 'grippes', count, _ ).
noun( 'gripsack', 'gripsacks', count, _ ).
noun( 'grist', '-', mass, _ ).
noun( 'gristle', '-', mass, _ ).
noun( 'grit', '-', mass, _ ).
noun( 'grits', 'grits', mass, _ ).
noun( 'grizzly', 'grizzlies', count, _ ).
noun( 'groan', 'groans', count, _ ).
noun( 'groat', 'groats', count, _ ).
noun( 'grocer', 'grocers', count, _ ).
noun( 'grocery', 'groceries', both, _ ).
noun( 'grog', '-', mass, _ ).
noun( 'groin', 'groins', count, _ ).
noun( 'groom', 'grooms', count, _ ).
noun( 'groove', 'grooves', count, _ ).
noun( 'groover', 'groovers', count, _ ).
noun( 'gross', 'gross', count, _ ).
noun( 'grossness', '-', mass, _ ).
noun( 'grot', 'grots', both, _ ).
noun( 'grotesque', 'grotesques', count, _ ).
noun( 'grotesqueness', '-', mass, _ ).
noun( 'grotto', 'grottos', count, _ ).
noun( 'grouch', 'grouches', count, _ ).
noun( 'ground', 'grounds', both, _ ).
noun( 'ground-bait', '-', mass, _ ).
noun( 'ground-fish', 'ground-fish', count, _ ).
noun( 'ground-plan', 'ground-plans', count, _ ).
noun( 'ground-rent', 'ground-rents', both, _ ).
noun( 'grounding', 'groundings', count, _ ).
noun( 'groundnut', 'groundnuts', count, _ ).
noun( 'groundsel', '-', mass, _ ).
noun( 'groundsheet', 'groundsheets', count, _ ).
noun( 'groundsman', 'groundsmen', count, _ ).
noun( 'groundwork', '-', mass, _ ).
noun( 'group', 'groups', count, _ ).
noun( 'grouping', 'groupings', count, _ ).
noun( 'grouse', 'grouse', count, _ ).
noun( 'grove', 'groves', count, _ ).
noun( 'groveller', 'grovellers', count, _ ).
noun( 'grower', 'growers', count, _ ).
noun( 'growl', 'growls', count, _ ).
noun( 'growler', 'growlers', count, _ ).
noun( 'grown-up', 'grown-ups', count, _ ).
noun( 'growth', 'growths', both, _ ).
noun( 'groyne', 'groynes', count, _ ).
noun( 'grub', 'grubs', both, _ ).
noun( 'grubbiness', '-', mass, _ ).
noun( 'grudge', 'grudges', count, _ ).
noun( 'gruel', '-', mass, _ ).
noun( 'gruesomeness', '-', mass, _ ).
noun( 'gruffness', '-', mass, _ ).
noun( 'grumble', 'grumbles', count, _ ).
noun( 'grumbler', 'grumblers', count, _ ).
noun( 'grumpiness', '-', mass, _ ).
noun( 'grunt', 'grunts', count, _ ).
noun( 'gryphon', 'gryphons', count, _ ).
noun( 'guano', 'guanos', both, _ ).
noun( 'guarantee', 'guarantees', count, _ ).
noun( 'guarantor', 'guarantors', count, _ ).
noun( 'guaranty', 'guaranties', count, _ ).
noun( 'guard', 'guards', both, _ ).
noun( 'guard-boat', 'guard-boats', count, _ ).
noun( 'guardhouse', 'guardhouses', count, _ ).
noun( 'guardian', 'guardians', count, _ ).
noun( 'guardianship', 'guardianships', count, _ ).
noun( 'guardrail', 'guardrails', count, _ ).
noun( 'guardroom', 'guardrooms', count, _ ).
noun( 'guardship', 'guardships', count, _ ).
noun( 'guardsman', 'guardsmen', count, _ ).
noun( 'guava', 'guavas', both, _ ).
noun( 'gudgeon', 'gudgeons', count, _ ).
noun( 'guelder rose', 'guelder roses', count, _ ).
noun( 'guerilla', 'guerillas', count, _ ).
noun( 'guerrilla', 'guerrillas', count, _ ).
noun( 'guess', 'guesses', count, _ ).
noun( 'guesstimate', 'guesstimates', count, _ ).
noun( 'guesswork', '-', mass, _ ).
noun( 'guest', 'guests', count, _ ).
noun( 'guest-night', 'guest-nights', count, _ ).
noun( 'guesthouse', 'guesthouses', count, _ ).
noun( 'guestroom', 'guestrooms', count, _ ).
noun( 'guffaw', 'guffaws', count, _ ).
noun( 'guidance', '-', mass, _ ).
noun( 'guide', 'guides', count, _ ).
noun( 'guidebook', 'guidebooks', count, _ ).
noun( 'guideline', 'guidelines', count, _ ).
noun( 'guild', 'guilds', count, _ ).
noun( 'guilder', 'guilders', count, _ ).
noun( 'guildhall', 'guildhalls', count, _ ).
noun( 'guile', '-', mass, _ ).
noun( 'guillemot', 'guillemots', count, _ ).
noun( 'guillotine', 'guillotines', count, _ ).
noun( 'guilt', '-', mass, _ ).
noun( 'guiltiness', '-', mass, _ ).
noun( 'guinea', 'guineas', count, _ ).
noun( 'guinea-fowl', 'guinea-fowl', count, _ ).
noun( 'guinea-pig', 'guinea-pigs', count, _ ).
noun( 'guise', 'guises', count, _ ).
noun( 'guitar', 'guitars', count, _ ).
noun( 'guitarist', 'guitarists', count, _ ).
noun( 'gulch', 'gulches', count, _ ).
noun( 'gulden', 'guldens', count, _ ).
noun( 'gulf', 'gulfs', count, _ ).
noun( 'gull', 'gulls', count, _ ).
noun( 'gullet', 'gullets', count, _ ).
noun( 'gullibility', '-', mass, _ ).
noun( 'gully', 'gullies', count, _ ).
noun( 'gulp', 'gulps', count, _ ).
noun( 'gum', 'gums', both, _ ).
noun( 'gumbo', 'gumbos', count, _ ).
noun( 'gumboil', 'gumboils', count, _ ).
noun( 'gumboot', 'gumboots', count, _ ).
noun( 'gumption', '-', mass, _ ).
noun( 'gumshoe', 'gumshoes', count, _ ).
noun( 'gun', 'guns', count, _ ).
noun( 'gun-carriage', 'gun-carriages', count, _ ).
noun( 'gunboat', 'gunboats', count, _ ).
noun( 'guncotton', '-', mass, _ ).
noun( 'gundog', 'gundogs', count, _ ).
noun( 'gunfire', '-', mass, _ ).
noun( 'gunman', 'gunmen', count, _ ).
noun( 'gunmetal', '-', mass, _ ).
noun( 'gunner', 'gunners', count, _ ).
noun( 'gunnery', '-', mass, _ ).
noun( 'gunny', '-', mass, _ ).
noun( 'gunplay', '-', mass, _ ).
noun( 'gunpoint', '-', count, _ ).
noun( 'gunpowder', '-', mass, _ ).
noun( 'gunroom', 'gunrooms', count, _ ).
noun( 'gunrunner', 'gunrunners', count, _ ).
noun( 'gunrunning', '-', mass, _ ).
noun( 'gunshot', 'gunshots', both, _ ).
noun( 'gunsmith', 'gunsmiths', count, _ ).
noun( 'gunwale', 'gunwales', count, _ ).
noun( 'gurgle', 'gurgles', both, _ ).
noun( 'guru', 'gurus', count, _ ).
noun( 'gush', 'gushes', count, _ ).
noun( 'gusher', 'gushers', count, _ ).
noun( 'gusset', 'gussets', count, _ ).
noun( 'gust', 'gusts', count, _ ).
noun( 'gustation', '-', mass, _ ).
noun( 'gusto', '-', mass, _ ).
noun( 'gut', 'guts', both, _ ).
noun( 'gutta-percha', '-', mass, _ ).
noun( 'gutter', 'gutters', count, _ ).
noun( 'guttersnipe', 'guttersnipes', count, _ ).
noun( 'guttural', 'gutturals', count, _ ).
noun( 'guvnor', 'guvnors', count, _ ).
noun( 'guy', 'guys', count, _ ).
noun( 'guzzler', 'guzzlers', count, _ ).
noun( 'gym', 'gyms', count, _ ).
noun( 'gymkhana', 'gymkhanas', count, _ ).
noun( 'gymnasium', 'gymnasiums', count, _ ).
noun( 'gymnast', 'gymnasts', count, _ ).
noun( 'gymnastics', 'gymnastics', mass, _ ).
noun( 'gymslip', 'gymslips', count, _ ).
noun( 'gynaecologist', 'gynaecologists', count, _ ).
noun( 'gynaecology', '-', mass, _ ).
noun( 'gynecologist', 'gynecologists', count, _ ).
noun( 'gyp', 'gyps', count, _ ).
noun( 'gypsum', '-', mass, _ ).
noun( 'gyration', 'gyrations', both, _ ).
noun( 'gyro', 'gyros', count, _ ).
noun( 'gyroscope', 'gyroscopes', count, _ ).
noun( 'h', '-', count, _ ).
noun( 'ha\'p\'orth', 'ha\'p\'orths', count, _ ).
noun( 'ha\'penny', 'ha\'pennies', count, _ ).
noun( 'habeas corpus', '-', mass, _ ).
noun( 'haberdasher', 'haberdashers', count, _ ).
noun( 'haberdashery', '-', mass, _ ).
noun( 'habit', 'habits', both, _ ).
noun( 'habitat', 'habitats', count, _ ).
noun( 'habitation', 'habitations', both, _ ).
noun( 'habitu_e', 'habitu_es', count, _ ).
noun( 'habituation', '-', mass, _ ).
noun( 'habitude', '-', mass, _ ).
noun( 'hacienda', 'haciendas', count, _ ).
noun( 'hack', 'hacks', count, _ ).
noun( 'hackney', 'hackneys', count, _ ).
noun( 'hacksaw', 'hacksaws', count, _ ).
noun( 'haddock', 'haddock', both, _ ).
noun( 'haematite', 'haematites', count, _ ).
noun( 'haemoglobin', '-', mass, _ ).
noun( 'haemophilia', '-', mass, _ ).
noun( 'haemophiliac', 'haemophiliacs', count, _ ).
noun( 'haemorrhage', 'haemorrhages', both, _ ).
noun( 'haft', 'hafts', count, _ ).
noun( 'hag', 'hags', count, _ ).
noun( 'haggis', 'haggises', count, _ ).
noun( 'hagiology', 'hagiologies', count, _ ).
noun( 'haha', 'hahas', count, _ ).
noun( 'hail', 'hails', both, _ ).
noun( 'hailstone', 'hailstones', count, _ ).
noun( 'hailstorm', 'hailstorms', count, _ ).
noun( 'hair', 'hairs', both, _ ).
noun( 'hair\'s-breadth', 'hair\'s-breadths', count, _ ).
noun( 'hair-breadth', 'hair-breadths', count, _ ).
noun( 'hair-dye', 'hair-dyes', count, _ ).
noun( 'hair-oil', 'hair-oils', both, _ ).
noun( 'hair-shirt', 'hair-shirts', count, _ ).
noun( 'hair-slide', 'hair-slides', count, _ ).
noun( 'hair-trigger', 'hair-triggers', count, _ ).
noun( 'hairbrush', 'hairbrushes', count, _ ).
noun( 'haircloth', 'haircloths', count, _ ).
noun( 'haircut', 'haircuts', count, _ ).
noun( 'hairdo', 'hairdos', count, _ ).
noun( 'hairdresser', 'hairdressers', count, _ ).
noun( 'hairdressing', '-', mass, _ ).
noun( 'hairiness', '-', mass, _ ).
noun( 'hairline', 'hairlines', count, _ ).
noun( 'hairnet', 'hairnets', count, _ ).
noun( 'hairpiece', 'hairpieces', count, _ ).
noun( 'hairpin', 'hairpins', count, _ ).
noun( 'hairsplitting', '-', mass, _ ).
noun( 'hairspring', 'hairsprings', count, _ ).
noun( 'hairstyle', 'hairstyles', count, _ ).
noun( 'hairstylist', 'hairstylists', count, _ ).
noun( 'hake', 'hake', both, _ ).
noun( 'halberd', 'halberds', count, _ ).
noun( 'halberdier', 'halberdiers', count, _ ).
noun( 'half', 'halves', count, _ ).
noun( 'half-blood', 'half-bloods', count, _ ).
noun( 'half-breed', 'half-breeds', count, _ ).
noun( 'half-brother', 'half-brothers', count, _ ).
noun( 'half-caste', 'half-castes', count, _ ).
noun( 'half-crown', 'half-crowns', count, _ ).
noun( 'half-holiday', 'half-holidays', count, _ ).
noun( 'half-hour', 'half-hours', count, _ ).
noun( 'half-pay', '-', mass, _ ).
noun( 'half-sister', 'half-sisters', count, _ ).
noun( 'half-time', '-', mass, _ ).
noun( 'half-track', 'half-tracks', count, _ ).
noun( 'half-truth', 'half-truths', count, _ ).
noun( 'half-volley', 'half-volleys', count, _ ).
noun( 'halfback', 'halfbacks', count, _ ).
noun( 'halfpenny', 'halfpennies', count, _ ).
noun( 'halfpennyworth', 'halfpennyworths', count, _ ).
noun( 'halftone', 'halftones', count, _ ).
noun( 'halfwit', 'halfwits', count, _ ).
noun( 'halibut', 'halibut', both, _ ).
noun( 'halitosis', '-', mass, _ ).
noun( 'hall', 'halls', count, _ ).
noun( 'hall-stand', 'hall-stands', count, _ ).
noun( 'hallelujah', 'hallelujahs', count, _ ).
noun( 'halliard', 'halliards', count, _ ).
noun( 'hallmark', 'hallmarks', count, _ ).
noun( 'halloo', 'halloos', count, _ ).
noun( 'hallow', 'hallows', count, _ ).
noun( 'hallucination', 'hallucinations', both, _ ).
noun( 'halma', '-', mass, _ ).
noun( 'halo', 'halos', count, _ ).
noun( 'halt', 'halts', count, _ ).
noun( 'halter', 'halters', count, _ ).
noun( 'halyard', 'halyards', count, _ ).
noun( 'ham', 'hams', both, _ ).
noun( 'hamadryad', 'hamadryads', count, _ ).
noun( 'hamburger', 'hamburgers', count, _ ).
noun( 'hamlet', 'hamlets', count, _ ).
noun( 'hammer', 'hammers', count, _ ).
noun( 'hammock', 'hammocks', count, _ ).
noun( 'hamper', 'hampers', count, _ ).
noun( 'hamster', 'hamsters', count, _ ).
noun( 'hamstring', 'hamstrings', count, _ ).
noun( 'hand', 'hands', count, _ ).
noun( 'hand-barrow', 'hand-barrows', count, _ ).
noun( 'hand-grenade', 'hand-grenades', count, _ ).
noun( 'hand-luggage', '-', mass, _ ).
noun( 'hand-me-down', 'hand-me-downs', count, _ ).
noun( 'hand-organ', 'hand-organs', count, _ ).
noun( 'hand-out', 'hand-outs', count, _ ).
noun( 'handbag', 'handbags', count, _ ).
noun( 'handbill', 'handbills', count, _ ).
noun( 'handbook', 'handbooks', count, _ ).
noun( 'handbrake', 'handbrakes', count, _ ).
noun( 'handcart', 'handcarts', count, _ ).
noun( 'handclap', 'handclaps', count, _ ).
noun( 'handcuff', 'handcuffs', count, _ ).
noun( 'handful', 'handfuls', count, _ ).
noun( 'handhold', 'handholds', count, _ ).
noun( 'handicap', 'handicaps', count, _ ).
noun( 'handicraft', 'handicrafts', both, _ ).
noun( 'handiness', '-', mass, _ ).
noun( 'handiwork', 'handiworks', both, _ ).
noun( 'handkerchief', 'handkerchiefs', count, _ ).
noun( 'handle', 'handles', count, _ ).
noun( 'handlebar', 'handlebars', count, _ ).
noun( 'handler', 'handlers', count, _ ).
noun( 'handmaid', 'handmaids', count, _ ).
noun( 'handover', 'handovers', count, _ ).
noun( 'handrail', 'handrails', count, _ ).
noun( 'handsaw', 'handsaws', count, _ ).
noun( 'handshake', 'handshakes', count, _ ).
noun( 'handshaking', '-', mass, _ ).
noun( 'handstand', 'handstands', count, _ ).
noun( 'handwork', '-', mass, _ ).
noun( 'handwriting', '-', mass, _ ).
noun( 'handyman', 'handymen', count, _ ).
noun( 'hang', '-', count, _ ).
noun( 'hang-up', 'hang-ups', count, _ ).
noun( 'hangar', 'hangars', count, _ ).
noun( 'hanger', 'hangers', count, _ ).
noun( 'hanger-on', 'hangers-on', count, _ ).
noun( 'hanging', 'hangings', both, _ ).
noun( 'hangman', 'hangmen', count, _ ).
noun( 'hangnail', 'hangnails', count, _ ).
noun( 'hangover', 'hangovers', count, _ ).
noun( 'hank', 'hanks', count, _ ).
noun( 'hankering', 'hankerings', count, _ ).
noun( 'hanky', 'hankies', count, _ ).
noun( 'hanky-panky', '-', mass, _ ).
noun( 'hansom', 'hansoms', count, _ ).
noun( 'hap', 'haps', count, _ ).
noun( 'happening', 'happenings', count, _ ).
noun( 'happiness', '-', mass, _ ).
noun( 'hara-kiri', '-', mass, _ ).
noun( 'harangue', 'harangues', count, _ ).
noun( 'harassment', '-', mass, _ ).
noun( 'harbinger', 'harbingers', count, _ ).
noun( 'harbour', 'harbours', count, _ ).
noun( 'harbourage', 'harbourages', count, _ ).
noun( 'hardback', 'hardbacks', count, _ ).
noun( 'hardboard', '-', mass, _ ).
noun( 'hardcover', 'hardcovers', count, _ ).
noun( 'hardihood', '-', mass, _ ).
noun( 'hardiness', '-', mass, _ ).
noun( 'hardliner', 'hardliners', count, _ ).
noun( 'hardness', '-', mass, _ ).
noun( 'hardship', 'hardships', both, _ ).
noun( 'hardtop', 'hardtops', count, _ ).
noun( 'hardware', '-', mass, _ ).
noun( 'hardwood', '-', mass, _ ).
noun( 'hare', 'hares', count, _ ).
noun( 'harebell', 'harebells', count, _ ).
noun( 'harelip', 'harelips', count, _ ).
noun( 'harem', 'harems', count, _ ).
noun( 'haricot', 'haricots', count, _ ).
noun( 'harlequin', 'harlequins', count, _ ).
noun( 'harlequinade', 'harlequinades', count, _ ).
noun( 'harlot', 'harlots', count, _ ).
noun( 'harm', '-', mass, _ ).
noun( 'harmattan', 'harmattans', count, _ ).
noun( 'harmonic', 'harmonics', count, _ ).
noun( 'harmonica', 'harmonicas', count, _ ).
noun( 'harmonium', 'harmoniums', count, _ ).
noun( 'harmonization', 'harmonizations', count, _ ).
noun( 'harmony', 'harmonies', both, _ ).
noun( 'harness', 'harnesses', count, _ ).
noun( 'harp', 'harps', count, _ ).
noun( 'harper', 'harpers', count, _ ).
noun( 'harpist', 'harpists', count, _ ).
noun( 'harpoon', 'harpoons', count, _ ).
noun( 'harpsichord', 'harpsichords', count, _ ).
noun( 'harpsichordist', 'harpsichordists', count, _ ).
noun( 'harpy', 'harpies', count, _ ).
noun( 'harridan', 'harridans', count, _ ).
noun( 'harrier', 'harriers', count, _ ).
noun( 'harrow', 'harrows', count, _ ).
noun( 'harshness', '-', mass, _ ).
noun( 'hart', 'harts', count, _ ).
noun( 'harum-scarum', 'harum-scarums', count, _ ).
noun( 'harvest', 'harvests', count, _ ).
noun( 'harvester', 'harvesters', count, _ ).
noun( 'has-been', 'has-beens', count, _ ).
noun( 'hash', '-', mass, _ ).
noun( 'hashish', '-', mass, _ ).
noun( 'hasp', 'hasps', count, _ ).
noun( 'hassle', 'hassles', count, _ ).
noun( 'hassock', 'hassocks', count, _ ).
noun( 'haste', '-', mass, _ ).
noun( 'hastiness', '-', mass, _ ).
noun( 'hat', 'hats', count, _ ).
noun( 'hatband', 'hatbands', count, _ ).
noun( 'hatch', 'hatches', count, _ ).
noun( 'hatchery', 'hatcheries', count, _ ).
noun( 'hatchet', 'hatchets', count, _ ).
noun( 'hatching', '-', mass, _ ).
noun( 'hatchway', 'hatchways', count, _ ).
noun( 'hate', 'hates', both, _ ).
noun( 'hatful', 'hatfuls', count, _ ).
noun( 'hatpin', 'hatpins', count, _ ).
noun( 'hatred', 'hatreds', both, _ ).
noun( 'hatter', 'hatters', count, _ ).
noun( 'hauberk', 'hauberks', count, _ ).
noun( 'haughtiness', '-', mass, _ ).
noun( 'haul', 'hauls', count, _ ).
noun( 'haulage', '-', mass, _ ).
noun( 'haulier', 'hauliers', count, _ ).
noun( 'haulm', '-', mass, _ ).
noun( 'haunch', 'haunches', count, _ ).
noun( 'haunt', 'haunts', count, _ ).
noun( 'hautboy', 'hautboys', count, _ ).
noun( 'hauteur', '-', mass, _ ).
noun( 'haven', 'havens', count, _ ).
noun( 'haversack', 'haversacks', count, _ ).
noun( 'havoc', '-', mass, _ ).
noun( 'haw', 'haws', count, _ ).
noun( 'haw-haw', 'haw-haws', count, _ ).
noun( 'hawk', 'hawks', count, _ ).
noun( 'hawker', 'hawkers', count, _ ).
noun( 'hawser', 'hawsers', count, _ ).
noun( 'hawthorn', 'hawthorns', count, _ ).
noun( 'hay', '-', mass, _ ).
noun( 'haycock', 'haycocks', count, _ ).
noun( 'hayfork', 'hayforks', count, _ ).
noun( 'haymaker', 'haymakers', count, _ ).
noun( 'haymaking', '-', mass, _ ).
noun( 'hayrick', 'hayricks', count, _ ).
noun( 'haystack', 'haystacks', count, _ ).
noun( 'haywire', '-', mass, _ ).
noun( 'hazard', 'hazards', both, _ ).
noun( 'haze', 'hazes', both, _ ).
noun( 'hazel', 'hazels', both, _ ).
noun( 'haziness', '-', mass, _ ).
noun( 'he-goat', 'he-goats', count, _ ).
noun( 'he-man', 'he-men', count, _ ).
noun( 'head', 'heads', count, _ ).
noun( 'head-hunter', 'head-hunters', count, _ ).
noun( 'headache', 'headaches', both, _ ).
noun( 'headband', 'headbands', count, _ ).
noun( 'headdress', 'headdresses', count, _ ).
noun( 'header', 'headers', count, _ ).
noun( 'headgear', '-', mass, _ ).
noun( 'heading', 'headings', count, _ ).
noun( 'headlamp', 'headlamps', count, _ ).
noun( 'headland', 'headlands', count, _ ).
noun( 'headlight', 'headlights', count, _ ).
noun( 'headline', 'headlines', count, _ ).
noun( 'headman', 'headmen', count, _ ).
noun( 'headmaster', 'headmasters', count, _ ).
noun( 'headmistress', 'headmistresses', count, _ ).
noun( 'headpiece', 'headpieces', count, _ ).
noun( 'headrest', 'headrests', count, _ ).
noun( 'headroom', '-', mass, _ ).
noun( 'headset', 'headsets', count, _ ).
noun( 'headship', 'headships', count, _ ).
noun( 'headstall', 'headstalls', count, _ ).
noun( 'headstone', 'headstones', count, _ ).
noun( 'headway', '-', mass, _ ).
noun( 'headwind', 'headwinds', count, _ ).
noun( 'headword', 'headwords', count, _ ).
noun( 'healer', 'healers', count, _ ).
noun( 'health', '-', mass, _ ).
noun( 'heap', 'heaps', count, _ ).
noun( 'hearer', 'hearers', count, _ ).
noun( 'hearing', 'hearings', both, _ ).
noun( 'hearing-aid', 'hearing-aids', count, _ ).
noun( 'hearsay', '-', mass, _ ).
noun( 'hearse', 'hearses', count, _ ).
noun( 'heart', 'hearts', both, _ ).
noun( 'heart\'s-ease', '-', mass, _ ).
noun( 'heart-disease', 'heart-diseases', both, _ ).
noun( 'heart-failure', '-', mass, _ ).
noun( 'heartache', 'heartaches', both, _ ).
noun( 'heartbeat', 'heartbeats', count, _ ).
noun( 'heartbreak', '-', mass, _ ).
noun( 'heartburn', '-', mass, _ ).
noun( 'heartburning', '-', mass, _ ).
noun( 'hearth', 'hearths', count, _ ).
noun( 'hearthrug', 'hearthrugs', count, _ ).
noun( 'heartlessness', '-', mass, _ ).
noun( 'heat', 'heats', both, _ ).
noun( 'heat-flash', 'heat-flashes', count, _ ).
noun( 'heater', 'heaters', count, _ ).
noun( 'heath', 'heaths', both, _ ).
noun( 'heathen', 'heathens', count, _ ).
noun( 'heather', 'heathers', both, _ ).
noun( 'heather-mixture', '-', mass, _ ).
noun( 'heating', '-', mass, _ ).
noun( 'heatspot', 'heatspots', count, _ ).
noun( 'heatstroke', '-', mass, _ ).
noun( 'heatwave', 'heatwaves', count, _ ).
noun( 'heave', 'heaves', count, _ ).
noun( 'heaven', 'heavens', count, _ ).
noun( 'heaviness', '-', mass, _ ).
noun( 'heavyweight', 'heavyweights', count, _ ).
noun( 'hecatomb', 'hecatombs', count, _ ).
noun( 'heck', 'hecks', count, _ ).
noun( 'heckler', 'hecklers', count, _ ).
noun( 'hectare', 'hectares', count, _ ).
noun( 'hedge', 'hedges', count, _ ).
noun( 'hedge-sparrow', 'hedge-sparrows', count, _ ).
noun( 'hedgehog', 'hedgehogs', count, _ ).
noun( 'hedgerow', 'hedgerows', count, _ ).
noun( 'hedonism', '-', mass, _ ).
noun( 'hedonist', 'hedonists', count, _ ).
noun( 'heed', '-', mass, _ ).
noun( 'heehaw', 'heehaws', count, _ ).
noun( 'heel', 'heels', count, _ ).
noun( 'hegemony', 'hegemonies', both, _ ).
noun( 'heifer', 'heifers', count, _ ).
noun( 'height', 'heights', both, _ ).
noun( 'heinousness', '-', mass, _ ).
noun( 'heir', 'heirs', count, _ ).
noun( 'heiress', 'heiresses', count, _ ).
noun( 'heirloom', 'heirlooms', count, _ ).
noun( 'helicopter', 'helicopters', count, _ ).
noun( 'heliograph', 'heliographs', count, _ ).
noun( 'heliotrope', 'heliotropes', count, _ ).
noun( 'heliport', 'heliports', count, _ ).
noun( 'helium', '-', mass, _ ).
noun( 'hell', 'hells', count, _ ).
noun( 'hellcat', 'hellcats', count, _ ).
noun( 'helm', 'helms', count, _ ).
noun( 'helmet', 'helmets', count, _ ).
noun( 'helmsman', 'helmsmen', count, _ ).
noun( 'helot', 'helots', count, _ ).
noun( 'help', 'helps', both, _ ).
noun( 'helper', 'helpers', count, _ ).
noun( 'helpfulness', '-', mass, _ ).
noun( 'helping', 'helpings', count, _ ).
noun( 'helplessness', '-', mass, _ ).
noun( 'helpmate', 'helpmates', count, _ ).
noun( 'helpmeet', 'helpmeets', count, _ ).
noun( 'helter-skelter', 'helter-skelters', count, _ ).
noun( 'helve', 'helves', count, _ ).
noun( 'hem', 'hems', count, _ ).
noun( 'hematite', 'hematites', count, _ ).
noun( 'hemisphere', 'hemispheres', count, _ ).
noun( 'hemline', 'hemlines', count, _ ).
noun( 'hemlock', 'hemlocks', both, _ ).
noun( 'hemming-stitch', 'hemming-stitches', count, _ ).
noun( 'hemoglobin', '-', mass, _ ).
noun( 'hemophilia', '-', mass, _ ).
noun( 'hemophiliac', 'hemophiliacs', count, _ ).
noun( 'hemorrhage', 'hemorrhages', both, _ ).
noun( 'hemp', '-', mass, _ ).
noun( 'hemstitch', 'hemstitches', count, _ ).
noun( 'hen', 'hens', count, _ ).
noun( 'hen-party', 'hen-parties', count, _ ).
noun( 'henbane', '-', mass, _ ).
noun( 'henchman', 'henchmen', count, _ ).
noun( 'hencoop', 'hencoops', count, _ ).
noun( 'henhouse', 'henhouses', count, _ ).
noun( 'henna', '-', mass, _ ).
noun( 'henroost', 'henroosts', count, _ ).
noun( 'hepatitis', '-', mass, _ ).
noun( 'heptagon', 'heptagons', count, _ ).
noun( 'herald', 'heralds', count, _ ).
noun( 'heraldry', '-', mass, _ ).
noun( 'herb', 'herbs', count, _ ).
noun( 'herbage', '-', mass, _ ).
noun( 'herbalist', 'herbalists', count, _ ).
noun( 'herd', 'herds', count, _ ).
noun( 'herdsman', 'herdsmen', count, _ ).
noun( 'hereafter', '-', count, _ ).
noun( 'hereditament', 'hereditaments', count, _ ).
noun( 'heredity', '-', mass, _ ).
noun( 'heresy', 'heresies', both, _ ).
noun( 'heretic', 'heretics', count, _ ).
noun( 'heritage', '-', count, _ ).
noun( 'hermaphrodite', 'hermaphrodites', count, _ ).
noun( 'hermit', 'hermits', count, _ ).
noun( 'hermitage', 'hermitages', count, _ ).
noun( 'hernia', 'hernias', both, _ ).
noun( 'hero', 'heroes', count, _ ).
noun( 'heroics', 'heroics', mass, _ ).
noun( 'heroin', '-', mass, _ ).
noun( 'heroine', 'heroines', count, _ ).
noun( 'heroism', '-', mass, _ ).
noun( 'heron', 'herons', count, _ ).
noun( 'heronry', 'heronries', count, _ ).
noun( 'herring', 'herring', both, _ ).
noun( 'herringbone', '-', mass, _ ).
noun( 'hertz', 'hertz', count, _ ).
noun( 'hesitance', '-', both, _ ).
noun( 'hesitancy', '-', mass, _ ).
noun( 'hesitation', 'hesitations', both, _ ).
noun( 'hessian', '-', mass, _ ).
noun( 'heterodoxy', '-', mass, _ ).
noun( 'heterosexual', 'heterosexuals', count, _ ).
noun( 'heterosexuality', '-', mass, _ ).
noun( 'heuristics', 'heuristics', mass, _ ).
noun( 'hewer', 'hewers', count, _ ).
noun( 'hexagon', 'hexagons', count, _ ).
noun( 'hexameter', 'hexameters', count, _ ).
noun( 'heyday', '-', count, _ ).
noun( 'hi-fi', 'hi-fis', count, _ ).
noun( 'hiatus', 'hiatuses', count, _ ).
noun( 'hibernation', '-', mass, _ ).
noun( 'hibiscus', '-', mass, _ ).
noun( 'hiccough', 'hiccoughs', count, _ ).
noun( 'hiccup', 'hiccups', count, _ ).
noun( 'hick', 'hicks', count, _ ).
noun( 'hickory', 'hickories', both, _ ).
noun( 'hide', 'hides', both, _ ).
noun( 'hide-and-seek', '-', mass, _ ).
noun( 'hide-out', 'hide-outs', count, _ ).
noun( 'hideaway', 'hideaways', count, _ ).
noun( 'hiding', 'hidings', both, _ ).
noun( 'hiding-place', 'hiding-places', count, _ ).
noun( 'hierarchy', 'hierarchies', count, _ ).
noun( 'hieroglyph', 'hieroglyphs', count, _ ).
noun( 'hieroglyphics', 'hieroglyphics', mass, _ ).
noun( 'high-fidelity', '-', mass, _ ).
noun( 'high-frequency', 'high-frequencies', both, _ ).
noun( 'high-mindedness', '-', mass, _ ).
noun( 'high-pressure', '-', mass, _ ).
noun( 'high-up', 'high-ups', count, _ ).
noun( 'highball', 'highballs', count, _ ).
noun( 'highboy', 'highboys', count, _ ).
noun( 'highbrow', 'highbrows', count, _ ).
noun( 'highflier', 'highfliers', count, _ ).
noun( 'highflyer', 'highflyers', count, _ ).
noun( 'highjack', 'highjacks', count, _ ).
noun( 'highland', 'highlands', count, _ ).
noun( 'highlight', 'highlights', count, _ ).
noun( 'highness', 'highnesses', both, _ ).
noun( 'highroad', 'highroads', count, _ ).
noun( 'highway', 'highways', count, _ ).
noun( 'highwayman', 'highwaymen', count, _ ).
noun( 'hijack', 'hijacks', count, _ ).
noun( 'hijacker', 'hijackers', count, _ ).
noun( 'hike', 'hikes', count, _ ).
noun( 'hiker', 'hikers', count, _ ).
noun( 'hilarity', '-', mass, _ ).
noun( 'hill', 'hills', count, _ ).
noun( 'hill-billy', 'hill-billies', count, _ ).
noun( 'hillock', 'hillocks', count, _ ).
noun( 'hillside', 'hillsides', count, _ ).
noun( 'hilt', 'hilts', count, _ ).
noun( 'hind', 'hinds', count, _ ).
noun( 'hindrance', 'hindrances', count, _ ).
noun( 'hindsight', '-', mass, _ ).
noun( 'hinge', 'hinges', count, _ ).
noun( 'hint', 'hints', count, _ ).
noun( 'hinterland', 'hinterlands', count, _ ).
noun( 'hip', 'hips', count, _ ).
noun( 'hip-bath', 'hip-baths', count, _ ).
noun( 'hip-flask', 'hip-flasks', count, _ ).
noun( 'hip-pocket', 'hip-pockets', count, _ ).
noun( 'hippie', 'hippies', count, _ ).
noun( 'hippo', 'hippos', count, _ ).
noun( 'hippodrome', 'hippodromes', count, _ ).
noun( 'hippopotamus', 'hippopotamuses', count, _ ).
noun( 'hippy', 'hippies', count, _ ).
noun( 'hire', '-', mass, _ ).
noun( 'hireling', 'hirelings', count, _ ).
noun( 'hiss', 'hisses', count, _ ).
noun( 'histogram', 'histograms', count, _ ).
noun( 'historian', 'historians', count, _ ).
noun( 'history', 'histories', both, _ ).
noun( 'histrionics', 'histrionics', mass, _ ).
noun( 'hit', 'hits', count, _ ).
noun( 'hitch', 'hitches', count, _ ).
noun( 'hitchhiker', 'hitchhikers', count, _ ).
noun( 'hive', 'hives', count, _ ).
noun( 'hoard', 'hoards', count, _ ).
noun( 'hoarder', 'hoarders', count, _ ).
noun( 'hoarding', 'hoardings', count, _ ).
noun( 'hoarfrost', '-', mass, _ ).
noun( 'hoariness', '-', mass, _ ).
noun( 'hoarseness', '-', mass, _ ).
noun( 'hoax', 'hoaxes', count, _ ).
noun( 'hoaxer', 'hoaxers', count, _ ).
noun( 'hob', 'hobs', count, _ ).
noun( 'hobble', 'hobbles', count, _ ).
noun( 'hobble-skirt', 'hobble-skirts', count, _ ).
noun( 'hobbledehoy', 'hobbledehoys', count, _ ).
noun( 'hobby', 'hobbies', count, _ ).
noun( 'hobbyhorse', 'hobbyhorses', count, _ ).
noun( 'hobgoblin', 'hobgoblins', count, _ ).
noun( 'hobnail', 'hobnails', count, _ ).
noun( 'hobo', 'hobos', count, _ ).
noun( 'hock', 'hocks', both, _ ).
noun( 'hockey', '-', mass, _ ).
noun( 'hocus-pocus', '-', mass, _ ).
noun( 'hod', 'hods', count, _ ).
noun( 'hodgepodge', '-', mass, _ ).
noun( 'hoe', 'hoes', count, _ ).
noun( 'hog', 'hogs', count, _ ).
noun( 'hogshead', 'hogsheads', count, _ ).
noun( 'hogwash', '-', mass, _ ).
noun( 'hoist', 'hoists', count, _ ).
noun( 'hold', 'holds', both, _ ).
noun( 'hold-up', 'hold-ups', count, _ ).
noun( 'holdall', 'holdalls', count, _ ).
noun( 'holder', 'holders', count, _ ).
noun( 'holding', 'holdings', count, _ ).
noun( 'holdup', 'holdups', count, _ ).
noun( 'hole', 'holes', count, _ ).
noun( 'holiday', 'holidays', count, _ ).
noun( 'holiday-maker', 'holiday-makers', count, _ ).
noun( 'holiness', '-', mass, _ ).
noun( 'holloa', 'holloas', count, _ ).
noun( 'hollow', 'hollows', count, _ ).
noun( 'holly', '-', mass, _ ).
noun( 'hollyhock', 'hollyhocks', count, _ ).
noun( 'holm-oak', 'holm-oaks', count, _ ).
noun( 'holocaust', 'holocausts', count, _ ).
noun( 'holograph', 'holographs', count, _ ).
noun( 'holster', 'holsters', count, _ ).
noun( 'holy', 'holies', count, _ ).
noun( 'holystone', '-', mass, _ ).
noun( 'homage', '-', mass, _ ).
noun( 'home', 'homes', count, _ ).
noun( 'home-farm', 'home-farms', count, _ ).
noun( 'homecoming', 'homecomings', count, _ ).
noun( 'homeland', 'homelands', count, _ ).
noun( 'homeliness', '-', mass, _ ).
noun( 'homeopath', 'homeopaths', count, _ ).
noun( 'homesickness', '-', mass, _ ).
noun( 'homespun', '-', mass, _ ).
noun( 'homestead', 'homesteads', count, _ ).
noun( 'homework', '-', mass, _ ).
noun( 'homicide', 'homicides', both, _ ).
noun( 'homiletics', 'homiletics', mass, _ ).
noun( 'homily', 'homilies', count, _ ).
noun( 'hominy', '-', mass, _ ).
noun( 'homo', 'homos', count, _ ).
noun( 'homo sapiens', '-', count, _ ).
noun( 'homoeopath', 'homoeopaths', count, _ ).
noun( 'homoeopathy', '-', mass, _ ).
noun( 'homogeneity', '-', mass, _ ).
noun( 'homograph', 'homographs', count, _ ).
noun( 'homonym', 'homonyms', count, _ ).
noun( 'homophone', 'homophones', count, _ ).
noun( 'homosexual', 'homosexuals', count, _ ).
noun( 'homosexuality', '-', mass, _ ).
noun( 'hone', 'hones', count, _ ).
noun( 'honesty', '-', mass, _ ).
noun( 'honey', 'honeys', both, _ ).
noun( 'honeybee', 'honeybees', count, _ ).
noun( 'honeycomb', 'honeycombs', both, _ ).
noun( 'honeydew', '-', mass, _ ).
noun( 'honeymoon', 'honeymoons', count, _ ).
noun( 'honeysuckle', '-', mass, _ ).
noun( 'honk', 'honks', count, _ ).
noun( 'honorarium', 'honorariums', count, _ ).
noun( 'honorific', 'honorifics', count, _ ).
noun( 'honour', 'honours', both, _ ).
noun( 'hooch', '-', mass, _ ).
noun( 'hood', 'hoods', count, _ ).
noun( 'hoodlum', 'hoodlums', count, _ ).
noun( 'hoodoo', 'hoodoos', count, _ ).
noun( 'hooey', '-', mass, _ ).
noun( 'hoof', 'hoofs', count, _ ).
noun( 'hook', 'hooks', count, _ ).
noun( 'hook-up', 'hook-ups', count, _ ).
noun( 'hookah', 'hookahs', count, _ ).
noun( 'hooker', 'hookers', count, _ ).
noun( 'hookworm', 'hookworms', both, _ ).
noun( 'hooky', '-', count, _ ).
noun( 'hooligan', 'hooligans', count, _ ).
noun( 'hooliganism', '-', mass, _ ).
noun( 'hoop', 'hoops', count, _ ).
noun( 'hoop-la', '-', mass, _ ).
noun( 'hoot', 'hoots', count, _ ).
noun( 'hooter', 'hooters', count, _ ).
noun( 'hop', 'hops', count, _ ).
noun( 'hop-field', 'hop-fields', count, _ ).
noun( 'hop-garden', 'hop-gardens', count, _ ).
noun( 'hop-picker', 'hop-pickers', count, _ ).
noun( 'hop-pole', 'hop-poles', count, _ ).
noun( 'hope', 'hopes', both, _ ).
noun( 'hopefulness', '-', mass, _ ).
noun( 'hopelessness', '-', mass, _ ).
noun( 'hopper', 'hoppers', count, _ ).
noun( 'hopscotch', '-', mass, _ ).
noun( 'horde', 'hordes', count, _ ).
noun( 'horizon', 'horizons', count, _ ).
noun( 'hormone', 'hormones', count, _ ).
noun( 'horn', 'horns', both, _ ).
noun( 'hornbeam', 'hornbeams', count, _ ).
noun( 'hornbill', 'hornbills', count, _ ).
noun( 'hornet', 'hornets', count, _ ).
noun( 'hornpipe', 'hornpipes', count, _ ).
noun( 'horology', '-', mass, _ ).
noun( 'horoscope', 'horoscopes', count, _ ).
noun( 'horridness', '-', mass, _ ).
noun( 'horror', 'horrors', both, _ ).
noun( 'horse', 'horses', count, _ ).
noun( 'horse-chestnut', 'horse-chestnuts', count, _ ).
noun( 'horse-laugh', 'horse-laughs', count, _ ).
noun( 'horse-pond', 'horse-ponds', count, _ ).
noun( 'horse-sense', '-', mass, _ ).
noun( 'horseback', '-', mass, _ ).
noun( 'horsebox', 'horseboxes', count, _ ).
noun( 'horseflesh', '-', mass, _ ).
noun( 'horsefly', 'horseflies', count, _ ).
noun( 'horsehair', '-', mass, _ ).
noun( 'horseman', 'horsemen', count, _ ).
noun( 'horsemanship', '-', mass, _ ).
noun( 'horsemeat', '-', mass, _ ).
noun( 'horseplay', '-', mass, _ ).
noun( 'horsepower', 'horsepower', both, _ ).
noun( 'horserace', 'horseraces', count, _ ).
noun( 'horseracing', '-', mass, _ ).
noun( 'horseradish', '-', mass, _ ).
noun( 'horseshoe', 'horseshoes', count, _ ).
noun( 'horsewhip', 'horsewhips', count, _ ).
noun( 'horsewoman', 'horsewomen', count, _ ).
noun( 'horticulture', '-', mass, _ ).
noun( 'horticulturist', 'horticulturists', count, _ ).
noun( 'hosanna', 'hosannas', count, _ ).
noun( 'hose', 'hoses', both, _ ).
noun( 'hosepipe', 'hosepipes', count, _ ).
noun( 'hosier', 'hosiers', count, _ ).
noun( 'hosiery', '-', mass, _ ).
noun( 'hospice', 'hospices', count, _ ).
noun( 'hospital', 'hospitals', count, _ ).
noun( 'hospitality', '-', mass, _ ).
noun( 'hospitalization', 'hospitalizations', both, _ ).
noun( 'host', 'hosts', count, _ ).
noun( 'hostage', 'hostages', count, _ ).
noun( 'hostel', 'hostels', count, _ ).
noun( 'hosteller', 'hostellers', count, _ ).
noun( 'hostelry', 'hostelries', count, _ ).
noun( 'hostess', 'hostesses', count, _ ).
noun( 'hostility', 'hostilities', both, _ ).
noun( 'hot-water-bottle', 'hot-water-bottles', count, _ ).
noun( 'hotbed', 'hotbeds', count, _ ).
noun( 'hotchpotch', 'hotchpotches', count, _ ).
noun( 'hotel', 'hotels', count, _ ).
noun( 'hotelier', 'hoteliers', count, _ ).
noun( 'hothead', 'hotheads', count, _ ).
noun( 'hothouse', 'hothouses', count, _ ).
noun( 'hotplate', 'hotplates', count, _ ).
noun( 'hound', 'hounds', count, _ ).
noun( 'hour', 'hours', count, _ ).
noun( 'hourglass', 'hourglasses', count, _ ).
noun( 'houri', 'houris', count, _ ).
noun( 'house', 'houses', count, _ ).
noun( 'house-party', 'house-parties', count, _ ).
noun( 'house-warming', 'house-warmings', count, _ ).
noun( 'houseboat', 'houseboats', count, _ ).
noun( 'housebreaker', 'housebreakers', count, _ ).
noun( 'housecoat', 'housecoats', count, _ ).
noun( 'housecraft', '-', mass, _ ).
noun( 'housedog', 'housedogs', count, _ ).
noun( 'housefather', 'housefathers', count, _ ).
noun( 'housefly', 'houseflies', count, _ ).
noun( 'houseful', 'housefuls', count, _ ).
noun( 'household', 'households', count, _ ).
noun( 'householder', 'householders', count, _ ).
noun( 'housekeeper', 'housekeepers', count, _ ).
noun( 'housemaid', 'housemaids', count, _ ).
noun( 'houseman', 'housemen', count, _ ).
noun( 'housemaster', 'housemasters', count, _ ).
noun( 'housemother', 'housemothers', count, _ ).
noun( 'houseroom', '-', mass, _ ).
noun( 'housetop', 'housetops', count, _ ).
noun( 'housewife', 'housewives', count, _ ).
noun( 'housewifery', '-', mass, _ ).
noun( 'housework', '-', mass, _ ).
noun( 'housewrecker', 'housewreckers', count, _ ).
noun( 'housing', '-', mass, _ ).
noun( 'hovel', 'hovels', count, _ ).
noun( 'hovercraft', 'hovercraft', count, _ ).
noun( 'how-d\'ye-do', 'how-d\'ye-dos', count, _ ).
noun( 'howdah', 'howdahs', count, _ ).
noun( 'howitzer', 'howitzers', count, _ ).
noun( 'howl', 'howls', count, _ ).
noun(